!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_special_funcs
!! NAME
!! m_special_funcs
!!
!! FUNCTION
!! This module contains routines and functions used to 
!! evaluate special functions frequently needed in Abinit.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MG,MT,FB)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! NOTES
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

module m_special_funcs

 use defs_basis
 use m_errors

 implicit none

 private

 public ::          &
&  jbessel_4spline, &  ! Spherical Bessel functions and derivatives employing a polynomial approximation for q->0
&  ylmc,            &  ! Complex Spherical harmonics for l<=3.
&  ylmcd               ! First derivative of complex Ylm wrt theta and phi up to l<=3

CONTAINS  !===========================================================
!!***

!!****f* m_special_funcs/jbessel_4spline
!! NAME
!!  jbessel_4spline
!!
!! FUNCTION
!!  Compute spherical Bessel functions and derivatives. 
!!  A polynomial approximation is employed for q-->0.
!!  
!! INPUT
!!  ll=l-order of the Bessel function
!!  tol=tolerance below which a Polynomial approximation is employed
!!   both for jl and its derivative (if required)
!!  order=1 if only first derivative is requested
!!        2 if first and second derivatives are requested
!!  xx=where to compute j_l
!!
!! OUTPUT
!!  bes=Spherical Bessel function j_l at xx
!!  besp= first derivative of j_l at xx (only if order>=1)
!!
!! TODO 
!! Remove inline definitions, they are obsolete in F200?
!!
!! PARENTS
!!      paw_mkrhox_spl,psp7nl
!!
!! CHILDREN
!!      leave_new,wrtout
!!
!! SOURCE

subroutine jbessel_4spline(bes,besp,ll,order,xx,tol)
!Arguments ---------------------------------------------
!scalars

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
 use interfaces_32_util
!End of the abilint section

 integer,intent(in) :: ll,order
 real(dp),intent(in) :: xx,tol
 real(dp),intent(out) :: bes,besp

!Local variables ---------------------------------------
!scalars
 real(dp) :: bespp
 character(len=500) :: msg
 real(dp) :: arg,bes0a,bes0ap,bes0b,bes0bp,bes1a,bes1ap,bes1b,bes1bp
 real(dp) :: bes2a,bes2ap,bes2b,bes2bp,bes3a,bes3ap,bes3b,bes3bp
! *********************************************************************

! === l=0,1,2 and 3 spherical Bessel functions (and derivatives) ===
  bes0a(arg)=1.0_dp-arg**2/6.0_dp*(1.0_dp-arg**2/20.0_dp)
  bes0b(arg)=sin(arg)/arg
  bes1a(arg)=(10.0_dp-arg*arg)*arg/30.0_dp
  bes1b(arg)=(sin(arg)-arg*cos(arg))/arg**2
  bes2a(arg)=arg*arg/15.0_dp-arg**4/210.0_dp
  bes2b(arg)=((3.0_dp-arg**2)*sin(arg)-3.0_dp*arg*cos(arg))/arg**3
  bes3a(arg)=arg*arg*arg/105.0_dp-arg**5/1890.0_dp+arg**7/83160.0_dp
  bes3b(arg)=(15.0_dp*sin(arg)-15.0_dp*arg*cos(arg)-6.0_dp*arg**2*sin(arg)+arg**3*cos(arg))/arg**4
  bes0ap(arg)=(-10.0_dp+arg*arg)*arg/30.0_dp
  bes0bp(arg)=-(sin(arg)-arg*cos(arg))/arg**2
  bes1ap(arg)=(10.0_dp-3.0_dp*arg*arg)/30.0_dp
  bes1bp(arg)=((arg*arg-2.0_dp)*sin(arg)+2.0_dp*arg*cos(arg))/arg**3
  bes2ap(arg)=(1.0_dp-arg*arg/7.0_dp)*2.0_dp*arg/15.0_dp
  bes2bp(arg)=((4.0_dp*arg*arg-9.0_dp)*sin(arg)+(9.0_dp-arg*arg)*arg*cos(arg))/arg**4
  bes3ap(arg)=(1.0_dp/35-arg*arg/378.0_dp+arg**4/11880.0_dp)*arg*arg
  bes3bp(arg)=((-60.0_dp+27.0_dp*arg*arg-arg**4)*sin(arg)+(60.0_dp*arg-7.0_dp*arg**3)*cos(arg))/arg**5

  if (order>2) stop "Wrong order in jbessel !"

  select case (ll)

   case (0)
    if (xx<TOL) then
     bes=bes0a(xx)
     if (order>=1) besp=bes0ap(xx)
    else
     bes=bes0b(xx)
     if (order>=1) besp=bes0bp(xx)
    end if

   case (1)
    if (xx<TOL) then
     bes=bes1a(xx)
     if (order>=1) besp=bes1ap(xx)
    else
     bes=bes1b(xx)
     if (order>=1) besp=bes1bp(xx)
    end if

   case (2)
    if (xx<TOL) then
     bes=bes2a(xx)
     if (order>=1) besp=bes2ap(xx)
    else
     bes=bes2b(xx)
     if (order>=1) besp=bes2bp(xx)
    end if

   case (3)
    if (xx<TOL) then
     bes=bes3a(xx)
     if (order>=1) besp=bes3ap(xx)
    else
     bes=bes3b(xx)
     if (order>=1) besp=bes3bp(xx)
    end if

   case (4:)
    call jbessel(bes,besp,bespp,ll,order,xx)

    case default
    write(msg,'(4a,i4)')ch10,&
&    ' jbessel_4spline : BUG - ',ch10,&
&    ' wrong value for ll = ',ll
    call wrtout(std_out,msg,'COLL') 
    call leave_new('COLL')
  end select

 end subroutine jbessel_4spline
!!***


!!****f* m_special_funcs/ylmc
!! NAME
!! ylmc
!!
!! FUNCTION
!!  Return a complex spherical harmonic with l <= 3
!!
!! INPUTS
!!  il=angular quantum number
!!  im=magnetic quantum number
!!  kcart=vector in cartesian coordinates defining the value of \theta and \psi
!!   where calculate the spherical harmonic
!!
!! OUTPUT
!!  ylm= spherical harmonic
!!
!! NOTES
!!  Note the use of double precision complex.
!!  Case l>3 not implemented.
!!
!! PARENTS
!!
!! CHILDREN
!!
!!
!! SOURCE

function ylmc(il,im,kcart)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: il,im
 complex(dpc) :: ylmc
!arrays
 real(dp),intent(in) :: kcart(3)

!Local variables-------------------------------
!scalars
 integer,parameter :: LMAX=3
 real(dp),parameter :: PPAD=tol8
 real(dp) :: cosphi,costh,costhreephi,costwophi,r,rxy,sinphi,sinth,sinthreephi,sintwophi
 character(len=500) :: msg

! *************************************************************************

  if (ABS(im)>ABS(il)) then
   write(msg,'(4a,i6,2a,i6,a,i6)')ch10,&
&   ' ylmc: ERROR -',ch10,&
&   '  value of m is,',im,ch10,&
&   '  however it should be between ',-il,' and ',il
   call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
  end if

  r=SQRT(kcart(1)**2+kcart(2)**2+kcart(3)**2)
  if (r<PPAD) r=r+PPAD
  rxy=SQRT(kcart(1)**2+kcart(2)**2)
  if (rxy<PPAD)rxy=r+PPAD
! 
! Get (th,phi) spherical coordinates.
  costh= kcart(3)/r
  sinth= rxy/r
  cosphi= kcart(1)/rxy
  sinphi= kcart(2)/rxy
  costwophi= two*cosphi**2 - one
  sintwophi= two*sinphi*cosphi
  costhreephi=cosphi*costwophi-sinphi*sintwophi
  sinthreephi=cosphi*sintwophi+sinphi*costwophi

  select case (il)

   case (0)
    ylmc= one/SQRT(four_pi)

   case (1)
    if (ABS(im)==0) then
     ylmc = SQRT(three/(four_pi))*costh
    else if (ABS(im)==1) then
     ylmc = -SQRT(three/(eight*pi))*sinth*CMPLX(cosphi,sinphi)
    end if

   case (2)
    if (ABS(im)==0) then
     ylmc = SQRT(5.d0/(16.d0*pi))*(three*costh**2-one)
    else if (ABS(im)==1) then
     ylmc = -SQRT(15.d0/(8.d0*pi))*sinth*costh*cmplx(cosphi,sinphi)
    else if (ABS(im)==2) then
     ylmc = SQRT(15.d0/(32.d0*pi))*(sinth)**2*CMPLX(costwophi,sintwophi)
    end if

   case (3)
    if (ABS(im)==0) then
     ylmc= SQRT(7.d0/(16.d0*pi))*(5.d0*costh**3 -3.d0*costh)
    else if (ABS(im)==1) then
     ylmc= -SQRT(21.d0/(64.d0*pi))*sinth*(5.d0*costh**2-one)*CMPLX(cosphi,sinphi)
    else if (ABS(im)==2) then
     ylmc= SQRT(105.d0/(32.d0*pi))*sinth**2*costh*CMPLX(costwophi,sintwophi)
    else if (ABS(im)==3) then
     ylmc=-SQRT(35.d0/(64.d0*pi))*sinth**3*CMPLX(costhreephi,sinthreephi)
    end if

    case default
    write(msg,'(4a,i6,2a,i6)')ch10,&
&    ' ylmc: ERROR -',ch10,&
&    '  The maximum allowed value for l is,',LMAX,ch10,&
&    '  however, l=',il
    call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
  end select
! 
! === Treat the case im < 0 ===
  if (im < 0) then
   ylmc=(-one)**(im)*CONJG(ylmc)
  end if

 end function ylmc
!!***

!!****f* m_special_funcs/ylmcd
!! NAME
!! ylmcd
!!
!! FUNCTION
!!  Computes dth and dphi, the first derivatives of complex Ylm as a function of 
!!  th and phi (the angles of the spherical coordinates)
!!  It works for all spherical harmonics with l <= 3
!!
!! INPUTS
!!  il=angular quantum number
!!  im=magnetic quantum number
!!  kcart=cartesian coordinates of the vector where the first derivatives of Ylm are evaluated
!!
!! OUTPUT
!!  dth =derivative of Y_lm with respect to \theta
!!  dphi=derivative of Y_lm with respect to \phi
!!
!! SIDE EFFECTS
!!
!! NOTES
!!  Note the use of double precision complex.
!!  Case l>3 not implemented.
!!
!! PARENTS
!!      m_commutator_vkbr
!!
!! CHILDREN
!!      leave_new,wrtout
!!
!! SOURCE

subroutine ylmcd(il,im,kcart,dth,dphi)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: il,im
 complex(dpc),intent(out) :: dphi,dth
!arrays
 real(dp),intent(in) :: kcart(3)

!Local variables-------------------------------
!scalars
 integer,parameter :: LMAX=3
 real(dp),parameter :: PPAD=tol8
 real(dp) :: cosphi,costh,costhreephi,costwophi,r,rxy,sinphi,sinth,sinthreephi,sintwophi
 character(len=500) :: msg

! *************************************************************************

  if (ABS(im)>ABS(il))then
   write(msg,'(4a,i6,2a,i6,a,i6)') ch10,&
&   ' ylmcd: ERROR -',ch10,&
&   '  value of m is,',im,ch10,&
&   '  however it should be between ',-il,' and ',il
   call wrtout(std_out,msg,'COLL') ; call leave_new('COLL')
  end if

  r=SQRT(kcart(1)**2+kcart(2)**2+kcart(3)**2)
  if (r<PPAD) r=r+PPAD
  rxy=SQRT(kcart(1)**2+kcart(2)**2)
  if (rxy<PPAD) rxy=r+PPAD

! Get (th,ph) spherical coordinates.
  costh= kcart(3)/r
  sinth= rxy/r
  cosphi= kcart(1)/rxy
  sinphi= kcart(2)/rxy
  costwophi= two*cosphi**2 - one
  sintwophi= two*sinphi*cosphi
  costhreephi=cosphi*costwophi-sinphi*sintwophi
  sinthreephi=cosphi*sintwophi+sinphi*costwophi

  select case (il)

   case (0)
    dth  = (0.d0,0.d0)
    dphi = (0.d0,0.d0)

   case (1)
    if (ABS(im)==0) then
     dth= -SQRT(three/(four_pi))*sinth
     dphi= (0.d0,0.d0)
    else if (abs(im)==1) then
     dth= -SQRT(3.d0/(8.d0*pi))*costh*CMPLX(cosphi,sinphi)
     dphi=-SQRT(3.d0/(8.d0*pi))*sinth*CMPLX(-sinphi,cosphi)
    end if

   case (2)
    if (ABS(im)==0) then
     dth= -SQRT(5.d0/(16.d0*pi))*6.d0*costh*sinth
     dphi= (0.d0,0.d0)
    else if (ABS(im)==1) then
     dth=  -SQRT(15.d0/(8.d0*pi))*(costh**2-sinth**2)*CMPLX(cosphi,sinphi)
     dphi= -SQRT(15.d0/(8.d0*pi))*costh*sinth*(0.d0,1.d0)*CMPLX(cosphi,sinphi)
    else if (abs(im)==2) then
     dth  = SQRT(15.d0/(32.d0*pi))*2.d0*costh*sinth*CMPLX(costwophi,sintwophi)
     dphi = SQRT(15.d0/(32.d0*pi))*sinth**2*(0.d0,2.d0)*CMPLX(costwophi,sintwophi)
    end if

   case (3)
    if (ABS(im)==0) then
     dth = SQRT(7.d0/(16*pi))*(-15.d0*costh**2*sinth + 3.d0**sinth)
     dphi= (0.d0,0.d0)
    else if (ABS(im)==1) then
     dth= -SQRT(21.d0/(64.d0*pi))*CMPLX(cosphi,sinphi)*(5.d0*costh**3-costh-10.d0*sinth**2*costh)
     dphi=-SQRT(21.d0/(64.d0*pi))*sinth*(5.d0*costh**2-1)*(0.d0,1.d0)*CMPLX(cosphi,sinphi)
    else if (ABS(im)==2) then
     dth =SQRT(105.d0/(32.d0*pi))*(2.d0*sinth*costh**2-sinth**3)*CMPLX(costwophi,sintwophi)
     dphi=SQRT(105.d0/(32*pi))*sinth**2*costh*(0.d0,2.d0)*CMPLX(costwophi,sintwophi)
    else if (abs(im)==3) then
     dth =-SQRT(35.d0/(64.d0*pi))*3.d0*sinth**2*costh*CMPLX(costhreephi,sinthreephi)
     dphi= SQRT(35.d0/(64.d0*pi))*sinth**3*(0.d0,3.d0)*CMPLX(costhreephi,sinthreephi)
    end if

    case default
    write(msg,'(4a,i6,2a,i6)')ch10,&
&    ' ylmcd: ERROR -',ch10,&
&    '  The maximum allowed value for l is,',LMAX,ch10,&
&    '  however, l=',il
    call wrtout(std_out,msg,'COLL') 
    call leave_new('COLL')
  end select
! 
! === Treat the case im < 0 ===
  if (im<0) then
   dth = (-one)**(im)*CONJG(dth)
   dphi= (-one)**(im)*CONJG(dphi)
  end if

 end subroutine ylmcd

end module m_special_funcs
!!***
