!{\src2tex{textfont=tt}}
!!****f* ABINIT/contstr25a
!! NAME
!! contstr25a
!!
!! FUNCTION
!! Carries out specialized metric tensor operations needed for contraction
!! of the 2nd strain derivative of the l=0,1,2,3 nonlocal Kleinman-Bylander
!! pseudopotential operation.  Derivatives are wrt a pair of cartesian
!! strain components.
!! Full advantage is taken of the full permutational symmetry of these
!! tensors.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2009 ABINIT group (DRH)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!!  istr1=1,...6 specifies cartesian strain component 11,22,33,32,31,21
!!  istr2=seccond strain component
!!  rank=angular momentum
!!  gm(3,3)=metric tensor (array is symmetric but stored as 3x3)
!!  gprimd(3,3)=reciprocal space dimensional primitive translations
!!  aa(2,*)=unique elements of complex right-hand tensor
!!  bb(2,*)=unique elements of complex left-hand tensor
!!
!! OUTPUT
!!  e2nl=contraction for nonlocal 2nd-order strain derivative energy
!!
!! NOTES
!! All tensors are stored in a compressed storage mode defined below;
!! input and output conform to this scheme.
!! When tensor elements occur repeatedly due to symmetry, the
!! WEIGHT IS INCLUDED in the output tensor element to simplify later
!! contractions with other tensors of the same rank and form, i.e. the
!! next contraction is then simply a dot product over the unique elements.
!!
!! PARENTS
!!      nonlop_pl
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine contstr25a(istr1,istr2,rank,gm,gprimd,e2nl,aa,bb)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istr1,istr2,rank
 real(dp),intent(out) :: e2nl
!arrays
 real(dp),intent(in) :: aa(2,((rank+3)*(rank+4))/2),bb(2,((rank+3)*(rank+4))/2)
 real(dp),intent(in) :: gm(3,3),gprimd(3,3)

!Local variables-------------------------------
!scalars
 integer,parameter :: mrank=3
 integer :: ii,jj,ka,kb,kd,kg
!arrays
 integer,save :: idx(12)=(/1,1,2,2,3,3,3,2,3,1,2,1/)
 real(dp) :: d2gm(3,3),dgm01(3,3),dgm10(3,3),tmp(2)
 real(dp),allocatable :: cm(:,:)

! *************************************************************************
 allocate(cm(((mrank+3)*(mrank+4))/2,((mrank+3)*(mrank+4))/2))

 ka=idx(2*istr1-1);kb=idx(2*istr1);kg=idx(2*istr2-1);kd=idx(2*istr2)

 do ii = 1,3
  dgm01(:,ii)=-(gprimd(ka,:)*gprimd(kb,ii)+gprimd(kb,:)*gprimd(ka,ii))
  dgm10(:,ii)=-(gprimd(kg,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kg,ii))
 end do

 d2gm(:,:)=0.d0
 do ii = 1,3
  if(ka==kg) d2gm(:,ii)=d2gm(:,ii)&
&  +gprimd(kb,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kb,ii)
  if(ka==kd) d2gm(:,ii)=d2gm(:,ii)&
&  +gprimd(kb,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(kb,ii)
  if(kb==kg) d2gm(:,ii)=d2gm(:,ii)&
&  +gprimd(ka,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(ka,ii)
  if(kb==kd) d2gm(:,ii)=d2gm(:,ii)&
&  +gprimd(ka,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(ka,ii)
 end do
 d2gm(:,:)=0.5d0*d2gm(:,:)

!
!The code below was written by a Mathematica program and formatted by
!a combination of editing scripts.  It is not intended to be read
!by human beings, and certainly not to be modified by one.  Conceivably
!it could be shortened somewhat by identifying common subexpressions.
!
 if(rank==3)then
  cm(1,1)=2*gm(1,1)**3*dgm01(1,1)*dgm10(1,1)
  cm(2,1)=(gm(1,1)*(240*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,1)&
&  *dgm10(1,1)+8640*gm(1,1)*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+1440*gm(1,1)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))))/1440.d0
  cm(3,1)=(gm(1,1)*(240*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+8640*gm(1,1)*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+1440*gm(1,1)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))))/1440.d0
  cm(4,1)=(gm(1,1)*(240*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *dgm01(1,1)*dgm10(1,1)+4320*gm(1,1)*gm(1,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+4320*gm(1,1)*gm(1,2)*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+1440*gm(1,1)**2*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))))/720.d0
  cm(5,1)=gm(1,1)**2*(6*gm(1,3)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))
  cm(6,1)=gm(1,1)**2*(6*gm(1,2)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))
  cm(7,1)=(80*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)&
&  *dgm10(1,1)+240*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+4320*gm(1,1)**2*gm(1,2)*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2)))/1440.d0
  cm(8,1)=(80*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+80*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +160*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2880*gm(1,1)**2*gm(1,3)*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+1440*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/480.d0
  cm(9,1)=(80*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+160*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+80*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+1440*gm(1,1)&
&  **2*gm(1,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2880*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3)))/480.d0
  cm(10,1)=(80*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+240*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4320*gm(1,1)**2*gm(1,3)*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/1440.d0
  cm(11,1)=(160*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+240*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)))/2880.d0
  cm(12,1)=(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+4*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&  *gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+gm(1,1)&
&  *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3)))/12.d0
  cm(13,1)=(3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +3*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3)))/18.d0
  cm(14,1)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+3*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3)))/18.d0
  cm(15,1)=(160*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+240*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/2880.d0
  cm(16,1)=((90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2)))/36.d0
  cm(17,1)=(3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3)))/36.d0
  cm(18,1)=(240*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+160*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3)))/2880.d0
  cm(19,1)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3)))/36.d0
  cm(20,1)=(160*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+240*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3)))/2880.d0
  cm(21,1)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/36.d0
  cm(1,2)=(gm(1,1)*(240*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,1)&
&  *dgm10(1,1)+8640*gm(1,1)*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+1440*gm(1,1)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))))/1440.d0
  cm(2,2)=6*gm(1,2)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,1)*(12*gm(2,2)**2*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)**2*dgm01(2,2)&
&  *dgm10(2,2)+gm(1,1)*gm(2,2)*(-3*dgm01(2,2)*dgm10(1,1)+48*dgm01(1,2)&
&  *dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)))+gm(1,2)**2*(6*gm(2,2)*dgm01(1,1)&
&  *dgm10(1,1)+gm(1,1)*(9*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)&
&  +9*dgm01(1,1)*dgm10(2,2)))+gm(1,1)*gm(1,2)*(30*gm(2,2)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*gm(1,1)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2)))
  cm(3,2)=(72*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,1)*dgm10(1,1)+72*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+72*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+144*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+12*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +864*gm(1,1)**2*gm(1,3)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+864*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+144*gm(1,1)**3*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/144.d0
  cm(4,2)=(72*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+144*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*dgm01(1,2)*dgm10(1,2)&
&  +72*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +36*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +36*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+72*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+12*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+432*gm(1,1)**2*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+432*gm(1,1)**2*gm(1,2)*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+864*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+144*gm(1,1)&
&  **3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(5,2)=(144*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+144*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+48*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3456*gm(1,1)&
&  **2*gm(1,2)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+864*gm(1,1)&
&  **2*gm(1,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+576*gm(1,1)&
&  **3*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))/288.d0
  cm(6,2)=3*gm(1,2)**3*dgm01(1,1)*dgm10(1,1)+15*gm(1,1)*gm(1,2)&
&  **2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*gm(1,2)&
&  *(15*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(2,2)*dgm10(1,1)&
&  +24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)))+gm(1,1)**2*(9*gm(2,2)&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+2*gm(1,1)*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(7,2)=gm(1,2)**3*(4*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&  +4*dgm01(1,1)*dgm10(2,2))+gm(1,2)*(6*gm(2,2)**2*dgm01(1,1)*dgm10(1,1)&
&  +6*gm(1,1)**2*dgm01(2,2)*dgm10(2,2)+gm(1,1)*gm(2,2)*(6*dgm01(2,2)&
&  *dgm10(1,1)+60*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)*dgm10(2,2)))&
&  +15*gm(1,2)**2*(gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+9*gm(1,1)&
&  *gm(2,2)*(gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(8,2)=(24*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*dgm01(1,1)*dgm10(1,1)+96*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)&
&  +24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +48*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +96*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +4*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+8*gm(1,1)&
&  *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+16*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*gm(1,1)*(12*gm(1,2)&
&  *gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+288*gm(1,1)**2*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+12*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+144*gm(1,1)&
&  **2*gm(1,2)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/48.d0
  cm(9,2)=(24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *dgm01(1,1)*dgm10(1,1)+192*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+24*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+48*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*gm(2,2)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +48*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+288*gm(1,1)**2*gm(1,3)*dgm01(2,2)*dgm10(2,2)&
&  +4*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +16*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +8*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+24*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*gm(1,1)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +288*gm(1,1)**2*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))&
&  /48.d0
  cm(10,2)=(24*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+24*(90*gm(1,3)&
&  **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+72*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+144*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+24*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+36*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+72*gm(1,1)*(12*gm(1,2)&
&  *gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+432*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/144.d0
  cm(11,2)=(2304*gm(1,2)**3*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+gm(1,2)**2*(2592*gm(1,1)*dgm01(2,2)*dgm10(2,2)+gm(2,2)&
&  *(864*dgm01(2,2)*dgm10(1,1)+10368*dgm01(1,2)*dgm10(1,2)+864*dgm01(1,1)&
&  *dgm10(2,2)))+gm(1,1)*gm(2,2)*(-864*gm(1,1)*dgm01(2,2)*dgm10(2,2)&
&  +gm(2,2)*(1728*dgm01(2,2)*dgm10(1,1)-3456*dgm01(1,2)*dgm10(1,2)&
&  +1728*dgm01(1,1)*dgm10(2,2)))+gm(1,2)*gm(2,2)*(1728*gm(2,2)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+3456*gm(1,1)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))))/288.d0
  cm(12,2)=(24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,2)*dgm10(1,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+2*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*dgm01(2,2)*dgm10(2,2)+3*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +24*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +4*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/12.d0
  cm(13,2)=(144*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+36*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+432*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+24*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,2)*dgm10(2,2)&
&  +36*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +72*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+36*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+72*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(14,2)=(12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&  *gm(2,3))*gm(3,3))*dgm01(1,2)*dgm10(1,2)+3*(-36*gm(1,2)*gm(2,3)&
&  *gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+18*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+(90*gm(1,3)&
&  **3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+9*(6*gm(1,2)*gm(1,3)&
&  *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/18.d0
  cm(15,2)=(6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(16,2)=(80*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)&
&  *dgm10(2,2)+4320*gm(1,2)*gm(2,2)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+240*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))/1440.d0
  cm(17,2)=(24*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+36*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+72*(30*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +72*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+144*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+24*(90*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+432*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/144.d0
  cm(18,2)=(48*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+72*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+144*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+1728*gm(1,2)*gm(2,2)**2*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+96*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+16*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3)))/288.d0
  cm(19,2)=(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +18*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+36*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+9*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(20,2)=(6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+18*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(21,2)=(3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(1,3)=(gm(1,1)*(240*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+8640*gm(1,1)*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+1440*gm(1,1)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))))/1440.d0
  cm(2,3)=(72*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,1)*dgm10(1,1)+72*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+72*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+144*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+12*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +864*gm(1,1)**2*gm(1,3)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+864*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+144*gm(1,1)**3*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/144.d0
  cm(3,3)=6*gm(1,3)**3*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(12*gm(3,3)**2*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)**2*dgm01(3,3)&
&  *dgm10(3,3)+gm(1,1)*gm(3,3)*(-3*dgm01(3,3)*dgm10(1,1)+48*dgm01(1,3)&
&  *dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)))+gm(1,3)**2*(6*gm(3,3)*dgm01(1,1)&
&  *dgm10(1,1)+gm(1,1)*(9*dgm01(3,3)*dgm10(1,1)+24*dgm01(1,3)*dgm10(1,3)&
&  +9*dgm01(1,1)*dgm10(3,3)))+gm(1,1)*gm(1,3)*(30*gm(3,3)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*gm(1,1)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))
  cm(4,3)=(72*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,1)*dgm10(1,1)+36*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+144*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *dgm01(1,3)*dgm10(1,3)+36*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+72*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +72*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+864*gm(1,1)&
&  **2*gm(1,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+12*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+432*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+432*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+144*gm(1,1)**3*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(5,3)=3*gm(1,3)**3*dgm01(1,1)*dgm10(1,1)+15*gm(1,1)*gm(1,3)&
&  **2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)*gm(1,3)&
&  *(15*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)*dgm10(1,1)&
&  +24*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)))+gm(1,1)**2*(9*gm(3,3)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*gm(1,1)*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(6,3)=(144*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+48*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +144*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3456*gm(1,1)**2*gm(1,3)*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+864*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+576*gm(1,1)**3*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/288.d0
  cm(7,3)=(24*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+72*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+24*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+144*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+36*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+72*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+24*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+432*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/144.d0
  cm(8,3)=(24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(1,1)*dgm10(1,1)+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+192*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+24*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+48*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+48*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +48*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+288*gm(1,1)**2*gm(1,2)*dgm01(3,3)*dgm10(3,3)&
&  +12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+8*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +16*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+24*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +288*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /48.d0
  cm(9,3)=(24*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*dgm01(1,1)*dgm10(1,1)+48*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+96*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+96*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+12*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +24*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+24*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+48*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+4*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+16*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+8*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+144*gm(1,1)**2*gm(1,3)*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+288*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(10,3)=gm(1,3)**3*(4*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&  +4*dgm01(1,1)*dgm10(3,3))+gm(1,3)*(6*gm(3,3)**2*dgm01(1,1)*dgm10(1,1)&
&  +6*gm(1,1)**2*dgm01(3,3)*dgm10(3,3)+gm(1,1)*gm(3,3)*(6*dgm01(3,3)&
&  *dgm10(1,1)+60*dgm01(1,3)*dgm10(1,3)+6*dgm01(1,1)*dgm10(3,3)))&
&  +15*gm(1,3)**2*(gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+9*gm(1,1)&
&  *gm(3,3)*(gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(11,3)=(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +12*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /36.d0
  cm(12,3)=(6*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*gm(3,3)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+24*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+4*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /12.d0
  cm(13,3)=(9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(90*gm(1,2)&
&  **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(1,3)&
&  *dgm10(1,3)+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +3*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(14,3)=(432*gm(1,3)*gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+144*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,3)*dgm10(1,3)+36*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+24*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+36*gm(3,3)&
&  *(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+72*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(3,3)+36*(6*gm(1,3)**2*gm(2,3)&
&  +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+72*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(15,3)=(2304*gm(1,3)**3*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+gm(1,3)**2*(2592*gm(1,1)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  *(864*dgm01(3,3)*dgm10(1,1)+10368*dgm01(1,3)*dgm10(1,3)+864*dgm01(1,1)&
&  *dgm10(3,3)))+gm(1,1)*gm(3,3)*(-864*gm(1,1)*dgm01(3,3)*dgm10(3,3)&
&  +gm(3,3)*(1728*dgm01(3,3)*dgm10(1,1)-3456*dgm01(1,3)*dgm10(1,3)&
&  +1728*dgm01(1,1)*dgm10(3,3)))+gm(1,3)*gm(3,3)*(1728*gm(3,3)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3456*gm(1,1)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))))/288.d0
  cm(16,3)=(3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(17,3)=(9*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+18*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+36*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(18,3)=(9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+12*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(19,3)=(432*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+24*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+72*gm(3,3)*(12*gm(1,3)*gm(2,3)&
&  +24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +144*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +24*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3)+36*(48*gm(1,2)*gm(2,3)&
&  *gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+72*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+24*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/144.d0
  cm(20,3)=(1728*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+96*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +72*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+144*(6*gm(1,3)**2*gm(2,3)&
&  +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+16*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /288.d0
  cm(21,3)=(80*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+4320*gm(1,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+240*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))/1440.d0
  cm(1,4)=(gm(1,1)*(240*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *dgm01(1,1)*dgm10(1,1)+4320*gm(1,1)*gm(1,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+4320*gm(1,1)*gm(1,2)*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+1440*gm(1,1)**2*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))))/720.d0
  cm(2,4)=(72*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+144*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*dgm01(1,2)*dgm10(1,2)&
&  +72*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +36*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +36*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+72*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+12*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+432*gm(1,1)**2*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+432*gm(1,1)**2*gm(1,2)*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+864*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+144*gm(1,1)&
&  **3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(3,4)=(72*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,1)*dgm10(1,1)+36*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+144*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *dgm01(1,3)*dgm10(1,3)+36*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+72*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +72*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+864*gm(1,1)&
&  **2*gm(1,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+12*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+432*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+432*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+144*gm(1,1)**3*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(4,4)=18*gm(1,2)**2*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,3)&
&  **2*(18*gm(2,2)*dgm01(1,1)*dgm10(1,1)+24*gm(1,1)*dgm01(1,2)*dgm10(1,2)&
&  +12*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))+gm(1,1)&
&  *(18*gm(2,3)**2*dgm01(1,1)*dgm10(1,1)+30*gm(2,2)*gm(3,3)*dgm01(1,1)&
&  *dgm10(1,1)+12*gm(1,2)*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+gm(1,2)*(48*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+24*gm(1,2)*dgm01(1,3)*dgm10(1,3)))+8*gm(1,1)**3*dgm01(2,3)&
&  *dgm10(2,3)+gm(1,1)**2*(48*gm(3,3)*dgm01(1,2)*dgm10(1,2)+48*gm(2,2)&
&  *dgm01(1,3)*dgm10(1,3)+24*gm(1,2)*dgm01(2,3)*dgm10(1,3)+24*gm(1,2)&
&  *dgm01(1,3)*dgm10(2,3)+gm(2,3)*(-12*dgm01(2,3)*dgm10(1,1)+48*dgm01(1,3)&
&  *dgm10(1,2)+48*dgm01(1,2)*dgm10(1,3)-12*dgm01(1,1)*dgm10(2,3)))&
&  +gm(1,3)*(12*gm(1,2)**2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,2)*(-12*gm(2,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(36*dgm01(2,3)&
&  *dgm10(1,1)+24*dgm01(1,3)*dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)&
&  +36*dgm01(1,1)*dgm10(2,3)))+gm(1,1)*(12*gm(2,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+48*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+24*gm(1,1)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))))
  cm(5,4)=(gm(1,2)*(864*gm(1,3)**2*dgm01(1,1)*dgm10(1,1)+3456*gm(1,1)&
&  *gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3456*gm(1,1)&
&  *(gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*dgm01(1,3)*dgm10(1,3)))&
&  +gm(1,1)*(864*gm(1,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,3)*(864*gm(2,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(864*dgm01(2,3)&
&  *dgm10(1,1)+1728*dgm01(1,3)*dgm10(1,2)+1728*dgm01(1,2)*dgm10(1,3)&
&  +864*dgm01(1,1)*dgm10(2,3)))+gm(1,1)*(1728*gm(3,3)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+864*gm(2,3)*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+576*gm(1,1)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3)))))/144.d0
  cm(6,4)=(144*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+3456*gm(1,1)&
&  **2*gm(1,3)*dgm01(1,2)*dgm10(1,2)+48*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +72*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+72*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+1728*gm(1,1)&
&  **2*gm(1,2)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+864*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+576*gm(1,1)&
&  **3*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3)))/144.d0
  cm(7,4)=(24*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(1,1)*dgm10(1,1)+144*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+12*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+72*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +72*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+36*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+36*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+36*gm(1,1)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+24*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+432*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(8,4)=12*gm(1,3)**3*dgm01(1,2)*dgm10(1,2)+gm(1,3)**2*(21*gm(1,2)&
&  *dgm01(2,3)*dgm10(1,1)+18*gm(1,2)*dgm01(1,3)*dgm10(1,2)+30*gm(1,1)&
&  *dgm01(2,3)*dgm10(1,2)+9*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+18*gm(1,2)*dgm01(1,2)*dgm10(1,3)+33*gm(2,2)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+21*gm(1,2)*dgm01(1,1)*dgm10(2,3)&
&  +30*gm(1,1)*dgm01(1,2)*dgm10(2,3))+gm(1,2)**2*(12*gm(3,3)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+6*gm(1,1)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))+gm(1,3)*(6*gm(2,3)**2*dgm01(1,1)*dgm10(1,1)&
&  +30*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)+3*gm(1,2)**2*dgm01(3,3)&
&  *dgm10(1,1)+30*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+60*gm(1,1)&
&  *gm(3,3)*dgm01(1,2)*dgm10(1,2)+6*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,2)&
&  +24*gm(1,2)**2*dgm01(1,3)*dgm10(1,3)+48*gm(1,1)*gm(1,2)*dgm01(2,3)&
&  *dgm10(1,3)+48*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(2,3)+24*gm(1,1)&
&  **2*dgm01(2,3)*dgm10(2,3)+gm(2,3)*(6*gm(1,2)*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+gm(1,1)*(0*dgm01(2,3)*dgm10(1,1)+60*dgm01(1,3)&
&  *dgm10(1,2)+60*dgm01(1,2)*dgm10(1,3)+0*dgm01(1,1)*dgm10(2,3)))&
&  +3*gm(1,2)**2*dgm01(1,1)*dgm10(3,3)+6*gm(1,1)*gm(1,2)*dgm01(1,2)&
&  *dgm10(3,3)+gm(2,2)*(24*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&  *(12*dgm01(3,3)*dgm10(1,1)+96*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&  *dgm10(3,3))))+gm(1,1)*(12*gm(2,3)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+18*gm(1,1)*gm(3,3)*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+gm(2,3)*(15*gm(3,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+12*gm(1,1)*(dgm01(3,3)*dgm10(1,2)+dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3)+dgm01(1,2)*dgm10(3,3)))+gm(2,2)&
&  *(27*gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*gm(1,1)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))))+gm(1,2)*(gm(2,3)&
&  *(6*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)*dgm10(1,1)&
&  +24*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)))+gm(1,1)*(gm(3,3)&
&  *(21*dgm01(2,3)*dgm10(1,1)+30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)&
&  *dgm10(1,3)+21*dgm01(1,1)*dgm10(2,3))+6*gm(1,1)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))))
  cm(9,4)=12*gm(1,2)**3*dgm01(1,3)*dgm10(1,3)+gm(1,3)**2*(12*gm(2,2)&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,2)*(3*dgm01(2,2)&
&  *dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))&
&  +6*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+gm(1,2)&
&  **2*(33*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +9*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+30*gm(1,1)&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3)))+gm(1,2)*(6*gm(2,3)&
&  **2*dgm01(1,1)*dgm10(1,1)+gm(2,2)*(24*gm(3,3)*dgm01(1,1)*dgm10(1,1)&
&  +60*gm(1,1)*dgm01(1,3)*dgm10(1,3))+gm(1,1)*gm(2,3)*(0*dgm01(2,3)&
&  *dgm10(1,1)+60*dgm01(1,3)*dgm10(1,2)+60*dgm01(1,2)*dgm10(1,3)&
&  +0*dgm01(1,1)*dgm10(2,3))+gm(1,1)*(gm(3,3)*(12*dgm01(2,2)*dgm10(1,1)&
&  +96*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2))+24*gm(1,1)&
&  *dgm01(2,3)*dgm10(2,3)))+gm(1,1)*(12*gm(2,3)**2*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+12*gm(1,1)*gm(3,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+gm(2,3)*(15*gm(2,2)*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+12*gm(1,1)*(dgm01(2,3)*dgm10(1,2)+dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)*dgm10(2,3)))+gm(2,2)&
&  *(27*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+18*gm(1,1)&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))))+gm(1,3)*(gm(1,2)&
&  **2*(21*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)&
&  *dgm10(1,3)+21*dgm01(1,1)*dgm10(2,3))+gm(2,2)*(6*gm(2,3)*dgm01(1,1)&
&  *dgm10(1,1)+30*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(21*dgm01(2,3)*dgm10(1,1)+30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)&
&  *dgm10(1,3)+21*dgm01(1,1)*dgm10(2,3)))+gm(1,2)*(6*gm(2,3)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*(48*dgm01(2,3)*dgm10(1,2)&
&  +6*dgm01(2,2)*dgm10(1,3)+6*dgm01(1,3)*dgm10(2,2)+48*dgm01(1,2)&
&  *dgm10(2,3)))+gm(1,1)*(gm(2,3)*(3*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)&
&  *dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))+6*gm(1,1)*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))))
  cm(10,4)=(24*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(1,1)*dgm10(1,1)+12*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+144*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+72*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+72*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+4*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +24*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+36*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+36*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+36*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +432*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /72.d0
  cm(11,4)=(12*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&  +gm(1,1)*gm(2,3)))*dgm01(1,2)*dgm10(1,2)+6*gm(2,2)*(-18*gm(1,3)&
&  *gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+2*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+3*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/18.d0
  cm(12,4)=(12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(1,2)+6*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)&
&  *dgm10(1,3)+6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+6*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+3*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +3*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+8*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,3)*dgm10(2,3)&
&  +12*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+12*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+12*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+3*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,1)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /6.d0
  cm(13,4)=(18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,2)*dgm10(1,2)+9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)*dgm10(1,3)&
&  +3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+9*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)&
&  *gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *dgm01(2,3)*dgm10(2,3)+9*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/9.d0
  cm(14,4)=(6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+18*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +9*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3*(90*gm(1,3)&
&  **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,3)**2*gm(2,3)&
&  +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*dgm01(2,3)*dgm10(2,3)+9*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +9*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(9*gm(1,3)**2*gm(2,2)&
&  -6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)&
&  **2+15*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +9*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/9.d0
  cm(15,4)=(12*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&  *gm(2,3))*gm(3,3))*dgm01(1,3)*dgm10(1,3)+6*gm(3,3)*(54*gm(1,3)&
&  *gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+2*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +9*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(16,4)=(3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+3*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+3*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3)))/18.d0
  cm(17,4)=(9*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)&
&  +18*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+18*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+3*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(18,4)=(9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+9*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))&
&  *dgm01(2,3)*dgm10(2,3)+6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&  *gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+6*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))&
&  /18.d0
  cm(19,4)=(3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+3*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+18*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+18*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+18*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/18.d0
  cm(20,4)=(4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&  *dgm10(2,3)+6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+9*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/18.d0
  cm(21,4)=(3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+3*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(1,5)=gm(1,1)**2*(6*gm(1,3)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))
  cm(2,5)=(144*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+144*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+48*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3456*gm(1,1)&
&  **2*gm(1,2)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+864*gm(1,1)&
&  **2*gm(1,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+576*gm(1,1)&
&  **3*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))/288.d0
  cm(3,5)=3*gm(1,3)**3*dgm01(1,1)*dgm10(1,1)+15*gm(1,1)*gm(1,3)&
&  **2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)*gm(1,3)&
&  *(15*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)*dgm10(1,1)&
&  +24*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)))+gm(1,1)**2*(9*gm(3,3)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*gm(1,1)*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(4,5)=gm(1,2)*(6*gm(1,3)**2*dgm01(1,1)*dgm10(1,1)+24*gm(1,1)&
&  *gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*gm(1,1)&
&  *(gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*dgm01(1,3)*dgm10(1,3)))&
&  +gm(1,1)*(6*gm(1,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,3)*(6*gm(2,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(6*dgm01(2,3)&
&  *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)&
&  +6*dgm01(1,1)*dgm10(2,3)))+gm(1,1)*(12*gm(3,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+6*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+4*gm(1,1)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))))
  cm(5,5)=gm(1,1)*(6*gm(1,3)**2*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)&
&  *gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)&
&  *(12*gm(3,3)*dgm01(1,1)*dgm10(1,1)+8*gm(1,1)*dgm01(1,3)*dgm10(1,3)))
  cm(6,5)=gm(1,1)*(6*gm(1,2)*(gm(1,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+gm(1,1)*(12*gm(2,3)&
&  *dgm01(1,1)*dgm10(1,1)+6*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+4*gm(1,1)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))))
  cm(7,5)=(48*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&  +gm(1,1)*gm(2,3)))*dgm01(1,1)*dgm10(1,1)+144*(6*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+16*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +96*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+72*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)&
&  *gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+1728*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))/288.d0
  cm(8,5)=(48*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,1)*dgm10(1,1)+48*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+128*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *dgm01(1,3)*dgm10(1,3)+16*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+96*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +32*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+48*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+1152*gm(1,1)&
&  **2*gm(1,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+576*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))/96.d0
  cm(9,5)=(48*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,1)*dgm10(1,1)+96*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+64*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)&
&  *dgm10(1,3)+16*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+48*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+64*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*gm(1,1)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +576*gm(1,1)**2*gm(1,3)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +48*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+1152*gm(1,1)**2*gm(1,2)*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3)))/96.d0
  cm(10,5)=8*gm(1,3)**3*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,3)**2*(9*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(3,3)&
&  *dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)))&
&  +gm(1,1)*gm(3,3)*(-3*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(6*dgm01(3,3)&
&  *dgm10(1,1)-12*dgm01(1,3)*dgm10(1,3)+6*dgm01(1,1)*dgm10(3,3)))&
&  +gm(1,1)*gm(1,3)*(12*gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+6*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(11,5)=(6*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))/36.d0
  cm(12,5)=(6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)+6*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+4*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+12*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+8*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))/12.d0
  cm(13,5)=(9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3)*dgm10(1,3)+3*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3)))/18.d0
  cm(14,5)=(3*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +9*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))/18.d0
  cm(15,5)=(32*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+24*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+36*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +24*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))/144.d0
  cm(16,5)=(240*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+160*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))/2880.d0
  cm(17,5)=(9*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(90*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))&
&  /36.d0
  cm(18,5)=(9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3)))/36.d0
  cm(19,5)=(3*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +18*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3)))/36.d0
  cm(20,5)=(6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))/36.d0
  cm(21,5)=(240*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+160*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))&
&  /2880.d0
  cm(1,6)=gm(1,1)**2*(6*gm(1,2)*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)))
  cm(2,6)=3*gm(1,2)**3*dgm01(1,1)*dgm10(1,1)+15*gm(1,1)*gm(1,2)&
&  **2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*gm(1,2)&
&  *(15*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(2,2)*dgm10(1,1)&
&  +24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)))+gm(1,1)**2*(9*gm(2,2)&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+2*gm(1,1)*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(3,6)=(144*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+48*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +144*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3456*gm(1,1)**2*gm(1,3)*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+864*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+576*gm(1,1)**3*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/288.d0
  cm(4,6)=6*gm(1,2)**2*(gm(1,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+gm(1,1)*gm(1,2)&
&  *(6*gm(2,3)*dgm01(1,1)*dgm10(1,1)+24*gm(1,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+gm(1,1)*(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)&
&  *dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)))&
&  +gm(1,1)*(24*gm(1,3)*(gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*dgm01(1,2)&
&  *dgm10(1,2))+gm(1,1)*(6*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+12*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +4*gm(1,1)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))))
  cm(5,6)=gm(1,1)*(6*gm(1,2)*(gm(1,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)))+gm(1,1)*(12*gm(2,3)&
&  *dgm01(1,1)*dgm10(1,1)+6*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+4*gm(1,1)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))))
  cm(6,6)=gm(1,1)*(6*gm(1,2)**2*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)&
&  *gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)&
&  *(12*gm(2,2)*dgm01(1,1)*dgm10(1,1)+8*gm(1,1)*dgm01(1,2)*dgm10(1,2)))
  cm(7,6)=8*gm(1,2)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,2)**2*(9*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(3*dgm01(2,2)&
&  *dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)))&
&  +gm(1,1)*gm(2,2)*(-3*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(6*dgm01(2,2)&
&  *dgm10(1,1)-12*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)*dgm10(2,2)))&
&  +gm(1,1)*gm(1,2)*(12*gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+6*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(8,6)=(48*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,1)*dgm10(1,1)+64*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *dgm01(1,2)*dgm10(1,2)+48*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+16*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +96*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +64*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+48*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +1152*gm(1,1)**2*gm(1,3)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +24*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+576*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3)))/96.d0
  cm(9,6)=(288*gm(1,2)**3*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,2)**2*(288*gm(2,3)*dgm01(1,1)*dgm10(1,1)+2016*gm(1,3)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*(576*dgm01(2,3)*dgm10(1,1)&
&  +1728*dgm01(1,3)*dgm10(1,2)+1728*dgm01(1,2)*dgm10(1,3)+576*dgm01(1,1)&
&  *dgm10(2,3)))+gm(1,1)*(gm(1,1)*(gm(2,3)*(576*dgm01(2,2)*dgm10(1,1)&
&  -2304*dgm01(1,2)*dgm10(1,2)+576*dgm01(1,1)*dgm10(2,2))+576*gm(1,3)&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+gm(2,2)*(-864*gm(2,3)&
&  *dgm01(1,1)*dgm10(1,1)+2016*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+gm(1,1)*(1152*dgm01(2,3)*dgm10(1,1)-576*dgm01(1,3)&
&  *dgm10(1,2)-576*dgm01(1,2)*dgm10(1,3)+1152*dgm01(1,1)*dgm10(2,3))))&
&  +gm(1,2)*(gm(1,3)*(2304*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&  *(288*dgm01(2,2)*dgm10(1,1)+6912*dgm01(1,2)*dgm10(1,2)+288*dgm01(1,1)&
&  *dgm10(2,2)))+gm(1,1)*(1440*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+1152*gm(1,1)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3)))))&
&  /96.d0
  cm(10,6)=(48*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&  *gm(2,3))*gm(3,3))*dgm01(1,1)*dgm10(1,1)+16*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +144*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+96*gm(1,1)&
&  *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+72*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+1728*gm(1,1)**2*gm(1,3)&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/288.d0
  cm(11,6)=(32*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)&
&  *dgm10(1,2)+24*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+36*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +24*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2)))/144.d0
  cm(12,6)=(8*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+6*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+4*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+3*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3)))/12.d0
  cm(13,6)=(12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+9*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+3*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3)))/18.d0
  cm(14,6)=(4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+6*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3)))/18.d0
  cm(15,6)=(6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(16,6)=(240*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+160*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))&
&  /2880.d0
  cm(17,6)=(9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+18*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+3*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(18,6)=(9*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3)))&
&  /36.d0
  cm(19,6)=(3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+18*(30*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(20,6)=(6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(21,6)=(240*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+160*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/2880.d0
  cm(1,7)=(80*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,1)&
&  *dgm10(1,1)+240*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+4320*gm(1,1)**2*gm(1,2)*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2)))/1440.d0
  cm(2,7)=gm(1,2)**3*(4*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&  +4*dgm01(1,1)*dgm10(2,2))+gm(1,2)*(6*gm(2,2)**2*dgm01(1,1)*dgm10(1,1)&
&  +6*gm(1,1)**2*dgm01(2,2)*dgm10(2,2)+gm(1,1)*gm(2,2)*(6*dgm01(2,2)&
&  *dgm10(1,1)+60*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)*dgm10(2,2)))&
&  +15*gm(1,2)**2*(gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+9*gm(1,1)&
&  *gm(2,2)*(gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(3,7)=(24*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+72*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+24*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+144*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+36*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+72*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+24*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+432*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/144.d0
  cm(4,7)=(24*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(1,1)*dgm10(1,1)+144*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+12*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+72*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +72*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+36*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+36*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+36*gm(1,1)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+24*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+432*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(5,7)=(48*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+144*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+16*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+96*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+72*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +1728*gm(1,1)**2*gm(1,2)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))&
&  /288.d0
  cm(6,7)=(2304*gm(1,2)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +gm(1,2)**2*(2592*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)*(864*dgm01(2,2)&
&  *dgm10(1,1)+10368*dgm01(1,2)*dgm10(1,2)+864*dgm01(1,1)*dgm10(2,2)))&
&  +gm(1,1)*gm(2,2)*(-864*gm(2,2)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&  *(1728*dgm01(2,2)*dgm10(1,1)-3456*dgm01(1,2)*dgm10(1,2)+1728*dgm01(1,1)&
&  *dgm10(2,2)))+gm(1,1)*gm(1,2)*(3456*gm(2,2)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+1728*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))))/288.d0
  cm(7,7)=2*gm(2,2)**3*dgm01(1,1)*dgm10(1,1)+gm(2,2)**2*(12*gm(1,2)&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*(-3*dgm01(2,2)&
&  *dgm10(1,1)+48*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)))&
&  +6*gm(1,2)**2*(gm(1,1)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+gm(2,2)*(12*gm(1,1)**2*dgm01(2,2)&
&  *dgm10(2,2)+gm(1,2)**2*(9*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)&
&  *dgm10(1,2)+9*dgm01(1,1)*dgm10(2,2))+30*gm(1,1)*gm(1,2)*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(8,7)=(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,2)*dgm10(1,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+6*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +6*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/12.d0
  cm(9,7)=(288*gm(2,2)**2*gm(2,3)*dgm01(1,1)*dgm10(1,1)+192*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(1,2)*dgm10(1,2)+24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+16*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+288*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+48*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*dgm01(2,2)*dgm10(2,2)&
&  +12*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +4*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +48*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +24*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +24*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+24*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/48.d0
  cm(10,7)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+36*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+18*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +3*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +18*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +9*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(11,7)=2*gm(2,2)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +3*gm(1,2)**3*dgm01(2,2)*dgm10(2,2)+gm(2,2)**2*(gm(1,2)*(3*dgm01(2,2)&
&  *dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))&
&  +9*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+15*gm(1,2)&
&  *gm(2,2)*(gm(1,1)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2)))
  cm(12,7)=(96*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+8*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +288*gm(2,2)**2*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +48*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)&
&  +4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+24*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+16*gm(2,2)*(-18*gm(1,3)&
&  *gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  +24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+48*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+144*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+24*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+12*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /48.d0
  cm(13,7)=(144*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *dgm01(1,2)*dgm10(1,2)+432*gm(2,2)**2*gm(2,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+144*gm(2,2)**3*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+864*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+72*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+12*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+36*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+72*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+12*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+432*gm(1,2)&
&  *gm(2,2)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+72*gm(2,2)&
&  *(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+36*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(14,7)=(12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+18*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(6*gm(1,2)**2*gm(1,3)&
&  +24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/18.d0
  cm(15,7)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(16,7)=gm(2,2)*(9*gm(1,2)**2*dgm01(2,2)*dgm10(2,2)+gm(2,2)&
&  **2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+gm(2,2)*(-3*gm(1,1)&
&  *dgm01(2,2)*dgm10(2,2)+6*gm(1,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))))
  cm(17,7)=(72*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(2,2)*dgm10(2,2)+12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+72*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+864*gm(2,2)**2*gm(2,3)*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+144*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +72*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +144*gm(2,2)**3*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +864*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +12*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/144.d0
  cm(18,7)=(144*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+864*gm(2,2)&
&  **2*gm(2,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+144*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+576*gm(2,2)**3*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+3456*gm(1,2)*gm(2,2)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+48*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/288.d0
  cm(19,7)=(24*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&  *gm(2,3))*gm(3,3))*dgm01(2,2)*dgm10(2,2)+4*(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +24*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*gm(2,2)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+144*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +72*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+432*gm(2,2)**2*gm(2,3)*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+72*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +36*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /144.d0
  cm(20,7)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&  *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(21,7)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/36.d0
  cm(1,8)=(80*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+80*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +160*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2880*gm(1,1)**2*gm(1,3)*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+1440*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/480.d0
  cm(2,8)=(24*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*dgm01(1,1)*dgm10(1,1)+96*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)&
&  +24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +48*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +96*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +4*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+8*gm(1,1)&
&  *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+16*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*gm(1,1)*(12*gm(1,2)&
&  *gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+288*gm(1,1)**2*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+12*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+144*gm(1,1)&
&  **2*gm(1,2)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/48.d0
  cm(3,8)=(24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(1,1)*dgm10(1,1)+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+192*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+24*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+48*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+48*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +48*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+288*gm(1,1)**2*gm(1,2)*dgm01(3,3)*dgm10(3,3)&
&  +12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+8*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +16*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+24*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +288*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /48.d0
  cm(4,8)=(24*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*dgm01(1,1)*dgm10(1,1)+48*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*dgm01(1,2)*dgm10(1,2)+12*(6*gm(1,3)**2*gm(2,3)&
&  +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+96*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)+12*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+48*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)&
&  *gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+48*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+576*gm(1,1)**2*gm(1,3)*dgm01(2,3)*dgm10(2,3)&
&  +4*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+24*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+8*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+24*gm(1,1)&
&  *(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+16*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(1,1)*(12*gm(1,2)&
&  *gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+12*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+12*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+12*gm(1,1)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +144*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /24.d0
  cm(5,8)=(48*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,1)*dgm10(1,1)+48*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+128*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *dgm01(1,3)*dgm10(1,3)+16*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+96*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +32*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+48*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+1152*gm(1,1)&
&  **2*gm(1,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+576*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))/96.d0
  cm(6,8)=(48*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,1)*dgm10(1,1)+64*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *dgm01(1,2)*dgm10(1,2)+48*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+16*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +96*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +64*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+48*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +1152*gm(1,1)**2*gm(1,3)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +24*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+576*gm(1,1)**2*gm(1,2)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3)))/96.d0
  cm(7,8)=(2*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,2)*dgm10(1,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+6*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +6*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/12.d0
  cm(8,8)=(gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+4*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+2*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+16*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)+4*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+8*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+4*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))&
&  *dgm01(2,3)*dgm10(2,3)+2*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+4*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+8*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))&
&  *dgm01(3,3)*dgm10(3,3)+(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&  *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+2*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/2.d0
  cm(9,8)=(2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+16*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(1,2)+2*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+4*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+16*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)&
&  +2*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +16*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+2*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+8*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+2*(30*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+8*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+8*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+4*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+2*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+2*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,1)*(12*gm(1,2)&
&  *gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+2*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/4.d0
  cm(10,8)=(288*gm(2,3)*gm(3,3)**2*dgm01(1,1)*dgm10(1,1)+288*gm(1,3)&
&  *gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+192*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *dgm01(1,3)*dgm10(1,3)+16*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&  *gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+48*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+8*gm(3,3)&
&  *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+48*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(3,3)+4*(90*gm(1,3)**2*gm(2,3)&
&  -36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+24*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+24*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+48*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+24*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(11,8)=(8*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+2*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+6*(6*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/12.d0
  cm(12,8)=(8*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(1,2)*dgm10(1,2)+2*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+16*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+8*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*gm(3,3)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +16*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+4*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+4*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+8*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+4*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+16*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+2*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+2*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*dgm01(3,3)*dgm10(3,3)+(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+2*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&  *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+2*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+2*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&  +2*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /4.d0
  cm(13,8)=(12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+3*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,3)&
&  *dgm10(1,3)+gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+12*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +12*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)&
&  +3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+2*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+12*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+6*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+3*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /6.d0
  cm(14,8)=(576*gm(1,3)*gm(3,3)**2*dgm01(1,2)*dgm10(1,2)+144*gm(2,3)&
&  *gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+96*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+12*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+16*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+48*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*dgm01(2,3)*dgm10(2,3)+12*gm(3,3)*(12*gm(1,3)&
&  *gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+24*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *dgm01(3,3)*dgm10(3,3)+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+4*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+24*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+48*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&  +24*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/24.d0
  cm(15,8)=(128*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(1,3)*dgm10(1,3)+576*gm(2,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+1152*gm(1,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+32*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+48*gm(3,3)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +16*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +96*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +48*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/96.d0
  cm(16,8)=(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&  *gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+2*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /12.d0
  cm(17,8)=(3*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*gm(3,3)*(12*gm(1,3)*gm(2,3)&
&  +24*gm(1,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)&
&  +6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+24*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/12.d0
  cm(18,8)=(3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+8*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&  +gm(1,1)*gm(2,3)))*dgm01(2,3)*dgm10(2,3)+2*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+8*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+2*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /12.d0
  cm(19,8)=(144*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+288*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+16*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+96*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *dgm01(2,3)*dgm10(2,3)+24*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*gm(3,3)*(12*gm(1,3)&
&  *gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+96*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+24*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3)&
&  +12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+24*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+48*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +4*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+24*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /48.d0
  cm(20,8)=(64*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&  *dgm10(2,3)+576*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+1152*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+64*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +24*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+48*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+96*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+16*(90*gm(1,3)**2*gm(2,3)&
&  -36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+48*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/96.d0
  cm(21,8)=(80*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&  *gm(2,3))*gm(3,3))*dgm01(3,3)*dgm10(3,3)+1440*gm(2,3)*gm(3,3)&
&  **2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2880*gm(1,3)&
&  *gm(3,3)**2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+160*gm(3,3)&
&  *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+80*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/480.d0
  cm(1,9)=(80*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&  +2*gm(1,2)*gm(2,3)))*dgm01(1,1)*dgm10(1,1)+160*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+80*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+1440*gm(1,1)**2*gm(1,3)*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2880*gm(1,1)**2*gm(1,2)*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3)))/480.d0
  cm(2,9)=(24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *dgm01(1,1)*dgm10(1,1)+192*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+24*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+48*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*gm(2,2)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +48*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+288*gm(1,1)**2*gm(1,3)*dgm01(2,2)*dgm10(2,2)&
&  +4*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +16*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +8*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+24*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*gm(1,1)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +288*gm(1,1)**2*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))&
&  /48.d0
  cm(3,9)=(24*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*dgm01(1,1)*dgm10(1,1)+48*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+96*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+96*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+12*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +24*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+24*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+48*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+4*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+16*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+8*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+144*gm(1,1)**2*gm(1,3)*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+288*gm(1,1)**2*gm(1,2)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(4,9)=12*gm(1,2)**3*dgm01(1,3)*dgm10(1,3)+gm(1,3)**2*(12*gm(2,2)&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,2)*(3*dgm01(2,2)&
&  *dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))&
&  +6*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+gm(1,2)&
&  **2*(33*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +9*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+30*gm(1,1)&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3)))+gm(1,2)*(6*gm(2,3)&
&  **2*dgm01(1,1)*dgm10(1,1)+gm(2,2)*(24*gm(3,3)*dgm01(1,1)*dgm10(1,1)&
&  +60*gm(1,1)*dgm01(1,3)*dgm10(1,3))+gm(1,1)*gm(2,3)*(0*dgm01(2,3)&
&  *dgm10(1,1)+60*dgm01(1,3)*dgm10(1,2)+60*dgm01(1,2)*dgm10(1,3)&
&  +0*dgm01(1,1)*dgm10(2,3))+gm(1,1)*(gm(3,3)*(12*dgm01(2,2)*dgm10(1,1)&
&  +96*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2))+24*gm(1,1)&
&  *dgm01(2,3)*dgm10(2,3)))+gm(1,1)*(12*gm(2,3)**2*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+12*gm(1,1)*gm(3,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+gm(2,3)*(15*gm(2,2)*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+12*gm(1,1)*(dgm01(2,3)*dgm10(1,2)+dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)*dgm10(2,3)))+gm(2,2)&
&  *(27*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+18*gm(1,1)&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))))+gm(1,3)*(gm(1,2)&
&  **2*(21*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)&
&  *dgm10(1,3)+21*dgm01(1,1)*dgm10(2,3))+gm(2,2)*(6*gm(2,3)*dgm01(1,1)&
&  *dgm10(1,1)+30*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(21*dgm01(2,3)*dgm10(1,1)+30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)&
&  *dgm10(1,3)+21*dgm01(1,1)*dgm10(2,3)))+gm(1,2)*(6*gm(2,3)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,1)*(48*dgm01(2,3)*dgm10(1,2)&
&  +6*dgm01(2,2)*dgm10(1,3)+6*dgm01(1,3)*dgm10(2,2)+48*dgm01(1,2)&
&  *dgm10(2,3)))+gm(1,1)*(gm(2,3)*(3*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)&
&  *dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))+6*gm(1,1)*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))))
  cm(5,9)=(48*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,1)*dgm10(1,1)+96*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+64*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)&
&  *dgm10(1,3)+16*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+48*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+64*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*gm(1,1)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +576*gm(1,1)**2*gm(1,3)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +48*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+1152*gm(1,1)**2*gm(1,2)*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3)))/96.d0
  cm(6,9)=(48*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,1)*dgm10(1,1)+128*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(1,2)*dgm10(1,2)&
&  +16*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +96*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +48*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+32*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+576*gm(1,1)**2*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+48*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+1152*gm(1,1)&
&  **2*gm(1,2)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3)))/96.d0
  cm(7,9)=(288*gm(2,2)**2*gm(2,3)*dgm01(1,1)*dgm10(1,1)+192*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(1,2)*dgm10(1,2)+24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+16*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+288*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+48*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*gm(1,1)&
&  *(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*dgm01(2,2)*dgm10(2,2)&
&  +12*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +4*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +48*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +24*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +24*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+24*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/48.d0
  cm(8,9)=(2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+16*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(1,2)+2*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+4*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+16*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)&
&  +2*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +16*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+2*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+8*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+2*(30*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+8*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+8*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+4*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+2*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+2*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,1)*(12*gm(1,2)&
&  *gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+2*gm(1,1)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/4.d0
  cm(9,9)=gm(2,2)**2*(12*gm(3,3)*dgm01(1,1)*dgm10(1,1)+48*gm(1,1)&
&  *dgm01(1,3)*dgm10(1,3)+24*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3)))+12*gm(1,2)**3*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+gm(1,2)**2*(24*gm(1,3)*dgm01(2,3)*dgm10(1,2)+6*gm(1,3)&
&  *dgm01(2,2)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)*dgm10(2,2)+gm(3,3)&
&  *(15*dgm01(2,2)*dgm10(1,1)+72*dgm01(1,2)*dgm10(1,2)+15*dgm01(1,1)&
&  *dgm10(2,2))+24*gm(1,3)*dgm01(1,2)*dgm10(2,3)+24*gm(1,1)*dgm01(2,3)&
&  *dgm10(2,3)+gm(2,3)*(6*dgm01(2,3)*dgm10(1,1)+24*dgm01(1,3)*dgm10(1,2)&
&  +24*dgm01(1,2)*dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)))+gm(1,2)*(12*gm(2,3)&
&  **2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*gm(1,3)&
&  **2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+48*gm(1,1)&
&  *gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+12*gm(1,1)&
&  *gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,3)&
&  *(gm(1,3)*(18*dgm01(2,2)*dgm10(1,1)-48*dgm01(1,2)*dgm10(1,2)&
&  +18*dgm01(1,1)*dgm10(2,2))+gm(1,1)*(24*dgm01(2,3)*dgm10(1,2)&
&  +48*dgm01(2,2)*dgm10(1,3)+48*dgm01(1,3)*dgm10(2,2)+24*dgm01(1,2)&
&  *dgm10(2,3))))+gm(2,2)*(6*gm(2,3)**2*dgm01(1,1)*dgm10(1,1)-6*gm(1,3)&
&  **2*dgm01(2,2)*dgm10(1,1)-3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)&
&  +72*gm(1,3)**2*dgm01(1,2)*dgm10(1,2)+120*gm(1,1)*gm(3,3)*dgm01(1,2)&
&  *dgm10(1,2)+96*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(1,2)+24*gm(1,2)&
&  **2*dgm01(1,3)*dgm10(1,3)-18*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(1,3)&
&  -6*gm(1,3)**2*dgm01(1,1)*dgm10(2,2)-3*gm(1,1)*gm(3,3)*dgm01(1,1)&
&  *dgm10(2,2)-18*gm(1,1)*gm(1,3)*dgm01(1,3)*dgm10(2,2)+96*gm(1,1)&
&  *gm(1,3)*dgm01(1,2)*dgm10(2,3)+48*gm(1,1)**2*dgm01(2,3)*dgm10(2,3)&
&  +gm(2,3)*(12*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +12*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)&
&  *(-18*dgm01(2,3)*dgm10(1,1)+96*dgm01(1,3)*dgm10(1,2)+96*dgm01(1,2)&
&  *dgm10(1,3)-18*dgm01(1,1)*dgm10(2,3)))+gm(1,2)*(48*gm(3,3)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+gm(1,3)*(48*dgm01(2,3)*dgm10(1,1)&
&  +24*dgm01(1,3)*dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)+48*dgm01(1,1)&
&  *dgm10(2,3))+60*gm(1,1)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))))&
&  +gm(1,1)*((6*gm(1,3)**2+12*gm(1,1)*gm(3,3))*dgm01(2,2)*dgm10(2,2)&
&  +gm(2,3)**2*(-6*dgm01(2,2)*dgm10(1,1)+72*dgm01(1,2)*dgm10(1,2)&
&  -6*dgm01(1,1)*dgm10(2,2))+gm(2,3)*(12*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+24*gm(1,1)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))))
  cm(10,9)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+4*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+3*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /12.d0
  cm(11,9)=(128*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(1,2)*dgm10(1,2)+576*gm(2,2)**2*gm(2,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+1152*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+48*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+24*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+16*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+48*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+32*gm(2,2)&
&  *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+48*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/96.d0
  cm(12,9)=(16*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*dgm01(1,2)*dgm10(1,2)+2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)&
&  *gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+8*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)&
&  +2*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+4*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+8*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +2*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)*dgm10(2,2)&
&  +(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(6*gm(1,3)**2*gm(2,3)&
&  +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+2*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+16*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)&
&  +4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+16*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)&
&  *gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+4*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+8*(6*gm(1,2)*gm(1,3)&
&  *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+2*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(6*gm(1,2)*gm(1,3)&
&  *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+2*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+2*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /4.d0
  cm(13,9)=gm(1,1)*gm(2,3)*(6*gm(1,3)*dgm01(2,2)*dgm10(2,2)+12*gm(2,3)&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+12*gm(1,2)**3*dgm01(2,3)&
&  *dgm10(2,3)+gm(2,2)**2*(12*gm(1,3)*dgm01(2,3)*dgm10(1,1)+12*gm(1,3)&
&  *dgm01(1,3)*dgm10(1,2)+12*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+12*gm(1,3)*dgm01(1,2)*dgm10(1,3)+24*gm(1,2)*dgm01(1,3)&
&  *dgm10(1,3)+18*gm(1,1)*dgm01(2,3)*dgm10(1,3)+6*gm(2,3)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*gm(1,3)*dgm01(1,1)*dgm10(2,3)&
&  +18*gm(1,1)*dgm01(1,3)*dgm10(2,3))+gm(1,2)**2*(33*gm(3,3)*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+gm(2,3)*(18*dgm01(2,3)*dgm10(1,2)&
&  +21*dgm01(2,2)*dgm10(1,3)+21*dgm01(1,3)*dgm10(2,2)+18*dgm01(1,2)&
&  *dgm10(2,3))+9*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))&
&  +gm(2,2)*(12*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)+27*gm(1,1)*gm(3,3)&
&  *dgm01(2,2)*dgm10(1,2)+6*gm(2,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+12*gm(1,3)**2*dgm01(1,2)*dgm10(2,2)+27*gm(1,1)*gm(3,3)&
&  *dgm01(1,2)*dgm10(2,2)+15*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,2)&
&  +15*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(2,3)+30*gm(1,2)**2*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+gm(2,3)*(gm(1,3)*(3*dgm01(2,2)&
&  *dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))&
&  +gm(1,2)*(6*dgm01(2,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,2)+48*dgm01(1,2)&
&  *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3))+gm(1,1)*(30*dgm01(2,3)*dgm10(1,2)&
&  +21*dgm01(2,2)*dgm10(1,3)+21*dgm01(1,3)*dgm10(2,2)+30*dgm01(1,2)&
&  *dgm10(2,3)))+gm(1,2)*(gm(3,3)*(12*dgm01(2,2)*dgm10(1,1)+96*dgm01(1,2)&
&  *dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2))+60*gm(1,1)*dgm01(2,3)*dgm10(2,3)&
&  +gm(1,3)*(60*dgm01(2,3)*dgm10(1,2)+0*dgm01(2,2)*dgm10(1,3)+0*dgm01(1,3)&
&  *dgm10(2,2)+60*dgm01(1,2)*dgm10(2,3))))+gm(1,2)*((6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*dgm01(2,2)*dgm10(2,2)+gm(2,3)**2*(3*dgm01(2,2)*dgm10(1,1)&
&  +24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))+gm(2,3)*(6*gm(1,3)&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+30*gm(1,1)*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))))
  cm(14,9)=(8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(1,2)*dgm10(1,2)+gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+3*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+6*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(6*gm(1,3)*gm(2,2)&
&  *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)&
&  *gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+6*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/6.d0
  cm(15,9)=(8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)+2*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(90*gm(1,3)&
&  **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /12.d0
  cm(16,9)=(80*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)&
&  +gm(1,1)*gm(2,3)))*dgm01(2,2)*dgm10(2,2)+1440*gm(2,2)**2*gm(2,3)&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+160*gm(2,2)*(-18*gm(1,3)&
&  *gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+2880*gm(1,2)*gm(2,2)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+80*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/480.d0
  cm(17,9)=(24*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(2,2)+12*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+48*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +24*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+96*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(2,3)*dgm10(2,3)+24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+96*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+48*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+24*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +144*gm(2,2)**2*gm(2,3)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +16*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+288*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+4*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(18,9)=(48*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(2,2)*dgm10(2,2)+24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+96*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+48*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +64*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,3)*dgm10(2,3)&
&  +576*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +64*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+1152*gm(1,2)*gm(2,2)**2*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+16*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/96.d0
  cm(19,9)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+4*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)&
&  +6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+24*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +3*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/12.d0
  cm(20,9)=(8*(90*gm(1,3)**2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)&
&  *gm(2,3))*gm(3,3))*dgm01(2,3)*dgm10(2,3)+2*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+4*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+6*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&  *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/12.d0
  cm(21,9)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(3,3)&
&  *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/12.d0
  cm(1,10)=(80*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+240*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4320*gm(1,1)**2*gm(1,3)*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/1440.d0
  cm(2,10)=(24*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,1)*dgm10(1,1)+24*(90*gm(1,3)&
&  **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+72*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+144*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+24*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+36*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+72*gm(1,1)*(12*gm(1,2)&
&  *gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+432*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/144.d0
  cm(3,10)=gm(1,3)**3*(4*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&  +4*dgm01(1,1)*dgm10(3,3))+gm(1,3)*(6*gm(3,3)**2*dgm01(1,1)*dgm10(1,1)&
&  +6*gm(1,1)**2*dgm01(3,3)*dgm10(3,3)+gm(1,1)*gm(3,3)*(6*dgm01(3,3)&
&  *dgm10(1,1)+60*dgm01(1,3)*dgm10(1,3)+6*dgm01(1,1)*dgm10(3,3)))&
&  +15*gm(1,3)**2*(gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+9*gm(1,1)&
&  *gm(3,3)*(gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(4,10)=(24*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(1,1)*dgm10(1,1)+12*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+144*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+72*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+72*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+4*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +24*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+36*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+36*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+36*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +432*gm(1,1)**2*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /72.d0
  cm(5,10)=(2304*gm(1,3)**3*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+gm(1,3)**2*(2592*gm(3,3)*dgm01(1,1)*dgm10(1,1)+gm(1,1)&
&  *(864*dgm01(3,3)*dgm10(1,1)+10368*dgm01(1,3)*dgm10(1,3)+864*dgm01(1,1)&
&  *dgm10(3,3)))+gm(1,1)*gm(3,3)*(-864*gm(3,3)*dgm01(1,1)*dgm10(1,1)&
&  +gm(1,1)*(1728*dgm01(3,3)*dgm10(1,1)-3456*dgm01(1,3)*dgm10(1,3)&
&  +1728*dgm01(1,1)*dgm10(3,3)))+gm(1,1)*gm(1,3)*(3456*gm(3,3)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+1728*gm(1,1)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))))/288.d0
  cm(6,10)=(48*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,1)*dgm10(1,1)+16*(90*gm(1,3)&
&  **3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+144*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +96*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+72*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+1728*gm(1,1)&
&  **2*gm(1,3)*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/288.d0
  cm(7,10)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+36*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+18*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +3*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +18*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +9*gm(1,1)*(12*gm(1,2)*gm(1,3)+24*gm(1,1)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(8,10)=(288*gm(2,3)*gm(3,3)**2*dgm01(1,1)*dgm10(1,1)+288*gm(1,3)&
&  *gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+192*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *dgm01(1,3)*dgm10(1,3)+16*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&  *gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+48*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+8*gm(3,3)&
&  *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+48*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(3,3)+4*(90*gm(1,3)**2*gm(2,3)&
&  -36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+24*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+24*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+48*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+24*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(9,10)=(2*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,1)&
&  *dgm10(1,1)+4*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+3*gm(1,1)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*gm(1,1)*(12*gm(1,2)*gm(1,3)&
&  +24*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /12.d0
  cm(10,10)=2*gm(3,3)**3*dgm01(1,1)*dgm10(1,1)+gm(3,3)**2*(12*gm(1,3)&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)*(-3*dgm01(3,3)&
&  *dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)))&
&  +6*gm(1,3)**2*(gm(1,1)*dgm01(3,3)*dgm10(3,3)+gm(1,3)*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+gm(3,3)*(12*gm(1,1)**2*dgm01(3,3)&
&  *dgm10(3,3)+gm(1,3)**2*(9*dgm01(3,3)*dgm10(1,1)+24*dgm01(1,3)&
&  *dgm10(1,3)+9*dgm01(1,1)*dgm10(3,3))+30*gm(1,1)*gm(1,3)*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(11,10)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(6*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(12,10)=(288*gm(2,3)*gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+96*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)+8*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +48*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+144*gm(1,3)*gm(3,3)**2*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+16*gm(3,3)&
&  *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+96*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+24*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3)+4*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+48*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /48.d0
  cm(13,10)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+18*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+9*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(14,10)=(144*gm(3,3)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(1,3)*dgm10(1,3)+432*gm(2,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+864*gm(1,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+432*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+72*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+72*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+12*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +36*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +72*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +36*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(15,10)=2*gm(3,3)**3*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +3*gm(1,3)**3*dgm01(3,3)*dgm10(3,3)+gm(3,3)**2*(gm(1,3)*(3*dgm01(3,3)&
&  *dgm10(1,1)+24*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3))&
&  +9*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+15*gm(1,3)&
&  *gm(3,3)*(gm(1,1)*dgm01(3,3)*dgm10(3,3)+gm(1,3)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))
  cm(16,10)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(17,10)=(432*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+72*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +144*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*(90*gm(1,2)&
&  **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*dgm01(3,3)&
&  *dgm10(3,3)+4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+36*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+72*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/144.d0
  cm(18,10)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+6*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(19,10)=(144*gm(3,3)**3*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+864*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+864*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+72*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +12*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+72*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+72*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/144.d0
  cm(20,10)=(576*gm(3,3)**3*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+3456*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+144*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(3,3)+864*gm(2,3)&
&  *gm(3,3)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+144*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+48*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/288.d0
  cm(21,10)=gm(3,3)*(9*gm(1,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  **2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+gm(3,3)*(-3*gm(1,1)&
&  *dgm01(3,3)*dgm10(3,3)+6*gm(1,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))))
  cm(1,11)=(160*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+240*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)))/2880.d0
  cm(2,11)=(2304*gm(1,2)**3*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+gm(1,2)**2*(2592*gm(1,1)*dgm01(2,2)*dgm10(2,2)+gm(2,2)&
&  *(864*dgm01(2,2)*dgm10(1,1)+10368*dgm01(1,2)*dgm10(1,2)+864*dgm01(1,1)&
&  *dgm10(2,2)))+gm(1,1)*gm(2,2)*(-864*gm(1,1)*dgm01(2,2)*dgm10(2,2)&
&  +gm(2,2)*(1728*dgm01(2,2)*dgm10(1,1)-3456*dgm01(1,2)*dgm10(1,2)&
&  +1728*dgm01(1,1)*dgm10(2,2)))+gm(1,2)*gm(2,2)*(1728*gm(2,2)*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+3456*gm(1,1)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))))/288.d0
  cm(3,11)=(6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +12*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /36.d0
  cm(4,11)=(12*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+6*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,2)*gm(1,3)&
&  *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+2*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+3*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/18.d0
  cm(5,11)=(6*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))/36.d0
  cm(6,11)=(32*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,2)&
&  *dgm10(1,2)+24*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+36*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +24*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2)))/144.d0
  cm(7,11)=2*gm(2,2)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +3*gm(1,2)**3*dgm01(2,2)*dgm10(2,2)+gm(2,2)**2*(gm(1,2)*(3*dgm01(2,2)&
&  *dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))&
&  +9*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+15*gm(1,2)&
&  *gm(2,2)*(gm(1,1)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2)))
  cm(8,11)=(8*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+2*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+6*(6*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/12.d0
  cm(9,11)=(128*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(1,2)*dgm10(1,2)+576*gm(2,2)**2*gm(2,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+1152*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+48*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+24*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+16*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+48*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+32*gm(2,2)&
&  *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+48*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/96.d0
  cm(10,11)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(6*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(11,11)=(gm(2,2)*(4608*gm(2,2)**2*dgm01(1,2)*dgm10(1,2)+288*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*dgm01(2,2)*dgm10(2,2)+6912*gm(1,2)*gm(2,2)&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))))/576.d0
  cm(12,11)=(64*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+1152*gm(2,2)**2*gm(2,3)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+48*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(2,2)*dgm10(2,2)+48*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+16*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +48*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+64*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +576*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +24*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/96.d0
  cm(13,11)=(3456*gm(2,2)**2*gm(2,3)*dgm01(1,2)*dgm10(1,2)+576*gm(2,2)&
&  **3*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+144*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(2,2)*dgm10(2,2)+72*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+48*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+864*gm(1,2)*gm(2,2)**2*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+1728*gm(1,2)*gm(2,2)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+72*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/144.d0
  cm(14,11)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+6*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +9*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /18.d0
  cm(15,11)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(16,11)=gm(2,2)**2*(6*gm(1,2)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(17,11)=(144*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+48*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +3456*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +144*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+576*gm(2,2)**3*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+864*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/288.d0
  cm(18,11)=(gm(2,2)*(288*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *dgm01(2,2)*dgm10(2,2)+3456*gm(2,2)*gm(2,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+2304*gm(2,2)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+3456*gm(1,2)*gm(2,2)*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))))/576.d0
  cm(19,11)=(48*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+16*(90*gm(2,3)&
&  **3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+96*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+144*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+1728*gm(2,2)**2*gm(2,3)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+72*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/288.d0
  cm(20,11)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(21,11)=(160*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+240*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/2880.d0
  cm(1,12)=(2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+4*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&  *gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+gm(1,1)&
&  *(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3)))/12.d0
  cm(2,12)=(24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,2)*dgm10(1,2)+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+2*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*dgm01(2,2)*dgm10(2,2)+3*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +24*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +4*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/12.d0
  cm(3,12)=(6*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +6*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*gm(3,3)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+24*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+6*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+4*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /12.d0
  cm(4,12)=(12*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(1,2)+6*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,3)&
&  *dgm10(1,3)+6*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+6*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+3*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +3*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+8*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,3)*dgm10(2,3)&
&  +12*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+2*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+12*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+2*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+12*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+3*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,1)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /6.d0
  cm(5,12)=(6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+8*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&  +2*gm(1,2)*gm(2,3)))*dgm01(1,3)*dgm10(1,3)+6*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +4*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+12*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)&
&  *gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2*gm(1,1)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))&
&  /12.d0
  cm(6,12)=(8*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+6*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+4*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +2*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+8*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+3*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +2*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3)))/12.d0
  cm(7,12)=(96*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+8*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +288*gm(2,2)**2*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +48*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)&
&  +4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+24*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+16*gm(2,2)*(-18*gm(1,3)&
&  *gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  +24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+48*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+144*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+24*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+12*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /48.d0
  cm(8,12)=(8*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(1,2)*dgm10(1,2)+2*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+16*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+4*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+8*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*gm(3,3)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +16*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+4*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+4*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+8*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+4*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+16*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+2*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+2*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*dgm01(3,3)*dgm10(3,3)+(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+2*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&  *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+2*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+2*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+4*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +4*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&  +2*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /4.d0
  cm(9,12)=(16*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*dgm01(1,2)*dgm10(1,2)+2*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)&
&  *gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+8*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*dgm01(1,3)*dgm10(1,3)&
&  +2*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+4*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+8*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +2*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)*dgm10(2,2)&
&  +(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(6*gm(1,3)**2*gm(2,3)&
&  +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+2*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +2*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+16*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)&
&  +4*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+16*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)&
&  *gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+4*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+8*(6*gm(1,2)*gm(1,3)&
&  *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+2*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+4*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(6*gm(1,2)*gm(1,3)&
&  *gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+2*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+2*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /4.d0
  cm(10,12)=(288*gm(2,3)*gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+96*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)+8*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +48*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+144*gm(1,3)*gm(3,3)**2*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+16*gm(3,3)&
&  *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+96*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+24*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3)+4*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+24*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+48*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /48.d0
  cm(11,12)=(64*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+1152*gm(2,2)**2*gm(2,3)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+48*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(2,2)*dgm10(2,2)+48*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+16*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +48*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+64*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +576*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +24*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/96.d0
  cm(12,12)=(4*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+4*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+4*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*dgm01(2,2)*dgm10(2,2)+2*gm(3,3)*(12*gm(1,3)*gm(2,3)&
&  +24*gm(1,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +2*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+16*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+8*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+8*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+4*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(3,3)&
&  *dgm10(3,3)+2*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +2*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+4*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/2.d0
  cm(13,12)=(48*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+576*gm(2,2)**2*gm(2,3)*dgm01(1,3)*dgm10(1,3)+8*gm(2,2)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(2,2)&
&  +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+24*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+24*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+48*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+16*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+48*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+12*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+144*gm(1,2)*gm(2,2)&
&  **2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+12*gm(2,2)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /24.d0
  cm(14,12)=(288*gm(2,3)**3*dgm01(1,3)*dgm10(1,3)+gm(2,3)**2*(216*gm(1,2)&
&  *dgm01(3,3)*dgm10(1,3)+720*gm(3,3)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+792*gm(1,1)*dgm01(3,3)*dgm10(2,3)+216*gm(1,2)*dgm01(1,3)&
&  *dgm10(3,3)+792*gm(1,1)*dgm01(2,3)*dgm10(3,3)+gm(1,3)*(504*dgm01(3,3)&
&  *dgm10(1,2)+432*dgm01(2,3)*dgm10(1,3)+432*dgm01(1,3)*dgm10(2,3)&
&  +504*dgm01(1,2)*dgm10(3,3)))+gm(2,3)*(576*gm(3,3)**2*dgm01(1,2)&
&  *dgm10(1,2)+(144*gm(1,2)**2+576*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +gm(1,3)**2*(72*dgm01(3,3)*dgm10(2,2)+576*dgm01(2,3)*dgm10(2,3)&
&  +72*dgm01(2,2)*dgm10(3,3))+gm(3,3)*(1440*gm(2,2)*dgm01(1,3)*dgm10(1,3)&
&  +288*gm(1,1)*dgm01(3,3)*dgm10(2,2)+2304*gm(1,1)*dgm01(2,3)*dgm10(2,3)&
&  +gm(1,3)*(1152*dgm01(2,3)*dgm10(1,2)+144*dgm01(2,2)*dgm10(1,3)&
&  +144*dgm01(1,3)*dgm10(2,2)+1152*dgm01(1,2)*dgm10(2,3))+288*gm(1,1)&
&  *dgm01(2,2)*dgm10(3,3)+gm(1,2)*(0*dgm01(3,3)*dgm10(1,2)+1440*dgm01(2,3)&
&  *dgm10(1,3)+1440*dgm01(1,3)*dgm10(2,3)+0*dgm01(1,2)*dgm10(3,3)))&
&  +gm(1,3)*(720*gm(2,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +144*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))&
&  +gm(3,3)*(144*gm(1,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +gm(1,3)*(144*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +gm(1,2)*(72*dgm01(3,3)*dgm10(2,2)+576*dgm01(2,3)*dgm10(2,3)&
&  +72*dgm01(2,2)*dgm10(3,3)))+288*(gm(1,2)*gm(3,3)*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)&
&  *dgm10(2,3))+gm(1,1)*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(1,2)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))&
&  +gm(2,2)*(432*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +gm(3,3)*(gm(1,3)*(504*dgm01(3,3)*dgm10(1,2)+720*dgm01(2,3)*dgm10(1,3)&
&  +720*dgm01(1,3)*dgm10(2,3)+504*dgm01(1,2)*dgm10(3,3))+360*gm(1,2)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+648*gm(1,1)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+gm(1,3)*(144*gm(1,2)*dgm01(3,3)&
&  *dgm10(3,3)+288*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))))&
&  /24.d0
  cm(15,12)=(64*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+1152*gm(2,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+576*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+64*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +48*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+48*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+16*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+96*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/96.d0
  cm(16,12)=(80*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+80*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +2880*gm(2,2)**2*gm(2,3)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +160*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+1440*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/480.d0
  cm(17,12)=(24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(2,2)*dgm10(2,2)+24*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+192*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+48*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+48*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+48*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+288*gm(1,2)*gm(2,2)**2*dgm01(3,3)*dgm10(3,3)+8*gm(2,2)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+288*gm(2,2)**2*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+24*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+16*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(18,12)=(48*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+48*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+48*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+128*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+32*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +1152*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +16*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+96*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+24*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +576*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /96.d0
  cm(19,12)=(288*gm(1,3)*gm(3,3)**2*dgm01(2,2)*dgm10(2,2)+288*gm(2,3)&
&  *gm(3,3)**2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+8*gm(3,3)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+192*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+48*gm(3,3)*(12*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +48*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+16*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+24*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*dgm01(3,3)*dgm10(3,3)+24*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+24*gm(2,2)&
&  *(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +4*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+24*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+48*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(20,12)=(128*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(2,3)*dgm10(2,3)+1152*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+32*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+576*gm(1,3)&
&  *gm(3,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+48*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+48*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+48*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+16*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+96*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/96.d0
  cm(21,12)=(80*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)+2880*gm(2,3)*gm(3,3)&
&  **2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+80*gm(3,3)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+1440*gm(1,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+160*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/480.d0
  cm(1,13)=(3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +3*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3)))/18.d0
  cm(2,13)=(144*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(1,2)*dgm10(1,2)+36*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+432*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+24*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*dgm01(2,2)*dgm10(2,2)&
&  +36*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +72*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+36*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+72*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+12*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(3,13)=(9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(1,3)*dgm10(1,3)+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,3)**2*gm(2,3)&
&  +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(6*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +3*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(4,13)=(18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(1,2)*dgm10(1,2)+9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(1,3)*dgm10(1,3)&
&  +3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+9*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)&
&  *gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *dgm01(2,3)*dgm10(2,3)+9*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  +48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/9.d0
  cm(5,13)=(9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(1,3)*dgm10(1,3)+3*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3)))/18.d0
  cm(6,13)=(12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&  +2*gm(1,2)*gm(2,3)))*dgm01(1,2)*dgm10(1,2)+9*(48*gm(1,2)*gm(1,3)&
&  *gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+3*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+9*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)&
&  +6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+6*gm(1,1)*(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +6*gm(1,1)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3)))/18.d0
  cm(7,13)=(144*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *dgm01(1,2)*dgm10(1,2)+432*gm(2,2)**2*gm(2,3)*(dgm01(1,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,2))+144*gm(2,2)**3*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+864*gm(1,2)*gm(2,2)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+72*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)&
&  *gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+12*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,2))+36*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+72*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+12*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+432*gm(1,2)&
&  *gm(2,2)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+72*gm(2,2)&
&  *(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+36*(6*gm(1,2)**3+30*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/72.d0
  cm(8,13)=(12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+3*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*dgm01(1,3)&
&  *dgm10(1,3)+gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+12*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +12*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)&
&  +3*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+2*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+12*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+6*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+3*(6*gm(1,2)**3+30*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /6.d0
  cm(9,13)=(96*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*dgm01(1,2)*dgm10(1,2)+12*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+576*gm(1,2)&
&  *gm(2,2)**2*dgm01(1,3)*dgm10(1,3)+144*gm(2,2)**2*gm(2,3)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+24*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +16*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*(6*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)&
&  +12*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+12*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+48*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)&
&  *gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+48*(6*gm(1,2)&
&  **3+30*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,3)*dgm10(2,3)+12*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+24*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+48*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(2,2)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +24*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +12*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)+48*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))&
&  /24.d0
  cm(10,13)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+18*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+9*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+9*(6*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *gm(1,3)*gm(2,2)+48*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(11,13)=(3456*gm(2,2)**2*gm(2,3)*dgm01(1,2)*dgm10(1,2)+576*gm(2,2)&
&  **3*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+144*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(2,2)*dgm10(2,2)+72*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+48*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+864*gm(1,2)*gm(2,2)**2*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+1728*gm(1,2)*gm(2,2)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+72*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/144.d0
  cm(12,13)=(48*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+576*gm(2,2)**2*gm(2,3)*dgm01(1,3)*dgm10(1,3)+8*gm(2,2)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+24*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(2,2)&
&  +12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+24*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+96*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  +24*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+24*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+48*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+16*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+12*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+48*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+12*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+144*gm(1,2)*gm(2,2)&
&  **2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+12*gm(2,2)*(12*gm(1,2)&
&  **2+24*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /24.d0
  cm(13,13)=(72*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+288*gm(2,2)**3*dgm01(1,3)*dgm10(1,3)+864*gm(2,2)&
&  **2*gm(2,3)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+72*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+72*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +72*gm(2,2)*(12*gm(1,2)**2+24*gm(1,1)*gm(2,2))*dgm01(2,3)*dgm10(2,3)&
&  +72*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+864*gm(1,2)*gm(2,2)**2*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+72*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/36.d0
  cm(14,13)=(6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+9*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+9*(6*gm(1,3)*gm(2,2)&
&  *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +9*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+9*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/9.d0
  cm(15,13)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+6*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+9*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(16,13)=(gm(2,2)*(240*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(2,2)*dgm10(2,2)+4320*gm(2,2)*gm(2,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+1440*gm(2,2)**2*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+4320*gm(1,2)*gm(2,2)*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))))/720.d0
  cm(17,13)=(72*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+36*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+144*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+72*gm(2,2)*(12*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +864*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +36*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+72*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+432*gm(2,2)**2*gm(2,3)*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+144*gm(2,2)**3*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +432*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /72.d0
  cm(18,13)=(864*gm(1,2)*gm(2,3)**2*dgm01(2,2)*dgm10(2,2)+576*gm(2,2)&
&  **3*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+gm(2,2)**2*(864*gm(1,3)&
&  *dgm01(2,3)*dgm10(2,2)+1728*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+864*gm(1,3)*dgm01(2,2)*dgm10(2,3)+3456*gm(1,2)*dgm01(2,3)&
&  *dgm10(2,3)+gm(2,3)*(1728*dgm01(2,3)*dgm10(1,2)+864*dgm01(2,2)&
&  *dgm10(1,3)+864*dgm01(1,3)*dgm10(2,2)+1728*dgm01(1,2)*dgm10(2,3)))&
&  +gm(2,2)*(3456*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+864*gm(2,3)&
&  **2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+gm(2,3)*(864*gm(1,3)&
&  *dgm01(2,2)*dgm10(2,2)+3456*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3)))))/144.d0
  cm(19,13)=(24*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(2,2)*dgm10(2,2)+12*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(90*gm(2,3)&
&  **3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+144*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+72*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+24*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+72*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+36*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+432*gm(2,2)**2*gm(2,3)*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+36*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+36*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(20,13)=(12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+6*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+9*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+9*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/18.d0
  cm(21,13)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(1,14)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+3*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3)))/18.d0
  cm(2,14)=(12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,2)*dgm10(1,2)+3*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+9*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+18*(30*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+9*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/18.d0
  cm(3,14)=(432*gm(1,3)*gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+144*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,3)*dgm10(1,3)+36*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+24*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+36*gm(3,3)&
&  *(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+72*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*gm(1,1)*(54*gm(1,2)*gm(1,3)&
&  -18*gm(1,1)*gm(2,3))*dgm01(3,3)*dgm10(3,3)+36*(6*gm(1,3)**2*gm(2,3)&
&  +6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+72*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(4,14)=(6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+18*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +9*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3*(90*gm(1,3)&
&  **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,3)**2*gm(2,3)&
&  +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*dgm01(2,3)*dgm10(2,3)+9*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +9*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(9*gm(1,3)**2*gm(2,2)&
&  -6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)&
&  **2+15*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +9*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/9.d0
  cm(5,14)=(3*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)&
&  +9*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+9*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*gm(1,1)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*gm(1,1)&
&  *(54*gm(1,2)*gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))/18.d0
  cm(6,14)=(4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+6*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*gm(1,1)*(54*gm(1,2)&
&  *gm(1,3)-18*gm(1,1)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3)))/18.d0
  cm(7,14)=(12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,2)*dgm10(1,2)+(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&  +3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+18*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(6*gm(1,2)**2*gm(1,3)&
&  +24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/18.d0
  cm(8,14)=(576*gm(1,3)*gm(3,3)**2*dgm01(1,2)*dgm10(1,2)+144*gm(2,3)&
&  *gm(3,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+96*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+12*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+16*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+48*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*dgm01(2,3)*dgm10(2,3)+12*gm(3,3)*(12*gm(1,3)&
&  *gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+24*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*(6*gm(1,2)&
&  **2*gm(1,3)+24*gm(1,1)*gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))&
&  *dgm01(3,3)*dgm10(3,3)+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+4*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+24*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+12*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+48*(9*gm(1,3)&
&  **2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)&
&  *(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&  +24*(6*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/24.d0
  cm(9,14)=(8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(1,2)*dgm10(1,2)+gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))+12*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)&
&  *dgm10(1,3)+3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+2*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+3*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+6*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+3*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*(6*gm(1,3)*gm(2,2)&
&  *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)&
&  *gm(1,3)*gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+6*(6*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(6*gm(1,1)*gm(1,3)&
&  *gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*(6*gm(1,2)**2*gm(1,3)+24*gm(1,1)&
&  *gm(1,3)*gm(2,2)+6*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/6.d0
  cm(10,14)=(144*gm(3,3)**3*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,2))+144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(1,3)*dgm10(1,3)+432*gm(2,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+864*gm(1,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+432*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+72*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+72*(6*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2+24*gm(1,1)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+12*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +36*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +72*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +36*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(11,14)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+6*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +9*(6*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)+24*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /18.d0
  cm(12,14)=(288*gm(2,3)**3*dgm01(1,3)*dgm10(1,3)+gm(2,3)**2*(216*gm(1,2)&
&  *dgm01(3,3)*dgm10(1,3)+720*gm(3,3)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+792*gm(1,1)*dgm01(3,3)*dgm10(2,3)+216*gm(1,2)*dgm01(1,3)&
&  *dgm10(3,3)+792*gm(1,1)*dgm01(2,3)*dgm10(3,3)+gm(1,3)*(504*dgm01(3,3)&
&  *dgm10(1,2)+432*dgm01(2,3)*dgm10(1,3)+432*dgm01(1,3)*dgm10(2,3)&
&  +504*dgm01(1,2)*dgm10(3,3)))+gm(2,3)*(576*gm(3,3)**2*dgm01(1,2)&
&  *dgm10(1,2)+(144*gm(1,2)**2+576*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +gm(1,3)**2*(72*dgm01(3,3)*dgm10(2,2)+576*dgm01(2,3)*dgm10(2,3)&
&  +72*dgm01(2,2)*dgm10(3,3))+gm(3,3)*(1440*gm(2,2)*dgm01(1,3)*dgm10(1,3)&
&  +288*gm(1,1)*dgm01(3,3)*dgm10(2,2)+2304*gm(1,1)*dgm01(2,3)*dgm10(2,3)&
&  +gm(1,3)*(1152*dgm01(2,3)*dgm10(1,2)+144*dgm01(2,2)*dgm10(1,3)&
&  +144*dgm01(1,3)*dgm10(2,2)+1152*dgm01(1,2)*dgm10(2,3))+288*gm(1,1)&
&  *dgm01(2,2)*dgm10(3,3)+gm(1,2)*(0*dgm01(3,3)*dgm10(1,2)+1440*dgm01(2,3)&
&  *dgm10(1,3)+1440*dgm01(1,3)*dgm10(2,3)+0*dgm01(1,2)*dgm10(3,3)))&
&  +gm(1,3)*(720*gm(2,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +144*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))&
&  +gm(3,3)*(144*gm(1,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +gm(1,3)*(144*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +gm(1,2)*(72*dgm01(3,3)*dgm10(2,2)+576*dgm01(2,3)*dgm10(2,3)&
&  +72*dgm01(2,2)*dgm10(3,3)))+288*(gm(1,2)*gm(3,3)*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)&
&  *dgm10(2,3))+gm(1,1)*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(1,2)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))&
&  +gm(2,2)*(432*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +gm(3,3)*(gm(1,3)*(504*dgm01(3,3)*dgm10(1,2)+720*dgm01(2,3)*dgm10(1,3)&
&  +720*dgm01(1,3)*dgm10(2,3)+504*dgm01(1,2)*dgm10(3,3))+360*gm(1,2)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+648*gm(1,1)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+gm(1,3)*(144*gm(1,2)*dgm01(3,3)&
&  *dgm10(3,3)+288*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))))&
&  /24.d0
  cm(13,14)=(6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,2)&
&  *dgm10(1,2)+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+9*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+9*(6*gm(1,3)*gm(2,2)&
&  *gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +9*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)*gm(2,3)+9*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+9*(6*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)+24*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/9.d0
  cm(14,14)=(288*gm(3,3)**3*dgm01(1,2)*dgm10(1,2)+72*gm(3,3)*(12*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3))*dgm01(1,3)*dgm10(1,3)+864*gm(2,3)*gm(3,3)&
&  **2*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+72*gm(3,3)&
&  *(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*dgm01(2,3)*dgm10(2,3)+864*gm(1,3)&
&  *gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+72*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+72*(9*gm(1,3)**2*gm(2,2)-6*gm(1,2)*gm(1,3)&
&  *gm(2,3)+9*gm(1,2)**2*gm(3,3)+gm(1,1)*(9*gm(2,3)**2+15*gm(2,2)&
&  *gm(3,3)))*dgm01(3,3)*dgm10(3,3)+24*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +72*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+72*(6*gm(1,3)&
&  **2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(15,14)=(576*gm(3,3)**3*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+864*gm(1,3)**2*gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  **2*(3456*gm(2,3)*dgm01(1,3)*dgm10(1,3)+864*gm(1,2)*dgm01(3,3)&
&  *dgm10(1,3)+1728*gm(1,1)*dgm01(3,3)*dgm10(2,3)+864*gm(1,2)*dgm01(1,3)&
&  *dgm10(3,3)+1728*gm(1,1)*dgm01(2,3)*dgm10(3,3)+gm(1,3)*(864*dgm01(3,3)&
&  *dgm10(1,2)+1728*dgm01(2,3)*dgm10(1,3)+1728*dgm01(1,3)*dgm10(2,3)&
&  +864*dgm01(1,2)*dgm10(3,3)))+gm(3,3)*(3456*gm(1,1)*gm(2,3)*dgm01(3,3)&
&  *dgm10(3,3)+864*gm(1,3)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))+gm(1,3)*(864*gm(1,2)*dgm01(3,3)*dgm10(3,3)+3456*gm(2,3)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))))/144.d0
  cm(16,14)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*gm(2,2)*(-18*gm(1,3)&
&  *gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/18.d0
  cm(17,14)=(432*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+36*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+144*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)&
&  +24*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+72*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+36*gm(3,3)*(12*gm(1,3)&
&  *gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+24*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(3,3)*dgm10(3,3)+4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+12*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +36*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+12*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+72*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(18,14)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+2*(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(19,14)=(144*gm(3,3)**3*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+432*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(2,3)*dgm10(2,3)+864*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+72*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+432*gm(1,3)&
&  *gm(3,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+72*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+12*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+36*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +12*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+36*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+72*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/72.d0
  cm(20,14)=(576*gm(3,3)**3*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+864*gm(1,3)*gm(2,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  **2*(3456*gm(1,3)*dgm01(2,3)*dgm10(2,3)+864*gm(1,2)*dgm01(3,3)&
&  *dgm10(2,3)+864*gm(1,2)*dgm01(2,3)*dgm10(3,3)+gm(2,3)*(864*dgm01(3,3)&
&  *dgm10(1,2)+1728*dgm01(2,3)*dgm10(1,3)+1728*dgm01(1,3)*dgm10(2,3)&
&  +864*dgm01(1,2)*dgm10(3,3))+1728*gm(2,2)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))+gm(3,3)*(3456*gm(1,3)*gm(2,2)*dgm01(3,3)&
&  *dgm10(3,3)+864*gm(2,3)**2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+gm(2,3)*(864*gm(1,2)*dgm01(3,3)*dgm10(3,3)+3456*gm(1,3)&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))))/144.d0
  cm(21,14)=(gm(3,3)*(240*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(3,3)*dgm10(3,3)+1440*gm(3,3)**2*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+4320*gm(2,3)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+4320*gm(1,3)*gm(3,3)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))))/720.d0
  cm(1,15)=(160*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+240*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/2880.d0
  cm(2,15)=(6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))&
&  +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(3,15)=(2304*gm(1,3)**3*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+gm(1,3)**2*(2592*gm(1,1)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  *(864*dgm01(3,3)*dgm10(1,1)+10368*dgm01(1,3)*dgm10(1,3)+864*dgm01(1,1)&
&  *dgm10(3,3)))+gm(1,1)*gm(3,3)*(-864*gm(1,1)*dgm01(3,3)*dgm10(3,3)&
&  +gm(3,3)*(1728*dgm01(3,3)*dgm10(1,1)-3456*dgm01(1,3)*dgm10(1,3)&
&  +1728*dgm01(1,1)*dgm10(3,3)))+gm(1,3)*gm(3,3)*(1728*gm(3,3)*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+3456*gm(1,1)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))))/288.d0
  cm(4,15)=(12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(1,3)*dgm10(1,3)+6*gm(3,3)&
&  *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+2*(90*gm(1,3)&
&  **3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+9*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)&
&  +24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+9*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+3*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(5,15)=(32*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+24*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+36*(6*gm(1,3)**3+30*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +24*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))/144.d0
  cm(6,15)=(6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(1,3))+4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+6*gm(1,1)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(7,15)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(8,15)=(128*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(1,3)*dgm10(1,3)+576*gm(2,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(1,3))+1152*gm(1,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(1,3))+32*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(48*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+48*gm(3,3)*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +16*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +96*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)*gm(3,3)+24*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +48*(6*gm(1,3)**3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/96.d0
  cm(9,15)=(8*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(1,3)*dgm10(1,3)+2*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(90*gm(1,3)&
&  **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+6*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(6*gm(1,3)&
&  **3+30*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+6*(48*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(6*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /12.d0
  cm(10,15)=2*gm(3,3)**3*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&  +3*gm(1,3)**3*dgm01(3,3)*dgm10(3,3)+gm(3,3)**2*(gm(1,3)*(3*dgm01(3,3)&
&  *dgm10(1,1)+24*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3))&
&  +9*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+15*gm(1,3)&
&  *gm(3,3)*(gm(1,1)*dgm01(3,3)*dgm10(3,3)+gm(1,3)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))
  cm(11,15)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(12,15)=(64*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+1152*gm(2,3)*gm(3,3)**2*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+576*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+64*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +48*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+48*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+16*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+24*gm(3,3)*(12*gm(1,3)**2+24*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+96*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)&
&  *gm(1,3)*gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/96.d0
  cm(13,15)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(1,3)&
&  *dgm10(1,3)+6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(1,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(1,3))+6*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+9*(6*gm(1,3)**2*gm(2,3)+6*gm(1,2)*gm(1,3)&
&  *gm(3,3)+24*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(14,15)=(576*gm(3,3)**3*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(1,3))+864*gm(1,3)**2*gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  **2*(3456*gm(2,3)*dgm01(1,3)*dgm10(1,3)+864*gm(1,2)*dgm01(3,3)&
&  *dgm10(1,3)+1728*gm(1,1)*dgm01(3,3)*dgm10(2,3)+864*gm(1,2)*dgm01(1,3)&
&  *dgm10(3,3)+1728*gm(1,1)*dgm01(2,3)*dgm10(3,3)+gm(1,3)*(864*dgm01(3,3)&
&  *dgm10(1,2)+1728*dgm01(2,3)*dgm10(1,3)+1728*dgm01(1,3)*dgm10(2,3)&
&  +864*dgm01(1,2)*dgm10(3,3)))+gm(3,3)*(3456*gm(1,1)*gm(2,3)*dgm01(3,3)&
&  *dgm10(3,3)+864*gm(1,3)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))+gm(1,3)*(864*gm(1,2)*dgm01(3,3)*dgm10(3,3)+3456*gm(2,3)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))))/144.d0
  cm(15,15)=(gm(3,3)*(4608*gm(3,3)**2*dgm01(1,3)*dgm10(1,3)+288*(12*gm(1,3)&
&  **2+24*gm(1,1)*gm(3,3))*dgm01(3,3)*dgm10(3,3)+6912*gm(1,3)*gm(3,3)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))))/576.d0
  cm(16,15)=(160*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+240*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/2880.d0
  cm(17,15)=(1728*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+96*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +16*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+72*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+144*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/288.d0
  cm(18,15)=(6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(19,15)=(576*gm(3,3)**3*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+3456*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+144*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)+48*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +864*gm(1,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/288.d0
  cm(20,15)=(gm(3,3)*(2304*gm(3,3)**2*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+288*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+3456*gm(2,3)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+3456*gm(1,3)*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))))/576.d0
  cm(21,15)=gm(3,3)**2*(6*gm(1,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(1,16)=((90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2)))/36.d0
  cm(2,16)=(80*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(2,2)&
&  *dgm10(2,2)+4320*gm(1,2)*gm(2,2)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+240*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))/1440.d0
  cm(3,16)=(3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(4,16)=(3*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+3*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+3*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3)))/18.d0
  cm(5,16)=(240*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+160*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2)))/2880.d0
  cm(6,16)=(240*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+160*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))&
&  /2880.d0
  cm(7,16)=(gm(2,2)*(240*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,2)&
&  *dgm10(2,2)+1440*gm(2,2)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+8640*gm(1,2)*gm(2,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))))/1440.d0
  cm(8,16)=(gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&  *gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+2*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))&
&  /12.d0
  cm(9,16)=(80*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+1440*gm(2,2)&
&  **2*gm(2,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+160*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+2880*gm(1,2)*gm(2,2)**2*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+80*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/480.d0
  cm(10,16)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(11,16)=gm(2,2)**2*(6*gm(1,2)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))
  cm(12,16)=(80*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+80*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +2880*gm(2,2)**2*gm(2,3)*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +160*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+1440*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/480.d0
  cm(13,16)=(gm(2,2)*(240*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(2,2)*dgm10(2,2)+4320*gm(2,2)*gm(2,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+1440*gm(2,2)**2*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+4320*gm(1,2)*gm(2,2)*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))))/720.d0
  cm(14,16)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+3*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*gm(2,2)*(-18*gm(1,3)&
&  *gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/18.d0
  cm(15,16)=(160*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+240*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/2880.d0
  cm(16,16)=2*gm(2,2)**3*dgm01(2,2)*dgm10(2,2)
  cm(17,16)=(gm(2,2)*(240*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+8640*gm(2,2)*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+1440*gm(2,2)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))))/1440.d0
  cm(18,16)=gm(2,2)**2*(6*gm(2,3)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))
  cm(19,16)=(80*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+240*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+4320*gm(2,2)**2*gm(2,3)*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/1440.d0
  cm(20,16)=(160*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+240*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/2880.d0
  cm(21,16)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(1,17)=(3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3)))/36.d0
  cm(2,17)=(24*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+36*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+72*(30*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +72*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+144*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+24*(90*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+432*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/144.d0
  cm(3,17)=(9*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+18*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+36*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(4,17)=(9*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+12*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&  *(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*dgm01(2,3)*dgm10(2,3)+18*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+18*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +18*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3*gm(2,2)&
&  *(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /18.d0
  cm(5,17)=(9*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +18*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(90*gm(1,2)**2*gm(1,3)&
&  -18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+3*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))&
&  /36.d0
  cm(6,17)=(9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+18*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+3*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(7,17)=(72*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(2,2)*dgm10(2,2)+12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+72*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+864*gm(2,2)**2*gm(2,3)*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+144*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +72*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +144*gm(2,2)**3*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +864*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +12*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/144.d0
  cm(8,17)=(3*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*gm(3,3)*(12*gm(1,3)*gm(2,3)&
&  +24*gm(1,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)&
&  +6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+24*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(3,3)*dgm10(3,3)&
&  +gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+2*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+4*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+2*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/12.d0
  cm(9,17)=(24*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(2,2)+12*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+48*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +24*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+96*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(2,3)*dgm10(2,3)+24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+96*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+48*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+24*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +144*gm(2,2)**2*gm(2,3)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +16*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+288*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+4*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+8*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(10,17)=(432*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+72*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +144*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *dgm01(3,3)*dgm10(3,3)+4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+24*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+36*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+72*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/144.d0
  cm(11,17)=(144*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+48*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +3456*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +144*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+576*gm(2,2)**3*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+864*gm(1,2)*gm(2,2)**2*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/288.d0
  cm(12,17)=(24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(2,2)*dgm10(2,2)+24*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+24*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+192*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+48*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+48*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+24*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+48*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+288*gm(1,2)*gm(2,2)**2*dgm01(3,3)*dgm10(3,3)+8*gm(2,2)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+288*gm(2,2)**2*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+24*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+16*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(13,17)=(72*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+36*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+12*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+144*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+72*gm(2,2)*(12*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +864*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +36*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+72*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+432*gm(2,2)**2*gm(2,3)*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+144*gm(2,2)**3*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +432*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /72.d0
  cm(14,17)=(432*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+36*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+144*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)&
&  +24*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+72*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+36*gm(3,3)*(12*gm(1,3)&
&  *gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+24*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))&
&  *dgm01(3,3)*dgm10(3,3)+4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+12*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +36*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+12*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+72*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(15,17)=(1728*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+96*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(-36*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +16*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+72*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+144*(48*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/288.d0
  cm(16,17)=(gm(2,2)*(240*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+8640*gm(2,2)*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+1440*gm(2,2)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))))/1440.d0
  cm(17,17)=6*gm(2,3)**3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +gm(2,2)*(12*gm(3,3)**2*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)**2*dgm01(3,3)&
&  *dgm10(3,3)+gm(2,2)*gm(3,3)*(-3*dgm01(3,3)*dgm10(2,2)+48*dgm01(2,3)&
&  *dgm10(2,3)-3*dgm01(2,2)*dgm10(3,3)))+gm(2,3)**2*(6*gm(3,3)*dgm01(2,2)&
&  *dgm10(2,2)+gm(2,2)*(9*dgm01(3,3)*dgm10(2,2)+24*dgm01(2,3)*dgm10(2,3)&
&  +9*dgm01(2,2)*dgm10(3,3)))+gm(2,2)*gm(2,3)*(30*gm(3,3)*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+12*gm(2,2)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))
  cm(18,17)=3*gm(2,3)**3*dgm01(2,2)*dgm10(2,2)+15*gm(2,2)*gm(2,3)&
&  **2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*gm(2,3)&
&  *(15*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(3*dgm01(3,3)*dgm10(2,2)&
&  +24*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3)))+gm(2,2)**2*(9*gm(3,3)&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+2*gm(2,2)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(19,17)=gm(2,3)**3*(4*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)&
&  *dgm10(2,3)+4*dgm01(2,2)*dgm10(3,3))+gm(2,3)*(6*gm(3,3)**2*dgm01(2,2)&
&  *dgm10(2,2)+6*gm(2,2)**2*dgm01(3,3)*dgm10(3,3)+gm(2,2)*gm(3,3)&
&  *(6*dgm01(3,3)*dgm10(2,2)+60*dgm01(2,3)*dgm10(2,3)+6*dgm01(2,2)&
&  *dgm10(3,3)))+15*gm(2,3)**2*(gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  +9*gm(2,2)*gm(3,3)*(gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(20,17)=(2304*gm(2,3)**3*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))+gm(2,3)**2*(2592*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  *(864*dgm01(3,3)*dgm10(2,2)+10368*dgm01(2,3)*dgm10(2,3)+864*dgm01(2,2)&
&  *dgm10(3,3)))+gm(2,2)*gm(3,3)*(-864*gm(2,2)*dgm01(3,3)*dgm10(3,3)&
&  +gm(3,3)*(1728*dgm01(3,3)*dgm10(2,2)-3456*dgm01(2,3)*dgm10(2,3)&
&  +1728*dgm01(2,2)*dgm10(3,3)))+gm(2,3)*gm(3,3)*(1728*gm(3,3)*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3456*gm(2,2)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))))/288.d0
  cm(21,17)=(80*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+4320*gm(2,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+240*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/1440.d0
  cm(1,18)=(240*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+160*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3)))/2880.d0
  cm(2,18)=(48*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)&
&  +2*gm(1,2)*gm(2,3)))*dgm01(2,2)*dgm10(2,2)+72*gm(2,2)*(24*gm(1,3)&
&  *gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+144*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+1728*gm(1,2)*gm(2,2)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+96*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+16*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))&
&  /288.d0
  cm(3,18)=(9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+12*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(4,18)=(9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+9*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+4*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))&
&  *dgm01(2,3)*dgm10(2,3)+6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)&
&  *gm(2,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+6*gm(2,2)*(54*gm(1,2)&
&  **2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)&
&  *gm(1,2)*gm(2,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))&
&  /18.d0
  cm(5,18)=(9*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+6*(-36*gm(1,2)&
&  *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(90*gm(1,2)&
&  **3-54*gm(1,1)*gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3)))/36.d0
  cm(6,18)=(9*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)&
&  -36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+6*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(90*gm(1,2)**3-54*gm(1,1)&
&  *gm(1,2)*gm(2,2))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3)))&
&  /36.d0
  cm(7,18)=(144*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,2)*dgm10(2,2)+864*gm(2,2)&
&  **2*gm(2,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+144*gm(2,2)&
&  *(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+576*gm(2,2)**3*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+3456*gm(1,2)*gm(2,2)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+48*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)&
&  *gm(2,2))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/288.d0
  cm(8,18)=(3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+12*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+8*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+2*gm(2,2)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+4*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +8*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+6*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +3*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)&
&  *gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +2*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/12.d0
  cm(9,18)=(48*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  +30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *dgm01(2,2)*dgm10(2,2)+24*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+96*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+48*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +64*gm(2,2)*(54*gm(1,2)**2-18*gm(1,1)*gm(2,2))*dgm01(2,3)*dgm10(2,3)&
&  +576*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +64*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+1152*gm(1,2)*gm(2,2)**2*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(48*gm(1,2)*gm(1,3)*gm(2,2)&
&  +6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+16*(-36*gm(1,2)*gm(1,3)*gm(2,2)&
&  +90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))/96.d0
  cm(10,18)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+18*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,2))+2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+6*(-36*gm(1,2)*gm(1,3)*gm(2,2)+90*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(11,18)=(gm(2,2)*(288*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *dgm01(2,2)*dgm10(2,2)+3456*gm(2,2)*gm(2,3)*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+2304*gm(2,2)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+3456*gm(1,2)*gm(2,2)*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))))/576.d0
  cm(12,18)=(48*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+48*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+48*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+128*gm(2,2)*(-18*gm(1,3)*gm(2,2)&
&  +54*gm(1,2)*gm(2,3))*dgm01(2,3)*dgm10(2,3)+32*gm(2,2)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +1152*gm(2,2)**2*gm(2,3)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +16*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+96*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+24*gm(2,2)*(24*gm(1,3)*gm(2,2)&
&  +12*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +576*gm(1,2)*gm(2,2)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /96.d0
  cm(13,18)=6*gm(1,2)*gm(2,3)**2*dgm01(2,2)*dgm10(2,2)+4*gm(2,2)&
&  **3*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+gm(2,2)**2*(6*gm(1,3)&
&  *dgm01(2,3)*dgm10(2,2)+12*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+6*gm(1,3)*dgm01(2,2)*dgm10(2,3)+24*gm(1,2)*dgm01(2,3)&
&  *dgm10(2,3)+gm(2,3)*(12*dgm01(2,3)*dgm10(1,2)+6*dgm01(2,2)*dgm10(1,3)&
&  +6*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)))+gm(2,2)*(24*gm(1,2)&
&  *gm(3,3)*dgm01(2,2)*dgm10(2,2)+6*gm(2,3)**2*(dgm01(2,2)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,2))+gm(2,3)*(6*gm(1,3)*dgm01(2,2)*dgm10(2,2)&
&  +24*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))))
  cm(14,18)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+2*(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +6*gm(2,2)*(-18*gm(1,3)*gm(2,2)+54*gm(1,2)*gm(2,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(15,18)=(6*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+4*(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+6*(-36*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/36.d0
  cm(16,18)=gm(2,2)**2*(6*gm(2,3)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))
  cm(17,18)=3*gm(2,3)**3*dgm01(2,2)*dgm10(2,2)+15*gm(2,2)*gm(2,3)&
&  **2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*gm(2,3)&
&  *(15*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(3*dgm01(3,3)*dgm10(2,2)&
&  +24*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3)))+gm(2,2)**2*(9*gm(3,3)&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+2*gm(2,2)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(18,18)=gm(2,2)*(6*gm(2,3)**2*dgm01(2,2)*dgm10(2,2)+12*gm(2,2)&
&  *gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)&
&  *(12*gm(3,3)*dgm01(2,2)*dgm10(2,2)+8*gm(2,2)*dgm01(2,3)*dgm10(2,3)))
  cm(19,18)=8*gm(2,3)**3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +gm(2,3)**2*(9*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(3*dgm01(3,3)&
&  *dgm10(2,2)+36*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3)))&
&  +gm(2,2)*gm(3,3)*(-3*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(6*dgm01(3,3)&
&  *dgm10(2,2)-12*dgm01(2,3)*dgm10(2,3)+6*dgm01(2,2)*dgm10(3,3)))&
&  +gm(2,2)*gm(2,3)*(12*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+6*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(20,18)=(32*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&  *dgm10(2,3)+24*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+36*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +24*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/144.d0
  cm(21,18)=(240*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+160*(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /2880.d0
  cm(1,19)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3)))/36.d0
  cm(2,19)=(2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&  +6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +18*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+36*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+9*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(3,19)=(432*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+24*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+72*gm(3,3)*(12*gm(1,3)*gm(2,3)&
&  +24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +144*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +24*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*(gm(1,3)*gm(2,2)+2*gm(1,2)&
&  *gm(2,3)))*dgm01(3,3)*dgm10(3,3)+36*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+72*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+4*(90*gm(1,3)&
&  **3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+24*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /144.d0
  cm(4,19)=(3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+3*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*dgm01(2,3)*dgm10(2,3)+18*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+18*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+18*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +9*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/18.d0
  cm(5,19)=(3*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+2*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))&
&  +18*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+9*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)*gm(1,3)&
&  *gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3)))/36.d0
  cm(6,19)=(3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+18*(30*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +12*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(48*gm(1,2)&
&  *gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(90*gm(1,2)&
&  **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(7,19)=(24*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,2)*dgm10(2,2)+4*(90*gm(2,3)&
&  **3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+24*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +24*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+144*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+72*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+432*gm(2,2)**2*gm(2,3)&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+72*gm(2,2)*(24*gm(1,3)&
&  *gm(2,2)+12*gm(1,2)*gm(2,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+36*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)**2*gm(2,3)&
&  -18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/144.d0
  cm(8,19)=(144*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+288*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+16*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+96*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *dgm01(2,3)*dgm10(2,3)+24*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+48*gm(3,3)*(12*gm(1,3)&
&  *gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+96*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +8*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+24*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*dgm01(3,3)*dgm10(3,3)&
&  +12*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+24*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+48*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +4*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +24*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+24*(-12*gm(1,3)**2*gm(2,2)&
&  +36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)&
&  **2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /48.d0
  cm(9,19)=(2*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,2))+4*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&  +2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+24*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)&
&  +6*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(2,3))+24*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+12*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +6*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +3*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+6*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)*gm(2,3))&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+3*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+6*(48*gm(1,2)*gm(1,3)*gm(2,2)+6*gm(1,2)&
&  **2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/12.d0
  cm(10,19)=(144*gm(3,3)**3*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,2))+864*gm(1,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+864*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+72*(-12*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)+30*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +12*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+72*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+72*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/144.d0
  cm(11,19)=(48*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,2)*dgm10(2,2)+16*(90*gm(2,3)&
&  **3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+96*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+144*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+1728*gm(2,2)**2*gm(2,3)*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+72*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/288.d0
  cm(12,19)=(288*gm(1,3)*gm(3,3)**2*dgm01(2,2)*dgm10(2,2)+288*gm(2,3)&
&  *gm(3,3)**2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+8*gm(3,3)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+192*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+48*gm(3,3)*(12*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +48*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+16*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+24*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+24*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*dgm01(3,3)*dgm10(3,3)+24*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+24*gm(2,2)&
&  *(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+12*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +4*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+24*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+48*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/48.d0
  cm(13,19)=(24*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(2,2)*dgm10(2,2)+12*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4*(90*gm(2,3)&
&  **3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+144*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+72*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+24*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+72*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+36*gm(2,2)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+432*gm(2,2)**2*gm(2,3)*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+36*(6*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+36*gm(2,2)*(24*gm(1,3)*gm(2,2)+12*gm(1,2)&
&  *gm(2,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/72.d0
  cm(14,19)=(144*gm(3,3)**3*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,2))+432*gm(2,3)*gm(3,3)**2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))&
&  *dgm01(2,3)*dgm10(2,3)+864*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+72*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+432*gm(1,3)&
&  *gm(3,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+72*(6*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+12*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+36*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +12*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+36*(48*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+72*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/72.d0
  cm(15,19)=(576*gm(3,3)**3*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,2))+3456*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+144*(48*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)+48*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +864*gm(1,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +144*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/288.d0
  cm(16,19)=(80*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,2)&
&  *dgm10(2,2)+240*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+4320*gm(2,2)**2*gm(2,3)*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/1440.d0
  cm(17,19)=gm(2,3)**3*(4*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)&
&  *dgm10(2,3)+4*dgm01(2,2)*dgm10(3,3))+gm(2,3)*(6*gm(3,3)**2*dgm01(2,2)&
&  *dgm10(2,2)+6*gm(2,2)**2*dgm01(3,3)*dgm10(3,3)+gm(2,2)*gm(3,3)&
&  *(6*dgm01(3,3)*dgm10(2,2)+60*dgm01(2,3)*dgm10(2,3)+6*dgm01(2,2)&
&  *dgm10(3,3)))+15*gm(2,3)**2*(gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  +9*gm(2,2)*gm(3,3)*(gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(18,19)=(2304*gm(2,3)**3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+gm(2,3)**2*(2592*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)&
&  *(864*dgm01(3,3)*dgm10(2,2)+10368*dgm01(2,3)*dgm10(2,3)+864*dgm01(2,2)&
&  *dgm10(3,3)))+gm(2,2)*gm(3,3)*(-864*gm(3,3)*dgm01(2,2)*dgm10(2,2)&
&  +gm(2,2)*(1728*dgm01(3,3)*dgm10(2,2)-3456*dgm01(2,3)*dgm10(2,3)&
&  +1728*dgm01(2,2)*dgm10(3,3)))+gm(2,2)*gm(2,3)*(3456*gm(3,3)*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+1728*gm(2,2)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))))/288.d0
  cm(19,19)=2*gm(3,3)**3*dgm01(2,2)*dgm10(2,2)+gm(3,3)**2*(12*gm(2,3)&
&  *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*(-3*dgm01(3,3)&
&  *dgm10(2,2)+48*dgm01(2,3)*dgm10(2,3)-3*dgm01(2,2)*dgm10(3,3)))&
&  +6*gm(2,3)**2*(gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(2,3)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+gm(3,3)*(12*gm(2,2)**2*dgm01(3,3)&
&  *dgm10(3,3)+gm(2,3)**2*(9*dgm01(3,3)*dgm10(2,2)+24*dgm01(2,3)&
&  *dgm10(2,3)+9*dgm01(2,2)*dgm10(3,3))+30*gm(2,2)*gm(2,3)*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(20,19)=2*gm(3,3)**3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +3*gm(2,3)**3*dgm01(3,3)*dgm10(3,3)+gm(3,3)**2*(gm(2,3)*(3*dgm01(3,3)&
&  *dgm10(2,2)+24*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3))&
&  +9*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+15*gm(2,3)&
&  *gm(3,3)*(gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(2,3)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))
  cm(21,19)=gm(3,3)*(9*gm(2,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  **2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+gm(3,3)*(-3*gm(2,2)&
&  *dgm01(3,3)*dgm10(3,3)+6*gm(2,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))))
  cm(1,20)=(160*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+240*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3)))/2880.d0
  cm(2,20)=(6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&  +12*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +2*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+18*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&  -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+3*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(3,20)=(1728*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+96*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(-36*gm(1,1)*gm(1,3)*gm(2,3)&
&  +gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +72*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+144*(6*gm(1,3)**2*gm(2,3)&
&  +48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+16*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /288.d0
  cm(4,20)=(4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(2,3)&
&  *dgm10(2,3)+6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+6*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+9*(6*gm(1,3)&
&  **2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)&
&  *(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/18.d0
  cm(5,20)=(6*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+4*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&  +9*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))/36.d0
  cm(6,20)=(6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+4*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)&
&  *gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(-36*gm(1,1)&
&  *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/36.d0
  cm(7,20)=(2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(2,3))+12*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(2,3))+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+18*(48*gm(1,3)*gm(2,2)*gm(2,3)&
&  +gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+9*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)&
&  *gm(2,3)-12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(8,20)=(64*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(2,3)&
&  *dgm10(2,3)+576*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+1152*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+64*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+48*(30*gm(1,3)&
&  **2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)**2*gm(3,3)&
&  +gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)&
&  +24*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+48*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+96*(6*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+16*(90*gm(1,3)**2*gm(2,3)&
&  -36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+48*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)&
&  *gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/96.d0
  cm(9,20)=(8*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(2,3)*dgm10(2,3)+2*gm(3,3)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+8*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+4*(-36*gm(1,2)&
&  *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&  *(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+2*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +3*(6*gm(2,3)**3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)&
&  +dgm01(1,1)*dgm10(3,3))+12*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+6*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +3*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +6*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)-12*gm(1,2)&
&  **2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/12.d0
  cm(10,20)=(576*gm(3,3)**3*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(2,3))+3456*gm(1,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+144*(6*gm(1,3)**2*gm(2,3)+48*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(3,3)+864*gm(2,3)&
&  *gm(3,3)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+144*gm(3,3)&
&  *(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+48*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/288.d0
  cm(11,20)=(4*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(2,3))+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(2,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(2,3))+6*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+9*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(12,20)=(128*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(2,3)*dgm10(2,3)+1152*gm(2,3)*gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(2,3))+32*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))+576*gm(1,3)&
&  *gm(3,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+48*(48*gm(1,3)&
&  *gm(2,2)*gm(2,3)+gm(1,2)*(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)&
&  *dgm10(3,3)+48*gm(3,3)*(12*gm(2,3)**2+24*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+48*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +24*gm(3,3)*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+16*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))+96*(6*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&  *(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/96.d0
  cm(13,20)=(12*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(2,3)*dgm10(2,3)+6*gm(3,3)*(54*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)*dgm10(2,3))&
&  +2*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(2,3))+6*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)&
&  *gm(3,3))*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+9*(6*gm(2,3)&
&  **3+30*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(3,3))+3*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+9*(6*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(6*gm(2,3)**2+24*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3))+9*(48*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)&
&  *(6*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/18.d0
  cm(14,20)=(576*gm(3,3)**3*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&  *dgm10(2,3))+864*gm(1,3)*gm(2,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  **2*(3456*gm(1,3)*dgm01(2,3)*dgm10(2,3)+864*gm(1,2)*dgm01(3,3)&
&  *dgm10(2,3)+864*gm(1,2)*dgm01(2,3)*dgm10(3,3)+gm(2,3)*(864*dgm01(3,3)&
&  *dgm10(1,2)+1728*dgm01(2,3)*dgm10(1,3)+1728*dgm01(1,3)*dgm10(2,3)&
&  +864*dgm01(1,2)*dgm10(3,3))+1728*gm(2,2)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3)))+gm(3,3)*(3456*gm(1,3)*gm(2,2)*dgm01(3,3)&
&  *dgm10(3,3)+864*gm(2,3)**2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+gm(2,3)*(864*gm(1,2)*dgm01(3,3)*dgm10(3,3)+3456*gm(1,3)&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))))/144.d0
  cm(15,20)=(gm(3,3)*(2304*gm(3,3)**2*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(2,3))+288*(12*gm(1,3)*gm(2,3)+24*gm(1,2)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+3456*gm(2,3)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+3456*gm(1,3)*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))))/576.d0
  cm(16,20)=(160*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+240*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/2880.d0
  cm(17,20)=(2304*gm(2,3)**3*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))+gm(2,3)**2*(2592*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)&
&  *(864*dgm01(3,3)*dgm10(2,2)+10368*dgm01(2,3)*dgm10(2,3)+864*dgm01(2,2)&
&  *dgm10(3,3)))+gm(2,2)*gm(3,3)*(-864*gm(2,2)*dgm01(3,3)*dgm10(3,3)&
&  +gm(3,3)*(1728*dgm01(3,3)*dgm10(2,2)-3456*dgm01(2,3)*dgm10(2,3)&
&  +1728*dgm01(2,2)*dgm10(3,3)))+gm(2,3)*gm(3,3)*(1728*gm(3,3)*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+3456*gm(2,2)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))))/288.d0
  cm(18,20)=(32*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(2,3)&
&  *dgm10(2,3)+24*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(2,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+36*(6*gm(2,3)**3+30*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&  +24*gm(2,2)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/144.d0
  cm(19,20)=2*gm(3,3)**3*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&  +3*gm(2,3)**3*dgm01(3,3)*dgm10(3,3)+gm(3,3)**2*(gm(2,3)*(3*dgm01(3,3)&
&  *dgm10(2,2)+24*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3))&
&  +9*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+15*gm(2,3)&
&  *gm(3,3)*(gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(2,3)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))
  cm(20,20)=(gm(3,3)*(4608*gm(3,3)**2*dgm01(2,3)*dgm10(2,3)+288*(12*gm(2,3)&
&  **2+24*gm(2,2)*gm(3,3))*dgm01(3,3)*dgm10(3,3)+6912*gm(2,3)*gm(3,3)&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))))/576.d0
  cm(21,20)=gm(3,3)**2*(6*gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(1,21)=((90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)))/36.d0
  cm(2,21)=(3*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&  +6*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(3,21)=(80*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+4320*gm(1,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+240*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))/1440.d0
  cm(4,21)=(3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+3*gm(3,3)*(54*gm(1,3)&
&  **2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(5,21)=(240*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+160*(90*gm(1,3)**3-54*gm(1,1)&
&  *gm(1,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))&
&  /2880.d0
  cm(6,21)=(240*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+160*(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))/2880.d0
  cm(7,21)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+3*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)&
&  *gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3)))/36.d0
  cm(8,21)=(80*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*dgm01(3,3)*dgm10(3,3)+1440*gm(2,3)&
&  *gm(3,3)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+2880*gm(1,3)&
&  *gm(3,3)**2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+160*gm(3,3)&
&  *(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+80*gm(3,3)*(54*gm(1,3)**2-18*gm(1,1)&
&  *gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/480.d0
  cm(9,21)=(gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+4*gm(3,3)*(54*gm(1,3)*gm(2,3)&
&  -18*gm(1,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))&
&  +2*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)&
&  *gm(3,3)))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(3,3)&
&  *(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+2*(90*gm(1,3)**2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)&
&  -18*gm(1,1)*gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3)))/12.d0
  cm(10,21)=(gm(3,3)*(240*(54*gm(1,3)**2-18*gm(1,1)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+1440*gm(3,3)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&  *dgm10(3,3))+8640*gm(1,3)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))))/1440.d0
  cm(11,21)=(160*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+240*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,2)&
&  +dgm01(2,2)*dgm10(3,3)))/2880.d0
  cm(12,21)=(80*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)&
&  **2-18*gm(2,2)*gm(3,3)))*dgm01(3,3)*dgm10(3,3)+2880*gm(2,3)*gm(3,3)&
&  **2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+80*gm(3,3)&
&  *(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&  *dgm10(3,3))+1440*gm(1,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+160*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/480.d0
  cm(13,21)=(3*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&  +3*gm(3,3)*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&  +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3)))/18.d0
  cm(14,21)=(gm(3,3)*(240*(54*gm(1,3)*gm(2,3)-18*gm(1,2)*gm(3,3))&
&  *dgm01(3,3)*dgm10(3,3)+1440*gm(3,3)**2*(dgm01(3,3)*dgm10(1,2)&
&  +dgm01(1,2)*dgm10(3,3))+4320*gm(2,3)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)&
&  +dgm01(1,3)*dgm10(3,3))+4320*gm(1,3)*gm(3,3)*(dgm01(3,3)*dgm10(2,3)&
&  +dgm01(2,3)*dgm10(3,3))))/720.d0
  cm(15,21)=gm(3,3)**2*(6*gm(1,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&  *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))
  cm(16,21)=((90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)))/36.d0
  cm(17,21)=(80*(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+4320*gm(2,3)*gm(3,3)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+240*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))/1440.d0
  cm(18,21)=(240*gm(3,3)*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*(dgm01(3,3)&
&  *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+160*(90*gm(2,3)**3-54*gm(2,2)&
&  *gm(2,3)*gm(3,3))*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))&
&  /2880.d0
  cm(19,21)=(gm(3,3)*(240*(54*gm(2,3)**2-18*gm(2,2)*gm(3,3))*dgm01(3,3)&
&  *dgm10(3,3)+1440*gm(3,3)**2*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&  *dgm10(3,3))+8640*gm(2,3)*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&  *dgm10(3,3))))/1440.d0
  cm(20,21)=gm(3,3)**2*(6*gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&  *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))
  cm(21,21)=2*gm(3,3)**3*dgm01(3,3)*dgm10(3,3)
 end if
!
!contraction to output scalar
!
 e2nl=0.d0
 do jj=1,((rank+3)*(rank+4))/2
  tmp(:)=0.d0
  do ii=1,((rank+3)*(rank+4))/2
   tmp(:)=tmp(:)+aa(:,ii)*cm(ii,jj)
  end do
  e2nl=e2nl+tmp(1)*bb(1,jj)+tmp(2)*bb(2,jj)
 end do
 e2nl=0.25d0*e2nl

 deallocate(cm)

end subroutine contstr25a
!!***
