!{\src2tex{textfont=tt}}
!! ===================================================
!! This module contains functions used to manipulate
!! variables of structured datatype cprj_type.
!! cprj_type variables are <p_lmn|Cnk> projected
!! quantities where |p_lmn> are non-local projectors
!!                  |Cnk> are wave functions
!! ===================================================

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

!!****f* ABINIT/cprj_alloc
!! NAME
!! cprj_alloc
!!
!! FUNCTION
!! Allocation of a cprj datastructure
!!
!! COPYRIGHT
!! Copyright (C) 2007-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  ncpgr=number of gradients to be allocated
!!  nlmn(:)=sizes of cprj%cp
!!
!! SIDE EFFECTS
!!  cprj(:,:) <type(cprj_type)>= cprj datastructure
!!
!! PARENTS
!!      calc_vHxc_braket,calc_wf_qp,cgwf3,csigme,ctocprj,dyfnl3,energy,getgh1c
!!      getgsc,loper3,nstwf3,optics_paw,optics_paw_core,outkss
!!      partial_dos_fractions_paw,paw_symcprj,pawmkrhoij,prep_nonlop,rdm,scfcv
!!      scfcv3,screening,sigma,suscep_stat,vtorho,vtorho3,vtowfk,vtowfk3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_alloc(cprj,ncpgr,nlmn)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ncpgr
!arrays
 integer,intent(in) :: nlmn(:)
 type(cprj_type),intent(inout) :: cprj(:,:)
!Local variables-------------------------------
 integer :: ii,jj,n1dim,n2dim,nn

! *************************************************************************

 n1dim=size(cprj,dim=1);n2dim=size(cprj,dim=2);nn=size(nlmn,dim=1)
 if (nn/=n1dim) stop "Error in cprj_alloc: wrong sizes !"
 do jj=1,n2dim
  do ii=1,n1dim
   nn=nlmn(ii)
   cprj(ii,jj)%nlmn=nn
   cprj(ii,jj)%ncpgr=ncpgr
   allocate(cprj(ii,jj)%cp(2,nn))
   if (ncpgr>0) allocate(cprj(ii,jj)%dcp(2,ncpgr,nn))
!  XG 080820 Was needed to get rid off problems with test paral#R with four procs
   cprj(ii,jj)%cp=zero
!  END XG 080820
   if (ncpgr>0) cprj(ii,jj)%dcp=zero
  end do
 end do
end subroutine cprj_alloc
!!***

!!****f* ABINIT/cprj_free
!! NAME
!! cprj_free
!!
!! FUNCTION
!! Deallocation of a cprj datastructure
!!
!! COPYRIGHT
!! Copyright (C) 2007-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! SIDE EFFECTS
!!  cprj(:,:) <type(cprj_type)>= cprj datastructure
!!
!! PARENTS
!!      calc_vHxc_braket,calc_wf_qp,cchi0,cchi0q0,cgwf3,csigme,ctocprj,dyfnl3
!!      energy,get_bands_sym_GW,getgh1c,getgsc,loper3,nstwf3,optics_paw
!!      optics_paw_core,outkss,partial_dos_fractions_paw,paw_symcprj,pawmkrhoij
!!      prep_nonlop,rdm,scfcv,scfcv3,screening,sigma,suscep_stat,vtorho,vtorho3
!!      vtowfk,vtowfk3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_free(cprj)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
!arrays
 type(cprj_type),intent(inout) :: cprj(:,:)
!Local variables-------------------------------
 integer :: ii,jj,n1dim,n2dim

! *************************************************************************

 n1dim=size(cprj,dim=1);n2dim=size(cprj,dim=2)
 do jj=1,n2dim
  do ii=1,n1dim
   deallocate(cprj(ii,jj)%cp)
   if (cprj(ii,jj)%ncpgr>0) deallocate(cprj(ii,jj)%dcp)
  end do
 end do
end subroutine cprj_free
!!***

!!****f* ABINIT/cprj_nullify
!! NAME
!! cprj_nullify
!!
!! FUNCTION
!! Nullification of a cprj datastructure
!!
!! COPYRIGHT
!! Copyright (C) 2007-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! SIDE EFFECTS
!!  cprj(:,:) <type(cprj_type)>= cprj datastructure
!!
!! PARENTS
!!      cgwf3,prep_nonlop
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_nullify(cprj)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
!arrays
 type(cprj_type),intent(inout) :: cprj(:,:)
!Local variables-------------------------------
 integer :: ii,jj,n1dim,n2dim

! *************************************************************************

 n1dim=size(cprj,dim=1);n2dim=size(cprj,dim=2)
 do jj=1,n2dim
  do ii=1,n1dim
   cprj(ii,jj)%cp(:,:)=zero
   if (cprj(ii,jj)%ncpgr>0) cprj(ii,jj)%dcp(:,:,:)=zero
  end do
 end do
end subroutine cprj_nullify
!!***


!!****f* ABINIT/cprj_copy
!! NAME
!! cprj_copy
!!
!! FUNCTION
!! Copy a cprj datastructure into another
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  cprj_in(:,:) <type(cprj_type)>= input cprj datastructure
!!
!! OUTPUT
!!  cprj_out(:,:) <type(cprj_type)>= output cprj datastructure
!!
!! NOTES
!!  MG: What about an option to report a pointer to cprj_in?
!!
!! PARENTS
!!      cprj_utils,dyfnl3,getgh1c,getgsc,prep_nonlop,vtowfk3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_copy(cprj_in,cprj_out)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
!arrays
 type(cprj_type),intent(in) :: cprj_in(:,:)
 type(cprj_type),intent(inout) :: cprj_out(:,:)
!Local variables-------------------------------
 integer :: ii,jj,kk,n1dim_in,n1dim_out,n2dim_in,n2dim_out,ncpgr_in,ncpgr_out,nlmn

! *************************************************************************

 n1dim_in=size(cprj_in,dim=1);n1dim_out=size(cprj_out,dim=1)
 n2dim_in=size(cprj_in,dim=2);n2dim_out=size(cprj_out,dim=2)
 ncpgr_in=cprj_in(1,1)%ncpgr;ncpgr_out=cprj_out(1,1)%ncpgr
 if (n1dim_in/=n1dim_out) stop "Error in cprj_copy: n1 wrong sizes ! "
 if (n2dim_in/=n2dim_out) stop "Error in cprj_copy: n2 wrong sizes ! "
 if (ncpgr_in/=ncpgr_out) stop "Error in cprj_copy: ncpgr wrong sizes ! "

 do jj=1,n2dim_in
  do ii=1,n1dim_in
   nlmn=cprj_in(ii,jj)%nlmn
   cprj_out(ii,jj)%nlmn =nlmn
   do kk=1,nlmn
    cprj_out(ii,jj)%cp(1:2,kk)=cprj_in(ii,jj)%cp(1:2,kk)
   end do
  end do
 end do

 if (ncpgr_in>0) then
  do jj=1,n2dim_in
   do ii=1,n1dim_in
    nlmn=cprj_in(ii,jj)%nlmn
    do kk=1,nlmn
     cprj_out(ii,jj)%dcp(1:2,1:ncpgr_in,kk)=cprj_in(ii,jj)%dcp(1:2,1:ncpgr_in,kk)
    end do
   end do
  end do
 end if

end subroutine cprj_copy
!!***


!!****f* ABINIT/cprj_axpby
!! NAME
!! cprj_axpby
!!
!! FUNCTION
!! Apply AXPBY (blas-like) operation with 2 cprj datastructures:
!!  cprjy(:,:) <- alpha.cprjx(:,:)+beta.cprjy(:,:)
!!  alpha and beta are REAL scalars
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  alpha,beta= alpha,beta REAL factors
!!  cprjx(:,:) <type(cprj_type)>= input cprjx datastructure
!!
!! SIDE EFFECTS
!!  cprjy(:,:) <type(cprj_type)>= input/output cprjy datastructure
!!
!! PARENTS
!!      cgwf3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_axpby(alpha,beta,cprjx,cprjy)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
 real(dp),intent(in) :: alpha,beta
!arrays
 type(cprj_type),intent(in) :: cprjx(:,:)
 type(cprj_type),intent(inout) :: cprjy(:,:)
!Local variables-------------------------------
 integer :: ii,jj,kk,n1dimx,n1dimy,n2dimx,n2dimy,ncpgrx,ncpgry,nlmn

! *************************************************************************

 n1dimx=size(cprjx,dim=1);n1dimy=size(cprjy,dim=1)
 n2dimx=size(cprjx,dim=2);n2dimy=size(cprjy,dim=2)
 ncpgrx=cprjx(1,1)%ncpgr;ncpgry=cprjy(1,1)%ncpgr
 if (n1dimx/=n1dimy) stop "Error in cprj_axpby: n1 wrong sizes ! "
 if (n2dimx/=n2dimy) stop "Error in cprj_axpby: n2 wrong sizes ! "
 if (ncpgrx/=ncpgry) stop "Error in cprj_axpby: ncpgr wrong sizes ! "

 do jj=1,n2dimx
  do ii=1,n1dimx
   nlmn=cprjx(ii,jj)%nlmn
   cprjy(ii,jj)%nlmn =nlmn
   do kk=1,nlmn
    cprjy(ii,jj)%cp(1:2,kk)=alpha*cprjx(ii,jj)%cp(1:2,kk) &
&    +beta *cprjy(ii,jj)%cp(1:2,kk)
   end do
  end do
 end do

 if (ncpgrx>0) then
  do jj=1,n2dimx
   do ii=1,n1dimx
    nlmn=cprjx(ii,jj)%nlmn
    do kk=1,nlmn
     cprjy(ii,jj)%dcp(1:2,1:ncpgrx,kk)=alpha*cprjx(ii,jj)%dcp(1:2,1:ncpgrx,kk) &
&     +beta *cprjy(ii,jj)%dcp(1:2,1:ncpgrx,kk)
    end do
   end do
  end do
 end if

end subroutine cprj_axpby
!!***


!!****f* ABINIT/cprj_zaxpby
!! NAME
!! cprj_zaxpby
!!
!! FUNCTION
!! Apply ZAXPBY (blas-like) operation with 2 cprj datastructures:
!!  cprjy(:,:) <- alpha.cprjx(:,:)+beta.cprjy(:,:)
!!  alpha and beta are COMPLEX scalars
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  alpha(2),beta(2)= alpha,beta COMPLEX factors
!!  cprjx(:,:) <type(cprj_type)>= input cprjx datastructure
!!
!! SIDE EFFECTS
!!  cprjy(:,:) <type(cprj_type)>= input/output cprjy datastructure
!!
!! PARENTS
!!      vtowfk3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_zaxpby(alpha,beta,cprjx,cprjy)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
 real(dp),intent(in) :: alpha(2),beta(2)
!arrays
 type(cprj_type),intent(in) :: cprjx(:,:)
 type(cprj_type),intent(inout) :: cprjy(:,:)
!Local variables-------------------------------
 integer :: ii,jj,kk,n1dimx,n1dimy,n2dimx,n2dimy,ncpgrx,ncpgry,nlmn

! *************************************************************************

 n1dimx=size(cprjx,dim=1);n1dimy=size(cprjy,dim=1)
 n2dimx=size(cprjx,dim=2);n2dimy=size(cprjy,dim=2)
 ncpgrx=cprjx(1,1)%ncpgr;ncpgry=cprjy(1,1)%ncpgr
 if (n1dimx/=n1dimy) stop "Error in cprj_zaxpby: n1 wrong sizes ! "
 if (n2dimx/=n2dimy) stop "Error in cprj_zaxpby: n2 wrong sizes ! "
 if (ncpgrx/=ncpgry) stop "Error in cprj_zaxpby: ncpgr wrong sizes ! "

 do jj=1,n2dimx
  do ii=1,n1dimx
   nlmn=cprjx(ii,jj)%nlmn
   cprjy(ii,jj)%nlmn =nlmn
   do kk=1,nlmn
    cprjy(ii,jj)%cp(1,kk)=alpha(1)*cprjx(ii,jj)%cp(1,kk) &
&    -alpha(2)*cprjx(ii,jj)%cp(2,kk) &
&    +beta(1) *cprjy(ii,jj)%cp(1,kk) &
&    -beta(2) *cprjy(ii,jj)%cp(2,kk)
    cprjy(ii,jj)%cp(2,kk)=alpha(1)*cprjx(ii,jj)%cp(2,kk) &
&    +alpha(2)*cprjx(ii,jj)%cp(1,kk) &
&    +beta(1) *cprjy(ii,jj)%cp(2,kk) &
&    +beta(2) *cprjy(ii,jj)%cp(1,kk)
   end do
  end do
 end do

 if (ncpgrx>0) then
  do jj=1,n2dimx
   do ii=1,n1dimx
    nlmn=cprjx(ii,jj)%nlmn
    do kk=1,nlmn
     cprjy(ii,jj)%dcp(1,1:ncpgrx,kk)=alpha(1)*cprjx(ii,jj)%dcp(1,1:ncpgrx,kk) &
&     -alpha(2)*cprjx(ii,jj)%dcp(2,1:ncpgrx,kk) &
&     +beta(1) *cprjy(ii,jj)%dcp(1,1:ncpgrx,kk) &
&     -beta(2) *cprjy(ii,jj)%dcp(2,1:ncpgrx,kk)
     cprjy(ii,jj)%dcp(2,1:ncpgrx,kk)=alpha(1)*cprjx(ii,jj)%dcp(2,1:ncpgrx,kk) &
&     +alpha(2)*cprjx(ii,jj)%dcp(1,1:ncpgrx,kk) &
&     +beta(1) *cprjy(ii,jj)%dcp(2,1:ncpgrx,kk) &
&     +beta(2) *cprjy(ii,jj)%dcp(1,1:ncpgrx,kk)
    end do
   end do
  end do
 end if

end subroutine cprj_zaxpby
!!***


!!****f* ABINIT/cprj_diskinit_r
!! NAME
!! cprj_diskinit_r
!!
!! FUNCTION
!! Initialize a cprj temporary file for READING
!! Nothing is done if mkmem=0
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  atind(natom)=index table for atoms (see iorder below)
!!  dimcp=first dimension of cprj arrays (1 or natom)
!!  iorder=0 if cprj ordering does not change during reading
!!         1 if cprj ordering changes during writing, depending on content of atind array:
!!              - if atind=atindx  (unsorted->type-sorted)
!!              - if atind=atindx1 (type-sorted->unsorted)
!!  mkmem=number of k points which can fit in memory; set to 0 if use disk
!!  natom=number of atoms in cell
!!  ncpgr=number of gradients of cprj
!!  nlmn(dimcp)=array of dimensions of cprj datastructure that will contain the read data
!!  nspinor=number of spinorial components of the wavefunctions
!!  uncp=unit number for cprj data (if used)
!!
!! PARENTS
!!      dyfnl3,nstdy3,optics_paw,optics_paw_core,partial_dos_fractions_paw
!!      pawmkrhoij,suscep_stat,vtorho3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_diskinit_r(atind,dimcp,iorder,mkmem,natom,ncpgr,nlmn,nspinor,uncp)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
!End of the abilint section

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: dimcp,iorder,mkmem,natom,ncpgr,nspinor,uncp
!arrays
 integer,intent(in) :: atind(natom),nlmn(dimcp)
!Local variables-------------------------------
 integer :: dimcp0,iatm,iatom,ncpgr0,nspinor0
 character(len=500) :: message
 integer,allocatable :: dimlmn(:)

! *************************************************************************

 if (mkmem==0) then

  rewind uncp;read(uncp) dimcp0,ncpgr0,nspinor0
  if (dimcp/=dimcp0.or.ncpgr/=ncpgr0.or.nspinor/=nspinor0) then
   write(message,'(a,a,a,a)')ch10,&
&   ' cprj_diskinit_r : BUG -',ch10,&
&   '  _PAW file was not created with the right options !'
   call wrtout(6,message,'COLL')
   call leave_new('COLL')
  end if

  allocate(dimlmn(dimcp))
  read(uncp) dimlmn(1:dimcp)
  do iatom=1,dimcp
   if (iorder==0) then
    iatm=iatom
   else
    iatm=min(atind(iatom),dimcp)
   end if
   if (dimlmn(iatom)/=nlmn(iatm)) then
    write(message,'(a,a,a,a)')ch10,&
&    ' cprj_diskinit_r : BUG -',ch10,&
&    '  _PAW file was not created with the right options !'
    call wrtout(6,message,'COLL')
    call leave_new('COLL')
   end if
  end do
  deallocate(dimlmn)

 end if

end subroutine cprj_diskinit_r
!!***


!!****f* ABINIT/cprj_diskinit_w
!! NAME
!! cprj_diskinit_w
!!
!! FUNCTION
!! Initialize a cprj temporary file for WRITING
!! Nothing is done if mkmem=0
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  atind(natom)=index table for atoms (see iorder below)
!!  dimcp=first dimension of cprj arrays (1 or natom)
!!  iorder=0 if cprj ordering does not change during reading
!!         1 if cprj ordering changes during writing, depending on content of atind array:
!!              - if atind=atindx  (type-sorted->unsorted)
!!              - if atind=atindx1 (unsorted->type-sorted)
!!  mkmem=number of k points which can fit in memory; set to 0 if use disk
!!  natom=number of atoms in cell
!!  ncpgr=number of gradients of cprj
!!  nlmn(dimcp)=array of dimensions of cprj datastructure that will contain the read data
!!  nspinor=number of spinorial components of the wavefunctions
!!  uncp=unit number for cprj data (if used)
!!
!! PARENTS
!!      ctocprj,vtorho,vtorho3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_diskinit_w(atind,dimcp,iorder,mkmem,natom,ncpgr,nlmn,nspinor,uncp)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: dimcp,iorder,mkmem,natom,ncpgr,nspinor,uncp
!arrays
 integer,intent(in) :: atind(natom),nlmn(dimcp)
!Local variables-------------------------------
 integer :: iatm,iatom
 integer,allocatable :: dimlmn(:)

! *************************************************************************

 if (mkmem==0) then

  rewind uncp
  write(uncp) dimcp,ncpgr,nspinor

  if (iorder==0) then
   write(uncp) nlmn(1:dimcp)
  else
   allocate(dimlmn(dimcp))
   do iatom=1,dimcp
    iatm=min(atind(iatom),dimcp)
    dimlmn(iatom)=nlmn(iatm)
   end do
   write(uncp) dimlmn(1:dimcp)
   deallocate(dimlmn)
  end if

 end if

end subroutine cprj_diskinit_w
!!***


!!****f* ABINIT/cprj_diskskip
!! NAME
!! cprj_diskskip
!!
!! FUNCTION
!! Skip records in a cprj temporary file for READING
!! Nothing is done if mkmem=0
!!
!! COPYRIGHT
!! Copyright (C) 2009-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  mkmem=number of k points which can fit in memory; set to 0 if use disk
!!  ncpgr=number of gradients of cprj
!!  nres=number of records to be skipped
!!  uncp=unit number for cprj data (if used)
!!
!! PARENTS
!!      vtowfk3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_diskskip(mkmem,ncpgr,nrec,uncp)

 use defs_basis
 use defs_datatypes

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: mkmem,ncpgr,nrec,uncp
!arrays
!Local variables-------------------------------
 integer :: ii

! *************************************************************************

 if (mkmem==0) then

  do ii=1,nrec
   read(uncp)
   if (ncpgr>0) read(uncp)
  end do

 end if

end subroutine cprj_diskskip
!!***


!!****f* ABINIT/cprj_get
!! NAME
!! cprj_get
!!
!! FUNCTION
!! Read the cprj for a given k-point from memory or from a temporary file
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  atind(natom)=index table for atoms (see iorder below)
!!  cprj(dimcp,nspinor*mband*mkmem*nsppol)=input cprj (used if mkmem/=0)
!!  dimcp=first dimension of cprj_k,cprj arrays (1 or natom)
!!  iband1=index of first band
!!  ibg=shift if cprj array to locate current k-point
!!  ikpt=index of current k-point
!!  iorder=0 if cprj ordering does not change during reading
!!         1 if cprj ordering changes during writing, depending on content of atind array:
!!              - if atind=atindx  (unsorted->type-sorted)
!!              - if atind=atindx1 (type-sorted->unsorted)
!!  isppol=index of current spin component
!!  mband=maximum number of bands
!!  mkmem=number of k points which can fit in memory; set to 0 if use disk
!!  mpi_enreg=informations about MPI parallelization
!!  natom=number of atoms in cell
!!  nband=number of bands to import (usually 1 or nband_k)
!!  nband_k=total number of bands for this k-point
!!  nspinor=number of spinorial components of the wavefunctions
!!  nsppol=1 for unpolarized, 2 for spin-polarized
!!  uncp=unit number for cprj data (used if mkmem=0)
!!
!! OUTPUT
!!  cprj_k(dimcp,nspinor*nband) <type(cprj_type)>= output cprj datastructure
!!
!! PARENTS
!!      dyfnl3,nstwf3,optics_paw,optics_paw_core,partial_dos_fractions_paw
!!      pawmkrhoij,suscep_stat,vtorho3,vtowfk3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_get(atind,cprj_k,cprj,dimcp,iband1,ibg,ikpt,iorder,isppol,&
&                    mband,mkmem,mpi_enreg,natom,nband,nband_k,nspinor,nsppol,uncp)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: dimcp,iband1,ibg,ikpt,iorder,isppol,mband,mkmem,natom,nband,nband_k,nspinor,nsppol,uncp
 type(MPI_type), intent(inout) :: mpi_enreg
!arrays
 integer,intent(in) :: atind(natom)
 type(cprj_type),intent(in) :: cprj(dimcp,nspinor*mband*mkmem*nsppol)
 type(cprj_type),intent(out) :: cprj_k(dimcp,nspinor*nband)
!Local variables-------------------------------
 integer :: iatm,iatom,ib,ibsp,isp,ispinor,jband,me,nband0,ncpgr
 character(len=500) :: message

! *************************************************************************

 ncpgr=cprj_k(1,1)%ncpgr

 if ((mpi_enreg%paral_compil_kpt==1) .and. &
& (mpi_enreg%paral_compil_fft==1)) then
  me=mpi_enreg%me_kpt
 else
  call xme_init(mpi_enreg,me)
 end if

 if (mkmem==0) then

  if (iband1==1) then
   read(uncp) nband0
   if (nband_k/=nband0) then
    write(message,'(a,a,a,a)')ch10,&
&    ' cprj_get : BUG -',ch10,&
&    '  _PAW file was not created with the right options !'
    call wrtout(6,message,'PERS')
    call leave_new('PERS')
   end if
  end if

  isp=0;jband=iband1-1
  do ib=1,nband
   jband=jband+1
   if (mpi_enreg%paral_compil_kpt==1) then
    if (abs(mpi_enreg%proc_distrb(ikpt,jband,isppol)-me)/=0) then
     isp=isp+nspinor
     cycle
    end if
   end if
   do ispinor=1,nspinor
    isp=isp+1
    if (iorder==0) then
     do iatom=1,dimcp
      if (ncpgr==0) then
       read(uncp) cprj_k(iatom,isp)%cp(:,:)
      else
       read(uncp) cprj_k(iatom,isp)%cp(:,:),cprj_k(iatom,isp)%dcp(:,:,:)
      end if
     end do
    else
     do iatom=1,dimcp
      iatm=min(atind(iatom),dimcp)
      if (ncpgr==0) then
       read(uncp) cprj_k(iatm,isp)%cp(:,:)
      else
       read(uncp) cprj_k(iatm,isp)%cp(:,:),cprj_k(iatm,isp)%dcp(:,:,:)
      end if
     end do
    end if
   end do
  end do

 else

  isp=0;ibsp=ibg+nspinor*(iband1-1);jband=iband1-1
  do ib=1,nband
   jband=jband+1
   if (mpi_enreg%paral_compil_kpt==1) then
    if (abs(mpi_enreg%proc_distrb(ikpt,jband,isppol)-me)/=0) then
     isp=isp+nspinor;ibsp=ibsp+nspinor
     cycle
    end if
   end if
   do ispinor=1,nspinor
    isp=isp+1;ibsp=ibsp+1
    if (iorder==0) then
     do iatom=1,dimcp
      cprj_k(iatom,isp)%cp(:,:)=cprj(iatom,ibsp)%cp(:,:)
      if (ncpgr>0) cprj_k(iatom,isp)%dcp(:,:,:)=cprj(iatom,ibsp)%dcp(:,:,:)
     end do
    else
     do iatom=1,dimcp
      iatm=min(atind(iatom),dimcp)
      cprj_k(iatm,isp)%cp(:,:)=cprj(iatom,ibsp)%cp(:,:)
      if (ncpgr>0) cprj_k(iatm,isp)%dcp(:,:,:)=cprj(iatom,ibsp)%dcp(:,:,:)
     end do
    end if
   end do
  end do

 end if

end subroutine cprj_get
!!***


!!****f* ABINIT/cprj_put
!! NAME
!! cprj_put
!!
!! FUNCTION
!! Write the cprj for a given set of (n,k) into memory or into a temporary file
!!
!! COPYRIGHT
!! Copyright (C) 2007-2009 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  are_gathered=TRUE if cprj_k arrays have already been gathered between procs,
!!               (band-fft parallelism only)
!!               Typically, TRUE after a call to prep_nonlop routine...
!!  atind(natom)=index table for atoms (see iorder below)
!!  cprj_k(dimcp,nspinor*nband) <type(cprj_type)>= input cprj datastructure
!!  dimcp=first dimension of cprj_k,cprjnk arrays (1 or natom)
!!  iband1=index of first band
!!  ibg=shift if cprjnk array to locate current k-point
!!  ikpt=index of current k-point
!!  iorder=0 if cprj ordering does not change during reading
!!         1 if cprj ordering changes during writing, depending on content of atind array:
!!              - if atind=atindx  (type-sorted->unsorted)
!!              - if atind=atindx1 (unsorted->type-sorted)
!!  isppol=index of current spin component
!!  mband=maximum number of bands
!!  mkmem=number of k points which can fit in memory; set to 0 if use disk
!!  mpi_enreg=informations about MPI parallelization
!!  natom=number of atoms in cell
!!  nband=number of bands to export (usually 1, nband_k or nblockbd)
!!  nband_k=total number of bands for this k-point
!!  nlmn(dimcp)=array of dimensions of cprj_k,cprjnk datastructures
!!  nspinor=number of spinorial components of the wavefunctions
!!  nsppol=1 for unpolarized, 2 for spin-polarized
!!  spaceComm_band=communicator used for bands in case of band-fft parallelism
!!  uncp=unit number for cprj data (used if mkmem=0)
!!
!! SIDE EFFECTS
!!  cprj(dimcp,nspinor*mband*mkmem*nsppol)=output cprj (used if mkmem/=0)
!!
!! PARENTS
!!      ctocprj,vtowfk,vtowfk3
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

 subroutine cprj_put(are_gathered,atind,cprj_k,cprj,dimcp,iband1,ibg,ikpt,iorder,isppol,&
&           mband,mkmem,mpi_enreg,natom,nband,nband_k,nlmn,nspinor,nsppol,spaceComm_band,uncp)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iband1,ibg,ikpt,iorder,isppol,dimcp,mband,mkmem
 integer,intent(in) :: natom,nband,nband_k,nspinor,nsppol,uncp,spaceComm_band
 logical,intent(in) :: are_gathered
 type(MPI_type), intent(inout) :: mpi_enreg
!arrays
 integer :: atind(natom),nlmn(dimcp)
 type(cprj_type),intent(out) :: cprj(dimcp,nspinor*mband*mkmem*nsppol)
 type(cprj_type),intent(in) :: cprj_k(dimcp,nspinor*nband)
!Local variables-------------------------------
 integer :: iatm,iatom,iband,ibsp,icpgr,ierr,ii,ilmn,isp,ispinor,jband,jj,lmndim,me,ncpgr
 real(dp),allocatable :: buffer1(:),buffer2(:)

! *************************************************************************

 ncpgr=cprj_k(1,1)%ncpgr

 if ((mpi_enreg%paral_compil_kpt==1) .and. &
& (mpi_enreg%paral_compil_fft==1)) then
  me=mpi_enreg%me_kpt
 else
  call xme_init(mpi_enreg,me)
 end if

 if (mpi_enreg%mode_para/='b'.or.are_gathered.or.mpi_enreg%nproc_band==1) then

  if (mkmem==0) then

   if (iband1==1) write(uncp) nband_k

   isp=0;jband=iband1-1
   do iband=1,nband
    jband=jband+1
    if (mpi_enreg%paral_compil_kpt==1) then
     if (abs(mpi_enreg%proc_distrb(ikpt,jband,isppol)-me)/=0) then
      isp=isp+nspinor
      cycle
     end if
    end if
    do ispinor=1,nspinor
     isp=isp+1
     if (iorder==0) then
      do iatom=1,dimcp
       if (ncpgr==0) then
        write(uncp) cprj_k(iatom,isp)%cp(:,:)
       else
        write(uncp) cprj_k(iatom,isp)%cp(:,:),cprj_k(iatom,isp)%dcp(:,:,:)
       end if
      end do
     else
      do iatom=1,dimcp
       iatm=min(atind(iatom),dimcp)
       if (ncpgr==0) then
        write(uncp) cprj_k(iatm,isp)%cp(:,:)
       else
        write(uncp) cprj_k(iatm,isp)%cp(:,:),cprj_k(iatm,isp)%dcp(:,:,:)
       end if
      end do
     end if
    end do
   end do

  else

   isp=0;ibsp=ibg+nspinor*(iband1-1);jband=iband1-1
   do iband=1,nband
    jband=jband+1
    if (mpi_enreg%paral_compil_kpt==1)then
     if (abs(mpi_enreg%proc_distrb(ikpt,jband,isppol)-me)/=0) then
      isp=isp+nspinor;ibsp=ibsp+nspinor
      cycle
     end if
    end if
    do ispinor=1,nspinor
     isp=isp+1;ibsp=ibsp+1
     if (iorder==0) then
      do iatom=1,dimcp
       cprj(iatom,ibsp)%cp(:,:)=cprj_k(iatom,isp)%cp(:,:)
       if (ncpgr>0) cprj(iatom,ibsp)%dcp(:,:,:)=cprj_k(iatom,isp)%dcp(:,:,:)
      end do
     else
      do iatom=1,dimcp
       iatm=min(atind(iatom),dimcp)
       cprj(iatom,ibsp)%cp(:,:)=cprj_k(iatm,isp)%cp(:,:)
       if (ncpgr>0) cprj(iatom,ibsp)%dcp(:,:,:)=cprj_k(iatm,isp)%dcp(:,:,:)
      end do
     end if
    end do
   end do

  end if

 else ! mode_para==b and nband>1

  lmndim=2*sum(nlmn(1:dimcp))*(1+ncpgr)*nspinor
  allocate(buffer1(lmndim),buffer2(lmndim*mpi_enreg%nproc_band))
  isp=0;ibsp=ibg+nspinor*(iband1-1)
  do iband=1,nband  ! must be nblockbd for band-fft parallelism
   jj=1
   do ispinor=1,nspinor
    isp=isp+1
    do iatom=1,dimcp
     if (iorder==0) then
      iatm=iatom
     else
      iatm=min(atind(iatom),dimcp)
     end if
     do ilmn=1,nlmn(iatm)
      buffer1(jj:jj+1)=cprj_k(iatm,isp)%cp(1:2,ilmn)
      jj=jj+2
     end do
     if (ncpgr>0) then
      do ilmn=1,nlmn(iatm)
       do icpgr=1,ncpgr
        buffer1(jj:jj+1)=cprj_k(iatm,isp)%dcp(1:2,icpgr,ilmn)
        jj=jj+2
       end do
      end do
     end if
    end do
   end do
   call xallgather_mpi(buffer1,lmndim,buffer2,spaceComm_band,ierr)
   jj=1
   do ii=1,mpi_enreg%nproc_band
    do ispinor=1,nspinor
     ibsp=ibsp+1
     do iatom=1,dimcp
      do ilmn=1,nlmn(iatom)
       cprj(iatom,ibsp)%cp(1:2,ilmn)=buffer2(jj:jj+1)
       jj=jj+2
      end do
     end do
     if (ncpgr>0) then
      do ilmn=1,nlmn(iatom)
       do icpgr=1,ncpgr
        cprj(iatom,ibsp)%dcp(1:2,icpgr,ilmn)=buffer2(jj:jj+1)
        jj=jj+2
       end do
      end do
     end if
    end do
   end do
  end do
  deallocate(buffer1,buffer2)

 end if ! mode_para=b, nband

end subroutine cprj_put
!!***

!!****f* ABINIT/cprj_exch
!! NAME
!! cprj_exch
!!
!! FUNCTION
!! Exchange a cprj_type between two processors inside a MPI communicator.
!!
!! COPYRIGHT
!! Copyright (C) 2007-2009 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  natom=Number of atoms (size of first dimension of Cprj_send and Cprj_recv).
!!  n2dim=Size of the second dimension.
!!  nlmn(natom)=Number of nlm partial waves for each atom.
!!  Cprj_send= The datatype to be transmitted.
!!  receiver=ID of the receiver in spaceComm.
!!  sender=ID of the sender in spaceComm.
!!  spaceComm=MPI Communicator.
!!
!! OUTPUT
!!  ierr=Error status.
!!  Cprj_recv=The datatype copied on proc. receiver.
!!
!! NOTES
!!  If sender==receiver, Cprj_send is copied into Cprj_recv.
!!  It should be easy to avoid this additional copy in the calling routine.
!!
!! PARENTS
!!      outkss
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

subroutine cprj_exch(natom,n2dim,nlmn,ncpgr,Cprj_send,Cprj_recv,sender,receiver,spaceComm,ierr)

 use defs_basis
 use defs_datatypes
 use m_errors, only : assert

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_65_nonlocal, except_this_one => cprj_exch
!End of the abilint section

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,n2dim,ncpgr
 integer,intent(in) :: sender,receiver,spaceComm
 integer,intent(out) :: ierr
!arrays
 integer,intent(in) :: nlmn(natom)
 type(cprj_type),intent(in) :: Cprj_send(:,:)
 type(cprj_type),intent(inout) :: Cprj_recv(:,:)

!Local variables-------------------------------
!scalars
 integer :: iat,jj,t2dim,tcpgr,n1dim,nn
 integer :: ntotcp,ipck,rank
 logical :: ltest
!arrays
 real(dp),allocatable :: buffer_cp(:,:),buffer_cpgr(:,:,:)

! *************************************************************************

 ierr=0
 if (sender==receiver) then
  call cprj_copy(Cprj_send,Cprj_recv)
  return
 end if

 call xcomm_rank(spaceComm,rank,ierr)

!#if defined DEBUG_MODE
 nn=size(nlmn,dim=1)
 if (rank==sender) then
  n1dim=size(Cprj_send,dim=1)
  t2dim=size(Cprj_send,dim=2)
  tcpgr=Cprj_send(1,1)%ncpgr
 end if
 if (rank==receiver) then
  n1dim=size(Cprj_recv,dim=1)
  t2dim=size(Cprj_recv,dim=2)
  tcpgr=Cprj_recv(1,1)%ncpgr
 end if
 call assert(   (nn==n1dim),'cprj_exch: size mismatch in natom!')
 call assert((t2dim==n2dim),'cprj_exch: size mismatch in dim=2!')
 call assert((tcpgr==ncpgr),'cprj_exch: size mismatch in ncpgr!')
!#endif

 ntotcp=n2dim*SUM(nlmn(:))

 allocate(buffer_cp(2,ntotcp))
 if (ncpgr/=0) allocate(buffer_cpgr(2,ncpgr,ntotcp))

!=== Pack Cprj_send ===
 if (rank==sender) then
  ipck=0
  do jj=1,n2dim
   do iat=1,natom
    nn=nlmn(iat)
    buffer_cp(:,ipck+1:ipck+nn)=Cprj_send(iat,jj)%cp(:,1:nn)
    if (ncpgr/=0) buffer_cpgr(:,:,ipck+1:ipck+nn)=Cprj_send(iat,jj)%dcp(:,:,1:nn)
    ipck=ipck+nn
   end do
  end do
 end if

!=== Transmit data ===
 call xexch_mpi(buffer_cp,2*ntotcp,sender,buffer_cp,receiver,spaceComm,ierr)
 if (ncpgr/=0) then
  call xexch_mpi(buffer_cpgr,2*ncpgr*ntotcp,sender,buffer_cpgr,receiver,spaceComm,ierr)
 end if

!=== UnPack buffers into Cprj_recv ===
 if (rank==receiver) then
  ipck=0
  do jj=1,n2dim
   do iat=1,natom
    nn=nlmn(iat)
    Cprj_recv(iat,jj)%cp(:,1:nn)=buffer_cp(:,ipck+1:ipck+nn)
    if (ncpgr/=0) Cprj_recv(iat,jj)%dcp(:,:,1:nn)=buffer_cpgr(:,:,ipck+1:ipck+nn)
    ipck=ipck+nn
   end do
  end do
 end if

 deallocate(buffer_cp)
 if (ncpgr/=0) deallocate(buffer_cpgr)

end subroutine cprj_exch
!!***

!!****f* ABINIT/cprj_bcast
!! NAME
!! cprj_bcast
!!
!! FUNCTION
!! Broadcast a cprj_type from master to all nodes inside a MPI communicator.
!!
!! COPYRIGHT
!! Copyright (C) 2007-2009 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  natom=Number of atoms (size of the first dimension of Cprj).
!!  n2dim=Size of the second dimension of Cprj.
!!  ncpgr=Number of gradients that have to be cast. It is a bit redundant but, it can be used to
!!   broad cast only the %cp"s without caring about the gradients. Just set it to 0 but be careful!
!!  nlmn(natom)=Number of nlm partial waves for each atom.
!!  master=ID of the sending node in spaceComm.
!!  spaceComm=MPI Communicator.
!!
!! OUTPUT
!!  ierr=Error status.
!!  Cprj(natom,n2dim)<cprj_type>=The datatype to be transmitted by master and received by the others nodes.
!!
!! PARENTS
!!
!! CHILDREN
!!      assert,xcast_mpi,xcomm_rank,xcomm_size
!!
!! SOURCE

subroutine cprj_bcast(natom,n2dim,nlmn,ncpgr,Cprj,master,spaceComm,ierr)

 use defs_basis
 use defs_datatypes
 use m_errors, only : assert

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
!End of the abilint section

 implicit none
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,n2dim,ncpgr
 integer,intent(in) :: master,spaceComm
 integer,intent(out) :: ierr
!arrays
 integer,intent(in) :: nlmn(natom)
 type(cprj_type),intent(inout) :: Cprj(natom,n2dim)

!Local variables-------------------------------
!scalars
 integer :: iat,jj,tcpgr,n1dim,nn
 integer :: ntotcp,ipck,rank,nprocs
 logical :: ltest
!arrays
 real(dp),allocatable :: buffer_cp(:,:),buffer_cpgr(:,:,:)

! *************************************************************************

 ierr=0
 call xcomm_size(spaceComm,nprocs,ierr)
 if (nprocs==1) return

 call xcomm_rank(spaceComm,rank,ierr)

!#if defined DEBUG_MODE
 nn=size(nlmn,dim=1)
 n1dim=size(Cprj,dim=1)
 call assert((nn==n1dim),'cprj_bcast: size mismatch in natom!')
!!!tcpgr=Cprj(1,1)%ncpgr
!!!call assert((tcpgr==ncpgr),'cprj_bcast: size mismatch in ncpgr!')
!#endif

 ntotcp=n2dim*SUM(nlmn(:))

 allocate(buffer_cp(2,ntotcp))
 if (ncpgr/=0) allocate(buffer_cpgr(2,ncpgr,ntotcp))

!=== Master packs Cprj ===
!Write a routine to pack/unpack?
 if (rank==master) then
  ipck=0
  do jj=1,n2dim
   do iat=1,natom
    nn=nlmn(iat)
    buffer_cp(:,ipck+1:ipck+nn)=Cprj(iat,jj)%cp(:,1:nn)
    if (ncpgr/=0) buffer_cpgr(:,:,ipck+1:ipck+nn)=Cprj(iat,jj)%dcp(:,:,1:nn)
    ipck=ipck+nn
   end do
  end do
 end if

!=== Transmit data ===
 call xcast_mpi(buffer_cp,master,spaceComm,ierr)
 if (ncpgr/=0) then
  call xcast_mpi(buffer_cpgr,master,spaceComm,ierr)
 end if

!=== UnPack the received buffer ===
 if (rank/=master) then
  ipck=0
  do jj=1,n2dim
   do iat=1,natom
    nn=nlmn(iat)
    Cprj(iat,jj)%cp(:,1:nn)=buffer_cp(:,ipck+1:ipck+nn)
    if (ncpgr/=0) Cprj(iat,jj)%dcp(:,:,1:nn)=buffer_cpgr(:,:,ipck+1:ipck+nn)
    ipck=ipck+nn
   end do
  end do
 end if

 deallocate(buffer_cp)
 if (ncpgr/=0) deallocate(buffer_cpgr)

end subroutine cprj_bcast
!!***

