!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_paw_toolbox
!! NAME
!!  m_paw_toolbox
!!
!! FUNCTION
!!  This module contains basic tools to initialize, nullify and free basic
!!  PAW objects.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MG, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! TODO 
!! This module should contain the definition of the data types as well.
!! At present these procedures are used only in GW but, hopefully, they will
!! become standard PAW methods. 
!!
!! NOTES
!!
!! * Routines tagged with "@type_name" are strongly connected to the definition of the data type. 
!!   Strongly connected means that the proper functioning of the implementation relies on the 
!!   assumption that the tagged procedure is consistent with the type declaration.
!!   Every time a developer changes the structure "type_name" adding new entries, he/she has to make sure 
!!   that all the strongly connected routines are changed accordingly to accomodate the modification of the data type. 
!!   Typical examples of strongly connected routines are creation, destruction or reset methods.
!!
!! * gcc 4.3 does not hanlde correctly the case in which there are routines with similar names defined 
!!   in the same module. This creates problems at link-time if the module contains procedures 
!!   beginning with the same name, e.g init_longname, init_longname_slightly_different.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_paw_toolbox

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors 

 implicit none

 private 

 public ::             &
&  reset_paw_ij_flags, &
&  destroy_paw_ij,     &
&  init_paw_ij,        &
&  destroy_paw_an,     &
&  init_pawfgr,        &
&  nullify_paw_ij,     &
&  nullify_paw_an,     &
&  init_paw_an,        &
&  print_pawtab,       &
&  print_paw_ij,       & !FIXME not yet tested
&  pawfgrtab_free,     &
&  pawfgrtab_init,     &
&  pawfgrtab_print,    &
&  pawfgrtab_nullify,  &
&  reset_paw_an_flags

CONTAINS  !===========================================================
!!***

!!****f* m_paw_toolbox/reset_paw_ij_flags
!! NAME
!! reset_paw_ij_flags
!!
!! FUNCTION
!!  Set all paw_ij flags set to 0.
!!
!! INPUTS
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine reset_paw_ij_flags(Paw_ij_flags)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_flags_type),intent(inout) :: Paw_ij_flags

! *************************************************************************

 ! @Paw_ij_flags_type
 Paw_ij_flags%has_dijfr     =0
 Paw_ij_flags%has_dijhartree=0
 Paw_ij_flags%has_dijhat    =0
 Paw_ij_flags%has_dijso     =0
 Paw_ij_flags%has_dijU      =0
 Paw_ij_flags%has_dijxc     =0
 Paw_ij_flags%has_dijxc_val =0
 Paw_ij_flags%has_dijR      =0
 Paw_ij_flags%has_dijL      =0
 Paw_ij_flags%has_dijLr3    =0

end subroutine reset_paw_ij_flags
!!***

!!****f* m_paw_toolbox/copy_paw_ij_flags
!! NAME
!! copy_paw_ij_flags
!!
!! FUNCTION
!!  Copy a Paw_ij_flags structure.
!!
!! INPUTS
!!  ij_flags_in
!! 
!! OUTPUT
!!  ij_flags_out
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine copy_paw_ij_flags(ij_flags_in, ij_flags_out)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_flags_type),intent(in)    :: ij_flags_in
 type(Paw_ij_flags_type),intent(inout) :: ij_flags_out

! *************************************************************************

 ! @Paw_ij_flags_type
 ij_flags_out%has_dijfr      = ij_flags_in%has_dijfr     
 ij_flags_out%has_dijhartree = ij_flags_in%has_dijhartree
 ij_flags_out%has_dijhat     = ij_flags_in%has_dijhat    
 ij_flags_out%has_dijso      = ij_flags_in%has_dijso     
 ij_flags_out%has_dijU       = ij_flags_in%has_dijU      
 ij_flags_out%has_dijxc      = ij_flags_in%has_dijxc     
 ij_flags_out%has_dijxc_val  = ij_flags_in%has_dijxc_val 
 ij_flags_out%has_dijR       = ij_flags_in%has_dijR
 ij_flags_out%has_dijL       = ij_flags_in%has_dijL
 ij_flags_out%has_dijLr3     = ij_flags_in%has_dijLr3

end subroutine copy_paw_ij_flags
!!***

!!****f* m_paw_toolbox/destroy_paw_ij
!! NAME
!! destroy_paw_ij
!!
!! FUNCTION
!!
!! INPUTS
!!  paw_ij(:)<type(paw_ij_type)>=paw arrays given on (i,j) channels
!!
!! OUTPUT
!!  See side effects
!!
!! SIDE EFFECTS
!!  All associated pointers in paw_ij(:) are deallocated
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_paw_ij(Paw_ij)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_type),intent(inout) :: Paw_ij(:)

!Local variables-------------------------------
 integer :: iat,natom
 character(len=500) :: msg

! *************************************************************************

 DBG_ENTER("COLL")

 ! @Paw_ij_type
 natom=SIZE(Paw_ij)
 do iat=1,natom
  if (associated(Paw_ij(iat)%dij       )) deallocate(Paw_ij(iat)%dij       )
  if (associated(Paw_ij(iat)%dijfr     )) deallocate(Paw_ij(iat)%dijfr     )
  if (associated(Paw_ij(iat)%dijhartree)) deallocate(Paw_ij(iat)%dijhartree)
  if (associated(Paw_ij(iat)%dijhat    )) deallocate(Paw_ij(iat)%dijhat    )
  if (associated(Paw_ij(iat)%dijU      )) deallocate(Paw_ij(iat)%dijU      )
  if (associated(Paw_ij(iat)%dijso     )) deallocate(Paw_ij(iat)%dijso     )
  if (associated(Paw_ij(iat)%dijxc     )) deallocate(Paw_ij(iat)%dijxc     )
  if (associated(Paw_ij(iat)%dijxc_val )) deallocate(Paw_ij(iat)%dijxc_val )
  if (associated(Paw_ij(iat)%dijR      )) deallocate(Paw_ij(iat)%dijR      )
  if (associated(Paw_ij(iat)%dijL      )) deallocate(Paw_ij(iat)%dijL      )
  if (associated(Paw_ij(iat)%dijLr3    )) deallocate(Paw_ij(iat)%dijLr3    )
  if (associated(Paw_ij(iat)%noccmmp   )) deallocate(Paw_ij(iat)%noccmmp   )
  if (associated(Paw_ij(iat)%nocctot   )) deallocate(Paw_ij(iat)%nocctot   )
  if (associated(Paw_ij(iat)%vpawx     )) deallocate(Paw_ij(iat)%vpawx     )

  ! === Reset all has_* flags ===
  Paw_ij(iat)%has_dijfr=0
  Paw_ij(iat)%has_dijhartree=0
  Paw_ij(iat)%has_dijhat    =0
  Paw_ij(iat)%has_dijso     =0
  Paw_ij(iat)%has_dijU      =0
  Paw_ij(iat)%has_dijxc     =0
  Paw_ij(iat)%has_dijxc_val =0
  Paw_ij(iat)%has_dijR      =0
  Paw_ij(iat)%has_dijL      =0
  Paw_ij(iat)%has_dijLr3    =0
 end do

 DBG_EXIT("COLL")

end subroutine destroy_Paw_ij
!!***

!!****f* m_paw_toolbox/init_paw_ij
!! NAME
!! init_paw_ij
!!
!! FUNCTION
!!  Initialize a Paw_ij data type.
!!
!! INPUTS
!!  cplex=1 if all on-site PAW quantities are real, 2 if they are complex
!!  cplex_dij=1 if dij are real, 2 if they are complex
!!  natom=Number of atoms.
!!  ntypat=Number of types of atoms in cell.
!!  nspinor=Number of spinor components
!!  nsppol=Number of independent spin polarizations.
!!  nspden=Number of spin-density components
!!  pawspnorb=1 if spin-orbit coupling is activated
!!  typat(natom)=Type of each atom
!!  Pawtab(ntypat)<type(pawtab_type)>=PAW tabulated starting data
!!
!! OPTIONAL INPUTS
!!  has_dijhat=1 to allocate Paw_ij%dijhat, 0 otherwise (default)
!!  has_dijxc=1 to allocate Paw_ij%dijxc, 0 otherwise (default)
!!  has_dijxc_val=1 to allocate Paw_ij%dijxc_val, 0 otherwise (default)
!!  has_dijhartree=1 to allocate Paw_ij%dijhartree, 0 otherwise (default)
!!  has_dijso=1 to allocate Paw_ij%dijso, used only if pawspnorb>0. 0 otherwise (default)
!!  has_dijU=1 to allocate Paw_ij%dijU, used only if Pawtab(itypat)%usepawu>0. 0 otherwise (default).
!!
!! OUTPUT
!!  Paw_ij(natom)<type(paw_ij_type)>=data structure containing PAW arrays given on (i,j) channels.
!!   In output all the basic dimensions are defined and the arrays are allocated
!!   according to the input variables.
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_paw_ij(Paw_ij,cplex,cplex_dij,nspinor,nsppol,nspden,pawspnorb,natom,ntypat,typat,Pawtab,&
& has_dijfr,has_dijhartree,has_dijhat,has_dijxc,has_dijxc_val,has_dijso,has_dijU) ! Optional

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,cplex_dij,nspinor,nspden,nsppol,pawspnorb,natom,ntypat
 integer,optional,intent(in) :: has_dijfr,has_dijhat,has_dijxc,has_dijxc_val
 integer,optional,intent(in) :: has_dijso,has_dijhartree,has_dijU
!arrays
 integer,intent(in) :: typat(natom)
 type(Paw_ij_type),intent(inout) :: Paw_ij(natom)
 type(Pawtab_type),intent(in) :: Pawtab(ntypat)

!Local variables-------------------------------
 integer :: iat,itypat,lmn2_size,ndij
 character(len=500) :: msg

! *************************************************************************

 DBG_ENTER("COLL")

 !allocate(Paw_ij(natom))
 !call nullify_paw_ij(Paw_ij)

 ! @Paw_ij_type
 do iat=1,natom
  itypat=typat(iat)
  lmn2_size              =Pawtab(itypat)%lmn2_size
  Paw_ij(iat)%cplex      =cplex
  !Paw_ij(iat)%cplex_dij  =nspinor
  !Paw_ij(iat)%cplex_dij  =MAX(cplex,1+pawspnorb,nspinor)
  Paw_ij(iat)%cplex_dij  =cplex_dij
  Paw_ij(iat)%nspden     =nspden
  Paw_ij(iat)%nsppol     =nsppol
  Paw_ij(iat)%lmn_size   =Pawtab(itypat)%lmn_size
  Paw_ij(iat)%lmn2_size  =lmn2_size
  Paw_ij(iat)%ndij       =MAX(nspinor**2,nspden)
  !Paw_ij(iat)%lmnmix_sz =  do we need this? It seems it is not used anymore and can be removed

  ndij=Paw_ij(iat)%ndij

  allocate(Paw_ij(iat)%dij(cplex_dij*lmn2_size,ndij))

  ! === Allocation for PAW+U ===
  if (Pawtab(itypat)%usepawu>0) then
   allocate(Paw_ij(iat)%noccmmp(2*Pawtab(itypat)%lpawu+1,2*Pawtab(itypat)%lpawu+1,nspden))
   allocate(Paw_ij(iat)%nocctot(nspden))
  end if

  ! === Allocation for PAW+LEXX ===
  if (Pawtab(itypat)%useexexch>0) then
   ! TODO solve issue with first dimension
   allocate(Paw_ij(iat)%vpawx(1,lmn2_size,nspden))
  end if

  ! ============================
  ! === Optional allocations ===
  ! ============================
  Paw_ij(iat)%has_dijfr=0
  if (PRESENT(has_dijfr)) then
   if (has_dijfr/=0) then
    Paw_ij(iat)%has_dijfr=1
    allocate(Paw_ij(iat)%dijfr(cplex*lmn2_size,ndij))
   end if
  end if

  Paw_ij(iat)%has_dijhartree=0
  if (PRESENT(has_dijhartree)) then
   if (has_dijhartree/=0) then
    Paw_ij(iat)%has_dijhartree=1
    allocate(Paw_ij(iat)%dijhartree(cplex*lmn2_size))
   end if
  end if

  Paw_ij(iat)%has_dijhat=0
  if (PRESENT(has_dijhat)) then
   if (has_dijhat/=0) then
    Paw_ij(iat)%has_dijhat=1
    allocate(Paw_ij(iat)%dijhat(cplex_dij*lmn2_size,ndij))
   end if
  end if

  Paw_ij(iat)%has_dijxc=0
  if (PRESENT(has_dijxc)) then
   if (has_dijxc/=0) then
    Paw_ij(iat)%has_dijxc=1
    allocate(Paw_ij(iat)%dijxc(cplex_dij*lmn2_size,ndij))
   end if
  end if

  Paw_ij(iat)%has_dijxc_val=0
  if (PRESENT(has_dijxc_val)) then
   if (has_dijxc_val/=0) then
    Paw_ij(iat)%has_dijxc_val=1
    allocate(Paw_ij(iat)%dijxc_val(cplex_dij*lmn2_size,ndij))
   end if
  end if

  Paw_ij(iat)%has_dijU=0
  if (PRESENT(has_dijU)) then
   if (has_dijU/=0.and.Pawtab(itypat)%usepawu>0) then
    Paw_ij(iat)%has_dijU=1
    allocate(Paw_ij(iat)%dijU(cplex_dij*lmn2_size,ndij))
   end if
  end if

  Paw_ij(iat)%has_dijso=0
  if (PRESENT(has_dijso)) then
   if (has_dijso/=0.and.pawspnorb>0) then
    Paw_ij(iat)%has_dijso=1
    allocate(Paw_ij(iat)%dijso(cplex_dij*lmn2_size,ndij))
   end if
  end if

 end do !iat

 DBG_EXIT("COLL")

end subroutine init_paw_ij
!!***

!!****f* m_paw_toolbox/destroy_paw_an
!! NAME
!! destroy_paw_an
!!
!! FUNCTION
!!
!! INPUTS
!!  Paw_an(:)<type(Paw_an_type)>=various arrays given on ANgular mesh or ANgular moments
!!
!! OUTPUT
!!  See side effects
!!
!! SIDE EFFECTS
!!  All associated pointers in Paw_an(:) are deallocated
!!
!! NOTES
!!  vh1 and vht1 are defined in the data structure but never used.
!!  Cannot test for association status since these quantities are
!!  not nullified before entering the calculation
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_Paw_an(Paw_an)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_type),intent(inout) :: Paw_an(:)

!Local variables-------------------------------
 integer :: iat,natom,itypat
 character(len=500) :: msg

! *************************************************************************

 DBG_ENTER("COLL")

 !@Paw_an_type

 natom=SIZE(Paw_an)

! do iat=1,natom
!  itypat=typat(iat)
!  deallocate(Paw_an(iat)%lmselect)
!  !deallocate(Paw_an(iat)%vh1,Paw_an(iat)%vht1)      !TODO nullify these arrays
!  deallocate(paw_an(iat)%vxc1,Paw_an(iat)%vxct1)
!  if (Paw_an(iat)%has_vxcval==1 ) deallocate(Paw_an(iat)%vxc1_val,Paw_an(iat)%vxct1_val)
!  if (Pawtab(itypat)%useexexch>0) deallocate(Paw_an(iat)%vxc_ex)
! end do

 do iat=1,natom
  if (associated(Paw_an(iat)%lmselect )) deallocate(Paw_an(iat)%lmselect )
  if (associated(Paw_an(iat)%vh1      )) deallocate(Paw_an(iat)%vh1      )
  if (associated(Paw_an(iat)%vht1     )) deallocate(Paw_an(iat)%vht1     )
  if (associated(Paw_an(iat)%vxc1     )) deallocate(Paw_an(iat)%vxc1     )
  if (associated(Paw_an(iat)%vxc1_val )) deallocate(Paw_an(iat)%vxc1_val )
  if (associated(Paw_an(iat)%vxct1    )) deallocate(Paw_an(iat)%vxct1    )
  if (associated(Paw_an(iat)%vxct1_val)) deallocate(Paw_an(iat)%vxct1_val)
  if (associated(Paw_an(iat)%vxc_ex   )) deallocate(Paw_an(iat)%vxc_ex   )
  if (associated(Paw_an(iat)%kxc1     )) deallocate(Paw_an(iat)%kxc1     )
  if (associated(Paw_an(iat)%kxct1    )) deallocate(Paw_an(iat)%kxct1    )

  ! === Reset all has_* flags ===
  Paw_an(iat)%has_vhartree=0
  Paw_an(iat)%has_vxc     =0
  Paw_an(iat)%has_vxcval  =0
  Paw_an(iat)%has_kxc     =0
 end do !iat

 DBG_EXIT("COLL")

end subroutine destroy_Paw_an
!!***

!!****f* m_paw_toolbox/init_pawfgr
!! NAME
!! init_pawfgr
!!
!! FUNCTION
!!  Initialize a pawfgr_type datatype, reporting also info on the FFT mesh
!!  according to the method used (norm-conserving or PAW)
!!
!! INPUTS
!! k0(3)=input k vector for k+G sphere
!! Dtset <type(dataset_type)>=all input variables for this dataset
!!   %dilatmx
!!   %usepaw
!!   %natom
!!   %ngfft
!!   %ngfftdg
!!   %nfft
!!   %mgfft
!!   %mgfftdg
!!   %dilatmx
!!   %pawecutdg
!!   %ecut
!!
!! OUTPUT
!!  ecut_eff=effective energy cutoff (hartree) for coarse planewave basis sphere
!!  ecutdg_eff=effective energy cutoff (hartree) for dense planewave basis sphere
!!  gmet(3,3)=reciprocal space metric (bohr^-2)
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  mgfft=maximum size of 1D FFTs
!!  ngfftc(18),ngfftf(18)=contain all needed information about 3D FFT, for coarse and dense FFT mesh, respectively.
!!   see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  Pawfgr<pawfgr_type>=For PAW, Fine rectangular GRid parameters and related data
!!
!! NOTES
!!  Pawfgr%gsqcut=gsqcutf_eff is never used thus it is not initialized inside this routine.
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_pawfgr(Dtset,k0,gmet,Pawfgr,mgfftf,nfftf,ecut_eff,ecutdg_eff,&
& gsqcutc_eff,gsqcutf_eff,ngfftc,ngfftf)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_56_recipspace
 use interfaces_66_paw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(out) :: nfftf,mgfftf
 real(dp),intent(in) :: gmet(3,3)
 real(dp),intent(out) :: ecut_eff,ecutdg_eff
 real(dp),intent(out) :: gsqcutf_eff,gsqcutc_eff
 type(dataset_type),intent(in) :: Dtset
 type(Pawfgr_type),intent(out) :: Pawfgr
!arrays
 integer,intent(out) :: ngfftc(18),ngfftf(18)
 real(dp),intent(in) :: k0(3)

!Local variables-------------------------------
 integer :: ii,nfftc_tot,nfftf_tot
 real(dp) :: boxcut,boxcutc
 character(len=500) :: msg

!************************************************************************

 DBG_ENTER("COLL")

 !@Pawfgr_type

 ngfftc(:)=Dtset%ngfft(:)

 SELECT CASE (Dtset%usepaw)

 CASE (0)
  ! === Norm-conserving pseudopotentials ===
  nfftf=Dtset%nfft ; mgfftf=Dtset%mgfft ; ngfftf(:)=Dtset%ngfft(:)
  Pawfgr%usefinegrid=0 ; allocate(Pawfgr%coatofin(0),Pawfgr%fintocoa(0))
  ecut_eff  =Dtset%ecut*Dtset%dilatmx**2
  ecutdg_eff=ecut_eff

 CASE (1)
  ! == PAW calculation ===
  if (Dtset%pawecutdg>=1.0000001_dp*Dtset%ecut) then
   ! * Use fine FFT grid generated according to pawecutdg.
   nfftf=Dtset%nfftdg ; mgfftf=Dtset%mgfftdg ; ngfftf(:)=Dtset%ngfftdg(:)
   nfftc_tot =ngfftc(1)*ngfftc(2)*ngfftc(3)
   nfftf_tot =ngfftf(1)*ngfftf(2)*ngfftf(3)
   Pawfgr%usefinegrid=1 ; allocate(Pawfgr%coatofin(nfftc_tot),Pawfgr%fintocoa(nfftf_tot))
   call indgrid(Pawfgr%coatofin,Pawfgr%fintocoa,nfftc_tot,nfftf_tot,ngfftc,ngfftf)
  else
   ! * Do not use fine FFT mesh. Simple transfer that can be done in parallel with only local info.
   nfftf=Dtset%nfft ; mgfftf=Dtset%mgfft ; ngfftf(:)=Dtset%ngfft(:)
   Pawfgr%usefinegrid=0 ; allocate(Pawfgr%coatofin(Dtset%nfft),Pawfgr%fintocoa(Dtset%nfft))
   do ii=1,Dtset%nfft
    Pawfgr%coatofin(ii)=ii ; Pawfgr%fintocoa(ii)=ii
   end do
  end if

  ! == Store useful dimensions in Pawfgr ===
  Pawfgr%natom=Dtset%natom
  Pawfgr%nfftc=Dtset%nfft ; Pawfgr%mgfftc=Dtset%mgfft ; Pawfgr%ngfftc(:)=Dtset%ngfft(:)
  Pawfgr%nfft=nfftf       ; Pawfgr%mgfft=mgfftf       ; Pawfgr%ngfft (:)=ngfftf(:)
  ecutdg_eff=Dtset%pawecutdg*Dtset%dilatmx**2
  ecut_eff  =Dtset%ecut*Dtset%dilatmx**2

 CASE DEFAULT
  write(msg,'(a,i4)')' Wrong value of usepaw: ',Dtset%usepaw
  MSG_BUG(msg)
 END SELECT
 !
 ! === Get boxcut for given gmet, ngfft, and ecut (center at k0) ===
 !     boxcut=ratio of basis sphere diameter to fft box side
 if (Dtset%usepaw==1) then
  write(msg,'(2a)')ch10,' Coarse grid specifications '!(used for wave-functions):'
  call wrtout(std_out,msg,'COLL') !; call wrtout(ab_out,msg,'COLL')
  call getcut(boxcutc,ecut_eff,gmet,gsqcutc_eff,Dtset%iboxcut,std_out,k0,ngfftc)
  write(msg,'(2a)')ch10,' Fine grid specifications (used for densities):'
  call wrtout(std_out,msg,'COLL') !; call wrtout(ab_out,msg,'COLL')
  call getcut(boxcut,ecutdg_eff,gmet,gsqcutf_eff,Dtset%iboxcut,std_out,k0,ngfftf)
  !FIXME this is never use, should ask Marc
  !Pawfgr%gsqcut=gsqcutf_eff
 else
  call getcut(boxcut,ecut_eff,gmet,gsqcutc_eff,Dtset%iboxcut,std_out,k0,ngfftc)
  gsqcutf_eff=gsqcutc_eff
 end if
 !
 ! === Check that boxcut>=2 if intxc=1; otherwise intxc must be set=0 ===
 if (boxcut<two .and. Dtset%intxc==1) then
  write(msg,'(a,es12.4,5a)')&
&  ' boxcut=',boxcut,' is < 2.0  => intxc must be 0;',ch10,&
&  ' Need larger ngfft to use intxc=1.',ch10,&
&  ' Action : you could increase ngfft, or decrease ecut, or put intxc=0.'
  MSG_ERROR(msg)
 end if

 DBG_EXIT("COLL")

end subroutine init_pawfgr
!!***

!!****f* m_paw_toolbox/nullify_paw_ij
!! NAME
!!  nullify_paw_ij
!!
!! FUNCTION
!!  Nullify the pointers in a paw_ij data type
!!
!! INPUTS
!!  See SIDE EFFECTS
!!
!! SIDE EFFECTS
!!  Paw_ij(:)<type(paw_ij_type)>=PAW arrays given on (i,j) channels. Nullified in output
!!
!! NOTES
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_paw_ij(Paw_ij)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_type),intent(inout) :: Paw_ij(:)

!Local variables-------------------------------
 integer :: iat,natom

! *************************************************************************

 !@Paw_ij_type

 natom=SIZE(Paw_ij(:))

 do iat=1,natom
  nullify(Paw_ij(iat)%dij       )
  nullify(Paw_ij(iat)%dijfr     )
  nullify(Paw_ij(iat)%dijhartree)
  nullify(Paw_ij(iat)%dijhat    )
  nullify(Paw_ij(iat)%dijU      )
  nullify(Paw_ij(iat)%dijso     )
  nullify(Paw_ij(iat)%dijxc     )
  nullify(Paw_ij(iat)%dijxc_val )
  nullify(Paw_ij(iat)%dijR      )
  nullify(Paw_ij(iat)%dijL      )
  nullify(Paw_ij(iat)%dijLr3    )
  nullify(Paw_ij(iat)%noccmmp   )
  nullify(Paw_ij(iat)%nocctot   )
  nullify(Paw_ij(iat)%vpawx     )

  ! === Set all has_* flags to zero ===
  Paw_ij(iat)%has_dijfr     =0
  Paw_ij(iat)%has_dijhartree=0
  Paw_ij(iat)%has_dijhat    =0
  Paw_ij(iat)%has_dijso     =0
  Paw_ij(iat)%has_dijU      =0
  Paw_ij(iat)%has_dijxc     =0
  Paw_ij(iat)%has_dijxc_val =0
  Paw_ij(iat)%has_dijR      =0
  Paw_ij(iat)%has_dijL      =0
  Paw_ij(iat)%has_dijLr3    =0
 end do !iat

end subroutine nullify_paw_ij
!!***

!!****f* m_paw_toolbox/nullify_paw_an
!! NAME
!!  nullify_paw_an
!!
!! FUNCTION
!!  Nullify the pointers in a paw_an data type
!!
!! INPUTS
!!  See SIDE EFFECTS
!!
!! SIDE EFFECTS
!!  Paw_an(:)<type(paw_an_type)>=PAW arrays given on ANgular mesh or ANgular moments.
!!  Nullified in output
!!
!! NOTES
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_paw_an(Paw_an)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_type),intent(inout) :: Paw_an(:)

!Local variables-------------------------------
 integer :: iat,natom

! *************************************************************************

 !@Paw_an_type
 natom=SIZE(Paw_an(:))

 do iat=1,natom
  nullify(Paw_an(iat)%lmselect )
  nullify(Paw_an(iat)%vh1      )
  nullify(Paw_an(iat)%vht1     )
  nullify(Paw_an(iat)%vxc1     )
  nullify(Paw_an(iat)%vxc1_val )
  nullify(Paw_an(iat)%vxct1_val)
  nullify(Paw_an(iat)%vxc_ex   )
  nullify(Paw_an(iat)%kxc1     )
  nullify(Paw_an(iat)%kxct1    )

  ! === Set all has_* flags to zero ===
  Paw_an(iat)%has_kxc      =0
  Paw_an(iat)%has_vhartree =0
  Paw_an(iat)%has_vxc      =0
  Paw_an(iat)%has_vxcval   =0
 end do

end subroutine nullify_paw_an
!!***

!!****f* m_paw_toolbox/init_paw_an
!! NAME
!!  init_paw_an
!!
!! FUNCTION
!!  Initialize a paw_an data type.
!!
!! INPUTS
!!  See SIDE EFFECTS
!!
!! SIDE EFFECTS
!!  Paw_an(:)<type(paw_an_type)>=PAW arrays given on ANgular mesh or ANgular moments.
!!  Initialized in output
!!
!! NOTES
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE


subroutine init_paw_an(natom,ntypat,nkxc1,nspden,cplex,pawxcdev,typat,Pawang,Pawtab,Paw_an,&
&                      has_vhartree,has_vxc,has_vxcval,has_kxc) ! Optional

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,nkxc1,ntypat,cplex,nspden,pawxcdev
 integer,optional,intent(in) :: has_vhartree,has_vxc,has_vxcval,has_kxc
!arrays
 integer,intent(in) :: typat(natom)
 type(Pawang_type),intent(in) :: Pawang
 type(Pawtab_type),intent(in) :: Pawtab(ntypat)
 type(Paw_an_type),intent(inout) :: Paw_an(:)

!Local variables-------------------------------
 integer :: iat,itypat,lm_size,v_size

! *************************************************************************

 DBG_ENTER("COLL")

 !@Paw_an_type

 !allocate(Paw_an(natom))
 !call nullify_paw_an(Paw_an)

 do iat=1,natom
  itypat=typat(iat)
  lm_size                =Pawtab(itypat)%lcut_size**2
  Paw_an(iat)%angl_size  =Pawang%angl_size
  Paw_an(iat)%cplex      =cplex
  Paw_an(iat)%lm_size    =lm_size
  Paw_an(iat)%mesh_size  =Pawtab(itypat)%mesh_size
  Paw_an(iat)%nkxc1      =nkxc1
  Paw_an(iat)%nspden     =nspden

  ! === Non-zero LM-moments of "one-center" densities/potentials ===
  ! * Filled in pawdenpot.
  allocate(Paw_an(iat)%lmselect(lm_size))

  v_size=Paw_an(iat)%lm_size ; if (pawxcdev==0) v_size=Paw_an(iat)%angl_size

 ! === XC potential inside the sphere ===
 ! * LM-moments of potential if pawxcdev/=0
 ! * (theta,phi) values of potential if pawxcdev=0
  Paw_an(iat)%has_vxc=0
  if (PRESENT(has_vxc)) then
   if (has_vxc>0) then
    Paw_an(iat)%has_vxc=1
    allocate(Paw_an(iat)%vxc1 (cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
    allocate(Paw_an(iat)%vxct1(cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
   end if
  end if

  ! ==========================
  ! === Optional arguments ===
  ! ==========================

  ! * XC potential inside PAW spheres generated by valence electrons.
  Paw_an(iat)%has_vxcval=0
  if (PRESENT(has_vxcval)) then
   if (has_vxcval>0) then
    Paw_an(iat)%has_vxcval=1
    allocate(Paw_an(iat)%vxc1_val (cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
    allocate(Paw_an(iat)%vxct1_val(cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
   end if
  end if

  ! * XC potential for local exact exchange inside the sphere.
  if (Pawtab(itypat)%useexexch>0) then
   allocate(Paw_an(iat)%vxc_ex(cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
  end if

  ! * Hartree potential LM-moments inside the sphere.
  Paw_an(iat)%has_vhartree=0
  if (PRESENT(has_vhartree)) then
   if (has_vhartree>0) then
    Paw_an(iat)%has_vhartree=1
! FIXME what about vht1?
!MG This is the coding PRESENTLY used in pawdenpot but the commented code should be the correct one
    allocate(Paw_an(iat)%vh1(cplex*Pawtab(itypat)%mesh_size,1,1))
    !£allocate(Paw_an(iat)%vh1 (cplex*Pawtab(itypat)%mesh_size,lm_size,nspden))
    !£allocate(Paw_an(iat)%vht1(cplex*Pawtab(itypat)%mesh_size,lm_size,nspden))
   end if
  end if

  ! xc kernels inside the sphere.
  Paw_an(iat)%has_kxc=0
  if (PRESENT(has_kxc)) then
   if (has_kxc>0) then
    Paw_an(iat)%has_kxc=1
    allocate(Paw_an(iat)%kxc1 (cplex*Pawtab(itypat)%mesh_size,v_size,nkxc1))
    allocate(Paw_an(iat)%kxct1(cplex*Pawtab(itypat)%mesh_size,v_size,nkxc1))
   end if
  end if

 end do !iat

 DBG_EXIT("COLL")

end subroutine init_paw_an
!!***

!!****f* m_paw_toolbox/print_pawtab
!! NAME
!! print_pawtab
!!
!! FUNCTION
!!
!! INPUTS
!!  Pawtab<pawtab_type> Only for PAW, TABulated data initialized at start
!!
!! OUTPUT
!!  Only writing  
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine print_pawtab(Pawtab,unitno,prtvol,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_66_paw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: prtvol,unitno
 character(len=4),intent(in),optional :: mode_paral
!arrays
 type(Pawtab_type) :: Pawtab(:)

!Local variables-------------------------------
!scalars
 integer :: ierr,ityp,ntypat,unt,verb
 character(len=4) :: mode
 character(len=500) :: msg

! *************************************************************************

 verb=0       ; if (PRESENT(prtvol))     verb=prtvol                         
 unt =std_out ; if (PRESENT(unitno))     unt =unitno     
 mode='COLL'  ; if (PRESENT(mode_paral)) mode=mode_paral

 ntypat=SIZE(Pawtab(:))

 write(*,*) ' ==================================== '
 write(*,*) ' ==== Info on PAW TABulated data ==== '
 write(*,*) ' ==================================== '

 do ityp=1,ntypat 
  ! Print out integer values (dimensions)
  write(*,*)'                                 '
  write(*,*)'  ****************************** ' 
  write(*,*)'  **** Atom type ',ityp,' **** ' 
  write(*,*)'  ****************************** ' 
  write(*,*)'  Number of (n,l) elements ....................... ',Pawtab(ityp)%basis_size
  write(*,*)'  Number of (l,m,n) elements ..................... ',Pawtab(ityp)%lmn_size
  write(*,*)'  Number of (i,j) elements (packed form) ......... ',Pawtab(ityp)%ij_size  
  write(*,*)'  Max L+1 leading to non-zero Gaunt .............. ',Pawtab(ityp)%l_size
  write(*,*)'  Max L+1 leading to non-zero Gaunt (pawlcutd) ... ',Pawtab(ityp)%lcut_size
  write(*,*)'  lmn2_size ...................................... ',Pawtab(ityp)%lmn2_size
  write(*,*)'  lmnmix_sz ...................................... ',Pawtab(ityp)%lmnmix_sz 
  write(*,*)'  Size of radial mesh ............................ ',Pawtab(ityp)%mesh_size 
  write(*,*)'  No of Q-points for tcorespl and tvalespl ....... ',Pawtab(ityp)%mqgrid 
  write(*,*)'  Radial shape function type ..................... ',Pawtab(ityp)%shape_type
  write(*,*)'  shape_lambda ................................... ',Pawtab(ityp)%shape_lambda 
  write(*,*)'  Use pseudized core density ..................... ',Pawtab(ityp)%usetcore
  write(*,*)'  Use pseudized valence density .................. ',Pawtab(ityp)%usetvale 
  write(*,*)'  Option for Vloc (0 Blochl, 1 Kresse) ........... ',Pawtab(ityp)%vlocopt  
  write(*,*)'  Use LDA+U ...................................... ',Pawtab(ityp)%usepawu
  if (Pawtab(ityp)%usepawu/=0) then
   write(*,*)'  L on which U is applied ........................ ',Pawtab(ityp)%lpawu 
  end if
  write(*,*)'  Use Local Exact exchange ....................... ',Pawtab(ityp)%useexexch
  if (Pawtab(ityp)%useexexch/=0) then
   write(*,*)'  L on which local exact-exchange is applied ..... ',Pawtab(ityp)%lexexch 
  end if
  if (Pawtab(ityp)%usepawu/=0.or.Pawtab(ityp)%useexexch/=0) then
   write(*,*)'  Number of (i,j) elements for PAW+U or EXX ..... ',Pawtab(ityp)%ij_proj
   write(*,*)'  Number of projectors on which U or EXX acts .... ',Pawtab(ityp)%nproju 
  end if
  write(*,*)'  Has nabla ...................................... ',Pawtab(ityp)%has_nabla
  !
  ! Real (real(dp)) scalars
  write(*,*)'  1/q d(tNcore(q))/dq for q=0 .....................',Pawtab(ityp)%dncdq0
  write(*,*)'  1/q d(tNvale(q))/dq for q=0 .....................',Pawtab(ityp)%dnvdq0
  write(*,*)'  XC energy for the core density ..................',Pawtab(ityp)%exccore
  write(*,*)'  Radius of the PAW sphere ........................',Pawtab(ityp)%rpaw
  write(*,*)'  Compensation charge radius (if >rshp, g(r)=0) ...',Pawtab(ityp)%rshp !(if r>rshp, g(r)=zero)
  if (Pawtab(ityp)%shape_type==2) then
   write(*,*)'  Sigma parameter in gaussian shape function ......',Pawtab(ityp)%shape_sigma !(shape_type=2)
  end if
  if (Pawtab(ityp)%usepawu/=0) then
   write(*,*)'  Value of the U parameter [eV] ...................',Pawtab(ityp)%upawu*Ha_eV
   write(*,*)'  Value of the J parameter [eV] ...................',Pawtab(ityp)%jpawu*Ha_eV
  end if
 end do
 !
 ! The other (huge) arrays are not reported..
 
end subroutine print_pawtab
!!***

!!****f* ABINIT/print_paw_ij
!! NAME
!! print_paw_ij
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! NOTES
!!  The implementation of the routine is not yet completed.   
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine print_paw_ij(Paw_ij,pawprtvol,unit,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: pawprtvol
 integer,intent(in),optional :: unit
 character(len=4),intent(in),optional :: mode_paral
!arrays
 type(Paw_ij_type),intent(in) :: Paw_ij(:)

!Local variables-------------------------------
 character(len=7),parameter :: dspin(6)=(/"up     ","down   ","up-up  ","dwn-dwn","up-dwn ","dwn-up "/)
!scalars
 integer :: cplex_dij,iatom,idij,lmn2_size,lmn_size,natom,nspden,nsploop,nsppol,unt
 integer :: opt_sym,tmp_cplex_dij
 character(len=4) :: mode
 character(len=500) :: msg
!arrays
 integer,allocatable :: idum(:)
 real(dp),pointer :: dij2p(:)

! *************************************************************************

 DBG_ENTER("COLL")

 unt    =std_out ; if (PRESENT(unit)) unt=unit
 !verbose=0      ; if (PRESENT(prtvol)) verbose=prtvol
 mode   ='COLL'  ; if (PRESENT(mode_paral)) mode=mode_paral

 ! TODO: to be consistent, unt and mode should be passed to print_ij
 ! moreover the pointers should be nullified when paw_ij is initialized

 natom  =SIZE(Paw_ij)
 nsppol =Paw_ij(1)%nsppol
 nspden =Paw_ij(1)%nspden
 nsploop=nsppol ; if (Paw_ij(1)%ndij==4) nsploop=4

 do iatom=1,natom

  lmn_size  = Paw_ij(iatom)%lmn_size
  lmn2_size = Paw_ij(iatom)%lmn2_size
  cplex_dij = Paw_ij(iatom)%cplex_dij

  ! ====================================
  ! === Loop over density components ===
  ! ====================================
  do idij=1,nsploop

   ! * Print title.
   if (ABS(pawprtvol)>=1) then
    if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
     if (nspden==2.and.nsppol==1) then
      write(msg,'(2a,i3,3a)')ch10,&
&      ' >>>>>>>>>> Atom ',iatom,':',ch10,&
&      ' (antiferromagnetism case: only one spin component)'
     else
      write(msg,'(2a,i3,3a)') ch10,&
&      ' >>>>>>>>>> Atom ',iatom,' (component ',TRIM(dspin(idij+2*(nsploop/4))),'):'
     end if
     call wrtout(unt,msg,mode)
    end if
   end if

   !if (abs(pawprtvol)>=1) then
   ! if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
   !  write(msg, '(a)') '   ************ Dij atomic (Dij0) ***********'
   !  call wrtout(unt,msg,mode)
   !  call print_ij(pawtab(itypat)%dij0,lmn2_size,1,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1)
   ! end if
   !end if

   if (abs(pawprtvol)>=1) then
    if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
     write(msg, '(a)')'   ************** Dij Hartree ***************'
     call wrtout(unt,msg,mode)
     call print_ij(Paw_ij(iatom)%dijhartree,lmn2_size,1,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1)
    end if
   end if

   if (associated(Paw_ij(iatom)%dijxc)) then
    if ((abs(pawprtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
      write(msg,'(a)')'   ****************** Dij_xc + Dijhat_xc ****************'
      call wrtout(unt,msg,mode)
      if (idij<=nsppol.or.idij==2) then
       opt_sym=2 ; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:cplex_dij,idij)
      else
       opt_sym=1 ; tmp_cplex_dij=cplex_dij
       dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:1,idij)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   if (associated(Paw_ij(iatom)%dijxc_val)) then
    if ((abs(pawprtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
      write(msg,'(a)')'   ****************** Dij_xcval ****************'
      call wrtout(unt,msg,mode)
      if (idij<=nsppol.or.idij==2) then
       opt_sym=2 ; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijxc_val(1:cplex_dij*lmn2_size:cplex_dij,idij)
      else
       opt_sym=1 ; tmp_cplex_dij=cplex_dij
       dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:1,idij)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   if (associated(Paw_ij(iatom)%dijhat)) then
    if ((abs(pawprtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
      write(msg,'(a)')'   ************* Dij_hat (Veff_ij) **********'
      call wrtout(unt,msg,mode)
      if (idij<=nsppol.or.idij==2) then
       opt_sym=2 ; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijhat(1:cplex_dij*lmn2_size:cplex_dij,idij)
       !call print_ij(dijhat(1:lmn2_size),lmn2_size,1,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1)
      else
        opt_sym=1 ; tmp_cplex_dij=cplex_dij
        dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:1,idij)
       !call print_ij(dijhat,lmn2_size,cplex_dij,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1,opt_sym=1)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   if (associated(Paw_ij(iatom)%dijso)) then
    if (abs(pawprtvol)>=1) then
     if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
      write(msg,'(a)')'   ************** Dij SpinOrbit ************'
      call wrtout(unt,msg,mode)
      dij2p =>  Paw_ij(iatom)%dijso(:,idij)
      call print_ij(dij2p,lmn2_size,cplex_dij,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1,opt_sym=3)
     end if
    end if
   end if

   if (associated(Paw_ij(iatom)%dijU)) then
    if ((abs(pawprtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.pawprtvol<0) then
      write(msg,'(a)')'   ************* Dij_LDA+U (dijpawu) **********'
      call wrtout(unt,msg,mode)
      if (idij<=nsppol.or.idij==2) then
       opt_sym=2 ; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijU(1:cplex_dij*lmn_size:cplex_dij,idij) 
       !call print_ij(dijpawu(1:lmn2_size),lmn2_size,1,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1)
      else
       opt_sym=1 ; tmp_cplex_dij=cplex_dij
       dij2p => Paw_ij(iatom)%dijU(1:cplex_dij*lmn2_size:1,idij)
       !call print_ij(dijpawu,lmn2_size,cplex_dij,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1,opt_sym=1)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   !TODO Dij_Exact_Exchange is not printed because there is no entry in the objects

  end do !idij
 end do !iat

 write(msg,'(a)')' '
 call wrtout(unt,msg,mode)

 DBG_ENTER("COLL")

end subroutine print_paw_ij
!!***

!!****f* m_paw_toolbox/pawfgrtab_nullify_
!! NAME
!!  pawfgrtab_nullify_ {PRIVATE}
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_paw_toolbox
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_nullify_(Pawfgrt)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Pawfgrtab_type),intent(inout) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom

! *************************************************************************

 !@Pawfgrtab_type
 natom=SIZE(Pawfgrt)
 do iat=1,natom
  nullify(Pawfgrt(iat)%ifftsph)
  nullify(Pawfgrt(iat)%gylm   )
  nullify(Pawfgrt(iat)%gylmgr )
  nullify(Pawfgrt(iat)%gylmgr2)
  nullify(Pawfgrt(iat)%nhatfr )
  nullify(Pawfgrt(iat)%rfgd   )
  nullify(Pawfgrt(iat)%vlocgr )
 end do

end subroutine pawfgrtab_nullify_
!!***

!!****f* m_paw_toolbox/pawfgrtab_free
!! NAME
!! pawfgrtab_free
!!
!! FUNCTION
!!  Free all dynamic memory stored in the datatype.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_free(Pawfgrt)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!arrays
 type(Pawfgrtab_type),intent(inout) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom

! *************************************************************************

 DBG_ENTER("COLL")

 !@Pawfgrtab_type
 natom=SIZE(Pawfgrt)
 do iat=1,natom
  if (associated(Pawfgrt(iat)%ifftsph)) deallocate(Pawfgrt(iat)%ifftsph)
  if (associated(Pawfgrt(iat)%gylm   )) deallocate(Pawfgrt(iat)%gylm   )
  if (associated(Pawfgrt(iat)%gylmgr )) deallocate(Pawfgrt(iat)%gylmgr )
  if (associated(Pawfgrt(iat)%gylmgr2)) deallocate(Pawfgrt(iat)%gylmgr2)
  if (associated(Pawfgrt(iat)%nhatfr )) deallocate(Pawfgrt(iat)%nhatfr )
  if (associated(Pawfgrt(iat)%rfgd   )) deallocate(Pawfgrt(iat)%rfgd   )
  if (associated(Pawfgrt(iat)%vlocgr )) deallocate(Pawfgrt(iat)%vlocgr )
 end do

 DBG_EXIT("COLL")

end subroutine pawfgrtab_free
!!***

!!****f* m_paw_toolbox/pawfgrtab_init
!! NAME
!! pawfgrtab_init
!!
!! FUNCTION
!!  Initialize a pawfgrtab datatype
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_init(Pawfgrt,cplex,l_size_atm,nspden)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,nspden
!arrays
 integer,intent(in) :: l_size_atm(:)
 type(Pawfgrtab_type),intent(out) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom
 character(len=500) :: msg

! *************************************************************************

 natom=SIZE(Pawfgrt)
 if (natom/=SIZE(l_size_atm)) then 
  msg='Sizes of assumed shape arrays do not match' 
  MSG_BUG(msg)
 end if

 !@Pawfgrtab_type
 call pawfgrtab_nullify_(Pawfgrt) 

 do iat=1,natom
  Pawfgrt(iat)%cplex             = cplex
  Pawfgrt(iat)%nspden            = nspden
  Pawfgrt(iat)%l_size            = l_size_atm(iat)
  Pawfgrt(iat)%nfgd              = 0  ; allocate(Pawfgrt(iat)%ifftsph(0)    )
  Pawfgrt(iat)%gylm_allocated    = 0  ; allocate(Pawfgrt(iat)%gylm(0,0)     )
  Pawfgrt(iat)%gylmgr_allocated  = 0  ; allocate(Pawfgrt(iat)%gylmgr(0,0,0) )
  Pawfgrt(iat)%gylmgr2_allocated = 0  ; allocate(Pawfgrt(iat)%gylmgr2(0,0,0))
  Pawfgrt(iat)%nhatfr_allocated  = 0  ; allocate(Pawfgrt(iat)%nhatfr(0,0)   )
  Pawfgrt(iat)%rfgd_allocated    = 0  ; allocate(Pawfgrt(iat)%rfgd(0,0)     )
  Pawfgrt(iat)%vlocgr_allocated  = 0  ; allocate(Pawfgrt(iat)%vlocgr(0,0)   )
 end do

end subroutine pawfgrtab_init
!!***

!!****f* m_paw_toolbox/pawfgrtab_print
!! NAME
!! pawfgrtab_print
!!
!! FUNCTION
!!  Reports basic info on the pawfgrtab datatype.
!!
!! INPUTS
!! Pawfgrt<pawfgrtab_type>=The datatype to be printed 
!! [mode_paral]=either "COLL" or "PERS", "COLL" if None.
!! [unitno]=Unit number for output, std_out if None.
!! [prtvol]=Verbosity level, lowest if None.
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_print(Pawfgrt,unitno,prtvol,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: prtvol,unitno
 character(len=4),intent(in),optional :: mode_paral
!arrays
 type(Pawfgrtab_type),intent(inout) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom,unt,verb
 character(len=4) :: mode
 character(len=500) :: msg

! *************************************************************************

 verb=0      ; if (PRESENT(prtvol))     verb=prtvol
 unt=std_out ; if (PRESENT(unitno))     unt=unitno
 mode='COLL' ; if (PRESENT(mode_paral)) mode=mode_paral

 natom=SIZE(Pawfgrt)

 write(msg,'(3a)')ch10,' === Content of the pawfgrtab datatype === ',ch10
 call wrtout(unt,msg,mode)
 do iat=1,natom
  write(msg,'(3(2a,i5))')ch10,&
&  ' > For atom number : ',iat,ch10,&
&  '    1+ Max l in Gaunt coefficients ',Pawfgrt(iat)%l_size,ch10,&
&  '    Number of fine FFT points in PAW sphere ',Pawfgrt(iat)%nfgd               
  call wrtout(unt,msg,mode)

  if (verb>=3) then 
   write(msg,'(6(2a,i2))')ch10,&
&   '    rfgd_allocated    : ',Pawfgrt(iat)%rfgd_allocated,ch10,&
&   '    gylm_allocated    : ',Pawfgrt(iat)%gylm_allocated,ch10,&   
&   '    gylmgr_allocated  : ',Pawfgrt(iat)%gylmgr_allocated,ch10,& 
&   '    gylmgr2_allocated : ',Pawfgrt(iat)%gylmgr2_allocated,ch10,&
&   '    nhatgr_allocated  : ',Pawfgrt(iat)%nhatfr_allocated,ch10,&
&   '    vlocgr_allocated  : ',Pawfgrt(iat)%vlocgr_allocated 
   call wrtout(unt,msg,mode)
  end if

! These huge arrays are not printed out!
! Pawfgrt(iat)%ifftsph
! Pawfgrt(iat)%rfgd
! Pawfgrt(iat)%gylm
! Pawfgrt(iat)%gylmgr
! Pawfgrt(iat)%gylmgr2
! Pawfgrt(ia)%nhatfr
! Pawfgrt(iat)%vlocgr
 end do

end subroutine pawfgrtab_print
!!***

!!****f* m_paw_toolbox/pawfgrtab_nullify
!! NAME
!! pawfgrtab_nullify
!!
!! FUNCTION
!!  Nullify all pointers in the Pawfgrtab_type
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_nullify(Pawfgrt)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Pawfgrtab_type),intent(inout) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom

! *************************************************************************

 !@Pawfgrtab_type
 natom=SIZE(Pawfgrt)
 do iat=1,natom
  nullify(Pawfgrt(iat)%ifftsph)
  nullify(Pawfgrt(iat)%gylm   )
  nullify(Pawfgrt(iat)%gylmgr )
  nullify(Pawfgrt(iat)%gylmgr2)
  nullify(Pawfgrt(iat)%nhatfr )
  nullify(Pawfgrt(iat)%rfgd   )
  nullify(Pawfgrt(iat)%vlocgr )
 end do

end subroutine pawfgrtab_nullify
!!***

!!****f* m_paw_toolbox/reset_paw_an_flags
!! NAME
!! reset_paw_an_flags
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!  See side effects
!!
!! SIDE EFFECTS
!!  All "has" flags in the Paw_an_type set to 0.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine reset_paw_an_flags(Paw_an_flags)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_flags_type),intent(inout) :: Paw_an_flags

! *************************************************************************

 ! @Paw_an_flags_type
 Paw_an_flags%has_kxc     =0
 Paw_an_flags%has_vhartree=0
 Paw_an_flags%has_vxc     =0
 Paw_an_flags%has_vxcval  =0

end subroutine reset_paw_an_flags
!!***

!!****f* m_paw_toolbox/copy_paw_an_flags
!! NAME
!! copy_paw_an_flags
!!
!! FUNCTION
!!  Copy a Paw_an_flags_type
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine copy_paw_an_flags(an_flags_in, an_flags_out)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_flags_type),intent(in)    :: an_flags_in
 type(Paw_an_flags_type),intent(inout) :: an_flags_out

! *************************************************************************

 ! @Paw_an_flags_type
 an_flags_out%has_kxc      = an_flags_in%has_kxc     
 an_flags_out%has_vhartree = an_flags_in%has_vhartree
 an_flags_out%has_vxc      = an_flags_in%has_vxc     
 an_flags_out%has_vxcval   = an_flags_in%has_vxcval  

end subroutine copy_paw_an_flags 

END MODULE m_paw_toolbox
!!***
