!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_bz_mesh
!! NAME
!!  m_bz_mesh
!!
!! FUNCTION
!!  This module provides the definition of the BZ_mesh_type structure gathering information 
!!  on the sampling of the Brillouin zone. It also contains useful tools to operate on k-points.
!!
!! COPYRIGHT
!! Copyright (C) 2008 ABINIT group (MG, GMR, VO, LR, RWG, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_bz_mesh

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors 

 use m_gwdefs,        only : GW_TOLQ, GW_Q0_DEFAULT !TODO remove GW_Q0_DEFAULT, GW_TOLQ should be local

 implicit none

 private  

 public ::                &
&  InitKmesh,             &  ! Main creation method.
&  NullifyBzMesh,         &  ! Nullify all pointers in BZ_mesh_type.
&  destroy_bz_mesh_type,  &  ! Free the structure.
&  print_bz_mesh,         &  ! Printout of basic info on the object.
&  get_BZ_item,           &  ! Get point in the  BZ as well as useful quantities.
&  get_IBZ_item,          &  ! Get point in the IBZ as well as useful quantities. 
&  get_BZ_diff,           &  ! Get the difference k1-k2 in the BZ (if any).
&  isamek,                &  ! Check whether two points are equal within an umklapp G0.
&  isequalk,              &  ! Check whether two points are equal within an umklapp G0 (does not report G0)
&  has_BZ_item,           &  ! Chesk if a point belongs to the BZ mesh.
&  has_IBZ_item,          &  ! Check if a point is in the IBZ
&  make_mesh,             &  ! Initialize the mesh starting from kptrlatt and shift (WARNING still under development)
&  findk,                 &  ! TODO to be removed
&  identk,                &  ! Find the BZ starting from the irreducible k-points.
&  get_ng0sh,             &  ! Calculate the smallest box in RSpace able to treat all possible umlapp processes.
&  make_path,             &  ! Construct a normalized path.
&  setup_qmesh,           &  ! Initialize the Q-mesh (WARNING, DO NOT USE IT. it is a temporary routine, it will be removed)
&  find_Qmesh,            &  ! Find the Q-mesh defined as the set of all possible k1-k2 differences.
&  findnq,                &  ! Helper routine returning the number of q-points. (TODO should be private)
&  findq,                 &  ! Helper routine returning the list of q-points. (TODO should be private)
&  findqg0                   ! Identify q + G0 = k1 - k2

CONTAINS  !=========================================================================================================================
!!***

!!****f* m_bz_mesh/InitKmesh
!! NAME
!! InitKmesh
!!
!! FUNCTION
!!  Initialize and construct a bz_mesh_type datatype
!!  gathering information on the mesh in the Brilloin zone. 
!!
!! INPUTS
!!  nkibz=number of irreducible k-points
!!  prtvol=verbosity level
!!  timrev=1 if time-reversal cannot be used, 2 otherwise
!!  kibz(3,nkibz)=irreducible k-points
!!  Cryst<Crystal_structure> = Info on unit cell and its symmetries
!!     %nsym=number of symmetry operations
!!     %symrec(3,3,nsym)=symmetry operations in reciprocal space
!!     %tnons(3,nsym)=fractional translations
!!
!! OUTPUT
!!  Kmesh<bz_mesh_type>=Datatype gathering information on the k point sampling.
!!
!! NOTES
!!
!! PARENTS
!!      m_electrons,m_gwannier,mlwfovlp_qp,mrgscr,setup_screening,setup_sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine InitKmesh(nkibz,kibz,Cryst,Kmesh,prtvol)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkibz 
 integer,optional,intent(in) :: prtvol
 type(BZ_mesh_type),intent(inout) :: Kmesh
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 real(dp),intent(in) :: kibz(3,nkibz)

!Local variables-------------------------------
!scalars
 integer :: ierr,ik_bz,ik_ibz,isym,nkbz,nkbzX,nsym,timrev,my_prtvol
 real(dp) :: shift1,shift2,shift3
 logical :: ltest,use_antiferro,do_wrap
 character(len=500) :: msg
!arrays
 integer,allocatable :: ktab(:),ktabi(:),ktabo(:),ktabr(:,:)
 integer,pointer :: symafm(:),symrec(:,:,:)
 real(dp) :: rm1t(3)
 real(dp),allocatable :: kbz(:,:),kbz_wrap(:,:),wtk(:)
 real(dp),pointer :: tnons(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 my_prtvol=0 ; if (PRESENT(prtvol)) my_prtvol=prtvol

 call NullifyBzMesh(Kmesh)

 ! === Initial tests on input arguments ===
 ltest=(Cryst%timrev==1.or.Cryst%timrev==2)
 write(msg,'(a,i4)')'Wrong value for timrev= ',Cryst%timrev
 call assert(ltest,msg,__FILE__,__LINE__)

 nsym          = Cryst%nsym
 timrev        = Cryst%timrev
 use_antiferro = Cryst%use_antiferro

 symrec => Cryst%symrec
 symafm => Cryst%symafm
 tnons  => Cryst%tnons
 !
 ! === Find BZ from IBZ and fill tables ===
 nkbzX=nkibz*nsym*timrev ! Maximum possible number
 allocate(kbz(3,nkbzX),wtk(nkibz),ktab(nkbzX),ktabi(nkbzX),ktabo(nkbzX))

 call identk(kibz,nkibz,nkbzX,nsym,timrev,symrec,symafm,use_antiferro,kbz,ktab,ktabi,ktabo,nkbz,wtk,my_prtvol)

 do_wrap=.FALSE.
 if (do_wrap) then ! Wrap the BZ points in the interval ]-1/2,1/2]
  allocate(kbz_wrap(3,nkbz))
  do ik_bz=1,nkbz
   call canon9(kbz(1,ik_bz),kbz_wrap(1,ik_bz),shift1)
   call canon9(kbz(2,ik_bz),kbz_wrap(2,ik_bz),shift2)
   call canon9(kbz(3,ik_bz),kbz_wrap(3,ik_bz),shift3)
  end do
  deallocate(kbz_wrap)
 end if
 !
 ! ================================================================
 ! ==== Create data structure to store information on k-points ====
 ! ================================================================
 !
 ! * Dimensions.
 Kmesh%nbz   =nkbz      ! Number of points in the full BZ
 Kmesh%nibz  =nkibz     ! Number of points in the IBZ
 Kmesh%nsym  =nsym      ! Number of operations
 Kmesh%timrev=timrev    ! 2 if time-reversal is used, 1 otherwise

 Kmesh%ntetra_irr=0     ! no tetrahedrons for the moment
 !
 ! * Arrays.
 Kmesh%gmet   = Cryst%gmet
 Kmesh%gprimd = Cryst%gprimd

 allocate(Kmesh%bz (3,nkbz))  ; Kmesh%bz   =  kbz(:,1:nkbz )  ! Red. coordinates of points in full BZ.
 allocate(Kmesh%ibz(3,nkibz)) ; Kmesh%ibz  = kibz(:,1:nkibz)  ! Red. coordinates of points in IBZ.

 allocate(Kmesh%tab (nkbz))   ; Kmesh%tab  = ktab (1:nkbz)    ! Index of the irred. point in the array IBZ.
 allocate(Kmesh%tabi(nkbz))   ; Kmesh%tabi = ktabi(1:nkbz)    !-1 if time reversal must be used to obtain this point,
 allocate(Kmesh%tabo(nkbz))   ; Kmesh%tabo = ktabo(1:nkbz)    ! Symm. operation that rotates k_IBZ onto \pm k_BZ
                                                              ! (depending on tabi)
 allocate(Kmesh%wt(nkibz))    ; Kmesh%wt(:)= wtk(1:nkibz)     ! Weight for each k_IBZ

 allocate(Kmesh%rottbm1(nkbz,timrev,nsym))                    ! Index of rotated point (IS)^{-1} kbz
 allocate(Kmesh%rottb  (nkbz,timrev,nsym))                    ! Index of rotated point IS kbz where I is either the
                                                              ! identity or time-reversal

 call setup_k_rotation(nsym,symrec,timrev,nkbz,Kmesh%bz,Kmesh%rottb,Kmesh%rottbm1)

 allocate(Kmesh%tabp(nkbz)) ! Phase factors for non-symmorphic operations $e{-i2\pik_BZ.\tau}$
 do ik_bz=1,nkbz
  isym  =Kmesh%tabo(ik_bz) 
  ik_ibz=Kmesh%tab (ik_bz)
  rm1t=MATMUL(TRANSPOSE(symrec(:,:,isym)),tnons(:,isym))
  Kmesh%tabp(ik_bz)=EXP(-(0.,1.)*two_pi*DOT_PRODUCT(kibz(:,ik_ibz),rm1t))
 end do

 deallocate(kbz,wtk,ktab,ktabi,ktabo)

 call print_bz_mesh(Kmesh,prtvol=my_prtvol)

 DBG_EXIT("COLL")

end subroutine InitKmesh
!!***

!!****f* m_bz_mesh/my_reset_BzMesh
!! NAME
!! my_reset_BzMesh
!!
!! FUNCTION
!! Reset dimensions in a BZ_mesh_type. [PRIVATE, @@]
!!
!! INPUTS
!! Kmesh<BZ_mesh_type>= The datatype whose dimensions have to be set to zero
!!
!! SIDE EFFECTS
!! All dimension reset to 0.
!!
!! PARENTS
!!      m_bz_mesh
!!
!! CHILDREN
!!
!! SOURCE

subroutine my_reset_BzMesh(Kmesh)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 type(BZ_mesh_type),intent(inout) :: Kmesh

! *********************************************************************

!dimensions
 Kmesh%nshift     = 0 
 Kmesh%ntetra_irr = 0
 Kmesh%nbz        = 0 
 Kmesh%nibz       = 0 

end subroutine my_reset_BzMesh
!!***

!!****f* m_bz_mesh/NullifyBzMesh
!! NAME
!! NullifyBzMesh
!!
!! FUNCTION
!! Nullify the pointers in a BZ_mesh_type. [PRIVATE]
!!
!! INPUTS
!! Kmesh<BZ_mesh_type>= The datatype whose pointers have to be nullified.
!!
!! SIDE EFFECTS
!! All pointers set to null().
!!
!! PARENTS
!!      m_bz_mesh,m_gwannier
!!
!! CHILDREN
!!
!! SOURCE

subroutine NullifyBzMesh(Kmesh)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(BZ_mesh_type),intent(inout) :: Kmesh

! *********************************************************************

! zero dimensions
 call my_reset_BzMesh(Kmesh)

!integer
 nullify(Kmesh%rottb     )
 nullify(Kmesh%rottbm1   )
 nullify(Kmesh%tab       )
 nullify(Kmesh%tabi      )
 nullify(Kmesh%tabo      )
 nullify(Kmesh%tetra_full)
 nullify(Kmesh%tetra_mult)
 nullify(Kmesh%tetra_wrap)

!real
 nullify(Kmesh%bz   )
 nullify(Kmesh%ibz  )
 nullify(Kmesh%shift)
 nullify(Kmesh%wt   )

!complex
 nullify(Kmesh%tabp)

end subroutine NullifyBzMesh
!!***

!!****f* m_bz_mesh/destroy_bz_mesh_type
!! NAME
!! destroy_bz_mesh_type
!!
!! FUNCTION
!! Deallocate all dynamics entities present in a BZ_mesh_type structure.
!!
!! INPUTS
!! Kmesh<BZ_mesh_type>=The datatype to be freed.
!!
!! SIDE EFFECTS
!! All allocated memory is released. 
!!
!! PARENTS
!!      m_electrons,m_gwannier,mlwfovlp_qp,mrgscr,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_bz_mesh_type(Kmesh)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(BZ_mesh_type),intent(inout) :: Kmesh

!Local variables-------------------------------
!scalars
 integer :: istat

! *********************************************************************

 DBG_ENTER("COLL")

!integer
 if (associated(Kmesh%rottb     )) deallocate(Kmesh%rottb     )
 if (associated(Kmesh%rottbm1   )) deallocate(Kmesh%rottbm1   )
 if (associated(Kmesh%tab       )) deallocate(Kmesh%tab       )
 if (associated(Kmesh%tabi      )) deallocate(Kmesh%tabi      )
 if (associated(Kmesh%tabo      )) deallocate(Kmesh%tabo      )
 if (associated(Kmesh%tetra_full)) deallocate(Kmesh%tetra_full)
 if (associated(Kmesh%tetra_mult)) deallocate(Kmesh%tetra_mult)
 if (associated(Kmesh%tetra_wrap)) deallocate(Kmesh%tetra_wrap)

!real
 if (associated(Kmesh%ibz    )) deallocate(Kmesh%ibz  )
 if (associated(Kmesh%bz     )) deallocate(Kmesh%bz   )
 if (associated(Kmesh%shift  )) deallocate(Kmesh%shift)
 if (associated(Kmesh%wt     )) deallocate(Kmesh%wt   )

!complex
 if (associated(Kmesh%tabp)) deallocate(Kmesh%tabp)

!set dimensions to zero
 call my_reset_BzMesh(Kmesh)

 DBG_EXIT("COLL")

end subroutine destroy_bz_mesh_type
!!***

!!****f* m_bz_mesh/print_bz_mesh
!! NAME
!! print_bz_mesh
!!
!! FUNCTION
!! Print the content of a bz_mesh_type datatype
!!
!! INPUTS
!! Kmesh<bz_mesh_type>=the datatype to be printed
!! unit[optional]=the unit number for output 
!! prtvol[optional]=verbosity level
!! mode_paral[optional]=either "COLL" or "PERS"
!!
!! OUTPUT
!!  Only printing.
!!
!! PARENTS
!!      m_bz_mesh,setup_screening
!!
!! CHILDREN
!!
!! SOURCE

subroutine print_BZ_mesh(Kmesh,unit,prtvol,mode_paral)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: prtvol,unit
 character(len=4),intent(in),optional :: mode_paral
 type(BZ_mesh_type),intent(in) :: Kmesh

!Local variables-------------------------------
!scalars
 integer,parameter :: nmaxk=50
 integer :: ii,ik,unt,verbose
 character(len=100) :: fmt
 character(len=4) :: mode
 character(len=500) :: msg

! *************************************************************************

 unt    =std_out ; if (PRESENT(unit      )) unt    =unit
 verbose=0       ; if (PRESENT(prtvol    )) verbose=prtvol
 mode   ='COLL'  ; if (PRESENT(mode_paral)) mode   =mode_paral

 write(msg,'(2a,i5,3a)')ch10,&
& ' Number of points in the IBZ : ',Kmesh%nibz,ch10,&
& ' Reduced Coordinates and Weights : ',ch10
 call wrtout(unt,msg,mode)

 write(fmt,*)'(1x,i5,a,2x,3es16.8,3x,f11.5)'
 do ik=1,Kmesh%nibz
! Add tol8 for portability reasons.
  write(msg,fmt) ik,') ',(Kmesh%ibz(ii,ik),ii=1,3),Kmesh%wt(ik)+tol8
  call wrtout(unt,msg,mode)
 end do

 select case (Kmesh%timrev)

 case (1)
  write(msg,'(2a,i2,3a,i5,a)')ch10,&
&  ' together with ',Kmesh%nsym,' symmetry operations (time-reversal not used) ',ch10,&
&  ' yields ',Kmesh%nbz,' points in the Brillouin Zone (BZ) :'

 case (2) 
  write(msg,'(2a,i2,3a,i5,a)')ch10,&
&  ' together with ',Kmesh%nsym,' symmetry operations and time-reversal ',ch10,&
&  ' yields ',Kmesh%nbz,' points in the Brillouin Zone (BZ) :'

 case default
  write(msg,'(a,i3)')' Wrong value for timrev = ',Kmesh%timrev
  MSG_BUG(msg)
 end select

 call wrtout(unt,msg,mode)

 if (verbose>0) then
  write(fmt,*)'(1x,i5,a,2x,3es16.8)'
  do ik=1,Kmesh%nbz
   if (verbose==1 .and. ik>nmaxk) then
    write(msg,'(a)')' prtvol=1, do not print more points.'
    call wrtout(unt,msg,mode) ; EXIT
   end if
   write(msg,fmt)ik,') ',(Kmesh%bz(ii,ik),ii=1,3)
   call wrtout(unt,msg,mode)
  end do
 end if
 !
 ! === Additional printing ===
 if (verbose>=10) then
  write(msg,'(2a)')ch10,&
&  '                  Full point  ------->    Irred point -->            through:  Symrec  Time-Rev (1=No,-1=Yes) '
  call wrtout(unt,msg,mode)
  write(fmt,*)'(2x,i5,2x,2(3f6.4,2x),i3,2x,i2)'
  do ik=1,Kmesh%nbz
   write(msg,fmt)ik,Kmesh%bz(:,ik),Kmesh%ibz(:,Kmesh%tab(ik)),Kmesh%tabo(ik),Kmesh%tabi(ik)
   call wrtout(unt,msg,mode)
  end do
 end if

 write(msg,'(a)')ch10
 call wrtout(unt,msg,mode)

end subroutine print_BZ_mesh
!!***

!!****f* m_bz_mesh/setup_k_rotation
!! NAME
!! setup_k_rotation
!!
!! FUNCTION
!! Set up tables giving the correspondence btw a k-point and its rotated image. 
!!
!! INPUTS
!! kbz(3,nbz)=k-points in reduced coordinates.
!! timrev=2 if time-reversal can be used, 1 otherwise.
!! nsym=Number of symmetry operations
!! nbz=Number of k-points
!! symrec(3,3,nsym)=Symmetry operations in reciprocal space in reduced coordinates.
!!
!! OUTPUT
!! krottb(k,I,S)=Index of (IS) k in the array bz
!! krottbm1(k,I,S)=Index of IS^{-1} k
!!
!! PARENTS
!!      m_bz_mesh
!!
!! CHILDREN
!!
!! SOURCE

subroutine setup_k_rotation(nsym,symrec,timrev,nbz,kbz,krottb,krottbm1)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nbz,nsym,timrev
!arrays
 integer,intent(in) :: symrec(3,3,nsym)
 integer,intent(inout) :: krottb(nbz,timrev,nsym),krottbm1(nbz,timrev,nsym)
 real(dp),intent(in) :: kbz(3,nbz)

!Local variables ------------------------------
!scalars
 integer :: ik,ikp,isym,itim
 logical :: found,isok
 character(len=500) :: msg
!arrays
 integer :: G0(3)
 real(dp) :: kbase(3),krot(3)

!************************************************************************

 DBG_ENTER("COLL")
 !
 ! === Set up k-rotation tables ===
 ! * Use spatial inversion instead of time reversal whenever possible.
 isok=.TRUE.
 do ik=1,nbz
  kbase(:)=kbz(:,ik)
  do itim=1,timrev
   do isym=1,nsym
    krot(:)=(3-2*itim)*MATMUL(symrec(:,:,isym),kbase)
    found=.FALSE.
    do ikp=1,nbz
     if (isamek(krot,kbz(:,ikp),G0)) then
      found=.TRUE.
      krottb  (ik ,itim,isym)=ikp
      krottbm1(ikp,itim,isym)=ik
     end if
    end do

    if (.not.found) then
     isok=.FALSE.
     write(msg,'(2(a,i4),2x,2(3f12.6,2a),i3,a,i2)')&
&     ' Initial k-point ',ik,'/',nbz,kbase(:),ch10,&
&     ' Rotated k-point (not found) ',krot(:),ch10,&
&     ' Through sym. operation ',isym,' and itim ',itim
     MSG_WARNING(msg)
    end if

   end do
  end do
 end do

 if (.not.isok) then
  MSG_ERROR('k-mesh not closed')
 end if

 DBG_EXIT("COLL")

end subroutine setup_k_rotation
!!***

!!****f* m_bz_mesh/get_BZ_item
!! NAME
!! get_BZ_item
!!
!! FUNCTION
!! Given the index of a point in the full BZ, report useful information.
!!
!! INPUTS
!! ikbz=The index of the required point in the BZ
!! Kmesh<BZ_mesh_type>=Datatype gathering information on the k point sampling. see defs_datatypes.F90
!!
!! OUTPUT
!! kbz(3)=the k-point in reduced coordinated
!! isym=index of the symmetry required to rotate ik_ibz onto ik_bz
!! itim=2 is time-reversal has to be used, 1 otherwise
!! ik_ibz=the index of corresponding point in the IBZ
!! ph_mkbzt=the phase factor for non-symmorphic operations 
!!  i.e e^{-i 2 \pi k_IBZ \cdot R{^-1}t}=e{-i 2\pi k_BZ cdot t}
!!
!! PARENTS
!!      cchi0,cchi0q0,csigme,m_coulombian,m_dyson_solver,m_electrons,m_wfs
!!      paw_symcprj
!!
!! CHILDREN
!!
!! SOURCE

subroutine get_BZ_item(Kmesh,ik_bz,kbz,ik_ibz,isym,itim,ph_mkbzt)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_bz
 integer,intent(out) :: ik_ibz,isym,itim
 complex(gwpc),intent(out),optional :: ph_mkbzt
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 real(dp),intent(out) :: kbz(3)

!Local variables-------------------------------
!scalars
 character(len=500) :: msg

! *********************************************************************

 if (ik_bz>Kmesh%nbz.or.ik_bz<=0) then
  write(msg,'(a,i3)')' Wrong value for ik_bz: ',ik_bz
  MSG_BUG(msg)
 end if

 kbz(:)=Kmesh%bz(:,ik_bz)

 ik_ibz=Kmesh%tab(ik_bz)
 isym=Kmesh%tabo(ik_bz)
 itim=(3-Kmesh%tabi(ik_bz))/2
 if (PRESENT(ph_mkbzt)) ph_mkbzt=Kmesh%tabp(ik_bz)

end subroutine get_BZ_item
!!***

!!****f* m_bz_mesh/get_IBZ_item
!! NAME
!! get_IBZ_item
!!
!! FUNCTION
!! Report useful information on a k-point in the IBZ starting from its senquential index in %ibz. 
!!
!! INPUTS
!! ik_bz=The index of the required point in the IBZ
!! Kmesh<bz_mesh_type>=datatype gathering information on the k point sampling. see defs_datatypes.F90
!!
!! OUTPUT
!! kibz(3)=the k-point in reduced coordinated
!! wtk=the weight
!!
!! TODO
!!  Add mapping ibz2bz, ibz2star
!!
!! PARENTS
!!      paw_symcprj
!!
!! CHILDREN
!!
!! SOURCE

subroutine get_IBZ_item(Kmesh,ik_ibz,kibz,wtk)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz
 real(dp),intent(out) :: wtk
 type(bz_mesh_type),intent(in) :: Kmesh
!arrays
 real(dp),intent(out) :: kibz(3)

!Local variables-------------------------------
!scalars
 character(len=500) :: msg

! *********************************************************************

 if (ik_ibz>Kmesh%nibz.or.ik_ibz<=0) then
  write(msg,'(a,i3)')' wrong value for ik_ibz: ',ik_ibz
  MSG_BUG(msg)
 end if

 kibz=Kmesh%ibz(:,ik_ibz)
 wtk =Kmesh%wt(ik_ibz)

end subroutine get_IBZ_item
!!***

!!****f* m_bz_mesh/get_BZ_diff
!! NAME
!! get_BZ_diff
!!
!! FUNCTION
!! Given two points k1 and k2 where k1 belongs to the BZ, check if the difference
!! k1-k2 still belongs to the BZ reporting useful quantities
!!
!! INPUTS
!!  Kmesh<bz_mesh_type>=datatype gathering information on the k-mesh
!!  k1(3)=the first k-points (supposed to be in the BZ)
!!  k2(3)=the second point
!!
!! OUTPUT
!!  idiff_bz=the idex of k1-k2 in the BZ
!!  G0(3)=the umklapp G0 vector required to bring k1-k2 back to the BZ
!!  nfound= the number of points in the BZ that are equal to k1-k2 (should be 1 if everything is OK)
!!
!! PARENTS
!!      cchi0,m_electrons
!!
!! CHILDREN
!!
!! SOURCE

subroutine get_BZ_diff(Kmesh,k1,k2,idiff_bz,G0,nfound)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(out) :: idiff_bz,nfound
 type(bz_mesh_type),intent(in) :: Kmesh
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: k1(3),k2(3)

!Local variables-------------------------------
!scalars
 integer :: ikp
 character(len=500) :: msg
!arrays
 integer :: umklp(3)
 real(dp) :: kdiff(3),ktrial(3)

! *********************************************************************

 if (.not.has_BZ_item(Kmesh,k1,ikp,umklp)) then
  write(msg,'(a,3f12.6)')' first point must be in BZ ',k1(:)
  MSG_ERROR(msg)
 end if

 kdiff(:)=k1 - k2
 nfound  =0 
 idiff_bz=0
 !
 ! === Find p such k1-k2=p+G0 where p in the BZ ===
 do ikp=1,Kmesh%nbz
  ktrial=Kmesh%bz(:,ikp)
  if (isamek(kdiff,ktrial,umklp)) then
   idiff_bz=ikp
   G0=umklp
   nfound=nfound+1
  end if
 end do
 !
 ! === Check if p has not found of found more than once ===
 ! * For extremely dense meshes, tol1q in defs_basis might be too large!
 if (nfound/=1) then
  if (nfound==0) then
   write(msg,'(a)')' k1-k2-G0 not found in BZ '
   MSG_WARNING(msg)
  else
   write(msg,'(a,i3)')' Multiple k1-k2-G0 found in BZ, nfound= ',nfound
   MSG_WARNING(msg)
  end if
  write(msg,'(4a,3(a,3f12.6,a))')&
&  ' k1    = ',k1   ,ch10,&
&  ' k2    = ',k2   ,ch10,&
&  ' k1-k2 = ',kdiff,ch10
  MSG_WARNING(msg)
 end if

end subroutine get_BZ_diff
!!***

!!****f* m_bz_mesh/isamek
!! NAME
!! isamek
!!
!! FUNCTION
!! Test two k-points for equality. 
!! Return .TRUE. is they are equal within a reciprocal lattice vector G0.
!!
!! INPUTS
!!  k1(3),k2(3)=The two k points to be compared.
!!
!! OUTPUT
!! Return .TRUE. if they are the same within a RL vector,
!!        .FALSE. if they are different.
!! G0(3)=if .TRUE. G0(3) is the reciprocal lattice vector such as k1=k2+G0
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

logical function isamek(k1,k2,G0)

 use defs_basis
 use m_gwdefs, only : GW_TOLQ

 implicit none

!Arguments ------------------------------------
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: k1(3),k2(3)

!Local variables-------------------------------
!scalars
 real(dp) :: f,x

! *************************************************************************

 ! === Statement function definition: f is zero only if x is integer ===
 f(x)=ABS(x-NINT(x))

 isamek=.FALSE. ; G0(:)=0
 if (f(k1(1)-k2(1))<GW_TOLQ) then
  if (f(k1(2)-k2(2))<GW_TOLQ) then
   if (f(k1(3)-k2(3))<GW_TOLQ) then
    isamek=.TRUE.
    G0(:)=NINT(k1(:)-k2(:))
   end if
  end if
 end if

end function isamek
!!***

!!****f* m_bz_mesh/isequalk
!! NAME
!! is_equalk
!!
!! FUNCTION
!! Return .TRUE. if two points are equal within a reciprocal lattice vector.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!  Will be filled automatically by the parent script
!!
!! CHILDREN
!!  Will be filled automatically by the parent script
!!
!! SOURCE

function isequalk(q1,q2)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 logical :: isequalk
 real(dp),intent(in) :: q1(3),q2(3)

!Local variables-------------------------------
 integer :: G0(3)
! *************************************************************************

 isequalk = isamek(q1,q2,G0)

end function isequalk
!!***

!!****f* m_bz_mesh/has_BZ_item
!! NAME
!! has_BZ_item
!!
!! FUNCTION
!!  check if item belongs to the BZ  within a reciprocal lattice vector
!!  and return the index number and the reciprocal vector g0.
!!
!! INPUTS
!!  Kmesh<bz_mesh_type>=datatype gathering information on the k-mesh
!!  item(3)=the k-point to be checked
!!
!! OUTPUT
!!  .TRUE. if item is the BZ within a RL vector
!!  ikbz=Index of the k-point in the Kmesh%bz array
!!  G0(3)=Umklapp vector.
!!
!! PARENTS
!!
!! FIXME
!!  Switch to routine version. Due to side-effects the present implemenation
!!  might be source of bugs in logical statements
!!
!! CHILDREN
!!
!! SOURCE

logical function has_BZ_item(Kmesh,item,ikbz,G0)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(out) :: ikbz
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: item(3)

!Local variables-------------------------------
!scalars
 integer :: ik_bz,yetfound
 character(len=500) :: msg

! *************************************************************************

 has_BZ_item=.FALSE. ; ikbz=0 ; G0(:)=0 ; yetfound=0
 do ik_bz=1,Kmesh%nbz
  if (isamek(item,Kmesh%bz(:,ik_bz),G0)) then
   has_BZ_item=.TRUE.
   ikbz=ik_bz
   yetfound=yetfound+1
   !EXIT
  end if
 end do

 if (yetfound/=0.and.yetfound/=1) then
  write(msg,'(a)')' multiple k-points found '
  MSG_BUG(msg)
 end if

end function has_BZ_item
!!***

!!****f* m_bz_mesh/has_IBZ_item
!! NAME
!! has_IBZ_item
!!
!! FUNCTION
!!  Check if item belongs to the IBZ within a reciprocal lattice vector
!!
!! INPUTS
!!  Kmesh<bz_mesh_type>=Datatype gathering information on the mesh in the BZ.
!!  item(3)=the k-point to be checked
!!
!! OUTPUT
!!  Return .TRUE. if item is the IBZ within a RL vector
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

logical function has_IBZ_item(Kmesh,item,ikibz,G0)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
! scalars
! arrays
!scalars
 integer,intent(out) :: ikibz
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 integer,intent(out) :: G0(3)
 real(dp),intent(in) :: item(3)

!Local variables-------------------------------
!scalars
 integer :: ik_ibz,yetfound
 character(len=500) :: msg

! *************************************************************************

 has_IBZ_item=.FALSE. ; ikibz=0 ; G0(:)=0 ; yetfound=0
 do ik_ibz=1,Kmesh%nibz
  if (isamek(item,Kmesh%ibz(:,ik_ibz),G0)) then
   has_IBZ_item=.TRUE.
   ikibz=ik_ibz
   yetfound=yetfound+1
   !EXIT
  end if
 end do

 if (yetfound/=0.and.yetfound/=1) then
  write(msg,'(a)')' multiple k-points found '
  MSG_BUG(msg)
 end if

end function has_IBZ_item
!!***

!!****f* m_bz_mesh/make_mesh
!! NAME
!! make_mesh
!!
!! FUNCTION
!! Initialize the BZ_mesh_type starting from qptrlatt and qshft
!!
!! INPUTS
!! Cryst<Crystal_structure>=Info on the crystalline structure.
!! nqshft=Number of shifts.
!! qptrlatt=
!! qshft(3,nqshft)=Shifts
!!
!! OUTPUT
!! Kmesh
!!
!! PARENTS
!!      m_gwannier
!!
!! CHILDREN
!!
!! SOURCE

subroutine make_mesh(Kmesh,Cryst,qptrlatt,nqshft,qshft)

 use defs_basis
 use defs_datatypes, only : BZ_mesh_type, Crystal_structure
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_32_util
 use interfaces_56_recipspace
!End of the abilint section

 implicit none

!Arguments -------------------------------
!scalars
 integer,intent(in) :: nqshft
 type(BZ_mesh_type),intent(out) :: Kmesh
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 integer,intent(in) :: qptrlatt(3,3) 
 real(dp),intent(in) :: qshft(3,nqshft)

!Local variables -------------------------
!scalars
 integer :: max_nqpt,facbrv,brav,mtetra,iqibz,nqibz,nqbz
 integer :: nsym,timrev,option,iscf,kptopt,msym,ntetra_irr
 real(dp) :: tetra_vol
 character(len=500) :: msg
!arrays
 integer :: bravais(11),dsifkpt(3),vacuum(3)
 integer,pointer :: symrec(:,:,:),symrel(:,:,:),symafm(:)
 integer,allocatable :: ibz2bz(:),bz2ibz(:)
 integer,allocatable :: ngqpt(:,:),tetra_full(:,:,:),tetra_mult(:),tetra_wrap(:,:,:)
 real(dp) :: gmet(3,3),rprimd(3,3),qlatt(3,3),gprimd(3,3),rlatt(3,3)
 real(dp),pointer :: tnons(:,:)
 real(dp),allocatable :: qibz(:,:),qbz(:,:),qshft_loc(:,:)
 real(dp),allocatable :: wtq(:),wtq_folded(:)

! *************************************************************************

 DBG_ENTER("COLL")

 ! Here we can select only particular symmetries if needed.
 ! Try to use getkgrid should be safer but it is difficult
 ! to encapsulate all the options.
 nsym   = Cryst%nsym
 timrev=0 ; if (Cryst%timrev==2) timrev=1  !FIXME there an incompatibly between Cryst%timrev and symkpt
 rprimd = Cryst%rprimd
 gprimd = Cryst%gprimd
 gmet   = Cryst%gmet

 symrel => Cryst%symrel
 tnons  => Cryst%tnons
 symrec => Cryst%symrec
 symafm => Cryst%symafm

 if (nqshft==1) then
  allocate(qshft_loc(3,nqshft))
  qshft_loc=qshft
 else 
  ! try to reduce the qpoint grid to a single qshift, otherwise stop
  msg = 'multiple shifts not implemented yet'
  MSG_ERROR(msg)
  iscf = 3
  msym=nsym
  kptopt=3
  dsifkpt(:)=1
  vacuum(:)=0
!  wtq(:) = one
!  call getkgrid(dsifkpt,std_out,iscf,kpt,kptopt,kptrlatt,kptrlen,&
!& msym,nkpt,nkpt_computed,nshiftk,nsym,rprimd,shiftk,symafm,&
!& symrel,tnons,vacuum,wtk)
 end if 

 max_nqpt = qptrlatt(1,1)*qptrlatt(2,2)*qptrlatt(3,3) &
&          +qptrlatt(1,2)*qptrlatt(2,3)*qptrlatt(3,1) &
&          +qptrlatt(1,3)*qptrlatt(2,1)*qptrlatt(3,2) &
&          -qptrlatt(1,2)*qptrlatt(2,1)*qptrlatt(3,3) &
&          -qptrlatt(1,3)*qptrlatt(2,2)*qptrlatt(3,1) &
&          -qptrlatt(1,1)*qptrlatt(2,3)*qptrlatt(3,2)

 allocate(qibz(3,max_nqpt),qbz(3,max_nqpt))

 !Save memory during the generation of the q-mesh in the full BZ  
 !Take into account the type of Bravais lattice
 !call symbrav to fill bravais
 !berryopt = 1
 !call symbrav(berryopt,bravais,iout,nsym,nptsym,ptsymrel,rmet,rprimd)

 option=1  ! just for printout
 brav=1
 facbrv=1
 !if (anaddb_dtset%brav==2) facbrv=2
 !if (anaddb_dtset%brav==3) facbrv=4

 call smpbz(brav,std_out,qptrlatt,max_nqpt,nqbz,nqshft,option,qshft_loc,qbz)
 ! 
 ! === Reduce the number of such points by symmetrization ===
 allocate(ibz2bz(nqbz),wtq(nqbz),wtq_folded(nqbz))
 wtq(:)=one/nqbz  ! Weights sum up to one
 option=1         ! Do not output

 call symkpt(gmet,ibz2bz,qbz,nqbz,nqibz,nsym,option,symrec,timrev,wtq,wtq_folded)

 ! Here I should initialize kmesh
 ! For the moment fill only the quantities I need for the DOS.
 Kmesh%nshift   = nqshft
 Kmesh%gmet     = gmet
 Kmesh%gprimd   = gprimd
 Kmesh%nsym     = nsym
 Kmesh%timrev   = Cryst%timrev
 Kmesh%kptrlatt = qptrlatt

 allocate(Kmesh%shift(3,nqshft))
 Kmesh%shift=qshft_loc

 Kmesh%nbz    = nqbz
 Kmesh%nibz   = nqibz
 allocate(Kmesh%bz (3,nqbz ))
 allocate(Kmesh%ibz(3,nqibz))
 allocate(Kmesh%wt(nqibz))

 Kmesh%bz=qbz(3,nqbz)
 do iqibz=1,nqibz
  Kmesh%ibz(:,iqibz) = qbz     (:,ibz2bz(iqibz))
  Kmesh%wt (iqibz)   = wtq_folded(ibz2bz(iqibz))
  qibz   (:,iqibz)   = qbz     (:,ibz2bz(iqibz))
 end do
 deallocate(ibz2bz,wtq,wtq_folded)

 ! ==========================
 ! === Tetrahedron method ===
 ! ==========================
 ! * convert kptrlatt to double and invert. 
 ! * qlatt here refer to the shortest qpt vectors
 rlatt(:,:)=qptrlatt(:,:)
 call matr3inv(rlatt,qlatt)
 !  
 ! === Make full kpoint grid and get equivalence to irred kpoints ===
 ! * Note: This routines badly scales wrt nqbz, 
 ! TODO should be rewritten, pass timrev and speed up by looping on shells.
 print*,'calling get_full_kgrid with ',nqbz
 call pclock(0)

 allocate(bz2ibz(nqbz))
 call get_full_kgrid(bz2ibz,qlatt,qibz,qbz,qptrlatt,nqibz,nqbz,nqshft,nsym,qshft_loc,symrel)

 print*,'after get_full_kgrid'
 call pclock(9999)
 !  
 ! === Get tetrahedra, ie indexes of the full q-points at their summits ===
 ! * tetra_full(:,1,it) : the indices of the irreducible k-points
 !   tetra_full(:,2,it) : the indices of the full k-points
 !   tetra_wrap(:,:,it) : a flag to wrap q-points outside the IBZ (+-1) to get the irreducible tetrahedra
 ! * the number of equivalent tetrahedra is counted in tetra_mult and the inequivalent few (ntetra < mtetra) are 
 !   packed into the beginning of tetra_full
 mtetra=6*nqbz
 allocate(tetra_full(4,2,mtetra),tetra_wrap(3,4,mtetra),tetra_mult(mtetra))
   
 call get_tetra(bz2ibz,gprimd,qlatt,qbz,mtetra,nqbz,ntetra_irr,tetra_full,tetra_mult,tetra_wrap,tetra_vol)

 write(*,*)' Number of irreducible tetrahedrons = ',ntetra_irr
 deallocate(bz2ibz)

 ! === Fill in tetrahedron variables and arrays ===
 Kmesh%ntetra_irr = ntetra_irr
 Kmesh%tetra_vol  = tetra_vol

 allocate(Kmesh%tetra_full(4,2,ntetra_irr))
 Kmesh%tetra_full = tetra_full(:,:,1:ntetra_irr)
 deallocate(tetra_full)

 allocate(Kmesh%tetra_wrap(3,4,ntetra_irr))
 Kmesh%tetra_wrap = tetra_wrap(:,:,1:ntetra_irr)
 deallocate(tetra_wrap)

 allocate(Kmesh%tetra_mult(ntetra_irr))
 Kmesh%tetra_mult = tetra_mult(1:ntetra_irr)
 deallocate(tetra_mult)

 deallocate(qshft_loc)

 DBG_EXIT("COLL")

end subroutine make_mesh
!!***

!!****f* m_bz_mesh/findk
!! NAME
!! findk
!!
!! FUNCTION
!! Check whether the k-point is in the set of the kbz
!!
!! INPUTS
!!  kbz(3,nkbz)=coordinates of k points in the BZ
!!  nkcalc= number of k points for GW calculation (input variable)
!!  nkbz=number of k points in Brillouin zone
!!  xkcalc(3,nkcalc)= coordinates of the k points
!!  umklp_opt=0 if no umklapp vector is admitted, 1 otherwise
!!
!! OUTPUT
!!  kcalc=index of the k points inside kbz
!!
!! TODO 
!!  Should be removed and replaced by Kmesh methods
!!
!! PARENTS
!!      m_gwannier,setup_sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine findk(nkcalc,nkbz,xkcalc,kbz,kcalc,umklp_opt)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbz,nkcalc,umklp_opt
!arrays
 integer,intent(out) :: kcalc(nkcalc)
 real(dp),intent(in) :: kbz(3,nkbz),xkcalc(3,nkcalc)

!Local variables-------------------------------
!scalars
 integer :: ik,jj
 real(dp) :: shift
 character(len=500) :: msg
!arrays
 real(dp) :: dummy(3),ktest(3)

! *************************************************************************

 write(msg,'(2a)')ch10,' findk : check if the k-points for sigma are in the set of BZ'
 call wrtout(std_out,msg,'COLL')

 select case (umklp_opt) 

 case (0)
  do jj=1,nkcalc
   kcalc(jj)=0
   do ik=1,nkbz
    if (all(abs(xkcalc(:,jj)-kbz(:,ik))<1.e-3)) kcalc(jj)=ik
   end do
   if (kcalc(jj)==0) then
    write(msg,'(a,3(f6.3,1x),a)')' k-point ',xkcalc(:,jj),' not in the set of kbz'
    MSG_ERROR(msg)
   end if
  end do

 case (1)
  do jj=1,nkcalc
   kcalc(jj)=0
   do ik=1,nkbz
    dummy=xkcalc(:,jj)-kbz(:,ik)
    call canon9(dummy(1),ktest(1),shift)
    call canon9(dummy(2),ktest(2),shift)
    call canon9(dummy(3),ktest(3),shift)
    if (all(abs(ktest)<1.e-3)) kcalc(jj)=ik
   end do
   if(kcalc(jj)==0) then
    write(msg,'(a,3(f6.3,1x),a)')&
&    ' k-point ',xkcalc(:,jj),' not in the set of kbz even though umklapp G0 are allowed '
    MSG_ERROR(msg)
   end if
  end do

 case DEFAULT
  MSG_BUG("Wrong umkl_opt")
 end select 

 write(msg,'(2a)')' check ok !',ch10
 call wrtout(std_out,msg,'COLL')

end subroutine findk
!!***

!!****f* m_bz_mesh/identk
!! NAME
!! identk
!!
!! FUNCTION
!! Identify k-points in the whole BZ starting from the IBZ. 
!! Generate also symmetry tables relating the BZ to the IBZ.
!!
!! INPUTS
!!  kibz(3,nkibz)=Coordinates of k-points in the IBZ.
!!  timrev=2 if time reversal symmetry can be used; 1 otherwise.
!!  nkibz=Number of k points in IBZ.
!!  nkbzmx=Maximum number of k points in BZ.
!!  nsym=Number of symmetry operations.
!!  symrec(3,3,nsym)=Symmetry operation matrices in reciprocal space.
!!  symafm(nsym)=(anti)ferromagnetic part of symmetry operations.
!!
!! OUTPUT
!!  kbz(3,nkbzmx)= k-points in whole BZ
!!  ktab(nkbzmx)= table giving for each k-point in the BZ (array kbz), 
!!   the corresponding irreducible point in the array (kibz)
!!   k_BZ= (IS) kIBZ where S is one of the symrec operations and I is the inversion or the identity
!!    where k_BZ = (IS) k_IBZ and S = \transpose R^{-1} 
!!  ktabi(nkbzmx)= for each k-point in the BZ defines whether inversion has to be 
!!   considered in the relation k_BZ=(IS) k_IBZ (1 => only S; -1 => -S)  
!!  ktabo(nkbzmx)= the symmetry operation S that takes k_IBZ to each k_BZ
!!  nkbz= no. of k-points in the whole BZ
!!  wtk(nkibz)= weight for each k-point in IBZ for symmetric quantities:
!!              no. of distinct ks in whole BZ/(timrev*nsym)
!!
!! NOTES
!!  The logic of the routine relies on the assumption that kibz really represent an irreducible set. 
!!  If symmetrical points are present in the input list, indeed, some the output weights will turn out to be zero.
!!  An initial check is done at the beginnig of the routine to trap this possible error.
!!
!! PARENTS
!!      m_bz_mesh,rdm
!!
!! CHILDREN
!!
!! SOURCE

subroutine identk(kibz,nkibz,nkbzmx,nsym,timrev,symrec,symafm,use_antiferro,kbz,ktab,ktabi,ktabo,nkbz,wtk,prtvol)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbzmx,nkibz,nsym,timrev,prtvol
 integer,intent(out) :: nkbz
 logical,intent(in) :: use_antiferro
!arrays
 integer,intent(in) :: symafm(nsym),symrec(3,3,nsym)
 integer,intent(out) :: ktab(nkbzmx),ktabi(nkbzmx),ktabo(nkbzmx)
 real(dp),intent(in) :: kibz(3,nkibz)
 real(dp),intent(out) :: kbz(3,nkbzmx),wtk(nkibz)

!Local variables ------------------------------
!scalars
 integer :: div4,found,id,ii,ik1,ik2,ikbz,ikibz,ikp,iold,isym,itim,jj,res
 logical :: isirred
 character(len=100) :: fmt
 character(len=500) :: msg
!arrays
 integer :: G0(3)
 real(dp) :: knew(3),k1(3),k2(3)

! *************************************************************************

 DBG_ENTER("COLL")

 ! === Check whether kibz really forms an irreducible set ===
 isirred=.TRUE.
 do ik1=1,nkibz-1
  k1=kibz(:,ik1)
  do ik2=ik1+1,nkibz
   k2=kibz(:,ik2)

   do itim=1,timrev
    do isym=1,nsym
     !if (use_antiferro.and.symafm(isym)==-1) CYCLE
     if (symafm(isym)==-1) CYCLE
     knew = (3-2*itim) * MATMUL(symrec(:,:,isym),k2)
     if (isamek(k1,knew,G0)) then
      isirred=.FALSE.
      write(msg,'(2(a,3f8.4),2(a,i2))')&
&      ' k1 = ',k1,' is symmetrical of k2 = ',k2,' through sym = ',isym,' itim = ',itim
      MSG_WARNING(msg)
     end if
    end do
   end do

  end do
 end do

 if (.not.isirred) then
  MSG_ERROR('input kibz is not irred.') 
 end if

 !
 ! === Loop over k-points in IBZ ===
 ! * Start with zero no. of k-points found.
 nkbz=0 
 do ikibz=1,nkibz
  wtk(ikibz) = zero

  ! === Loop over time-reversal I and symmetry operations S  ===
  ! * Use spatial inversion instead of time reversal whenever possible.
  do itim=1,timrev
   do isym=1,nsym

    !if (use_antiferro.and.symafm(isym)==-1) CYCLE
    if (symafm(isym)==-1) CYCLE

    ! === Form IS k ===
    ! FIXME this is a temporary hacking to pass tests under gfortran, should use MATMUL
    call dosym(REAL(symrec(:,:,isym),dp),itim,kibz(:,ikibz),knew)
    !knew(:)=(3-2*itim)*MATMUL(symrec(:,:,isym),kibz(:,ikibz))

    ! * Check whether it has already been found (to within a RL vector).
    iold=0
    do ikbz=1,nkbz
     if (isamek(knew,kbz(:,ikbz),G0)) iold=iold+1
    end do
    ! * If not yet found add to kbz and increase the weight.
    if (iold==0) then
     nkbz=nkbz+1
     wtk(ikibz)=wtk(ikibz)+one
     if (nkbz>nkbzmx) then
      write(msg,'(a,i6,a)')&
&      ' nkbzmx too small, nkbzmx = ',nkbzmx,', increase nkbzmx !'
      MSG_BUG(msg)
     end if
     kbz(:,nkbz) = knew(:)
     ktab (nkbz) = ikibz
     ktabo(nkbz) = isym
     ktabi(nkbz) = 3-2*itim
    end if
   end do 
  end do 

 end do !ikibz

 ! * Weights are normalized to 1.
 wtk = wtk/SUM(wtk)

 ! ================================
 ! === Printout of the results ===
 ! ================================
 if (prtvol>0) then

  write(msg,'(2a,i3,2a,10x,2a)')ch10,&
&  ' number of k-points in the irreducible wedge (IBZ) ',nkibz,ch10,&
&  ' k-points [reciprocal lattice units]','weights',ch10
  call wrtout(std_out,msg,'COLL')
  write(fmt,*)'(i5,3f12.6,3x,f12.6)'
  do jj=1,nkibz
   write(msg,fmt) jj,(kibz(ii,jj),ii=1,3),wtk(jj)
   call wrtout(std_out,msg,'COLL')
  end do
  write(msg,'(2a,i2,3a,i5,a)')ch10,&
&  ' together with ',nsym,' symmetry operations and inversion',ch10,&
&  ' have yielded ',nkbz,' k-points in Brillouin Zone (BZ):'
  call wrtout(std_out,msg,'COLL')

  write(fmt,*)'(i5,2x,4(3f7.3,2x))'
  div4=nkbz/4 ; res=MOD(nkbz,4)
  do id=0,div4-1
   jj=4*id+1 
   write(msg,fmt)jj,((kbz(ii,ikp),ii=1,3),ikp=jj,jj+3)
   call wrtout(std_out,msg,'COLL')
  end do
  if (res/=0) then
   write(fmt,*)'(i5,2x,',res,'(3f7.3,2x),a)'
   write(msg,fmt)4*div4+1,((kbz(ii,ik1),ii=1,3),ik1=4*div4+1,nkbz),ch10
   call wrtout(std_out,msg,'COLL')
  end if

 end if !prtvol

 DBG_EXIT("COLL")

end subroutine identk
!!***

!!****f* m_bz_mesh/get_ng0sh
!! NAME
!! get_ng0sh
!!
!! FUNCTION
!!  Given two lists of k-points, kbz1 and kbz2, calculate every possible difference k1-k2.
!!  For each difference, find the umklapp G0 vector and the point k3 in the array kfold 
!!  such as k1-k2 = k3 + G0.
!!  The optimal value of G0 shells is returned, namely the smallest box around Gamma 
!!  which suffices to treat all possible umklapp processes.
!!
!! INPUTS
!!  gmet(3,3)=Reciprocal space metric ($\textrm{bohr}^{-2}$).
!!  nfold=Number of points in the array kfold.
!!  nk1, nk2=Number of points in the arrays kbz1, kbz2.
!!  kbz1(3,nk1)=Reduced coordinates of the first set of points.
!!  kbz2(3,nk2)=Reduced coordinates of the second set of points.
!!  kfold(3,nkfol)=Reduced coordinated of the points in the BZ.
!!  tolq0=Tolerance below which a q-point is treated as zero.
!!  mg0sh=Integer defining the Max number of shells to be considered.
!!
!! OUTPUT
!!  opt_ng0(3)=Minimal reduced components of the G0 vectors to account for umklapps.
!!
!! PARENTS
!!      setup_screening,setup_sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine get_ng0sh(nk1,kbz1,nk2,kbz2,nkfold,kfold,gmet,tolq0,mg0sh,opt_ng0)

 use defs_basis

 use m_numeric_tools, only : is_zero
 use m_geometry,      only : normv

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_28_numeric_noabirule
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: mg0sh,nk1,nk2,nkfold
 real(dp),intent(in) :: tolq0
!arrays
 integer,intent(out) :: opt_ng0(3)
 real(dp),intent(in) :: gmet(3,3),kbz1(3,nk1),kbz2(3,nk2),kfold(3,nkfold)

!Local variables-------------------------------
!scalars
 integer :: i1,i2,ig,ig1,ig2,ig3,ii,ikf,itr,ng0
 logical :: found
 character(len=500) :: msg
!arrays
 integer :: gtrial(3)
 integer,allocatable :: iperm(:),g0(:,:) 
 real(dp) :: k1mk2(3)
 real(dp),allocatable :: norm(:)

!************************************************************************
 
 ng0=(2*mg0sh+1)**3
 allocate(g0(3,ng0),norm(ng0),iperm(ng0))

 ii=1
 do ig3=-mg0sh,mg0sh
  do ig2=-mg0sh,mg0sh
   do ig1=-mg0sh,mg0sh
    g0(1,ii)=ig1
    g0(2,ii)=ig2
    g0(3,ii)=ig3
    norm(ii)=normv(g0(:,ii),gmet,'G')
    iperm(ii)=ii
    ii=ii+1
   end do
  end do
 end do
 !
 ! === Order g0-vectors in order of increasing module ===
 ! * Should speed up the search in the while statement below.
 call sort_dp(ng0,norm,iperm,tol14)

 opt_ng0(:)=0 
 do i2=1,nk2
  ! This is used in case of screening calculation.
  ! If q is small treat it as zero. In this case, indeed, 
  ! we use q=0 to calculate the oscillator matrix elements. 
  if (is_zero(kbz2(:,i2),tolq0)) CYCLE
  do i1=1,nk1 
   k1mk2(:)=kbz1(:,i1)-kbz2(:,i2)
   itr=1 ; found=.FALSE.
   do while (itr<=ng0.and..not.found)
    do ikf=1,nkfold
     if (ALL(ABS(k1mk2(:)-kfold(:,ikf)-g0(:,iperm(itr))) < GW_TOLQ)) then 
      found=.TRUE. 
      gtrial(:)=ABS(g0(:,iperm(itr)))
      opt_ng0(1)=MAX(opt_ng0(1),gtrial(1))
      opt_ng0(2)=MAX(opt_ng0(2),gtrial(2))
      opt_ng0(3)=MAX(opt_ng0(3),gtrial(3))
      EXIT
     end if
    end do
    itr=itr+1
   end do
   if (.not.found) then 
    write(msg,'(a,i5,3a,i2,2(2a,i4,3es16.8),a)')&
&    ' Not able to found umklapp G0 vector among ',ng0,' vectors',ch10,&
&    ' Increase mg0sh such as k1-k2 = kf+G0, present value = ',mg0sh,ch10,&
&    ' point1 = ',i1,kbz1(:,i1),ch10,&
&    ' point2 = ',i2,kbz2(:,i2),ch10
    MSG_ERROR(msg)
   end if
  end do
 end do 

 write(msg,'(a,3i2)')' optimal value for ng0sh = ',opt_ng0(:)
 call wrtout(std_out,msg,'COLL')

 deallocate(g0,norm,iperm)

end subroutine get_ng0sh
!!***

!!****f* m_bz_mesh/make_path
!! NAME
!! make_path
!!
!! FUNCTION
!!  Create a normalized path given the extrema.
!!
!! INPUTS
!!  nbounds=Number of extrema defining the path.
!!  bounds(3,nbounds)=The points defining the path in reduced coordinates.
!!  met(3,3)=Metric matrix. 
!!  space='R' for real space, G for reciprocal space.
!!  ndiv_small=Number of divisions to be used for the smallest segment.
!!
!! OUTPUT
!!  npt_tot=Total number of points in the normalized circuit.
!!  ndiv(nbounds-1)=Number of division for each segment
!!  See also SIDE EFFECTS
!!
!! SIDE EFFECTS
!!  In input path(:,:) is a NULL pointer.
!!  It is allocated inside the routine. When the subroutine returns, path(3,npt_tot) will
!!  contain the path in reduced coordinates.
!! 
!! PARENTS
!!      m_gwannier
!!
!! CHILDREN
!!
!! SOURCE

subroutine make_path(nbounds,bounds,met,space,ndiv_small,ndiv,npt_tot,path)

 use defs_basis

 use m_geometry, only : normv

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nbounds,ndiv_small
 integer,intent(out) :: npt_tot
 character(len=1),intent(in) :: space
!arrays
 integer,intent(out) :: ndiv(nbounds-1)
 real(dp),intent(in) :: bounds(3,nbounds),met(3,3)
 real(dp),pointer :: path(:,:)

!Local variables-------------------------------
!scalars
 integer :: idx,ii,jp,n,np
 real(dp) :: nfact
 logical :: ltest
 character(len=500) :: msg
!arrays
 real(dp) :: diff(3),lng(nbounds-1)

! *************************************************************************

 ltest=ndiv_small>0
 call assert(ltest,'ndiv_small <=0',__FILE__,__LINE__)

 do ii=1,nbounds-1 
  diff(:)=bounds(:,ii+1)-bounds(:,ii)
  lng(ii) = normv(diff,met,space)
 end do

 ! Avoid division by zero if any k(:,i+1)=k(:,i).
 nfact=MINVAL(lng)
 if (ABS(nfact)<tol6) then 
  write(msg,'(3a)')&
&  ' Found two equal consecutive points in the path ',ch10,&
&  ' This is not allowed, please modify the path in your input file'
  MSG_ERROR(msg)
 end if

 nfact=nfact/ndiv_small
 ndiv(:)=NINT(lng(:)/nfact) 
 npt_tot=SUM(ndiv)+1 !1 for the first point

 write(msg,'(2a,i6,2a)')ch10,&
& ' Total number of points in the path : ',npt_tot,ch10,&
& ' Number of divisions for each segment of the normalized path : '
 call wrtout(std_out,msg,'COLL') 

 do ii=1,nbounds-1
  write(msg,'(2(3f8.5,a),i5,a)')bounds(:,ii),' ==> ',bounds(:,ii+1),' ( ndiv : ',ndiv(ii),' )' 
  call wrtout(std_out,msg,'COLL')
 end do 
 write(msg,'(a)')ch10
 call wrtout(std_out,msg,'COLL') 

 ! Allocate and construct the path.
 allocate(path(3,npt_tot))

 write(msg,'(2a)')ch10,' Normalized Path : '
 call wrtout(std_out,msg,'COLL')
 idx=0
 do ii=1,nbounds-1
  do jp=1,ndiv(ii)
   idx=idx+1
   path(:,idx)=bounds(:,ii)+(jp-1)*(bounds(:,ii+1)-bounds(:,ii))/ndiv(ii)
   write(msg,'(i4,4x,3(f8.5,1x))')idx,path(:,idx)
   call wrtout(std_out,msg,'COLL')
  end do 
 end do 
 path(:,npt_tot)=bounds(3,nbounds)
 write(msg,'(i4,4x,3(f8.5,1x))')npt_tot,path(:,npt_tot)
 call wrtout(std_out,msg,'COLL')

end subroutine make_path
!!***


!!****f* m_bz_mesh/setup_Qmesh
!! NAME
!! setup_Qmesh
!!
!! FUNCTION
!! Initialize and construct a bz_mesh_type datatype 
!! gathering information on the q-mesh used for GW calculations (special treatment temporarily needed for Q-points)
!!
!! INPUTS
!! nqibz=number of irreducible q-points
!! nsym=number of symmetry operations
!! prtvol=verbosity level
!! timrev=1 if time-reversal cannot be used, 2 otherwise
!! qibz(3,nqibz)=irreducible q-points
!! symrec(3,3,nsym)=symmetry operations in reciprocal space
!!
!! OUTPUT
!! Qmesh<bz_mesh_type>=datatype gathering information on the q point sampling. see defs_datatypes.F90
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_bz_mesh,setup_sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine setup_Qmesh(nqibz,Cryst,prtvol,qibz,Qmesh)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_42_geometry
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nqibz,prtvol
 type(BZ_mesh_type),intent(inout) :: Qmesh
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 real(dp),intent(in) :: qibz(3,nqibz)

!Local variables-------------------------------
!scalars
 integer :: iq_bz,iq_ibz,isym,itim,nqbz,nqbzX,nsym,timrev
 real(dp) :: ucvol
 logical :: ltest
 character(len=500) :: msg
!arrays
 integer,allocatable :: qtab(:),qtabi(:),qtabo(:)
 integer,pointer :: symrec(:,:,:)
 real(dp) :: Sq(3),gmet(3,3),gprimd(3,3),rmet(3,3)
 real(dp),allocatable :: qbz(:,:),wtq(:)

! *************************************************************************

 DBG_ENTER("COLL")

 call NullifyBzMesh(Qmesh)

 ltest=(Cryst%timrev==1.or.Cryst%timrev==2)
 call assert(ltest,"timrev should be 1 or 2",__FILE__,__LINE__)

 nsym   =  Cryst%nsym
 timrev =  Cryst%timrev
 symrec => Cryst%symrec

 call metric(gmet,gprimd,-1,rmet,Cryst%rprimd,ucvol)
 Qmesh%gmet   = gmet
 Qmesh%gprimd = gprimd

 nqbzX=nqibz*nsym*timrev
 allocate(qbz(3,nqbzX),qtab(nqbzX),qtabo(nqbzX),qtabi(nqbzX),wtq(nqibz))
 qbz(:,:)=0 ; qtab(:)=0 ; qtabo(:)=0 ; qtabi(:)=0

 call identq(qibz,nqibz,nqbzX,REAL(symrec,dp),nsym,timrev,wtq,qbz,qtab,qtabi,qtabo,nqbz,verbose=1)

 !£ use_antiferro = Cryst%use_antiferro
 !£ symafm => Cryst%symafm
 !£ symrec => Cryst%symrec
 !£ call identk(qibz,nqibz,nqbzX,nsym,timrev,symrec,symafm,use_antiferro,qbz,qtab,qtabi,qtabo,nqbz,wtq,my_prtvol)

 do iq_bz=1,nqbz
  isym=qtabo(iq_bz) ; iq_ibz=qtab(iq_bz) ; itim=(3-qtabi(iq_bz))/2
  call dosym(REAL(symrec(:,:,isym),dp),itim,qibz(:,iq_ibz),Sq(:))
  if (ANY(ABS(qbz(:,iq_bz)-Sq(:) )>1.0d-4)) then
   write(*,*) Sq,qbz(:,iq_bz) 
   write(msg,'(a,3f6.3,a,3f6.3,2a,9i3,2a)')&
&   ' qpoint ',qbz(:,iq_bz),' is the symmetric of ',qibz(:,iq_ibz),ch10,&
&   ' through operation ',symrec(:,:,isym),ch10,&
&   ' however a non zero umklapp G_o vector is required and this is not yet allowed'
   MSG_ERROR(msg)
  end if
 end do 
 !
 ! ==== Create data structure to store information on q-points ====
 ! * Dimensions
 Qmesh%nbz    = nqbz
 Qmesh%nibz   = nqibz      
 Qmesh%timrev = timrev
 Qmesh%nsym   = nsym

 Qmesh%ntetra_irr=0   ! no tetrahedrons for the moment
 !
 ! * Arrays
 allocate(Qmesh%ibz(3,nqibz)) ; Qmesh%ibz(:,:) = qibz(:,1:nqibz) 
 allocate(Qmesh%wt(nqibz))    ; Qmesh%wt(:)    = wtq(1:nqibz)

 allocate(Qmesh%bz(3,nqbz))   ; Qmesh%bz(:,:)  = qbz(:,1:nqbz)
 allocate(Qmesh%tab(nqbz))    ; Qmesh%tab(:)   = qtab (1:nqbz)
 allocate(Qmesh%tabi(nqbz))   ; Qmesh%tabi(:)  = qtabi(1:nqbz)
 allocate(Qmesh%tabo(nqbz))   ; Qmesh%tabo(:)  = qtabo(1:nqbz)

 !
 ! TODO For the time being these arrays are not used however they should be defined just
 ! to be consistent
 !nullify(Qmesh%tabp)       
 !nullify(Qmesh%rottb)
 !nullify(Qmesh%rottbm1)

 !call print_bz_mesh(Qmesh,prtvol=prtvol)
 deallocate(qbz,qtab,qtabi,qtabo,wtq)

 DBG_EXIT("COLL")

end subroutine setup_Qmesh
!!***

!!****f* m_bz_mesh/find_Qmesh
!! NAME
!! find_Qmesh
!!
!! FUNCTION
!!  Find the q-mesh defined as all the possible differences between k-points
!!  Find the irreducible q-points using a special treatment for the Gamma point.
!!  Then call setup_kmesh to initialize the Qmesh datatype
!!
!! INPUTS
!!  Cryst<crystal_structure>=datatype gathering info on the unit cell and symmetries
!!    %nsym=number of symmetry operations
!!    %symrec(3,3,nsym)=symmetry operations in reciprocal space
!!  prtvol=verbosity level
!!  timrev=1 if time-reversal cannot be used, 2 otherwise
!!  Kmesh<bz_mesh_type>=datatype gathering information on the k-mesh
!!
!! OUTPUT
!!  Qmesh<bz_mesh_type>=datatype gathering information on the q point sampling. see defs_datatypes.F90
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_electrons,mrgscr,setup_screening
!!
!! CHILDREN
!!
!! SOURCE

subroutine find_Qmesh(Cryst,Kmesh,Qmesh,prtvol)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: prtvol
 type(BZ_mesh_type),intent(in) :: Kmesh
 type(BZ_mesh_type),intent(out) :: Qmesh
 type(Crystal_structure),intent(in) :: Cryst

!Local variables-------------------------------
!scalars
 integer :: nqibz
 logical :: avoid_zero
 character(len=500) :: msg
!arrays
 real(dp),allocatable :: qibz(:,:)

! *************************************************************************

 ! * Find the number of q-points such that q = k1-k2.
 call findnq(Kmesh%nbz,Kmesh%bz,Cryst%nsym,Cryst%symrec,nqibz,Cryst%timrev) 
 !
 ! * Find the coordinates of the q-points in the IBZ.
 allocate(qibz(3,nqibz)) 
 !avoid_zero=.TRUE.
 avoid_zero=.FALSE.

 call findq(Kmesh%nbz,Kmesh%bz,Cryst%nsym,Cryst%symrec,Cryst%gprimd,nqibz,qibz,Cryst%timrev,avoid_zero)
 !
 ! * Create the Qmesh object starting from the IBZ.
 call setup_Qmesh(nqibz,Cryst,prtvol,qibz,Qmesh)

 ! TODO Here I should call setup_Kmesh, just to keep it simple but I have to 
 ! solve some problems with the small q
 !call InitKmesh(nqibz,qibz,Cryst,Qmesh,prtvol)

 deallocate(qibz)

end subroutine find_Qmesh
!!***

!!****f* m_bz_mesh/findnq
!! NAME
!! findnq
!!
!! FUNCTION
!! Identify the number of q-points in the IBZ by which the k-points in BZ differ
!! (count the q points in the k-point difference set)
!!
!! INPUTS
!!  kbz(3,nkbz)=coordinates of k points in BZ
!!  timrev=2 if time-reversal symmetry is used, 1 otherwise
!!  nkbz=number of k points in Brillouin zone
!!  nsym=number of symmetry operations
!!  symrec(3,3,nsym)=symmetry operations in reciprocal space
!!
!! OUTPUT
!!  nqibz=number of q points
!!
!! PARENTS
!!      m_bz_mesh,rdm
!!
!! CHILDREN
!!
!! SOURCE

subroutine findnq(nkbz,kbz,nsym,symrec,nqibz,timrev)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: timrev,nkbz,nsym
 integer,intent(out) :: nqibz
!arrays
 integer,intent(in) :: symrec(3,3,nsym)
 real(dp),intent(in) :: kbz(3,nkbz)

!Local variables ------------------------------
!scalars
 integer :: ifound,ik,isym,iq,istat,memory_exhausted,nqall,nqallm,itim
 character(len=500) :: msg
!arrays
 integer :: gtemp(3),g0(3)
 real(dp) :: qposs(3),qrot(3)
 real(dp),allocatable :: qall(:,:)
!************************************************************************

 ! === Infinite do-loop to be able to allocate sufficient memory ===
 nqallm=1000 ; memory_exhausted=0
 do 
  allocate(qall(3,nqallm), STAT=istat)
  ABI_CHECK(istat==0,'out-of-memory qall')
  nqall=0
  ! === Loop over all k-points in BZ, forming k-k1 ===
  do ik=1,nkbz
   qposs(:)=kbz(:,ik)-kbz(:,1)

   ! * Check whether this q (or its equivalent) has already been found.
   ! * Use spatial inversion instead of time reversal whenever possible.
   ifound=0
   do iq=1,nqall
    do itim=1,timrev
     do isym=1,nsym
      !FIXME this is for g95
      call dosym(REAL(symrec(:,:,isym),dp),itim,qall(:,iq),qrot)
      if (isamek(qrot,qposs,g0)) ifound=ifound+1
     end do
    end do
   end do

   if (ifound==0) then
    nqall=nqall+1
    !
    ! === If not yet found, check that the allocation is big enough ===
    if (nqall>nqallm) then
     memory_exhausted=1 ; deallocate(qall)
     nqallm=nqallm*2    ; EXIT ! Exit the do ik=1 loop
    end if
    ! * Add it to the list.
    qall(:,nqall)=qposs(:)
   end if
  end do

  if (memory_exhausted==0) EXIT
 end do !infinite loop

 deallocate(qall)
 nqibz=nqall

end subroutine findnq
!!***

!!****f* m_bz_mesh/findq
!! NAME
!! findq
!!
!! FUNCTION
!! Identify the q-points by which the k-points in BZ differ
!!
!! INPUTS
!!  gprimd(3,3)=dimensional reciprocal space primitive translations
!!  kbz(3,nkbz)=coordinates of k points in BZ
!!  timrev=2 if time-reversal symmetry is used, 1 otherwise
!!  nkbz=number of k points in Brillouin zone
!!  nsym=number of symmetry operations
!!  nqibz=number of q points in the IBZ by which k points differ (computed in findnq)
!!  symrec(3,3,nsym)=symmetry operations in reciprocal space
!!
!! OUTPUT
!!  qibz(3,nqibz)=coordinates of q points by which k points differ
!!
!! PARENTS
!!      m_bz_mesh,rdm
!!
!! CHILDREN
!!
!! SOURCE


subroutine findq(nkbz,kbz,nsym,symrec,gprimd,nqibz,qibz,timrev,avoid_zero)

 use defs_basis

 use m_numeric_tools, only : is_zero

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_32_util
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbz,nqibz,nsym,timrev
 logical,intent(in) :: avoid_zero
!arrays
 integer,intent(in) :: symrec(3,3,nsym)
 real(dp),intent(in) :: gprimd(3,3),kbz(3,nkbz)
 real(dp),intent(out) :: qibz(3,nqibz)

!Local variables ------------------------------
!scalars
 integer :: ii,ik,iq,iqp,isym,itim,jj,nq0
 real(dp) :: shift,tolq0
 logical :: found
 character(len=500) :: msg
!arrays
 integer :: g0(3),gtemp(3)
 real(dp) :: gmet(3,3),qposs(3),qrot(3)

!************************************************************************

 ! Compute reciprocal space metrics
 do ii=1,3
  gmet(ii,:)=gprimd(1,ii)*gprimd(1,:)+&
&            gprimd(2,ii)*gprimd(2,:)+&
&            gprimd(3,ii)*gprimd(3,:)
 end do

 tolq0=0.001_dp !old behaviour
 !
 ! === Loop over k-points in BZ, form k-k1 and translate in first BZ ===
 ! iq is the no. of q-points found, zero at the beginning
 iq=0
 do ik=1,nkbz
  qposs(:)=kbz(:,ik)-kbz(:,1)
  ! === Check whether this q (or its equivalent) has already been found ===
  ! * Use spatial inversion instead of time reversal whenever possible.
  found=.FALSE.
  do iqp=1,iq
   do itim=1,timrev
    do isym=1,nsym
     !FIXME this is for g95
     call dosym(REAL(symrec(:,:,isym),dp),itim,qibz(:,iqp),qrot)
     if (isamek(qrot,qposs,g0)) found=.TRUE.
    end do
   end do
  end do
  if (.not.found) then
   iq=iq+1
   if (iq>nqibz) then 
    write(msg,'(a,i5)')' iq > nqibz= ',nqibz
    MSG_BUG(msg)
   end if 
   qibz(:,iq)=qposs(:)
  end if
 end do

 if (iq/=nqibz) then 
  write(msg,'(2(a,i5))')' iq= ',iq,' /= nqibz= ',nqibz
  MSG_BUG(msg)
 end if 
 !
 ! Translate q-points (calculated for info) to 1st BZ
 !
!MG it seems that bz1 sometimes does not work, 
!in SiO2 for example I got 0.333333   -0.666667    0.000000
!it is better to use canon9 also because if an irred qpoint 
!lies outside the 1BZ then most probably we have to consider
!an umklapp G0 vector to reconstruct the full BZ. The 
!correct treatment of this case is not yet implemented yet, see csigme.F90
!Anyway I should check this new method because likely mrscr will complain
!The best idea consists in  writing a new subroutine canon10 
!wich reduce the q point in the interval [-1/2,1/2[ which is
!supposed to be the scope of bz1. Just to obtain the same results as 
!the automatic tests
!FIXME for the moment use old version, easy for debugging
!XG090614 : I have modified bz1 to provide q point in the interval [-1/2,1/2[

! do iq=1,nqibz
!  call bz1(qibz(:,iq),gtemp,gmet)
! end do

 do iq=1,nqibz
  do ii=1,3
   call canon9(qibz(ii,iq),qibz(ii,iq),shift)
  end do
 end do 

 !write(msg,'(a)')' q-points [reduced coordinates]'
 !call wrtout(std_out,msg,'COLL') 
 !call wrtout(ab_out,msg,'COLL')

 nq0=0
 do jj=1,nqibz
  if (is_zero(qibz(:,jj),tolq0).and.avoid_zero) then
   !qibz(1,jj)=0.000010
   !qibz(2,jj)=0.000020
   !qibz(3,jj)=0.000030
   qibz(:,jj)=GW_Q0_DEFAULT
   nq0=nq0+1
  end if
  !write(msg,'(3f12.6)') (qibz(ii,jj),ii=1,3)
  !call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')
 end do

! if (nq0/=1) then 
!  write(msg,'(a,i2,5a)')&
!&  ' Found ',nq0,' "small" qpoints ',ch10,&
!&  ' Check the q-mesh and, if it is correct, decrease the tolerance value ',ch10,&
!&  ' below which two points are considered equivalent. '
!  MSG_ERROR(msg)
! end if

end subroutine findq
!!***

!!****f* m_bz_mesh/findqg0
!! NAME
!! findqg0
!!
!! FUNCTION
!! Identify q + g0 = k - kp
!!
!! INPUTS
!!  kmkp(3)= k - kp input vector
!!  nqbz=number of q points
!!  qbz(3,nqbz)=coordinates of q points
!!  mG0(3)= For each reduced direction gives the max G0 component to account for umklapp processes
!!
!! OUTPUT
!!  iq=index of q vector in BZ table
!!  g0(3)=reciprocal space vector, to be used in igfft
!!
!! PARENTS
!!      csigme,rdm,set_gwdistrb,sigma
!!
!! CHILDREN
!!
!! SOURCE


subroutine findqg0(iq,g0,kmkp,nqbz,qbz,mG0)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nqbz
 integer,intent(out) :: iq
!arrays
 integer,intent(in) :: mG0(3)
 integer,intent(out) :: g0(3)
 real(dp),intent(in) :: kmkp(3),qbz(3,nqbz)

!Local variables-------------------------------
!FIXME if I use 1.0d-4 the jobs crash, should understand why
!scalars
 integer :: iqbz,jg01,jg02,jg03,nq0found
 real(dp) :: tolq0=1.0D-3
 character(len=500) :: msg
!arrays
 real(dp) :: kp(3),qpg0(3)

! *************************************************************************

 iq=0
 if (ALL(ABS(kmkp(:))<epsilon(one))) then
  ! === Find q close to 0 ===
  nq0found=0
  do iqbz=1,nqbz
   if (ALL(ABS(qbz(:,iqbz))<tolq0)) then 
    iq=iqbz
    nq0found=nq0found+1
   end if 
  end do
  if (iq==0) then 
   write(msg,'(a)')'Wrong list of q-points: q=0 not present. '
   MSG_BUG(msg)
  end if
  if (nq0found/=1) then 
   write(msg,'(3a)')&
&   ' The input q-list contains more than one small q-point',ch10,&
&   ' ACTION: reduce the value of tolq0 '
   MSG_BUG(msg)
  end if 
  g0(:)= 0 ; RETURN
 else
  ! === q is not zero, find q such as k-kp=q+G0 ===
  do iqbz=1,nqbz
   do jg01=-mG0(1),mG0(1)
    do jg02=-mG0(2),mG0(2)
     do jg03=-mG0(3),mG0(3)
      ! * Form q-G0 and check if it is the one.
      qpg0(1)= qbz(1,iqbz)+jg01
      qpg0(2)= qbz(2,iqbz)+jg02
      qpg0(3)= qbz(3,iqbz)+jg03
      if (ALL(ABS(qpg0(:)-kmkp(:))<GW_TOLQ)) then
       if (iq/=0) then
        write(std_out,*)iqbz,qbz(:,iqbz),jg01,jg02,jg03
        write(std_out,*)iq,qbz(:,iq),g0
        msg=' Found duplicated q+g0.'
        MSG_ERROR(msg)
       end if
       iq=iqbz
       g0(1)=jg01
       g0(2)=jg02
       g0(3)=jg03
      end if
     end do
    end do
   end do
  end do
  if (iq==0) then
   write(msg,'(3a,3f9.5)')&
&   ' q = k-kp+G0 not found. ',ch10,&
&   ' kmkp = ',kmkp(:) 
   MSG_ERROR(msg)
  end if
 end if

end subroutine findqg0

END MODULE m_bz_mesh
!!***
