!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_commutator_vkbr
!! NAME
!!  m_commutator_vkbr
!!
!! FUNCTION
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_commutator_vkbr

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors

 use m_gwdefs, only : czero_gw

 implicit none

 private

 public ::            &
&  apply_gradvnl,     & ! Return mat. elements of [r,V_nl] between two wavefunctions using Legendre polynomials.
&  apply_gradvnl_Ylm, & ! Return mat. elements of [r,V_nl] between two wavefunctions using spherical harmonics.
&  ccgradvnl,         & ! Compute (grad_K+grad_K') Vnl(K,K') using Legendre polynomials.
&  ccgradvnl_ylm,     & ! Compute (grad_K+grad_K') Vnl(K,K') using complex spherical harmonics.
&  destroy_Hur,       & ! Deallocate the dynamic memory in the HUr_commutator object.
&  nullify_Hur,       &
&  paw_inabla,        & ! FIXME change name
&  make_Hur_commutator

 type,public :: KB_form_factors
!scalars
  integer :: mpw       ! Maximum number of plane waves over k-points
  integer :: nkibz     ! Number of irreducible k-points.
  integer :: lnmax     ! Max. number of (l,n) components over all type of pseudos.
  integer :: ntypat    ! Number of type of atoms.

!arrays
  integer,pointer :: sign_dyad(:,:)  
  ! sign_dyad(lnmax,ntypat). 
  ! sign of the KB dyadic product.

  real(dp),pointer :: ff(:,:,:,:)   
  ! ff(npw,lnmax,ntypat,nkibz) 
  ! KB form factor.

  real(dp),pointer :: ffd(:,:,:,:)   
  ! ffd(npw,lnmax,ntypat,nkibz) 
  ! Derivative of ff wrt k+G of ff.
 end type KB_form_factors


 type,public :: HUr_commutator
  integer :: lmn_size
  integer :: lmn2_size
  integer :: nsppol
  !integer :: nsel

  integer,pointer :: ij_select(:,:,:)
  ! ijselect(lmn_size,lmn_size,nsppol) 
  ! Selection rules of ij matrix elements
  ! Do not take into account selectron on x-y-x for the moment

  real(dp),pointer :: commutator(:,:,:)
  ! commutator(3,nsel,nsppol)
 end type HUr_commutator

CONTAINS  !=========================================================================================================================
!!***

!!****f* m_commutator_vkbr/nullify_Hur
!! NAME
!! nullify_Hur
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      cchi0q0
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine nullify_Hur(Hur)

 use defs_basis

 implicit none

!Arguments ------------------------------------
 type(HUr_commutator),intent(inout) :: Hur(:)

!Local variables-------------------------------
 integer :: natom,iat
! *************************************************************************

 natom=SIZE(Hur)

 do iat=1,natom
  nullify(Hur(iat)%ij_select )
  nullify(Hur(iat)%commutator)
 end do

end subroutine nullify_Hur
!!***

!!****f* m_commutator_vkbr/destroy_Hur
!! NAME
!! destroy_Hur
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      cchi0q0
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine destroy_Hur(Hur)

 use defs_basis

 implicit none

!Arguments ------------------------------------
 type(HUr_commutator),intent(inout) :: Hur(:)

!Local variables-------------------------------
 integer :: natom,iat
! *************************************************************************

 natom=SIZE(Hur)

 do iat=1,natom
  if (associated(Hur(iat)%ij_select )) deallocate(Hur(iat)%ij_select )
  if (associated(Hur(iat)%commutator)) deallocate(Hur(iat)%commutator)
 end do

end subroutine destroy_Hur
!!***

!!****f* m_commutator_vkbr/vkb_init
!! NAME
!!  vkb_init
!!
!! FUNCTION
!!  Calculate KB form factors and derivatives required to evalute
!!  the matrix elements of the commutator [Vnl,r]-. 
!!  This term enters the expression for the oscillator strengths in 
!!  the optical limit q-->0. Pseudopotentials with more than one
!!  projector per angular channel are supported.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! TODO 
!!  Replace old implementation with this new routine. Matrix elements
!!  of the commutator should be calculated on-the-fly in screening only
!!  if really needed. This is the first step toward the elimination
!!  of the KSS file. Modifications in cchi0q0 are needed.
!!
!! PARENTS
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine vkb_init(Cryst,ntypat,Kmesh,Psps,mkmem,mpw,npwarr,ecut,kg,Vkb)

 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_56_recipspace
 use interfaces_65_nonlocal
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ntypat,mpw,mkmem
 real(dp),intent(in) :: ecut
 type(crystal_structure),     intent(in)    :: Cryst
 type(Pseudopotential_type),  intent(in)    :: Psps
 type(Bz_mesh_type),          intent(in)    :: Kmesh
 type(KB_form_factors),       intent(inout) :: Vkb
!arrays
 integer,target,intent(in) :: kg(3,mpw*mkmem),npwarr(Kmesh%nibz)

!Local variables ------------------------------
!scalars
 integer :: dimffnl,ider,idir,npw_k,ikibz,itypat,istat,nkpg
 integer :: il,ilmn,iln,iln0,nlmn,ikg
 real(dp) :: fact,effmass,ecutsm
 character(len=500) :: msg
!arrays
 integer,pointer :: kg_k(:,:)
 real(dp),pointer :: kpoint(:)
 real(dp),allocatable :: ffnl(:,:,:,:),kpg_dum(:,:),modkplusg(:)
 real(dp),allocatable :: ylm(:,:),ylm_gr(:,:,:),ylm_k(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 ! == Test the arguments ===
 call assert((Psps%usepaw==0),"You should not be here!",__FILE__,__LINE__)
 call assert((mkmem/=0),"mkmem==0 not implemented",__FILE__,__LINE__)
 !
 ! === Save Vkb dyadic sign (integer-valued) ===
 ! * Notice how the ordering is chosen correctly unlike in outkss.
 ! * More than one projector per angular channel is allowed but changes in cchi0q0 are needed
 !  allocate(vkbsign(Psps%mpsang,ntypat))  THIS THE OLD IMPLEMENTATION
 allocate(Vkb%sign_dyad(Psps%lnmax,ntypat)) 
 Vkb%sign_dyad(:,:)=0

 do itypat=1,ntypat
  nlmn=COUNT(Psps%indlmn(3,:,itypat)>0)
  iln0=0 
  do ilmn=1,nlmn
   iln=Psps%indlmn(5,ilmn,itypat)
   if (iln>iln0) then
    iln0=iln
    Vkb%sign_dyad(iln,itypat)=NINT(DSIGN(one,Psps%ekb(ilmn,itypat)))
   end if
  end do
 end do

 Vkb%nkibz  = Kmesh%nibz
 Vkb%lnmax  = Psps%lnmax
 Vkb%ntypat = ntypat
 Vkb%mpw    = mpw

 ! === Allocate KB form factor and derivative wrt k+G ===
 ! * Also here we use correct ordering for dimensions
 allocate(Vkb%ff (mpw,Vkb%lnmax,ntypat,Vkb%nkibz), stat=istat)
 allocate(Vkb%ffd(mpw,Vkb%lnmax,ntypat,Vkb%nkibz), stat=istat)
 Vkb%ff(:,:,:,:)=zero ; Vkb%ffd(:,:,:,:)=zero
 
 ider=1 ; dimffnl=2 ! To retrieve the first derivative.
 idir=0 ; nkpg=0 ; ikg=0

 do ikibz=1,Kmesh%nibz

  npw_k = npwarr(ikibz)
  kpoint => Kmesh%ibz(:,ikibz)
  kg_k => kg(:,1+ikg:ikg+npw_k)
  ikg=ikg+npw_k
  !
  ! Quantities used only if useylm==1
  allocate(ylm(npw_k,Psps%mpsang**2*Psps%useylm))
  allocate(ylm_gr(npw_k,3+6*(ider/2),Psps%mpsang**2*Psps%useylm))
  allocate(ylm_k(npw_k,Psps%mpsang**2*Psps%useylm))
  allocate(kpg_dum(npw_k,nkpg)) 

  allocate(ffnl(npw_k,dimffnl,Psps%lmnmax,ntypat))

  call mkffnl(Psps%dimekb,dimffnl,Psps%ekb,ffnl,Psps%ffspl,Cryst%gmet,Cryst%gprimd,ider,idir,Psps%indlmn,&
&  kg_k,kpg_dum,kpoint,Psps%lmnmax,Psps%lnmax,Psps%mpsang,Psps%mqgrid_ff,nkpg,npw_k,& 
&  ntypat,Psps%pspso,Psps%qgrid_ff,Cryst%rmet,Psps%usepaw,Psps%useylm,ylm_k,ylm_gr)

  deallocate(kpg_dum,ylm,ylm_gr,ylm_k)

  allocate(modkplusg(npw_k))
  effmass=one ; ecutsm=zero

  call mkkin(ecut,ecutsm,effmass,Cryst%gmet,kg_k,modkplusg,kpoint,npw_k)
                                                                         
  modkplusg(:)=SQRT(half/pi**2*modkplusg(:))
  modkplusg(:)=MAX(modkplusg(:),tol10)

  do itypat=1,ntypat
   nlmn=COUNT(Psps%indlmn(3,:,itypat)>0)
   iln0=0 
   do ilmn=1,nlmn
    il= Psps%indlmn(1,ilmn,itypat)+1
    iln=Psps%indlmn(5,ilmn,itypat)

    if (iln>iln0) then
     iln0=iln

     if (ABS(Psps%ekb(ilmn,itypat))>1.0d-10) then
      SELECT CASE (il)
      CASE (1)
       Vkb%ff (1:npw_k,iln,itypat,ikibz) = ffnl(:,1,ilmn,itypat)
       Vkb%ffd(1:npw_k,iln,itypat,ikibz) = ffnl(:,2,ilmn,itypat)*modkplusg(:)/two_pi

      CASE (2)
       Vkb%ff (1:npw_k,iln,itypat,ikibz) =   ffnl(:,1,ilmn,itypat)*modkplusg(:)
       Vkb%ffd(1:npw_k,iln,itypat,ikibz) = ((ffnl(:,2,ilmn,itypat)*modkplusg(:)**2)+&
&                                            ffnl(:,1,ilmn,itypat))/two_pi
      CASE (3)
       Vkb%ff (1:npw_k,iln,itypat,ikibz) =  ffnl(:,1,ilmn,itypat)*modkplusg(:)**2
       Vkb%ffd(1:npw_k,iln,itypat,ikibz) = (ffnl(:,2,ilmn,itypat)*modkplusg(:)**3+&
&                                         2*ffnl(:,1,ilmn,itypat)*modkplusg(:))/two_pi
      CASE (4)
       Vkb%ff (1:npw_k,iln,itypat,ikibz) =  ffnl(:,1,ilmn,itypat)*modkplusg(:)**3
       Vkb%ffd(1:npw_k,iln,itypat,ikibz) = (ffnl(:,2,ilmn,itypat)*modkplusg(:)**4+&
                                          3*ffnl(:,1,ilmn,itypat)*modkplusg(:)**2)/two_pi
      CASE DEFAULT
       msg=' l greater than g not implemented. '
       MSG_ERROR(msg)
      END SELECT

      fact = SQRT(four_pi/Cryst%ucvol*(2*il-1)*ABS(Psps%ekb(ilmn,itypat)))
      Vkb%ff (:,iln,itypat,ikibz) = fact * Vkb%ff (:,iln,itypat,ikibz)
      Vkb%ffd(:,iln,itypat,ikibz) = fact * Vkb%ffd(:,iln,itypat,ikibz)

     else ! ekb==0
      Vkb%ff (:,iln,itypat,ikibz)=zero
      Vkb%ffd(:,iln,itypat,ikibz)=zero
     end if

    end if
   end do
  end do

  deallocate(ffnl,modkplusg)

 end do !ikibz

 DBG_EXIT("COLL")

end subroutine vkb_init
!!***

!!****f* ABINIT/ccgradvnl_ylm
!! NAME
!! ccgradvnl_ylm
!!
!! FUNCTION
!!  Compute Vnl(K) and grad_K Vnl(K) three reciprocal lattice units components
!!  using spherical harmonics instead of Legendre polynomials
!!  Needed for chi0(q=0)
!!
!! COPYRIGHT
!!  Copyright (C) 2006-2009 ABINIT group (FB, MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  Cryst<crystal_structure>=Unit celle and symmetries
!!    %rprimd(3,3)=the three primitive vectors
!!    %ntypat=number of types of atoms
!!    %natom=number of atoms
!!    %xcart(3,natom)=cartesian coordinates of nuclei
!!    %typat(natom)=type of each atom
!!  gprimd(3,3)=dimensional primitive translations for reciprocal space ($\textrm{bohr}^{-1}$)
!!  gvec(3,npwwfn)=integer coordinates of each plane wave in reciprocal space
!!  kibz(3,nkibz)=coordinates of all k points in the irreducible Brillouin zone
!!  mpsang=1+maximum angular momentum for nonlocal pseudopotentials
!!  nkibz=number of k points in the irreducible Brillouin zone
!!  npwwfn=number of planewaves for wavefunctions
!!  vkb(npwwfn,ntypat,mpsang,nkibz)=KB projector function
!!  vkbd(npwwfn,ntypat,mpsang,nkibz)=derivative of the KB projector function in reciprocal space
!!  vkbsign(mpsang,ntypat)=sign of each KB dyadic product
!!
!! OUTPUT
!!  l_fnl(npwwfn,mpsang*mpsang,natom,nkibz),
!!  l_fnld(3,npwwfn,mpsang*mpsang,natom,nkibz)
!!
!! SIDE EFFECTS
!!
!! NOTES
!!  Subroutine taken from the EXC code  
!!  All the calculations are done in double precision, but the output arrays l_fnl and l_fnld 
!!  are in single precision, should use double precision after modification of the
!!  other subroutines 
!!  the subroutine does not work wity pseudo with more that one projector per angular state TODO
!!
!! PARENTS
!!      cchi0q0
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine ccgradvnl_ylm(npwwfn,nkibz,Cryst,gvec,gprimd,kibz,&
& mpsang,vkbsign,vkb,vkbd,l_fnl,l_fnld)

 use defs_datatypes
 use defs_abitypes
 use m_special_funcs, only : ylmc, ylmcd

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: mpsang,nkibz,npwwfn
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 integer,intent(in) :: gvec(3,npwwfn)
 real(dp),intent(in) :: gprimd(3,3) 
 real(dp),intent(in) :: kibz(3,nkibz)
 real(dp),intent(in) :: vkb(npwwfn,Cryst%ntypat,mpsang,nkibz)
 real(dp),intent(in) :: vkbd(npwwfn,Cryst%ntypat,mpsang,nkibz),vkbsign(mpsang,Cryst%ntypat)
 complex(gwpc),intent(out) :: l_fnl(npwwfn,mpsang**2,Cryst%natom,nkibz)
 complex(gwpc),intent(out) :: l_fnld(3,npwwfn,mpsang**2,Cryst%natom,nkibz)

!Local variables-------------------------------
!scalars
 integer,parameter :: nlx=4
 integer :: i,ia,ig,ik,il,im,iml,is,lmax
 real(dp),parameter :: ppad=tol8
 real(dp) :: cosphi,costh,factor,mkg,mkg2,sinphi,sinth,sq,xdotg
 complex(dpc) :: dphi,dth,sfac
 character(len=500) :: msg
!arrays
 real(dp) :: gcart(3),kcart(3),kg(3)
 real(dp) :: b1(3),b2(3),b3(3),a1(3),a2(3),a3(3)
 real(dp),pointer :: xcart(:,:)
 complex(dpc) :: dylmcart(3),dylmcrys(3),gradphi(3),gradth(3)
!************************************************************************

 DBG_ENTER("COLL")

 write(msg,'(a)')' ccgradvnl_ylm : limit q->0, including term <n,k|[Vnl,iqr]|n"k> using Y_lm'
 call wrtout(std_out,msg,'COLL')

 lmax=mpsang
 if (mpsang>nlx) then
  write(msg,'(3a)')&
&  ' Number of angular momentum components bigger than programmed.',ch10,&
&  ' Taking into account only s p d f ' 
  MSG_WARNING(msg)
  lmax=nlx
 end if

 a1=Cryst%rprimd(:,1) ; b1=two_pi*gprimd(:,1)
 a2=Cryst%rprimd(:,2) ; b2=two_pi*gprimd(:,2)
 a3=Cryst%rprimd(:,3) ; b3=two_pi*gprimd(:,3)

 xcart => Cryst%xcart
 !
 ! === Calculate Kleiman-Bylander factor and first derivative ===
 l_fnl=(0.0_gwp,0.0_gwp) ; l_fnld=(0.0_gwp,0.0_gwp)
 do ik=1,nkibz
  do ig=1,npwwfn
   !  
   ! === Get kcart =k+G in Cartesian coordinates ===
   kg(:)= kibz(:,ik)+REAL(gvec(:,ig))
   kcart(:)=kg(1)*b1(:)+kg(2)*b2(:)+kg(3)*b3(:)
   ! * Solve the problem with sinth=0. or sinphi=0
   if (ABS(kcart(2))<ppad) kcart(2)=kcart(2)+ppad

   mkg2=kcart(1)**2+kcart(2)**2+kcart(3)**2
   mkg=SQRT(mkg2)
   sq=SQRT(kcart(1)**2+kcart(2)**2)

   gcart(:)=  REAL(gvec(1,ig))*b1(:)&
&            +REAL(gvec(2,ig))*b2(:)&
&            +REAL(gvec(3,ig))*b3(:)

   ! === Calculate spherical coordinates (th,phi) ===
   costh = kcart(3)/mkg
   sinth = sq/mkg
   cosphi= kcart(1)/sq
   sinphi= kcart(2)/sq
   
   gradth(1)  = kcart(1)*kcart(3)/mkg**3/sinth
   gradth(2)  = kcart(2)*kcart(3)/mkg**3/sinth
   gradth(3)  = -(one/mkg-kcart(3)**2/mkg**3)/sinth
   gradphi(1) = -(one/sq - kcart(1)**2/sq**3)/sinphi
   gradphi(2) = kcart(2)*kcart(1)/sq**3/sinphi
   gradphi(3) = czero
   
   do ia=1,Cryst%natom
    is=Cryst%typat(ia)
    xdotg=gcart(1)*xcart(1,ia)+gcart(2)*xcart(2,ia)+gcart(3)*xcart(3,ia)
    ! Remember that in the GW code the reciprocal vectors 
    ! are defined such as a_i*b_j = 2pi delta_ij, no need to introduce 2pi
    sfac=CMPLX(COS(xdotg),SIN(xdotg)) 

    do il=1,lmax
     factor=SQRT(four_pi/REAL(2*(il-1)+1))
     do im= 1,2*(il-1)+1
      ! Index of im and il
      iml=im+(il-1)*(il-1)
      !     
      ! Calculate the first KB factor, note that l_fnl is simple precision complex
      l_fnl(ig,iml,ia,ik)=factor*sfac*ylmc(il-1,im-il,kcart)*vkb(ig,is,il,ik)*vkbsign(il,is)
      !     
      ! Calculate the second KB factor (involving first derivatives)
      ! dYlm/dK = dYlm/dth * grad_K th + dYlm/dphi + grad_K phi
      call ylmcd(il-1,im-il,kcart,dth,dphi)
      dylmcart(:) = dth*gradth(:) + dphi*gradphi(:)
      !     
      ! Cartesian to crystallographic axis
      dylmcrys(:) = ( dylmcart(1)*a1(:)&
&                    +dylmcart(2)*a2(:)&
&                    +dylmcart(3)*a3(:) ) /(two_pi)
      
      ! Note that l_fnld is simple precision complex, it could be possible use double precision
      do i=1,3
       l_fnld(i,ig,iml,ia,ik) = factor*sfac*&
&       ( kg(i)/mkg*ylmc(il-1,im-il,kcart)*vkbd(ig,is,il,ik) + dylmcrys(i)*vkb(ig,is,il,ik) )
      end do 

     end do !im
    end do !il
   end do !ia
  end do !ig
 end do !ik

 DBG_EXIT("COLL")

end subroutine ccgradvnl_ylm
!!***

!!****f* m_commutator_vkbr/apply_gradvnl_Ylm
!! NAME
!! apply_gradvnl_Ylm
!! 
!! FUNCTION
!!  Calculate the matrix elements of the gradient of the non-local operator 
!!  when Legendre polynomial are employed. Wavefunctions are supposed to be complex.
!!
!! INPUTS
!!  gradvnl(3,npwwfn,npwwfn)= the gradient at this k-point
!!  npwwfn=number of G vectors for wavefunctions
!!  wfg1(npwwfn),wfg1(npwwfn)= bra and ket in reciprocal space
!!  fnl(npwwfn,mpsang**2,natom)= Kleynmann-Bylander form factor for this k-point
!!  fnld(3,npwwfn,mpsang**2,natom)= Derivative of the KB form factor for this k-point
!!
!! OUTPUT
!!  
!! PARENTS
!!
!! CHILDREN
!!
subroutine apply_gradvnl_Ylm(npwwfn,wfg1,wfg2,natom,mpsang,fnl,fnld,res)
    
 use defs_basis

 implicit none

!Arguments ------------------------------------
 integer,intent(in) :: npwwfn,natom,mpsang
 complex(gwpc),intent(in) :: wfg1(npwwfn),wfg2(npwwfn)
 complex(gwpc),intent(in) :: fnl(npwwfn,mpsang**2,natom) 
 complex(gwpc),intent(in) :: fnld(3,npwwfn,mpsang**2,natom)
 complex(gwpc),intent(out) :: res(3)

!Local variables-------------------------------
 integer :: iat,ig,ilm
 complex(gwpc) :: cta1,cta4
 complex(gwpc) :: cta2(3),cta3(3)
! *************************************************************************
 res(:)=czero_gw

 do iat=1,natom
  do ilm=1,mpsang*mpsang
   cta1=czero_gw ; cta2(:)=czero_gw
   cta4=czero_gw ; cta3(:)=czero_gw
   ! === Here we take advantage of the property Y_(l-m)= (-i)^m Y_lm^* ===
   do ig=1,npwwfn
    cta1   = cta1    + wfg1(ig) * fnl(ig,ilm,iat)
    cta2(:)= cta2(:) + wfg2(ig) * fnld(:,ig,ilm,iat)
    cta3(:)= cta3(:) + wfg1(ig) * fnld(:,ig,ilm,iat)
    cta4   = cta4    + wfg2(ig) * fnl(ig,ilm,iat)
   end do
   res(:)= res(:) +CONJG(cta1)*cta2(:) +CONJG(cta3(:))*cta4
  end do !ilm
 end do !iat

end subroutine apply_gradvnl_Ylm
!!***

!!****f* ABINIT/ccgradvnl
!! NAME
!! ccgradvnl
!!
!! FUNCTION
!! Compute the (grad_K+grad_K') Vnl(K,K') (three reciprocal lattice units components)
!! Needed for chi0(q=0)
!!
!! COPYRIGHT
!! Copyright (C) 1999-2009 ABINIT group (GMR, VO, LR, RWG, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!  
!! INPUTS
!!  gvec(3,npwwfn)=integer coordinates of each plane wave in reciprocal space
!!  kibz(3,nkibz)=coordinates of all k points in the irreducible Brillouin zone
!!  mpsang=1+maximum angular momentum for nonlocal pseudopotentials
!!  Cryst<Crystal_structure>=data type gathering information on unit cell and symmetries
!!    %natom=number of atoms
!!    %typat(natom)=type of each atom
!!    %ntypat=number of types of atoms
!!    %xcart(3,natom)=cartesian coordinates of nuclei
!!  nkibz=number of k points in the irreducible Brillouin zone
!!  npwwfn=number of planewaves for wavefunctions
!!  vkb(npwwfn,ntypat,mpsang,nkibz)=KB projector function
!!  vkbd(npwwfn,ntypat,mpsang,nkibz)=derivative of the KB projector function in reciprocal space
!!  vkbsign(mpsang,ntypat)=sign of each KB dyadic product
!!
!! OUTPUT
!!  gradvnl =(grad_K + grad_K') Vnl(K,K') in reciprocal lattice units
!!  
!! PARENTS
!!      cchi0q0
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine ccgradvnl(npwwfn,nkibz,gvec,gprimd,kibz,Cryst,mpsang,vkbsign,vkb,vkbd,gradvnl)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: mpsang,nkibz,npwwfn
 type(Crystal_structure),intent(in) :: Cryst
!arrays
 integer,intent(in) :: gvec(3,npwwfn)
 real(dp),intent(in) :: kibz(3,nkibz),vkb(npwwfn,Cryst%ntypat,mpsang,nkibz)
 real(dp),intent(in) :: vkbd(npwwfn,Cryst%ntypat,mpsang,nkibz),vkbsign(mpsang,Cryst%ntypat)
 real(dp),intent(in) :: gprimd(3,3)
 complex(gwpc),intent(out) :: gradvnl(3,npwwfn,npwwfn,nkibz)

!Local variables ------------------------------
!scalars
 integer :: i,ia,ig,igd1,igd2,igd3,igp,ik,il,is,lmax
 real(dp) :: mkg,mkg2,mkgkgp,mkgp,mkgp2,rgdx,rgdy,rgdz,taugd,x,x2,x3,x4,x5,x6
 real(dp) :: x7,xcostheta
 complex(dpc) :: cs,ct
 character(len=500) :: msg
!arrays
 complex(dpc) :: sfac(Cryst%ntypat)
 integer,parameter :: nlx=8
 real(dp) :: pl(nlx),pld(nlx)
 real(dp) :: kg(3),kgp(3)
 real(dp) :: b1(3),b2(3),b3(3)
 real(dp),pointer :: xcart(:,:)
!************************************************************************

 write(msg,'(a)')' limit q->0, including term <n,k|[Vnl,iqr]|n"k>'
 call wrtout(std_out,msg,'COLL')

 b1=two_pi*gprimd(:,1)
 b2=two_pi*gprimd(:,2)
 b3=two_pi*gprimd(:,3)

 xcart => Cryst%xcart

 lmax=mpsang
 if(mpsang>nlx) then
  write(msg,'(3a)')&
& ' Number of angular momentum components bigger than programmed! ',ch10,&
& ' Taking into account only s p d f g h i j '
  MSG_WARNING(msg)
  lmax=nlx
 end if
 !
 ! Legendre polynomials and their first derivatives
 ! s p d f g h i j  so up to PL_7 = pl(8)
 !
 pl(1)  = one
 pld(1) = zero
 !pl(2) = costheta
 pld(2) = 1.0
 !pl(3) = 1/2 ( 3 costheta**2 - 1 )
 !pld(3)= 3 costheta

 gradvnl(:,:,:,:) = (0.0,0.0)

 do ik=1,nkibz

  do ig=1,npwwfn
   kg(:)=kibz(:,ik) + real(gvec(:,ig))
   mkg2=scpdt(kg,kg,b1,b2,b3)
   mkg=sqrt(mkg2)
   ! The next to solve the problem with k=Gamma.
   if (mkg < 0.0001) cycle

   do igp=1,npwwfn
    kgp(:)=kibz(:,ik) + real(gvec(:,igp))
    mkgp2=scpdt(kgp,kgp,b1,b2,b3)
    mkgp=sqrt(mkgp2)
    ! The next to solve the problem with k=Gamma.
    if (mkgp < 0.0001) cycle

    mkgkgp = mkg*mkgp
    xcostheta = scpdt(kg,kgp,b1,b2,b3) / mkgkgp
    x = xcostheta
    x2 = x * x
    x3 = x2 * x
    x4 = x3 * x
    x5 = x4 * x
    x6 = x5 * x
    x7 = x6 * x
    !
    ! Calculate legendre polynomial PL_0 = pl(1)
    pl(2) = x
    pl(3) = (3.0/2.0) * x2 - (1.0/2.0)
    pl(4) = (5.0/2.0) * x3 - (3.0/2.0) * x
    pl(5) = (35.0/8.0) * x4 - (30.0/8.0) * x2 + (3.0/8.0)
    pl(6) = (63.0/8.0) * x5 - (70.0/8.0) * x3 + (15.0/8.0) * x
    pl(7) = (231.0/16.0) * x6 - (315.0/16.0) * x4 + (105.0/16.0) * x2 - (5.0/16.0)
    pl(8) = (429.0/16.0) * x7 - (693.0/16.0) * x5 + (315.0/16.0) * x3 - (35.0/16.0) * x
    !
    ! Calculate legendre polynomial derivative
    pld(3) = 3.0 * x
    pld(4) = (15.0/2.0) * x2 - (3.0/2.0)
    pld(5) = (35.0/2.0) * x3 - (15.0/2.0) * x
    pld(6) = (315.0/8.0) * x4 - (210.0/8.0) * x2 + (15.0/8.0)
    pld(7) = (693.0/8.0) * x5 - (315.0/4.0) * x3 + (105.0/8.0) * x
    pld(8) = (3003.0/16.0) * x6 - (3465.0/16.0) * x4 + (945.0/16.0) * x2 - (35.0/16.0)

    igd1 = gvec(1,ig)-gvec(1,igp)
    igd2 = gvec(2,ig)-gvec(2,igp)
    igd3 = gvec(3,ig)-gvec(3,igp)
    rgdx = igd1*b1(1)+igd2*b2(1)+igd3*b3(1)
    rgdy = igd1*b1(2)+igd2*b2(2)+igd3*b3(2)
    rgdz = igd1*b1(3)+igd2*b2(3)+igd3*b3(3)

    do is=1,Cryst%ntypat
     sfac(is) = czero
     do ia=1,Cryst%natom
      if (Cryst%typat(ia)/=is) CYCLE
      taugd = rgdx*xcart(1,ia)+rgdy*xcart(2,ia)+ &
&      rgdz*xcart(3,ia)
      sfac(is) = sfac(is) + cmplx(cos(taugd),-sin(taugd))
     end do
    end do

    do i = 1, 3
     gradvnl(i,ig,igp,ik) = 0.0
     do is=1,Cryst%ntypat
      do il = 1, lmax
       ct =( kg(i)*(1/mkgkgp - xcostheta/mkg2 ) + &
&       kgp(i)*(1/mkgkgp - xcostheta/mkgp2 ) ) * &
&       pld(il) * vkbsign(il,is) * vkb(ig,is,il,ik) * vkb(igp,is,il,ik)
       
       cs = pl(il) * vkbsign(il,is) * &
&       ( kg(i)/mkg * vkbd(ig,is,il,ik) * vkb(igp,is,il,ik) + &
&       kgp(i)/mkgp * vkb(ig,is,il,ik) * vkbd(igp,is,il,ik) )
       
       gradvnl(i,ig,igp,ik) = gradvnl(i,ig,igp,ik) + sfac(is) * (ct + cs)
      end do !il
     end do !is
    end do !i

   end do !igp
  end do !ig
 end do !ik

 contains
!!***

!!****f* ccgradvnl/scpdt
!! NAME
!! scpdt
!!
!! FUNCTION
!! Compute scalar product of two vectors u and v in reciprocal space
!!
!! INPUTS
!!  b1(3),b2(3),b3(3)=the three primitive vectors in reciprocal space
!!  u(3),v(3)=the two vectors
!!
!! OUTPUT
!!  function scpdt=scalar product of u and v in reciprocal space
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function scpdt(u,v,b1,b2,b3)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 real(dp) :: scpdt
!arrays
 real(dp),intent(in) :: b1(3),b2(3),b3(3),u(3),v(3)
! *************************************************************************
 scpdt=&
& (u(1)*b1(1)+u(2)*b2(1)+u(3)*b3(1))*(v(1)*b1(1)+v(2)*b2(1)+v(3)*b3(1))+&
& (u(1)*b1(2)+u(2)*b2(2)+u(3)*b3(2))*(v(1)*b1(2)+v(2)*b2(2)+v(3)*b3(2))+&
& (u(1)*b1(3)+u(2)*b2(3)+u(3)*b3(3))*(v(1)*b1(3)+v(2)*b2(3)+v(3)*b3(3))

 end function scpdt

end subroutine ccgradvnl
!!***

!!****f* m_commutator_vkbr/apply_gradvnl
!! NAME
!! apply_gradvnl
!!
!! FUNCTION
!!  Calculate the matrix elements of the gradient of the non-local operator 
!!  when Legendre polynomial are employed. Wavefunctions are supposed to be complex.
!!
!! INPUTS
!!  gradvnl(3,npwwfn,npwwfn)= the gradient at this k-point
!!  npwwfn=number of G vectors for wavefunctions
!!  wfg1(npwwfn),wfg1(npwwfn)= bra and ket in reciprocal space
!!
!! OUTPUT
!!  res(3)= sum_{G_1,G_2} c(G_1)^\dagger c(G_2) V^{nl}_{G_1,G_2}
!!
!!
!! TODO  
!!  The following routines should be redefined as functions to allow inlining. 
!!  Unfortunately g95 seems to crash in cchi0q0 
!!
!! PARENTS 
!!      cchi0q0
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine apply_gradvnl(npwwfn,wfg1,wfg2,gradvnl,res) 
    
 use defs_basis

 implicit none

!Arguments ------------------------------------
 integer,intent(in) :: npwwfn
 complex(gwpc),intent(in) :: wfg1(npwwfn),wfg2(npwwfn)
 complex(gwpc),intent(in) :: gradvnl(3,npwwfn,npwwfn)
 complex(gwpc),intent(out) :: res(3)

!Local variables-------------------------------
 integer :: ig1,ig2 
 complex(gwpc) :: ct
! *************************************************************************
 
 res(:)=czero_gw
 do ig1=1,npwwfn
  do ig2=1,npwwfn
   ct=CONJG(wfg1(ig1))*wfg2(ig2)
   res(:)=res(:)+ct*gradvnl(:,ig1,ig2)
  end do
 end do

end subroutine apply_gradvnl
!!***

!!****f* m_commutator_vkbr/paw_inabla
!! NAME
!! paw_inabla
!!
!! FUNCTION
!!  Calculate the PAW onsite contribution to the matrix elements of the  i\nabla operator.
!!  in cartesian coordinates. Take also into account the contribution arising from the U 
!!  part of the Hamiltonian (if any)
!!
!! INPUTS
!!  natom=number of atoms in unit cell
!!  Pawtab(ntypat)=Only for PAW, TABulated data initialized at start
!!    %lmn_size Number of (l,m,n) elements for the paw basis
!!    %nabla_ij(3,lmn_size,lmn_size)) Onsite contribution
!!     <phi_i|nabla|phi_j>-<tphi_i|nabla|tphi_j> for each type
!!  HUr(natom)=Commutator of the LDA+U part of the Hamiltonian with the position operator.
!!
!! OUTPUT
!!  onsite(2,3)=Onsite contribution to  $i<ug1|\nabla|ug2>$
!!
!! PARENTS
!!      cchi0q0
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine paw_inabla(isppol,natom,typat,Pawtab,Cprj_b1,Cprj_b2,HUr,onsite)
    
 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,isppol
 real(dp),intent(out) :: onsite(2,3)
!arrays
 integer,intent(in) :: typat(natom)
 type(Pawtab_type),intent(in) :: Pawtab(:)
 type(Cprj_type),intent(in) :: Cprj_b1(natom),Cprj_b2(natom)
 type(HUr_commutator),intent(in) :: Hur(natom)

!Local variables-------------------------------
 integer :: ig,iatom,itypat,lmn_size,ilmn,jlmn,has_u,isel
 real(dp) :: re_p,im_p
!arrays
 real(dp) :: hurc_ij(3)
 real(dp),pointer :: nabla_ij(:,:,:)
! *************************************************************************
 
 onsite(:,:)=zero
 do iatom=1,natom 
  itypat=typat(iatom)
  lmn_size=Pawtab(itypat)%lmn_size
  nabla_ij => Pawtab(itypat)%nabla_ij(:,:,:) 
  has_u=Pawtab(itypat)%usepawu
  !
  !=== Unpacked loop over lmn channels ====
  do jlmn=1,lmn_size
   do ilmn=1,lmn_size

    re_p =  Cprj_b1(iatom)%cp(1,ilmn)*Cprj_b2(iatom)%cp(1,jlmn) &
&          +Cprj_b1(iatom)%cp(2,ilmn)*Cprj_b2(iatom)%cp(2,jlmn) 
 
    im_p =  Cprj_b1(iatom)%cp(1,ilmn)*Cprj_b2(iatom)%cp(2,jlmn) &
&          -Cprj_b1(iatom)%cp(2,ilmn)*Cprj_b2(iatom)%cp(1,jlmn)

    onsite(1,1)=onsite(1,1) - im_p*nabla_ij(1,ilmn,jlmn)
    onsite(1,2)=onsite(1,2) - im_p*nabla_ij(2,ilmn,jlmn)
    onsite(1,3)=onsite(1,3) - im_p*nabla_ij(3,ilmn,jlmn)

    onsite(2,1)=onsite(2,1) + re_p*nabla_ij(1,ilmn,jlmn)
    onsite(2,2)=onsite(2,2) + re_p*nabla_ij(2,ilmn,jlmn)
    onsite(2,3)=onsite(2,3) + re_p*nabla_ij(3,ilmn,jlmn)

    if (has_u/=0) then 
     ! here change the sign since in cchi0 I have to calculate -inabla
     ! this has to be done in a clear way
     isel=Hur(iatom)%ij_select(ilmn,jlmn,isppol)
     if (isel>0) then
      hurc_ij(:)=Hur(iatom)%commutator(:,isel,isppol)

      onsite(1,1)=onsite(1,1) + im_p*hurc_ij(1)
      onsite(1,2)=onsite(1,2) + im_p*hurc_ij(2)
      onsite(1,3)=onsite(1,3) + im_p*hurc_ij(3)
                                                           
      onsite(2,1)=onsite(2,1) - re_p*hurc_ij(1)
      onsite(2,2)=onsite(2,2) - re_p*hurc_ij(2)
      onsite(2,3)=onsite(2,3) - re_p*hurc_ij(3)
     end if
    end if

   end do !ilmn
  end do !jlmn
 end do !iatom

end subroutine paw_inabla
!!***

!!****f* m_commutator_vkbr/make_Hur_commutator
!! NAME
!! make_Hur_commutator
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      cchi0q0
!!
!! CHILDREN
!!      pawpupot,pawr
!!
!! SOURCE

subroutine make_Hur_commutator(nsppol,pawprtvol,Cryst,Psps,Pawtab,Pawang,Pawrad,Paw_ij,Hur)
    
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_66_paw
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsppol,pawprtvol
 type(Crystal_structure),intent(in)    :: Cryst
 type(Pawang_type),intent(in) :: Pawang
 type(Pseudopotential_type),intent(in) :: Psps
!arrays
 type(Pawtab_type),intent(in) :: Pawtab(Psps%ntypat*Psps%usepaw)
 type(Pawrad_type),intent(in) :: Pawrad(Psps%ntypat*Psps%usepaw)
 type(Paw_ij_type),intent(in) :: Paw_ij(Cryst%natom*Psps%usepaw)  
 type(HUr_commutator),intent(inout) :: Hur(Cryst%natom)

!Local variables-------------------------------
!scalars 
 integer :: iatom,ij_idx,isel,itypat,isppol,lmn2_size_max,lmn2_size,lmn_size,lpawu
 integer :: jlmn,jl,jm,jlm,jln,k0lmn,k0lm,k0ln,ilmn,il,im,ilm,iln
 integer :: klmn,klm,kln,m2,m1,left_lmn,right_lmn,tot_lmn,nmax
 real(dp) :: VUKS
!arrays
 integer :: nsel(3,nsppol)
 real(dp) :: sumr_ij(3)
 real(dp),allocatable :: rcart_onsite(:,:,:)
 real(dp),allocatable :: rij_tmp(:,:,:),vpawu(:,:)

! *************************************************************************

 ! * Get onsite matrix elements of the position operator.
 lmn2_size_max=MAXVAL(Pawtab(:)%lmn2_size) 
 allocate(rcart_onsite(3,lmn2_size_max,Cryst%natom))

 call pawr(Pawtab,Pawrad,Pawang,Psps,Cryst%natom,Cryst%ntypat,Cryst%typat,Cryst%xcart,lmn2_size_max,rcart_onsite)

 do iatom=1,Cryst%natom
  itypat=Cryst%typat(iatom)
  if (Pawtab(itypat)%usepawu==0) CYCLE
  lmn2_size=Pawtab(itypat)%lmn2_size
  lmn_size =Pawtab(itypat)%lmn_size
  lpawu=Pawtab(itypat)%lpawu
  Hur(iatom)%lmn2_size=lmn2_size
  Hur(iatom)%lmn_size =lmn_size
  Hur(iatom)%nsppol   =nsppol

  allocate(rij_tmp(3,lmn_size**2,nsppol))
  rij_tmp=zero
  allocate(vpawu(2*lpawu+1,2*lpawu+1))

  do isppol=1,nsppol ! spinor not implemented

   ! Get Vpawu^{\sigma}_{m1,m2}
   call pawpupot(isppol,Paw_ij(iatom),pawprtvol,Pawtab(itypat),vpawu,VUKS)

   ! === Loop on (jl,jm,jn) channels ===
   ij_idx=0
   do jlmn=1,lmn_size
    jl =Psps%indlmn(1,jlmn,itypat)
    jm =Psps%indlmn(2,jlmn,itypat)
    jlm=Psps%indlmn(4,jlmn,itypat)
    jln=Psps%indlmn(5,jlmn,itypat)

    k0lmn=jlmn*(jlmn-1)/2 
    k0lm =jlm *(jlm -1)/2
    k0ln =jln *(jln -1)/2
    !
    ! === Loop on (il,im,in) channels === 
    ! * Looping over all ij components. Elements are not symmetric.
    do ilmn=1,lmn_size
     il =Psps%indlmn(1,ilmn,itypat)
     im =Psps%indlmn(2,ilmn,itypat)
     ilm=Psps%indlmn(4,ilmn,itypat)
     iln=Psps%indlmn(5,ilmn,itypat)

     ij_idx=ij_idx+1

     ! === Selection rules ===
     if (il/=lpawu.and.jl/=lpawu) CYCLE 

     sumr_ij(:)=zero 
     do m2=1,2*lpawu+1
      do m1=1,2*lpawu+1

       if (m1==(im-lpawu-1).and.il==lpawu) then 
        left_lmn =ilmn-(il+im+1)+m2
        right_lmn=jlmn
        if (right_lmn>=left_lmn) then 
         tot_lmn=right_lmn*(right_lmn-1)/2 + left_lmn
        else 
         tot_lmn=left_lmn*(left_lmn-1)/2 + right_lmn
        end if
        sumr_ij=sumr_ij+vpawu(m1,m2)*rcart_onsite(:,tot_lmn,iatom)
       end if

       if (m2==(jm-lpawu-1).and.jl==lpawu) then 
        left_lmn =ilmn
        right_lmn=jlmn-(jl+jm+1)+m1
        if (right_lmn>=left_lmn) then 
         tot_lmn=right_lmn*(right_lmn-1)/2 + left_lmn
        else 
         tot_lmn=left_lmn*(left_lmn-1)/2 + right_lmn
        end if
        sumr_ij=sumr_ij+vpawu(m1,m2)*rcart_onsite(:,tot_lmn,iatom)
       end if

      end do !m1
     end do !m2

     rij_tmp(:,ij_idx,isppol)=sumr_ij(:)

    end do !ilmn
   end do !jlmn
  end do !isppol

  deallocate(vpawu)

  ! === Save values in packed form ===
  allocate(Hur(iatom)%ij_select(lmn_size,lmn_size,nsppol))
  Hur(iatom)%ij_select=0
  nsel(:,:)=COUNT(ABS(rij_tmp)>tol6,DIM=2)
  nmax=MAXVAL(nsel)
  allocate(Hur(iatom)%commutator(3,nmax,nsppol))
  do isppol=1,nsppol
   ij_idx=0
   isel  =0
   do jlmn=1,lmn_size
    do ilmn=1,lmn_size
     ij_idx=ij_idx+1
     if (ANY (ABS(rij_tmp(:,ij_idx,isppol))>tol6) ) then
      isel=isel+1
      Hur(iatom)%ij_select(ilmn,jlmn,isppol)=isel
      Hur(iatom)%commutator(:,isel,isppol)=rij_tmp(:,ij_idx,isppol)
     end if
    end do
   end do
  end do

  deallocate(rij_tmp)
 end do !iatom

 deallocate(rcart_onsite)

end subroutine make_Hur_commutator

END MODULE m_commutator_vkbr
!!***
