!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_gsphere
!! NAME
!!  m_gsphere
!!
!! FUNCTION
!!  This module defines two objects:
!!
!!   1) The Gsphere data type defining the set of G-vectors 
!!      centered on Gamma used to describe wavefunctions and W in the GW code. 
!!   2) The Gpairs_q object used to define the set of independent G1-G2 Fourier
!!      components of a two-point function which is invariant under the symmeties 
!!      of the space group
!!
!!  Methods used to initialize and destroy these two objects are defined here.
!!
!! COPYRIGHT
!! Copyright (C) 1999-2009 ABINIT group (MG, GMR, VO, LR, RWG, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! TODO 
!! Change name (gvectors >>> Gsphere)
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_gsphere

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors

 use m_gwdefs,   only : GW_TOLQ
 use m_geometry, only : normv

 implicit none

 private 

 public ::                &
&   init_Gvectors_type,   &  ! Initialize the G-sphere.
&   print_Gvectors,       &  ! Printout of basic dimensions.
&   destroy_Gvectors,     &  ! Free dynamics memory allocated in the object.
&   init_Gpairs_type,     &  ! Init the G-pairs object.
&   destroy_Gpairs_type,  &  ! Free the memory allocate in the G-pairs object. 
&   nullify_Gpairs_type

CONTAINS  !=========================================================================================================================
!!***

!!****f* m_gsphere/setup_G_rotation
!! NAME
!! setup_G_rotation
!!
!! FUNCTION
!! Set up tables indicating rotation of G-vectors.
!!
!! INPUTS
!! gvec(3,npw)=Coordinates of plane waves, supposed to be ordered in increasing modulus
!! timrev=2 if time reversal can be used, 1 otherwise.
!! nsym=Number of symmetry operations.
!! npw=Number of planewaves in the sphere.
!! symrec(3,3,nsym)=Symmetry operations in reciprocal space.
!! g2sh FIXME to be desribed
!! only_one_kpt
!! nsh
!! shlim
!!
!! OUTPUT
!!  grottb  (npw,2,nsym)= grottb(G,I,S) is the index of (SI) G in the array gvec. 
!!  grottbm1(npw,2,nsym)= index of IS^{-1} G.
!!
!! NOTES: 
!!  I is either the identity or the inversion (time reversal in reciprocal space).
!!  S is one of the symmetry operation in reciprocal space belonging to the Space group.
!!
!! PARENTS
!!      m_gsphere
!!
!! CHILDREN
!!
!! SOURCE

subroutine setup_G_rotation(only_one_kpt,nsym,symrec,timrev,npw,gvec,g2sh,nsh,shlim,grottb,grottbm1)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: npw,nsh,nsym,timrev
 logical,intent(in) :: only_one_kpt
!arrays
 integer,intent(in) :: g2sh(npw),gvec(3,npw),shlim(nsh+1),symrec(3,3,nsym)
 integer,intent(inout) :: grottb  (npw,timrev,nsym)
 integer,intent(inout) :: grottbm1(npw,timrev,nsym)

!Local variables ------------------------------
!scalars
 integer :: ee,ig1,ig2,ish1,isym,itim,ss
 logical :: found
 character(len=500) :: msg
!arrays
 integer :: gbase(3),grot(3)

!************************************************************************

 DBG_ENTER("COLL")
 !
 ! === Set up G-rotation table ===
 ! * This loop might be CPU consuming in isolated systems.
 !   Therefore we skip it in case of one single k-point.
 if (only_one_kpt) then
  ! As only_one_kpt is true, the only symmetry needed is identity
  do ig1=1,npw
   grottb  (ig1,1,1)=ig1
   grottbm1(ig1,1,1)=ig1
   !TODO check if inversion can enter somewhere!!!
  end do
 else 
  ! === Several k-points ===
  do ig1=1,npw
   ish1=g2sh(ig1) ; ss=shlim(ish1) ; ee=shlim(ish1+1)-1
   gbase(:)=gvec(:,ig1)

   do itim=1,timrev
    do isym=1,nsym
     grot=(3-2*itim)*MATMUL(symrec(:,:,isym),gbase)
     found=.FALSE.
     ! * Loop on the shell of ig1 to speed up the search.
     do ig2=ss,ee 
      if (ALL(ABS(grot(:)-gvec(:,ig2))==0)) then
       found=.TRUE.
       grottb  (ig1,itim,isym)=ig2
       grottbm1(ig2,itim,isym)=ig1
      end if
     end do 
     if (.not.found) then
      write(msg,'(3a,i5,a,i5,1x,2(3i5,a),a,i3,a,i3)')&
&      ' G-shell not closed',ch10,&
&      '  Initial G vector ',ig1,'/',npw,gbase(:),' Rotated G vector ',grot(:),ch10,&
&      '  Through sym ',isym,' and itim ',itim
      MSG_ERROR(msg)
     end if
    end do 
   end do 

  end do !ig1 
 end if !only_one_kpt

 DBG_EXIT("COLL")

end subroutine setup_G_rotation
!!***

!!****f* m_gsphere/init_Gvectors_type 
!! NAME
!! init_Gvectors_type
!!
!! FUNCTION
!!  Initialize a Gvectors data type
!!
!! NOTES
!!  gvec are supposed to be ordered with increasing norm.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr,setup_screening,setup_sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_Gvectors_type(only_one_kpt,Gsph,Cryst,ng,gvec,gmet,gprimd)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ng
 logical,intent(in) :: only_one_kpt
 type(Crystal_structure),intent(in) :: Cryst
 type(Gvectors_type),intent(out) :: Gsph
!arrays
 integer,intent(in) :: gvec(3,ng)
 real(dp),intent(in) :: gmet(3,3),gprimd(3,3)

!Local variables-------------------------------
!scalars
 integer :: ig,istat,isym,nsh,nsym,timrev
 real(dp) :: eps,norm,norm_old
 logical :: ltest
 character(len=500) :: msg
!arrays
 integer :: sg(3)
 integer,allocatable :: shlim(:)
 integer,pointer :: symrec(:,:,:)
 real(dp),allocatable :: shlen(:)
 real(dp),pointer :: tnons(:,:)

!************************************************************************

 DBG_ENTER("COLL")

 call nullify_Gvectors_(Gsph)

 ! === Copy info on symmetries ===
 nsym   =  Cryst%nsym
 timrev =  Cryst%timrev
 symrec => Cryst%symrec
 tnons  => Cryst%tnons
 !
 ! === Initialize the object ===
 Gsph%ng     = ng
 Gsph%nsym   = nsym
 Gsph%timrev = timrev

 Gsph%gmet   = gmet
 Gsph%gprimd = gprimd

 allocate(Gsph%gvec(3,ng)) 
 Gsph%gvec(:,:)=gvec(:,:)
 !
 ! === Calculate phase exp{-i2\pi G.\tau} ===
 allocate(Gsph%phmGt(ng,nsym))
 do ig=1,ng
  do isym=1,nsym
   Gsph%phmGt(ig,isym)=EXP(-j_dpc*two_pi*DOT_PRODUCT(gvec(:,ig),tnons(:,isym)))
  end do 
 end do
 !
 ! === Calculate phase phsgt= exp{-i2\pi SG\cdot t} ===
 ! TODO Here we can store only one of this arrays but I have to rewrite screeening!
 allocate(Gsph%phmSGt(ng,nsym))
 do ig=1,ng
  do isym=1,nsym
   sg(:)=MATMUL(symrec(:,:,isym),gvec(:,ig))
   Gsph%phmSGt(ig,isym)=EXP(-j_dpc*two_pi*DOT_PRODUCT(sg,tnons(:,isym)))
  end do
 end do
 !
 ! === Calculate number of shells and corresponding starting index ===
 ! * Shells are useful to speed up search algorithms see e.g setup_G_rotation.
 ! * The last shell ends at ng+1, thus gvec is supposed to be closed.
 ltest=ALL(gvec(:,1)==0)
 call assert(ltest,'First G should be 0',__FILE__,__LINE__)

 allocate(Gsph%g2sh(ng)) ; Gsph%g2sh(1)=1 ! This table is useful if we dont loop over shell

 ! For each shell, gives the index of the initial G-vector.
 allocate(shlim(ng+1))  
 shlim(1)=1 

 ! For each shell, gives the radius of the shell.
 allocate(shlen(ng))  
 shlen(1)=zero 

 nsh=1 ; norm_old=zero
 do ig=2,ng
  norm=two_pi*SQRT(DOT_PRODUCT(gvec(:,ig),MATMUL(gmet,gvec(:,ig))))
  eps=norm*tol8
  if (ABS(norm-norm_old)>eps) then 
   norm_old=norm
   nsh=nsh+1
   shlim(nsh)=ig
   shlen(nsh)=norm 
  end if
  Gsph%g2sh(ig)=nsh
 end do
 shlim(nsh+1)=ng+1

 ! === Save info on the shells ===
 Gsph%nsh=nsh
 allocate(Gsph%shlim(nsh+1)) ; Gsph%shlim=shlim(1:nsh+1)
 allocate(Gsph%shlen(nsh  )) ; Gsph%shlen=shlen(1:nsh)
 deallocate(shlim,shlen)
 !
 ! === Calculate tables for rotated G"s ===
 allocate(Gsph%rottb  (ng,timrev,nsym),STAT=istat)   
 allocate(Gsph%rottbm1(ng,timrev,nsym),STAT=istat) 

 call setup_G_rotation(only_one_kpt,nsym,symrec,timrev,Gsph%ng,Gsph%gvec,&
& Gsph%g2sh,Gsph%nsh,Gsph%shlim,Gsph%rottb,Gsph%rottbm1)

 call print_Gvectors(Gsph,unit=std_out,prtvol=1)

 DBG_EXIT("COLL")

end subroutine init_Gvectors_type
!!***

!!****f* m_gsphere/print_Gvectors
!! NAME
!! print_Gvectors
!!
!! FUNCTION
!!  Print the content of a gvectors data type
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      m_gsphere
!!
!! CHILDREN
!!
!! SOURCE

subroutine print_Gvectors(Gsph,unit,prtvol,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: prtvol,unit
 character(len=4),intent(in),optional :: mode_paral
 type(Gvectors_type),intent(in) :: Gsph

!Local variables-------------------------------
!scalars
 integer :: ii,ish,nsc,unt,verbose
 real(dp) :: fact,kin
 character(len=100) :: fmt
 character(len=4) :: mode
 character(len=500) :: msg

! *************************************************************************

 unt    =std_out ; if (PRESENT(unit      )) unt    =unit
 verbose=0       ; if (PRESENT(prtvol    )) verbose=prtvol
 mode   ='COLL'  ; if (PRESENT(mode_paral)) mode   =mode_paral

 write(msg,'(3a,2(a,i8,a))')ch10,&
& ' ==== Info on the G-sphere ==== ',ch10,&
& '  Number of G vectors ... ',Gsph%ng,ch10,&
& '  Number of shells ...... ',Gsph%nsh,ch10
 call wrtout(unt,msg,mode)

 SELECT CASE (Gsph%timrev) 
 CASE (1)
  write(msg,'(a)')' Time reversal symmetry is used'
 CASE (2)
  write(msg,'(a)')' Time reversal symmetry is used'
 CASE DEFAULT
  MSG_BUG("Wrong timrev")
 END SELECT
 call wrtout(unt,msg,mode)

 if (verbose/=0) then 
  fact=half*two_pi**2
  write(msg,'(a)')' Shell   Tot no. of Gs   Cutoff [Ha]'
  call wrtout(unt,msg,mode)
  do ish=1,Gsph%nsh
   nsc=Gsph%shlim(ish+1)-1 
   kin=half*Gsph%shlen(ish)**2
   write(msg,'(2x,i4,10x,i6,5x,f8.3)')ish,nsc,kin
   call wrtout(unt,msg,'COLL')
  end do
  write(msg,'(a)')ch10 ; call wrtout(unt,msg,mode)
 end if

end subroutine print_Gvectors 
!!***

!!****f* m_gsphere/destroy_Gvectors
!! NAME
!! destroy_Gvectors
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_Gvectors(Gsph)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(Gvectors_type),intent(inout) :: Gsph

! *************************************************************************

 DBG_ENTER("COLL")

 if (associated(Gsph%g2sh   ))  deallocate(Gsph%g2sh   )
 if (associated(Gsph%gvec   ))  deallocate(Gsph%gvec   )
 if (associated(Gsph%rottb  ))  deallocate(Gsph%rottb  )
 if (associated(Gsph%rottbm1))  deallocate(Gsph%rottbm1)
 if (associated(Gsph%shlim  ))  deallocate(Gsph%shlim  )
 if (associated(Gsph%shlen  ))  deallocate(Gsph%shlen  )
 if (associated(Gsph%phmGt  ))  deallocate(Gsph%phmGt  )
 if (associated(Gsph%phmSGt ))  deallocate(Gsph%phmSGt )

 DBG_EXIT("COLL")

end subroutine destroy_Gvectors
!!***

!!****f* m_gsphere/nullify_Gvectors_
!! NAME
!! nullify_Gvectors_
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      m_gsphere
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_Gvectors_(Gsph)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(Gvectors_type),intent(inout) :: Gsph

! *************************************************************************

 nullify(Gsph%g2sh   )
 nullify(Gsph%gvec   )
 nullify(Gsph%rottb  )
 nullify(Gsph%rottbm1)
 nullify(Gsph%shlim  )
 nullify(Gsph%shlen  )
 nullify(Gsph%phmGt  )
 nullify(Gsph%phmSGt )

end subroutine nullify_Gvectors_
!!***

!!****f* m_gsphere/findggp_
!! NAME
!! findggp_
!!
!! FUNCTION
!!  Fing the independent (G1,G2) pairs that are sufficient to reconstruct using 
!!  symmetry properties the Fourier components f_q(G1,G2) of a two-point function 
!!  which has the same symmetry of the crystal 
!!
!! INPUTS
!!  Gsphere<Gvectors_type>=Info on the G-sphere
!!   %ng=number of G vectors in the f matrix
!!   %gmet(3,3)=metric tensor in reciprocal space
!!   %gvec(3,ng)=G vectors in reduced coordinates
!!  nsym=number of symmetry operations
!!  qpt(3)=q-point in reciprocal coordinated
!!  symrec(3,3,nsym)=symmetry operations in reciprocal space
!!
!! OUTPUT
!!  Gpairs_q<Gpairs_type>= Structure containing information on the irreducible pairs
!!   %niggp=nuber of independent (G1,G1) pairs   
!!   %fp2ip(2,ng,ng)= for given T1 and T1 reports the sequential index of 
!!     the indendent pair (G1,G2) such as (T1,T2)= S (G1,G2) 
!!   %fptabo(ng,ng)= index of the symmetry operation S in the array symrec such as (T1,T2)= S (G1,G2)
!!   %ip2fp(2,niggp)= index of G1 and G2 in the array gvec for each niggp independent pair. 
!!
!! NOTES
!!
!! PARENTS
!!      m_gsphere
!!
!! CHILDREN
!!
!! SOURCE

subroutine findggp_(nsym,symrec,Gsphere,qpt,Gpairs_q)
    
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_28_numeric_noabirule
 use interfaces_56_recipspace
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsym
 type(Gvectors_type),intent(in) :: Gsphere
 type(Gpairs_type),intent(inout) :: Gpairs_q 
!arrays
 integer,intent(in) :: symrec(3,3,nsym)
 real(dp),intent(in) :: qpt(3) 

!Local variables-------------------------------
!scalars
 integer :: igpi1,igpi2,isym,itim,ig,ig1,ig2,igpi,ng,niggp
 integer :: dummy,timrev_,iid,ic,istat
 real(dp) :: diff2,norm1,norm2,eps,eps1,eps2
 real(dp) :: a1,a2,a3,d1,d2,d3,g12,g23,g31
 logical :: found,found_identity
 character(len=500) :: msg           
!arrays
 integer :: identity(3,3),symqpt(4,2,nsym),ltg(nsym)
 integer :: gposs1(3),gposs2(3),gxx1(3),gxx2(3)
 integer, allocatable :: iperm(:)
 real(dp) :: gmet(3,3)
 real(dp),allocatable :: gnorm(:),gdiff2(:)
 integer,allocatable :: ip2fp(:,:)
 integer,pointer :: gvec(:,:)
 character(len=50),parameter :: my_name='findggp_'

!************************************************************************

 DBG_ENTER("COLL")
 !
 ! === Check the presence of identity and save its index ===
 identity(:,:)=RESHAPE((/1,0,0,0,1,0,0,0,1/),(/3,3/))
 found_identity=.FALSE.
 do isym=1,nsym
  if (ALL(symrec(:,:,isym)==identity)) then 
   iid=isym
   found_identity=.TRUE. ; EXIT
  end if
 end do 

 if (.not.found_identity) then 
  write(msg,'(3a)')&
&  ' Only the inversion was found in the set of symmetries read from the KSS file ',ch10,&
&  ' Likely you are using an old version of the KSS file! '
  MSG_ERROR(msg)
 else 
  write(msg,'(a,i2)')' findggp_ : found identity with index: ',iid
  call wrtout(std_out,msg,'COLL')
 end if 
 !
 ! === Only at Gamma time-reversal can be included ===
 timrev_=1 ; if (ALL(ABS(qpt)<GW_TOLQ).and.Gsphere%timrev==2) timrev_=2

 write(msg,'(2a,3f8.5)')ch10,' Analyzing symmetries at point : ',qpt(:)
 if (timrev_==2) msg=TRIM(msg)//' (including time-reversal) '
 call wrtout(std_out,msg,'COLL')
 !
 ! === Find operations in the little group === 
 ! ltg is 1 if the symmetry  preserves q with a zero umklapp vector
 call symq3(nsym,qpt,symqpt,symrec,dummy,prtvol=0)

 ltg(:)=0 
 do itim=1,timrev_
  do isym=1,nsym
   if (symqpt(4,itim,isym)==1.and.ALL(symqpt(1:3,itim,isym)==0)) ltg(isym)=1
  end do 
 end do

 if (SUM(ltg)==1.and.timrev_==1) then 
  ! === In this case do not allocate anything, set niggp=ng**2 and exit ===
  write(msg,'(a)')' findggp_ : not enough symmetries to reduce the number of G-pairs'
  call wrtout(std_out,msg,'COLL')
  Gpairs_q%niggp=Gsphere%ng**2
  RETURN
 end if 
 !
 ! === We can use symmetries to reduce the number of pairs ===
 ng=Gsphere%ng
 gmet =  Gsphere%gmet(:,:)
 gvec => Gsphere%gvec(1:3,1:ng)

 allocate(Gpairs_q%fp2ip(2,ng,ng), STAT=istat)
 ABI_CHECK(istat==0,'out-of-memory fp2ip')

 allocate(Gpairs_q%fptabo(ng,ng), STAT=istat)
 ABI_CHECK(istat==0,'out-of-memory fptabo')

 allocate(ip2fp(2,ng**2), STAT=istat) ! still do not know number of irred pairs 
 ABI_CHECK(istat==0,'out-of-memory ip2fp')
 !
 ! === Precalculate G norm to speed the loop over G-pairs ===
 ! Use the scalar-valued function to work around a bug in sunstudio12.
 allocate(gnorm(ng))
 do ig=1,ng
  gnorm(ig)=normv(Gsphere%gvec(:,ig),gmet,'G')
  !gnorm(ig)=normv(DBLE(Gsphere%gvec(:,ig)),gmet,'G')
 end do
 !
 ! === Precalculate |G1-G2|^2 square to speed loop over G pairs ===
 allocate(gdiff2(ng**2), STAT=istat)
 ABI_CHECK(istat==0,'out-of-memory gdiff2')

 allocate(iperm(ng**2), STAT=istat)
 ABI_CHECK(istat==0,'out-of-memory iperm')

 ic=1
 g12=two*gmet(1,2) ; g23=two*gmet(2,3) ; g31=two*gmet(3,1)
 do ig1=1,ng
  a1=gvec(1,ig1) ; a2=gvec(2,ig1) ; a3=gvec(3,ig1)
  do ig2=1,ng
   d1=gvec(1,ig2)-a1 ; d2=gvec(2,ig2)-a2 ; d3=gvec(3,ig2)-a3
   gdiff2(ic)= d1*(gmet(1,1)*d1+g12*d2) &
&             +d2*(gmet(2,2)*d2+g23*d3) &
&             +d3*(gmet(3,3)*d3+g31*d1)
   iperm(ic)=ic ; ic=ic+1
  end do
 end do

 ! === Sort differences ===
 call sort_dp(ng**2,gdiff2,iperm,tol10)
 !
 ! === Loop over all all possible pairs (G1,G2), finding the irreducible ones === 
 ! * Note that the pairs are addressed by ascending order of the norm of G1
 niggp=0 ! no. of pairs found
 do ic=1,ng**2

  ig1=(iperm(ic)-1)/ng+1
  ig2=iperm(ic)-(ig1-1)*ng
  diff2=gdiff2(ic)

  norm1=gnorm(ig1) ; eps1=tol8*norm1
  norm2=gnorm(ig2) ; eps2=tol8*norm2
  gposs1(:)=gvec(:,ig1)
  gposs2(:)=gvec(:,ig2)
  !
  ! === Check if this pair is the image through the same operation of a pair already found ===
  ! * Consider only vectors with the same length
  found=.FALSE.
  if (niggp>0) then
   ip : do igpi=niggp,1,-1

    if (diff2-gdiff2(igpi)>eps1+eps2+48*tol10) EXIT  ! This makes the algorithm scale as N**2
    igpi1=ip2fp(1,igpi) ; if (ABS(norm1-gnorm(igpi1))>eps1) CYCLE
    igpi2=ip2fp(2,igpi) ; if (ABS(norm2-gnorm(igpi2))>eps2) CYCLE

    do itim=1,timrev_
     do isym=1,nsym
      ! === Calculate IS G1 and IS G2 ===
      ! * Do this only for operations in the little group such as Sq=q
      ! TODO Calculate SG outside the loop, requires more memory but should be faster!
      !       avoid check on ltg, could make a table full==> little_group
      if (ltg(isym)==1) then 
       gxx1=(3-2*itim)*MATMUL(symrec(:,:,isym),gvec(:,igpi1))
       if (ALL(ABS(gxx1-gposs1)==0)) then 
        gxx2=(3-2*itim)*MATMUL(symrec(:,:,isym),gvec(:,igpi2))
        if (ALL(ABS(gxx2-gposs2)==0)) then 
         found=.TRUE.
         Gpairs_q%fp2ip(1,ig1,ig2)=igpi1
         Gpairs_q%fp2ip(2,ig1,ig2)=igpi2
         Gpairs_q%fptabo(ig1,ig2)=isym*(3-2*itim) ! Minus is time-reversal is considered (Only at Gamma)
         EXIT ip
        end if
       end if 
      end if 
     end do !isym 
    end do !itim
   end do ip
  end if
  
  if (.not.found) then
   ! === Increment counter and fill tables ===
   niggp=niggp+1
   gdiff2(niggp)=diff2
   ip2fp(1,niggp)=ig1
   ip2fp(2,niggp)=ig2
   Gpairs_q%fp2ip(1,ig1,ig2)=ig1
   Gpairs_q%fp2ip(2,ig1,ig2)=ig2
   Gpairs_q%fptabo(ig1,ig2)=iid  ! This irreducible pair comes from the identity!
  end if
 end do !ic

 deallocate(gnorm)
 deallocate(gdiff2,iperm)

 if (niggp>ng**2) then 
  MSG_BUG('niggp>ng**2')
 end if 

 write(msg,'(2a,i8,a,i8,a)')ch10,&
& ' findggp_ : number of independent (G,G'') pairs found = ',niggp,' / ',ng**2,ch10
 call wrtout(std_out,msg,'COLL')
 !
 ! === Save finale values ===
 Gpairs_q%niggp=niggp
 allocate(Gpairs_q%ip2fp(2,niggp),STAT=istat)
 ABI_CHECK(istat==0,'out-of-memory ip2fp')

 Gpairs_q%ip2fp=ip2fp(1:2,1:niggp)
 deallocate(ip2fp)

 call DEBUG_Gpairs__(Gpairs_q,Gsphere,nsym,symrec)
 DBG_EXIT("COLL")

end subroutine findggp_
!!***

!!****f* m_gsphere/init_Gpairs_type
!! NAME
!! init_Gpairs_type
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr,screening
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_Gpairs_type(Gpairs_q,qpt,Gsphere,Cryst)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(Gvectors_type),intent(in) :: Gsphere
 type(Crystal_structure),intent(in) :: Cryst
 type(Gpairs_type),intent(out) :: Gpairs_q
!arrays
 real(dp),intent(in) :: qpt(3)

!Local variables ------------------------------
!scalars
 integer :: ng,istat,nsym
 integer,pointer :: symrec(:,:,:)
!************************************************************************

 call destroy_Gpairs_type(Gpairs_q) 

 nsym   =  Cryst%nsym
 symrec => Cryst%symrec

 ng=Gsphere%ng
 !
 ! === Dimensions ===
 Gpairs_q%ng            = ng
 Gpairs_q%ngpairs       = ng**2
 Gpairs_q%nsym          = Gsphere%nsym
 Gpairs_q%timrev        = Gsphere%timrev
 Gpairs_q%can_use_timrev= .FALSE.

 ! === The point under investigation ===
 Gpairs_q%qpt=qpt
 if (ALL(ABS(qpt)<GW_TOLQ).and.Gsphere%timrev==2) Gpairs_q%can_use_timrev=.TRUE.
 
 call findggp_(nsym,symrec,Gsphere,qpt,Gpairs_q)

end subroutine init_Gpairs_type
!!***

!!****f* m_gsphere/destroy_Gpairs_type
!! NAME
!! destroy_Gpairs_type
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      m_gsphere,mrgscr,screening
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_Gpairs_type(Gpairs_q)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(Gpairs_type),intent(inout) :: Gpairs_q

!************************************************************************
 
 DBG_ENTER("COLL")

 if (associated(Gpairs_q%fp2ip )) deallocate(Gpairs_q%fp2ip )
 if (associated(Gpairs_q%fptabo)) deallocate(Gpairs_q%fptabo)
 if (associated(Gpairs_q%ip2fp )) deallocate(Gpairs_q%ip2fp )

 DBG_EXIT("COLL")

end subroutine destroy_Gpairs_type
!!***

!!****f* m_gsphere/nullify_Gpairs_type
!! NAME
!! nullify_Gpairs_type
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr,screening
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_Gpairs_type(Gpairs_q)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(Gpairs_type),intent(inout) :: Gpairs_q

!************************************************************************

 nullify(Gpairs_q%fp2ip )
 nullify(Gpairs_q%fptabo)
 nullify(Gpairs_q%ip2fp )

end subroutine nullify_Gpairs_type
!!***

!!****f* m_gsphere/DEBUG_Gpairs__
!! NAME
!! DEBUG_Gpairs__
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      m_gsphere
!!
!! CHILDREN
!!
!! SOURCE

subroutine DEBUG_Gpairs__(Gpairs_q,Gsphere,nsym,symrec)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsym
 type(Gpairs_type),intent(in) :: Gpairs_q
 type(Gvectors_type),intent(in) :: Gsphere
!arrays
 integer,intent(in) :: symrec(3,3,nsym)

!Local variables-------------------------------
!scalars
 integer :: ig1,ig2,ng,ir1,ir2,isym,itim
 character(len=500) :: msg           
!arrays
 integer :: girr1(3),girr2(3),gxx1(3),gxx2(3),gtest1(3),gtest2(3)
 integer,pointer :: gvec(:,:)

!************************************************************************

 ng   =  Gpairs_q%ng
 gvec => Gsphere%gvec

 do ig1=1,ng
  gtest1=gvec(:,ig1)
  do ig2=1,ng
   gtest2=gvec(:,ig2)
   ir1=Gpairs_q%fp2ip(1,ig1,ig2)
   ir2=Gpairs_q%fp2ip(2,ig1,ig2)
   girr1=gvec(:,ir1)
   girr2=gvec(:,ir2)
   isym=Gpairs_q%fptabo(ig1,ig2) ; itim=1 
   if (isym<0) then 
    isym=-isym
    itim=2
   end if
   gxx1=(3-2*itim)*MATMUL(symrec(:,:,isym),girr1)
   gxx2=(3-2*itim)*MATMUL(symrec(:,:,isym),girr2)
   if (ANY((gtest1-gxx1)/=0).or.ANY((gtest2-gxx2)/=0)) then 
     write(msg,'(a,2i8,a)')' G1-G2 pair',ig1,ig2,' has no corresponding irreducible pair ' 
     write(*,*)' G1, G2 ',gtest1,gtest2
     write(*,*)' independent pair? ',girr1,girr2
     write(*,*)' operation ',isym,symrec(:,:,isym),' with itim ',itim
     MSG_BUG(msg)
    end if 
   end do
  end do 

end subroutine DEBUG_Gpairs__

END MODULE m_gsphere
!!***
