!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_io_screening
!! NAME
!!  m_io_screening
!!
!! FUNCTION
!!  This module contains the definition of the header of the 
!!  _SCR and _SUSC file as well as methods used to read/write/echo.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUT
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_io_screening

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors

 use m_header,  only : hdr_copy

 implicit none

 private

 public ::          &
&  scr_hdr_io,      &  ! I/O of the header (read/write/echo).
&  print_ScrHdr,    &  ! Print the SCR related part of the header.
&  init_ScrHdr,     &  ! Initialize the header.
&  scrhdr_comm,     &  ! Transmit the header.
&  free_scrhdr,     &  ! Free the header.
&  copy_scrhdr,     &  ! Deep copy of the SCR|SUSC header.
&  merge_ScrHdr,    &  ! Merge two or more SCR headers.
&  write_screening, &  ! Write a q-slice of the matrix in G-space.
&  read_screening      ! Read the content of the (SCR|SUSC) file placed after the header. 


 ! TODO use smaller record size to avoid problems in supercell calculations
 logical,private :: switch_to_580=.TRUE.

!!***

!!****t* m_io_screening/ScrHdr_type
!! NAME
!!  ScrHdr_type
!!
!! FUNCTION
!!  The structure defining the header of the _SCR or _SUSC file.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SOURCE

!£ type,public ;; ScrHdr_type
!£
!£  !Other variables that can be added are, for the moment, commented out. 
!£  !Most of them are related to the Abinit implementation  and should not compare in the ETSF specs
!£
!£  !Index of the qlwl section?
!£  !gwcomp, gwencomp  ! Info on the extrapolar algorithm
!£ 
!£  integer :: ID           ! Matrix identifier: O if not yet defined, 1 for chi0, 
!£                            ! 2 for chi, 3 for epsilon, 4 for espilon^{-1}
!£  integer :: ikxc         ! Kxc kernel used, 0 for None (RPA), >0 for static TDDFT (=ixc), <0 for frequency-dependent TDDFT 
!£  integer :: inclvkb      ! q-->0 treatment, 0 for None, 1-2 for transversal gauge, 3 for longitudinal
!£  integer :: headform     ! format of the SCR header
!£  integer :: fform        ! File format:
!£  integer :: gwcalctyp    ! Calculation type (G0W0, G0W, GW ...)
!£  integer :: nI,nJ        ! Number of spin components (rows,columns) in chi|eps^-1. (1,1) if collinear. 
!£                          !  The internal representation of the matrix is eps(nI*npwe,nJ*npwe) 
!£  integer :: nqibz        ! Number of q-points in the IBZ.
!£  integer :: nqlwl        ! Number of points for the treatment of the long wavelength limit.
!£  integer :: nomega       ! Total number of frequencies.
!£  integer :: nbnds_used   ! Number of bands used during the screening calculation (only for info)
!£  integer :: npwe         ! Number of G vectors reported on the file.
!£  integer :: npwwfn_used  ! Number of G vectors for wavefunctions used during the screening calculation (only for info)
!£  integer :: spmeth       ! Method used to approximate the delta function in the expression for Im Chi_0
!£  integer :: test_type    ! 0 for None, 1 for TEST-PARTICLE, 2 for TEST-ELECTRON (only for TDDFT)
!£  integer :: tordering    ! 0 if not defined, 1 for Time-Ordered, 2 for Advanced, 3 for Retarded.
!£
!£  real(dp) :: soenergy    ! Scissor Energy, zero if not used
!£  real(dp) :: spsmear     ! Smearing of the delta in case of spmeth==2
!£  real(dp) :: zcut        ! Imaginary shift to avoid the poles along the real axis.
!£
!£  type(Hdr_type) :: Hdr   ! The abinit header.
!£
!£!arrays
!£  character(len=80) :: title(2)
!£  ! Title describing the content of the file.
!£
!£  integer,pointer  :: gvec(:,:)                 
!£  ! gvec(3,npwe) 
!£  ! G vectors in r.l.u.
!£
!£  real(dp),pointer :: qibz(:,:)
!£  ! qibz(3,nqibz)
!£  ! q-points in r.l.u.
!£
!£  real(dp),pointer :: qlwl(:,:)                 
!£  ! qlwl(3,nqlwl)
!£  ! q-points for the long wave-length limit treatment (r.l.u)
!£
!£  complex(dpc),pointer :: lwing(:,:,:)         
!£  ! lwing(npwe,nomega,nqlwl)
!£  ! Lower wings for the different q"s -->0 
!£
!£  complex(dpc),pointer :: omega(:)             
!£  ! omega(nomega) 
!£  ! All frequencies calculated both along the real and the imaginary axis.
!£
!£  complex(dpc),pointer :: uwing(:,:,:)   
!£  ! uwing(npwe,nomega,nqlwl)
!£  ! Upper wings for the different q"s -->0 
!£
!£ end type ScrHdr_type
!£!!***

 integer,private,parameter :: HSCR_LATEST_HEADFORM=57 ! This is still the old one used in abinit

 integer,public,parameter :: DP_NOTYPE           = 0
 integer,public,parameter :: DP_IRREDUCIBILE_POL = 1
 integer,public,parameter :: DP_REDUCIBLE_POL    = 2
 integer,public,parameter :: DP_EPSILON          = 4
 integer,public,parameter :: DP_INVERSE_EPSILON  = 8
 
 !integer,public,parameter :: DP_RPA_KERNEL = 0  
 integer,public,parameter :: DP_TDDFT_RPA       = 1  
 integer,public,parameter :: DP_TDDFT_ADIABATIC = 2  
 integer,public,parameter :: DP_TDDFT_TDEPEDENT = 4 

 !integer,public,parameter :: DP_TDDFT_TKERNEL_TYPE = 4 

 integer,public,parameter :: DP_ONESHOT       = 1
 integer,public,parameter :: DP_ELECTRON_ONLY = 2
 integer,public,parameter :: DP_WAVEFUNCTIONS = 4

 integer,public,parameter :: DP_TIME_ORDERED   = 1
 integer,public,parameter :: DP_ADVANCED       = 2
 integer,public,parameter :: DP_RETARDED       = 4


 integer,public,parameter :: DP_TEST_NONE      = 1
 integer,public,parameter :: DP_TEST_PARTICLE  = 2
 integer,public,parameter :: DP_TEST_ELECTRON  = 4


CONTAINS  !=========================================================================================================================
!!***

!!****f* m_io_screening/scr_hdr_io
!! NAME
!!  scr_hdr_io
!!
!! FUNCTION
!! This subroutine deals with the I/O of the ScrHdr_type structured variables (read/write/echo).
!! According to the value of rdwr, it reads the header of a file, writes it, or echo the value 
!! of the structured variable to a file. Note that, when reading, different records of ScrHdr
!! are allocated here, according to the values of the read variables. Records of ScrHdr should be 
!! deallocated correctly by a call to hdr_clean when ScrHdr is not used anymore.
!!
!! INPUTS
!!  accesswff=Option defining the file format of the external file.
!!  localrdwf=(parallel case) if 1, the file is local to each machine.
!!  spaceComm=MPI communicator.
!!  master=rank of the master node in spaceComm, usually 0
!!  rdwr= if 1, read the ScrHdr structured variable from the header of the file,
!!        if 2, write the header to unformatted file
!!        if 3, echo part of the header to formatted file (records 1 and 2)
!!        if 4, echo the header to formatted file
!!        if 5, read the ScrHdr without rewinding (unformatted)
!!        if 6, write the ScrHdr without rewinding (unformatted)
!!  unt=unit number of the file (unformatted if rdwr=1, 2, 5 or 6 formatted if rdwr=3,4)
!!
!! OUTPUT
!!  (see side effects)
!!
!! SIDE EFFECTS
!!  The following variables are both input or output :
!!  fform=kind of the array in the file
!!   if rdwr=1,5 : will be output ; if the reading fail, return fform=0
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!  ScrHdr <type(ScrHdr_type)>=the header structured variable
!!   if rdwr=1,5 : will be output
!!   if rdwr=2,3,4,6 : should be input, will be written or echo to file
!!
!! NOTES
!! In all cases, the file is supposed to be open already
!! When reading (rdwr=1) or writing (rdwr=2), rewind the file
!! When echoing (rdwr=3) does not rewind the file.
!! When reading (rdwr=5) or writing (rdwr=6), DOES NOT rewind the file
!!
!! In writing mode, the routine is supposed to called by the master node.
!! no check is done, it is up to the developer.
!!
!! PARENTS
!!      m_io_screening,m_screening,mrgscr,screening
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine scr_hdr_io(fform,rdwr,unt,spaceComm,master,accesswff,localrdwf,HScr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_59_io_mpi
 use interfaces_61_ionetcdf
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(inout) :: fform
 integer,intent(in) :: rdwr,unt,accesswff,localrdwf,spaceComm,master
 type(ScrHdr_type),intent(inout) :: Hscr

!Local variables-------------------------------
!scalars
 integer :: iqibz,iqlwl,iomega,rank,ierr,nprocs,prtvol
 logical :: ltest,I_read,master_casts
 character(len=500) :: msg
!arrays

! *************************************************************************

 DBG_ENTER("COLL")

 ltest=(accesswff==0.or.accesswff==2)
 call assert(ltest,'Wrong value for accesswff',__FILE__,__LINE__)

 ! === Initialize MPI info for spaceComm ===
 call xcomm_rank(spaceComm,rank,  ierr)
 call xcomm_size(spaceComm,nprocs,ierr)

 I_read      =(localrdwf==1.or.rank==master)
 master_casts=(localrdwf==0.and.nprocs>1)

 if (rdwr==1.or.rdwr==5) then
  ! === Reading the header of an unformatted file ===

  if (I_read) then
   ! * Read the abinit header, rewinding of the file (if any) is done here.
   ! TODO write a wrapper, using accesswff as input
   if (accesswff==0) then
    call hdr_io(fform,Hscr%Hdr,rdwr,unt)
   else if (accesswff==2) then
    call hdr_io_etsf(fform,Hscr%Hdr,rdwr,unt)
   end if

   ! * Reset the variables absent in old versions.
   Hscr%fform=fform
   call reset_HSCR_(Hscr)

   select case (fform)
    case (1002)
     ! Old epsilon^-1 file used in version <5.6
     read(unt,ERR=10)Hscr%title
     read(unt,ERR=10)Hscr%npwe,Hscr%npwwfn_used,Hscr%nbnds_used,Hscr%nqibz,Hscr%nomega

     allocate(Hscr%gvec(3,Hscr%npwe))
     read(unt,ERR=10)Hscr%gvec(:,:)

     allocate(Hscr%qibz(3,Hscr%nqibz))
     read(unt,ERR=10)Hscr%qibz(:,:)

     allocate(Hscr%omega(Hscr%nomega))
     read(unt,ERR=10)Hscr%omega(:)

     ! Quantities not present in the old file format, nqlwl is set to 0
     allocate(Hscr%qlwl(3,Hscr%nqlwl)) 
     allocate(Hscr%lwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 
     allocate(Hscr%uwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 

    case (1102)
     ! File format for epsilon^-1, espilon, chi0 
     read(unt,ERR=10)Hscr%title
     read(unt,ERR=10)Hscr%ID,Hscr%nI,Hscr%nJ,Hscr%tordering,Hscr%test_type
     read(unt,ERR=10)Hscr%npwe,Hscr%npwwfn_used,Hscr%nbnds_used,Hscr%nqibz,Hscr%nomega,Hscr%nqlwl

     allocate(Hscr%gvec(3,Hscr%npwe))
     read(unt,ERR=10)Hscr%gvec(:,:)

     allocate(Hscr%qibz(3,Hscr%nqibz))
     read(unt,ERR=10)Hscr%qibz(:,:)

     allocate(Hscr%omega(Hscr%nomega))
     read(unt,ERR=10)Hscr%omega(:)

     ! Read data for q-->0 limit.
     allocate(Hscr%qlwl(3,Hscr%nqlwl))
     allocate(Hscr%lwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 
     allocate(Hscr%uwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 

     if (Hscr%nqlwl>0) then
      read(unt,ERR=10)Hscr%qlwl(:,:)
      read(unt,ERR=10)Hscr%lwing(:,:,:)
      read(unt,ERR=10)Hscr%uwing(:,:,:)
     end if

    case default
     write(msg,'(a,i5)')' scr_hdr_io: Wrong fform read = ',fform
     MSG_BUG(msg)
   end select

  end if !I_read

  ! === Cast data to other processors ===
  if (master_casts) then       
   call xcast_mpi(fform,master,spaceComm,ierr)
   call scrhdr_comm(HScr,master,rank,spaceComm)
  end if

 else if (rdwr==2.or.rdwr==6) then
  ! === Writing the header of an unformatted file ===

  ! Write the abinit header, rewinding of the file (if any) is done here.
  if (accesswff==0) then
   call hdr_io(fform,Hscr%Hdr,rdwr,unt)
  else if (accesswff==2) then
   call hdr_io_etsf(fform,Hscr%Hdr,rdwr,unt)
  end if

  ! TODO should always use the latest version.
  write(unt)Hscr%title

  select case (fform)
   case (1002) ! Old epsilon^-1 file used in version <5.6
    write(unt)Hscr%npwe,Hscr%npwwfn_used,Hscr%nbnds_used,Hscr%nqibz,Hscr%nomega

   case (1102) ! File format for epsilon^-1, espilon, chi0 
    !TODO add new variables
    write(unt)Hscr%ID,Hscr%nI,Hscr%nJ,Hscr%tordering,Hscr%test_type
    write(unt)Hscr%npwe,Hscr%npwwfn_used,Hscr%nbnds_used,Hscr%nqibz,Hscr%nomega,Hscr%nqlwl 

   case default
    write(msg,'(a,i5)')' scr_hdr_io: Wrong value for fform = ',fform
    MSG_BUG(msg)
  end select

  write(unt)Hscr%gvec(:,:)
  write(unt)Hscr%qibz(:,:)
  write(unt)Hscr%omega(:)

  ! === Add q-points for heads and wings for q-->0  ====
  if (fform>1002.and.Hscr%nqlwl>0) then 
   write(unt)Hscr%qlwl(:,:)
   write(unt)Hscr%lwing(:,:,:)
   write(unt)Hscr%uwing(:,:,:)
  end if
  
 else if (rdwr==3.or.rdwr==4) then
  ! === Echo the header to a formatted file ===

  write(unt,'(a)')&
&  ' ==============================================================================='
  if (rdwr==3) then 
   prtvol=0
   write(unt,'(a)') ' ECHO of part of the ABINIT-SCR file header '
  end if
  if (rdwr==4) then 
   prtvol=1
   write(unt,'(a)') ' ECHO of the ABINIT-SCR file header '
  end if

  call print_ScrHdr(Hscr,unit=unt,prtvol=prtvol,mode_paral='COLL')

  if (rdwr==3) write(unt,'(a)')' End the ECHO of part of the ABINIT-SCR file header '
  if (rdwr==4) write(unt,'(a)')' End the ECHO of the ABINIT-SCR file header '
  write(unt,'(a)')&
&  ' ==============================================================================='

  ! * Echo the abinit header. Rewinding of the file (if any) is done here.
  if (prtvol>0) then
   if (accesswff==0) then
    call hdr_io(fform,Hscr%Hdr,rdwr,unt)
   else if (accesswff==2) then
    call hdr_io_etsf(fform,Hscr%Hdr,rdwr,unt)
   end if
  end if

 else 
  write(msg,'(a,i5)')' scr_hdr_io: Wrong value for rdwr = ',rdwr
  MSG_BUG(msg)
 end if ! read/write/echo

 DBG_EXIT("COLL")

 RETURN
 !
 ! === Something went wrong while reading! ===
 10 continue
 write(msg,'(a)')' The header of the (SCR|SUSC) file seems to be corrupted.'
 MSG_ERROR(msg)

end subroutine scr_hdr_io
!!***

!!****f* m_io_screening/print_ScrHdr
!! NAME
!! print_ScrHdr
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      m_io_screening,mrgscr,setup_sigma
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine print_ScrHdr(Hscr,header,unit,prtvol,mode_paral)

 use defs_basis

 use m_numeric_tools, only : print_arr

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: prtvol,unit
 character(len=4),intent(in),optional :: mode_paral
 character(len=*),intent(in),optional :: header
 type(ScrHdr_type),intent(in) :: Hscr

!Local variables-------------------------------
!scalars
 integer :: iomega,iqibz,iqlwl,unt,verbose
 character(len=4) :: mode
 character(len=500) :: msg

! *************************************************************************

 unt=std_out ; if (PRESENT(unit))       unt    =unit
 verbose=0   ; if (PRESENT(prtvol))     verbose=prtvol
 mode='COLL' ; if (PRESENT(mode_paral)) mode   =mode_paral

 if (PRESENT(header)) then 
  msg=' ==== '//TRIM(ADJUSTL(header))//' ==== '
  call wrtout(unt,msg,mode)
 end if

 write(msg,'(1x,a)')TRIM(Hscr%title(1))
 call wrtout(unt,msg,mode)
 write(msg,'(1x,a)')TRIM(Hscr%title(2))
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Identifier                ',Hscr%ID
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Kxc kernel                ',Hscr%ikxc
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Treatment of q-->0 limit  ',Hscr%inclvkb
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' headform                  ',Hscr%headform
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' fform                     ',Hscr%fform
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' gwcalctyp                 ',Hscr%gwcalctyp
 call wrtout(unt,msg,mode)
 write(msg,'(a,2i8)')' Number of components      ',Hscr%nI,Hscr%nJ
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Number of q-points        ',Hscr%nqibz
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Number of q-directions    ',Hscr%nqlwl
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Number of frequencies     ',Hscr%nomega
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Number of bands used      ',Hscr%nbnds_used
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Dimension of matrix       ',Hscr%npwe
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Number of planewaves used ',Hscr%npwwfn_used
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Spectral method           ',Hscr%spmeth
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Test_type                 ',Hscr%test_type
 call wrtout(unt,msg,mode)
 write(msg,'(a,i8)') ' Time-ordering             ',Hscr%tordering
 call wrtout(unt,msg,mode)

 write(msg,'(a,es16.6)')' Scissor Energy             ',Hscr%soenergy
 call wrtout(unt,msg,mode)
 write(msg,'(a,es16.6)')' Spectral smearing          ',Hscr%spsmear
 call wrtout(unt,msg,mode)
 write(msg,'(a,es16.6)')' Complex Imaginary Shift    ',Hscr%zcut
 call wrtout(unt,msg,mode)

 if (verbose==0) then
  write(msg,'(a)')' The header contains additional records.'
  call wrtout(unt,msg,mode)
 else

  write(msg,'(2a)')ch10,' q-points [r.l.u.]:'
  call wrtout(unt,msg,mode)
  do iqibz=1,Hscr%nqibz 
   write(msg,'(i5,3f12.6)')iqibz,Hscr%qibz(:,iqibz)
   call wrtout(unt,msg,mode)
  end do

  write(msg,'(2a)')ch10,' Frequencies used [eV]:'
  call wrtout(unt,msg,mode)
  do iomega=1,Hscr%nomega
   write(msg,'(i3,2f7.2)')iomega,REAL(Hscr%omega(iomega))*Ha_eV,AIMAG(Hscr%omega(iomega))*Ha_eV
   call wrtout(unt,msg,mode)
  end do

  if (Hscr%nqlwl>0) then
   write(msg,'(2a)')ch10,' q-points for long-wavelength limit [r.l.u.]:'
   call wrtout(unt,msg,mode)

   do iqlwl=1,Hscr%nqlwl
    write(msg,'(i5,3f12.6)')iqlwl,Hscr%qlwl(:,iqlwl)
    call wrtout(unt,msg,mode)
    if (verbose>0) then
     do iomega=1,Hscr%nomega
      write(msg,'(2x,a,i4,a,2f9.4,a)')&
&      ' Upper and lower wings at the ',iomega,'th omega',Hscr%omega(iomega)*Ha_eV,' [eV]'
      call wrtout(unt,msg,mode)
      call print_arr(Hscr%uwing(:,iomega,iqlwl),max_r=9,unit=ab_out)
      call print_arr(Hscr%lwing(:,iomega,iqlwl),max_r=9,unit=ab_out)
      write(msg,'(a)')ch10 
      call wrtout(unt,msg,mode)
     end do
    end if
   end do
  end if
  ! G-vectors are not printed out.
 end if 

end subroutine print_ScrHdr
!!***

!!****f* m_io_screening/reset_HSCR_
!! NAME
!!  reset_HSCR_ [PRIVATE]
!!
!! FUNCTION
!!  Initialize variables of Hscr using default values.
!!  in order to maintain backward compatibility.
!!
!! INPUTS
!!
!! OUTPUT
!!  (see side effects)
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!      m_io_screening
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine reset_HSCR_(Hscr)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(ScrHdr_type),intent(inout) :: Hscr
! *************************************************************************

 if (Hscr%fform==1002) then 
  Hscr%ID        = 4         ! 4 for e^-1 since it was the only format available.
  Hscr%ikxc      = 0         ! 0 for None (RPA),
  Hscr%headform  = 56        ! Oldest one.
  Hscr%inclvkb   = 0         ! q-->0 treatment, 0 for None
  Hscr%gwcalctyp =-1         ! Not present in old fformat
  Hscr%nI        = 1         ! Collinear case.
  Hscr%nJ        = 1     
  Hscr%nqlwl     = 0         ! No. of q-->0 points.
  Hscr%spmeth    = 0         ! Was not released yet.
  Hscr%test_type = 0         ! None
  Hscr%tordering = 1         ! 1 for Time-Ordered, 2 for Advanced, 3 for Retarded.

  Hscr%soenergy  = zero      ! Not available.
  Hscr%spsmear   = zero      ! Not released yet.  
  Hscr%zcut      = 3.67493260d-03  ! Default, 0.1eV.
 end if

end subroutine reset_HSCR_
!!***

!!****f* m_io_screening/my_nullify_HScr
!! NAME
!!  my_nullify_HScr [PRIVATE]
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!  (see side effects)
!!
!! SIDE EFFECTS
!!  All pointers set to null
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine my_nullify_HScr(Hscr)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(ScrHdr_type),intent(inout) :: Hscr

!Local variables-------------------------------

! *************************************************************************

!integer 
 nullify(Hscr%gvec)

!real
 nullify(Hscr%qibz)
 nullify(Hscr%qlwl)

!complex
 nullify(Hscr%lwing)
 nullify(Hscr%omega)
 nullify(Hscr%uwing)

end subroutine my_nullify_HScr
!!***

!!****f* m_io_screening/init_ScrHdr
!! NAME
!!  init_ScrHdr
!!
!! FUNCTION
!!  Initialize the Hscr datatype and most of its content from the 
!!  Epsilonm1_parameters datat type Ep.
!!
!! INPUTS
!!  ID=Identifier used to define the type of Response function (e^-1, chi0)
!!  ikxc=Integer flag definining the type of XC kernel (0 if None i.e RPA)
!!  test_type=Integer flag defining the type of probing charge (0 for None)
!!  tordering=The time-ordering of the Response function.
!!  gvec(3,Ep%npwe)=The G-vectors used. 
!!  Ep<Epsilonm1_parameters>=Parameters defining the calculation of the screening.
!!  Hdr_abinit<Hdr_type>=The abinit header.
!!
!! OUTPUT
!!  Hscr<type(ScrHdr_type)>=the header, initialized.
!!
!! NOTES
!!
!! PARENTS
!!      screening
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine init_ScrHdr(ID,ikxc,test_type,tordering,title,ngvec,gvec,Ep,Hdr_abinit,Hscr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ID,ikxc,test_type,tordering,ngvec
 type(Epsilonm1_parameters),intent(in) :: Ep
 type(Hdr_type),intent(in) :: Hdr_abinit
 type(ScrHdr_type),intent(inout) :: HScr
!arrays
 integer,intent(in) :: gvec(3,ngvec)
 character(len=80),intent(in) :: title(2)

!Local variables-------------------------------

! *************************************************************************

 call assert((ngvec==Ep%npwe),'ngvec/=Ep%npwe',__FILE__,__LINE__)

 !£call my_nullify_HScr(HScr)

 ! === Deep copy of the abinit header ===
 call hdr_copy(Hdr_abinit,Hscr%Hdr)

 ! === Initialize quantities related to the screening file ===
 Hscr%ID         =ID           
 Hscr%ikxc       =ikxc         
 Hscr%inclvkb    =Ep%inclvkb      
 Hscr%fform      =1002
 !£Hscr%fform      =1102
 Hscr%headform   =HSCR_LATEST_HEADFORM
 Hscr%gwcalctyp  =Ep%gwcalctyp
 Hscr%nI         =Ep%nI
 Hscr%nJ         =Ep%nJ  
 Hscr%nqibz      =Ep%nqcalc  ! Ep%nqcalc==Ep%nqibz unless we splitted the calculation into different runs. 
 Hscr%nqlwl      =Ep%nqlwl        
 Hscr%nomega     =Ep%nomega       
 Hscr%nbnds_used =Ep%nbnds   
 Hscr%npwe       =Ep%npwe 
 Hscr%npwwfn_used=Ep%npwwfn 
 Hscr%spmeth     =Ep%spmeth
 Hscr%test_type  =test_type    
 Hscr%tordering  =tordering    

 Hscr%soenergy   =Ep%soenergy
 Hscr%spsmear    =Ep%spsmear
 Hscr%zcut       =Ep%zcut

 Hscr%title(:)=title(:)

 allocate(Hscr%gvec(3,Ep%npwe))
 Hscr%gvec(:,:)=gvec(1:3,1:Ep%npwe)

 allocate(Hscr%qibz(3,Ep%nqcalc))
 Hscr%qibz(:,:)=Ep%qcalc(:,:)
 
 allocate(Hscr%qlwl(3,Ep%nqlwl))                 
 Hscr%qlwl(:,:)=Ep%qlwl(:,:)

 allocate(Hscr%omega(Ep%nomega))             
 Hscr%omega(:)=Ep%omega(:) 

 allocate(Hscr%lwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 
 allocate(Hscr%uwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 

 !TODO these quantities should be initialized correctly from input arguments
 if (Ep%nqlwl>0) then
  Hscr%lwing=czero
  Hscr%uwing=czero
 end if

end subroutine init_ScrHdr
!!***

!!****f* m_io_screening/scrhdr_comm
!! NAME
!! scrhdr_comm
!!
!! FUNCTION
!! This subroutine transmit the header structured datatype initialized 
!! on one processor (or a group of processor), to the other processors. 
!! It also allocates the needed part of the header.
!!
!! INPUTS
!!  master=ID of the master node.
!!  rank=ID of the node that receives the data.
!!  spaceComm=MPI communicator.
!!
!! OUTPUT
!!  (no output)
!!
!! SIDE EFFECTS
!!  Hscr<type(ScrHdr_type)>=the SCR header. For the master, it is already
!!   initialized entirely, while for the other procs, everything has
!!   to be transmitted.
!!
!! NOTES
!! This routine is called only in the case of MPI version of the code.
!!
!! PARENTS
!!      m_io_screening
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine scrhdr_comm(Hscr,master,rank,spaceComm)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_59_io_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer, intent(in) :: master,rank,spaceComm
 type(ScrHdr_type),intent(inout) :: Hscr

!Local variables-------------------------------
 integer :: ierr
 character(len=500) :: msg

! *************************************************************************

 DBG_ENTER("COLL")

!scalars
 call xcast_mpi(Hscr%ID,         master,spaceComm,ierr)
 call xcast_mpi(Hscr%ikxc,       master,spaceComm,ierr)
 call xcast_mpi(Hscr%inclvkb,    master,spaceComm,ierr)
 call xcast_mpi(Hscr%headform,   master,spaceComm,ierr)
 call xcast_mpi(Hscr%fform,      master,spaceComm,ierr)
 call xcast_mpi(Hscr%gwcalctyp,  master,spaceComm,ierr)
 call xcast_mpi(Hscr%nI,         master,spaceComm,ierr)
 call xcast_mpi(Hscr%nJ,         master,spaceComm,ierr)
 call xcast_mpi(Hscr%nqibz,      master,spaceComm,ierr)
 call xcast_mpi(Hscr%nqlwl,      master,spaceComm,ierr)
 call xcast_mpi(Hscr%nomega,     master,spaceComm,ierr)
 call xcast_mpi(Hscr%nbnds_used, master,spaceComm,ierr)
 call xcast_mpi(Hscr%npwe,       master,spaceComm,ierr)
 call xcast_mpi(Hscr%npwwfn_used,master,spaceComm,ierr)
 call xcast_mpi(Hscr%spmeth,     master,spaceComm,ierr)
 call xcast_mpi(Hscr%test_type,  master,spaceComm,ierr)
 call xcast_mpi(Hscr%tordering,  master,spaceComm,ierr)

 call xcast_mpi(Hscr%soenergy,   master,spaceComm,ierr)
 call xcast_mpi(Hscr%spsmear,    master,spaceComm,ierr)
 call xcast_mpi(Hscr%zcut,       master,spaceComm,ierr)

 ! Communicate the Abinit header.
 call hdr_comm(Hscr%Hdr,master,rank,spaceComm)

!arrays
 call xcast_mpi(Hscr%title,      master,spaceComm,ierr)

 if (rank/=master) then 
  allocate(Hscr%gvec(3,Hscr%npwe))
  allocate(Hscr%qibz(3,Hscr%nqibz))
  allocate(Hscr%qlwl(3,Hscr%nqlwl))
  allocate(Hscr%omega(Hscr%nomega))
  allocate(Hscr%lwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 
  allocate(Hscr%uwing(Hscr%npwe,Hscr%nomega,Hscr%nqlwl)) 
 end if

 call xcast_mpi(Hscr%gvec, master,spaceComm,ierr)
 call xcast_mpi(Hscr%qibz, master,spaceComm,ierr)
 call xcast_mpi(Hscr%qlwl, master,spaceComm,ierr)
 call xcast_mpi(Hscr%omega,master,spaceComm,ierr)
 if (Hscr%nqlwl>0) then
  call xcast_mpi(Hscr%lwing,master,spaceComm,ierr)
  call xcast_mpi(Hscr%uwing,master,spaceComm,ierr)
 end if

 DBG_EXIT("COLL")

end subroutine scrhdr_comm
!!***

!!****f* m_io_screening/free_scrhdr
!! NAME
!! free_scrhdr
!!
!! FUNCTION
!! Deallocate the components of the header structured datatype
!!
!! INPUTS
!! hdr <type(hdr_type)>=the header
!!
!! OUTPUT
!!  (only deallocate)
!!
!! PARENTS
!!      m_io_screening,m_screening,mrgscr,screening
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine free_scrhdr(Hscr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_62_iowfdenpot
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(ScrHdr_type),intent(inout) :: Hscr

!Local variables-------------------------------

! ************************************************************************* 

 DBG_ENTER("COLL")

 call hdr_clean(Hscr%Hdr)

 deallocate(Hscr%gvec)
 deallocate(Hscr%qibz)
 deallocate(Hscr%qlwl)
 deallocate(Hscr%omega)

 deallocate(Hscr%lwing)
 deallocate(Hscr%uwing)

 DBG_EXIT("COLL")

end subroutine free_scrhdr
!!***

!!****f* m_io_screening/copy_ScrHdr
!! NAME
!! copy_ScrHdr
!!
!! FUNCTION
!! Make a deep copy of the header of the _SCR or _SUSC file.
!!
!! INPUTS
!!
!! PARENTS
!!      m_io_screening,m_screening
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine copy_ScrHdr(Hscr_in,Hscr_cp)

 use defs_basis
 use m_copy, only : deep_copy

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(ScrHdr_type),intent(in) :: Hscr_in
 type(ScrHdr_type),intent(inout) :: Hscr_cp

!Local variables-------------------------------
!scalars
 character(len=500) :: msg
! *************************************************************************

 if (Hscr_in%headform>57) then 
  write(msg,'(3a,i10,2a)')&
&  ' copy_ScrHdr deals only with versions of the header up to 57. ',ch10,&
&  ' However headform= ',Hscr_in%headform,ch10,&
&  ' Change the source to add the changes done in the new version. '  
  MSG_BUG(msg)
 end if

 ! === Integer values ===
 Hscr_cp%ID          = Hscr_in%ID           
 Hscr_cp%ikxc        = Hscr_in%ikxc         
 Hscr_cp%inclvkb     = Hscr_in%inclvkb      
 Hscr_cp%headform    = Hscr_in%headform        
 Hscr_cp%fform       = Hscr_in%fform        
 Hscr_cp%gwcalctyp   = Hscr_in%gwcalctyp    
 Hscr_cp%nI          = Hscr_in%nI
 Hscr_cp%nJ          = Hscr_in%nJ
 Hscr_cp%nqibz       = Hscr_in%nqibz        
 Hscr_cp%nqlwl       = Hscr_in%nqlwl        
 Hscr_cp%nomega      = Hscr_in%nomega       
 Hscr_cp%nbnds_used  = Hscr_in%nbnds_used   
 Hscr_cp%npwe        = Hscr_in%npwe         
 Hscr_cp%npwwfn_used = Hscr_in%npwwfn_used  
 Hscr_cp%spmeth      = Hscr_in%spmeth       
 Hscr_cp%test_type   = Hscr_in%test_type    
 Hscr_cp%tordering   = Hscr_in%tordering    

 ! === Real variables ====
 Hscr_cp%soenergy = Hscr_in%soenergy    
 Hscr_cp%spsmear  = Hscr_in%spsmear     
 Hscr_cp%zcut     = Hscr_in%zcut        

 ! === Copy the abinit Header ===
 call hdr_copy(Hscr_in%Hdr,Hscr_cp%Hdr)

 Hscr_cp%title(:) = Hscr_in%title(:)

 ! Copy pointers.
 call deep_copy( Hscr_in%gvec , Hscr_cp%gvec  )

 call deep_copy( Hscr_in%qibz , Hscr_cp%qibz  )
 call deep_copy( Hscr_in%qlwl , Hscr_cp%qlwl  )

 call deep_copy( Hscr_in%lwing, Hscr_cp%lwing )
 call deep_copy( Hscr_in%omega, Hscr_cp%omega )
 call deep_copy( Hscr_in%uwing, Hscr_cp%uwing )

end subroutine copy_ScrHdr
!!***

!!****f* m_io_screening/merge_ScrHdr
!! NAME
!! merge_ScrHdr
!!
!! FUNCTION
!! This subroutine merges diffrent header structured variable (Scrhdr)
!!
!! INPUTS
!!  Hscr_in(:) <ScrHdr_type)>=the header structured variable
!!  Hscr_out<ScrHdr_type>=the merged header.
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine merge_ScrHdr(Hscr_in,Hscr_out)

 use defs_basis
 use m_numeric_tools, only : remove_copies, imax_loc
 use m_bz_mesh,       only : isequalk

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_62_iowfdenpot
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(ScrHdr_type),intent(out) :: Hscr_out
 type(ScrHdr_type),intent(in) :: Hscr_in(:)

!Local variables-------------------------------
!scalars
 integer :: nhds,restart,restartpaw,ihd,ii,nqtot,nqneq
 logical :: ltest,isok
 character(len=500) :: msg
!arrays
 real(dp),allocatable :: qset(:,:)
! *************************************************************************

 DBG_ENTER("COLL")

 nhds=SIZE(Hscr_in(:))

 ! === Make an initial copy of the header ===
 ! * If multiple headers, select the header containing q-->0 so that we copy also heads and wings
 ii=imax_loc(Hscr_in(:)%nqlwl)
 call copy_ScrHdr(Hscr_in(ii),Hscr_out)
 do ihd=1,nhds
  if (ihd/=ii.and.(Hscr_in(ihd)%nqlwl/=0)) then
   msg='Only a single header should contain heads and wings'
   MSG_ERROR(msg)
  end if
 end do

 if (nhds==1) RETURN

 ! === Check the consistency of the abinit Headers ===
 ! * FFT grid might be q-point dependent so we stop only when restart==0
 isok=.TRUE.
 do ihd=2,nhds
  call hdr_check(Hscr_in(1)%fform,Hscr_in(ihd)%fform,Hscr_in(1)%Hdr,Hscr_in(ihd)%Hdr,'COLL',restart,restartpaw)
  if (restart==0) then
   isok=.FALSE.
   write(msg,'(a,i3,a)')' Abinit header no.',ihd,' is not consistent with the first header '
   MSG_WARNING(msg)
  end if
 end do

 if (.not.isok) then
  MSG_ERROR('Cannot continue, Check headers')
 end if

 ! === Now check variables related to polarizability|epsilon^{-1} ===
 ! 1) Tests quantities that must be equal 
 ii = assert_eq(Hscr_in(:)%ID,       'Headers have different Identifiers')
 ii = assert_eq(Hscr_in(:)%ikxc,     'Headers have different ikxc'       )
 ii = assert_eq(Hscr_in(:)%headform, 'Headers have different headform'   )
 ii = assert_eq(Hscr_in(:)%fform,    'Headers have different fform'      )
 ii = assert_eq(Hscr_in(:)%gwcalctyp,'Headers have different gwcalctyp'  )
 ii = assert_eq(Hscr_in(:)%nI,       'Headers have different nI'         )
 ii = assert_eq(Hscr_in(:)%nJ,       'Headers have different nJ'         )
 ii = assert_eq(Hscr_in(:)%nomega,   'Headers have different nomega'     )
 ii = assert_eq(Hscr_in(:)%test_type,'Headers have different test_type'  )
 ii = assert_eq(Hscr_in(:)%tordering,'Headers have different tordering'  )

 ! This is not mandatory but makes life easier!
 ii = assert_eq(Hscr_in(:)%npwe,'Headers have different number of G-vectors'  )

 do ihd=2,nhds
  if (ANY(ABS(Hscr_in(ihd)%omega-Hscr_in(1)%omega)>tol6)) then
   write(msg,'(a,i3,a)')' Frequencies in the first and the ',ihd,'-th header differ '
   MSG_ERROR(msg)
  end if

  if (ANY(Hscr_in(ihd)%gvec(:,:)-Hscr_in(1)%gvec(:,:)/=0)) then 
   write(msg,'(a,i3,a)')' Incompatible G-vector list found in the ',ihd,'-th header '
   MSG_ERROR(msg)
  end if
 end do !ihd

 ! === If error is not fatal, just warn ===
 if (ANY(Hscr_in(:)%npwwfn_used/=Hscr_in(1)%npwwfn_used)) then
  write(msg,'(a)')'  Files have been produced with a different number of planewaves for the wavefunctions. '
  MSG_COMMENT(msg)
 end if

 if (ANY(Hscr_in(:)%nbnds_used/=Hscr_in(1)%nbnds_used)) then
  write(msg,'(a)')'  Files have been produced with a different number of bands. '
  MSG_COMMENT(msg)
 end if

 if (ANY(Hscr_in(:)%spmeth/=Hscr_in(1)%spmeth)) then
  write(msg,'(a)')'  Files have been produced with different algorithms. '
  MSG_COMMENT(msg)
 end if

 if (ANY(ABS(Hscr_in(:)%soenergy-Hscr_in(1)%soenergy)>tol6)) then
  write(msg,'(a)')' Files have benn produced with different values of soenergy. '
  MSG_COMMENT(msg)
 end if

 if (ANY(ABS(Hscr_in(:)%spsmear-Hscr_in(1)%spsmear)>tol6)) then
  write(msg,'(a)')' Files have been produced with different values of spsmear. '
  MSG_COMMENT(msg)
 end if

 if (ANY(ABS(Hscr_in(:)%zcut-Hscr_in(1)%zcut)>tol6)) then
  write(msg,'(a)')' Files have been produced with different values of zcut. '
  MSG_COMMENT(msg)
 end if

 ! === Now merge the list of q-points ===
 nqtot=SUM(Hscr_in(:)%nqibz)
 allocate(qset(3,nqtot))
 ii=0
 do ihd=1,nhds
  qset(:,ii+1:ii+Hscr_in(ihd)%nqibz)=Hscr_in(ihd)%qibz(:,:)
  ii=ii+Hscr_in(ihd)%nqibz
 end do

 call remove_copies(nqtot,qset,nqneq,isequalk)

 if (nqneq/=nqtot) then
  write(msg,'(6a,i3,a,i3,a)')ch10,&
&  ' merge_ScrHdr: COMMENT ',ch10,&
&  ' Headers contain duplicated q-points ',ch10,&
&  ' Found ',nqneq,' distinct q-points among the total ',nqtot,' points reported in the headers. ' 
  !MSG_COMMENT(msg)
  call wrtout(std_out,msg,'COLL')
 end if

 Hscr_out%nqibz = nqneq
 deallocate(Hscr_out%qibz)
 allocate(Hscr_out%qibz(3,nqneq))
 Hscr_out%qibz(:,:)=qset(:,1:nqneq)
 deallocate(qset)

 DBG_EXIT("COLL")

end subroutine merge_ScrHdr
!!***

!!****f* m_io_screening/write_screening
!! NAME
!! write_screening
!!
!! FUNCTION
!! For a single q-point, write either \tilde epsilon^{-1} on the _SCR file 
!! or chi0 on the _SUSC file. The file is supposed to have been open in the calling routine.
!!
!! INPUTS
!!  unt=The unit number of the file to be written (supposed to be already open)
!!  epsm1(npwe,npwe,nomega)=The matrix to be written, for different frequencies, and a single q-point.
!!  nomega=Number of frequencies
!!  npwe=Number of plane waves in epsm1.
!!  omega(nomega)=Set of frequencies.
!!  accesswff=Integer flag defining the format of the output file. Available options:
!!    1--> Plain Fortran file 
!!    3--> ETSF format (TODO not yet coded)
!!
!! OUTPUT
!!  (only writing on file)
!!
!! PARENTS
!!      m_screening,mrgscr,screening
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine write_screening(unt,accesswff,npwe,nomega,omega,epsm1)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nomega,npwe,unt,accesswff
!arrays
 complex(gwpc),intent(in) :: epsm1(npwe,npwe,nomega) 
 complex(dpc),intent(in) :: omega(nomega)

!Local variables-------------------------------
!scalars
 integer :: ipwe,iomega,istat
 logical :: ltest
 character(len=500) :: msg
!arrays
 complex(dpc),allocatable :: epsm1d(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 ltest=(accesswff==0.or.accesswff==3)
 call assert(ltest,'Wrong value for accesswff',__FILE__,__LINE__)
 ltest=(accesswff==0)
 call assert(ltest,'accesswff=3 not yet implemented',__FILE__,__LINE__)
 !
 ! === Write a record for each omega ===
 ! * Always use double precision.
 if (switch_to_580) then
  allocate(epsm1d(npwe,1),STAT=istat) 
 else
  allocate(epsm1d(npwe,npwe),STAT=istat) 
 endif
 if (istat/=0) then  
  write(msg,'(a)')' out of memory'
  MSG_ERROR(msg)
 end if

 do iomega=1,nomega

  if (switch_to_580) then
   do ipwe=1,npwe
    epsm1d(:,1)=epsm1(:,ipwe,iomega) !spc ==> dpc
    write(unt)epsm1d(1:npwe,1)
   end do
  else 
   epsm1d(:,:)=epsm1(:,:,iomega) !spc ==> dpc
   write(unt)epsm1d(1:npwe,1:npwe)
  end if
 
 end do
 deallocate(epsm1d)

 DBG_EXIT("COLL")

end subroutine write_screening
!!***

!!****f* m_io_screening/read_screening
!! NAME
!! read_screening
!!
!! FUNCTION
!! Read either a screening (\tilde epsilon^{-1}) file in the SCR format or 
!! the irreducible polarizability (chi0) in the SUSC format.
!!
!! INPUTS
!!  accesswff=Integer flag defining the format of the output file. Available options:
!!    1--> Plain Fortran file 
!!    3--> ETSF format (TODO not yet coded)
!!  iqiA[optional]=Used if only a particular q-point is required. In this case iqiA define the index
!!   of the required q-point in the array qibz(3,Hscr%nqibz)
!!  nqibzA=number of asked q-points (used to dimension the output arrays). 
!!   Equal to Hscr%nqibz if the full matrix is required
!!  localrdwf= input variable (for parallel case) if 1, the SCR file is local to each machine
!!  MPI_enreg= datatype gathering information on parallelism
!!  npweA=number of asked planewaves
!!  nomegaA=number of asked frequencies
!!
!! OUTPUT
!!  epsm1(npweA,npweA,nomegaA,nqibzA) = \tilde\epsilon^{-1}(Ng,Ng,Nw,Nq)
!!
!! NOTES
!!  * If the epsilon matrix read is bigger than npweA x npweA, it will be truncated; 
!!    if it is smaller, an error will occur
!!  * If the number of frequencies asked for is smaller than that reported in the file, the matrix 
!!    will be truncated. If nomegaA > Hscr%nomega an error will occur 
!! 
!! TODO 
!!  Ome might use a temporary fortran file open in direct mode in case of out-of-memory solution. 
!!  It should speed-up the reading of the matrix.
!!
!! PARENTS
!!      m_screening,mrgscr
!!
!! CHILDREN
!!      assert,free_scrhdr,leave_test,scr_hdr_io,wrtout,xcast_mpi,xcomm_init
!!      xmaster_init,xme_init
!!
!! SOURCE

subroutine read_screening(fname,npweA,nqibzA,nomegaA,epsm1,MPI_enreg,accesswff,localrdwf,&
& iqiA) ! Optional

 use defs_basis
 use m_io_tools, only : get_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_14_hidewrite
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none
    
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: accesswff,localrdwf,nomegaA,npweA,nqibzA
 integer,optional,intent(in) :: iqiA 
 character(len=fnlen),intent(in) :: fname
 type(MPI_type),intent(in) :: MPI_enreg
!arrays
 complex(gwpc),intent(inout) :: epsm1(npweA,npweA,nomegaA,nqibzA)
  
!Local variables-------------------------------
!scalars
 integer :: ipwe,fform,iomega,ios,iqibz,istat,unt
 integer :: rdwr,spaceComm,rank,master,ierr
 logical :: ltest,i_read,master_casts,read_qslice
 character(len=500) :: msg
 type(ScrHdr_type) :: Hscr
!arrays
 complex(dpc),allocatable :: epsm1d(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 call xcomm_init  (MPI_enreg,spaceComm)  
 call xme_init    (MPI_enreg,rank     )          
 call xmaster_init(MPI_enreg,master   )  
 !
 ! === Define IO for parallel execution ===
 i_read      =(localrdwf==1.or.rank==master)
 master_casts=(localrdwf==0.and.MPI_enreg%nproc>1)

 if (i_read) then
  unt=get_unit()
  open(unit=unt,file=fname,status='old',form='unformatted',iostat=ios)
  if (ios/=0) then 
   write(msg,'(3a)')' Opening file ',TRIM(fname),' as old unformatted '
   MSG_ERROR(msg)
  end if  
 end if
 ! 
 ! === Read the header of the file ===
 ! * If localrdwf==0, master broadcasts the header to spaceComm.
 rdwr=1 
 call scr_hdr_io(fform,rdwr,unt,spaceComm,master,accesswff,localrdwf,Hscr)

 ! === Slice or full array? ===
 read_qslice=.FALSE.
 if (PRESENT(iqiA)) then 
  read_qslice=.TRUE.
  write(msg,'(a,i5,2a)')&
&  ' read_screening : reading the slice corresponding to iq = ',iqiA,' from file : ',TRIM(fname)
  call wrtout(std_out,msg,'COLL')
  if (iqiA<=0.or.iqiA>Hscr%nqibz) then
   MSG_BUG('iqiA out of range')
  end if
 end if 

 ! === Do some check ===
 if (Hscr%npwe>npweA) then
  write(msg,'(a,i8,2a,i8)')&
&  '  Total number of G-vectors reported on file = ',Hscr%npwe,ch10,&
&  '  Reading a smaller matrix of dimension      = ',npweA
  MSG_COMMENT(msg)
 end if

 ltest=(Hscr%npwe>=npweA)
 if (.not.ltest) then
  write(msg,'(2(a,i8,a))')&
&  '  dimension of matrix        = ',Hscr%npwe,ch10,&
&  '  requiring a too big matrix = ',npweA,ch10
  MSG_ERROR(msg)
 end if

 if (Hscr%nqibz<nqibzA) then 
  write(msg,'(a)')' Requiring too much q-points '
  MSG_ERROR(msg)
 end if 

 if (Hscr%nomega<nomegaA) then
  write(msg,'(a)')' Requiring too much frequencies '
  MSG_ERROR(msg)
 end if
 !
 ! === Now read epsilon^-1 ===
 ! Allocate a single column to save memory when switch_to_580 is set to .TRUE.
 if (i_read) then
  if (switch_to_580) then 
   allocate(epsm1d(Hscr%npwe,1),STAT=istat) 
  else
   allocate(epsm1d(Hscr%npwe,Hscr%npwe),STAT=istat) 
  endif
 endif
 call assert((istat==0),'out of memory in epsm1d',__FILE__,__LINE__)
 !
 ! === Two coding for different case just to keep it readable ===
 ! * Communication is done inside the loops to avoid problems with 
 !   the size of the MPI packet. Much slower but safer.
 ! TODO re-merge the two cases.
 SELECT CASE (read_qslice)

 CASE (.TRUE.)
  ! === Read only a slice of the full array (useful if the entire array is huge) ===
  !if (dim_wings==1) STOP 'not implemented'
  !TODO this has to be done in a cleaner way.

  do iqibz=1,Hscr%nqibz
   if (iqibz==iqiA) then 
    do iomega=1,nomegaA
     if (switch_to_580) then 
      if (i_read) then
       do ipwe=1,Hscr%npwe
        read(unt,ERR=10) epsm1d(1:Hscr%npwe,1)
        if(ipwe<=npweA) epsm1(1:npweA,ipwe,iomega,1)=epsm1d(1:npweA,1)
       end do
      end if
     else
      if (i_read) then
       read(unt,ERR=10) epsm1d(1:Hscr%npwe,1:Hscr%npwe)
       epsm1(1:npweA,1:npweA,iomega,1)=epsm1d(1:npweA,1:npweA)
      endif
     end if
     if (master_casts) then
      call xcast_mpi(epsm1(:,:,iomega,1),master,spaceComm,ierr)
     endif
    end do
    ! Skip other frequencies
    if (i_read) then 
     if (switch_to_580) then 
      do iomega=nomegaA+1,Hscr%nomega
       do ipwe=1,Hscr%npwe
        read(unt,ERR=10)
       end do
      end do
     else
      do iomega=nomegaA+1,Hscr%nomega ; read(unt,ERR=10) ; end do
     end if
    end if
   else 
    ! Skip other q-points i.e epsm1d(1:Hscr%npwe,1:Hscr%npwe)
    if (i_read) then 
     if (switch_to_580) then 
      do iomega=1,Hscr%nomega
       do ipwe=1,Hscr%npwe
        read(unt,ERR=10)
       end do
      end do
     else 
      do iomega=1,Hscr%nomega ; read(unt,ERR=10) ; end do
     end if
    end if
   end if ! iqibz==iqiA 
  end do ! iqibz

 CASE (.FALSE.)
  ! === Read the entire array ===

  do iqibz=1,Hscr%nqibz
   do iomega=1,nomegaA
     write(*,*) iqibz,iomega
    if (i_read) then 
     if (switch_to_580) then
      do ipwe=1,Hscr%npwe
       read(unt,ERR=10) epsm1d(1:Hscr%npwe,1)
       if(ipwe<=npweA) epsm1(1:npweA,ipwe,iomega,iqibz)=epsm1d(1:npweA,1)
      end do
     else
      read(unt,ERR=10) epsm1d(1:Hscr%npwe,1:Hscr%npwe)
      epsm1(1:npweA,1:npweA,iomega,iqibz)=epsm1d(1:npweA,1:npweA)
     end if
    end if
    if (master_casts) then
     call xcast_mpi(epsm1(:,:,iomega,iqibz),master,spaceComm,ierr)
    endif
   end do
   ! Skip other frequencies
   if (i_read) then 
    if (switch_to_580) then
     do iomega=nomegaA+1,Hscr%nomega
      do ipwe=1,Hscr%npwe
       read(unt,ERR=10)
      end do
     end do
    else
     do iomega=nomegaA+1,Hscr%nomega ; read(unt,ERR=10) ; end do
    end if
   end if
  end do !iqibz

 END SELECT
 !
 ! === Free memory and close file ===
 if (i_read) then 
  deallocate(epsm1d)
  close(unt)                    
  call free_scrhdr(Hscr)
 end if

 if (master_casts) then 
  call leave_test(MPI_enreg) 
 end if

 DBG_EXIT("COLL")

 RETURN
 !
 ! === Something went wrong! ===
10 write(msg,'(a)')' File seems to be corrupted. '
 MSG_ERROR(msg)

end subroutine read_screening

END MODULE m_io_screening
!!***
