!{\src2tex{textfont=tt}}
!!****f* ABINIT/ppmodel_methods
!! NAME
!! ppmodel_methods
!!
!! FUNCTION
!!  Module containing the definition of the PPmodel_type used to deal with 
!!  the plasmonpole technique. Methods to operate on the object are also provided.
!!
!! COPYRIGHT
!!  Copyright (C) 2008-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! NOTES
!!
!! PARENTS
!!  
!!
!! CHILDREN
!!  
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_ppmodel

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors

 implicit none

 private

 public ::              &
!£&  PPmodel_type,        &
&  PPmodel_symmetrize,  &    ! Symmetrize PPm parameters in the BZ.
&  nullify_PPmodel,     &    ! Nullify all pointers
&  init_PPmodel,        &    ! Initialize dimensions and pointers
&  destroy_PPmodel,     &    ! Destruction method.
&  setup_ppmodel,       &    ! Main Driver 
&  getem1_from_PPm,     &    ! Reconstruct e^{-1}(w) from PPm.
&  get_PPm_eigenvalues       

 integer,public,parameter :: PPM_NO_PLASMONPOLE  = 0
 integer,public,parameter :: PPM_GODBY_NEEDS     = 1
 integer,public,parameter :: PPM_HYBERTSEN_LOUIE = 2
 integer,public,parameter :: PPM_LINDEN_HORSH    = 3
 integer,public,parameter :: PPM_ENGEL_FARID     = 4

CONTAINS  !=========================================================================================================================
!!***

!!****f* ABINIT/PPmodel_symmetrize
!! NAME
!!  PPmodel_symmetrize
!!
!! FUNCTION
!!  Symmetrize the plasmonpole matrix elements in the full BZ zone
!!
!! INPUTS
!!  PPm<PPmodel_type>=data type containing information on the plasmonpole technique 
!!  Gsph<Gvectors_type>=data related to the G-sphere
!!    %grottb
!!    %phmSGt 
!!  Qmesh<BZ_mesh_type>=Info on the q-mesh
!!    %nbz=number if q-points in the BZ
!!    %tab(nbz)=index of the symmeric q-point in the IBZ, for each point in the BZ
!!    %tabo(nbz)=the operation that rotates q_ibz onto \pm q_bz (depending on tabi) 
!!    %tabi(nbz)=-1 if time-reversal has to be considered, 1 otherwise
!!  iq_bz=Index of the q-point in the BZ where PPmodel parameters have to be symmetrized
!!
!! OUTPUT
!!  botsq 
!!  otq
!!  eig (only if PPm%ppmodel==3)
!!
!! SIDE EFFECTS
!!
!! NOTES
!!  In the present implementation we are not considering a possible umklapp vector G0.
!!  In this case,indeed, the equation is different since we have to consider G-G0. 
!!  There is however a check in sigma
!! 
!!  * Remember the symmetry properties of \tilde\espilon^{-1}
!!    If q_bz=Sq_ibz+G0:
!! 
!!    $\epsilon^{-1}_{SG1-G0,SG2-G0}(q_bz) = e^{+iS(G2-G1).\tau}\epsilon^{-1}_{G1,G2)}(q)
!!
!!    If time-reversal symmetry can be used then :
!!    $\epsilon^{-1}_{G1,G2}(-q_bz) = e^{+i(G1-G2).\tau}\epsilon^{-1}_{-S^{-1}(G1+Go),-S^{-1}(G2+G0)}^*(q)
!! 
!! * Notice that eig is used only if PPm%model==3
!!
!! TODO
!!  Symmetrization can be skipped if iq_bz correspond to a point in the IBZ
!!
!! PARENTS
!!      csigme
!!
!! CHILDREN
!!      assert,getem1_from_ppm,wrtout,zgees,zhpev
!!
!! SOURCE

subroutine PPmodel_symmetrize(PPm,Gsph,Qmesh,iq_bz,botsq,otq,eig) 

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iq_bz 
 type(PPmodel_type),intent(in) :: PPm
 type(Gvectors_type),intent(in) :: Gsph
 type(BZ_mesh_type),intent(in) :: Qmesh
 type(Jpt_gwpc_2D),intent(inout) :: botsq,otq,eig
!arrays
!Local variables-------------------------------
!scalars
 integer :: ii,jj
 integer :: iq_ibz,iiq,isymq,iq_curr
 character(len=500) :: msg
!arrays
 integer,pointer :: grottb(:)
 complex(gwpc),pointer :: phsgt(:)
! *********************************************************************

 ! Here there is a problem with the small q, still cannot use BZ methods
 iq_ibz=Qmesh%tab(iq_bz) ; isymq=Qmesh%tabo(iq_bz) ; iiq=(3-Qmesh%tabi(iq_bz))/2

 iq_curr=iq_ibz ; if (PPm%mqmem==0) iq_curr=1 

 grottb => Gsph%rottb (1:PPm%npwc,iiq,isymq)
 phsgt  => Gsph%phmSGt(1:PPm%npwc,isymq) 

 ! === Symmetrize PPM parameters, equations depend on model ===
 SELECT CASE (PPm%model)

 CASE (PPM_GODBY_NEEDS, PPM_HYBERTSEN_LOUIE)
  ! * Plasmon pole frequencies otq are invariant under symmetry 
  do jj=1,PPm%npwc
   do ii=1,PPm%npwc
    botsq%datum(grottb(ii),grottb(jj))=PPm%bigomegatwsq%datum(ii,jj,iq_curr)*phsgt(ii)*CONJG(phsgt(jj))
    otq%datum  (grottb(ii),grottb(jj))=PPm%omegatw%datum(ii,jj,iq_curr)  
   end do
  end do

 CASE (PPM_LINDEN_HORSH)
  ! * For notations see pag 22 of Quasiparticle Calculations in solid (Aulbur et al)
  !  If q_bz=Sq_ibz+G0 then:
  !
  ! $\omega^2_{ii}(q_bz) = \omega^2_{ii}(q)$        (otq array)
  ! $\alpha_{ii}(q_bz)   = \alpha_{ii}(q)$          (botq array
  ! $\Phi_{SG-G0}(q_bz)  = \Phi_{G}(q) e^{-iSG.t}$  (eigenvectors of e^{-1}, eig array) 
  !   
  do ii=1,PPm%npwc ! DM bands index
   otq%datum  (ii,1)=PPm%omegatw%datum     (ii,1,iq_curr)
   botsq%datum(ii,1)=PPm%bigomegatwsq%datum(ii,1,iq_curr)
   do jj=1,PPm%npwc
    eig%datum(grottb(jj),ii)=PPm%eigpot(jj,ii,iq_curr)*phsgt(jj)
   end do
  end do
  if (iiq==2) eig%datum(:,:)=CONJG(eig%datum(:,:)) ! Time-reversal

 CASE (PPM_ENGEL_FARID)
  ! * For notations see pag 23 of Quasiparticle Calculations in solid (Aulbur et al)
  ! If q_bz=Sq_ibz+G0 then:
  !   
  ! $\omega^2_{ii}(q_bz) = \omega^2_{ii}(q)$        (otq array)
  ! $y_{SG-G0}(q_bz)     = y_{G}(q) e^{-iSG.t}$     (y=Lx) 
  !   
  do ii=1,PPm%npwc ! DM bands index
   otq%datum(ii,1)=PPm%omegatw%datum(ii,1,iq_curr)
   do jj=1,PPm%npwc
    botsq%datum(grottb(jj),ii)=PPm%bigomegatwsq%datum(jj,ii,iq_curr)*phsgt(jj)
   end do
  end do

 CASE DEFAULT 
  write(msg,'(a,i6)')' Wrong value for PPm%model = ',PPm%model
  MSG_BUG(msg)
 END SELECT
 !
 ! * Take into account time-reversal symmetry.
 if (iiq==2) botsq%datum(:,:)=CONJG(botsq%datum(:,:))

end subroutine PPmodel_symmetrize
!!***

!!****f* ABINIT/nullify_PPmodel
!! NAME
!!  nullify_PPmodel
!!
!! FUNCTION
!!  Nullify dynamic entities in a PPmodel_type object
!!
!! PARENTS
!!      m_ppmodel
!!
!! CHILDREN
!!      assert,getem1_from_ppm,wrtout,zgees,zhpev
!!
!! SOURCE

subroutine nullify_PPmodel(PPm)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_jolly_pointers

 implicit none

!Arguments ------------------------------------
 type(PPmodel_type),intent(inout) :: PPm
!Local variables-------------------------------
 integer :: istat
! *********************************************************************

 call destroy_jpt(PPm%bigomegatwsq,istat)
 call destroy_jpt(PPm%omegatw     ,istat)
 nullify(PPm%eigpot)

end subroutine nullify_PPmodel
!!***

!!****f* ABINIT/destroy_PPmodel
!! NAME
!!  destroy_PPmodel
!!
!! FUNCTION
!!  Free a PPmodel structure
!!
!! PARENTS
!!      mrgscr,sigma
!!
!! CHILDREN
!!      assert,getem1_from_ppm,wrtout,zgees,zhpev
!!
!! SOURCE

subroutine destroy_PPmodel(PPm)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_jolly_pointers

 implicit none

!Arguments ------------------------------------
 type(PPmodel_type),intent(inout) :: PPm
!Local variables-------------------------------
 integer :: istat
! *********************************************************************

 call destroy_jpt(PPm%bigomegatwsq,istat)
 call destroy_jpt(PPm%omegatw     ,istat)
 if (associated(PPm%eigpot)) deallocate(PPm%eigpot)

end subroutine destroy_PPmodel
!!***

!!****f* ABINIT/init_PPmodel
!! NAME
!!  init_PPmodel
!!
!! FUNCTION
!!  Initialize dimensions and other useful variables related to the PPmodel
!!
!! PARENTS
!!      mrgscr,sigma
!!
!! CHILDREN
!!      assert,getem1_from_ppm,wrtout,zgees,zhpev
!!
!! SOURCE

subroutine init_PPmodel(PPm,Er,ppmodel,drude_plsmf,gwcalctyp10)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_jolly_pointers

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(in) :: ppmodel,gwcalctyp10
 real(dp),intent(in) :: drude_plsmf
 type(PPmodel_type),intent(out) :: PPm
 type(Epsilonm1_results),intent(in) :: Er

!Local variables-------------------------------
 integer :: dim_q,istat,dt_shape(2,3)
 logical :: ltest
 character(len=500) :: msg      
! *********************************************************************

 DBG_ENTER("COLL")

 call nullify_PPmodel(PPm)

 PPm%nqibz = Er%nqibz
 PPm%mqmem = Er%mqmem
 ltest = (PPm%mqmem==0.or.PPm%mqmem==PPm%nqibz)
 call assert(ltest,'Wrong value for mqmem',__FILE__,__LINE__)
 PPm%npwc  = Er%npwe
 PPm%model = ppmodel

 PPm%drude_plsmf = drude_plsmf

 SELECT CASE (PPm%model)

 CASE (PPM_NO_PLASMONPOLE) 
  write(*,*)'WARNING, called with ppmodel==0'
  PPm%dm2_botsq = 0
  PPm%dm2_otq   = 0
  PPm%dm_eig    = 0 
  RETURN

 CASE (PPM_GODBY_NEEDS, PPM_HYBERTSEN_LOUIE)
  PPm%dm2_botsq = PPm%npwc
  PPm%dm2_otq   = PPm%npwc
  PPm%dm_eig    = 1 ! Should be set to 0, but g95 doesnt like zero-sized arrays

 CASE (PPM_LINDEN_HORSH)
  PPm%dm2_botsq = 1  
  PPm%dm2_otq   = 1
  PPm%dm_eig    = PPm%npwc

 CASE (PPM_ENGEL_FARID)
  PPm%dm2_botsq = PPm%npwc
  PPm%dm2_otq   = 1
  PPm%dm_eig    = 1 ! Should be set to 0, but g95 doesnt like zero-sized arrays

 CASE DEFAULT
  write(msg,'(a,i6)')' Wrong PPm%model = ',PPm%model
  MSG_BUG(msg)
 END SELECT
 !
 ! === Do we store full the q-mesh or out-of-memory solution? ===
 dim_q=PPm%nqibz ; if (PPm%mqmem==0) dim_q=1

 allocate(PPm%eigpot       (PPm%dm_eig,PPm%dm_eig,   dim_q), STAT=istat)

 !
 ! === If developer, save memory by using the same piece of memory for epsm1 and 
 ! the PPm parameters
 ! This is a VERY bad coding habit and, therefore, should be used only by developers.
 ltest = .not.(PPm%save_memory_devel .and. (PPm%model/=1 .or. .not.(gwcalctyp10==0.or.gwcalctyp10==8)) )
 call assert(ltest,'save_memory_devel should be set to FALSE (hardcoded)',__FILE__,__LINE__)
 if(PPm%save_memory_devel) then
  PPm%bigomegatwsq%datum => Er%epsm1(:,:,1,:) 
  PPm%omegatw%datum      => Er%epsm1(:,:,2,:)
 else
  dt_shape(1,:) = 1 ; dt_shape(2,1) = PPm%npwc   ; dt_shape(2,2) = PPm%dm2_botsq ; dt_shape(2,3) = dim_q
  call allocate_jpt(PPm%bigomegatwsq,dt_shape,istat)
  dt_shape(1,:) = 1 ; dt_shape(2,1) = PPm%npwc   ; dt_shape(2,2) = PPm%dm2_otq   ; dt_shape(2,3) = dim_q
  call allocate_jpt(PPm%omegatw,dt_shape,istat)
 endif

 DBG_EXIT("COLL")
 
end subroutine init_PPmodel
!!***

!!****f* m_ppmodel/setup_ppmodel
!! NAME
!! setup_ppmodel
!!
!! FUNCTION
!!  Initialize some values of several arrays of the Er% datastructure 
!!  that are used in case of plasmonpole calculations
!!  Just a wrapper around different plasmonpole routines.
!!
!! INPUTS
!!  paral_kgb=variable related to band parallelism
!!  Qmesh<bz_mesh_type>=the q-mesh used for the inverse dielectric matrix
!!    %nibz=number of irreducible q-points
!!    %ibz(3,%nibz)=the irred q-point
!!  Er<epsilonm1_results>=the inverse dielectric matrix 
!!    %nomega=number of frequencies in $\epsilon^{-1}$
!!    %epsm1=the inverse dielctric matrix 
!!    %omega=frequencies in epsm1
!!    %npwe=number of G vectors for the correlation part
!!  MPI_enreg<MPI_type>=informations about MPI parallelization
!!  ngfftf(18)=contain all needed information about the 3D fine FFT mesh, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  gmet(3,3)=reciprocal space metric ($\textrm{bohr}^{-2}$).
!!  gprimd(3,3)=dimensional primitive translations for reciprocal space ($\textrm{bohr}^{-1}$)
!!  nfftf=the number of points in the FFT mesh (for this processor)
!!  rhor_tot(nfftf)=the total charge in real space
!!  PPm<PPmodel_type>: 
!!    %ppmodel=the type of  plasmonpole model 
!!
!! OUTPUT
!!  
!!
!! SIDE EFFECTS
!!  PPm<PPmodel_type>: 
!!  == if ppmodel 1 or 2 ==
!!   %omegatw and %bigomegatwsq=PPmodel parameters 
!!  == if ppmodel 3 ==
!!   %omegatw, %bigomegatwsq and %eigpot=PPmodel parameters
!!  == if ppmodel 4 ==
!!   %omegatw and %bigomegatwsq=PPmodel parameters 
!!
!! NOTES
!! TODO: rhor_tot should be replaced by rhog_tot
!! FFT parallelism won"t work 
!! Solve Issue with MPI_enreg
!!
!! PARENTS
!!      csigme,mrgscr,sigma
!!
!! CHILDREN
!!      assert,getem1_from_ppm,wrtout,zgees,zhpev
!!
!! SOURCE

subroutine setup_ppmodel(PPm,paral_kgb,Qmesh,Er,MPI_enreg,nfftf,gvec,ngfftf,gmet,gprimd,rhor_tot,&
& iqiA) !Optional

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_51_manage_mpi
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftf,paral_kgb
 integer,intent(in),optional :: iqiA
 type(BZ_mesh_type),intent(in) :: Qmesh
 type(Epsilonm1_results),intent(inout) :: Er
 type(MPI_type),intent(inout) :: MPI_enreg
 type(PPmodel_type),intent(inout) :: PPm
!arrays
 integer,intent(in) :: gvec(3,Er%npwe),ngfftf(18)
 real(dp),intent(in) :: gmet(3,3),gprimd(3,3)
 real(dp),intent(inout) :: rhor_tot(nfftf)

!Local variables-------------------------------
!scalars
 integer :: istat,master,npwc2,npwc3,nqiA,rank,spaceComm
 logical :: ltest,single_q
 character(len=500) :: msg

! *************************************************************************

 DBG_ENTER("COLL")

 call xcomm_init  (MPI_enreg,spaceComm)  
 call xme_init    (MPI_enreg,rank     )          
 call xmaster_init(MPI_enreg,master   )  
 !
 ! === if iqiA is present, then consider only one qpoint to save memory ===
 ! * This means the object has been already initialized
 nqiA=Qmesh%nibz ; single_q=.FALSE.
 if (PRESENT(iqiA)) then 
  nqiA=1 ; single_q=.TRUE.
  ltest=PRESENT(iqiA)
  call assert(ltest,'For single q-point mode, also iqiA must be present',__FILE__,__LINE__)
 end if
 !
 ! Allocate plasmonpole parameters 
 ! TODO ppmodel==1 by default, should be set to 0 if AC and CD
 SELECT CASE (PPm%model)

 CASE (PPM_NO_PLASMONPOLE)
  write(msg,'(a)')' Skipping Plasmompole model calculation' 
  call wrtout(std_out,msg,'COLL') 
  RETURN

 CASE (PPM_GODBY_NEEDS) 
  ! * Note: q-dependency enters only through epsilon^-1

  ! save_memory_devel is to be used only for developers (aggressive memory savings)!
  if(PPm%save_memory_devel) then
   call cppm1par(Er%npwe,nqiA,Er%nomega,Er%epsm1,Er%omega,PPm%drude_plsmf)
  else
   call cppm1par(Er%npwe,nqiA,Er%nomega,Er%epsm1,Er%omega,PPm%drude_plsmf,PPm%omegatw%datum,PPm%bigomegatwsq%datum)
  endif

 CASE (PPM_HYBERTSEN_LOUIE)
  if (.not.single_q) then
   call cppm2par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,PPm%omegatw%datum,&
&   ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh,gmet)
  else
   call cppm2par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,PPm%omegatw%datum,&
&   ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh,gmet,iqiA)
  end if

 CASE (PPM_LINDEN_HORSH)
  ! TODO Check better double precision, this routine is in a messy state
  if (.not.single_q) then
   call cppm3par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,&
&   PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,PPm%eigpot,Qmesh)
  else
   call cppm3par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,&
&   PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,PPm%eigpot,Qmesh,iqiA)
  end if

 CASE (PPM_ENGEL_FARID)
  ! TODO Check better double precision, this routine is in a messy state
  if (.not.single_q) then
   call cppm4par(paral_kgb,Er%npwe,nqiA,Er%epsm1,Er%nomega,PPm%bigomegatwsq%datum,&
&   PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh)
  else
   call cppm4par(paral_kgb,Er%npwe,nqiA,Er%epsm1,Er%nomega,PPm%bigomegatwsq%datum,&
&   PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh,iqiA)
  end if

 CASE DEFAULT
  write(msg,'(a,i6)')' Wrong PPm%model=',PPm%model
  MSG_BUG(msg)
 END SELECT

 DBG_EXIT("COLL")

end subroutine setup_ppmodel
!!***

!!****f* m_ppmodel/getem1_from_PPm
!! NAME
!!  getem1_from_PPm
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      m_ppmodel,mrgscr
!!
!! CHILDREN
!!      assert,getem1_from_ppm,wrtout,zgees,zhpev
!!
!! SOURCE

subroutine getem1_from_PPm(PPm,iqibz,zcut,nomega,omega,Vcp,em1q)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iqibz,nomega
 type(PPmodel_type),intent(in) :: PPm
 type(Coulombian_type),intent(in) :: Vcp
 real(dp),intent(in) :: zcut
!arrays
 complex(dpc),intent(in) :: omega(nomega)
 complex(dpc),intent(out) :: em1q(PPm%npwc,PPm%npwc,nomega)

!Local variables-------------------------------
!scalars
 integer :: istat,ig1,ig2,io,idm
 real(dp) :: den 
 complex(dpc) :: qpg1,qpg2,ug1,ug2
 complex(dpc) :: delta,num,em1ggp,otw,zzpq,yg1,yg2,bot1,bot2,chig1g2
 character(len=500) :: msg
 logical :: ltest
!arrays

! *************************************************************************

 ltest=(PPm%mqmem/=0)
 call assert(ltest,'mqmem==0 case in getem1_from_PPm not implemented',__FILE__,__LINE__)

 !TODO zcut should be an entry in PPm
 delta=CMPLX(zero,zcut)

 select case (PPm%model)

 case (PPM_GODBY_NEEDS, PPM_HYBERTSEN_LOUIE)
 do io=1,nomega
 
  do ig2=1,PPm%npwc
   do ig1=1,PPm%npwc
    !den = omega(io)**2-REAL(PPm%omegatw(ig1,ig2,iqibz)**2)
    !if (den**2<zcut**2) den = omega(io)**2-REAL( (PPm%omegatw(ig1,ig2,iqibz)-delta)**2 )
    den = omega(io)**2-REAL( (PPm%omegatw%datum(ig1,ig2,iqibz)-delta)**2 )
    em1ggp = PPm%bigomegatwsq%datum(ig1,ig2,iqibz)/den
    if (ig1==ig2) em1ggp=em1ggp+one
    em1q(ig1,ig2,io)=em1ggp
    !em1q(ig1,ig2,io)=em1ggp*Vcp%vc_sqrt(ig1,iqibz)*Vcp%vc_sqrt(ig2,iqibz)
   end do
  end do

 end do !io

 case (PPM_LINDEN_HORSH)

  !TODO Check coefficients 
  do io=1,nomega

   do ig2=1,PPm%npwc
    do ig1=1,PPm%npwc

     em1ggp=czero
     do idm=1,PPm%npwc
      !den=omega(io)**2-(omegatw(ig1,ig2,iqibz)-delta)**2
      !em1w(io)=em1w(io)+eigvec(ig1,idm,iqibz)*conjg(eigvec(ig2,idm,iqibz))*bigomegatwsq(ig1,ig2,iqibz)/den
      ug1 =PPm%eigpot(ig1,idm,iqibz)
      ug2 =PPm%eigpot(ig2,idm,iqibz)
      otw =PPm%bigomegatwsq%datum(idm,1,iqibz)*PPm%omegatw%datum(idm,1,iqibz)
      zzpq=PPm%bigomegatwsq%datum(idm,1,iqibz)
      den=half*REAL(zzpq*otw*( one/(omega(io)-otw+delta) - one/(omega(io)+otw-delta) ))
      em1ggp=em1ggp+ug1*CONJG(ug2)*den
      !eigenvalues(idm,io)=one + half*REAL(zzpq*otw*( one/(omega(io)-otw+delta) - one/(omega(io)+otw-delta) ))
     end do

     if (ig2==ig1) em1ggp=em1ggp+one
     em1q(ig1,ig2,io)=em1ggp

    end do !ig1
   end do !ig2

  end do !iomega 

 case (PPM_ENGEL_FARID)
 ! Make e^-1

 do io=1,nomega

  do ig2=1,PPm%npwc
   qpg2=one/Vcp%vc_sqrt(ig2,iqibz)
   do ig1=1,PPm%npwc
    qpg1=one/Vcp%vc_sqrt(ig1,iqibz)

    chig1g2=czero 
    do idm=1,PPm%npwc
     otw =PPm%omegatw%datum(idm,1,iqibz)
     bot1=PPm%bigomegatwsq%datum(ig1,idm,iqibz)
     bot2=PPm%bigomegatwsq%datum(ig2,idm,iqibz)
     yg1=SQRT(otw/four_pi)*qpg1*bot1
     yg2=SQRT(otw/four_pi)*qpg2*bot2
     chig1g2=chig1g2 + yg1*CONJG(yg2)/(omega(io)**2-(otw-delta)**2)
    end do

    em1ggp=four_pi*chig1g2/(qpg1*qpg2)
    if (ig1==ig2) em1ggp=em1ggp+one
    em1q(ig1,ig2,io)=em1ggp !*Vcp%vc_sqrt(ig1,iqibz)*Vcp%vc_sqrt(ig2,iqibz)

   end do !ig1
  end do !ig2

 end do !iomega

 case default
  write(msg,'(a,i6)')" Wrong PPm%model = ",PPm%model
  MSG_BUG(msg)
 end select

end subroutine getem1_from_PPm
!!***

!!****f* m_ppmodel/get_PPm_eigenvalues
!! NAME
!!  get_PPm_eigenvalues
!!
!! FUNCTION
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      mrgscr
!!
!! CHILDREN
!!      assert,getem1_from_ppm,wrtout,zgees,zhpev
!!
!! SOURCE

subroutine get_PPm_eigenvalues(PPm,iqibz,zcut,nomega,omega,Vcp,eigenvalues)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_linalg
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iqibz,nomega
 type(PPmodel_type),intent(in) :: PPm
 type(Coulombian_type),intent(in) :: Vcp
 real(dp),intent(in) :: zcut
!arrays
 complex(dpc),intent(in) :: omega(nomega)
 complex(dpc),intent(out) :: eigenvalues(PPm%npwc,nomega)

!Local variables-------------------------------
!scalars
 integer :: info,lwork,istat,negw,ig1,ig2,io,idm,idx,sdim,iomega
 real(dp) :: den 
 complex(dpc) :: num,em1ggp,otw,zzpq,yg1,yg2,bot1,bot2,chig1g2
 character(len=500) :: msg
 logical :: ltest
!arrays
 real(dp),allocatable :: ww(:),rwork(:)
 complex(dpc),allocatable :: work(:),Adpp(:),eigvec(:,:),wwc(:),vs(:,:),Afull(:,:)
 complex(dpc),allocatable :: em1q(:,:,:)
 logical,allocatable :: bwork(:)
 logical :: sortcplx !BUG in abilint

! *************************************************************************

 ltest=(PPm%mqmem/=0)
 call assert(ltest,'mqmem==0 case in get_PPm_eigenvalues not implemented',__FILE__,__LINE__)

 allocate(em1q(PPm%npwc,PPm%npwc,nomega))

 call getem1_from_PPm(PPm,iqibz,zcut,nomega,omega,Vcp,em1q)
 !print*,em1q

 do iomega=1,nomega

  print*,'iomega',iomega

  if (ABS(REAL(omega(iomega)))>0.00001) then
   !if (.TRUE.) then
   ! === Eigenvalues for a generic complex matrix ===
                                                                                                         
   lwork=4*2*PPm%npwc
   allocate(wwc(PPm%npwc),work(lwork),rwork(PPm%npwc),bwork(PPm%npwc))
   allocate(vs(PPm%npwc,PPm%npwc),STAT=istat)
   allocate(Afull(PPm%npwc,PPm%npwc),STAT=istat)
 print*,'done'
                                                                                                         
   Afull=em1q(:,:,iomega)
                                                                                                         
   !for the moment no sort, maybe here I should sort using the real part?
   call ZGEES('V','N',sortcplx,PPm%npwc,Afull,PPm%npwc,sdim,wwc,vs,PPm%npwc,work,lwork,rwork,bwork,info)
   if (info/=0) then 
    write(msg,'(2a,i10)')' get_PPm_eigenvalues : Error in ZGEES, diagonalizing complex matrix, info = ',info
    call wrtout(std_out,msg,'COLL') 
   end if
                                                                                                         
 print*,'done'
   eigenvalues(:,iomega)=wwc(:)
                                                                                                         
   deallocate(wwc,work,rwork,bwork)
   deallocate(vs)
   deallocate(Afull)

  else 
   ! === Hermitian Case ===
   lwork=2*PPm%npwc-1
   allocate(ww(PPm%npwc),work(lwork),rwork(3*PPm%npwc-2))
   allocate(eigvec(PPm%npwc,PPm%npwc))
   allocate(Adpp(PPm%npwc*(PPm%npwc+1)/2),STAT=istat)
   if (istat/=0) STOP ' get_PPm_eigenvalues : out of memory in Adpp'

   print*,'in hermitian'
 
   idx=0
   do ig2=1,PPm%npwc
    do ig1=1,ig2
     idx=idx+1
     Adpp(idx)=em1q(ig1,ig2,iomega)
    end do
   end do

   ! For the moment we require also the eigenvectors.
   call ZHPEV('V','U',PPm%npwc,Adpp,ww,eigvec,PPm%npwc,work,rwork,info)

   if (info/=0) then 
    write(msg,'(2a,i10)')' get_PPm_eigenvalues : Error diagonalizing matrix, info = ',info
    call wrtout(std_out,msg,'COLL') 
   end if
   negw = (COUNT((REAL(ww)<tol6)))
   if (negw/=0) then 
    write(msg,'(3a,i5,a,i3,a,f8.4)')&
&    ' get_PPm_eigenvalues : WARNING - ',ch10,&
&    ' Found negative eigenvalues. No. ',negw,' at iqibz= ',iqibz,' minval= ',MINVAL(REAL(ww))
   end if

   eigenvalues(:,iomega)=ww(:)

   deallocate(ww,work,rwork)
   deallocate(eigvec)
   deallocate(Adpp)
  end if

 end do !iomega

 deallocate(em1q)

end subroutine get_PPm_eigenvalues

END MODULE m_ppmodel
!!***
