!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_qparticles
!! NAME
!!  m_qparticles
!!
!! FUNCTION
!!  This module contains tools dealing with the IO of the QP file and other procedures
!!  related to the calculation of the quasiparticle amplitudes in the KS basis set.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (FB, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_qparticles

 use defs_basis
 use m_errors  

 implicit none

 private 

 public ::       &
&  wrqps,        &   ! Write a QPS file.
&  rdqps,        &   ! Read a QPS file.
&  show_QP,      &   ! Report the components of a QP amplitude in terms of KS eigenstates.
&  rdgw              ! Read GW corrections from an external file.

CONTAINS  !=========================================================================================================================
!!***

!!****f* m_qparticles/wrqps
!! NAME
!! wrqps
!!
!! FUNCTION
!!  Write the _QPS file containing information on the quasi-particles energies and wavefunctions.
!!
!! INPUTS
!!  fname=The name of the file
!!  Sp<Sigma_parameters>=Parameters characterizing the self-energy calculation.  
!!     %nsppol=1 for unpolarized, 2 for spin-polarized
!!     %nbnds=number of bands used for sigma
!!  Sr<Sigma_results>=Structure containing the results of the sigma run.
!!     %en_qp_diago(nbnds,nibz,nsppol)= NEW quasi-particle energies
!!     %eigvec_qp(nbnds,nbnds,nibz,nsppol)= NEW QP amplitudes in the KS basis set 
!!      obtained by diagonalizing H0 + Herm(Sigma).
!!  m_lda_to_qp(nbnds,nbnds,nibz,nsppol)= expansion of the OLD QP amplitudes in terms of KS wavefunctions
!!  Kmesh<Bz_mesh_type>=information on the k-point sampling.
!!     %nibz=number of irreducible k-points
!!     %ibz(3,kibz)=reduced coordinates of the irreducible k-points
!!  nfftot=Total number of FFT points for density
!!  ngfftf(18)=Info on the FFT mesh for the density.
!!  nscf=number of self consistent cycles performed
!!  nspden=number of spin-density components
!!
!! OUTPUT
!!  Only writing
!!
!! PARENTS
!!      sigma
!!
!! CHILDREN
!!      assert,wrtout
!!
!! SOURCE

subroutine wrqps(fname,Sp,Kmesh,nspden,nscf,nfftot,ngfftf,Sr,m_lda_to_qp,rho_qp)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 use m_io_tools,    only : get_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftot,nscf,nspden
 character(len=fnlen),intent(in) :: fname
 type(BZ_mesh_type),intent(in) :: Kmesh
 type(Sigma_parameters),intent(in) :: Sp
 type(Sigma_results),intent(in) :: Sr
!arrays
 integer,intent(in) :: ngfftf(18)
 real(dp),intent(in) :: rho_qp(nfftot,nspden)
 complex(dpc),intent(in) :: m_lda_to_qp(Sp%nbnds,Sp%nbnds,Kmesh%nibz,Sp%nsppol)

!Local variables-------------------------------
!scalars
 integer :: ib,ik,is,unqps
 character(len=500) :: msg

!arrays
 complex(dpc),allocatable :: mtmp(:,:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 unqps=get_unit()
 open(unit=unqps,file=fname,form='formatted',status='unknown')

 write(msg,'(3a)')ch10,' writing QP data on file : ',TRIM(fname) 
 call wrtout(std_out,msg,'COLL') 
 call wrtout(ab_out,msg,'COLL')

 write(unqps,*)nscf+1
 write(unqps,*)Kmesh%nibz
 write(unqps,*)Sp%nbnds
 write(unqps,*)Sp%nsppol
 !
 ! === Calculate the new m_lda_to_qp ===
 allocate(mtmp(Sp%nbnds,Sp%nbnds,Sp%nsppol))
 do is=1,Sp%nsppol
  do ik=1,Kmesh%nibz
   mtmp(:,:,is)=MATMUL(m_lda_to_qp(:,:,ik,is),Sr%eigvec_qp(:,:,ik,is))
   write(unqps,*)Kmesh%ibz(:,ik)
   do ib=1,Sp%nbnds
    write(unqps,*)Sr%en_qp_diago(ib,ik,is)
    write(unqps,*)mtmp(:,ib,is)
   end do 
  end do 
 end do 
 deallocate(mtmp)
 !
 ! === Write FFT dimensions and QP density ===
 write(unqps,*)ngfftf(1:3)
 write(unqps,*)rho_qp(:,:)
 close(unqps)

 DBG_EXIT("COLL")

end subroutine wrqps
!!***

!!****f* m_qparticles/rdqps
!! NAME
!! rdqps
!!
!! FUNCTION
!!  Read a _QPS file
!!
!! INPUTS
!!  nfftot=Total number of FFT points for density
!!  ngfftf(18)=Info on the FFT mesh for the density.
!!  nspden=Number of SPin-DENsity components. 
!!  usepaw=1 if we are using PAW.
!!  fname=Name of the file
!!  dimrho=1 if density has to be read, 0 otherwise
!!  prtvol=Verbosity level
!!  BSt<Bandstructure_type>=Structure containing the initial band structure.
!!     %nsppol=1 for unpolarized, 2 for spin-polarized.
!!     %mband=Max number of bands used
!!     %nkpt=number of irreducible k-points.
!!     %kptns(3,nkpt)=reduced coordinates of each irreducible k-point.
!!  ucvol=Volume of the unit cell
!!
!! OUTPUT
!!  nbsc=number of bands used to describe the QP amplitudes 
!!  nscf=number of iterations that have been performed (==0 if we start from a KS calculation)  
!!  m_lda_to_qp(mband,mband,nibz,nsppol)=matrix giving the decomposition of the QP
!!   wavefunction in the mainfold generated by the KS wavefunctions 
!!   (i.e $ m_lda_to_qp(ib,jb,k,s) := <\psi_{ib,k,s}^{KS}|\psi_{jb,k,s}^{QP}>$
!!  rhor_out(nfftot,nspden)=quasiparticle density
!!
!! SIDE EFFECTS
!!  BSt<Bandstructure_type>=Structure containing the initial band structure.
!!     %en_qp(mband,nkpt,nsppol)=QP energies at iteration nscf 
!!
!! PARENTS
!!      mlwfovlp_qp,screening,sigma
!!
!! CHILDREN
!!      assert,wrtout
!!
!! SOURCE

subroutine rdqps(BSt,fname,usepaw,nspden,dimrho,nscf,prtvol,&
& nfftot,ngfftf,ucvol,paral_kgb,MPI_enreg,nbsc,m_lda_to_qp,rhor_out)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 use m_io_tools, only : get_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_14_hidewrite
 use interfaces_32_util
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftot,nspden,usepaw,paral_kgb,dimrho,prtvol
 integer,intent(out) :: nbsc,nscf
 real(dp),intent(in) :: ucvol
 character(len=fnlen),intent(in) :: fname
 type(Bandstructure_type),intent(inout) :: BSt
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 integer,intent(in) :: ngfftf(18)
 real(dp),intent(inout) :: rhor_out(nfftot,nspden*dimrho)
 complex(dpc),intent(out) :: m_lda_to_qp(BSt%mband,BSt%mband,BSt%nkpt,BSt%nsppol)

!Local variables-------------------------------
!scalars
 integer :: ib,ii,ik,isppol,nbR,nkR,nsR,unqps,rank,ispden,master
 integer :: ifft,n1,n2,n3,ir1,ir2,ir3
 integer :: cplex,optin,optout,nfft_found
 real(dp) :: uerr,rho_intp,nelect_qps,ratio
 logical,parameter :: use_FFT_interpolation=.TRUE.
 logical :: lfile,ltest
 character(len=500) :: msg
!arrays
 integer :: ngfft_found(18)
 real(dp) :: kibz(3),rr(3),rhogdum(1,1)
 real(dp),allocatable :: en_tmp(:)
 real(dp),allocatable :: rhor_tmp(:,:)
 complex(dpc),allocatable :: mtmp(:,:),utest(:,:)
 logical,allocatable :: kmask(:)

! *************************************************************************

 DBG_ENTER("COLL")

 ltest=ALL(BSt%nband(:)==BSt%nband(1))
 call assert(ltest,'Number of bands must be constant',__FILE__,__LINE__)
 ltest=(dimrho==0.or.dimrho==1)
 call assert(ltest,'dimrho must be 0 or 1',__FILE__,__LINE__)

 master=0
 call xmpi_me(rank)
 !
 ! * Check whether file exists or not.
 write(msg,'(5a)')ch10,&
& ' rdqps: reading QP wavefunctions of the previous step ',ch10,&
& '        looking for file ',TRIM(fname)
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')

 inquire(file=fname,exist=lfile)
 if (.not.lfile) then
  write(msg,'(2a)')' file not found, 1st iteration initialized with KS eigenelements ',ch10
  call wrtout(std_out,msg,'COLL') 
  call wrtout(ab_out, msg,'COLL')
  nscf=0
  RETURN
 end if

 unqps=get_unit()
 open(unit=unqps,file=fname,form='formatted',status='unknown')
 ! TODO the _QPS file should contain additional information

 read(unqps,*)nscf

 write(msg,'(a,i4,a)')' Number of iteration(s) already performed: ',nscf,ch10
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')

 read(unqps,*)nkR

 if (nkR/=BSt%nkpt) then 
  write(msg,'(3a,i4,a,i4)')&
&  ' Wrong number of k-points ',ch10,&
&  ' Expected: ',BSt%nkpt,', Found: ',nkR 
  MSG_ERROR(msg)
 end if

 read(unqps,*)nbR
 nbsc=MIN(nbR,BSt%mband)

 if (nbsc/=BSt%mband) then 
  write(msg,'(3a,i4,a,i4)')&
&  ' QPS file contains less bands than that used in the present calculation ',ch10,&
&  ' Required: ',BSt%mband,', Found: ',nbR
  MSG_WARNING(msg)
 end if

 if (nbsc/=nbR) then 
  write(msg,'(3a,i4,a)')&
&  ' The QPS file contains more bands than that used in the present calculation ',ch10,&
&  ' only the first ',nbR,' bands will be read'
  MSG_COMMENT(msg)
 end if

 allocate(mtmp(nbR,nbR),en_tmp(nbR))
 read(unqps,*)nsR

 ltest=(nsR==BSt%nsppol)
 call assert(ltest,'QPS file generated with different nsppol',__FILE__,__LINE__)
 ! 
 ! === Read energies and transformation for each k-point and spin ===
 ! TODO: The format of the QPS file must be standardized !
 ! For example we might add the occupation numbers.
 do isppol=1,BSt%nsppol 
  do ik=1,BSt%nkpt
   read(unqps,*)kibz(:)
   write(msg,'(a,i5,a,3(f6.3,1x),4x,a,i2)')' Reading ik ',ik,')  k = ',kibz(:),' is = ',isppol
   call wrtout(std_out,msg,'COLL')
   ltest=(ALL(ABS(kibz(:)-BSt%kptns(:,ik))<0.001)) 
   call assert(ltest,'Wrong k-point read',__FILE__,__LINE__)
   do ib=1,nbR
    read(unqps,*)en_tmp(ib)
    read(unqps,*)mtmp(:,ib)
   end do 

   ! === Store transformation and update energies ===
   m_lda_to_qp(1:nbsc,1:nbsc,ik,isppol)=mtmp(1:nbsc,1:nbsc)
   BSt%eig(1:nbsc,ik,isppol)=en_tmp(1:nbsc)

   ! * Chech if matrix is unitary.
   allocate(utest(nbsc,nbsc))
   utest=TRANSPOSE(mtmp(1:nbsc,1:nbsc)) !this is just for the buggy gfortran
   utest=MATMUL(CONJG(utest),mtmp(1:nbsc,1:nbsc))
   do ii=1,nbsc
    utest(ii,ii)=utest(ii,ii)-one
   end do
   uerr=MAXVAL(ABS(utest))
   if (uerr>tol6) then
    write(msg,'(a,es16.8)')' KS -> QP matrix is not unitary, MAX error = ',uerr
    MSG_WARNING(msg)
   end if
   deallocate(utest)

  end do !ik
 end do !isppol 

 deallocate(mtmp,en_tmp)
 ! 
 ! === Read the QP density ===
 ! * Be careful since the FFT grid might differ.
 ! * In case perform FFT interpolation to have rhor on the input mesh.
 if (dimrho==1) then
  read(unqps,*)n1,n2,n3

  if (ngfftf(1)==n1.and.ngfftf(2)==n2.and.ngfftf(3)==n3) then
   read(unqps,*)rhor_out(:,:)
  else 
   write(msg,'(2a,a,5(i3,a),i3)')&
&   ' FFT meshes differ. Performing Fourier interpolation. ',ch10,&
&   ' Found: ',n1,' x',n2,' x',n3,'; Expected: ',ngfftf(1),' x',ngfftf(2),' x',ngfftf(3)
   MSG_COMMENT(msg)

   allocate(rhor_tmp(n1*n2*n3,nspden))
   read(unqps,*)rhor_tmp(:,:)

   if (use_FFT_interpolation) then
    ngfft_found(1)=n1
    ngfft_found(2)=n2
    ngfft_found(3)=n3
    ngfft_found(4)=2*(ngfft_found(1)/2)+1 ! 4:18 are not used, anyway!
    ngfft_found(5)=2*(ngfft_found(2)/2)+1
    ngfft_found(6)=ngfft_found(3)
    ngfft_found(7:18)=ngfftf(7:18)
    nfft_found=PRODUCT(ngfft_found(1:3)) !no FFT para

    cplex =1 ! Real quantities.
    optin =0 ! Input is taken from rhor.
    optout=0 ! Output is only in real space. 

    call four_intepol(cplex,nspden,optin,optout,nfft_found,ngfft_found,nfftot,ngfftf,&
&    paral_kgb,MPI_enreg,rhor_tmp,rhor_out,rhogdum,rhogdum)
    !write(77,*)rhor_out
   else
    ! * Linear interpolation.
    do ispden=1,nspden
     do ir3=0,ngfftf(3)-1
      rr(3)=DBLE(ir3)/n3
      do ir2=0,ngfftf(2)-1
       rr(2)=DBLE(ir2)/n2
       do ir1=0,ngfftf(1)-1
        rr(1)=DBLE(ir1)/n1
        call interpol3d(rr,n1,n2,n3,rho_intp,rhor_tmp(:,ispden))
        ifft = 1 +ir1 +ir2*ngfftf(1) +ir3*ngfftf(1)*ngfftf(2)
        rhor_out(ifft,ispden)=rho_intp
       end do
      end do
     end do
    end do
    !write(78,*)rhor_out
   end if

   deallocate(rhor_tmp)
  end if

  ! === Test the normalization of the QPS density ===
  ! * There might be errors due to the interpolation or the truncation of the G basis set
  ! * Density will be renormalized in the caller since for PAW we still have to add the onsite contribution.
  if (usepaw==0) then
   nelect_qps=SUM(rhor_out(:,1))*ucvol/nfftot ; ratio=BSt%nelect/nelect_qps
   write(msg,'(3(a,f9.4))')&
&   ' Number of electrons calculated using the QPS density = ',nelect_qps,' Expected = ',BSt%nelect,' ratio = ',ratio
   call wrtout(std_out,msg,'COLL') 
   !!rhor_out(:,:)=ratio*rhor_out(:,:)
  end if
 end if !dimrho=1

 close(unqps)

 !write(msg,'(a)')' rdqps : File successfully read '
 !call wrtout(std_out,msg,'COLL')

 DBG_EXIT("COLL")

end subroutine rdqps
!!***

!!****f* m_qparticles/show_QP
!! NAME
!! show_QP
!!
!! FUNCTION
!! Print in a nice format (?) the expansion coefficients of the quasiparticle 
!! amplitudes in terms of the KS eigenvectors
!!
!! INPUTS
!!  Bst<Bandstructure_type>=Description of the band structure.
!!    %nsppol=1 for unpolarized, 2 for spin-polarized.
!!    %mband=Max number of bands (in GW doesn"t depend on k an spin) 
!!    %nkpt=number of irreducible k-points.
!!    %eig(mband,nkpt,nsppol)= QP energies for each k-point, band and spin.
!!  m_lda_to_qp(nbnds,nbnds,nkibz,nsppol)=matrix giving the decomposition of the QP
!!   amplitued in the mainfold generated by the KS wavefunctions 
!!   (i.e $ m_lda_to_qp(ib,jb,k,s) := \langle \psi_{ib,k,s}^{KS}| \psi_{jb,k,s}^{QP}\rangle $
!!  fromb,tob=initial and final band index for QP, only states in this range are printed
!!  prtvol=Verbosity level (not used)
!!  unit=Unit number of the output file
!! tolmat[Optional]=Only components whose coefficient has modulus larger than tolmat are shown (default is 0.01)
!!
!! OUTPUT
!!  Only printing
!!
!! NOTES
!!  Only master node should call this routine.
!!
!! TODO 
!! use m_fstrings
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      assert,wrtout
!!
!! SOURCE

subroutine show_QP(Bst,m_lda_to_qp,fromb,tob,unit,prtvol,tolmat,kmask)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_27_toolbox_oop
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: fromb,tob
 integer,optional,intent(in) :: prtvol,unit
 real(dp),optional,intent(in) :: tolmat
 type(Bandstructure_type),intent(in) :: Bst
!arrays
 logical,optional,intent(in) :: kmask(Bst%nkpt)
 complex(dpc),intent(in) :: m_lda_to_qp(Bst%mband,Bst%mband,Bst%nkpt,Bst%nsppol)

!Local variables-------------------------------
!scalars
 integer :: nbra=5
 integer :: ib_start,ib_stop,verbose
 integer :: counter,ib_KS,ib_QP,ii,ikibz,isp,nspace,unt,nband_k
 real(dp) :: my_tolmat
 character(len=10) :: bks,bqp,k_tag,spin_tag
 character(len=500) :: KS_row,KS_ket,tmpstr,QP_ket,fspace,msg

! *********************************************************************

 unt      =std_out   ; if (PRESENT(unit  )) unt      =unit
 verbose  =0         ; if (PRESENT(prtvol)) verbose  =prtvol
 ib_start =1         ; if (PRESENT(fromb )) ib_start =fromb
 ib_stop  =Bst%mband ; if (PRESENT(tob   )) ib_stop  =tob 
 my_tolmat=0.001     ; if (PRESENT(tolmat)) my_tolmat=ABS(tolmat)

 ! * I suppose nband_k is constant thus the check is done here.
 if (ib_start<=0       ) ib_start=1
 if (ib_start>Bst%mband) ib_start=Bst%mband
 if (ib_stop<=0        ) ib_stop=1
 if (ib_stop>Bst%mband ) ib_stop=Bst%mband

 ! Have to follow rules 7.f.
 write(unt,'(/,a,/,a,/,a,f5.3,a,/,a)')&
  ' '//REPEAT('*',76),&
& ' ***** QP amplitudes expressed as linear combination of KS eigenstates. *****',&
& ' ***** Only KS components whose modulus is larger than ',my_tolmat,' are shown ***** ',&
& ' '//REPEAT('*',76)

 if (PRESENT(kmask)) then 
  if (.not.ALL(kmask))write(unt,'(/,a,i3,a)')' Only ',COUNT(kmask),' k-points are reported '  
 end if

 do isp=1,Bst%nsppol 
  call int2char(isp,spin_tag) 
  write(unt,'(/,a,i2,a,/)')' >>>>> Begin block for spin ',isp,' <<<<< '

  do ikibz=1,Bst%nkpt
   if (PRESENT(kmask)) then 
    if (.not.kmask(ikibz)) CYCLE
   end if
   call int2char(ikibz,k_tag) 
   nband_k=Bst%nband(ikibz+(isp-1)*Bst%nkpt)
   write(unt,'(a,i4,a,3es16.8,a,f6.3,/)')&
&   ' k-point: ',ikibz,') ',Bst%kptns(:,ikibz),'; wtk= ',Bst%wtk(ikibz)

   do ib_QP=ib_start,ib_stop
    call int2char(ib_QP,bqp) 
    QP_ket=' |QP: b='//TRIM(bqp)//'; s='//TRIM(spin_tag)//'> = '
    write(unt,'(a)')TRIM(QP_ket)
    nspace=LEN(TRIM(QP_ket)) 
   
    counter=0 ; KS_row=REPEAT('',nspace+2)
    do ib_KS=1,Bst%mband
     if (ABS(m_lda_to_qp(ib_KS,ib_QP,ikibz,isp))<my_tolmat) CYCLE
     counter=counter+1
     call int2char(ib_KS,bks) 
     write(tmpstr,'(3a)')' |',TRIM(bks),'>'
     write(KS_ket,'(1x,2f7.3,a,1x)')m_lda_to_qp(ib_KS,ib_QP,ikibz,isp),TRIM(tmpstr)
     KS_row=TRIM(KS_row)//TRIM(KS_ket)
     if (MOD(counter,nbra)==0) then  ! nbra KS kets per row
      write(unt,'(a)')TRIM(KS_row)
      KS_row=REPEAT('',nspace+2)
     end if
    end do

    if (MOD(counter,nbra)/=0) write(unt,'(a)')TRIM(KS_row) ! Last row, if any
    write(unt,'(a)')''
   end do !ib_QP

  end do !ikibz
 end do !isp

 write(unt,'(a,/)')' '//REPEAT('*',76)

end subroutine show_QP
!!***

!!****f* m_qparticles/rdgw
!! NAME
!! rdgw
!!
!! FUNCTION
!!  This subroutine reads the GW corrections from a _GW file
!!
!! INPUTS
!!  Bst<Bandstructure_type>=type describing the Band structure.
!!    %nbnds=number of bands.
!!    %nkpt=number of irred k-points.
!!    %nsppol=number of spin
!!    %kptns(3,nkpt)=irreducible k-points
!!
!! OUTPUT
!!  gwenergy(nkpt,nbnds,nsppol) : QP energies as read or derived from the data contained
!!   in the external file
!! 
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      assert,wrtout
!!
!! SOURCE
!!

subroutine rdgw(Bst,fname,gwenergy)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 use m_numeric_tools, only : linfit
 use m_io_tools,      only : get_unit
 use m_electrons,     only : get_valence_idx

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 character(len=*),intent(in) :: fname
 type(Bandstructure_type),intent(inout) :: Bst
!arrays
 real(dp),intent(out) :: gwenergy(Bst%mband,Bst%nkpt,Bst%nsppol)

!Local variables ------------------------------
!scalars
 integer :: ib,ibr,ik,ikibz,ikr,ios,is,nn,nbR,nkR,nsR,unt,nbv
 real(dp) :: alpha,beta,degw,egw,smrt
 logical :: ltest
 character(len=500) :: msg
!arrays
 integer,allocatable :: vbik(:,:) 
 real(dp) :: kread(3)
 real(dp),allocatable :: gwcorr(:,:,:)
!************************************************************************

 ltest=ALL(Bst%nband(:)==Bst%mband) 
 call assert(ltest,'nband should be costant',__FILE__,__LINE__)

 write(msg,'(2a)')' reading GW corrections from file ',TRIM(fname)
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')
 
 unt=get_unit()
 open(unt,file=fname,status='old',iostat=ios)
 if (ios/=0) then 
  write(msg,'(3a)')' Opening file: ',TRIM(fname),' as old'
  MSG_ERROR(msg)
 end if

 read(unt,*)nkR,nsR 
 if (nkR/=Bst%nkpt) then  
  write(msg,'(a,i4,a,i4,3a)')&
&  ' Found less k-points than that required ',nkR,'/',Bst%nkpt,ch10,&
&  ' Some k-points will be skipped. ',ch10
  MSG_WARNING(msg)
 end if 

 ltest=(nsR==Bst%nsppol)
 call assert(ltest,'mismatch in nsppol',__FILE__,__LINE__)

 allocate(gwcorr(Bst%mband,Bst%nkpt,Bst%nsppol)) 
 gwcorr(:,:,:)=zero

 do is=1,Bst%nsppol
  do ikr=1,nkR
   read(unt,*)kread
   read(unt,*)nbR
   ikibz=0
   do ik=1,Bst%nkpt
    if (ALL(ABS(kread(:)-Bst%kptns(:,ik))<0.0001)) ikibz=ik
   end do
   do ib=1,nbR
    read(unt,*)ibr,egw,degw
    if (ibr<=Bst%mband .and.ikibz/=0) gwcorr(ibr,ikibz,is)=degw/Ha_eV
   end do
  end do 
 end do
 close(unt)

 allocate(vbik(BSt%nkpt,BSt%nsppol))
 vbik(:,:) = get_valence_idx(BSt)

 do is=1,Bst%nsppol
  do ik=1,Bst%nkpt
   ! nbv is the index of the (valence band| Fermi band) for each spin
   nbv=vbik(ik,is) ; nn=Bst%mband-nbv

   do ib=nbv+1,Bst%mband
    if (gwcorr(ib,ik,is)==0) then
     nn=ib-1-nbv
     if (nn>1) then
      write(msg,'(a)')' linear extrapolating (conduction) GW corrections beyond the read values'
      call wrtout(std_out,msg,'COLL') 
      smrt=linfit(nn,gwenergy(nbv+1:nbv+nn,ik,is),gwcorr(nbv+1:nbv+nn,ik,is),alpha,beta)
     else
      write(msg,'(a)')' assuming constant (conduction) GW corrections beyond the read values'
      call wrtout(std_out,msg,'COLL') 
      alpha=zero 
      beta =gwcorr(nbv+nn,ik,is)
     end if
     EXIT !ib loop
    end if
   end do !ib

   do ib=nbv+nn+1,Bst%mband
    gwcorr(ib,ik,is)= alpha*gwenergy(ib,ik,is) + beta
   end do

   nn=nbv
   do ib=nbv,1,-1
    if (gwcorr(ib,ik,is)==0) then
     nn=nbv-ib
     if (nn>1) then
      write(msg,'(a)')' linear extrapolating (valence) GW corrections beyond the read values'
      call wrtout(std_out,msg,'COLL') 
      smrt=linfit(nn,gwenergy(nbv-nn+1:nbv,ik,is),gwcorr(nbv-nn+1:nbv,ik,is),alpha,beta)
     else
      write(msg,'(a)')' assuming constant (valence) GW corrections beyond the read values'
      call wrtout(std_out,msg,'COLL') 
      alpha=zero 
      beta =gwcorr(nbv,ik,is)
     end if
     exit !ib
    end if
   end do !ib

   do ib=1,nbv-nn
    gwcorr(ib,ik,is)=alpha*gwenergy(ib,ik,is) + beta
   end do

  end do !ik
 end do !is

 write(msg,'(a)')' k  s     GW corrections [eV] '
 call wrtout(std_out,msg,'COLL')
 do is=1,Bst%nsppol
  do ik=1,Bst%nkpt
   write(*,'(i3,1x,i3,10f7.2/50(10x,10f7.2/))')ik,is,(Ha_eV*gwcorr(ib,ik,is),ib=1,Bst%mband)
  end do
 end do 
 gwenergy(:,:,:)=gwenergy(:,:,:)+gwcorr(:,:,:)

 write(msg,'(a)')' k   s    GW eigenvalues [eV]'
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')
 do is=1,Bst%nsppol
  do ik=1,Bst%nkpt
   write(std_out,'(i3,7x,10f7.2/50(10x,10f7.2/))')ik,is,(Ha_eV*gwenergy(ib,ik,is),ib=1,Bst%mband)
  end do
 end do
 write(std_out,*)

 deallocate(gwcorr,vbik)

end subroutine rdgw

END MODULE m_qparticles
!!***
