!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_wfs
!! NAME
!!  m_wfs
!!
!! FUNCTION
!!  This module contains tools to operate on the wavefunctions_information object.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2009 ABINIT group (FB, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_wfs

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors

 use m_gwdefs, only : czero_gw, cone_gw

 implicit none

 private 

 public ::           & 
&  init_Wfs,         &  ! Creation method. 
&  destroy_Wfs,      &  ! Destructor.
&  reinit_Wfs,       &  ! Reinitialize memory storage of u(r).
&  get_wfr,          &  ! Get wavefunction in real space from the object.
&  duplicate_Wfs,    &  ! Copy the content on the object taking into account spreading.
&  nullify_Wfs,      &  ! Set all pointers to null()
&  print_Wfs,        &  ! Printout of basic info.
&  fft_onewfn,       &  ! Perform a single FFT from G to R.
&  calc_wfwfg,       &
&  calc_wf_qp,       &
&  calc_wf_qp_Wfval

CONTAINS  !=========================================================================================================================
!!***

!!****f* m_wfs/init_Wfs
!! NAME
!! init_Wfs
!!
!! FUNCTION
!!  Initialize the object.
!!
!! INPUTS
!!  ngfft(18)=All needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  nkibz=Number of irreducible k-points.
!!  npwwfn=Number of plane waves for u(G).
!!  nsppol=Number of independent spin polarizations.
!!  nspden=Number of density components.
!!  nspinor=Number of spinorial components.
!!  my_minb,my_maxb = min and MAX band treated by this processor.
!!  MPI_enreg=Datatype gathering information on the parallelism.
!!  gwmem=Option for memory storage.
!!  paral_kgb=Option for band-FFT parallelism (not yet available)
!!  gvec(3,npwwfn)=G-vectors in reduced coordinates.
!!
!! OUTPUT
!!  Initialize the object with basic dimensions, allocate also memory for u(g) and u(r) according to gwmem.
!!  %wfg(npwwfn,my_minb:my_maxb,nk,nsppol) in reciprocal space are always allocated.
!!  %wfr(nfft,my_minb:my_maxb,nk,nsppol) in real space only if gwmem = ?1
!!  igfft0(Wf%npwwfn)=Index of G in the FFT box.
!!  is_already_stored=Table storing allocation status of wfr.
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine init_Wfs(Wfs,gwmem,paral_kgb,npwwfn,&
& my_minb,my_maxb,nkibz,nsppol,nspden,nspinor,ngfft,gvec,MPI_enreg)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_53_ffts
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: gwmem,paral_kgb,my_maxb,my_minb
 integer,intent(in) :: nkibz,npwwfn,nsppol,nspden,nspinor
 type(MPI_type),intent(in) :: MPI_enreg
 type(wavefunctions_information),intent(out) :: Wfs
!array
 integer,intent(in) :: ngfft(18)
 integer,intent(in) :: gvec(3,npwwfn)

!Local variables ------------------------------
!scalars
 integer :: istat
 real(dp) :: wgsize,wrsize
 character(len=500) :: msg
!arrays
 logical,allocatable :: mask(:)
!************************************************************************

 DBG_ENTER("COLL")

 call nullify_Wfs(Wfs)

 ! === Basic dimensions ===
 Wfs%nk        = nkibz
 Wfs%my_minb   = my_minb
 Wfs%my_maxb   = my_maxb
 Wfs%nsppol    = nsppol
 Wfs%nspden    = nspden
 Wfs%nspinor   = nspinor
 Wfs%npwwfn    = npwwfn

 Wfs%gwmem     = gwmem
 Wfs%paral_kgb = paral_kgb

 Wfs%ngfft     = ngfft(:)
 Wfs%nfftot    = PRODUCT(ngfft(1:3)) 
 Wfs%nfft      = Wfs%nfftot !MG TODO At present no FFT para

 ! === Index of the G-sphere in the FFT box ===
 allocate(Wfs%igfft0(Wfs%npwwfn))

 allocate(mask(Wfs%npwwfn))
 call kgindex(Wfs%igfft0,gvec,mask,MPI_enreg,ngfft,Wfs%npwwfn)
 if (.not.ALL(mask)) then 
  MSG_ERROR("FFT para not yet implemented")
 end if
 deallocate(mask)

 allocate(Wfs%is_already_stored(my_minb:my_maxb,nkibz,nsppol))
 Wfs%is_already_stored(:,:,:)=.FALSE.

 ! === Allocate u(g) and, if required, also u(r) ===
 wgsize=nspinor*npwwfn*(my_maxb-my_minb+1)*nkibz*nsppol
 write(msg,'(a,f12.1,a)')' Memory required for wfg= ',2*gwpc*wgsize*b2Mb,' [Mb]'
 call wrtout(std_out,msg,'PERS')

 allocate(Wfs%wfg(npwwfn*nspinor,my_minb:my_maxb,nkibz,nsppol), STAT=istat)
 ABI_CHECK(istat==0,'out-of-memory in wfg')
 Wfs%wfg(:,:,:,:)=czero_gw

 select case (MODULO(Wfs%gwmem,10))
 case (0)
  msg=' Wavefunctions in real space are NOT stored in memory! '
  MSG_COMMENT(msg)

 case (1)
  wrsize=Wfs%nfft*nspinor*(my_maxb-my_minb+1)*nkibz*nsppol
  write(msg,'(3a,f12.1,a)')&
&  ' init_Wfs: Wavefunctions in real space are stored in memory. ',ch10,&
&  ' Memory required for wfr= ',2*gwpc*wrsize*b2Mb,' [Mb]'
  call wrtout(std_out,msg,'PERS') 

  allocate(Wfs%wfr(Wfs%nfft*nspinor,my_minb:my_maxb,nkibz,nsppol), STAT=istat)
  ABI_CHECK(istat==0,'out-of-memory in wfr')

  Wfs%wfr=czero_gw

 case DEFAULT
  msg=' Wfs%gwmem/=x0,x1 not yet implemented '
  MSG_BUG(msg)
 end select

 DBG_EXIT("COLL")

end subroutine init_Wfs
!!***

!!****f* m_wfs/destroy_Wfs
!! NAME
!!  destroy_Wfs
!!
!! FUNCTION
!!  Free the dynamic entities in the data type
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine destroy_Wfs(Wfs)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(wavefunctions_information),intent(inout) :: Wfs
!************************************************************************

 DBG_ENTER("COLL")

 if (associated(Wfs%igfft0           )) deallocate(Wfs%igfft0           )
 if (associated(Wfs%is_already_stored)) deallocate(Wfs%is_already_stored)
 if (associated(Wfs%wfg              )) deallocate(Wfs%wfg              )
 if (associated(Wfs%wfr              )) deallocate(Wfs%wfr              )

 DBG_EXIT("COLL")

end subroutine destroy_Wfs
!!***

!!****f* m_wfs/reinit_Wfs
!! NAME
!!  reinit_Wfs
!!
!! FUNCTION
!!  reinitialize the storage mode
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine reinit_Wfs(Wfs)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(wavefunctions_information),intent(inout) :: Wfs
!************************************************************************

 Wfs%is_already_stored(:,:,:)=.FALSE.

end subroutine reinit_Wfs
!!***

!!****f* m_wfs/get_wfr
!! NAME
!!  get_wfr
!!
!! FUNCTION
!!  Get a wave function is real space, either by doing a FFT G-->R
!!  or by just retrieving the data already stored in the data type
!!
!! INPUTS
!!  Wfs<wavefunctions_information>=the data type
!!  MPI_enreg=Info on the parallelism
!!  ib=bands index
!!  ik=Index of the k-point in the IBZ 
!!  is=spin index
!!
!! OUTPUT
!!  wfr(Wfs%nfft)=the required wavefunction in real space
!!
!! PARENTS
!!      calc_density,calc_vHxc_braket,cchi0,cchi0q0,csigme,get_bands_sym_GW
!!      m_wfs
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine get_wfr(Wfs,MPI_enreg,ib,ik,is,wfr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ib,ik,is
 type(wavefunctions_information),intent(inout) :: Wfs
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 complex(gwpc),intent(out) :: wfr(Wfs%nfft*Wfs%nspinor)

!Local variables ------------------------------
!scalars
 integer,parameter :: tim_fourdp=5
 integer :: istat,npwwfn,nfft,nspinor
 character(len=500) :: msg
!************************************************************************

!£#if defined DEBUG_MODE
 if (ib>Wfs%my_maxb.or.ib<Wfs%my_minb) then
  write(msg,'(3a,i4,a,2i4)')&
&  ' Band index out of range. ',ch10,&
&  ' Requiring ib= ',ib,' while range is ',Wfs%my_minb,Wfs%my_maxb 
  MSG_BUG(msg)
 end if

 if (ik>Wfs%nk) then
  write(msg,'(3a,i4,a,i4)')&
&  ' k-point index out of range. ',ch10,&
&  ' Requiring ik= ',ik,' while nkpt= ',Wfs%nk
  MSG_BUG(msg)
 end if

 if (is>Wfs%nsppol) then
  write(msg,'(3a,i4,a,i4)')&
&  ' Spin index out of range. ',ch10,&
&  ' Requiring is= ',is,' while nsppol= ',Wfs%nsppol 
  MSG_BUG(msg)
 end if
!£#endif

 ! MG Here it would be better using a pointer instead of an array
 ! If wfr is big a lot of time would be wasted to copy %wfr

 if (.not.Wfs%is_already_stored(ib,ik,is)) then
  npwwfn = Wfs%npwwfn
  nfft   = Wfs%nfft
  nspinor= Wfs%nspinor

  call fft_onewfn(Wfs%paral_kgb,nspinor,npwwfn,nfft,Wfs%wfg(:,ib,ik,is),wfr,&
&  Wfs%igfft0,Wfs%ngfft,tim_fourdp,MPI_enreg)

  if (Wfs%gwmem==1) then
   Wfs%wfr(:,ib,ik,is)=wfr(:)
   Wfs%is_already_stored(ib,ik,is)=.TRUE.
   !write(*,'(a,3(i4,x),a)')' wavefunction',ib,ik,is,'stored'
  end if

 else 
  ! * wfr is_already_stored, just copy it back
  wfr(:)=Wfs%wfr(:,ib,ik,is)
 end if 

end subroutine get_wfr
!!***

!!****f* m_wfs/duplicate_Wfs
!! NAME
!!  duplicate_Wfs
!!
!! FUNCTION
!!  Copy the content of a wavefunctions_information data type 
!!  taking into account the spreading among processors
!!
!! INPUTS
!!  MPI_enreg=information about MPI parallelization
!!  Kmesh<BZ_mesh_type>=Structure reporting information on the k-point sampling 
!!  kcalc
!!
!! OUTPUT
!!  See sides effect
!!
!! SIDES EFFECT
!!  Wfs=
!!  Wfs_braket=
!!  
!!
!! PARENTS
!!      sigma
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine duplicate_Wfs(MPI_enreg,Wfs,Wfs_braket,kcalc,Kmesh)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_14_hidewrite
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(MPI_type),intent(in) :: MPI_enreg
 type(wavefunctions_information),intent(inout) :: Wfs,Wfs_braket
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 integer,intent(in) :: kcalc(Wfs_braket%nk)

!Local variables ------------------------------
!scalars
 integer :: ib,ik,ikibz,is,my_maxb,my_minb,nsppol,nkcalc
 integer :: rank,spaceComm,ikbs_proc_rank,ierr
 character(len=500) :: msg
!************************************************************************

 call xcomm_init  (MPI_enreg,spaceComm)
 call xme_init    (MPI_enreg,rank     )
 !write(*,*) me,MPI_enreg%proc_distrb(:,:,:)

 ! For band parallelism the processors must communicate
 ! the wavefunctions where the GW corrections are required
 if (MPI_enreg%gwpara==2) then
  write(msg,'(3a,2(a,i3))')ch10,&
&  ' band parallelism : communicating wavefunctions ',ch10,&
&  ' from band = ',Wfs_braket%my_minb,' up to band = ',Wfs_braket%my_maxb
  call wrtout(std_out,msg,'PERS')
 end if

 nsppol  = Wfs_braket%nsppol
 nkcalc  = Wfs_braket%nk
 my_minb = Wfs_braket%my_minb
 my_maxb = Wfs_braket%my_maxb

 do is=1,nsppol
  do ik=1,nkcalc
   ikibz=Kmesh%tab(kcalc(ik))
   !write(*,*)' === me',me,ik,kcalc(ik),ikibz

   do ib=my_minb,my_maxb
    ikbs_proc_rank=MPI_enreg%proc_distrb(ikibz,ib,is)
    !write(*,*)'== rank',ib,ikbs_proc_rank
    if (rank==ikbs_proc_rank) Wfs_braket%wfg(:,ib,ik,is)=Wfs%wfg(:,ib,ikibz,is)
    if (MPI_enreg%gwpara==2) then 
     call xcast_mpi(Wfs_braket%wfg(:,ib,ik,is),ikbs_proc_rank,spaceComm,ierr)
    end if
   end do

  end do
 end do

 call leave_test(MPI_enreg)

end subroutine duplicate_Wfs
!!***

!!****f* m_wfs/nullify_Wfs
!! NAME
!!  nullify_Wfs
!!
!! FUNCTION
!!  Nullify the pointers of the data structure.
!!
!! PARENTS
!!      m_wfs,screening
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine nullify_Wfs(Wfs)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(wavefunctions_information),intent(inout) :: Wfs
!************************************************************************

 nullify(Wfs%igfft0           )
 nullify(Wfs%is_already_stored)
 nullify(Wfs%wfg              )
 nullify(Wfs%wfr              )

end subroutine nullify_Wfs
!!***

!!****f* m_wfs/print_Wfs
!! NAME
!! print_Wfs
!!
!! FUNCTION
!!  Print the content of a wavefunctions_information datatype
!!
!! INPUTS
!!  Wfs<wavefunctions_information>=The datatype.
!!  [unitno]=Unit number for output
!!  [prtvol]=Verbosity level
!!  [mode_paral]=Either "COLL" or "PERS"
!!
!! OUTPUT
!!  Only printing 
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine print_Wfs(Wfs,unitno,prtvol,mode_paral)
    
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 type(wavefunctions_information),intent(in) :: Wfs
 integer,optional,intent(in) :: unitno,prtvol
 character(len=4),optional,intent(in) :: mode_paral

!local variables-------------------------------
 integer :: verb,unt
 character(len=4) :: mode
 character(len=500) :: msg      
! *************************************************************************

 verb=0       ; if (PRESENT(prtvol    )) verb=prtvol
 unt =std_out ; if (PRESENT(unitno    )) unt =unitno
 mode='COLL'  ; if (PRESENT(mode_paral)) mode=mode_paral

 write(msg,'(2a,3(a,i5,a),a,i5)')&
& ' ==== Content of the Wfs datatype ==== ',ch10,&
& '   Number of irreducible k-points ........ ',Wfs%nk,ch10,&
& '   Number of spinorial components ........ ',Wfs%nspinor,ch10,&
& '   Number of spin-density components ..... ',Wfs%nspden,ch10,&
& '   Number of spin polarizations .......... ',Wfs%nsppol
 call wrtout(unt,msg,mode)
 write(msg,'(5(a,i5,a),a,2i5)')&
& '   Number of reciprocal lattice vectors .. ',Wfs%npwwfn,ch10,&
& '   Memory storage option (gwmem) ......... ',Wfs%gwmem,ch10,&
& '   Total number of FFT points ....... .... ',Wfs%nfftot,ch10,&
& '   Number of FFT points treated by me .... ',Wfs%nfft,ch10,&
& '   Parallelism over k-b-g (paral_kgb) .... ',Wfs%paral_kgb,ch10,&
& '   min and Max band index stored by me ... ',Wfs%my_minb,Wfs%my_maxb
 call wrtout(unt,msg,mode)

 call print_ngfft(Wfs%ngfft,'FFT mesh for wavefunctions',unt,mode,verb)

end subroutine print_Wfs
!!***

!!****f* m_wfs/fft_onewfn
!! NAME
!! fft_onewfn
!!
!! FUNCTION
!! Calculate ONE wavefunction in real space using FFT
!!
!! INPUTS
!! nspinor=number of spinorial components
!! igfft(npwwfn)=index of each plane wave in FFT grid
!! ngfft(18)=contain all needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!! nkibz=number of k points
!! npwwfn=number of plane waves
!! nsppol=number of independent spin polarizations 
!! tim_fourdp=4 if called from within screening ; =5 if called from within sigma
!! wfg(npwwfn,my_minb:my_maxb,nkibz,nsppol)=wavefunctions in reciprocal space treated by this processor.
!! my_minb,my_maxb = min and max band treated by this processor
!! MPI_enreg= datatype containing information on parallelism to be passed to fourdp
!!
!! OUTPUT
!!  wfr(ngfft(1)*ngfft(2)*ngfft(3)*nspinor)=wavefunctions in real space.
!!
!! PARENTS
!!      calc_density,calc_vHxc_braket,m_wfs
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine fft_onewfn(paral_kgb,nspinor,npwwfn,nfftot,wfg,wfr,igfft,ngfft,tim_fourdp,MPI_enreg)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_ffts
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: paral_kgb,npwwfn,nfftot,tim_fourdp,nspinor
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 integer,intent(in) :: igfft(npwwfn),ngfft(18)
 complex(gwpc),intent(in) :: wfg(npwwfn*nspinor)
 complex(gwpc),intent(out) :: wfr(ngfft(1)*ngfft(2)*ngfft(3)*nspinor)

!Local variables-------------------------------
!scalars
 integer :: ispinor,ig,master,me,spaceComm,rspad,gspad
!arrays
 real(dp),allocatable :: wfg_dp(:,:),wfr_dp(:,:)

! *************************************************************************

 !call xcomm_init  (MPI_enreg,spaceComm) 
 !call xme_init    (MPI_enreg,me)         
 !call xmaster_init(MPI_enreg,master)  
 !MPI_enreg%me_fft=0 ; MPI_enreg%nproc_fft=1

 allocate(wfg_dp(2,nfftot),wfr_dp(2,nfftot))

 do ispinor=1,nspinor
  gspad=(ispinor-1)*npwwfn
  rspad=(ispinor-1)*nfftot
  !
  ! === Fill FFT array from PW array ===
  wfg_dp(:,:)=zero
  do ig=1,npwwfn
   wfg_dp(1,igfft(ig))=REAL (wfg(ig+gspad))
   wfg_dp(2,igfft(ig))=AIMAG(wfg(ig+gspad))
  end do
  !
  ! === Take FFT to give wfn in real space ===
  ! here wfr_dp doesnt has same shape as fofr
  call fourdp(2,wfg_dp(:,:),wfr_dp(:,:),+1,MPI_enreg,nfftot,ngfft,paral_kgb,tim_fourdp)
  
  wfr(1+rspad:nfftot+rspad)=CMPLX(wfr_dp(1,:),wfr_dp(2,:),kind=gwpc)
 end do

 deallocate(wfg_dp,wfr_dp)

end subroutine fft_onewfn
!!***

!!****f* m_wfs/calc_wfwfg
!! NAME
!! calc_wfwfg
!!
!! FUNCTION
!!  Calculate the Fourier transform of the product u_{bk}^*(r).u_{b"k}(r) 
!!  at an arbitrary k in the BZ.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      cchi0,cchi0q0
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine calc_wfwfg(MPI_enreg,paral_kgb,tim_fourdp,ktabr_k,ktabi_k,nfftot,ngfft_gw,wfr_jb,wfr_kb,wfg2_jk)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_ffts
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ktabi_k,nfftot,paral_kgb,tim_fourdp
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 integer,intent(in) :: ktabr_k(nfftot),ngfft_gw(18)
 complex(gwpc),intent(in) :: wfr_jb(nfftot),wfr_kb(nfftot)
 complex(gwpc),intent(out) :: wfg2_jk(nfftot)

!Local variables-------------------------------
!arrays
 real(dp),allocatable :: wfg2_tmp(:,:),wfr2_tmp(:,:)

! *************************************************************************

 ! There is no need to take into account phases arising from non-symmorphic
 ! operations since the wavefunctions are evaluated at the same k-point.

 allocate(wfg2_tmp(2,nfftot),wfr2_tmp(2,nfftot))

 wfr2_tmp(1,:)=   real(wfr_jb(ktabr_k(:))) *  real(wfr_kb(ktabr_k(:)))&
&               +aimag(wfr_jb(ktabr_k(:))) * aimag(wfr_kb(ktabr_k(:)))

 wfr2_tmp(2,:)=  real(wfr_jb(ktabr_k(:))) * aimag(wfr_kb(ktabr_k(:)))&
&              -aimag(wfr_jb(ktabr_k(:))) *  real(wfr_kb(ktabr_k(:)))

 ! Conjugate the product if time-reversal is used to reconstruct this k-point
 if (ktabi_k==-1) wfr2_tmp(2,:)=-wfr2_tmp(2,:)

 call fourdp(2,wfg2_tmp,wfr2_tmp,-1,MPI_enreg,nfftot,ngfft_gw,paral_kgb,tim_fourdp)

 wfg2_jk(:)= wfg2_tmp(1,:)+(0.,1.)*wfg2_tmp(2,:)

 deallocate(wfg2_tmp,wfr2_tmp)

end subroutine calc_wfwfg
!!***

!!****f* m_wfs/calc_wf_qp
!! NAME
!! calc_wf_qp
!!
!! FUNCTION
!!  Calculate QP amplitudes in real or reciprocal space starting from the 
!!  KS wavefunctions and the corresponding expansion coefficients.
!!  Take into account possible spreading of bands.
!!
!! INPUTS
!!  b1gw, b2gw = Min and max band index over k-point and spin for GW corrections.
!!  nkibz=number of k-points.
!!  nsize= number of points in real space or number of G vectors.
!!  nsppol=number of spin.
!!  nbnds=number of bands in the present GW calculation.
!!  my_minb, my_maxb = Indeces of the bands treated by this processor.
!!  m_lda_to_qp(nbnds,nbnds,nkibz,nsppol)= expansion of the QP amplitudes in terms of KS wavefunctions.
!!
!! OUTPUT
!!  wf(nsize,my_minb:my_maxb,nkibz,nsppol)= Updated QP amplitudes for this processor.
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE
!!

subroutine calc_wf_qp(MPI_enreg,nkibz,nbnds,nsize,nsppol,nspinor,&
& m_lda_to_qp,my_minb,my_maxb,b1gw,b2gw,wf)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_14_hidewrite
 use interfaces_51_manage_mpi
 use interfaces_linalg
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nbnds,nkibz,nsize,nsppol,nspinor,my_minb,my_maxb,b1gw,b2gw
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 complex(dpc),intent(in) :: m_lda_to_qp(nbnds,nbnds,nkibz,nsppol)
 complex(gwpc),intent(inout) :: wf(nsize*nspinor,my_minb:my_maxb,nkibz,nsppol)

!Local variables-------------------------------
!scalars
 integer :: ib,ik,is,ierr,lowerb,upperb,rangeb,ispinor,spad
 integer :: spaceComm,sizegw,rank,shift,indx_kibz,ilmn,nlmn
 character(len=500) :: msg
!arrays
 complex(gwpc),allocatable :: umat_k(:,:)
 complex(gwpc),allocatable :: wf_ks(:,:),wf_qp(:,:)
!************************************************************************

 DBG_ENTER("COLL")

 call xcomm_init(MPI_enreg,spaceComm)
 call xme_init  (MPI_enreg,rank)
 !
 ! === Determine the range of bands this processor has to treat ===
 lowerb=0  ! NO overlap between [b1gw,b2gw] and [my_minb,my_maxb]
 if (b1gw<=my_maxb) lowerb=MAX(b1gw,my_minb)
 upperb=0  ! NO overlap between [b1gw,b2gw] and [my_minb,my_maxb]
 if (b2gw>=my_minb) upperb=MIN(b2gw,my_maxb)
 rangeb=0
 if (lowerb/=0.and.upperb/=0) rangeb=upperb-lowerb+1
 sizegw=b2gw-b1gw+1

 if (rangeb/=0) then
  write(msg,'(2a,i3,3a,i3,1x,i3,a,3i3,2a,3i3)')ch10,&
&  ' proc. ',rank,' will update its wavefunctions ',ch10,&
&  ' my_bands indeces: ',my_minb,my_maxb,' gwrange: ',b1gw,b2gw,sizegw,ch10,&
&  ' lowerb, upperb, rangeb: ',lowerb,upperb,rangeb
  call wrtout(std_out,msg,'PERS') 
 end if

 allocate(umat_k(lowerb:upperb,b1gw:b2gw))
 allocate(wf_qp(nsize*nspinor,b1gw:b2gw))  
 allocate(wf_ks(nsize,lowerb:upperb))
 wf_qp(:,:)=czero_gw ; wf_ks(:,:)=czero_gw 
 !
 ! === Calculate : $\Psi^{QP}_{r,b} = \sum_n \Psi^{KS}_{r,n} M_{n,b}$ ===
 do is=1,nsppol
  do ik=1,nkibz

   umat_k(:,:)=m_lda_to_qp(lowerb:upperb,b1gw:b2gw,ik,is)
   wf_qp(:,:)=czero_gw

   if (rangeb/=0) then
    do ispinor=1,nspinor
     spad=nsize*(ispinor-1)
     wf_ks(:,lowerb:upperb)=wf(spad+1:spad+nsize,lowerb:upperb,ik,is)
#if defined HAVE_GW_DPC
     call ZGEMM('N','N',nsize,sizegw,rangeb,cone_gw,wf_ks(:,lowerb:upperb),nsize,&
&     umat_k,rangeb,czero_gw,wf_qp(spad+1:spad+nsize,b1gw:b2gw),nsize)
#else
     call CGEMM('N','N',nsize,sizegw,rangeb,cone_gw,wf_ks(:,lowerb:upperb),nsize,&
&     umat_k,rangeb,czero_gw,wf_qp(spad+1:spad+nsize,b1gw:b2gw),nsize)
#endif
    end do
   end if
   !
   ! =======================================
   ! === Update the input wave functions ===
   ! =======================================
   !
   select case (MPI_enreg%gwpara)

   case (0,1)
    ! == Each node has the full set ==
    wf(:,b1gw:b2gw,ik,is)=wf_qp(:,b1gw:b2gw)

   case (2)
    ! == Bands are spreaded across the nodes ==
    ! * Sum up all the partial QP amplitudes.
    ! * Keep the band in memory only if you are the right processor.
    call xsum_mpi(wf_qp(:,b1gw:b2gw),spaceComm,ierr)
    do ib=b1gw,b2gw
     if (rank==MPI_enreg%proc_distrb(ik,ib,is)) wf(:,ib,ik,is)=wf_qp(:,ib)
    end do

   case DEFAULT
    write(msg,'(a,i3,a)')' gwpara = ',MPI_enreg%gwpara,' not allowed '
    MSG_BUG(msg)
   end select

  end do !ik
 end do !is

 deallocate(umat_k)
 deallocate(wf_ks,wf_qp)

 DBG_EXIT("COLL")

end subroutine calc_wf_qp
!!***

!!****f* m_wfs/calc_wf_qp_Wfval
!! NAME
!! calc_wf_qp_Wfval
!!
!! FUNCTION
!!  Calculate QP amplitudes in real or reciprocal space starting from the 
!!  KS wavefunctions and the corresponding expansion coefficients,
!!  Consider the case of two separated sets of wavefunctions: Wf and Wfval
!!
!! INPUTS
!!  b1gw, b2gw = Min and max band index over k-point and spin for GW corrections.
!!  nkibz=number of k-points.
!!  nsize= number of points in real space or number of G vectors.
!!  nsppol=number of spin.
!!  nbnds=number of bands in the present GW calculation.
!!  my_minb, my_maxb = Indeces of the bands treated by this processor.
!!  m_lda_to_qp(nbnds,nbnds,nkibz,nsppol)= expansion of the QP amplitudes in terms of KS wavefunctions.
!!
!! OUTPUT
!!  wf(nsize,my_minb:my_maxb,nkibz,nsppol)= Updated QP amplitudes for this processor.
!!
!! PARENTS
!!      screening
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE
!!

subroutine calc_wf_qp_Wfval(MPI_enreg,nkibz,nbnds,nsize,nsppol,nspinor,&
& m_lda_to_qp,my_minb,my_maxb,b1gw,b2gw,wf,nbvw,wfval)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_14_hidewrite
 use interfaces_51_manage_mpi
 use interfaces_linalg
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nbnds,nkibz,nsize,nsppol,nspinor,my_minb,my_maxb,b1gw,b2gw,nbvw
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 complex(dpc),intent(in) :: m_lda_to_qp(nbnds,nbnds,nkibz,nsppol)
 complex(gwpc),intent(inout) :: wf(nsize,my_minb:my_maxb,nkibz,nsppol)
 complex(gwpc),intent(inout) :: wfval(nsize,nbvw,nkibz,nsppol)

!Local variables-------------------------------
!scalars
 integer :: ib,ik,is,ierr,lowerb,upperb,rangeb,b1gw_c
 integer :: spaceComm,sizegw,rank,shift,indx_kibz,ilmn,nlmn
 character(len=500) :: msg
!arrays
 complex(gwpc),allocatable :: umat_k(:,:)
 complex(gwpc),allocatable :: wf_qp(:,:)
 complex(gwpc),allocatable :: wf_qp_valval(:,:),wf_qp_val(:,:),wf_qp_valcon(:,:)
!************************************************************************

 DBG_ENTER("COLL")

 call xcomm_init(MPI_enreg,spaceComm)
 call xme_init  (MPI_enreg,rank)
 !
 ! === Determine the range of bands this processor has to treat ===
 b1gw_c=MAX(b1gw,nbvw+1) ! Avoid double counting of the valence bands
 lowerb=0  ! NO overlap between [b1gw,b2gw] and [my_minb,my_maxb]
 if (b1gw_c<=my_maxb) then
  lowerb=MAX(b1gw_c,my_minb)
 end if
 upperb=0  ! NO overlap between [b1gw,b2gw] and [my_minb,my_maxb]
 if (b2gw>=my_minb) upperb=MIN(b2gw,my_maxb)
 rangeb=0
 if (lowerb/=0.and.upperb/=0) rangeb=upperb-lowerb+1
 sizegw=b2gw-b1gw_c+1
 
 if (rangeb>0) then
  write(msg,'(2a,i3,3a,i3,1x,i3,a,3i3,2a,3(i3,1x))')ch10,&
&  ' proc ',rank,' will update its wavefunctions ',ch10,&
&  ' my_bands indeces: ',my_minb,my_maxb,' gwrange: ',b1gw_c,b2gw,sizegw,ch10,&
&  ' lowerb, upperb, rangeb: ',lowerb,upperb,rangeb
  call wrtout(std_out,msg,'PERS') 
 end if
 
 allocate(wf_qp_valval(nsize,nbvw))
 allocate(wf_qp_val(nsize,nbvw))  

 if (sizegw>0) then
  allocate(wf_qp_valcon(nsize,b1gw_c:b2gw))  
  allocate(wf_qp(nsize,b1gw_c:b2gw))  
 end if
 !
 ! === Calculate : $\Psi^{QP}_{r,b} = \sum_n \Psi^{KS}_{r,n} M_{n,b}$ ===
 do is=1,nsppol
  do ik=1,nkibz
   !
   ! I) Treat the valence bands
   !
   wf_qp_valval(:,:)=czero_gw ; wf_qp_val(:,:)=czero_gw 
   allocate(umat_k(nbvw,nbvw))
   umat_k(:,:)=m_lda_to_qp(1:nbvw,1:nbvw,ik,is)
 
#if defined HAVE_GW_DPC
   call ZGEMM('N','N',nsize,nbvw,nbvw,cone_gw,wfval(:,1:nbvw,ik,is),nsize,&
&   umat_k,nbvw,czero_gw,wf_qp_valval(:,1:nbvw),nsize)
#else
   call CGEMM('N','N',nsize,nbvw,nbvw,cone_gw,wfval(:,1:nbvw,ik,is),nsize,&
&   umat_k,nbvw,czero_gw,wf_qp_valval(:,1:nbvw),nsize)
#endif
   deallocate(umat_k)
  
   if (rangeb>0) then
    allocate(umat_k(lowerb:upperb,1:nbvw))
    umat_k(:,:)=m_lda_to_qp(lowerb:upperb,1:nbvw,ik,is)
#if defined HAVE_GW_DPC
    call ZGEMM('N','N',nsize,nbvw,rangeb,cone_gw,wf(:,lowerb:upperb,ik,is),nsize,&
&    umat_k,rangeb,czero_gw,wf_qp_val(:,1:nbvw),nsize)
#else
    call CGEMM('N','N',nsize,nbvw,rangeb,cone_gw,wf(:,lowerb:upperb,ik,is),nsize,&
&    umat_k,rangeb,czero_gw,wf_qp_val(:,1:nbvw),nsize)
#endif
    deallocate(umat_k)
   end if

   if (MPI_enreg%gwpara==2) then 
    ! Bands are spreaded among processors:
    ! * Sum up all the partial QP amplitudes.
    ! * Keep the band in memory only if you are the right processor.
    call xsum_mpi(wf_qp_val(:,1:nbvw),spaceComm,ierr)
    wf_qp_valval(:,1:nbvw)=wf_qp_valval(:,1:nbvw) + wf_qp_val(:,1:nbvw)
   else
    ! Each node has the full set
    wf_qp_valval(:,1:nbvw)=wf_qp_valval(:,1:nbvw) + wf_qp_val(:,1:nbvw)
   end if
   !
   ! II) Treat the NON-valence bands
   !
   if (sizegw>0) then
    wf_qp_valcon(:,:)=czero_gw
    wf_qp(:,:)=czero_gw
    allocate(umat_k(1:nbvw,b1gw_c:b2gw))
    umat_k(:,:)=m_lda_to_qp(1:nbvw,b1gw_c:b2gw,ik,is)
 
#if defined HAVE_GW_DPC
    call ZGEMM('N','N',nsize,sizegw,nbvw,cone_gw,wfval(:,1:nbvw,ik,is),nsize,&
&    umat_k,nbvw,czero_gw,wf_qp_valcon(:,b1gw_c:b2gw),nsize)
#else
    call CGEMM('N','N',nsize,sizegw,nbvw,cone_gw,wfval(:,1:nbvw,ik,is),nsize,&
&    umat_k,nbvw,czero_gw,wf_qp_valcon(:,b1gw_c:b2gw),nsize)
#endif
    deallocate(umat_k)
 
    if (rangeb>0) then
     allocate(umat_k(lowerb:upperb,b1gw_c:b2gw))
     umat_k(:,:)=m_lda_to_qp(lowerb:upperb,b1gw_c:b2gw,ik,is)
#if defined HAVE_GW_DPC
     call ZGEMM('N','N',nsize,sizegw,rangeb,cone_gw,wf(:,lowerb:upperb,ik,is),nsize,&
&     umat_k,rangeb,czero_gw,wf_qp(:,b1gw_c:b2gw),nsize)
#else
     call CGEMM('N','N',nsize,sizegw,rangeb,cone_gw,wf(:,lowerb:upperb,ik,is),nsize,&
&     umat_k,rangeb,czero_gw,wf_qp(:,b1gw_c:b2gw),nsize)
#endif
     deallocate(umat_k)
    end if
    !
    ! === Update the input wave functions ===
    if (MPI_enreg%gwpara==2) then 
     ! Bands are spreaded among processors:
     ! * Sum up all the partial QP amplitudes.
     ! * Keep the band in memory only if you are the right processor.
     call xsum_mpi(wf_qp(:,b1gw_c:b2gw),spaceComm,ierr)
     do ib=b1gw_c,b2gw
      if (rank==MPI_enreg%proc_distrb(ik,ib,is)) wf(:,ib,ik,is)=wf_qp(:,ib)+wf_qp_valcon(:,ib)
     end do
    else
     ! Each node has the full set
     wf(:,b1gw_c:b2gw,ik,is)=wf_qp_valcon(:,b1gw_c:b2gw)+wf_qp(:,b1gw_c:b2gw)
    end if
 
   endif !sizegw>0
 
   wfval(:,:,ik,is)=wf_qp_valval(:,:)
   wf   (:,my_minb:b1gw_c-1,ik,is)=wf_qp_valval(:,my_minb:b1gw_c-1)
 
  end do !ik
 end do !is

 call leave_test(mpi_enreg)

 if (allocated(wf_qp       )) deallocate(wf_qp       )
 if (allocated(wf_qp_valval)) deallocate(wf_qp_valval)
 if (allocated(wf_qp_val   )) deallocate(wf_qp_val   )
 if (allocated(wf_qp_valcon)) deallocate(wf_qp_valcon)

 DBG_EXIT("COLL")

end subroutine calc_wf_qp_Wfval
!!***

!!****f* m_wfs/rotate_wfg
!! NAME
!! rotate_wfg
!!
!! FUNCTION
!! Rotate the Fourier components of a wave function in reciprocal space to obtain
!! the wave function at a symmetric k-point (assuming a nondegenerate state)
!!
!! INPUTS
!! Wf<wavefunctions_information>=datatype gathering information of wavefunctions in the IBZ
!! ik_bz=the BZ k-point asked for
!! iband=the required band index
!! isppol=the spin polarization
!! grottbm1(Wf%npwwfn,Kmesh%timreversal,Kmsh%nsym)=Index of (IS)^{-1}G 
!! phmgt(Wf%npwwfn,Kmesh%nsym)=phase e^{-iG.t}
!!
!! OUTPUT
!! wfg_rot(Wf%npwwfn)=the Fourier coefficients of the wave function at point ik_bz
!! Notes that possible umklapp G0 vectors are not yet supported
!!
!! PARENTS
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine rotate_wfg(Wfs,Kmesh,iband,ik_bz,isppol,grottbm1,phmGt,wfg_rot)

 use defs_basis
 use m_bz_mesh, only : get_BZ_item

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_bz,iband,isppol
 type(wavefunctions_information),intent(in) :: Wfs
 type(bz_mesh_type),intent(in) :: Kmesh
!arrays
 integer,target,intent(in) :: grottbm1(:,:,:)
 complex(gwpc),intent(in) :: phmGt(:,:)
 complex(gwpc),intent(out) :: wfg_rot(Wfs%npwwfn)

!Local variables ------------------------------
!scalars
 integer :: isym,itim,ik_ibz,ig
 real(dp) :: kbz(3)
 complex(gwpc) :: phnons_k 
!arrays
 integer,pointer :: Sm1G(:)
 complex(gwpc),pointer :: wfg_irr(:)
!************************************************************************

 if (Wfs%npwwfn/=SIZE(grottbm1,DIM=1))    STOP ' rotate_wfg : Wf%npwwfn /= grottbm1 DIM 1 '
 if (Kmesh%timrev/=SIZE(grottbm1,DIM=2)) STOP ' rotate_wfg : timrev /= grottbm1 DIM 2 ' 
 if (Kmesh%nsym/=SIZE(grottbm1,DIM=3))   STOP ' rotate_wfg : nsym /= grottbm1 DIM 3 ' 
 if (Wfs%npwwfn/=SIZE(phmGt,DIM=1))      STOP ' rotate_wfg : Wf%npwwfn /= phmGt DIM 1 '
 if (Kmesh%nsym/=SIZE(phmGt,DIM=2))     STOP ' rotate_wfg : nsym /= phmGt DIM 2 ' 
 
 ! ****WARNING WARNING ****
 !FIXME Umklapp not yet implemented I have to store the umklapp somewhere
 ! but it requires boring modification of the code
 MSG_BUG("Still under development")

 call get_BZ_item(Kmesh,ik_bz,kbz,ik_ibz,isym,itim,phnons_k)
 write(*,*) " Rotate_wfg WARNING Umklapp not implemented yet "
 !
 ! === Rotate irreducible wave function ===
 ! $u_{Sk}(G) = e^{-i(Sk+G).\tau} u_k(S^{-1}G)$
 ! $u_{-k}(G) = u_k(-G)^*$
 ! $u_{k+G0}(G) = u_k(G+G0)$   
 !
 Sm1G => grottbm1(:,itim,isym)
 wfg_irr => Wfs%wfg(:,iband,ik_ibz,isppol)
 do ig=1,Wfs%npwwfn
  wfg_rot(ig)=wfg_irr(Sm1G(ig))*phmGt(ig,isym)
 end do
 wfg_rot=wfg_rot*phnons_k
 if (itim==2) wfg_rot=CONJG(wfg_rot)

end subroutine rotate_wfg
!!***

!!****f* m_wfs/rotate_wfr
!! NAME
!! rotate_wfr
!! 
!! FUNCTION
!! Rotate the periodic part of a wave function in real space to obtain
!! the lattice-period part of the Bloch function at a symmetric k-point (assuming a nondegenerate state)
!!
!! INPUTS
!! Wfs<wavefunctions_information>=datatype gathering information of wavefunctions in the IBZ
!! ik_bz=the BZ k-point asked for
!! iband=the required band index
!! isppol=the spin polarization
!! irottb(Wf%nfft,Kmesh%timreversal,Kmsh%nsym)=Index of (IS)^{-1}G 
!! phmgt(Wf%npwwfn,Kmesh%nsym)=phase e^{-iG.t}
!!
!! OUTPUT
!! wfr_rot(Wf%npwwfn)=the Fourier coefficients of the wave function at point ik_bz
!! Notes that possible umklapp G0 vectors are not yet supported
!!
!! PARENTS
!!
!! CHILDREN
!!      get_bz_item,get_wfr
!!
!! SOURCE

subroutine rotate_wfr(Wfs,Kmesh,iband,ik_bz,isppol,irottb,MPI_enreg,ur_rot)

 use defs_basis
 use m_bz_mesh, only : get_BZ_item

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_bz,iband,isppol
 type(Wavefunctions_information),intent(inout) :: Wfs
 type(BZ_mesh_type),intent(in) :: Kmesh
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 integer,intent(in),target :: irottb(Wfs%nfft,Kmesh%nsym)
 complex(gwpc),intent(out) :: ur_rot(Wfs%nfft)

!Local variables ------------------------------
!scalars
 integer :: isym,itim,ik_ibz,ir
 real(dp) :: kbz(3)
 complex(gwpc) :: ph_mkbzt
!arrays
 integer,pointer :: Rm1rt(:)
 complex(gwpc),allocatable :: ur_irr(:)
!************************************************************************

 MSG_BUG("Still under development")
 
 call get_BZ_item(Kmesh,ik_bz,kbz,ik_ibz,isym,itim,ph_mkbzt)
 !
 ! === Rotate irreducible wave function ===
 ! $u_{Sk}(r) = e^{-i(Sk.\tau} u_k(R^{-1}(r-\tau))$
 ! $u_{-k}(r) = u_k(r)^*$
 !
 allocate(ur_irr(Wfs%nfft))
 call get_wfr(Wfs,MPI_enreg,iband,ik_ibz,isppol,ur_irr)
 Rm1rt => irottb(:,isym)

 do ir=1,Wfs%nfft
  ur_rot(ir)=ur_irr(Rm1rt(ir)) 
 end do
 ur_rot(:)=ur_rot(:)*ph_mkbzt
 if (itim==2) ur_rot=CONJG(ur_rot)

 deallocate(ur_irr)

end subroutine rotate_wfr

END MODULE m_wfs
!!***
