!{\src2tex{textfont=tt}}
!!****f* ABINIT/make_epsm1_driver
!! NAME
!! make_epsm1_driver
!!
!! FUNCTION
!!  Driver routine to calculate the symmetrical inverse dielectric matrix starting
!!  from the irreducible polarizability. The routine considers a single q-point, and 
!!  performs the following tasks:
!!
!!  1) Calculate $\tilde\epsilon^{-1}$ using different approximations:
!!      * RPA
!!      * ALDA within TDDFT
!!
!!  2) Use a special treatment of non-Analytic behavior of heads and wings in reciprocal space
!!     calculating these quantities for different small q-directions specified by the user
!!     (Not yet operative)
!!
!!  3) Output the electron energy loss function and the macroscopic dielectric function with and 
!!     without nonlocal field effects (only if the frequencies dependency along the real axis has 
!!     been calculated
!!
!! COPYRIGHT
!!  Copyright (C) 2008-2009 ABINIT group (MG)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  iqibz=index of the q-point in the array Vcp%qibz where epsilon^-1 has to be calculated
!!  npwe=Number of G-vectors in chi0.
!!  nI,nJ=Number of rows/columns in chi0_ij (1,1 in collinear case)
!!  nomega=Number of frequencies.
!!  approx_type=Integer flag defining the type of approximation
!!   == 0 for RPA   ==
!!   == 1 for TDDFT ==
!!  option_test=Only for TDDFT:
!!   == 0 for TESTPARTICLE ==
!!   == 1 for TESTELECTRON ==
!!  Vcp<Coulombian_type>=Structure gathering data on the Coulombian interaction
!!   %nqibz=Number of q-points.
!!   %qibz(3,nqibz)=q-points in the IBZ.  
!!  FIXME treatment of kxcg has to be rewritten.
!!  dim_kxcg=Integer defining the dimension of the kernel in reciprocal space
!!  kxcg(npwe,npwe*dim_kxcg)=TDDFT kernel, required only if approx_type==1
!!  MPI_enreg=MPI-parallelisation information
!!
!! OUTPUT
!!  Different files are written according to the type of calculation
!!  See also side effects
!!
!! SIDE EFFECTS
!!  chi0(npwe*nI,npwe*nJ,nomega): in input the irreducible polarizability, in output 
!!   the symmetrized inverse dielectric matrix.
!!
!! NOTES
!!
!! PARENTS
!!      m_screening,screening
!!
!! CHILDREN
!!      hermitianize,init_spectra,matcginv,metric,print_arr,split_work2,timab
!!      wrtout,xcomm_init,xcomm_rank,xcomm_size,xsum_mpi
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"


subroutine make_epsm1_driver(iqibz,dim_wing,npwe,nI,nJ,nomega,omega,&
& approx_type,option_test,Vcp,dim_kxcg,kxcg,MPI_enreg,lwing,uwing,chi0,Spectra)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_errors

 use m_gwdefs,        only : GW_TOLQ0, czero_gw
 use m_numeric_tools, only : print_arr,hermitianize
 use m_io_tools,      only : get_unit
 use m_geometry,      only : normv
 use m_spectra,       only : init_spectra, destroy_spectra

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_12_hide_mpi
 use interfaces_14_hidewrite
 use interfaces_18_timing
 use interfaces_32_util
 use interfaces_42_geometry
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iqibz,nI,nJ,npwe,nomega,dim_wing,approx_type,option_test,dim_kxcg
 type(Coulombian_type),intent(in) :: Vcp
 type(MPI_type),intent(in) :: MPI_enreg
 type(Spectra_type),intent(out) :: Spectra
!arrays

 complex(gwpc),intent(in) :: kxcg(npwe,npwe*dim_kxcg) !FIXME this has to be rewritten LDA is (npwe,1)
 complex(dpc),intent(in) :: omega(nomega)
 complex(dpc),intent(inout) :: lwing(npwe*nI,nomega,dim_wing)
 complex(dpc),intent(inout) :: uwing(npwe*nJ,nomega,dim_wing)
 complex(gwpc),intent(inout) :: chi0(npwe*nI,npwe*nJ,nomega)

!Local variables-------------------------------
!scalars
 integer :: i1,i2,ig1,ig2,io,istat,ierr,irank,master,iqlwl,my_nqlwl
 integer :: nor,rank,skxc,unt,nprocs,spaceComm,use_MPI
 real(dp) :: ucvol
 logical :: is_qeq0
 character(len=500) :: msg

!arrays
 integer :: omega_distrb(nomega)
 integer,allocatable :: istart(:),istop(:)
 real(dp) :: tsec(2)
 real(dp) :: gmet(3,3),gprimd(3,3),rmet(3,3)
 real(dp),allocatable :: eelf(:,:) 
 complex(gwpc),allocatable :: chitmp(:,:) 
 complex(dpc),allocatable :: epsm_lf(:,:),epsm_nlf(:,:)
 complex(gwpc),pointer :: vc_sqrt(:)
 complex(gwpc),allocatable :: chi0_save(:,:)

! *************************************************************************

 DBG_ENTER("COLL")

 if (nI/=1.or.nJ/=1) then 
  MSG_ERROR("nI or nJ=/1 not yet implemented")
 end if

 call timab(309,1,tsec) ! chi/eps

 call xcomm_init(MPI_enreg,spaceComm) 
 call xcomm_size(spaceComm,nprocs,ierr)
 call xcomm_rank(spaceComm,rank,ierr)
 master=0

 ! TODO Use my own data type to pass the communicator, get rid of MPI_enreg
 !call xcomm_size(MPI_Comm%space,nprocs,ierr)
 !call xcomm_rank(MPI_Comm%space,rank,ierr)
 !master = MPIComm%master

 call metric(gmet,gprimd,-1,rmet,Vcp%rprimd,ucvol)

 is_qeq0 = (normv(Vcp%qibz(:,iqibz),gmet,'G')<GW_TOLQ0)

 use_MPI = 0  ! Parallelism is not yet used 
 !if (nprocs>=nomega) use_MPI = 1

 if (use_MPI==1) then ! * Initialize distribution table for frequencies.
  allocate(istart(nprocs),istop(nprocs))
  call split_work2(nomega,nprocs,istart,istop,1)
  omega_distrb=-999
  do irank=0,nprocs-1
   i1 = istart(irank+1)
   i2 = istop (irank+1)
   if (i1<=i2) omega_distrb(i1:i2) = irank
  end do
  deallocate(istart,istop)
 end if

 ! * Initialize container for spectral results
 do nor=1,nomega
  if (ABS(AIMAG(omega(nor)))>1.e-3) EXIT
 end do
 nor = nor -1 ; if (nor==0) nor = 1 ! only imag !?

 !if (is_qeq0.and.dim_wing>1) then 
 if (dim_wing>=1) then 
  write(*,*)' Analyzing long wavelength limit for several q'
  call init_spectra(Spectra,nor,REAL(omega(1:nor)),Vcp%nqlwl,Vcp%qlwl)
  my_nqlwl = dim_wing
  ABI_CHECK(dim_wing==SIZE(Vcp%vcqlwl_sqrt,DIM=2),"WRONG DIMS")
 else
  call init_spectra(Spectra,nor,REAL(omega(1:nor)),1,Vcp%qibz(:,iqibz))
  my_nqlwl = 1
 end if


!FB: all processors have to perform this operation
!    in order to have the epsm1 matrix when performing
!    a sigma calculation starting with the file _SUS
!if (rank==master) then ! presently only master has chi0 in screening

 ! Temporary arrays to store spectra.
 allocate(epsm_lf(nomega,my_nqlwl),epsm_nlf(nomega,my_nqlwl),eelf(nomega,my_nqlwl)) 
 epsm_lf =czero
 epsm_nlf=czero
 eelf = zero

 if (my_nqlwl>1) allocate(chi0_save(npwe*nI,npwe*nJ))

 select case (approx_type)
  case (0)
   ! * RPA: \tepsilon=(\delta-Vc^{1/2}*chi0Vc^{1/2}).
   ! * vc_sqrt contains vc^{1/2}(q,G), complex-valued to allow for a possible cutoff
   do io=1,nomega
    if (my_nqlwl>1)  chi0_save = chi0(:,:,io)

    ! * Loop over small q"s (if any) to treat the nonanalytical behavior.
    do iqlwl=my_nqlwl,1,-1

     if (my_nqlwl>1) then 
      chi0(:,:,io) = chi0_save         ! restore pristine polarizability
      chi0(:,1,io) = lwing(:,io,iqlwl) ! change the wings
      chi0(1,:,io) = uwing(:,io,iqlwl)
     end if

     if (iqibz==1) then 
      vc_sqrt => Vcp%vcqlwl_sqrt(:,iqlwl)  ! Use Coulomb term for q-->0
     else 
      vc_sqrt => Vcp%vc_sqrt(:,iqibz)  
     end if

     do ig2=1,npwe
!     (XG 090513) The following line has a problem with g95 compiler installed on max
!     chi0(:,ig2,io)=-vc_sqrt(:)*chi0(:,ig2,io)*vc_sqrt(ig2)
      do ig1=1,npwe*nI
       chi0(ig1,ig2,io)=-vc_sqrt(ig1)*chi0(ig1,ig2,io)*vc_sqrt(ig2)
      enddo
      chi0(ig2,ig2,io)=one+chi0(ig2,ig2,io) 
     end do

     epsm_nlf(io,iqlwl)=chi0(1,1,io) ! * chi0(io), now contains \tepsilon(io).
     write(msg,'(a,i4,a,2f9.4,a)')' Symmetrical epsilon(G,G'') at the ',io,' th omega',omega(io)*Ha_eV,' [eV]'
     call wrtout(std_out,msg,'COLL')
     call print_arr(chi0(:,:,io))
     !
     ! === Invert tepsilon and calculate macroscopic dielectric constant ===
     ! * epsm_lf(w)=1/epsm1(G=0,Gp=0,w). 
     ! * Since G=Gp=0 there is no difference btw symmetrical and not symmetrical 

     call matcginv(chi0(:,:,io),npwe,npwe)

     epsm_lf(io,iqlwl) = one/chi0(1,1,io)
     eelf   (io,iqlwl) = -AIMAG(chi0(1,1,io))
     write(msg,'(a,i4,a,2f9.4,a)')' Symmetrical epsilon^-1(G,G'') at the ',io,' th omega',omega(io)*Ha_eV,' [eV]'
     call wrtout(std_out,msg,'COLL')
     call print_arr(chi0(:,:,io))

     ! Save wings of e^-1 overwriting input values.
     if (dim_wing>0) then
      lwing(:,io,iqlwl) = chi0(:,1,io) 
      uwing(:,io,iqlwl) = chi0(1,:,io) 
     end if
end do !iqlwl 
 
   end do ! nomega

  case (1)
   ! === Vertex correction from Adiabatic TDDFT ===
   MSG_ERROR("recheck TDDFT code and parallel")
   skxc=assert_eq(SIZE(kxcg,1),SIZE(kxcg,2),'kxcg not square',__FILE__,__LINE__) 
   if (skxc/=npwe) STOP 'wrong size in kxcg'

   allocate(chitmp(npwe,npwe),STAT=istat) 
   if (istat/=0) then 
    write(msg,'(a,f8.2,a)')"out-of-memory, requiring ",npwe**2*gwpc*b2Mb," Mb"
    MSG_ERROR(msg)
   end if
   do io=1,nomega
    ! * Calculate chi0*fxc.
    chitmp(:,:)=MATMUL(chi0(:,:,io),kxcg(:,:))
    ! * Calculate (1-chi0*Vc-chi0*Kxc) and put it in chitmp.
    do ig1=1,npwe
     do ig2=1,npwe
      chitmp(ig1,ig2)=-chitmp(ig1,ig2)-vc_sqrt(ig2)**2*chi0(ig1,ig2,io) 
     end do
     chitmp(ig1,ig1)=chitmp(ig1,ig1)+one
    end do
    ! === Invert (1-chi0*Vc-chi0*Kxc) and Multiply by chi0 ===
    call matcginv(chitmp,npwe,npwe)
    chitmp=MATMUL(chitmp,chi0(:,:,io))
    ! === Save result, now chi0 contains chi ===
    chi0(:,:,io)=chitmp(:,:)
    write(std_out,'(a,i2,a,i1,a)')' chi(q= ',iqibz,',omega= ',io,',G,G")'
    call print_arr(chi0(:,:,io),mode_paral='PERS')
   end do

   select case (option_test)
    case (0) 
     ! === Calculate symmetrized TESTPARTICLE epsilon^-1 ===
     write(msg,'(a)')' calculating TESTPARTICLE epsilon^-1(G,G") = 1 + Vc*chi'
     call wrtout(std_out,msg,'COLL')
     do io=1,nomega
      do ig1=1,npwe
       chi0(ig1,:,io)=(vc_sqrt(ig1)*vc_sqrt(:))*chi0(ig1,:,io)
       chi0(ig1,ig1,io)=one+chi0(ig1,ig1,io)
      end do 
     end do 
    case (1)
     ! === Calculate symmetrized TESTELECTRON epsilon^-1 ===
     write(msg,'(a)')' calculating TESTELECTRON epsilon^-1(G,G") = 1 + (Vc + fxc)*chi'
     call wrtout(std_out,msg,'COLL')
     do io=1,nomega
      chitmp=MATMUL(kxcg(:,:),chi0(:,:,io))
      ! === Perform hermitianization (why ?) ===
      call hermitianize(chitmp)
      do ig1=1,npwe
       chi0(ig1,:,io)=(vc_sqrt(ig1)*vc_sqrt(:))*chi0(ig1,:,io)+chitmp(ig1,:)
       chi0(ig1,ig1,io)=one+chi0(ig1,ig1,io)
      end do 
     end do
    case default
     write(msg,'(a,i3)')'Wrong value for option_test= ',option_test
     MSG_BUG(msg)
   end select

   deallocate(chitmp)
   !
   ! === chi0 now contains symmetrical epsm1 ===
   ! * Calculate macroscopic dielectric constant epsm_lf(w)=1/epsm1(G=0,Gp=0,w) ===
   epsm_lf(:,1)=one/chi0(1,1,:)
   do io=1,nomega
    !write (msg,'(a,i2,a,i1,a)')' Symmetrical epsilon^-1(q=',iqibz,',omega=',io,',G,G")'
    !call wrtout(std_out,msg,'COLL')
    call print_arr(chi0(:,:,io),mode_paral='PERS')
   end do

 case default
  write(msg,'(a,i3)')'Wrong value for approx_type= ',approx_type 
  MSG_BUG(msg)
 end select

!FB: See comment above
!end if !master

 if (use_MPI==1) then ! * Collect results on each node.
  do io=1,nomega
   if (omega_distrb(io)/=rank) then 
    chi0(:,:,io)  = czero_gw
    lwing(:,io,:) = zero
    uwing(:,io,:) = zero
    !epsm_lf(io,:) = czero 
    !epsm_nlf(io,:) = czero 
    !eelf(io,:) = zero 
   end if
   call xsum_mpi(chi0(:,:,io), spaceComm,ierr)
   call xsum_mpi(lwing(:,io,:),spaceComm,ierr)
   call xsum_mpi(uwing(:,io,:),spaceComm,ierr)
  end do
  call xsum_mpi(epsm_lf, spaceComm,ierr )
  call xsum_mpi(epsm_nlf,spaceComm,ierr)
  call xsum_mpi(eelf,    spaceComm,ierr)
  !call leave_test(MPI_enreg)
  !£call xbarrier(spaceComm)
 end if

 ! * Save results in Spectra%, mind the slicing.
 Spectra%emacro_nlf(:,:) = epsm_nlf(1:nor,:)
 Spectra%emacro_lf (:,:) = epsm_lf (1:nor,:)
 Spectra%eelf      (:,:) = eelf    (1:nor,:)

 deallocate(epsm_lf,epsm_nlf,eelf)
 if (allocated(chi0_save)) deallocate(chi0_save) 

 call timab(309,2,tsec) ! chi/eps

 DBG_EXIT("COLL")

end subroutine make_epsm1_driver
!!***

