!{\src2tex{textfont=tt}}
!!****f* ABINIT/mkphdos
!!
!! NAME
!! mkphdos
!!
!! FUNCTION
!! Function to calculate the phonon density of states as well as 
!! the contributions associated to the different types of atoms in the unit cell.
!! Two methods are implemented: gaussian method and linear interpolation based on 
!! tetrahedrons.
!!
!! COPYRIGHT
!! Copyright (C) 1999-2009 ABINIT group (MG,MJV)
!! This file is distributed under the terms of the
!! GNU General Public Licence, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! acell(3) =length scales by which rprim is to be multiplied
!! amu(ntypat)=mass of the atoms (atomic mass unit)
!! anaddb_dtset= (derived datatype) contains all the input variables
!! atmfrc(2,3,natom,3,natom,nrpt) = Interatomic Forces in real space
!! dielt(3,3)=dielectric tensor
!! dyewq0(3,3,natom)=Ewald part of the dynamical matrix, at q=0.
!! filnam=prefix for the output file containing the PHDOS, presently not used 
!! gmet(3,3)= metric tensor in reciprocal space.
!! gprim(3,3)=normalized coordinates in reciprocal space
!! indsym(4,nsym,natom)=label given by subroutine symatm, indicating atom
!!  label which gets rotated into given atom by given symmetry
!!  (first three elements are related primitive translation--
!!  see symatm where this is computed)
!! mpert =maximum number of ipert
!! msym = maximum number of symmetries
!! natom=number of atoms in the unit cell
!! nrpt=number of R points in the Big Box
!! nsym=number of symmetries
!! ntypat=number of atom types
!! rcan(3,natom)=atomic position in canonical coordinates
!! rmet(3,3)=metric tensor in real space.
!! rprim(3,3)=dimensionless primitive translations in real space
!! rpt(3,nprt)=canonical coordinates of the R points in the unit cell
!!  These coordinates are normalized (=> * acell(3)!!)
!! symrec(3,3,nsym)=symmetry operations
!! symrel(3,3,nsym)=symmetry operations
!! tcpui=initial cpu time
!! trans(3,natom)=atomic translations : xred = rcan + trans
!! twalli=initial wall clock time
!! typat(natom)=integer label of each type of atom (1,2,...)
!! ucvol=unit cell volume
!! wghatm(natom,natom,nrpt)=weights associated to a pair of atoms and to a R vector
!! xred(3,natom)= relative coords of atoms in unit cell (dimensionless)
!! zeff(3,3,natom)=effective charge on each atom, versus electric
!!  field and atomic displacement
!!
!! OUTPUT
!!
!! NOTES
!! 
!! On the use of the q-grids : 
!! Two different q-meshes are used in this subroutine. The first one is the coarse 
!! mesh where the interatomic forces have been calculated during the DFPT run. 
!! This q-grid is used to obtain an initial guess for the max and min frequency 
!! value of the phonon spectrum. These values are, indeed, required to dimension 
!! the array containing the PHDOS. The second (dense) grid is used to perform the 
!! PHDOS calculation. If the Fourier interpolation on the second dense q-grid 
!! generates a phonon frequency outside the initially calculated frequency mesh,
!! the mesh is enlarged and the calculation is restarted.
!!
!! PARENTS
!!      
!!
!! CHILDREN
!!      
!!
!! SOURCE
subroutine mkphdos(acell,amu,anaddb_dtset,atmfrc,dielt,dyewq0,filname,gmet,gprim,indsym,&
& mpert,msym,natom,nrpt,nsym,ntypat,phonon_dos,rcan,rmet,rprim,rpt,symrec,symrel,tcpui,  &
& trans,twalli,typat,ucvol,wghatm,xred,zeff)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_phdos

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
 use interfaces_18_timing
 use interfaces_32_util
 use interfaces_42_geometry
 use interfaces_56_recipspace
 use interfaces_62_occeig
 use interfaces_72_response
 use interfaces_77_ddb, except_this_one => mkphdos
!End of the abilint section

 implicit none

!Arguments -------------------------------
!scalars
 integer,intent(in) :: mpert,msym,natom,nrpt,nsym,ntypat
 real(dp),intent(in) :: tcpui,twalli,ucvol
 character(len=fnlen),intent(in) :: filname
 type(anaddb_dataset_type),intent(in) :: anaddb_dtset
 type(phonon_dos_type),intent(inout) :: phonon_dos
!arrays
 integer,intent(in) :: indsym(4,nsym,natom),symrec(3,3,nsym),symrel(3,3,nsym)
 integer,intent(in) :: typat(natom)
 real(dp),intent(in) :: acell(3),amu(ntypat),dielt(3,3),gmet(3,3),gprim(3,3)
 real(dp),intent(in) :: rcan(3,natom),rmet(3,3),rprim(3,3),rpt(3,nrpt)
 real(dp),intent(in) :: trans(3,natom),wghatm(natom,natom,nrpt),xred(3,natom)
 real(dp),intent(in) :: zeff(3,3,natom)
 real(dp),intent(inout) :: atmfrc(2,3,natom,3,natom,nrpt),dyewq0(3,3,natom)

!Local variables -------------------------
 character(len=50),parameter :: sub_name='mkphdos.F90'
!scalars
 integer :: facbrv,iat,idir,ii,imesh,imode,io,iq,istat,itype,mtetra,nkpt_fullbz
 integer :: nmesh,nqbz,nqpt_max,nqshft,ntetra_ibz,option,timrev
 real(dp) :: Ha_meV,bzvol,dum,gaussfactor,gaussprefactor
 real(dp) :: gaussval,low_bound,max_occ,norm,pnorm
 real(dp) :: qphnrm,tcpu,twall,upr_bound,vtetra,xx
 logical :: out_of_bounds
 character(len=500) :: msg
!arrays
 integer :: igqpt2(3),qptrlatt(3,3)
 integer,allocatable :: bz2ibz(:),ibz2bz(:),ngqpt(:,:),tetra_full(:,:,:)
 integer,allocatable :: tetra_mult(:),tetra_wrap(:,:,:)
 real(dp) :: d2cart(2,3,mpert,3,mpert),displ(2*3*natom*3*natom),eigval(3*natom)
 real(dp) :: eigvec(2,3,natom,3*natom),gprimd(3,3),kpq(3,1),phfrq(3*natom)
 real(dp) :: qlatt(3,3),qphon(3),rlatt(3,3),rprimd(3,3)
 real(dp),allocatable :: dtweightde(:,:),full_eigvec(:,:,:,:,:),full_phfrq(:,:)
 real(dp),allocatable :: kpt_fullbz(:,:)
 real(dp),allocatable :: qbz(:,:)
 real(dp),allocatable :: qibz(:,:),qshft(:,:),tmp_phfrq(:),tweight(:,:),wtq(:)
 real(dp),allocatable :: wtq_folded(:),wtqibz(:)

! *********************************************************************

#ifdef DEBUG_MODE 
 write(std_out,*)' mkphdos : enter '
#endif

 if (anaddb_dtset%prtdos/=1.and.anaddb_dtset%prtdos/=2) then 
  write(msg,'(6a,i5)')ch10,&
&  ' mkphdos : BUG -',ch10,&
&  ' The argument anaddb%prtdos should be 1 or 2 ',ch10,&
&  ' however prtdos = ',anaddb_dtset%prtdos 
  call wrtout(std_out,msg,'COLL') 
  call leave_new('COLL')
 end if 
 if (anaddb_dtset%dosdeltae<=zero) then 
  write(msg,'(6a,es16.8)')ch10,&
&  ' mkphdos : BUG -',ch10,&
&  ' The argument anaddb%dosdeltae should be positive ',ch10,&
&  ' however dosdeltae = ',anaddb_dtset%dosdeltae 
  call wrtout(std_out,msg,'COLL') 
  call leave_new('COLL')
 end if 
 if (anaddb_dtset%prtdos==1.and.anaddb_dtset%dossmear<=zero) then 
  write(msg,'(6a,es16.8)')ch10,&
&  ' mkphdos : BUG -',ch10,&
&  ' The argument anaddb%dossmear should be positive ',ch10,&
&  ' however dossmear = ',anaddb_dtset%dosdeltae 
  call wrtout(std_out,msg,'COLL') 
  call leave_new('COLL')
 end if 
!
!If timing is needed at some point ...
 call timein(tcpu,twall)
 write(msg,'(a,2(a,f11.4),a)')ch10,&
& ' mkphdos : begin at tcpu',tcpu-tcpui,' and twall',twall-twalli,' sec'
 call wrtout(std_out,msg,'COLL')

!initialize container type, but with minimal values
 call init_phondos(phonon_dos,ntypat,natom,anaddb_dtset%prtdos,&
& 1,1,1,smallest_real,greatest_real,anaddb_dtset%dosdeltae,anaddb_dtset%dossmear)

 call mkrdim(acell,rprim,rprimd)
 call matr3inv(rprimd,gprimd)
 bzvol=ABS ( gprimd(1,1)*(gprimd(2,2)*gprimd(3,3)-gprimd(3,2)*gprimd(2,3)) &
& -gprimd(2,1)*(gprimd(1,2)*gprimd(3,3)-gprimd(3,2)*gprimd(1,3)) &
& +gprimd(3,1)*(gprimd(1,2)*gprimd(2,3)-gprimd(2,2)*gprimd(1,3)))
!
!=== Parameters defining the gaussian approximant ===
 Ha_mev=Ha_eV*1000

 if (anaddb_dtset%prtdos==1) then 
  gaussprefactor=one/(phonon_dos%dossmear*sqrt(two_pi))
  gaussfactor=one/(sqrt2*phonon_dos%dossmear)
  write(msg,'(4a,f8.5,2a,f8.5)')ch10,&
&  ' mkphdos : calculating phonon DOS using gaussian method :',ch10,&
&  '  gaussian smearing [meV] = ',phonon_dos%dossmear*Ha_meV,ch10,&
&  '  frequency step    [meV] = ',phonon_dos%omega_step*Ha_meV
 else if (anaddb_dtset%prtdos==2) then 
  write(msg,'(2a)')ch10,&
&  ' mkphdos : calculating phonon DOS using tetrahedron method '
 end if 
 call wrtout(std_out,msg,'COLL')
!
!=== Initial lower and upper bound of the phonon spectrum ===
 low_bound=greatest_real 
 upr_bound=smallest_real
!
!Save memory during the generation of the q-mesh in the full BZ  
!Take into account the type of Bravais lattice
 facbrv=1
 if (anaddb_dtset%brav==2) facbrv=2
 if (anaddb_dtset%brav==3) facbrv=4

 nmesh=2 ; allocate(ngqpt(3,nmesh))
 do imesh=1,nmesh

  if (imesh==1) then 
!  
!  === Coarse q-mesh used during RF calculation ===
   ngqpt(:,imesh)=anaddb_dtset%ngqpt(1:3)
   nqshft=anaddb_dtset%nqshft 
   allocate(qshft(3,nqshft))
!  TODO this has to be fixed  there is a small inconsistency in the dimension of q1shft
   qshft(:,1:nqshft)=anaddb_dtset%q1shft(:,1:nqshft)
  else 
!  
!  Dense q-mesh used for the Fourier interpolation. 
   ngqpt(1:3,imesh)=anaddb_dtset%ng2qpt(1:3)
   nqshft=1 !always 1 
   allocate(qshft(3,nqshft))
   qshft(:,1)=anaddb_dtset%q2shft(:)  !small inconsistency in the dimension of q1shft
  end if 

  nqpt_max=(ngqpt(1,imesh)*ngqpt(2,imesh)*ngqpt(3,imesh)*nqshft)/facbrv
  allocate(qibz(3,nqpt_max),qbz(3,nqpt_max))

  qptrlatt(:,:)=0
  qptrlatt(1,1)=ngqpt(1,imesh)
  qptrlatt(2,2)=ngqpt(2,imesh)
  qptrlatt(3,3)=ngqpt(3,imesh)
  option=1 

! here I noticed a problem in the declaration of q1shft in the anaddb datatype 
! FIXME we write on unit 6 just to avoid problem with automatic tests
  call smpbz(anaddb_dtset%brav,6,qptrlatt,nqpt_max,nqbz,nqshft,option,qshft,qbz)
! 
! Reduce the number of such points by symmetrization.
  allocate(ibz2bz(nqbz),wtq(nqbz),wtq_folded(nqbz))
  wtq(:)=one/nqbz         ! Weights sum up to one
  timrev=1 ; option=1     ! TODO timrev should be input 

! this call will set phonon_dos%nqibz
  call symkpt(gmet,ibz2bz,qbz,nqbz,phonon_dos%nqibz,nsym,option,symrec,timrev,wtq,wtq_folded)
  write (*,*) 'phonon_dos%nqibz = ', phonon_dos%nqibz

  allocate(wtqibz(phonon_dos%nqibz))
  do iq=1,phonon_dos%nqibz
   wtqibz(iq)=wtq_folded(ibz2bz(iq))
   qibz(:,iq)=qbz(:,ibz2bz(iq))
  end do
  deallocate(wtq_folded,qshft)

  if (anaddb_dtset%prtdos==2.and.imesh==2) then
!  
!  Second mesh with tetrahedron method
!  convert kptrlatt to double and invert, qlatt here refer to the shortest qpt vectors
   rlatt(:,:)=qptrlatt(:,:)
   call matr3inv(rlatt,qlatt)

   allocate(qshft(3,nqshft))
   qshft(:,1)=anaddb_dtset%q2shft(:)  ! FIXME small inconsistency in the dimension of q1shft
   nkpt_fullbz=nqbz 
   allocate(bz2ibz(nkpt_fullbz))
   allocate(kpt_fullbz(3,nkpt_fullbz))
!  
!  Make full kpoint grid and get equivalence to irred kpoints.
!  This routines scales badly wrt nkpt_fullbz, should introduce checl on the norm.
   call get_full_kgrid(bz2ibz,qlatt,qibz,kpt_fullbz,qptrlatt,phonon_dos%nqibz,nkpt_fullbz,nqshft,nsym,qshft,symrel)
!  
!  Get tetrahedra, ie indexes of the full q-points at their summits
!  tetra_full(:,1,i) contains the irred qpt  number
!  tetra_full(:,2,i) contains the full  qpt number
!  tetra_wrap(:,:,i) contains a flag to wrap q-points outside the IBZ (+-1) to get the irreducible tetrahedra
!  the number of equivalent tetrahedra is counted in tetra_mult and the inequivalent few (ntetra < mtetra) are 
!  packed into the beginning of tetra_full
   mtetra=6*nqbz
   allocate(tetra_full(4,2,mtetra),tetra_wrap(3,4,mtetra),tetra_mult(mtetra))
   
   call get_tetra(bz2ibz,gprimd,qlatt,kpt_fullbz,mtetra,nqbz,ntetra_ibz,tetra_full,tetra_mult,tetra_wrap,vtetra)
   write(*,*)' Number of irreducible tetrahedrons = ',ntetra_ibz
   deallocate(bz2ibz)
!  
!  === Arrays Used to store the entire spectrum, Required to calculate tetra weights ===
   allocate(full_phfrq(3*natom,phonon_dos%nqibz),full_eigvec(2,3,natom,3*natom,phonon_dos%nqibz),stat=istat)
   if (istat/=0) call memerr(sub_name,'full_eigvec',18*natom**2*phonon_dos%nqibz,'dp')

  end if  ! end if anaddb_dtset%prtdos==2.and.imesh==2

  do 
!  
!  This infinite loop is used to be sure that the frequency mesh is large enough to contain 
!  the entire phonon spectrum. The mesh is enlarged if, during the Fourier interpolation,
!  a phonon frequency turns out to be outside the interval [omega_min:omega_max]
!  
   if (associated(phonon_dos%omega)) deallocate(phonon_dos%omega)
   if (associated(phonon_dos%phdos)) deallocate(phonon_dos%phdos)
   if (associated(phonon_dos%pjdos)) deallocate(phonon_dos%pjdos)

   out_of_bounds=.FALSE.
   phonon_dos%omega_min=low_bound ; if (ABS(phonon_dos%omega_min)<tol5) phonon_dos%omega_min=-tol5
   phonon_dos%omega_max=upr_bound 
   phonon_dos%nomega=NINT((phonon_dos%omega_max-phonon_dos%omega_min)/phonon_dos%omega_step)+1
!  ensure the integration will be ok
   phonon_dos%nomega=max(6,phonon_dos%nomega)

   if (imesh/=1) then 
    write(*,*)'nomega = ',phonon_dos%nomega,' omega_min [cm-1] =',phonon_dos%omega_min*Ha_cmm1,&
&    '                    omega_max [cm-1] =',phonon_dos%omega_max*Ha_cmm1
   end if 
   allocate(phonon_dos%omega(phonon_dos%nomega))
   do io=1,phonon_dos%nomega
    phonon_dos%omega(io)=phonon_dos%omega_min+phonon_dos%omega_step*(io-1)
   end do
   allocate(phonon_dos%phdos(phonon_dos%nomega),phonon_dos%pjdos(phonon_dos%nomega,3,natom))
   phonon_dos%phdos(:)=zero ; phonon_dos%pjdos(:,:,:)=zero
!  
!  === Sum over irreducible q-points ===
   do iq=1,phonon_dos%nqibz
    qphon(:)=qibz(:,iq) ; qphnrm=one
!   
!   Get d2cart using interatomic forces and the long-range coulomb interaction through Ewald summation
    call gtdyn9(acell,atmfrc,dielt,anaddb_dtset%dipdip,dyewq0,d2cart,gmet,gprim,&
&    mpert,natom,nrpt,qphnrm,qphon,rcan,rmet,rprim,rpt,trans,ucvol,wghatm,xred,zeff)
!   
!   Get eigenvectors and eigenvalues of the dynamical matrix, eigvec are normalized to one
    call phfrq3(amu,displ,d2cart,eigval,eigvec,indsym,mpert,msym,natom,nsym,ntypat,&
&    phfrq,qphnrm,qphon,rprimd,anaddb_dtset%symdynmat,symrel,typat,ucvol,xred)
    
    dum=MINVAL(phfrq) ; phonon_dos%omega_min=MIN(phonon_dos%omega_min,dum)
    dum=MAXVAL(phfrq) ; phonon_dos%omega_max=MAX(phonon_dos%omega_max,dum)
!   write (*,*) 'phonon_dos%omega_min,phonon_dos%omega_max,MINVAL(phfrq),MAXVAL(phfrq),low_bound,upr_bound'
!   write (*,*) phonon_dos%omega_min,phonon_dos%omega_max,MINVAL(phfrq),MAXVAL(phfrq),low_bound,upr_bound
    out_of_bounds = (phonon_dos%omega_min<low_bound .or. phonon_dos%omega_max>upr_bound) 

    if (imesh>1.and..not.out_of_bounds) then
     select case (anaddb_dtset%prtdos)

      case (1)
!      === Accumulate PHDOS and PJDOS using gaussian method ===
       do imode=1,3*natom 
        do io=1,phonon_dos%nomega
         xx=(phonon_dos%omega(io)-phfrq(imode))*gaussfactor
         gaussval=gaussprefactor*exp(-xx*xx)
         phonon_dos%phdos(io)=phonon_dos%phdos(io) + wtqibz(iq)*gaussval
         do iat=1,natom
          do idir=1,3
           pnorm=eigvec(1,idir,iat,imode)**2+eigvec(2,idir,iat,imode)**2
           phonon_dos%pjdos(io,idir,iat)=phonon_dos%pjdos(io,idir,iat)+ pnorm*wtqibz(iq)*gaussval
          end do
         end do
        end do 
       end do 

      case (2) 
!      === Tetrahedrons ===
!      * Save phonon frequencies and eigenvectors. 
!      Summation is done after the loops over the two meshes.
       full_phfrq(:,iq)=phfrq(:)
       full_eigvec(:,:,:,:,iq)=eigvec(:,:,:,:)
     end select
    end if !Second mesh and not out of boundaries

   end do !irred q-points

   if (out_of_bounds) then 
    upr_bound=phonon_dos%omega_max+ABS(phonon_dos%omega_max/ten)
    low_bound=phonon_dos%omega_min-ABS(phonon_dos%omega_min/ten)
    write(msg,'(6a)')ch10,&
&    ' mkphdos : COMMENT : ',ch10,&
&    ' at least one phonon frequency falls outside the frequency mesh chosen',ch10,&
&    ' restarting the calculation with a larger frequency mesh ' 
    if (imesh>1) call wrtout(std_out,msg,'COLL')
   else
    EXIT !infinite loop
   end if 
  end do !infinite loop

  deallocate(ibz2bz,qibz,qbz,wtq,wtqibz)
 end do !imesh
 deallocate(ngqpt)

 if (associated(phonon_dos%phdos_int)) deallocate (phonon_dos%phdos_int)
 if (associated(phonon_dos%pjdos_int)) deallocate (phonon_dos%pjdos_int)

 allocate(phonon_dos%phdos_int(phonon_dos%nomega)) 
 phonon_dos%phdos_int(:)=zero 

 if (anaddb_dtset%prtdos==2) then 
! === Integrate using tetrahedrons ===
! * All the data are contained in full_phfrq and full_eigvec. 
! * low_bound and upr_bound contain the entire spectrum calculated on the dense mesh. 
  allocate(tmp_phfrq(phonon_dos%nqibz)) 
  allocate(tweight(phonon_dos%nqibz,phonon_dos%nomega),dtweightde(phonon_dos%nqibz,phonon_dos%nomega))
  allocate(phonon_dos%pjdos_int(phonon_dos%nomega,3,natom)) 
  phonon_dos%phdos(:)=zero ; phonon_dos%pjdos(:,:,:)=zero ; phonon_dos%pjdos_int(:,:,:)=zero
  max_occ=one 

  do imode=1,3*natom 
   tmp_phfrq(:)=full_phfrq(imode,:)
!  
!  === Calculate general integration weights at each irred kpoint as in Blochl et al PRB 49 16223 ===
   call get_tetra_weight(tmp_phfrq,low_bound,upr_bound,max_occ,mtetra,phonon_dos%nomega,phonon_dos%nqibz,&
&   ntetra_ibz,bzvol,tetra_full,tetra_mult,tweight,dtweightde,vtetra)

   do io=1,phonon_dos%nomega
    do iq=1,phonon_dos%nqibz
     phonon_dos%phdos(io)=phonon_dos%phdos(io)+dtweightde(iq,io)
     phonon_dos%phdos_int(io)=phonon_dos%phdos_int(io)+tweight(iq,io)
     do iat=1,natom
      do idir=1,3
       pnorm=full_eigvec(1,idir,iat,imode,iq)**2 + full_eigvec(2,idir,iat,imode,iq)**2
       phonon_dos%pjdos(io,idir,iat)=phonon_dos%pjdos(io,idir,iat) + pnorm*dtweightde(iq,io)
       phonon_dos%pjdos_int(io,idir,iat)=phonon_dos%pjdos_int(io,idir,iat) + pnorm*tweight(iq,io)         
      end do
     end do
    end do
   end do

  end do 
  deallocate(tmp_phfrq)
  deallocate(tweight,dtweightde)
 end if 

!=== calculate IPDOS ===

 if (associated(phonon_dos%pjdos_xyz_typ)) deallocate (phonon_dos%pjdos_xyz_typ)
 if (associated(phonon_dos%pjdos_typ)) deallocate (phonon_dos%pjdos_typ)

 allocate(phonon_dos%pjdos_xyz_typ(phonon_dos%nomega,3,ntypat))
 phonon_dos%pjdos_xyz_typ(:,:,:)=zero
 allocate(phonon_dos%pjdos_typ(phonon_dos%nomega,ntypat))
 phonon_dos%pjdos_typ(:,:)=zero
 allocate(phonon_dos%pjdos_typ_int(phonon_dos%nomega,ntypat))
 phonon_dos%pjdos_typ_int(:,:)=zero

 do iat=1,natom 
  itype=typat(iat)
  do io=1,phonon_dos%nomega
   phonon_dos%pjdos_xyz_typ(io,:,itype)=phonon_dos%pjdos_xyz_typ(io,:,itype)+phonon_dos%pjdos(io,:,iat)
   phonon_dos%pjdos_typ(io,itype)=phonon_dos%pjdos_typ(io,itype)+sum(phonon_dos%pjdos(io,:,iat))
  end do
  if (anaddb_dtset%prtdos==2) then 
   do io=1,phonon_dos%nomega
    phonon_dos%pjdos_typ_int(io,itype)=phonon_dos%pjdos_typ_int(io,itype)+SUM(phonon_dos%pjdos_int(io,:,iat))
   end do
  end if 
 end do


!Evaluate IDOS using simple simpson integration
!TODO should avoid the simpson rule using derf.F90, just to be consistent
 if (anaddb_dtset%prtdos==1) then 
  call simpson_int(phonon_dos%nomega,phonon_dos%omega_step,phonon_dos%phdos,phonon_dos%phdos_int)
  do itype=1,ntypat
   call simpson_int(phonon_dos%nomega,phonon_dos%omega_step,phonon_dos%pjdos_typ(:,itype),phonon_dos%pjdos_typ_int(:,itype))
  end do
 end if 

 if (anaddb_dtset%prtdos==2) then
  deallocate(tetra_full,tetra_wrap,tetra_mult)
  deallocate(full_phfrq,full_eigvec)
 end if

end subroutine mkphdos
!!***
