!{\src2tex{textfont=tt}}
!!****f* ABINIT/rdpawps1xml
!! NAME
!! rdpawps1xml
!!
!! FUNCTION
!! Read the header of a PAW pseudopotential XML file generated by AtomPAW
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  filename= input file name (atomicdata XML)
!!  unit= input unit number
!!
!! OUTPUT
!!  pshead
!!    %atomic_charge= Total atomic charge
!!    %basis_size= Number of elements for the paw nl basis
!!    %core_charge= Core charge
!!    %core_meshsz= Dimension of radial mesh for core density
!!    %exctype_atompaw= AtomPaw s expression of exchange-correlation
!!    %lambda= Lambda in gaussian type g(r)
!!    %lmn_size= Number of elements for the paw basis
!!    %mesh_type= type of radial grid (regular or log)
!!    %occ(basis_size)= occupation for each basis function
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %pspxc_abinit= Abinit s code number for the exchange-correlation
!!    %rad_step= Step corresponding to radial mesh
!!    %rc_hat= radius for shape function
!!    %rc_sph= Default PAW sphere radius
!!    %shape_type= Shape function type
!!    %sigma= Sigma for gaussian type g(r)
!!    %stateid(basis_size)= "state id" for each basis function
!!    %title= Title for pseudopotential
!!    %vale_meshsz= Dimension of radial mesh for pseudo valence density
!!    %vloc_meshsz= Dimension of radial mesh for vloc=vhtnzc
!!    %wav_meshsz= Dimension of radial mesh for phi, tphi ...
!!  version= version number for AtomPAW2Abinit
!!
!! PARENTS
!!      atompaw2abinit
!!
!! CHILDREN
!!      rdfromline
!!
!! SOURCE

 subroutine rdpawps1xml(filename,pshead,funit,version)

 use defs_basis
 use defs_pawps

 implicit none

!Arguments ---------------------------------------------
 integer :: funit
 character*(fnlen) :: filename
 character*5 :: version
 type(pshead_type) :: pshead

!Local variables ---------------------------------------
 integer :: ib,ierr
 logical :: endfile,found,symbolfound,statesfound,gridfound,biggridfound,exchfound,shapefound

 real(dp) :: rc,rr1,rr2,testsin,testexp
 real(dp) :: ff(20)
 character*(strlen) :: line,readline,strg,strg2

! *************************************************************************

!Inits
 pshead%rc_sph=-1.d0
 pshead%rc_hat=-1.d0
 pshead%basis_size=0
 pshead%core_meshsz=0
 pshead%vale_meshsz=0
 pshead%vloc_meshsz=0
 pshead%mesh_type=1
 symbolfound=.false.
 statesfound=.false.
 gridfound=.false.
 biggridfound=.false.
 exchfound=.false.
 shapefound=.false.

!Open the atomicdata XML file for reading
 open(unit=funit,file=filename,form='formatted',status='old')

!Start a reading loop
 endfile=.false.
 do while (.not.endfile)
   read(funit,'(a)',err=10,end=10) readline
   line=adjustl(readline);goto 20
10 line="";endfile=.true.
20 continue

!--Read TITLE, ATOMIC CHARGE AND CORE CHARGE
   if (line(1:12)=='<atom symbol') then
    call rdfromline("atom symbol",line,strg,ierr)
    write(unit=pshead%title,fmt='(4a)') &
&    "Paw atomic data for element ",trim(ADJUSTL(strg)),&
&    " - Generated by AtomPAW + AtomPAW2Abinit v",version
    call rdfromline(" Z",line,strg,ierr)
    read(unit=strg,fmt=*) pshead%atomic_charge
    call rdfromline(" core",line,strg,ierr)
    read(unit=strg,fmt=*) pshead%core_charge
    symbolfound=.true.
   endif

!--Read BASIS SIZE, ORBITALS, RC AND OCCUPATIONS/STATE IDs
   if (line(1:16)=='<valence_states>') then
    do while (line(1:17)/='</valence_states>')
     read(funit,'(a)') readline;line=adjustl(readline)
     if (line(1:6)=='<state') then
      pshead%basis_size=pshead%basis_size+1
      if (pshead%basis_size>20) then
       write(6,'(a)') "Error in rdpawps1xml: basis size too large (>20) !"
       close(funit);stop
      endif
      call rdfromline(" l",line,strg,ierr)
      read(unit=strg,fmt=*) pshead%orbitals(pshead%basis_size)
      call rdfromline(" f",line,strg,ierr)
      read(unit=strg,fmt=*) pshead%occ(pshead%basis_size)
      call rdfromline(" id",line,strg,ierr)
      read(unit=strg,fmt=*) pshead%stateid(pshead%basis_size)
      call rdfromline(" rc",line,strg,ierr)
      read(unit=strg,fmt=*) rc
      if (rc>pshead%rc_sph) pshead%rc_sph=rc
     endif
    enddo
    statesfound=.true.
   endif

!--Read MESH_STEP AND NUMBER OF POINTS
   if (line(1:12)=='<radial_grid') then
    call rdfromline(" id",line,strg,ierr)
    if (trim(strg)=="lin1") then
     call rdfromline(" d",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%rad_step
     call rdfromline("iend=",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%core_meshsz
     pshead%core_meshsz=pshead%core_meshsz+1
     pshead%wav_meshsz=pshead%core_meshsz
     pshead%log_step=zero
     pshead%mesh_type=1
     gridfound=.true.
    else if (trim(strg)=="lin2") then
     call rdfromline("iend=",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%vloc_meshsz
     pshead%vloc_meshsz=pshead%vloc_meshsz+1
     pshead%vale_meshsz=pshead%vloc_meshsz
     biggridfound=.true.
    else if (trim(strg)=="log1") then
     call rdfromline(" a=",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%rad_step
     call rdfromline(" d=",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%log_step
     call rdfromline("iend=",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%core_meshsz
     pshead%core_meshsz=pshead%core_meshsz+1
     pshead%wav_meshsz=pshead%core_meshsz
     pshead%mesh_type=2
     gridfound=.true.
    else if (trim(strg)=="log2") then
     call rdfromline("iend=",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%vloc_meshsz
     pshead%vloc_meshsz=pshead%vloc_meshsz+1
     pshead%vale_meshsz=pshead%vloc_meshsz
     biggridfound=.true.
    endif
   endif

!--Read EXCHANGE-CORRELATION TYPE
   if (line(1:14)=='<xc_functional') then
    call rdfromline(" type",line,strg,ierr)
    call rdfromline(" name",line,strg2,ierr)
    write(unit=pshead%exctype_atompaw,fmt='(3a)') trim(ADJUSTL(strg)),"-",trim(ADJUSTL(strg2))
    if (trim(ADJUSTL(strg))=="LDA".and.trim(ADJUSTL(strg2))=="PW") then
     pshead%pspxc_abinit=7
    else if (trim(ADJUSTL(strg))=="GGA".and.trim(ADJUSTL(strg2))=="PBE") then
     pshead%pspxc_abinit=11
    else
     write(6,'(a)') "Error in rdpawps1xml: unknown XC type !"
     close(funit);stop
    endif
    exchfound=.true.
   endif

!--Read SHAPE TYPE
   if (line(1:15)=='<shape_function') then
    if (pshead%rc_sph<0.d0) then
     write(6,'(a)') "Error in rdpawps1xml: rc must be read before shape function !"
     close(funit);stop
    endif
    if (pshead%core_meshsz==0) then
     write(6,'(a)') "Error in rdpawps1xml: grid must be read before shape function !"
     close(funit);stop
    endif
    call rdfromline(" type",line,strg,ierr)
    if (trim(strg)=="gauss") then
     pshead%shape_type=1
     pshead%lambda=2
     call rdfromline(" rc",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%sigma
     pshead%rc_hat=pshead%rc_sph
    elseif (trim(strg)=="sinc") then
     pshead%shape_type=2
     pshead%lambda=0
     pshead%sigma=0.d0
     call rdfromline(" rc",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%rc_hat
    elseif (trim(strg)=="bessel") then
     pshead%shape_type=3
     pshead%lambda=0
     pshead%sigma=0.d0
     call rdfromline(" rc",line,strg2,ierr)
     read(unit=strg2,fmt=*) pshead%rc_hat
    else
     pshead%shape_type=-1
     pshead%lambda=0
     pshead%sigma=0.d0
    endif
    shapefound=.true.
   endif

!End of reading loop
 enddo

!Close the XML atomicdata file
 close(funit)

!Test flags: is anything OK ?
 found=symbolfound.and.statesfound.and.gridfound.and.exchfound.and.shapefound
 if (.not.symbolfound) &
&  write(6,'(a,i2,a)') "Error in rdpawps1xml: ATOM SYMBOL not found !"
 if (.not.statesfound) &
&  write(6,'(a,i2,a)') "Error in rdpawps1xml: VALENCE STATES not found !"
 if (.not.gridfound) &
&  write(6,'(a,i2,a)') "Error in rdpawps1xml: GRID not found !"
 if (.not.exchfound) &
&  write(6,'(a,i2,a)') "Error in rdpawps1xml: EXCHANGE/CORRELATION not found !"
 if (.not.shapefound) &
&  write(6,'(a,i2,a)') "Error in rdpawps1xml: SHAPE FUNCTION TYPE not found !"
 if (.not.found) stop

!Deduce mesh size(s)
 if (rc>0.d0) then
  if (pshead%mesh_type==1) then
   pshead%sph_meshsz=1+nint(pshead%rc_sph/pshead%rad_step)
  else
   pshead%sph_meshsz=1+nint(log(one+pshead%rc_sph/pshead%rad_step)/pshead%log_step)
  endif
 else
  write(6,'(a)') "Error in rdpawps1xml: unconsistent rc !"
 endif

!Tests of consistency
 if (pshead%core_meshsz<pshead%wav_meshsz) then
  write(6,'(a)') "Error in rdpawps1xml:"
  write(6,'(a)') "   Mesh size for tcore density (CORETAIL_POINTS)"
  write(6,'(a)') "   must be greater or equal than MESH_SIZE !"
  stop
 endif
!Radius of PAW spheres
 if (pshead%mesh_type==1) then
  if (pshead%rc_sph>pshead%rad_step*dble(pshead%wav_meshsz-1)+tol8) then
   write(6,'(a)') "Error in rdpawps1xml:"
   write(6,'(a)') "   Radius for PAW spheres (RC)"
   write(6,'(a)') "   must be less (or equal) than R(MESH_SIZE) !"
   stop
  endif
 else
  if (pshead%rc_sph>pshead%rad_step*(exp(pshead%log_step*dble(pshead%wav_meshsz-1))-1.d0)+tol8) then
   write(6,'(a)') "Error in rdpawps1xml:"
   write(6,'(a)') "   Radius for PAW spheres (RC)"
   write(6,'(a)') "   must be less (or equal) than R(MESH_SIZE) !"
   stop
  endif
 end if

!Here must compute lmn_size
 pshead%lmn_size=0
 do ib=1,pshead%basis_size
  pshead%lmn_size=pshead%lmn_size+2*pshead%orbitals(ib)+1
 enddo

 end subroutine
!!***
