!{\src2tex{textfont=tt}}
!!****f* ABINIT/rdpawps2xml
!! NAME
!! rdpawps2xml
!!
!! FUNCTION
!! Read the body of a PAW pseudopotential file generated by AtomPAW
!! (the header must have been read before)
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  filename= input file name (atomicdata XML)
!!  funit= input unit number
!!  pawarray
!!    %indlmn(6,lmn_size)= Gives l,m,n,lm,ln,s for i=lmn
!!  pawrad= radial grid definitions
!!  pshead
!     %atomic_charge= Total atomic charge
!!    %basis_size= Number of elements for the paw nl basis
!!    %core_charge= Core charge
!!    %core_meshsz= Dimension of radial mesh for core density
!!    %lmn_size= Number of elements for the paw basis
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %shape_type= Shape function type
!!    %sph_meshsz= Dimension of radial mesh corresponding to PAW spheres
!!    %stateid(basis_size)= "state id" for each basis function
!!    %vale_meshsz= Dimension of radial mesh for pseudo valence
!!    %vloc_meshsz= Dimension of radial mesh for vloc=htnzc
!!    %vlocopt= Option for Vloc
!!    %wav_meshsz= Dimension of radial mesh for phi, tphi ...
!!
!! OUTPUT
!!  pawarray
!!    %kij(lmn2_size)= Kinetic overlap operator
!!    %shapefunc(wav_meshsz)= shape function
!!  pawps
!!    %coreden4pr2(core_meshsz)= Core density multiplied by 4Pi.r2
!!    %tcoreden4pr2(core_meshsz)= Pseudized core density multiplied by 4Pi.r2
!!    %tvaleden4pr2(vale_meshsz)= Pseudized valence density multiplied by 4Pi.r2
!!    %phi(wav_meshsz,basis_size)= PAW atomic wavefunctions on the radial grid
!!    %tphi(wav_meshsz,basis_size)= PAW atomic pseudo-wavefunctions on the radial grid
!!    %tproj(prj_msz_max,basis_size)= PAW projectors on the radial grid
!!    %vbare(sph_meshsz)= bare local potential (part of VH(tnzc))
!!    %vhtnzc(core_meshsz)= Hartree potential of the ps-density of the nucleus + core electrons
!!
!! PARENTS
!!      atompaw2abinit
!!
!! CHILDREN
!!      deducer0,rdfromline
!!
!! SOURCE

 subroutine rdpawps2xml(filename,pawarray,pawps,pawrad,pshead,funit)

 use defs_basis
 use defs_pawps

 implicit none

!Arguments ---------------------------------------------
 integer :: funit
 character*(fnlen)   :: filename
 type(pawarray_type) :: pawarray
 type(pawps_type)    :: pawps
 type(pawrad_type)   :: pawrad
 type(pshead_type)   :: pshead

!Local variables ---------------------------------------
 integer :: ib,ierr,ilmn,ilm,iln,ir1,ir2,ir,jlmn,jlm,jln,j0lmn,klmn
 logical :: found,endfile,corefound,tcorefound,tvalefound,kijfound,shapefound,usevbare,vbarefound,vionfound
 logical,allocatable :: phifound(:),tphifound(:),tprojfound(:)
 real(dp), allocatable :: kij(:,:),tmp(:)
 character*(strlen) :: line,readline,strg

!-------------------------------------------------------

! ==================================================
! ====== 1-READ  ATOMICDATA FILE FROM AtomPAW ======
! ==================================================

!Init some flags
 allocate(phifound(pshead%basis_size));phifound=.false.
 allocate(tphifound(pshead%basis_size));tphifound=.false.
 allocate(tprojfound(pshead%basis_size));tprojfound=.false.
 corefound=.false.;tcorefound=.false.;tvalefound=.false.
 kijfound=.false.;shapefound=(pshead%shape_type/=-1)
 vionfound=.false.;vbarefound=.false.
 usevbare=(pshead%vlocopt==0)

!Open the atomicdata file for reading
 open(unit=funit,file=filename,form='formatted',status='old')

!Start a reading loop
 endfile=.false.
 do while (.not.endfile)
   read(funit,'(a)',err=10,end=10) readline
   line=adjustl(readline);goto 20
10 line="";endfile=.true.
20 continue

!--Read WAVE FUNCTIONS PHI
  if (line(1:16)=='<ae_partial_wave') then
   call rdfromline(" state",line,strg,ierr)
   ib=0;found=.false.
   do while (ib<pshead%basis_size.and.(.not.found))
    ib=ib+1
    if (trim(pshead%stateid(ib))==trim(strg)) then
     found=.true.;phifound(ib)=.true.
     read(funit,*) (pawps%phi(ir,ib),ir=1,pshead%wav_meshsz)
    endif
   enddo
  endif

!--Read PSEUDO WAVE FUNCTIONS TPHI
  if (line(1:20)=='<pseudo_partial_wave') then
   call rdfromline(" state",line,strg,ierr)
   ib=0;found=.false.
   do while (ib<pshead%basis_size.and.(.not.found))
    ib=ib+1
    if (trim(pshead%stateid(ib))==trim(strg)) then
     found=.true.;tphifound(ib)=.true.
     read(funit,*) (pawps%tphi(ir,ib),ir=1,pshead%wav_meshsz)
    endif
   enddo
  endif

!--Read PROJECTORS TPROJ
  if (line(1:19)=='<projector_function') then
   call rdfromline(" state",line,strg,ierr)
   ib=0;found=.false.
   do while (ib<pshead%basis_size.and.(.not.found))
    ib=ib+1
    if (trim(pshead%stateid(ib))==trim(strg)) then
     found=.true.;tprojfound(ib)=.true.
     pawps%tproj(:,ib)=zero
     read(funit,*) (pawps%tproj(ir,ib),ir=1,pshead%wav_meshsz)
    endif
   enddo
  endif

!--Read core density CORE_DENSITY
  if (line(1:16)=='<ae_core_density') then
   read(funit,*) (pawps%coreden4pr2(ir),ir=1,pshead%core_meshsz)
   pawps%coreden4pr2=4*pi*pawps%coreden4pr2/sqrt(4.d0*pi)
   pawps%coreden4pr2(1:pshead%core_meshsz)=pawps%coreden4pr2(1:pshead%core_meshsz) &
&                                         *(pawrad%rad(1:pshead%core_meshsz)**2)
   corefound=.true.
  endif

!--Read pseudized core density CORETAIL_DENSITY
  if (line(1:20)=='<pseudo_core_density') then
   read(funit,*) (pawps%tcoreden4pr2(ir),ir=1,pshead%core_meshsz)
   pawps%tcoreden4pr2=4*pi*pawps%tcoreden4pr2/sqrt(4.d0*pi)
   pawps%tcoreden4pr2(1:pshead%core_meshsz)=pawps%tcoreden4pr2(1:pshead%core_meshsz) &
&                                          *(pawrad%rad(1:pshead%core_meshsz)**2)
   tcorefound=.true.
  endif

!--Read pseudized valence density PSEUDO_VALENCE_DENSITY
  if (line(1:23)=='<pseudo_valence_density') then
   if (pshead%vale_meshsz>0) then
    read(funit,*) (pawps%tvaleden4pr2(ir),ir=1,pshead%vale_meshsz)
    pawps%tvaleden4pr2=4*pi*pawps%tvaleden4pr2/sqrt(4.d0*pi)
    pawps%tvaleden4pr2(1:pshead%vale_meshsz)=pawps%tvaleden4pr2(1:pshead%vale_meshsz) &
&                                           *(pawrad%rad(1:pshead%vale_meshsz)**2)
   endif
   tvalefound=.true.
  endif

!--Read Vbare potential VLOCFUN
  if (line(1:20)=='<zero_potential') then
   if (pshead%vlocopt==0) then
    allocate(tmp(pshead%wav_meshsz))
    read(funit,*) (tmp(ir),ir=1,pshead%wav_meshsz)
    pawps%vbare(1:pshead%sph_meshsz)=tmp(1:pshead%sph_meshsz)/sqrt(4.d0*pi)
   endif
   vbarefound=.true.
  endif

!--Read Vloc for Abinit potential VLOC_ION
  if (line(1:37)=='<kresse_joubert_local_ionic_potential') then
   if (pshead%vlocopt==1) then
    read(funit,*) (pawps%vhtnzc(ir),ir=1,pshead%vloc_meshsz)
    pawps%vhtnzc=pawps%vhtnzc/sqrt(4.d0*pi)
   endif
   vionfound=.true.
  endif

!--Read numeric SHAPE FUNCTION
  if ((pshead%shape_type==-1).and.(line(1:15)=='<shape_function')) then
   read(funit,*) (pawarray%shapefunc(ir),ir=1,pshead%wav_meshsz)
   pawarray%shapefunc=pawarray%shapefunc/sqrt(4.d0*pi)
   shapefound=.true.
  endif

!--Read Kinetic term KINETIC_ENERGY_MATRIX
  if (line(1:28)=='<kinetic_energy_differences>') then
   allocate(kij(pshead%basis_size,pshead%basis_size))
   read(funit,*) kij(:,:)
   pawarray%kij=zero
   do jlmn=1,pshead%lmn_size
    j0lmn=jlmn*(jlmn-1)/2
    jlm=pawarray%indlmn(4,jlmn);jln=pawarray%indlmn(5,jlmn)
    do ilmn=1,jlmn
     klmn=j0lmn+ilmn
     ilm=pawarray%indlmn(4,ilmn);iln=pawarray%indlmn(5,ilmn)
     if (ilm==jlm) pawarray%kij(klmn)=kij(iln,jln)
    enddo
   enddo
   deallocate(kij);kijfound=.true.
  endif

!End of reading loop
 enddo

!Close the atomicdata file
 close(funit)

!Test flags: is anything OK ?
 do ib=1,pshead%basis_size
  if (.not.phifound(ib)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: PHI(",ib,") not found !"
  if (.not.tphifound(ib)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: TPHI(",ib,") not found !"
  if (.not.tprojfound(ib)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: TPROJ(",ib,") not found !"
 enddo
 if (.not.corefound) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: CORE DENSITY not found !"
 if (.not.tcorefound) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: CORETAIL DENSITY not found !"
 if ((pshead%vale_meshsz>0).and.(.not.tvalefound)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: PSEUDO VALENCE DENSITY not found !"
 if (.not.kijfound) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: KINETIC ENERGY MATRIX not found !"
 if (.not.shapefound) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: SHAPE FUNCTION not found !"
 if (usevbare.and.(.not.vbarefound)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: VLOC FUNC not found !"
 if ((.not.usevbare).and.(.not.vionfound)) &
&  write(6,'(a,i2,a)') "Error in rdpawps2xml: VLOC ION not found !"
 found=(usevbare.and.vbarefound).or.((.not.usevbare).and.vionfound)
 found=found.and.corefound.and.tcorefound.and.tvalefound.and.kijfound.and.shapefound
 do ib=1,pshead%basis_size
  found=found.and.phifound(ib).and.tphifound(ib).and.tprojfound(ib)
 enddo
 if (.not.found) stop
 deallocate(phifound,tphifound,tprojfound)

!===========================================================
!Convert wave functions and projectors
 do ib=1,pshead%basis_size
  pawps%phi(1:pshead%wav_meshsz,ib)  =pawps%phi(1:pshead%wav_meshsz,ib)  *pawrad%rad(1:pshead%wav_meshsz)
  pawps%tphi(1:pshead%wav_meshsz,ib) =pawps%tphi(1:pshead%wav_meshsz,ib) *pawrad%rad(1:pshead%wav_meshsz)
  pawps%tproj(1:pshead%wav_meshsz,ib)=pawps%tproj(1:pshead%wav_meshsz,ib)*pawrad%rad(1:pshead%wav_meshsz)
 enddo

!===========================================================
!Test pseudo valence density
 if (pshead%vale_meshsz>0) then
  write(6,'(/,a,a,f5.2,a,a,g11.4)') 'Info:',&
&   '  At r=',pawrad%rad(pshead%vale_meshsz),' a.u.,',&
&   ' Pseudo_valence_density= ', pawps%tvaleden4pr2(pshead%vale_meshsz) &
&                               /pawrad%rad(pshead%vale_meshsz)**2/(4*pi)
  write(6,'(a)') '  This quantity must be as small as possible.'
endif

!===========================================================
!Test potential
 if (pshead%vlocopt==1) then
  write(6,'(/,a,a,f5.2,a,a,g11.4)') 'Info:',&
&   '  At r_vloc=',pawrad%rad(pshead%vloc_meshsz),' a.u.,',&
&   ' VHartree(ntild(Zv+Zc))= -Zv/r + ', pawps%vhtnzc(pshead%vloc_meshsz) &
&    +(pshead%atomic_charge-pshead%core_charge)/pawrad%rad(pshead%vloc_meshsz)
  write(6,'(a)') '  This quantity must be as small as possible.'
endif

 end subroutine
!!***
