!{\src2tex{textfont=tt}}
!!****f* ABINIT/atompaw2abinit
!! NAME
!! atompaw2abinit
!!
!! FUNCTION
!! Main routine for building a psp file for PAW calculations
!! in Abinit, starting from the code atomPAW (N. Holzwarth).
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! NOTES
!! The present main routine drives the following operations :
!!
!! 1) Read useful data from the output file of AtomPAW
!! 2) Calculate quantities that do not depend from self-consistent PAW
!!    iterations
!! 3) Write an output file to be read as an input PAW atomic data
!!    file in abinit
!!
!!       Units: Energies=Hartree, Lengths=Bohr
!!
!! PARENTS
!!
!! CHILDREN
!!    calc_dij0,calc_rhoij0,calc_shapef,calc_tcore,initmesh,initpawps,opt_proj
!!    rdpawps1,rdpawps1xml,rdpawps2,rdpawps2xml,wrpawps
!!
!! SOURCE

 program atompaw2abinit

 use defs_basis
 use defs_pawps

 implicit none

!==================================================================
! DECLARATIONS

 character*10 :: version='3.2.0', verdate='july 2007', abinitver='5.4.3+'

 integer, parameter :: unit_atompaw=21, unit_abinit=22

 integer  :: loggrd_size_def=350
 real(dp) :: loggrd_step_def=0.035d0

 real(dp) :: ecut_rso_def=10.d0,&
&            gfact_rso_def=2.d0,&
&            werror_rso_def=1.d-4

 character*(fnlen) :: empty='',&
&                     file_abinit='abinit.pawps',&
&                     file_atompaw='(atom).atomicdata',&
                      file_xml='(atom.exc).xml'
 character*(fnlen) :: response,filefmt
!These types are defined in defs_pawps
 type(loggrd_type)   :: loggrd
 type(pawarray_type) :: pawarray
 type(pawps_type)    :: pawps
 type(pawrad_type)   :: pawrad
 type(pawrso_type)   :: pawrso
 type(pshead_type)   :: pshead

!-------------------------------------------------------
!BEGIN EXECUTABLE SECTION
!
!------------------------------------------------------------------
!---- Write header and read file names
!------------------------------------------------------------------

 write(6,'(a)') '========================================================'
 write(6,'(3a)') '==   atompaw2abinit - v',trim(version),':                         =='
 write(6,'(a)') '==   Converts a PAW pseudopotential file generated    =='
 write(6,'(a)') '==   by "atompaw" code (from N. Holzwarth)            =='
 write(6,'(a)') '==   into a PAW atomic data file readable             =='
 write(6,'(3a)') '==   by Abinit (v',trim(abinitver),')                              =='
 write(6,'(a)') '==                                                    =='
 write(6,'(a)') '==   "atompaw" can be found at:  http://pwpaw.wfu.edu =='
 write(6,'(a)') '========================================================'

 write(6,'(/)',ADVANCE='NO')
 write(6,'(80a)',ADVANCE='NO') &
    'Input atompaw datafile [',&
       trim(file_atompaw),' or ',trim(file_xml),'] ? '
 read (5,'(A)') response
 if (response/=empty) file_atompaw=trim(response)

 write(6,'(3a)',ADVANCE='NO') &
&   'Output psp file name (for Abinit) [',trim(file_abinit),'] ? '
 read (5,'(A)') response
 if (response/=empty) file_abinit=trim(response)
 write(6,*)

!To be activated later:
! write(6,'(a,/,a)',ADVANCE='NO') &
!&   'Do you want to keep the format of Vloc potential (recommended) ',&
!&   '(or transfer it to VH(tild_n_zc) format)  [y] ? '
! read (5,'(A)') response
! pshead%vlocopt=1;if (response==empty.or.response=="y".or.response=="Y") pshead%vlocopt=0
 pshead%vlocopt=1

 write(6,'(a,/,a)',ADVANCE='NO') &
&   'Do you want to improve non-local projector',&
&   'by using "Real Space Optimization" (King-Smith et al.) [n] ? '
 read (5,'(A)') response
 pawrso%userso=(response=="y".or.response=="Y")

 pawrso%ecut=ecut_rso_def;pawrso%gfact=gfact_rso_def;pawrso%werror=werror_rso_def
 if (pawrso%userso) then
  write(6,'(a,f4.1,a,f3.1,a,g6.1,a)',ADVANCE='NO') &
&    'Real Space optim.: Ecut, Gamma/Gmax, Wl(error) [',&
&    ecut_rso_def,', ',gfact_rso_def,', ',werror_rso_def,'] ? '
  read (5,'(A)') response
  if (response/=empty) read(unit=response,fmt=*) pawrso%ecut,pawrso%gfact,pawrso%werror
  write(6,*)
 endif

 write(6,*)

!------------------------------------------------------------------
!---- Read PAW pseudopotential file from AtomPAW
!------------------------------------------------------------------

 allocate(pshead%orbitals(20),pshead%occ(20),pshead%stateid(20))

!---- Read the header (sizes of arrays)
 open(unit=unit_atompaw,file=file_atompaw,form='formatted',status='old')
 read(unit_atompaw,'(a)') filefmt
 close(unit_atompaw)
 if (filefmt(3:5)=='xml') then
  call rdpawps1xml(file_atompaw,pshead,unit_atompaw,version)
 else
  call rdpawps1(file_atompaw,pshead,unit_atompaw,version)
 end if

!---- Ask for an eventual transfer on an other grid
 if (pshead%mesh_type==1) then
  write(6,'(a,/,a)',ADVANCE='NO') &
&    'Do you want to transfer atomic data from AtomPAW',&
&    'on a (reduced) logarithmic grid    (recommended) [y] ? '
  read (5,'(A)') response
  loggrd%uselog=(response==empty.or.response=="y".or.response=="Y")
  if (loggrd%uselog) then
   loggrd%meshsz=loggrd_size_def;loggrd%log_step=loggrd_step_def
   write(6,'(a,i3,a,f5.3,a)',ADVANCE='NO') &
&    'Logarithmic grid: Number of pts, logarithmic step [',&
&    loggrd_size_def,', ',loggrd_step_def,'] ? '
  read (5,'(A)') response
   if (response/=empty) read(unit=response,fmt=*) loggrd%meshsz,loggrd%log_step
   write(6,*)
  endif
 else
  loggrd%uselog=.false.
 end if

!---- Initialize some useful data
 allocate(pawarray%indlmn(6,pshead%lmn_size))
 call initpawps(pawarray,pshead,version)

!---- Allocate memory
 allocate(pawps%phi(pshead%wav_meshsz,pshead%basis_size))
 allocate(pawps%tphi(pshead%wav_meshsz,pshead%basis_size))
 allocate(pawps%tproj(pshead%prj_msz_max,pshead%basis_size))
 allocate(pawps%coreden4pr2(pshead%core_meshsz))
 allocate(pawps%tcoreden4pr2(pshead%core_meshsz))
 allocate(pawps%tvaleden4pr2(pshead%vale_meshsz))
 allocate(pawps%dij0(pshead%lmn2_size))
 allocate(pawps%rhoij0(pshead%lmn2_size))
 allocate(pawarray%shapefunc(pshead%wav_meshsz))
 allocate(pawarray%shpnrm(pshead%l_size))
 allocate(pawarray%kij(pshead%lmn2_size))
 allocate(pawarray%tvaleden4pr2(pshead%wav_meshsz))
 allocate(pawarray%hatden4pr2(pshead%sph_meshsz))
 if (pshead%vlocopt==0) allocate(pawps%vbare(pshead%sph_meshsz))
 if (pshead%vlocopt==1) allocate(pawps%vhtnzc(pshead%vloc_meshsz))

!---- Initialize radial mesh
 allocate(pawrad%rad(pshead%max_meshsz),pawrad%radfact(pshead%max_meshsz))
 call initmesh(pawrad,pshead%max_meshsz,pshead%mesh_type,pshead%rad_step,pshead%log_step)

!---- Read the body
 if (filefmt(3:5)=='xml') then
  call rdpawps2xml(file_atompaw,pawarray,pawps,pawrad,pshead,unit_atompaw)
 else
  call rdpawps2(file_atompaw,pawarray,pawps,pawrad,pshead,unit_atompaw)
 end if

!------------------------------------------------------------------
!---- Compute new data needed by Abinit
!------------------------------------------------------------------
 call calc_shapef(pawarray,pawrad,pshead)
 call calc_valden(pawarray,pawps,pawrad,pshead)
 call calc_dij0(pawarray,pawps,pawrad,pshead)
 call calc_rhoij0(pawps,pshead)
 call opt_proj(pawps,pawrad,pawrso,pshead)

!------------------------------------------------------------------
!---- Write PAW pseudopotential file for Abinit
!------------------------------------------------------------------

 call wrpawps(file_abinit,loggrd,pawarray,pawps,pawrad,pawrso,pshead,unit_abinit)

!------------------------------------------------------------------
!---- Deallocate memory
!------------------------------------------------------------------

 deallocate(pshead%orbitals,pshead%occ,pshead%stateid)
 deallocate(pawps%phi,pawps%tphi,pawps%tproj)
 deallocate(pawps%coreden4pr2,pawps%tcoreden4pr2,pawps%tvaleden4pr2)
 deallocate(pawps%dij0,pawps%rhoij0)
 deallocate(pawarray%shapefunc,pawarray%shpnrm,pawarray%kij)
 deallocate(pawarray%tvaleden4pr2,pawarray%hatden4pr2)
 if (pshead%vlocopt==0) deallocate(pawps%vbare)
 if (pshead%vlocopt==1) deallocate(pawps%vhtnzc)
 deallocate(pawarray%indlmn,pawrad%rad,pawrad%radfact)

 write(6,'(/,a)') 'Program ended.'

 end program
!!***
