!{\src2tex{textfont=tt}}
!!****f* ABINIT/splinefit
!! NAME
!! splinefit
!!
!! FUNCTION
!! Spline routine (using naturally bound conditions and regular grid for input)
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  n1=number of input points
!!  h1=step of regular input grid x_i
!!  f1=values of function to fit (y_i)
!!
!! OUTPUT
!!  n2=number of output points
!!  r2=abscisses of output grid X_prime_i
!!  f2=values of fitted function (y_prime_i)
!!
!! PARENTS
!!      wrpawps
!!
!! NOTES
!!   Code follows [W.H. Press et al., Numerical
!!   Recipes (Cambridge University Press, New York, 1986].
!!
!! SOURCE

 subroutine splinefit(n1,h1,f1,n2,r2,f2)

 use defs_basis

 implicit none

!Arguments ---------------------------------------------
 integer :: n1,n2
 real(dp) :: h1
 real(dp) :: f1(n1),r2(n2),f2(n2)

!Local variables ---------------------------------------
 integer :: i1
 real(dp) :: yp1,ypn
 real(dp) :: r1(n1),work1(n1),work2(n1),work3(n2-1),work4(n2-1)

!-------------------------------------------------------

  do i1=1,n1
   r1(i1)=dble(i1-1)*h1
  enddo
 if (r2(n2)>r1(n1)) then
  print*, 'Error:'
  print *,'  splinefit: spline problem'
  print *,"  r1(n1)=",r1(n1)
  print *,"  r2(n2)=",r2(n2)
  stop
 endif
 f2(1)=f1(1)
 work3(1:n2-1)=r2(2:n2)
 yp1=(f1(3)-f1(1))/(2.d0*h1)
 ypn=(f1(n1)-f1(n1-1))/h1
 call spline(r1,f1,n1,yp1,ypn,work1,work2)
 call splint(n1,r1,f1,work1,n2-1,work3,work4)
 f2(2:n2)=work4(1:n2-1)

 end subroutine


!======================================================================
!======================================================================

subroutine spline(x, y, n, yp1, ypn, y2, work)

! Cubic spline routine based on p. 88 of Numerical Recipes.
! Given arrays x and y of length n containing y=f(x) with x s in
! ascending order and given yp1 and ypn for first derivative of interpolating
! function at the endpoints, returns array y2 of length n which contains
! the second derivatives of the interpolating function at the tabulated
! points x.  If yp1 and/or ypn are 1.e30 or larger, routine sets corresponding
! boundary condition for a natural spline, with (0._q,0._q) second derivative on
! that boundary.
! The cubic spline fit to the function is then given by
!
!  y = A y  + B y    + C y'' + D y''
!	  j	 j+1	  j	  j+1
!
! with A=(x(j+1)-x)/(x(j+1)-x(j)), B=1-A=(x-x(j))/(x(j+1)-x(j)),
! C=(A^3-A)(x(j+1)-x(j))^2/6, and D=(B^3-B)(x(j+1)-x(j))^2/6.
!
! The first derivative is therefore (with dx = x(j+1)-x(j))
!
!  y prime = (y(j+1)-y(j))/dx + (3A^2-1)dx y''(j)/6 + (3B^2-1)dx y''(j+1)/6
!
! and the second derivative is
!
!  y'' = A y''(j) + B y''(j+1)
!
! Input:
!  x(n)=x values in ascending order.
!  y(n)=y values at x points.
!  n=number of incoming data points.
!  yp1=y prime at x(1) or else > 1e30 (latter uses natural spline).
!  ypn=y prime at x(n) or else > 1e30 (as above).
!  Note that use of a natural spline has little to recommend it.
!  work(n)=work space.
! Output:
!  y2(n)=spline fit array of y'' values.

 use defs_basis

 implicit none

!Arguments -------------------------------------
 integer :: n
 real(dp) :: yp1,ypn
 real(dp) :: x(n),y(n),y2(n),work(n)

!Local variables -------------------------------
 integer :: i,k
 real(dp) :: sig,p,qn,workn

!-----------------------------------------------

      if (yp1>1.0d+30) then
!       lower boundary condition is either natural ...
        y2(1) = 0.0d0
        work(1) = 0.0d0
      else
!       or else to have a specified first derivative.
        y2(1) =  - 0.50d0
        work(1) = (3.d0/(x(2)-x(1)))*((y(2)-y(1))/(x(2)-x(1))-yp1)
      end if
!     Decomposition loop of tridiagonal algorithm:
      do i=2,n-1
        sig = (x(i)-x(i-1))/(x(i+1)-x(i-1))
        p = sig*y2(i-1) + 2.0d0
        y2(i) = (sig-1.0d0)/p
        work(i)=(6.0d0*((y(i+1)-y(i))/(x(i+1)-x(i))-(y(i)-y(i-1))&
&        /(x(i)-x(i-1)))/(x(i+1)-x(i-1))-sig*work(i-1))/p
      enddo
      if (ypn>1.0d+30) then
!       Set upper boundary condition to be natural ...
        qn = 0.0d0
        workn = 0.0d0
      else
!       Or else to have a specified first derivative
        qn = 0.50d0
        workn=(3.d0/(x(n)-x(n-1)))*(ypn-(y(n)-y(n-1))/(x(n)-x(n-1)))
      end if
      y2(n) = (workn-qn*work(n-1))/(qn*y2(n-1)+1.0d0)
!     Backsubstitution loop of tridiagonal algorithm:
      do k=n-1,1,-1
        y2(k)=y2(k)*y2(k+1)+work(k)
      enddo

end subroutine


!======================================================================
!======================================================================

subroutine splint (nspline,xspline,yspline,ysplin2,nfit,xfit,yfit)

! Calculate spline interpolation
!  ON INPUT:
!  nspline: number of grid points of input mesh
!  xspline(nspline): input mesh
!  yspline(nspline): function on input mesh
!  ysplin2(nspline): second derivative of yspline on input mesh
!  nfit: number of points of output mesh
!  xfit(nfit): output mesh
!  ON OUTPUT:
!  yfit(nfit): function on output mesh

 use defs_basis

 implicit none

!Arguments -------------------------------------
 integer :: nspline,nfit
 real(dp) :: xspline(nspline), yspline(nspline),&
&           ysplin2(nspline),xfit(nfit),yfit(nfit)

!Local variables -------------------------------
 integer :: klo,i,k,khi
 real(dp) :: h,a,b

!-----------------------------------------------

      klo = 1
      do i=1, nfit
        do k=klo+1, nspline
            if(xspline(k).ge.xfit(i)) then
               if(xspline(k-1).le.xfit(i)) then
                  khi = k
                  klo = k-1
               else
                  if (k-1.eq.1 .and. i.eq.1) then
                     stop '  SPLINT: xfit(1) < xspline(1)'
                  else
                     stop '  SPLINT: xfit not properly ordered'
                  end if
               end if
               h= xspline(khi) - xspline(klo)
               a= (xspline(khi)-xfit(i))/h
               b= (xfit(i)-xspline(klo))/h

               yfit(i) = a*yspline(klo) + b*yspline(khi)&
&                   +( (a**3-a)*ysplin2(klo) +&
&                   (b**3-b)*ysplin2(khi) ) *h*h/6
               go to 10
            end if
         end do
!        This is for the unlikely event that rmax exceed r(mesh)
         yfit(i)=0.d0
 10     continue
      end do

end subroutine



!!***
