!{\src2tex{textfont=tt}}
!!****m* ABINIT/defs_mover
!! NAME
!! defs_mover
!!
!! FUNCTION
!! This module contains definition of an structured datatypes used
!! by the mover routine.
!! If you want to add one new datatype, please, examine first
!! whether another datatype might meet your need (e.g. adding some
!! records to it).
!! Then, if you are sure your new structured datatype is needed,
!! write it here, and DOCUMENT it properly (not all datastructure
!! here are well documented, it is a shame ...).
!! Do not forget : you will likely be the major winner if you
!! document properly.
!! Proper documentation of a structured datatype means :
!!  (1) Mention it in the list just below
!!  (2) Describe it in the NOTES section
!!  (3) Put it in alphabetical order in the the main section of
!!      this module
!!  (4) Document each of its records, except if they are described
!!      elsewhere
!!      (this exception is typically the case of the dataset
!!      associated with input variables, for which there is a help
!!      file)
!!
!! List of datatypes :
!! * ab_movehistory  : Historical record of previous states of
!!                     during the optimization of geometry
!! * ab_movetype     : Subset of dtset with data for move atoms
!! * ab_xfh_type     : Old history style
!!
!! List of subroutines :
!! * ab_movehistory_nullify  : Nullify an ab_movehistory
!! * ab_movetype_nullify     : Nullify an ab_movetype
!! * ab_movetype_print       : Print the contents of an ab_movetype
!!
!! COPYRIGHT
!! Copyright (C) 2001-2010 ABINIT group (XG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

module defs_mover

 use defs_basis
 
implicit none
!!***

 integer, parameter :: mover_BEFORE=0
 integer, parameter :: mover_AFTER=1

!Structures
!!***

!!****t* defs_mover/ab_movehistory
!! NAME
!! ab_movehistory
!!
!! FUNCTION
!! This function has several vectors, and index scalars to store
!! a proper history of previous evaluations of forces and
!! stresses,velocities,positions and energies
!!
!! NOTES
!! The vectors are not allocated because in some cases
!! not all the vectors are needed, in particular a history
!! of stresses is only needed if optcell/=0, and a history
!! of velocities is needed for ionmov==1
!!
!! Store acell, rprimd and strten even with optcell/=0
!! represent a waste of 12x (dp)[Usually 8 Bytes] per
!! iteration, the reason to store all the records is
!! because some routines (eg bfgs.F90) uses the metric (gmet)
!! for initialize the hessian and we need rprimd for that.
!!
!! SOURCE

type ab_movehistory

! scalars
! Index of the last element on all records
integer :: ihist
! Maximun size of the historical records
integer :: mxhist
! Booleans to know if some arrays are changing
logical :: isVused  ! If velocities are changing
logical :: isARused ! If Acell and Rprimd are changing

! arrays
! Vector of (x,y,z)X(mxhist)
real(dp), pointer :: histA(:,:)
! Vector of (mxhist) values of energy
real(dp), pointer :: histE(:)
! Vector of (x,y,z)X(x,y,z)X(mxhist)
real(dp), pointer :: histR(:,:,:)
! Vector of (stress [6])X(mxhist)
real(dp), pointer :: histS(:,:)
! Vector of (x,y,z)X(natom)X(mxhist) values of velocity
real(dp), pointer :: histV(:,:,:)
! Vector of (x,y,z)X(natom)X(xcart,xred,fcart,fred)X(mxhist)
real(dp), pointer :: histXF(:,:,:,:)

end type ab_movehistory
!!***

!!****t* defs_mover/ab_movetype
!! NAME
!! ab_movetype
!!
!! FUNCTION
!! This datatype has the purpouse of store all the data taked
!! usually from dtset needed for the different predictors
!! to update positions, acell, etc.
!!
!! NOTES
!!  At present 26 variables are present in ab_movetype
!!  if a new variable is added in ab_movetype it should
!!  be added also for nullify in ab_movetype_nullify
!!
!!
!! SOURCE

type ab_movetype

! scalars
! DIIS memory (Use by pred_diisrelax only)
integer,pointer  :: diismemory
! Delta Time for IONs
real(dp),pointer  :: dtion
! include a JELLium SLAB in the cell
integer,pointer  :: jellslab
! Number of ATOMs
integer,pointer  :: natom
! Number of CONstraint EQuations
integer,pointer  :: nconeq
! Number of SYMmetry operations
integer,pointer  :: nsym
! Number of Types of atoms
integer,pointer  :: ntypat
! OPTimize the CELL shape and dimensions
integer,pointer  :: optcell
! RESTART Xcart and Fred
integer,pointer  :: restartxf
! Molecular Dynamics Final Temperature
real(dp),pointer :: mdftemp
! Molecular Dynamics Initial Temperature
real(dp),pointer :: mditemp
! NOT DOCUMENTED
real(dp),pointer :: noseinert
! Phonon Temperature (Use by pred_scphon only)
real(dp),pointer :: scphon_temp
! STRess PRECONditioner
real(dp),pointer :: strprecon
! VIScosity
real(dp),pointer :: vis

! arrays
! Indices of AToms that are FIXed
integer,  pointer :: iatfix(:,:)         ! iatfix(3,natom)
! SYMmetry in REaL space
integer,  pointer :: symrel(:,:,:)       ! symrel(3,3,nsym)
! Supercell_multiplicity (Use by pred_scphon only)
integer,  pointer :: scphon_supercell(:) ! scphon_supercell(3)
! Mass of each atom (NOT IN DTSET)
real(dp), pointer :: amass(:)            ! amass(natom)
! STRess TARGET
real(dp), pointer :: strtarget(:)        ! strtarget(6)
! TYPe of ATom
integer,  pointer :: typat(:)            ! typat(natom)
! Filename for Hessian matrix
character(len=fnlen), pointer :: fnameabi_hes
! Filename for _PCINFO file
character(len=fnlen), pointer :: filnam_ds3 ! dtfil%filnam_ds(3)
! Filename for _PHFRQ file
character(len=fnlen), pointer :: fnameabi_phfrq ! dtfil%fnameabi_phfrq
! Filename for _PHVEC file
character(len=fnlen), pointer :: fnameabi_phvec ! dtfil%fnameabi_phvec
! Z number of each NUCLeus
real(dp), pointer :: znucl(:)            ! znucl(npsp)

end type ab_movetype
!!***

!!****t* defs_mover/ab_xfh_type
!! NAME
!! ab_xfh_type
!!
!! FUNCTION
!! Datatype with the old structure for storing history
!! used in gstate and brdmin,delocint, and others
!!
!!
!! NOTES
!!
!! This is a transitional structure, to bridge between
!! the old code and the new one base on ab_movehistory
!!
!! SOURCE

type ab_xfh_type

!  mxfh = last dimension of the xfhist array
!  nxfh = actual number of (x,f) history pairs, see xfhist array
integer :: nxfh,nxfhr,mxfh

!  xfhist(3,natom+4,2,mxfh) = (x,f) history array, also including
!   rprim and stress
real(dp),pointer :: xfhist(:,:,:,:)

end type ab_xfh_type
!!***

!!****t* defs_mover/ab_delocint
!! NAME
!! ab_delocint
!!
!! FUNCTION
!! Datatype with the important variables in pred_delocint
!!
!! NOTES
!!
!!   deloc <type(ab_delocint)>=Important variables for 
!!   |                           pred_delocint
!!   | 
!!   | nang     = Number of angles
!!   | nbond    = Number of bonds
!!   | ncart    = Number of cartesian directions 
!!   |             (used for constraints)
!!   | ndihed   = Number of dihedrals
!!   | nrshift  = Dimension of rshift
!!   | ninternal= Number of internal coordinates
!!   |            ninternal=nbond+nang+ndihed+ncart
!!   |
!!   | angs(2,3,nang)  = Indexes to characterize angles
!!   | bonds(2,2,nbond)= For a bond between iatom and jatom
!!   |                   bonds(1,1,nbond) = iatom
!!   |                   bonds(2,1,nbond) = icenter
!!   |                   bonds(1,2,nbond) = jatom
!!   |                   bonds(2,2,nbond) = irshift
!!   | carts(2,ncart)  = Index of total primitive internal,
!!   |                   and atom (carts(2,:))
!!   | dihedrals(2,4,ndihed)= Indexes to characterize dihedrals
!!   | 
!!   | rshift(3,nrshift)= Shift in xred that must be done to find
!!   |                    all neighbors of a given atom within a
!!   |                    given number of neighboring shells
!!
!! SOURCE

type ab_delocint

! scalars
 integer :: nang
 integer :: nbond
 integer :: ncart
 integer :: ndihed
 integer :: nrshift
 integer :: ninternal

! arrays
 integer,pointer :: angs(:,:,:)
 integer,pointer :: bonds(:,:,:)
 integer,pointer :: carts(:,:)
 integer,pointer :: dihedrals(:,:,:)
 real(dp),pointer :: rshift(:,:)

end type ab_delocint
!!***

CONTAINS  !=============================================================
!!***

!!****f* defs_mover/ab_movetype_nullify
!! NAME
!! ab_movetype_nullify
!!
!! FUNCTION
!! Nullify all the pointers in a ab_mover
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  ab_mover <type(ab_movetype)> = The ab_mover to nullify
!!
!! PARENTS
!!      mover
!!
!! CHILDREN
!!
!! NOTES
!!  At present 26 variables are present in ab_movetype
!!  if a new variable is added in ab_movetype it should
!!  be added also for nullify here
!!
!! SOURCE

subroutine ab_movetype_nullify(ab_mover)


implicit none

!Arguments ------------------------------------

 type(ab_movetype),intent(inout) :: ab_mover

! ***************************************************************

! Number of Geometry histories (Use by pred_diisrelax only)
 nullify(ab_mover%diismemory)
! Delta Time for IONs
 nullify(ab_mover%dtion)
! include a JELLium SLAB in the cell
 nullify(ab_mover%jellslab)
! Number of ATOMs
 nullify(ab_mover%natom)
! Number of CONstraint EQuations
 nullify(ab_mover%nconeq)
! Number of SYMmetry operations
 nullify(ab_mover%nsym)
! Number of Types of atoms
 nullify(ab_mover%ntypat)
! OPTimize the CELL shape and dimensions
 nullify(ab_mover%optcell)
! RESTART Xcart and Fred
 nullify(ab_mover%restartxf)
! Molecular Dynamics Final Temperature
 nullify(ab_mover%mdftemp)
! Molecular Dynamics Initial Temperature
 nullify(ab_mover%mditemp)
! NOT DOCUMENTED
 nullify(ab_mover%noseinert)
! Phonon Temperature (Use by pred_scphon only)
 nullify(ab_mover%scphon_temp)
! STRess PRECONditioner
 nullify(ab_mover%strprecon)
! VIScosity
 nullify(ab_mover%vis)

!arrays
! Indices of AToms that are FIXed
 nullify(ab_mover%iatfix)
! SYMmetry in REaL space
 nullify(ab_mover%symrel)
! Supercell_multiplicity (Use by pred_scphon only)
 nullify(ab_mover%scphon_supercell)
! Mass of each atom (NOT IN DTSET)
 nullify(ab_mover%amass)
! STRess TARGET
 nullify(ab_mover%strtarget)
! TYPe of ATom
 nullify(ab_mover%typat)
! Filename for Hessian matrix
 nullify(ab_mover%fnameabi_hes)
! Filename for _PCINFO file
 nullify(ab_mover%filnam_ds3)
! Filename for _PHFRQ file
 nullify(ab_mover%fnameabi_phfrq)
! Filename for _PHVEC file
 nullify(ab_mover%fnameabi_phvec)
! Z number of each NUCLeus
 nullify(ab_mover%znucl)


end subroutine ab_movetype_nullify
!!***

!!****f* defs_mover/ab_movetype_print
!! NAME
!! ab_movetype_print
!!
!! FUNCTION
!! Nullify all the pointers in a ab_mover
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  ab_mover <type(ab_movetype)> = The ab_mover to nullify
!!
!! PARENTS
!!
!! CHILDREN
!!
!! NOTES
!!  At present 17 variables are present in ab_movetype
!!  if a new variable is added in ab_movetype it should
!!  be added also for print here
!!
!! SOURCE

subroutine ab_movetype_print(ab_mover,iout)


implicit none

!Arguments ------------------------------------
 type(ab_movetype),intent(inout) :: ab_mover
 integer,intent(in) :: iout

!Local variables-------------------------------
!arrays
character(len=1200) :: message
character(len=110)   :: fmt

! ***************************************************************

fmt='(a,e12.5,a,a,I5,a,a,I5,a,a,I5,a,a,I5,a,a,I5,a,a,I5,a,a,e12.5,a,a,e12.5,a,a,e12.5,a,a,e12.5,a,a,e12.5,a)'

write(message,fmt)&
& 'Delta Time for IONs',ab_mover%dtion,ch10, &
& 'include a JELLium SLAB in the cell',ab_mover%jellslab,ch10, &
& 'Number of ATOMs',ab_mover%natom,ch10, &
& 'Number of CONstraint EQuations',ab_mover%nconeq,ch10, &
& 'Number of SYMmetry operations',ab_mover%nsym,ch10, &
& 'OPTimize the CELL shape and dimensions',ab_mover%optcell,ch10, &
& 'RESTART Xcart and Fred',ab_mover%restartxf,ch10, &
& 'Molecular Dynamics Final Temperature',ab_mover%mdftemp,ch10, &
& 'Molecular Dynamics Initial Temperature',ab_mover%mditemp,ch10, &
& 'NOT DOCUMENTED',ab_mover%noseinert,ch10, &
& 'STRess PRECONditioner',ab_mover%strprecon,ch10, &
& 'VIScosity',ab_mover%vis,ch10

! ! arrays
! ! Indices of AToms that are FIXed
! integer,  pointer :: iatfix(:,:)
! ! SYMmetry in REaL space
! integer,  pointer :: symrel(:,:,:)
! ! Mass of each atom (NOT IN DTSET)
! real(dp), pointer :: amass(:)
! ! STRess TARGET
! real(dp), pointer :: strtarget(:)
! Filename for Hessian matrix
! character(len=fnlen), pointer :: fnameabi_hes

 write(iout,*) 'CONTENTS of ab_mover'
 write(iout,'(a)') message

end subroutine ab_movetype_print
!!***

!!****f* defs_mover/ab_movehistory_nullify
!! NAME
!! ab_movehistory_nullify
!!
!! FUNCTION
!! Nullify all the pointers in a hist
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  hist <type(ab_movehistory)> = The hist to nullify
!!
!! PARENTS
!!      mover
!!
!! CHILDREN
!!
!! NOTES
!!  At present 17 variables are present in ab_movetype
!!  if a new variable is added in ab_movetype it should
!!  be added also for nullify here
!!
!! SOURCE

subroutine ab_movehistory_nullify(hist)


implicit none

!Arguments ------------------------------------

 type(ab_movehistory),intent(inout) :: hist

! ***************************************************************

! Vector of (x,y,z)X(mxhist)
 nullify(hist%histA)
! Vector of (mxhist) values of energy
 nullify(hist%histE)
! Vector of (x,y,z)X(x,y,z)X(mxhist)
 nullify(hist%histR)
! Vector of (stress [6])X(mxhist)
 nullify(hist%histS)
! Vector of (x,y,z)X(natom)X(mxhist) values of velocity
 nullify(hist%histV)
! Vector of (x,y,z)X(natom)X(xcart,xred,fcart,fred)X(mxhist)
 nullify(hist%histXF)

end subroutine ab_movehistory_nullify
!!***

!----------------------------------------------------------------------

end module defs_mover
!!***
