!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_xmpi
!! NAME
!!  m_xmpi
!!
!! FUNCTION
!!  This module provides MPI named constants, tools for inquiring the MPI environment 
!!  and a set of generic interfaces wrapping the most commonly used MPI primitives.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MG, MB, XG, YP, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! NOTES
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_xmpi

 use defs_basis,  only : sp, dp, spc, dpc

#if defined HAVE_MPI && defined HAVE_MPI2
 use mpi
#endif

 implicit none

 private

#if defined HAVE_MPI && defined HAVE_MPI1
 include 'mpif.h'
#endif

#ifdef HAVE_MPI
 ! MPI constants used in abinit. Make sure that a corresponding fake value is provided for the sequential version.
 integer,public,parameter :: xmpi_world          = MPI_COMM_WORLD
 integer,public,parameter :: xmpi_self           = MPI_COMM_SELF
 integer,public,parameter :: xmpi_undefined      = MPI_UNDEFINED  
 integer,public,parameter :: xmpi_undefined_rank = MPI_UNDEFINED  ! MPI_UNDEFINED_RANK is not portable.

#else
 ! Fake replacements for the sequential version.
 integer,public,parameter :: xmpi_world          = 0
 integer,public,parameter :: xmpi_self           = 0
 integer,public,parameter :: xmpi_undefined      =-32765
 integer,public,parameter :: xmpi_undefined_rank =-32766
#endif

 integer,private,save :: xmpi_tag_ub=32767 
 ! The tag upper bound value must be at least 32767. An MPI implementation is free to make 
 ! the value of MPI_TAG_UB larger than this hence xmpi_tag_ub is redefined when MPI is init in xmpi_init.

!----------------------------------------------------------------------

!!***

!!****t* m_xmpi/mpicomm_t
!! NAME
!!  mpicomm_t
!!
!! FUNCTION
!!  Datatype used to store data associated to an MPI communicator.
!!
!! SOURCE

 type,public :: mpicomm_t
  integer :: id    = xmpi_undefined
  ! The MPI communicator identifier.

  integer :: my_rank = xmpi_undefined_rank
  ! The rank of the node inside comm.

  integer :: master  = 0
  ! The rank of master node.

  integer :: nprocs = xmpi_undefined
  ! The number of processors in the communicator.

  !$integer,pointer :: ranks_in_world(:)   SET2NULL
  ! The MPI ranks in MPI_COMM_WORLD of the nodes beloging to the communicator.

 end type mpicomm_t
!!***

! Public procedures.
 public :: xmpi_init      ! Initialize the MPI environment.
 public :: xmpi_end       ! Terminate the MPI environment.
 public :: xcomm_rank     ! The rank of the node inside the communicator.
 public :: xcomm_size     ! The number of processors inside the communicator.
 public :: xbarrier_mpi   ! Hides MPI_BARRIER from MPI library.
 public :: xmpi_name      ! Return the name of this processor (usually the hostname).
 public :: xerror_string  ! Return a string describing the error from ierr.

 ! MPI generic interfaces.
 public :: xallgather_mpi
 public :: xallgatherv_mpi
 public :: xalltoall_mpi
 public :: xalltoallv_mpi
 public :: xcast_mpi
 public :: xexch_mpi
 public :: xmax_mpi
 public :: xmin_mpi
 public :: xrecv_mpi
 public :: xsend_mpi
 public :: xsum_master
 public :: xsum_mpi

!----------------------------------------------------------------------

interface xallgather_mpi
  module procedure xallgather_mpi_int
  module procedure xallgather_mpi_char
  module procedure xallgather_mpi_int1d
  module procedure xallgather_mpi_dp1d
  module procedure xallgather_mpi_dp2d
  module procedure xallgather_mpi_dp3d
  module procedure xallgather_mpi_dp4d
end interface xallgather_mpi

!----------------------------------------------------------------------

interface xallgatherv_mpi
  module procedure xallgatherv_mpi_int2d
  module procedure xallgatherv_mpi_int
  module procedure xallgatherv_mpi_dp
  module procedure xallgatherv_mpi_dp2d
  module procedure xallgatherv_mpi_dp3d
  module procedure xallgatherv_mpi_dp4d
end interface xallgatherv_mpi

!----------------------------------------------------------------------

interface xalltoall_mpi
  module procedure xalltoall_mpi_dp2d
end interface xalltoall_mpi

!----------------------------------------------------------------------

interface xalltoallv_mpi
  module procedure xalltoallv_mpi_dp2d
  module procedure xalltoallv_mpi_int2d
  module procedure xalltoallv_mpi_dp1d
  module procedure xalltoallv_mpi_dp1d2
end interface xalltoallv_mpi

!----------------------------------------------------------------------

interface xcast_mpi
  module procedure xcast_mpi_intv
  module procedure xcast_mpi_int1d
  module procedure xcast_mpi_int2d
  module procedure xcast_mpi_int3d
  module procedure xcast_mpi_dpv
  module procedure xcast_mpi_dp1d
  module procedure xcast_mpi_dp2d
  module procedure xcast_mpi_dp3d
  module procedure xcast_mpi_dp4d
  module procedure xcast_mpi_spv
  module procedure xcast_mpi_sp1d
  module procedure xcast_mpi_sp2d
  module procedure xcast_mpi_sp3d
  module procedure xcast_mpi_sp4d
  module procedure xcast_mpi_cplxv
  module procedure xcast_mpi_cplx1d
  module procedure xcast_mpi_cplx2d
  module procedure xcast_mpi_cplx3d
  module procedure xcast_mpi_cplx4d
  module procedure xcast_mpi_dcv
  module procedure xcast_mpi_dc1d
  module procedure xcast_mpi_dc2d
  module procedure xcast_mpi_dc3d
  module procedure xcast_mpi_dc4d
  module procedure xcast_mpi_ch0d
  module procedure xcast_mpi_ch1d
end interface xcast_mpi

!----------------------------------------------------------------------

interface xexch_mpi
  module procedure xexch_mpi_intn
  module procedure xexch_mpi_int2d
  module procedure xexch_mpi_dpn
  module procedure xexch_mpi_dp2d
  module procedure xexch_mpi_dp3d
  module procedure xexch_mpi_dp4d_tag
  module procedure xexch_mpi_dp5d_tag
  module procedure xexch_mpi_spc_1d
  module procedure xexch_mpi_dpc_1d
  module procedure xexch_mpi_dpc_2d
end interface xexch_mpi

!----------------------------------------------------------------------

interface xmax_mpi
  module procedure xmax_mpi_intv
  module procedure xmax_mpi_dpv
end interface xmax_mpi

!----------------------------------------------------------------------

interface xmin_mpi
  module procedure xmin_mpi_intv
  module procedure xmin_mpi_dpv
end interface xmin_mpi

!----------------------------------------------------------------------

interface xrecv_mpi
  module procedure xrecv_mpi_intv
  module procedure xrecv_mpi_dp2d
  module procedure xrecv_mpi_dp3d
end interface xrecv_mpi

!----------------------------------------------------------------------

interface xsend_mpi
  module procedure xsend_mpi_intv
  module procedure xsend_mpi_dp2d
  module procedure xsend_mpi_dp3d
end interface xsend_mpi

!----------------------------------------------------------------------

interface xsum_master
  module procedure xsum_master_dp1d
  module procedure xsum_master_dp2d
  module procedure xsum_master_dp3d
  module procedure xsum_master_dp4d
  module procedure xsum_master_dp5d
  module procedure xsum_master_dp6d
  module procedure xsum_master_dp7d
  module procedure xsum_master_int4d
  module procedure xsum_master_c2cplx
  module procedure xsum_master_c3cplx
  module procedure xsum_master_c4cplx
  module procedure xsum_master_c1dpc
  module procedure xsum_master_c2dpc
  module procedure xsum_master_c3dpc
  module procedure xsum_master_c4dpc
  module procedure xsum_master_c5dpc
end interface xsum_master

!----------------------------------------------------------------------

interface xsum_mpi
  module procedure xsum_mpi_int
  module procedure xsum_mpi_intv
  module procedure xsum_mpi_intv2
  module procedure xsum_mpi_intn
  module procedure xsum_mpi_int2t
  module procedure xsum_mpi_int2d
  module procedure xsum_mpi_int3d
  module procedure xsum_mpi_int4d
  module procedure xsum_mpi_dp
  module procedure xsum_mpi_dpvt
  module procedure xsum_mpi_dpv
  module procedure xsum_mpi_dpn
  module procedure xsum_mpi_dp2d
  module procedure xsum_mpi_dp3d
  module procedure xsum_mpi_dp4d
  module procedure xsum_mpi_dp5d
  module procedure xsum_mpi_dp6d
  module procedure xsum_mpi_dp2t
  module procedure xsum_mpi_dp3d2t
  module procedure xsum_mpi_dp4d2t
  module procedure xsum_mpi_c0dc
  module procedure xsum_mpi_c1dc
  module procedure xsum_mpi_c2dc
  module procedure xsum_mpi_c3dc
  module procedure xsum_mpi_c4dc
  module procedure xsum_mpi_c5dc
  module procedure xsum_mpi_c1cplx
  module procedure xsum_mpi_c2cplx
  module procedure xsum_mpi_c3cplx
  module procedure xsum_mpi_c4cplx
  module procedure xsum_mpi_c5cplx
  module procedure xsum_mpi_log1d
  module procedure xsum_mpi_log2d
end interface xsum_mpi

!----------------------------------------------------------------------

CONTAINS  !===========================================================
!!***

!!****f* m_xmpi/xmpi_init
!! NAME
!!  xmpi_init
!!
!! FUNCTION
!!  Hides MPI_INIT from MPI library.
!!
!! INPUTS
!!  None
!!
!! PARENTS
!!      abinit,aim,anaddb,conducti,cut3d,fftprof,lwf,mrgddb,mrggkk,mrgscr,newsp
!!      optic,ujdet
!!
!! CHILDREN
!!      mpi_error_string
!!
!! SOURCE

subroutine xmpi_init()


 implicit none

!Local variables-------------------
 integer :: ierr
#if defined HAVE_MPI
 integer :: attribute_val
 logical :: lflag
#endif

! *************************************************************************

 ierr=0
#if defined HAVE_MPI
 call MPI_INIT(ierr)

 call MPI_ATTR_GET(MPI_COMM_WORLD, MPI_TAG_UB, attribute_val, lflag, ierr) ! Deprecated in MPI2
 !call MPI_Comm_get_attr(MPI_COMM_WORLD, MPI_TAG_UB, attribute_val, lflag, ierr)

 if (lflag) xmpi_tag_ub = attribute_val
 !write(*,*)"xmpi_tag_ub= ",xmpi_tag_ub
#endif

end subroutine xmpi_init
!!***

!----------------------------------------------------------------------

!!****f* m_xmpi/xmpi_end
!! NAME
!!  xmpi_end
!!
!! FUNCTION
!!  Hides MPI_FINALIZE from MPI library.
!!
!! INPUTS
!!  None
!!
!! PARENTS
!!      abinit,aim,anaddb,conducti,cut3d,fftprof,leave_new,lwf,mrgddb,mrggkk
!!      mrgscr,newsp,optic,ujdet
!!
!! CHILDREN
!!      mpi_error_string
!!
!! SOURCE

subroutine xmpi_end()


 implicit none

!Local variables-------------------
 integer :: ierr

! *************************************************************************

 ierr=0
#if defined HAVE_MPI
 call MPI_FINALIZE(ierr)
#endif

end subroutine xmpi_end
!!***

!----------------------------------------------------------------------

!!****f* m_xmpi/xcomm_rank
!! NAME
!!  xcomm_rank
!!
!! FUNCTION
!!  Hides MPI_COMM_RANK from MPI library.
!!
!! INPUTS
!!  spaceComm=MPI communicator.
!!
!! OUTPUT
!!  xcomm_rank=The rank of the node inside spaceComm
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function xcomm_rank(spaceComm)


 implicit none

!Arguments-------------------------
 integer,intent(in) :: spaceComm
 integer :: xcomm_rank

!Local variables-------------------
 integer :: ierr

! *************************************************************************

 ierr=0; xcomm_rank=0
#if defined HAVE_MPI
 if ( spaceComm/=MPI_COMM_NULL ) then
   call MPI_COMM_RANK(spaceComm,xcomm_rank,ierr)
 end if
#endif

end function xcomm_rank
!!***

!----------------------------------------------------------------------

!!****f* m_xmpi/xcomm_size
!! NAME
!!  xcomm_size
!!
!! FUNCTION
!!  Hides MPI_COMM_SIZE from MPI library.
!!
!! INPUTS
!!  spaceComm=MPI communicator.
!!
!! OUTPUT
!!  xcomm_size=The number of processors inside spaceComm.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function xcomm_size(spaceComm)


 implicit none

!Arguments-------------------------
 integer,intent(in) :: spaceComm
 integer :: xcomm_size 
 
!Local variables-------------------------------
!scalars
 integer :: ierr

! *************************************************************************

 ierr=0; xcomm_size=1
#if defined HAVE_MPI
 if ( spaceComm/=MPI_COMM_NULL ) then
   call MPI_COMM_SIZE(spaceComm,xcomm_size,ierr)
 end if
#endif

end function xcomm_size
!!***

!----------------------------------------------------------------------


!!****f* m_xmpi/xbarrier_mpi
!! NAME
!!  xbarrier_mpi
!!
!! FUNCTION
!!  Hides MPI_BARRIER from MPI library.
!!
!! INPUTS
!!
!! PARENTS
!!      atomden,calc_exch,calc_sigx_me,cohsex_me,datafordmft,debug_tools,denfgr
!!      dmft_solve,exc_iterative_diago,exceig,gw2wfk,haydock,ks_ddiago
!!      leave_new,m_coulombian,m_green,m_io_kss,m_io_screening,m_melemts
!!      m_screening,m_self,m_wfs,outkss,outwf,pawmkaewf,respfn,sigma,spectra
!!      tddft,trashme,vtorho,vtorhorec
!!
!! CHILDREN
!!      mpi_error_string
!!
!! SOURCE

subroutine xbarrier_mpi(spaceComm)


 implicit none

!Arguments-------------------------
 integer,intent(in) :: spaceComm

!Local variables-------------------
 integer   :: ier
#if defined HAVE_MPI
 integer :: nprocs
#endif

! *************************************************************************

 ier = 0
#if defined HAVE_MPI
 if (spaceComm /= MPI_COMM_NULL) then
   call MPI_COMM_SIZE(spaceComm,nprocs,ier)
   if(nprocs>1)then
     call MPI_BARRIER(spaceComm,ier)
   end if
 end if
#endif

end subroutine xbarrier_mpi
!!***

!----------------------------------------------------------------------

!!****f* m_xmpi/xmpi_name
!! NAME
!!  xmpi_name
!!
!! FUNCTION
!!  Hides MPI_GET_PROCESSOR_NAME from MPI library.
!!
!! OUTPUT
!!  name= the host name transformed to integer variable.
!!  ierr=Status error.
!!
!! PARENTS
!!      m_gpu_detect
!!
!! CHILDREN
!!      mpi_error_string
!!
!! SOURCE

subroutine xmpi_name(name_ch, ierr)


 implicit none

!Arguments-------------------------
 integer,intent(out) ::  ierr
 character(20),intent(out) :: name_ch

!Local variables-------------------
 integer :: name,len
! character*(MPI_MAX_PROCESSOR_NAME) :: name_ch

! *************************************************************************
!Get the name of this processor (usually the hostname)

 name = 0
 ierr = 0

#if defined HAVE_MPI
 call MPI_GET_PROCESSOR_NAME(name_ch, len, ierr)
 name_ch = trim(name_ch)

#else
 name_ch ='0'
#endif

end subroutine xmpi_name
!!***

!----------------------------------------------------------------------

!!****f* m_xmpi/xerror_string
!! NAME
!!  xerror_string
!!
!! FUNCTION
!!  Hides MPI_ERROR_STRING from MPI library.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      print_ierr
!!
!! CHILDREN
!!      mpi_error_string
!!
!! SOURCE

subroutine xerror_string(ierr,err_string,ilen,ierror)


 implicit none

!Arguments-------------------------
 integer,intent(in) :: ierr
 integer,intent(out) :: ilen,ierror
 character(len=*),intent(out) :: err_string

! *************************************************************************

 ilen=0
#if defined HAVE_MPI
 call MPI_Error_string(ierr,err_string,ilen,ierror)
#else
 ierror=1
 err_string="Sorry, no MPI_Error_string routine is available to interpret the error message"
#endif

end subroutine xerror_string
!!***

!----------------------------------------------------------------------

! Include files providing wrappers for some of the most commonly used MPI primitives.

#include "xallgather_mpi.F90"

#include "xallgatherv_mpi.F90"

#include "xalltoall_mpi.F90"

#include "xalltoallv_mpi.F90"

#include "xcast_mpi.F90"

#include "xexch_mpi.F90"

#include "xmax_mpi.F90"

#include "xmin_mpi.F90"

#include "xrecv_mpi.F90"

#include "xsend_mpi.F90"

#include "xsum_master.F90"

#include "xsum_mpi.F90"

END MODULE m_xmpi
!!***
