/* dev_spec.cu*/

/*
 * Copyright (C) 2008-2010 ABINIT Group (MMancini)
 *
 * This file is part of the ABINIT software package. For license information,
 * please see the COPYING file in the top-level directory of the ABINIT source
 * distribution.
 *
 */

#include <stdio.h>
#include <string.h>
/*=========================================================================*/
/*________________________ GPU_function called by HOST_____________________*/
/*=========================================================================*/

// display CUDA device info
static __host__ void  prt_dev_info()
{
  int deviceCount;
  cudaGetDeviceCount(&deviceCount);
  for (int dev = 0; dev < deviceCount; ++dev)
    {
      cudaDeviceProp deviceProp;
      cudaGetDeviceProperties(&deviceProp, dev);
      printf("\n___________________________________________________________________\n");
      printf(  "__________  Graphic Card Properties  ______________________________\n");
      printf("\n  Device %d: \"%s\"\n", dev, deviceProp.name);
      printf("  Revision number:                               %d.%d\n", deviceProp.major,deviceProp.minor);
      printf("  Total amount of global memory:                 %3.1f Mbytes\n", deviceProp.totalGlobalMem/1048576.);
      printf("  Clock rate:                                    %3.1f GHz\n", deviceProp.clockRate/1000000.);
      printf("  Max GFLOP:                                     %d GFP\n", 8*deviceProp.multiProcessorCount * deviceProp.clockRate/1000000);
      printf("  Total amount of constant memory:               %d bytes\n",(int) deviceProp.totalConstMem); 
      printf("  Total amount of shared memory per block:       %d bytes\n",(int) deviceProp.sharedMemPerBlock);
      printf("  Total number of registers available per block: %d\n", deviceProp.regsPerBlock);
      printf("___________________________________________________________________\n");

      if( (int) deviceProp.totalConstMem<0) break;
      //if(deviceProp.major==9999){printf("EXIT: PROBLEM WITH AVAILABLE DEVICES \n");exit(0);}
    }
}


__host__  void check_err(int line ){
  cudaError_t cudaError;
  cudaError = cudaGetLastError();
  if(cudaError != cudaSuccess)
    { fprintf(stderr, "CUDA Runtime API Error reported : %s %d\n", cudaGetErrorString(cudaError),line);
      exit(EXIT_FAILURE);
    }
  return;
}


// Gives the number of GPU devices ---------
static __host__ int get_gpu_ndev_c()
{
  int deviceCount;
  cudaGetDeviceCount(&deviceCount);
  return deviceCount;
}

// Gives the max mememory available for a GPU device ---------
static __host__ float get_gpu_max_mem_c(int device)
{
  cudaDeviceProp deviceProp;
  cudaGetDeviceProperties(&deviceProp, device);
  float a = (float) deviceProp.totalGlobalMem;
  return a;
}

// Set the device if it exists   -----------------
static __host__ void set_dev_c(int* device)
{
  int deviceCount;
  //  cudaError_t cudaError;
  cudaGetDeviceCount(&deviceCount);
  if(deviceCount>*device){
    cudaSetDevice(*device);
    /*     cudaError = cudaGetLastError(); */
    /*     fprintf(stderr, "CUDA Runtime API Error reported : %s\n", cudaGetErrorString(cudaError)); */
  }
  else *device=-1; 
  return;
}


// Unset the devices  -----------------
static __host__ void  unset_dev_c()
{
  cudaThreadExit();
  return;
}



/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*~~~~~~~~~~~~~~~~~~~~~~~~ INTERFACE WITH FORTRAN ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

/* extern "C"  */
/* void prt_gpu_info_() */
/* { */
/*   /\*____ display CUDA devices info ____*\/ */
/*   prt_dev_info(); */
/*   return; */
/* } */


extern "C" 
void get_gpu_ndev_(int* ndevice)
{
  /*____ give the number of devices  ____*/
  int bo = get_gpu_ndev_c();
  *ndevice = bo;
  return;
}


extern "C" 
void get_gpu_max_mem_(int* device, float* max_mem)
{
  /*____ give the max memory available on device ____*/
   float bo = get_gpu_max_mem_c(*device);
   *max_mem = bo;
   return;
}



extern "C" 
void set_dev_(int* gpudevice)
{
   if(*gpudevice >-1)
      set_dev_c(gpudevice);
  return;
}


extern "C" 
void unset_dev_()
{
  unset_dev_c();
  return;
}


extern "C"
void  get_dev_info_(int* device,
		    char* name,
		    int* lenname,
		    int vers[2], 
		    float* globalmem,
		    float* clockrate,
		    int* gflop,
		    int* constmem,
		    int* sharemem,
		    int* regist
		    )                          
{
  cudaDeviceProp deviceProp;
  cudaGetDeviceProperties(&deviceProp, *device);
  strcpy(name,deviceProp.name); 
  *lenname = strlen( name );
  vers[0] = deviceProp.major;
  vers[1] = deviceProp.minor;
  *globalmem = deviceProp.totalGlobalMem/1048576.;
  *clockrate = deviceProp.clockRate/1000000.;
  *gflop = 8*deviceProp.multiProcessorCount * deviceProp.clockRate/1000000;
  *constmem = deviceProp.totalConstMem; 
  *sharemem =  deviceProp.sharedMemPerBlock;
  *regist = deviceProp.regsPerBlock;
}


extern "C" 
void c_get_ndevice_(int* ndev)
{
  int deviceCount;
  cudaGetDeviceCount(&deviceCount);
  for (int idev = 0; idev < deviceCount; ++idev)
    {
      cudaDeviceProp deviceProp;
      cudaGetDeviceProperties(&deviceProp, idev);
      if(deviceProp.major>0 && deviceProp.major<99 &&
	 deviceProp.minor>0 && deviceProp.minor<99)
	{       
	*ndev+=1;
	}
    }
}


