!{\src2tex{textfont=tt}}
!!****f* ABINIT/m_bs_defs
!! NAME
!!  m_bs_defs
!!
!! FUNCTION
!!  This module defines basic structures used for Bethe-Salpeter calculations.
!!
!! COPYRIGHT
!! Copyright (C) 1992-2009 EXC group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida)
!! Copyright (C) 2009-2010 ABINIT group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida, M.Giantomassi)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_bs_defs

 use defs_basis 
 use m_errors 

 use m_io_tools,  only : get_unit

 implicit none

 private 
!!***

! Approximation for the excitonic Hamiltonian
!$ integer,public,parameter :: BSE_HRPA = 1   
!$ integer,public,parameter :: BSE_HGW  = 2
!$ integer,public,parameter :: BSE_HEXC = 3

! Treatment of the resontant and coupling blocks of the Hamiltonian
!$? integer,public,parameter :: BSE_HRPA = 1   
!$? integer,public,parameter :: BSE_HGW  = 2

!$
!$! Treatment of W(G1,G2)
!$ integer,public,parameter :: BSE_WGG_DIAGONAL=1
!$ integer,public,parameter :: BSE_WGG_FULL    =2

!$! Treatment of W(omega)
!$ integer,public,parameter :: BSE_WFREQ_STATIC=1
!$ integer,public,parameter :: BSE_WFREQ_PPM   =2
!$ integer,public,parameter :: BSE_WFREQ_FULL  =2

!----------------------------------------------------------------------

!!****t* m_bs_defs/excparam
!! NAME
!!  excparam
!!
!! FUNCTION
!!  The excparam derived data type contains the parameters controlling the BS calculation. 
!!
!! SOURCE

type,public :: excparam

!scalars
  logical :: exc               ! Exciton calculation.
  logical :: gw                ! Update energies using GW results.
  logical :: rpa               ! Use RPA expression for the polarizability.
  logical :: lf                ! Include local-field effects (bare coulomb term)
  logical :: nlf               ! Do NOT include local-field effects.
  logical :: resonant          ! Consider only the resonant part of the exciton Hamiltonian.
  logical :: coupling          ! Include off-diagonal terms coupling resonant and anti-resonant transitions.
  logical :: wdiag             ! Use diagonal approximation for W.
  logical :: wfull             ! Use full inverse dielectric matrix.
  logical :: use_haydock       ! Use haydock iterative method to calculate epsilon.
  !$logical :: use_cg          ! Use conjugate gradient minimization techniques to calculate the eigenstates.
  logical :: gwterm            ! Add QP corrections to the band structure either with scissors or GW file.
  logical :: exchangeterm      ! Include exchange terms in the BS Hamiltonian.
  logical :: coulombterm       ! Include W term in the BS Hamiltonian.
  logical :: soshift           ! Use shissor operator to open the gap
  logical :: inv               ! If true, use time reversal.

  character(len=3) :: type     ! Type of the calculation (RPA,GW,EXC)
  character(len=3) :: lftype   ! Local-fied type (LF,NLF)
  character(len=5) :: wtype    ! Treatment of W (WFULL,diago only)
  character(len=8) :: htype    ! Type of the Hamiltonian 

  integer :: npweps            ! No. of G in the Screening.
  integer :: npwwfn            ! No. of G for wave functions.
  !$integer :: npwx              ! No. of G for the exchange part.
  integer :: npwvec            ! MAX between npwwfn and npweps
  integer :: nsh               ! Number of shells corresponding to npwvec=MAX(npwwfn,npweps)
  integer :: nbnds             ! Total number of bands considered.
  integer :: nbndv             ! No. of valence states treated (homo-lomo+1)
  integer :: nbndc             ! No. of conduction states (humo-lumo+1)
  integer :: lomo,homo         ! Lowest and highest occupied orbital considered.
  integer :: lumo,humo         ! Lowest and highest unoccupied orbital considered.
  integer :: niter             ! No. of iterations for Haydock.
  integer :: nkibz, nkbz       ! No. of k-points in the IBZ and BZ (resp.)
  integer :: nqbz       ! No. of q-points in the IBZ and BZ (resp.)
  integer :: nopk, ninvk       ! 
  integer :: nop, ninv         ! No. of symmetry operations, ninv is (1,2) depeding if time reversal cannot or can be used.
  integer :: nomega            ! No. of frequencies for epsilon.
  integer :: nh                ! Rank of the exciton matrix.
  integer :: nq                ! Number of "small" q for optical limit.

  real(dp) :: nel              ! No. of electrons per unit cell.
  real(dp) :: soenergy         ! Scissors energy
  real(dp) :: omegai           ! First omega for epsilon.
  real(dp) :: omegae           ! Last omega for epsilon (defaults to 10eV)
  real(dp) :: domega           ! Step of the frequency mesh.
  real(dp) :: broad            ! Lorentzian Broadening.
  real(dp) :: ircut            ! Infrared cutoff for transitions
  real(dp) :: uvcut            ! Ultraviolete cutoff for transitions.
  real(dp) :: stripecut        ! Cutoff on (t,t") transitions.
  real(dp) :: haydock_tol      ! Tolerance for stopping the Haydock algorithm.

!arrays
  integer :: mG0(3)            ! For each reduced direction gives the max G0 component
                               ! to account for umklapp processes
  integer,pointer :: shlim(:)   SET2NULL

  real(dp) :: q(3,6)                ! Q-point for optical limit.
  real(dp) :: qcc(3,6)
  real(dp) :: qcc2pioa(3,6)

  complex(dpc),pointer :: omega(:)  SET2NULL
  ! Frequency mesh for epsilon.

end type excparam
!!***

 public :: nullify_bs_parameters 
 public :: destroy_bs_parameters
 public :: print_bs_parameters

!!****t* m_bs_defs/excfiles
!! NAME
!!  excfiles
!!
!! FUNCTION
!!  The excfiles derived data type contains file names and unit numbers used to store 
!!  temporary or final results of the Bethe-Salpeer calculation. 
!!
!! SOURCE

type,public :: excfiles

  logical :: hx   ! existence of the files
  logical :: cx   ! existence of the files
  logical :: ox   ! existence of the files

  character(len=fnlen) :: exh     ! Name of the file with the excitonic resonant part of the Hamiltonian (Hermitian).
  character(len=fnlen) :: exc     ! Name of the file with the coupling part of the Hamiltonian (symmetric).
  character(len=fnlen) :: exovl   ! Name of the file containing the overlap of the eigenvectors.
  !TODO the overlap should be contained in the BS_EIG file.

  logical :: in_eig_exists         ! .TRUE. if the bseig_in file exists (see setup_bethe_salpeter.F90)
  character(len=fnlen) :: out_eig  ! Name of the output file with the eigenvalues and the eigenvectors of the Hamiltonian.
  character(len=fnlen) :: in_eig   ! Name of the input file with the eigenvalues and the eigenvectors of the Hamiltonian.

end type excfiles
!!***

public :: print_bs_files    ! Printout of the excfiles data type.


!!****t* m_bs_defs/transition
!! NAME
!!  transition
!!
!! FUNCTION
!!  The transition derived data type is used to store the correspondence 
!!  between the transition index and the set of quantum numbers (ik_bz,v,c) 
!!  The energy of the transition is stored as well.
!!
!! SOURCE                                                                                   

type,public :: transition
 integer :: k   ! Index of the k-point in the BZ
 integer :: v   ! Valence band index.
 integer :: c   ! Conduction band index.
 !integer :: s  ! Spin of the transition (s1,s2) for spinor?
 complex(dpc) :: en  ! Transition energy 
end type transition

 public :: init_transitions     ! Creation method.
 public :: write_transitions    ! Dump the structure on file.
 public :: repr_trans           ! Returns a string representing the transition or a couple of transitions.
 !public :: repr_1trans           ! Returns a string representing a single transition.
 !public :: repr_2trans           ! Returns a string representing two transitions.

 interface repr_trans 
  module procedure repr_1trans
  module procedure repr_2trans
 end interface repr_trans 

CONTAINS  !========================================================================================================
!!***

!!****f* m_bs_defs/nullify_bs_parameters
!! NAME
!!  nullify_bs_parameters
!!
!! FUNCTION
!!  Set all pointers in a structure of type excparam to NULL.
!!
!! SIDE EFFECTS
!!  Bsp<excparam>=All pointers are initialized to NULL.
!!
!! PARENTS
!!      setup_bse
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine nullify_bs_parameters(BSp)
    
 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(excparam),intent(inout) :: BSp

!************************************************************************

 !@excparam
 nullify(Bsp%shlim)
 nullify(Bsp%omega)

end subroutine nullify_bs_parameters
!!***

!----------------------------------------------------------------------

!!****f* m_bs_defs/destroy_bs_parameters
!! NAME
!!  destroy_bs_parameters
!!
!! FUNCTION
!!  Free all memory allocated in a structure of type excparam
!!
!! SIDE EFFECTS
!!  Bsp<excparam>=All associated pointers are deallocated.
!!
!! PARENTS
!!      bethe_salpeter
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine destroy_bs_parameters(BSp)
    
 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(excparam),intent(inout) :: BSp

!************************************************************************
 
 !@excparam
 if (associated(Bsp%shlim)) deallocate(Bsp%shlim)
 if (associated(Bsp%omega)) deallocate(Bsp%omega) 

end subroutine destroy_bs_parameters
!!***

!----------------------------------------------------------------------

!!****f* m_bs_defs/print_bs_parameters
!! NAME
!!  print_bs_parameters
!!
!! FUNCTION
!!  Printout of the parameters used for the BS calculation.
!!
!! INPUTS
!!  p<excparam>=Datatype storing the parameters of the Bethe-Salpeter calculation.
!!
!! OUTPUT
!!  Only printing.
!!
!! PARENTS
!!      setup_bse
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine print_bs_parameters(BSp,header,unit,mode_paral,prtvol) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: unit,prtvol
 character(len=4),optional,intent(in) :: mode_paral 
 character(len=*),optional,intent(in) :: header
 type(excparam),intent(inout) :: BSp

!Local variables ------------------------------
!scalars
 integer :: my_unt,my_prtvol,iq,ii
 logical :: linv,linvk
 character(len=4) :: my_mode
 character(len=500) :: msg      

! ********************************************************************* 

 my_unt   =std_out; if (PRESENT(unit      )) my_unt   =unit
 my_prtvol=0      ; if (PRESENT(prtvol    )) my_prtvol=prtvol 
 my_mode  ='COLL' ; if (PRESENT(mode_paral)) my_mode  =mode_paral

 msg=' ==== Parameters for the Bethe-Salpeter  ==== '
 if (PRESENT(header)) msg=' ==== '//TRIM(ADJUSTL(header))//' ==== '
 call wrtout(my_unt,msg,my_mode)

 if (BSp%ninv == 1) then
   linv = .false.
 else
   BSp%ninv = 2
   linv = .true.
 end if
      
 if (BSp%ninvk == 1) then
   linvk = .false.
 else
   BSp%ninvk = 2
   linvk = .true.
 end if

 write(msg,'(6(a,i5,a),a,i5)')&
&  ' dimension of the matrices,     npweps = ',BSp%npweps,ch10,&
&  ' cut-off on the wavefunctions,  npwwfn = ',BSp%npwwfn,ch10,&
&  ' highest empty band included,   nbands = ',BSp%nbnds,ch10, &
&  ' lowest occupied band included, lomo   = ',BSp%lomo,ch10,  &
&  ' number of k-points in IBZ,     nkibz  = ',BSp%nkibz,ch10, &
&  ' number of symmetry operations, nsymk  = ',BSp%nopk,ch10,&
&  ' number of symmetry operations, nsym   = ',BSp%nop
 call wrtout(my_unt,msg,my_mode)

 write(msg,'(2(a,l5,a),3(a,f6.2,a),a,f6.2)')&
&  ' inversion,                     invk   = ',linvk,ch10,&
&  ' inversion,                     inv    = ',linv,ch10,&
!&  ' polarization vector (r.l.u.)   polvec = ',BSp%qpg,ch10,&
&  ' minimum energy [eV]            Emin   = ',BSp%omegai*Ha_eV,ch10,&
&  ' maximum energy [eV]            Emax   = ',BSp%omegae*Ha_eV,ch10,&
&  ' energy step [eV]               dE     = ',BSp%domega*Ha_eV,ch10,&
&  ' lorentzian broadening [eV]     eta    = ',BSp%broad*Ha_eV
 call wrtout(my_unt,msg,my_mode)

 write(msg,'(5(a,i5,a),a,i5)')&
&  ' lowest occupied state                   ',BSp%lomo,ch10,&
&  ' highest occupied state                  ',BSp%homo,ch10,&
&  ' lowest unoccupied state                 ',BSp%lumo,ch10,&
&  ' highest unoccupied state                ',BSp%nbnds,ch10,&
&  ' number of valence bands                 ',BSp%nbndv,ch10,&
&  ' number of conduction bands              ',BSp%nbndc
 call wrtout(my_unt,msg,my_mode)

 msg=' KS energies'
 if (BSp%GWTERM) msg=' GW corrections included'
 call wrtout(my_unt,msg,my_mode)

 msg=' Local fields effects (v) excluded'
 if (BSp%EXCHANGETERM) msg=' Local fields effects (v) included'
 call wrtout(my_unt,msg,my_mode)

 msg=' Excitonic effects (W) excluded'
 if (BSp%COULOMBTERM) msg=' Excitonic effects (W) included'
 call wrtout(my_unt,msg,my_mode)

 if (BSp%COUPLING) then
   msg=' Coupling term included'
 else
   msg=' Resonant term only'
 end if
 call wrtout(my_unt,msg,my_mode)

 msg=' Full W included'
 if (BSp%WDIAG) msg=' Only W diagonal included'
 call wrtout(my_unt,msg,my_mode)

 write(msg,'(a)')ch10
 call wrtout(my_unt,msg,my_mode)

 msg = ' Calculating epsilon_Macro(q-->0,w), along the following polarization directions:' 
 call wrtout(my_unt,msg,my_mode)

 do iq=1,BSp%nq
   write(msg,'(3(a,3f10.6,2a),a)')&
&    ' q = (',(BSp%q(ii,iq),ii=1,3),  ') [r.l.u.]',ch10,&
&    ' q = (',(BSp%qcc(ii,iq),ii=1,3),') [c.c.]',ch10,&
&    ' q = (',(BSp%qcc2pioa(ii,iq),ii=1,3),') ',ch10
   call wrtout(my_unt,msg,my_mode)
 end do

end subroutine print_bs_parameters
!!***

!----------------------------------------------------------------------

!!****f* m_bs_defs/init_transitions
!! NAME
!!  init_transitions
!!
!! FUNCTION
!!  Main creation method for the transition structured datatype.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      setup_bse
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine init_transitions(BSp,nsppol,gw_energy,ktab,stripewidth,trans,transtab,gw_gap,en_order)                  
                                                                        
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsppol
 integer,intent(out) :: stripewidth
 complex(dpc),intent(out) :: gw_gap
 logical,intent(in) :: en_order
 type(excparam),intent(inout) :: BSp
!arrays
 integer,intent(in) :: ktab(BSp%nkbz) 
 integer,intent(out) :: transtab(BSp%nkbz,BSp%lomo:BSp%homo,BSp%lumo:BSp%humo) 
 complex(dpc),intent(in) :: gw_energy(BSp%nbnds,BSp%nkibz,nsppol)
 type(transition),intent(out) :: trans(BSp%nkbz*BSp%nbndv*BSp%nbndc) 

!Local variables ------------------------------
!scalars
 integer :: it,jt,ik_bz,ik_ibz,iv,ic,stripewidthtemp 
 real(dp) :: trans_energy, trendiff 
 complex(dpc) :: tene
 logical :: DEBUG
 character(len=500) :: msg
 type(transition) :: transtemp 

!************************************************************************
      
 ABI_CHECK(BSp%nbndv==BSp%homo-BSp%lomo+1,"Wrong nbndv")
 ABI_CHECK(BSp%nbndc==BSp%humo-BSp%lumo+1,"Wrong nbndc")
 ABI_CHECK(nsppol==1,"nsppol==2 not coded")

! Find transitions                                                  
 gw_gap=gw_energy(BSp%humo,1,1)-gw_energy(BSp%lomo,1,1) 

 it=0
 do ik_bz=1,BSp%nkbz 
   ik_ibz=ktab(ik_bz) 

   do iv=BSp%lomo,BSp%homo
     do ic=BSp%lumo,BSp%humo
       tene = gw_energy(ic,ik_ibz,1)-gw_energy(iv,ik_ibz,1)  ! TODO SPIN SUPPORT is missing
       trans_energy = DBLE(tene)

       if (trans_energy < BSp%uvcut .and. trans_energy > BSp%ircut) then 
         it = it + 1 
         trans(it)%k  = ik_bz 
         trans(it)%v  = iv 
         trans(it)%c  = ic 
         trans(it)%en = tene
         if (DBLE(tene) < DBLE(gw_gap)) gw_gap = tene
       end if 
     end do 
   end do 

 end do ! ik_bz

 BSp%nh = it 
 write(msg,'(a,i0)')' Number of transitions found: ',BSp%nh 
 call wrtout(std_out,msg,"COLL")

 ! TODO This part should be removed, not optimal and makes parallelism more complicated.
 if (en_order) then
   msg=' Putting transitions in energy ascending order'
   call wrtout(std_out,msg,"COLL")
   do it=BSp%nh,2,-1 
     do jt=1,it-1 
       if (DBLE(trans(jt)%en) > DBLE(trans(jt+1)%en)) then  
         ! WARNING this coding doesn"t work if transtemp contains pointer.
         transtemp   = trans(jt) 
         trans(jt  ) = trans(jt+1) 
         trans(jt+1) = transtemp 
       end if 
     end do 
   end do
 else
   msg=' Transitions are in c-v-k order (k slower)'                                                                        
   call wrtout(std_out,msg,"COLL")
 end if 
                                                                        
 ! Create transition table transtab                                  
 transtab(:,:,:) = 0 
 do it=1,BSp%nh 
   transtab(trans(it)%k,trans(it)%v,trans(it)%c) = it
 end do 
                                                                        
 ! Write transitions and found stripe width                          
 DEBUG = .FALSE.
 if (BSp%nh > 4000) DEBUG = .FALSE. 
 if (DEBUG) then 
   write(*,'(" trans   k   v   c  energy [eV]  stripewidth")')
 end if 
                                                                        
 do it=1,BSp%nh 

   do jt=it,BSp%nh 
     trendiff = trans(jt)%en - trans(it)%en 
     if (trendiff <= BSp%stripecut) then 
       stripewidthtemp = jt-it 
       if (stripewidthtemp > stripewidth) stripewidth = stripewidthtemp           
     end if 
   end do 
                                                                         
   if (DEBUG) then 
     write(*,110) transtab(trans(it)%k,trans(it)%v,trans(it)%c),&
&                 trans(it)%k, trans(it)%v, trans(it)%c,&
&                 trans(it)%en*Ha_eV, stripewidthtemp              
110  format(i6,3i4,2f12.2,7x,i6) 
   end if 
 end do 

 ! The stripewidth refers to the number of diagonals on the right (or of the principal).
 write(msg,'(a,i0)')' Dimension of the stripe ', stripewidth 
 call wrtout(std_out,msg,"COLL")
 write(msg,'(a,2f6.2)')" GW  direct gap     [eV] ",gw_gap*Ha_eV
 call wrtout(std_out,msg,"COLL")

end subroutine init_transitions                                           
!!***

!----------------------------------------------------------------------

!!****f* m_bs_defs/write_transitions
!! NAME
!!  write_transitions
!!
!! FUNCTION
!!  Dump the content of the transition structured datatype on an external file.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine write_transitions(Trans,BSp,nsppol,gw_energy,kbz,ktab)
 
 use defs_basis

 implicit none 

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsppol
 type(excparam),intent(in) :: BSp
!arrays
 integer,intent(in) :: ktab(BSp%nkbz) 
 real(dp),intent(in) :: kbz(3,BSp%nkbz) 
 complex(dpc),intent(in) :: gw_energy(BSp%nbnds,BSp%nkibz,nsppol)
 type(transition),intent(in) :: trans(BSp%nh) 

!Local variables ------------------------------
!scalars
 integer :: it,ik_bz,ik_ibz,iv,ic,unt
 real(dp) :: trans_energy 
! character(len=500) :: msg

!************************************************************************

 !msg=' Writing transitions and electronic structure on file' 
 !call wrtout(std_out,msg,"COLL")

 ABI_CHECK(nsppol==1,"nsppol==2 not coded")

 unt= get_unit()
 open(unt,file='out.trans',form='unformatted') 

 write(unt) BSp%nh 
 do it = 1, BSp%nh 
   ik_bz = trans(it)%k 
   iv    = trans(it)%v 
   ic    = trans(it)%c 
   trans_energy = trans(it)%en 
   write(unt) ik_bz, iv, ic, trans_energy 
 end do 

 write(unt) BSp%nkbz, BSp%lomo, BSp%homo, BSp%nbnds 

 do ik_bz=1,BSp%nkbz 
   ik_ibz = ktab(ik_bz) 
   write(unt) kbz(:,ik_bz),DBLE(gw_energy(ik_ibz,1:BSp%nbnds,1))
 end do 
 close(unt) 

end subroutine write_transitions
!!***

!----------------------------------------------------------------------

!!****f* m_bs_defs/repr_1trans
!! NAME
!!  repr_1trans
!!
!! FUNCTION
!!  Returns a string with info on the (k,v,c) transition.
!!
!! INPUTS
!!  Trans<transition>=structure datatype containing indececes and info on the optical transition.
!!  [prtvol]=Verbosity level. Defaults to 0.
!!
!! OUTPUT
!!  str(len=500)=The string representing the transition.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function repr_1trans(Trans,prtvol) result(str)
 
 use defs_basis

 implicit none 

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: prtvol
 character(len=500) :: str
 type(transition),intent(in) :: Trans 

!Local variables ------------------------------
!scalars
 integer :: my_prtvol

!************************************************************************

 my_prtvol=0; if (PRESENT(prtvol)) my_prtvol=prtvol

 if (my_prtvol==0) then
   write(str,'(3(a,i3))')" k= ",Trans%k," v= ",Trans%v," c= ",Trans%c
 else 
   write(str,'(3(a,i3),a,2f6.2)')" k= ",Trans%k," v= ",Trans%v," c= ",Trans%c," ene= ",Trans%en*Ha_eV
 end if

end function repr_1trans
!!***

!----------------------------------------------------------------------

!!****f* m_bs_defs/repr_2trans
!! NAME
!!  repr_2trans
!!
!! FUNCTION
!!  Returns a string with info on two transitions
!!
!! INPUTS
!!  Trans1, Trans2<transition>=structure datatypes containing indececes and info on the optical transitions
!!  [prtvol]=Verbosity level. Defaults to 0.
!!
!! OUTPUT
!!  str(len=500)=The string representing the transition.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function repr_2trans(Trans1,Trans2,prtvol) result(str)
 
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none 

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: prtvol
 character(len=500) :: str
 type(transition),intent(in) :: Trans1,Trans2

!Local variables ------------------------------
!scalars
 integer :: my_prtvol

!************************************************************************

 my_prtvol=0; if (PRESENT(prtvol)) my_prtvol=prtvol

 str = repr_1trans(Trans1,my_prtvol)

 str = TRIM(str)//"|"//TRIM(repr_1trans(Trans2,my_prtvol))

end function repr_2trans
!!***

!----------------------------------------------------------------------

!!****f* m_bs_defs/print_bs_files
!! NAME
!!  print_bs_files
!!
!! FUNCTION
!!  Printout of the content of the excfiles structure.
!!
!! INPUTS
!!  BS_files<excfiles>=An object of type excfile storing the filenames used in the Bethe-Salpeter code.
!!  [unit]=Unit number for output
!!  [prtvol]=Verbosity level
!!  [mode_paral]=Either "COLL" or "PERS"
!!  [header]=String to be printed as header for additional info.
!!
!! OUTPUT
!!  Only printing.
!!
!! PARENTS
!!      setup_bse
!!
!! CHILDREN
!!      wrtout
!!
!! SOURCE

subroutine print_bs_files(BS_files,header,unit,mode_paral,prtvol)                                   
 
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none 

!Arguments ------------------------------------
!scalars
 type(excfiles),intent(in) :: BS_files
 integer,optional,intent(in) :: unit,prtvol
 character(len=4),optional,intent(in) :: mode_paral 
 character(len=*),optional,intent(in) :: header
!arrays

!Local variables ------------------------------
!scalars
 integer :: my_unt,my_prtvol
 character(len=4) :: my_mode
 character(len=500) :: msg      
! ********************************************************************* 

 !@excfiles
 my_unt   =std_out; if (PRESENT(unit      )) my_unt   =unit
 my_prtvol=0      ; if (PRESENT(prtvol    )) my_prtvol=prtvol 
 my_mode  ='COLL' ; if (PRESENT(mode_paral)) my_mode  =mode_paral
                                                                    
 msg=' ==== Files used for the Bethe-Salpeter calculation  ==== '
 if (PRESENT(header)) msg=' ==== '//TRIM(ADJUSTL(header))//' ==== '
 call wrtout(my_unt,msg,my_mode)

 !TODO this routine is a stub, sill working on file handling.
 msg = " Output BS_EIG file: "//BS_files%out_eig
 call wrtout(my_unt,msg,my_mode)
 if (BS_files%in_eig_exists) then 
   msg = "Will read BS_EIG from: "//BS_files%in_eig
   call wrtout(my_unt,msg,my_mode)
 end if

end subroutine print_bs_files
!!***

!----------------------------------------------------------------------

END MODULE m_bs_defs
!!***
