!{\src2tex{textfont=tt}}
!!****m* ABINIT/defs_elphon
!!
!! NAME
!! defs_elphon
!!
!! FUNCTION
!! This module contains the datastructures for elphon
!!  the different (huge) matrices will either be allocated and
!!  used, or be written to disk. All combinations should be feasible.
!!
!! COPYRIGHT
!! Copyright (C) 2004-2010 ABINIT group (MVer, MG)
!! This file is distributed under the terms of the
!! GNU General Public Licence, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! NOTES
!!  Contains two datastructures:
!!   elph_type contains data and dimensions for the kpoints near the
!!    fermi surface and the $g_{k k+q}$ matrix elements
!!   phon_type contains the necessary data to interpolate the
!!    phonon bandstructure and eigenvectors in reciprocal space (ie.
!!    interatomic force constants and corresponding real space grid info).
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

module defs_elphon

  use defs_basis
  use m_kptrank

  implicit none

!!***

!----------------------------------------------------------------------
!!****t* defs_elphon/elph_kgrid_type
!! NAME
!! elph_kgrid_type
!!
!! FUNCTION
!! elph_kgrid_type contains k-point grid data and dimensions
!!  this is a sub object of elph_type
!!
!! SOURCE

  type elph_kgrid_type

   integer :: nkpt                            ! number of k-points in full grid
   integer :: nkptirr                         ! number of k-points in irreducible grid

   type(kptrank_type) :: kptrank_t            ! ranking of all kpoints on phonon calculation grid, and inverse rank

   integer, pointer :: irr2full(:)      ! correspondence of irred kpoints to a full one
   integer, pointer :: full2irr(:,:)    ! correspondence of full k to one irred kpoints through sym and timrev
   integer, pointer :: full2full(:,:,:) ! symmetry mapping of kpoints 

   real(dp),pointer :: kpt(:,:)               ! coordinates of the full kpoints from phonon calculation
   real(dp),pointer :: kptirr(:,:)            ! irreducible k-points, for preliminary set up
   real(dp),pointer :: wtk(:,:,:)             ! integration weights (see also gkk_intweight)
   real(dp),pointer :: wtkirr(:)              ! weights for irreducible kpoints, to sum over _whole_ BZ (not just Fermi Surface)
  end type elph_kgrid_type
!!***


!!****t* defs_elphon/elph_type
!! NAME
!! elph_type
!!
!! FUNCTION
!! elph_type contains data and dimensions for the kpoints near the
!! fermi surface and the $g_{k k+q}$ matrix elements
!!
!! SOURCE

  type elph_type
    
   type(elph_kgrid_type) :: k_phon               ! object for k-grid of phonon calculation
   type(elph_kgrid_type) :: k_fine               ! object for fine k-grid for FS integration

   integer :: natom,nbranch,nFSband 
   integer :: minFSband,maxFSband                !Index of lower and upper bands used for the FS integration

   integer :: ngkkband                           !Number of bands kept in final gkk matrix elements:
                                                 !either 1 if sum is performed immediately
                                                 !or = nband if all elements are kept based on flag ep_keepbands

   integer :: nqpt_full                          !number of q in full BZ
   integer :: nqptirred                          !number of irred q-points


   integer :: unita2f,unit_gkk2,unit_gkk_rpt
   integer :: unitgkq                            !units for file output
     
   integer :: gkqwrite                              
   integer :: gkk2write
   integer :: gkk_rptwrite

   integer :: ep_scalprod                        !flag to perform the scalar product
   integer :: symgkq                             !flag to symmetrize gkq matrix elements
   integer :: ep_keepbands                       !flag to sum over bands or not
   integer :: tuniformgrid                       !flag to expect uniform grid of q or not

   
   integer :: na2f                               !dimensions and increments for a2F function
   integer :: nsppol                             ! number of spin polarization channels
   integer :: nspinor                            ! number of spinorial components
   real(dp) :: omega_min,omega_max 
   real(dp) :: a2fsmear,domega
   real(dp) :: occ_factor                        ! normalization for integrals over FS, for num of spins, spinors, etc...

   real(dp) :: mustar                            !mustar parameter
   real(dp) :: fermie                            !Fermi energy (Ha), either comes from wfk file or from anaddb input file

   character(len=fnlen) :: elph_base_name        !base name for output files

   integer,pointer :: qirredtofull(:)            !mapping between the qpoints found in the GGK file
                                                 !and the array of qpoints generated by the code

   real(dp),pointer :: wtq(:)                    !weight for each qpoint in the full grid spqt
                                                 !if a point is not in the IBZ ==>  wtq=0 
                                                 !MG we can also use indqpt

   real(dp),pointer :: n0(:)                     !DOS at the Fermi level (states/Ha/spin)
   real(dp),pointer :: qpt_full(:,:)             !special q points obtained by the Monkhorst & Pack method,
                                                 !in reduced coordinates


   real(dp),pointer :: gkk_intweight(:,:,:)      ! (nFSband,nkpt_fine,nsppol)
                                                 !integration weights for gkk matrix elements on FS:
                                                 !if ep_keepbands == 0 all are 1
                                                 !if ep_keepbands == 1 then = to wtk_phon in elphon
                                                 !DOES NOT INCLUDE FACTOR OF 1/nkpt_phon

   real(dp),pointer :: gkk_qpt(:,:,:,:,:,:)      ! (2, ngkkband*ngkkband, nbranch*nbranch, nkpt_phon, nsppol, nqptirred)
                                                 !Now gkq contains gkk2 matrices on basic qpts, 
                                                 !summed over bands if ngkkband==1


   real(dp),pointer :: gkk_rpt(:,:,:,:,:,:)      ! (2, ngkkband**2, nbranch**2, nkpt_phon, nsppol, nrpt)
                                                 !For the moment, gkk_rpt in memory is out of the question
   real(dp),pointer :: gkk2(:,:,:,:,:,:)         ! (nbranch, ngkkband,ngkkband, nkpt_phon, nkpt_phon, nsppol)

   real(dp),pointer :: gamma_qpt(:,:,:,:)        !gamma matrices integrated over kpoint coeff
                                                 !  and bands: still depends on qpt
                                                 ! dims= 2, nbranch**2, nsppol, nqpt
   real(dp),pointer :: gamma_rpt(:,:,:,:)
                                                 ! dims= 2, nbranch**2, nsppol, nrpt
!NOTE: choice to put nsppol before or after nqpt is a bit arbitrary
!   abinit uses nband,nkpt,nsppol, but here for convenience nkpt_phon,nsppol,nqpt 
!   as interpolation is on qpt

   real(dp),pointer :: phfrq(:,:)                !phonon frequencies
   real(dp),pointer :: a2f(:,:,:)                !a2f function

   real(dp),pointer :: qgrid_data(:,:,:,:)       !e-ph values calculated over the irreducible part of the q-grid:
                                                 !first entry  =  index of the q-point,
                                                 !second index =  branch index
                                                 !the third slice contains the frequency, the linewidth and lambda(q,nu)
                                                 !for that particular phonon mode
                                                 ! dims= nqptirred,elph_ds%nbranch,nsppol,3 

  end type elph_type
!!***

!----------------------------------------------------------------------

!!****t* defs_elphon/phon_type
!! NAME
!! phon_type
!!
!! FUNCTION
!! phon_type contains the necessary data to interpolate the
!! phonon bandstructure and eigenvectors in reciprocal space 
!! (ie. interatomic force constants and corresponding real space grid info).
!!
!! SOURCE

 type phon_type

  integer :: dipdip
  integer :: mpert,nsym
  integer :: natom,ntypat,nrpt
  integer,pointer :: indsym(:,:,:),symrel(:,:,:),typat(:)
  real(dp) :: ucvol
  real(dp),pointer :: acell(:),amu(:),atmfrc(:,:,:,:,:,:)
  real(dp),pointer :: dielt(:,:),dyewq0(:,:,:),gprim(:,:),gmet(:,:)
  real(dp),pointer :: xred(:,:),zeff(:,:,:)
  real(dp),pointer :: rcan(:,:),rmet(:,:),rprim(:,:),rprimd(:,:),rpt(:,:)
  real(dp),pointer :: trans(:,:),wghatm(:,:,:)

 end type phon_type

!!***

!----------------------------------------------------------------------


!!****t* defs_elphon/elph_tr_type
!! NAME
!! phon_type
!!
!! FUNCTION
!! elph_tr_ds contains the necessary data for the transport properties
!!
!! SOURCE

  type elph_tr_type

     integer :: ifltransport
     integer :: unitgkq_trin,unitgkq_trout
     integer :: gkqwrite,gkqexist
     integer :: onegkksize
     character(len=fnlen) :: ddkfilename
     real(dp),pointer :: el_veloc(:,:,:,:)        ! nkpt nband 3 nsppol
! the 9 = 3x3 is for the full tensorial transport coefficients
     real(dp),pointer :: eta_trin(:,:,:,:,:)      ! 9 bands**2 nFSkpt nsppol qpt 
     real(dp),pointer :: eta_trout(:,:,:,:,:) 

     real(dp),pointer :: gamma_qpt_tr(:,:,:,:,:)    ! 2 9 branches**2 nsppol qpt
     real(dp),pointer :: gamma_qpt_trin(:,:,:,:,:)  !idem
     real(dp),pointer :: gamma_qpt_trout(:,:,:,:,:) !idem

     real(dp),pointer :: gamma_rpt_tr(:,:,:,:,:)    !idem
     real(dp),pointer :: gamma_rpt_trin(:,:,:,:,:)  !idem
     real(dp),pointer :: gamma_rpt_trout(:,:,:,:,:) !idem

     real(dp),pointer :: a2f_1d_tr(:,:,:)           ! nfreq 9 nsppol
     real(dp),pointer :: a2f_1d_trin(:,:,:)
     real(dp),pointer :: a2f_1d_trout(:,:,:)

     real(dp),pointer ::  FSelecveloc_sq(:,:)       ! 3 nsppol



  end type elph_tr_type

end module defs_elphon
!!***
