!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_header
!! NAME
!! m_header
!!
!! FUNCTION
!! This module contains the definition of the abinit header (TODO)
!! and methods acting on the data type.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2010 ABINIT group (XG, MG)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_header

 use defs_basis
 use m_errors

 use m_copy,          only : deep_copy
 use defs_datatypes,  only : bandstructure_type, pseudopotential_type, pawtab_type
 use defs_abitypes,   only : hdr_type, dataset_type

 implicit none

 private

 public :: hdr_init                ! Initializes the header structured datatype and most of its content from dtset and psps.
 public :: hdr_nullify             ! Set all pointers to null.
 public :: hdr_clean               ! Deallocates the components of the header.
 public :: hdr_copy                ! Deep copy of the Header.
 public :: hdr_get_nelect_byocc    ! Returns the number of electrons calculated from Hdr%occ
 public :: isknown_headform        ! Returns .TRUE. is headform is in HDR_KNOWN_HEADFORMS.

 integer,public ,parameter :: HDR_NHEADFORMS=9
 ! Number of abinit headform used so far.

 integer,public ,parameter :: HDR_KNOWN_HEADFORMS( HDR_NHEADFORMS ) = (/23,34,40,41,42,44,53,56,57/)
 ! The list of abinit headform used so far.

 integer,public ,parameter :: HDR_LATEST_HEADFORM = HDR_KNOWN_HEADFORMS( HDR_NHEADFORMS )
 ! The latest headform to be used for writing.

CONTAINS  !===========================================================
!!***

!!****f* m_header/hdr_init
!! NAME
!! hdr_init
!!
!! FUNCTION
!! This subroutine initializes the header structured datatype
!! and most of its content from dtset and psps, and put default values for
!! evolving variables.
!!
!! INPUTS
!! bstruct <type(bandstructure_type)>=band structure information
!!  including Brillouin zone description
!! codvsn=code version
!! dtset <type(dataset_type)>=all input variables for this dataset
!! pawtab(ntypat*usepaw) <type(pawtab_type)>=paw tabulated starting data
!! pertcase=index of the perturbation, or 0 if GS calculation
!! psps <type(pseudopotential_type)>=all the information about psps
!!
!! OUTPUT
!! hdr <type(hdr_type)>=the header, initialized, and for most part of
!!   it, contain its definite values, except for evolving variables
!!
!! PARENTS
!!      bloch_interp,gstate,gstateimg,gw2wfk,loper3,newsp,nonlinear,respfn
!!      setup_bse,setup_screening,setup_sigma
!!
!! CHILDREN
!!      deep_copy,rhoij_alloc,rhoij_copy
!!
!! SOURCE

subroutine hdr_init(bstruct,codvsn,dtset,hdr,pawtab,pertcase,psps)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_abiutil
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: pertcase
 character(len=6),intent(in) :: codvsn
 type(bandstructure_type),intent(in) :: bstruct
 type(dataset_type),intent(in) :: dtset
 type(hdr_type),intent(out) :: hdr
 type(pseudopotential_type),intent(in) :: psps
!arrays
 type(pawtab_type),intent(in) :: pawtab(dtset%ntypat*psps%usepaw)

!Local variables-------------------------------
!scalars
 integer :: bantot,date,itypat,natom,nkpt,npsp,nsppol,nsym,ntypat
 integer :: idx,isppol,ikpt,iband,ipsp
 character(len=500) :: msg
 character(len=8) :: date_time
!arrays
 integer,allocatable :: nlmn(:)

! *************************************************************************

 !@hdr_type

!More checking would be needed ...
 if(dtset%ntypat/=psps%ntypat)then
  write(msg,'(a,2i6)')' dtset%ntypat and psps%ntypat differs. They are :',dtset%ntypat,psps%ntypat
  MSG_ERROR(msg)
 end if

 if(dtset%npsp/=psps%npsp)then
  write(msg,'(a,2i6)')' dtset%npsp and psps%npsp differs. They are :',dtset%npsp,psps%npsp
  MSG_ERROR(msg)
 end if

 call date_and_time(date_time)
 read(date_time,'(i8)')date

 natom  = dtset%natom
 npsp   = dtset%npsp
 nsym   = dtset%nsym
 ntypat = dtset%ntypat

 nkpt   = bstruct%nkpt
 nsppol = bstruct%nsppol
 bantot = bstruct%bantot

!Allocate all components of hdr
 allocate(hdr%istwfk(nkpt))
 allocate(hdr%kptns(3,nkpt))
 allocate(hdr%lmn_size(npsp))
 allocate(hdr%nband(nkpt*nsppol))
 allocate(hdr%npwarr(nkpt)) ! Warning : npwarr here has only one dim
 allocate(hdr%occ(bantot))
 allocate(hdr%pspcod(npsp))
 allocate(hdr%pspdat(npsp))
 allocate(hdr%pspso(npsp))
 allocate(hdr%pspxc(npsp))
 allocate(hdr%so_psp(npsp))
 allocate(hdr%symafm(nsym))
 allocate(hdr%symrel(3,3,nsym))
 allocate(hdr%title(npsp))
 allocate(hdr%tnons(3,nsym))
 allocate(hdr%typat(natom))
 allocate(hdr%wtk(nkpt))
 allocate(hdr%xred(3,natom))
 allocate(hdr%zionpsp(npsp))
 allocate(hdr%znuclpsp(npsp))
 allocate(hdr%znucltypat(ntypat))
 if(psps%usepaw==1)then
  allocate(hdr%pawrhoij(natom))
  allocate(nlmn(ntypat))
  do itypat=1,ntypat
   nlmn(itypat)=pawtab(itypat)%lmn_size
  end do
  !Values of nspden/nspinor/nsppol are dummy ones; they are overwritten later (by hdr_update)
  call rhoij_alloc(dtset%pawcpxocc,nlmn,dtset%nspden,dtset%nspinor,dtset%nsppol,hdr%pawrhoij,dtset%typat)
  deallocate(nlmn)
 end if

!Transfer data from dtset
 hdr%intxc    =dtset%intxc
 hdr%ixc      =dtset%ixc
 hdr%natom    =natom
 hdr%npsp     =npsp
 hdr%nspden   =dtset%nspden
 hdr%nspinor  =dtset%nspinor
 hdr%nsym     =nsym
 hdr%ntypat   =ntypat
 hdr%occopt   =dtset%occopt
 if(psps%usepaw==1)then
  hdr%ngfft(:) =dtset%ngfftdg(1:3)
 else if (dtset%usewvl == 1) then
  hdr%ngfft(:) = dtset%wvl%ni(:)
 else
  hdr%ngfft(:) =dtset%ngfft(1:3)
 end if
 hdr%so_psp(:)=dtset%so_psp(:)
 hdr%symafm(1:min(size(dtset%symafm),size(hdr%symafm)))=dtset%symafm(1:min(size(dtset%symafm),size(hdr%symafm)))
 hdr%symrel(:,:,1:min(size(dtset%symrel,3),size(hdr%symrel,3))) =dtset%symrel(:,:,1:min(size(dtset%symrel,3),size(hdr%symrel,3)))
 hdr%typat(1:dtset%natom)  =dtset%typat(1:dtset%natom) ! PMA : in tests/v2/t11 size(dtset%typat) is bigger dtset%natom
 hdr%ecut     =dtset%ecut
 hdr%ecutsm   =dtset%ecutsm
 hdr%ecut_eff =dtset%ecut * (dtset%dilatmx)**2
 hdr%qptn(:)  =dtset%qptn(:)
 hdr%stmbias  =dtset%stmbias
 hdr%tnons(:,1:min(size(dtset%tnons,2),size(hdr%tnons,2)))    =dtset%tnons(:,1:min(size(dtset%tnons,2),size(hdr%tnons,2)))
 hdr%tphysel  =dtset%tphysel
 hdr%tsmear   =dtset%tsmear
 hdr%rprimd(:,:)=dtset%rprimd_orig(:,:,1)      ! Evolving data
 hdr%xred(:,1:dtset%natom)=dtset%xred_orig(:,1:dtset%natom,1)          ! Evolving data

!Transfer wavelets data.
 hdr%usewvl     = dtset%usewvl
!hdr%nwvlarr will be set later since the number
!of wavelets have not yet been computed.

!DEBUG
!write(6,*)' hdr_init : before transfer data from bstruct'
!ENDDEBUG

!Transfer data from bstruct
 hdr%bantot        =bantot
 hdr%nkpt          =nkpt
 hdr%nsppol        =nsppol
 hdr%istwfk(1:nkpt)=bstruct%istwfk(1:nkpt)
 hdr%nband(1:nkpt*nsppol) =bstruct%nband(1:nkpt*nsppol)
 hdr%npwarr(:)     =bstruct%npwarr(:)
 hdr%kptns(:,:)    =bstruct%kptns(:,:)
 hdr%wtk(:)        =bstruct%wtk(:)

 !call get_eneocc_vect(bstruct,'occ',hdr%occ)  ! Evolving data

 hdr%occ(:)=zero; idx=0
 do isppol=1,nsppol
  do ikpt=1,nkpt
   do iband=1,hdr%nband(ikpt+(isppol-1)*nkpt)
    idx=idx+1
    hdr%occ(idx)=bstruct%occ(iband,ikpt,isppol)
   end do
  end do
 end do

!DEBUG
!write(6,*)' hdr_init : before transfer data from psps'
!ENDDEBUG

!Transfer data from psps
 hdr%pspcod(:)=psps%pspcod(:)
 hdr%pspdat(:)=psps%pspdat(:)
 hdr%pspso (:)=psps%pspso (:)
 hdr%pspxc (:)=psps%pspxc (:)
 hdr%znuclpsp(:)=psps%znuclpsp(:)
 hdr%znucltypat(:)=psps%znucltypat(:)
 hdr%zionpsp(:)=psps%zionpsp(:)
 do ipsp = 1, psps%npsp, 1
    write(hdr%title(ipsp), "(A)") psps%title(ipsp)(1:132)
 end do

!Transfer paw data
 hdr%usepaw=psps%usepaw
 if(psps%usepaw==1) then
  hdr%ecutdg   =dtset%pawecutdg
  hdr%lmn_size(1:npsp)=pawtab(1:npsp)%lmn_size
 else
  hdr%ecutdg=hdr%ecut
  hdr%lmn_size(:)=psps%lmnmax
 end if

!Initialize other known data
 hdr%codvsn   =codvsn
 hdr%date     =date
 hdr%headform = HDR_LATEST_HEADFORM ! Initialize with the latest headform
 hdr%pertcase =pertcase

!Default for other data  (all evolving data)
 hdr%etot     =1.0d20
 hdr%fermie   =1.0d20
 hdr%residm   =1.0d20

end subroutine hdr_init
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_nullify
!! NAME
!! hdr_nullify
!!
!! FUNCTION
!! This subroutine set all the pointers to null()
!!
!! INPUTS
!! hdr <type(hdr_type)>=the header
!!
!! PARENTS
!!      m_io_screening
!!
!! CHILDREN
!!      deep_copy,rhoij_alloc,rhoij_copy
!!
!! SOURCE

subroutine hdr_nullify(hdr)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(hdr_type),intent(inout) :: hdr

! *************************************************************************
 !@hdr_type

! integer
 nullify(hdr%istwfk)
 nullify(hdr%lmn_size)
 nullify(hdr%nband)
 nullify(hdr%npwarr)
 nullify(hdr%pspcod)
 nullify(hdr%pspdat)
 nullify(hdr%pspso)
 nullify(hdr%pspxc)
 nullify(hdr%so_psp)
 nullify(hdr%symafm)
 nullify(hdr%symrel)
 nullify(hdr%typat)

! real
 nullify(hdr%kptns)
 nullify(hdr%occ)
 nullify(hdr%tnons)
 nullify(hdr%wtk)
 nullify(hdr%xred)
 nullify(hdr%zionpsp)
 nullify(hdr%znuclpsp)
 nullify(hdr%znucltypat)

! string arrays
 nullify(hdr%title)

! types
 nullify(hdr%pawrhoij)

end subroutine hdr_nullify
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_clean
!! NAME
!! hdr_clean
!!
!! FUNCTION
!! This subroutine deallocates the components of the header structured datatype
!!
!! INPUTS
!! hdr <type(hdr_type)>=the header
!!
!! OUTPUT
!!  (only deallocate)
!!
!! PARENTS
!!      bethe_salpeter,bloch_interp,compare_interpol,conducti_nc,conducti_paw
!!      cut3d,elphon,gstate,gstateimg,gw2wfk,gw_tools,initaim,inpgkk,inwffil
!!      ioarr,linear_optics_paw,loper3,m_ebands,m_gwannier,m_io_kss
!!      m_io_screening,m_wannier2abinit,macroave,mrggkk,newsp,nonlinear,optic
!!      rdm,read_el_veloc,read_gkk,respfn,screening,sigma,suscep
!!
!! CHILDREN
!!      deep_copy,rhoij_alloc,rhoij_copy
!!
!! SOURCE

subroutine hdr_clean(hdr)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_abiutil
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(hdr_type),intent(inout) :: hdr

!Local variables-------------------------------

! *************************************************************************

 DBG_ENTER("COLL")

 !@hdr_type

 !integer
 if (associated(hdr%istwfk  ))  deallocate(hdr%istwfk  )
 if (associated(hdr%lmn_size))  deallocate(hdr%lmn_size)
 if (associated(hdr%nband   ))  deallocate(hdr%nband   )
 if (associated(hdr%npwarr  ))  deallocate(hdr%npwarr  )
                            
 if (associated(hdr%pspcod))  deallocate(hdr%pspcod)
 if (associated(hdr%pspdat))  deallocate(hdr%pspdat)
 if (associated(hdr%pspso ))  deallocate(hdr%pspso )
 if (associated(hdr%pspxc ))  deallocate(hdr%pspxc )
 if (associated(hdr%so_psp))  deallocate(hdr%so_psp)
 if (associated(hdr%symafm))  deallocate(hdr%symafm)
 if (associated(hdr%symrel))  deallocate(hdr%symrel)
 if (associated(hdr%typat ))  deallocate(hdr%typat )
                            
 !real
 if (associated(hdr%kptns     ))  deallocate(hdr%kptns     )
 if (associated(hdr%occ       ))  deallocate(hdr%occ       )
 if (associated(hdr%tnons     ))  deallocate(hdr%tnons     )
 if (associated(hdr%wtk       ))  deallocate(hdr%wtk       )
 if (associated(hdr%xred      ))  deallocate(hdr%xred      )
 if (associated(hdr%zionpsp   ))  deallocate(hdr%zionpsp   )
 if (associated(hdr%znuclpsp  ))  deallocate(hdr%znuclpsp  )
 if (associated(hdr%znucltypat))  deallocate(hdr%znucltypat)
                            
 !string arrays
 if(associated(hdr%title))  deallocate(hdr%title)

 if (hdr%usepaw==1 .and. associated(hdr%pawrhoij) ) then
  call rhoij_free(hdr%pawrhoij)
  deallocate(hdr%pawrhoij)
 end if

 DBG_EXIT("COLL")

end subroutine hdr_clean
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_copy
!! NAME
!! hdr_copy
!!
!! FUNCTION
!! Make a deep copy of the abinit header.
!!
!! INPUTS
!!  Hdr_in=The header to be copied.
!!
!! OUTPUT
!!  Hdr_cp=The deep copy of Hdr_in.
!!
!! NOTES
!!  The present version deals with versions of the header up to 56.
!!
!! PARENTS
!!      m_io_kss,m_io_screening,m_wannier2abinit
!!
!! CHILDREN
!!      deep_copy,rhoij_alloc,rhoij_copy
!!
!! SOURCE

subroutine hdr_copy(Hdr_in,Hdr_cp)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_abiutil
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(hdr_type),intent(in) :: Hdr_in
 type(hdr_type),intent(inout) :: Hdr_cp

!Local variables-------------------------------
!scalars
 integer :: cplex
 character(len=500) :: msg
!arrays
 integer,pointer :: nlmn(:)
! *************************************************************************

!TODO add method to nullify header, deep_copy might crash if
!a pointer in the Hdr is not initialized to null(), tipically
!this happens for PAW quantities but we always check Hdr%usepaw

 !@hdr_type
 if (Hdr_in%headform>57) then
  write(msg,'(3a,i3,2a)')&
&  ' hdr_copy deals with versions of the header only up to 57. ',ch10,&
&  ' However headform = ',Hdr_in%headform,ch10,&
&  ' Change the source to add the changes done in the new version. '
  MSG_ERROR(msg)
 end if

!=== Integer values ===
 Hdr_cp%bantot   = Hdr_in%bantot
 Hdr_cp%date     = Hdr_in%date
 Hdr_cp%headform = Hdr_in%headform
 Hdr_cp%intxc    = Hdr_in%intxc
 Hdr_cp%ixc      = Hdr_in%ixc
 Hdr_cp%natom    = Hdr_in%natom
 Hdr_cp%nkpt     = Hdr_in%nkpt
 Hdr_cp%npsp     = Hdr_in%npsp
 Hdr_cp%nspden   = Hdr_in%nspden
 Hdr_cp%nspinor  = Hdr_in%nspinor
 Hdr_cp%nsppol   = Hdr_in%nsppol
 Hdr_cp%nsym     = Hdr_in%nsym
 Hdr_cp%ntypat   = Hdr_in%ntypat
 Hdr_cp%occopt   = Hdr_in%occopt
 Hdr_cp%pertcase = Hdr_in%pertcase
 Hdr_cp%usepaw   = Hdr_in%usepaw
 Hdr_cp%usewvl   = Hdr_in%usewvl

 ! === Integer arrays ===
 Hdr_cp%ngfft(:)   = Hdr_in%ngfft(:)
 Hdr_cp%nwvlarr(:) = Hdr_in%nwvlarr(:)

!=== Integer pointers ====
 call deep_copy( Hdr_in%istwfk,  Hdr_cp%istwfk   )
 call deep_copy( Hdr_in%lmn_size,Hdr_cp%lmn_size )
 call deep_copy( Hdr_in%nband,   Hdr_cp%nband    )
 call deep_copy( Hdr_in%npwarr,  Hdr_cp%npwarr   )
 call deep_copy( Hdr_in%pspcod,  Hdr_cp%pspcod )
 call deep_copy( Hdr_in%pspdat,  Hdr_cp%pspdat )
 call deep_copy( Hdr_in%pspso ,  Hdr_cp%pspso  )
 call deep_copy( Hdr_in%pspxc ,  Hdr_cp%pspxc  )
 call deep_copy( Hdr_in%so_psp,  Hdr_cp%so_psp )
 call deep_copy( Hdr_in%symafm,  Hdr_cp%symafm )
 call deep_copy( Hdr_in%symrel,  Hdr_cp%symrel )
 call deep_copy( Hdr_in%typat ,  Hdr_cp%typat  )

!=== Real variables ====
 Hdr_cp%ecut        = Hdr_in%ecut
 Hdr_cp%ecutdg      = Hdr_in%ecutdg
 Hdr_cp%ecutsm      = Hdr_in%ecutsm
 Hdr_cp%ecut_eff    = Hdr_in%ecut_eff
 Hdr_cp%etot        = Hdr_in%etot
 Hdr_cp%fermie      = Hdr_in%fermie
 Hdr_cp%residm      = Hdr_in%residm
 Hdr_cp%stmbias     = Hdr_in%stmbias
 Hdr_cp%tphysel     = Hdr_in%tphysel
 Hdr_cp%tsmear      = Hdr_in%tsmear

 Hdr_cp%qptn(:)     = Hdr_in%qptn(:)
 Hdr_cp%rprimd(:,:) = Hdr_in%rprimd(:,:)

!=== Real pointers ===
 call deep_copy( Hdr_in%kptns     ,Hdr_cp%kptns     )
 call deep_copy( Hdr_in%occ       ,Hdr_cp%occ       )
 !write(*,*)"DEBUG: ",Hdr_in%occ
 !write(*,*)"DEBUG: ",Hdr_in%bantot
 !if (associated(Hdr_cp%occ)) write(*,*)"DEBUG: ",Hdr_cp%occ
 call deep_copy( Hdr_in%tnons     ,Hdr_cp%tnons     )
 call deep_copy( Hdr_in%wtk       ,Hdr_cp%wtk       )
 call deep_copy( Hdr_in%xred      ,Hdr_cp%xred      )
 call deep_copy( Hdr_in%zionpsp   ,Hdr_cp%zionpsp   )
 call deep_copy( Hdr_in%znuclpsp  ,Hdr_cp%znuclpsp  )
 call deep_copy( Hdr_in%znucltypat,Hdr_cp%znucltypat)

!=== Character pointers ===
 Hdr_cp%codvsn = Hdr_in%codvsn
! THIS DOES NOT WORK ON XLF: Hdr_cp%title string length becomes huge and segfaults
! call deep_copy( Hdr_in%title,Hdr_cp%title )
 allocate (Hdr_cp%title(Hdr_cp%npsp))
 Hdr_cp%title = Hdr_in%title

!=== For PAW have to copy Pawrhoij ====
!* TODO alchemy requires a different treatment but for the moment it is not available within PAW.
 if (Hdr_in%usepaw==1) then
  cplex = Hdr_in%Pawrhoij(1)%cplex
  nlmn => Hdr_in%lmn_size(1:Hdr_in%ntypat)
  allocate(Hdr_cp%Pawrhoij(Hdr_in%natom))
  call rhoij_alloc(cplex,nlmn,Hdr_in%nspden,Hdr_in%nspinor,Hdr_in%nsppol,Hdr_cp%Pawrhoij,Hdr_in%typat)
  call rhoij_copy(Hdr_in%Pawrhoij,Hdr_cp%Pawrhoij)
 end if

end subroutine hdr_copy
!!***

!----------------------------------------------------------------------

!!****f* m_header/hdr_get_nelect_byocc
!! NAME
!! hdr_get_nelect_byocc
!!
!! FUNCTION
!!  Return the number of electrons from the occupation numbers
!!  thus taking into account a possible additional charge or alchemy.
!!
!! INPUTS
!!  Hdr<hdr_type>
!!
!! OUTPUT
!!  nelect=Number of electrons in the unit cell.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function hdr_get_nelect_byocc(Hdr) result(nelect)

 use defs_basis

 implicit none

!Arguments ---------------------------------------------
!scalars
 type(hdr_type),intent(in) :: Hdr
 real(dp) :: nelect

!Local variables ---------------------------------------
!scalars
 integer :: idx,isppol,ikibz,nband_k

! *************************************************************************

!* Cannot use znucl because we might have additional charge or alchemy.
 nelect=zero ; idx=0
 do isppol=1,Hdr%nsppol
  do ikibz=1,Hdr%nkpt
   nband_k=Hdr%nband(ikibz+(isppol-1)*Hdr%nkpt)
   nelect = nelect + Hdr%wtk(ikibz)*SUM(Hdr%occ(idx+1:idx+nband_k))
   idx=idx+nband_k
  end do
 end do

!Might also check also Hdr%znuclpsp(:) to avoid round off errors

end function hdr_get_nelect_byocc
!!***

!----------------------------------------------------------------------

!!****f* m_header/isknown_headform
!! NAME
!!  isknown_headform
!!
!! FUNCTION
!!  Returns .TRUE. if headform is one of the allowed values.
!!
!! INPUTS
!!  headform
!!
!! SOURCE

function isknown_headform(headform) result(ans)

 use defs_basis

!Arguments ------------------------------------

 integer,intent(in) :: headform
 logical :: ans

! *************************************************************************

 ans = ANY(headform == HDR_KNOWN_HEADFORMS)

end function isknown_headform
!!***

!----------------------------------------------------------------------

END MODULE m_header
!!***
