!{\src2tex{textfont=tt}}
!!****m* ABINIT/m_paw_toolbox
!! NAME
!!  m_paw_toolbox
!!
!! FUNCTION
!!  This module contains basic tools to initialize, nullify and free basic
!!  PAW objects.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2010 ABINIT group (MG, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! TODO
!! This module should contain the definition of the data types as well.
!! At present these procedures are used only in GW but, hopefully, they will
!! become standard PAW methods.
!!
!! NOTES
!!
!! * Routines tagged with "@type_name" are strongly connected to the definition of the data type.
!!   Strongly connected means that the proper functioning of the implementation relies on the
!!   assumption that the tagged procedure is consistent with the type declaration.
!!   Every time a developer changes the structure "type_name" adding new entries, he/she has to make sure
!!   that all the strongly connected routines are changed accordingly to accommodate the modification of the data type.
!!   Typical examples of strongly connected routines are creation, destruction or reset methods.
!!
!! * gcc 4.3 does not hanlde correctly the case in which there are routines with similar names defined
!!   in the same module. This creates problems at link-time if the module contains procedures
!!   beginning with the same name, e.g init_longname, init_longname_slightly_different.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_paw_toolbox

 use defs_basis
 use defs_datatypes
 use m_errors

 use m_fstrings,    only : toupper
 use m_crystal,     only : crystal_structure
 use defs_abitypes, only : dataset_type

 implicit none

 private

 public :: reset_paw_ij_flags
 public :: destroy_paw_ij
 public :: init_paw_ij
 public :: destroy_paw_an
 public :: destroy_pawtab
 public :: init_pawfgr
 public :: nullify_paw_ij
 public :: nullify_paw_an
 public :: nullify_pawtab
 public :: init_paw_an
 public :: print_pawtab
 public :: print_paw_ij     !FIXME not yet tested
 public :: pawfgrtab_free
 public :: pawfgrtab_init
 public :: pawfgrtab_print
 public :: pawfgrtab_nullify
 public :: reset_paw_an_flags
 public :: init_pawang
 public :: destroy_pawang
 public :: get_dimcprj !  Helper function returning the number of lmn components in the <p_{lmn}^i|\psi> for the i-th atom.
 
CONTAINS  !===========================================================
!!***

!!****f* m_paw_toolbox/reset_paw_ij_flags
!! NAME
!! reset_paw_ij_flags
!!
!! FUNCTION
!!  Set all paw_ij flags set to 0.
!!
!! SIDE EFFECTS
!!  Paw_ij_flags<type(paw_ij_flags_type)>=flags in a paw_ij structure
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine reset_paw_ij_flags(Paw_ij_flags)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_flags_type),intent(inout) :: Paw_ij_flags

! *************************************************************************

 ! @Paw_ij_flags_type
 Paw_ij_flags%has_dij       =0
 Paw_ij_flags%has_dijfr     =0
 Paw_ij_flags%has_dijhartree=0
 Paw_ij_flags%has_dijhat    =0
 Paw_ij_flags%has_dijso     =0
 Paw_ij_flags%has_dijU      =0
 Paw_ij_flags%has_dijxc     =0
 Paw_ij_flags%has_dijxc_val =0
 Paw_ij_flags%has_dijR      =0
 Paw_ij_flags%has_dijL      =0
 Paw_ij_flags%has_dijLr3    =0
 Paw_ij_flags%has_exexch_pot=0
 Paw_ij_flags%has_pawu_occ  =0

end subroutine reset_paw_ij_flags
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/copy_paw_ij_flags
!! NAME
!! copy_paw_ij_flags
!!
!! FUNCTION
!!  Copy a Paw_ij_flags structure.
!!
!! INPUTS
!!  ij_flags_in<type(paw_ij_flags_type)>=input a paw_ij_flags structure
!!
!! OUTPUT
!!  ij_flags_out<type(paw_ij_flags_type)>=output a paw_ij_flags structure
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine copy_paw_ij_flags(ij_flags_in, ij_flags_out)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_flags_type),intent(in)    :: ij_flags_in
 type(Paw_ij_flags_type),intent(inout) :: ij_flags_out

! *************************************************************************

 ! @Paw_ij_flags_type
 ij_flags_out%has_dij        = ij_flags_in%has_dij
 ij_flags_out%has_dijfr      = ij_flags_in%has_dijfr
 ij_flags_out%has_dijhartree = ij_flags_in%has_dijhartree
 ij_flags_out%has_dijhat     = ij_flags_in%has_dijhat
 ij_flags_out%has_dijso      = ij_flags_in%has_dijso
 ij_flags_out%has_dijU       = ij_flags_in%has_dijU
 ij_flags_out%has_dijxc      = ij_flags_in%has_dijxc
 ij_flags_out%has_dijxc_val  = ij_flags_in%has_dijxc_val
 ij_flags_out%has_dijR       = ij_flags_in%has_dijR
 ij_flags_out%has_dijL       = ij_flags_in%has_dijL
 ij_flags_out%has_dijLr3     = ij_flags_in%has_dijLr3
 ij_flags_out%has_exexch_pot = ij_flags_in%has_exexch_pot
 ij_flags_out%has_pawu_occ  = ij_flags_in%has_pawu_occ

end subroutine copy_paw_ij_flags
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/destroy_paw_ij
!! NAME
!!  destroy_paw_ij
!!
!! FUNCTION
!!  Deallocate pointers and nullify flags in a paw_ij structure
!!
!! SIDE EFFECTS
!!  paw_ij(:)<type(paw_ij_type)>=paw arrays given on (i,j) channels
!!
!! PARENTS
!!      bethe_salpeter,ldau_self,m_energy,nstpaw3,respfn,scfcv,scfcv3,screening
!!      sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_paw_ij(Paw_ij)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_type),intent(inout) :: Paw_ij(:)

!Local variables-------------------------------
 integer :: iat,natom

! *************************************************************************

 DBG_ENTER("COLL")

 ! @Paw_ij_type
 natom=SIZE(Paw_ij)
 do iat=1,natom
  if (associated(Paw_ij(iat)%dij       )) deallocate(Paw_ij(iat)%dij       )
  if (associated(Paw_ij(iat)%dijfr     )) deallocate(Paw_ij(iat)%dijfr     )
  if (associated(Paw_ij(iat)%dijhartree)) deallocate(Paw_ij(iat)%dijhartree)
  if (associated(Paw_ij(iat)%dijhat    )) deallocate(Paw_ij(iat)%dijhat    )
  if (associated(Paw_ij(iat)%dijU      )) deallocate(Paw_ij(iat)%dijU      )
  if (associated(Paw_ij(iat)%dijso     )) deallocate(Paw_ij(iat)%dijso     )
  if (associated(Paw_ij(iat)%dijxc     )) deallocate(Paw_ij(iat)%dijxc     )
  if (associated(Paw_ij(iat)%dijxc_val )) deallocate(Paw_ij(iat)%dijxc_val )
  if (associated(Paw_ij(iat)%dijR      )) deallocate(Paw_ij(iat)%dijR      )
  if (associated(Paw_ij(iat)%dijL      )) deallocate(Paw_ij(iat)%dijL      )
  if (associated(Paw_ij(iat)%dijLr3    )) deallocate(Paw_ij(iat)%dijLr3    )
  if (associated(Paw_ij(iat)%noccmmp   )) deallocate(Paw_ij(iat)%noccmmp   )
  if (associated(Paw_ij(iat)%nocctot   )) deallocate(Paw_ij(iat)%nocctot   )
  if (associated(Paw_ij(iat)%vpawx     )) deallocate(Paw_ij(iat)%vpawx     )

  ! === Reset all has_* flags ===
  Paw_ij(iat)%has_dij       =0
  Paw_ij(iat)%has_dijfr     =0
  Paw_ij(iat)%has_dijhartree=0
  Paw_ij(iat)%has_dijhat    =0
  Paw_ij(iat)%has_dijso     =0
  Paw_ij(iat)%has_dijU      =0
  Paw_ij(iat)%has_dijxc     =0
  Paw_ij(iat)%has_dijxc_val =0
  Paw_ij(iat)%has_dijR      =0
  Paw_ij(iat)%has_dijL      =0
  Paw_ij(iat)%has_dijLr3    =0
  Paw_ij(iat)%has_exexch_pot=0
  Paw_ij(iat)%has_pawu_occ  =0
 end do

 DBG_EXIT("COLL")

end subroutine destroy_Paw_ij
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/init_paw_ij
!! NAME
!! init_paw_ij
!!
!! FUNCTION
!!  Initialize a Paw_ij data type.
!!
!! INPUTS
!!  cplex=1 if all on-site PAW quantities are real, 2 if they are complex
!!  cplex_dij=1 if dij are real, 2 if they are complex
!!  natom=Number of atoms.
!!  ntypat=Number of types of atoms in cell.
!!  nspinor=Number of spinor components
!!  nsppol=Number of independent spin polarizations.
!!  nspden=Number of spin-density components
!!  pawspnorb=1 if spin-orbit coupling is activated
!!  typat(natom)=Type of each atom
!!  Pawtab(ntypat)<type(pawtab_type)>=PAW tabulated starting data
!!
!! OPTIONAL INPUTS
!!  has_dij=1 to allocate Paw_ij%dij, 0 otherwise (default)
!!  has_dijhat=1 to allocate Paw_ij%dijhat, 0 otherwise (default)
!!  has_dijxc=1 to allocate Paw_ij%dijxc, 0 otherwise (default)
!!  has_dijxc_val=1 to allocate Paw_ij%dijxc_val, 0 otherwise (default)
!!  has_dijhartree=1 to allocate Paw_ij%dijhartree, 0 otherwise (default)
!!  has_dijso=1 to allocate Paw_ij%dijso, used only if pawspnorb>0. 0 otherwise (default)
!!  has_dijU=1 to allocate Paw_ij%dijU, used only if Pawtab(itypat)%usepawu>0. 0 otherwise (default).
!!  has_exexch_pot=1 to allocate potnetial used in PAW+(local exact exchange) formalism, 0 otherwise (default)
!!  has_pawu_occ=1 to allocate occupations used in PAW+U formalism, 0 otherwise (default)
!!
!! OUTPUT
!!  Paw_ij(natom)<type(paw_ij_type)>=data structure containing PAW arrays given on (i,j) channels.
!!   In output all the basic dimensions are defined and the arrays are allocated
!!   according to the input variables.
!!
!! PARENTS
!!      bethe_salpeter,ldau_self,m_energy,nstpaw3,paw_qpscgw,respfn,scfcv
!!      scfcv3,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_paw_ij(Paw_ij,cplex,cplex_dij,nspinor,nsppol,nspden,pawspnorb,natom,ntypat,typat,Pawtab,&
&                      has_dij,has_dijfr,has_dijhartree,has_dijhat,& ! Optional
&                      has_dijxc,has_dijxc_val,has_dijso,has_dijU,&  ! Optional
&                      has_exexch_pot,has_pawu_occ)                  ! Optional

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,cplex_dij,nspinor,nspden,nsppol,natom,ntypat,pawspnorb
 integer,optional,intent(in) :: has_dij,has_dijfr,has_dijhat,has_dijxc,has_dijxc_val
 integer,optional,intent(in) :: has_dijso,has_dijhartree,has_dijU,has_exexch_pot,has_pawu_occ
!arrays
 integer,intent(in) :: typat(natom)
 type(Paw_ij_type),intent(inout) :: Paw_ij(natom)
 type(Pawtab_type),intent(in) :: Pawtab(ntypat)

!Local variables-------------------------------
 integer :: iat,itypat,lmn2_size,ndij

! *************************************************************************

 DBG_ENTER("COLL")

 !allocate(Paw_ij(natom))
 !call nullify_paw_ij(Paw_ij)

 ! @Paw_ij_type
 do iat=1,natom
  itypat=typat(iat)
  lmn2_size              =Pawtab(itypat)%lmn2_size
  Paw_ij(iat)%cplex      =cplex
  !Paw_ij(iat)%cplex_dij  =nspinor
  !Paw_ij(iat)%cplex_dij  =MAX(cplex,1+pawspnorb,nspinor)
  Paw_ij(iat)%cplex_dij  =cplex_dij
  Paw_ij(iat)%nspden     =nspden
  Paw_ij(iat)%nsppol     =nsppol
  Paw_ij(iat)%lmn_size   =Pawtab(itypat)%lmn_size
  Paw_ij(iat)%lmn2_size  =lmn2_size
  Paw_ij(iat)%ndij       =MAX(nspinor**2,nspden)
  !Paw_ij(iat)%lmnmix_sz =  do we need this? It seems it is not used anymore and can be removed

  ndij=Paw_ij(iat)%ndij

  ! ==================================
  ! === Allocations (all optional) ===
  ! ==================================

  ! === Allocation for total Dij ===
  Paw_ij(iat)%has_dij=0
  if (PRESENT(has_dij)) then
    if (has_dij/=0) then
      Paw_ij(iat)%has_dij=1
      allocate(Paw_ij(iat)%dij(cplex_dij*lmn2_size,ndij))
    end if
  end if

  ! === Allocation for total Dij_Hartree ===
  Paw_ij(iat)%has_dijhartree=0
  if (PRESENT(has_dijhartree)) then
    if (has_dijhartree/=0) then
      Paw_ij(iat)%has_dijhartree=1
      allocate(Paw_ij(iat)%dijhartree(cplex*lmn2_size))
    end if
  end if

  ! === Allocation for total Dij_hat ===
  Paw_ij(iat)%has_dijhat=0
  if (PRESENT(has_dijhat)) then
    if (has_dijhat/=0) then
      Paw_ij(iat)%has_dijhat=1
      allocate(Paw_ij(iat)%dijhat(cplex_dij*lmn2_size,ndij))
    end if
  end if

  ! === Allocation for total Dij_XC ===
  Paw_ij(iat)%has_dijxc=0
  if (PRESENT(has_dijxc)) then
    if (has_dijxc/=0) then
      Paw_ij(iat)%has_dijxc=1
      allocate(Paw_ij(iat)%dijxc(cplex_dij*lmn2_size,ndij))
    end if
  end if

  ! === Allocation for total Dij_XC_val ===
  Paw_ij(iat)%has_dijxc_val=0
  if (PRESENT(has_dijxc_val)) then
    if (has_dijxc_val/=0) then
      Paw_ij(iat)%has_dijxc_val=1
      allocate(Paw_ij(iat)%dijxc_val(cplex_dij*lmn2_size,ndij))
    end if
  end if

  ! === Allocation for total Dij_U_val ===
  Paw_ij(iat)%has_dijU=0
  if (PRESENT(has_dijU)) then
    if (has_dijU/=0.and.Pawtab(itypat)%usepawu>0) then
      Paw_ij(iat)%has_dijU=1
      allocate(Paw_ij(iat)%dijU(cplex_dij*lmn2_size,ndij))
    end if
  end if

  ! === Allocation for total Dij_SO ===
  Paw_ij(iat)%has_dijso=0
  if (PRESENT(has_dijso)) then
    if (has_dijso/=0.and.pawspnorb>0) then
      Paw_ij(iat)%has_dijso=1
      allocate(Paw_ij(iat)%dijso(cplex_dij*lmn2_size,ndij))
     end if
  end if

  ! === Allocation for frozen part of 1st-order Dij ===
  Paw_ij(iat)%has_dijfr=0
  if (PRESENT(has_dijfr)) then
    if (has_dijfr/=0) then
      Paw_ij(iat)%has_dijfr=1
      allocate(Paw_ij(iat)%dijfr(cplex_dij*lmn2_size,ndij))
    end if
  end if

  ! === Allocation for PAW+U occupations ===
  Paw_ij(iat)%has_pawu_occ=0
  if (PRESENT(has_pawu_occ)) then
    if (has_pawu_occ/=0.and.Pawtab(itypat)%usepawu>0) then
      Paw_ij(iat)%has_pawu_occ=1
      allocate(Paw_ij(iat)%noccmmp(cplex_dij,2*Pawtab(itypat)%lpawu+1,2*Pawtab(itypat)%lpawu+1,ndij))
      allocate(Paw_ij(iat)%nocctot(ndij))
    end if
  end if

  ! === Allocation for PAW+LEXX potential ===
  Paw_ij(iat)%has_exexch_pot=0
  if (PRESENT(has_exexch_pot)) then
    if (has_exexch_pot/=0.and.Pawtab(itypat)%useexexch>0) then
      Paw_ij(iat)%has_exexch_pot=1
    ! TODO solve issue with first dimension
      allocate(Paw_ij(iat)%vpawx(1,lmn2_size,nspden))
     end if
  end if

 end do !iat

 DBG_EXIT("COLL")

end subroutine init_paw_ij
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/destroy_paw_an
!! NAME
!! destroy_paw_an
!!
!! FUNCTION
!!  Deallocate pointers and nullify flags in a paw_an structure
!!
!! SIDE EFFECTS
!!  Paw_an(:)<type(Paw_an_type)>=various arrays given on ANgular mesh or ANgular moments
!!
!! SIDE EFFECTS
!!  All associated pointers in Paw_an(:) are deallocated
!!
!! NOTES
!!  vh1 and vht1 are defined in the data structure but never used.
!!  Cannot test for association status since these quantities are
!!  not nullified before entering the calculation
!!
!! PARENTS
!!      bethe_salpeter,respfn,scfcv,scfcv3,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_Paw_an(Paw_an)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_type),intent(inout) :: Paw_an(:)

!Local variables-------------------------------
 integer :: iat,natom
!integer :: itypat

! *************************************************************************

 DBG_ENTER("COLL")

 !@Paw_an_type

 natom=SIZE(Paw_an)

! do iat=1,natom
!  itypat=typat(iat)
!  deallocate(Paw_an(iat)%lmselect)
!  !deallocate(Paw_an(iat)%vh1,Paw_an(iat)%vht1)      !TODO nullify these arrays
!  deallocate(paw_an(iat)%vxc1,Paw_an(iat)%vxct1)
!  if (Paw_an(iat)%has_vxcval==1 ) deallocate(Paw_an(iat)%vxc1_val,Paw_an(iat)%vxct1_val)
!  if (Pawtab(itypat)%useexexch>0) deallocate(Paw_an(iat)%vxc_ex)
! end do

 do iat=1,natom
  if (associated(Paw_an(iat)%lmselect )) deallocate(Paw_an(iat)%lmselect )
  if (associated(Paw_an(iat)%vh1      )) deallocate(Paw_an(iat)%vh1      )
  if (associated(Paw_an(iat)%vht1     )) deallocate(Paw_an(iat)%vht1     )
  if (associated(Paw_an(iat)%vxc1     )) deallocate(Paw_an(iat)%vxc1     )
  if (associated(Paw_an(iat)%vxc1_val )) deallocate(Paw_an(iat)%vxc1_val )
  if (associated(Paw_an(iat)%vxct1    )) deallocate(Paw_an(iat)%vxct1    )
  if (associated(Paw_an(iat)%vxct1_val)) deallocate(Paw_an(iat)%vxct1_val)
  if (associated(Paw_an(iat)%vxc_ex   )) deallocate(Paw_an(iat)%vxc_ex   )
  if (associated(Paw_an(iat)%kxc1     )) deallocate(Paw_an(iat)%kxc1     )
  if (associated(Paw_an(iat)%kxct1    )) deallocate(Paw_an(iat)%kxct1    )

  ! === Reset all has_* flags ===
  Paw_an(iat)%has_kxc     =0
  Paw_an(iat)%has_vhartree=0
  Paw_an(iat)%has_vxc     =0
  Paw_an(iat)%has_vxcval  =0
 end do !iat

 DBG_EXIT("COLL")

end subroutine destroy_Paw_an
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/destroy_pawtab
!! NAME
!!  destroy_pawtab
!!
!! FUNCTION
!!  Deallocate pointers and nullify flags in a pawtab structure
!!
!! SIDE EFFECTS
!!  Pawtab(:)<type(pawtab_type)>=PAW arrays tabulated.
!!  All associated pointers in Pawtab(:) are deallocated
!!
!! PARENTS
!!      mrgddb,rdddb9
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_pawtab(Pawtab)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Pawtab_type),intent(inout) :: Pawtab(:)

!Local variables-------------------------------
 integer :: ityp,ntypat

! *************************************************************************

 !@Pawtab_type
 ntypat=SIZE(Pawtab(:))
 if (ntypat==0) return

 do ityp=1,ntypat

  if (associated(Pawtab(ityp)%indklmn)) deallocate(Pawtab(ityp)%indklmn)
  if (associated(Pawtab(ityp)%klmntomn)) deallocate(Pawtab(ityp)%klmntomn)
  if (associated(Pawtab(ityp)%kmix)) deallocate(Pawtab(ityp)%kmix)
  if (associated(Pawtab(ityp)%lnproju)) deallocate(Pawtab(ityp)%lnproju)
  if (associated(Pawtab(ityp)%coredens)) deallocate(Pawtab(ityp)%coredens)
  if (associated(Pawtab(ityp)%dij0)) deallocate(Pawtab(ityp)%dij0)
  if (associated(Pawtab(ityp)%dltij)) deallocate(Pawtab(ityp)%dltij)
  if (associated(Pawtab(ityp)%dshpfunc)) deallocate(Pawtab(ityp)%dshpfunc)
  if (associated(Pawtab(ityp)%eijkl)) deallocate(Pawtab(ityp)%eijkl)
  if (associated(Pawtab(ityp)%fk)) deallocate(Pawtab(ityp)%fk)
  if (associated(Pawtab(ityp)%gnorm)) deallocate(Pawtab(ityp)%gnorm)
  if (associated(Pawtab(ityp)%kij)) deallocate(Pawtab(ityp)%kij)
  if (associated(Pawtab(ityp)%nabla_ij)) deallocate(Pawtab(ityp)%nabla_ij)
  if (associated(Pawtab(ityp)%phi)) deallocate(Pawtab(ityp)%phi)
  if (associated(Pawtab(ityp)%phiphj)) deallocate(Pawtab(ityp)%phiphj)
  if (associated(Pawtab(ityp)%phiphjint)) deallocate(Pawtab(ityp)%phiphjint)
  if (associated(Pawtab(ityp)%ph0phiint)) deallocate(Pawtab(ityp)%ph0phiint)
  if (associated(Pawtab(ityp)%qgrid_shp)) deallocate(Pawtab(ityp)%qgrid_shp)
  if (associated(Pawtab(ityp)%qijl)) deallocate(Pawtab(ityp)%qijl)
  if (associated(Pawtab(ityp)%rad_for_spline)) deallocate(Pawtab(ityp)%rad_for_spline)
  if (associated(Pawtab(ityp)%rhoij0)) deallocate(Pawtab(ityp)%rhoij0)
  if (associated(Pawtab(ityp)%shape_alpha)) deallocate(Pawtab(ityp)%shape_alpha)
  if (associated(Pawtab(ityp)%shape_q)) deallocate(Pawtab(ityp)%shape_q)
  if (associated(Pawtab(ityp)%shapefunc)) deallocate(Pawtab(ityp)%shapefunc)
  if (associated(Pawtab(ityp)%shapefncg)) deallocate(Pawtab(ityp)%shapefncg)
  if (associated(Pawtab(ityp)%sij)) deallocate(Pawtab(ityp)%sij)
  if (associated(Pawtab(ityp)%tcoredens)) deallocate(Pawtab(ityp)%tcoredens)
  if (associated(Pawtab(ityp)%tcorespl)) deallocate(Pawtab(ityp)%tcorespl)
  if (associated(Pawtab(ityp)%tphi)) deallocate(Pawtab(ityp)%tphi)
  if (associated(Pawtab(ityp)%tphitphj)) deallocate(Pawtab(ityp)%tphitphj)
  if (associated(Pawtab(ityp)%tvalespl)) deallocate(Pawtab(ityp)%tvalespl)
  if (associated(Pawtab(ityp)%Vee)) deallocate(Pawtab(ityp)%Vee)
  if (associated(Pawtab(ityp)%Vex)) deallocate(Pawtab(ityp)%Vex)
  if (associated(Pawtab(ityp)%zioneff)) deallocate(Pawtab(ityp)%zioneff)

  ! === Reset all has_* flags ===
  Pawtab(ityp)%has_kij  =0
  Pawtab(ityp)%has_nabla=0
  Pawtab(ityp)%usetcore =0
  Pawtab(ityp)%usetvale =0
 end do

end subroutine destroy_pawtab
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/init_pawfgr
!! NAME
!! init_pawfgr
!!
!! FUNCTION
!!  Initialize a pawfgr_type datatype, reporting also info on the FFT mesh
!!  according to the method used (norm-conserving or PAW)
!!
!! INPUTS
!!  k0(3)=input k vector for k+G sphere
!!  Dtset <type(dataset_type)>=all input variables for this dataset
!!   %dilatmx
!!   %usepaw
!!   %natom
!!   %ngfft
!!   %ngfftdg
!!   %nfft
!!   %mgfft
!!   %mgfftdg
!!   %dilatmx
!!   %pawecutdg
!!   %ecut
!!  gmet(3,3)=reciprocal space metric (bohr^-2)
!!
!! OUTPUT
!!  ecut_eff=effective energy cutoff (hartree) for coarse planewave basis sphere
!!  ecutdg_eff=effective energy cutoff (hartree) for dense planewave basis sphere
!!  gsqcutc_eff=(PAW) Fourier cutoff on G^2 for "large sphere" of radius double for the coarse FFT grid
!   gsqcutf_eff=Fourier cutoff on G^2 for "large sphere" of radius double for the dense FFT grid
!!  nfftf=(effective) number of FFT grid points (for this proc), for dense FFT mesh
!!  mgfftf=maximum size of 1D FFTs, for dense FFT mesh
!!  ngfftc(18),ngfftf(18)=contain all needed information about 3D FFT, for coarse and dense FFT mesh, resp.
!!                        see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  Pawfgr<pawfgr_type>=For PAW, Fine rectangular GRid parameters and related data
!!
!! PARENTS
!!      bethe_salpeter,gstate,gstateimg,respfn,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_pawfgr(Dtset,Pawfgr,mgfftf,nfftf,ecut_eff,ecutdg_eff,ngfftc,ngfftf,&
&                      gsqcutc_eff,gsqcutf_eff,gmet,k0) ! optional
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_56_recipspace
 use interfaces_66_paw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(out) :: nfftf,mgfftf
 real(dp),intent(out) :: ecut_eff,ecutdg_eff
 real(dp),intent(out),optional :: gsqcutf_eff,gsqcutc_eff
 type(dataset_type),intent(in) :: Dtset
 type(Pawfgr_type),intent(out) :: Pawfgr
!arrays
 real(dp),intent(in),optional :: gmet(3,3)
 integer,intent(out) :: ngfftc(18),ngfftf(18)
 real(dp),intent(in),optional :: k0(3)

!Local variables-------------------------------
 integer :: ii,nfftc_tot,nfftf_tot
 real(dp) :: boxcut,boxcutc
 character(len=500) :: msg

!************************************************************************

 DBG_ENTER("COLL")

 !@Pawfgr_type

 if ((present(gsqcutc_eff).or.present(gsqcutf_eff)).and.&
&    ((.not.present(gmet)).or.(.not.present(k0)))) then
   msg='To compute gsqcut[c,f]_eff, both k0 and gmet must be present as argument !'
   MSG_BUG(msg)
 end if

 ngfftc(:)=Dtset%ngfft(:)

 SELECT CASE (Dtset%usepaw)

 CASE (0)
  ! === Norm-conserving pseudopotentials ===
  nfftf=Dtset%nfft ; mgfftf=Dtset%mgfft ; ngfftf(:)=Dtset%ngfft(:)
  Pawfgr%usefinegrid=0 ; allocate(Pawfgr%coatofin(0),Pawfgr%fintocoa(0))
  ecut_eff  =Dtset%ecut*Dtset%dilatmx**2
  ecutdg_eff=ecut_eff

 CASE (1)
  ! == PAW calculation ===
  if (Dtset%pawecutdg>=1.0000001_dp*Dtset%ecut) then
   ! * Use fine FFT grid generated according to pawecutdg.
   nfftf=Dtset%nfftdg ; mgfftf=Dtset%mgfftdg ; ngfftf(:)=Dtset%ngfftdg(:)
   nfftc_tot =ngfftc(1)*ngfftc(2)*ngfftc(3)
   nfftf_tot =ngfftf(1)*ngfftf(2)*ngfftf(3)
   Pawfgr%usefinegrid=1 ; allocate(Pawfgr%coatofin(nfftc_tot),Pawfgr%fintocoa(nfftf_tot))
   call indgrid(Pawfgr%coatofin,Pawfgr%fintocoa,nfftc_tot,nfftf_tot,ngfftc,ngfftf)
  else
   ! * Do not use fine FFT mesh. Simple transfer that can be done in parallel with only local info.
   nfftf=Dtset%nfft ; mgfftf=Dtset%mgfft ; ngfftf(:)=Dtset%ngfft(:)
   Pawfgr%usefinegrid=0 ; allocate(Pawfgr%coatofin(Dtset%nfft),Pawfgr%fintocoa(Dtset%nfft))
   do ii=1,Dtset%nfft
    Pawfgr%coatofin(ii)=ii ; Pawfgr%fintocoa(ii)=ii
   end do
  end if

  ! == Store useful dimensions in Pawfgr ===
  Pawfgr%nfftc=Dtset%nfft ; Pawfgr%mgfftc=Dtset%mgfft ; Pawfgr%ngfftc(:)=Dtset%ngfft(:)
  Pawfgr%nfft=nfftf       ; Pawfgr%mgfft=mgfftf       ; Pawfgr%ngfft (:)=ngfftf(:)
  ecutdg_eff=Dtset%pawecutdg*Dtset%dilatmx**2
  ecut_eff  =Dtset%ecut*Dtset%dilatmx**2

 CASE DEFAULT
  write(msg,'(a,i4)')' Wrong value of usepaw: ',Dtset%usepaw
  MSG_BUG(msg)
 END SELECT
 !
 ! === Get boxcut for given gmet, ngfft, and ecut (center at k0) ===
 !     boxcut=ratio of basis sphere diameter to fft box side
 boxcut=-one
 if (Dtset%usepaw==1) then
   if (present(gsqcutc_eff)) then
     write(msg,'(2a)')ch10,' Coarse grid specifications '!(used for wave-functions):'
     call wrtout(std_out,msg,'COLL') !; call wrtout(ab_out,msg,'COLL')
     call getcut(boxcutc,ecut_eff,gmet,gsqcutc_eff,Dtset%iboxcut,std_out,k0,ngfftc)
   end if
   if (present(gsqcutf_eff)) then
     write(msg,'(2a)')ch10,' Fine grid specifications (used for densities):'
     call wrtout(std_out,msg,'COLL') !; call wrtout(ab_out,msg,'COLL')
     call getcut(boxcut,ecutdg_eff,gmet,gsqcutf_eff,Dtset%iboxcut,std_out,k0,ngfftf)
   end if
 else if (present(gsqcutc_eff)) then
   call getcut(boxcut,ecut_eff,gmet,gsqcutc_eff,Dtset%iboxcut,std_out,k0,ngfftc)
   gsqcutf_eff=gsqcutc_eff
 end if
 !
 ! === Check that boxcut>=2 if intxc=1; otherwise intxc must be set=0 ===
 if (boxcut>=zero .and. boxcut<two .and. Dtset%intxc==1) then
  write(msg,'(a,es12.4,5a)')&
&  ' boxcut=',boxcut,' is < 2.0  => intxc must be 0;',ch10,&
&  ' Need larger ngfft to use intxc=1.',ch10,&
&  ' Action : you could increase ngfft, or decrease ecut, or put intxc=0.'
  MSG_ERROR(msg)
 end if

 DBG_EXIT("COLL")

end subroutine init_pawfgr
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/nullify_paw_ij
!! NAME
!!  nullify_paw_ij
!!
!! FUNCTION
!!  Nullify pointers and flags in a paw_ij structure
!!
!! SIDE EFFECTS
!!  Paw_ij(:)<type(paw_ij_type)>=PAW arrays given on (i,j) channels. Nullified in output
!!
!! PARENTS
!!      bethe_salpeter,ldau_self,m_energy,nstpaw3,paw_qpscgw,respfn,scfcv
!!      scfcv3,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_paw_ij(Paw_ij)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_ij_type),intent(inout) :: Paw_ij(:)

!Local variables-------------------------------
 integer :: iat,natom

! *************************************************************************

 !@Paw_ij_type

 natom=SIZE(Paw_ij(:))

 do iat=1,natom
  nullify(Paw_ij(iat)%dij       )
  nullify(Paw_ij(iat)%dijfr     )
  nullify(Paw_ij(iat)%dijhartree)
  nullify(Paw_ij(iat)%dijhat    )
  nullify(Paw_ij(iat)%dijU      )
  nullify(Paw_ij(iat)%dijso     )
  nullify(Paw_ij(iat)%dijxc     )
  nullify(Paw_ij(iat)%dijxc_val )
  nullify(Paw_ij(iat)%dijR      )
  nullify(Paw_ij(iat)%dijL      )
  nullify(Paw_ij(iat)%dijLr3    )
  nullify(Paw_ij(iat)%noccmmp   )
  nullify(Paw_ij(iat)%nocctot   )
  nullify(Paw_ij(iat)%vpawx     )

  ! === Set all has_* flags to zero ===
  Paw_ij(iat)%has_dij       =0
  Paw_ij(iat)%has_dijfr     =0
  Paw_ij(iat)%has_dijhartree=0
  Paw_ij(iat)%has_dijhat    =0
  Paw_ij(iat)%has_dijso     =0
  Paw_ij(iat)%has_dijU      =0
  Paw_ij(iat)%has_dijxc     =0
  Paw_ij(iat)%has_dijxc_val =0
  Paw_ij(iat)%has_dijR      =0
  Paw_ij(iat)%has_dijL      =0
  Paw_ij(iat)%has_dijLr3    =0
  Paw_ij(iat)%has_exexch_pot=0
  Paw_ij(iat)%has_pawu_occ  =0
 end do !iat

end subroutine nullify_paw_ij
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/nullify_paw_an
!! NAME
!!  nullify_paw_an
!!
!! FUNCTION
!!  Nullify pointers and flags in a paw_an structure
!!
!! SIDE EFFECTS
!!  Paw_an(:)<type(paw_an_type)>=PAW arrays given on ANgular mesh or ANgular moments.
!!                               Nullified in output
!!
!! PARENTS
!!      bethe_salpeter,paw_qpscgw,respfn,scfcv,scfcv3,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_paw_an(Paw_an)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_type),intent(inout) :: Paw_an(:)

!Local variables-------------------------------
 integer :: iat,natom

! *************************************************************************

 !@Paw_an_type
 natom=SIZE(Paw_an(:))

 do iat=1,natom
  nullify(Paw_an(iat)%lmselect )
  nullify(Paw_an(iat)%vh1      )
  nullify(Paw_an(iat)%vht1     )
  nullify(Paw_an(iat)%vxc1     )
  nullify(Paw_an(iat)%vxct1    )
  nullify(Paw_an(iat)%vxc1_val )
  nullify(Paw_an(iat)%vxct1_val)
  nullify(Paw_an(iat)%vxc_ex   )
  nullify(Paw_an(iat)%kxc1     )
  nullify(Paw_an(iat)%kxct1    )

  ! === Set all has_* flags to zero ===
  Paw_an(iat)%has_kxc      =0
  Paw_an(iat)%has_vhartree =0
  Paw_an(iat)%has_vxc      =0
  Paw_an(iat)%has_vxcval   =0
 end do

end subroutine nullify_paw_an
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/nullify_pawtab
!! NAME
!!  nullify_pawtab
!!
!! FUNCTION
!!  Nullify pointers and flags in a pawtab structure
!!
!! SIDE EFFECTS
!!  Pawtab(:)<type(pawtab_type)>=PAW arrays tabulated.
!!                               Nullified in output
!!
!! PARENTS
!!      mrgddb,rdddb9
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_pawtab(Pawtab)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Pawtab_type),intent(inout) :: Pawtab(:)

!Local variables-------------------------------
 integer :: ityp,ntypat

! *************************************************************************

 !@Pawtab_type
 ntypat=SIZE(Pawtab(:))
 if (ntypat==0) return

 do ityp=1,ntypat

  nullify(Pawtab(ityp)%indklmn)
  nullify(Pawtab(ityp)%klmntomn)
  nullify(Pawtab(ityp)%kmix)
  nullify(Pawtab(ityp)%lnproju)
  nullify(Pawtab(ityp)%coredens)
  nullify(Pawtab(ityp)%dij0)
  nullify(Pawtab(ityp)%dltij)
  nullify(Pawtab(ityp)%dshpfunc)
  nullify(Pawtab(ityp)%eijkl)
  nullify(Pawtab(ityp)%fk)
  nullify(Pawtab(ityp)%gnorm)
  nullify(Pawtab(ityp)%kij)
  nullify(Pawtab(ityp)%nabla_ij)
  nullify(Pawtab(ityp)%phi)
  nullify(Pawtab(ityp)%phiphj)
  nullify(Pawtab(ityp)%phiphjint)
  nullify(Pawtab(ityp)%ph0phiint)
  nullify(Pawtab(ityp)%qgrid_shp)
  nullify(Pawtab(ityp)%qijl)
  nullify(Pawtab(ityp)%rad_for_spline)
  nullify(Pawtab(ityp)%rhoij0)
  nullify(Pawtab(ityp)%shape_alpha)
  nullify(Pawtab(ityp)%shape_q)
  nullify(Pawtab(ityp)%shapefunc)
  nullify(Pawtab(ityp)%shapefncg)
  nullify(Pawtab(ityp)%sij)
  nullify(Pawtab(ityp)%tcoredens)
  nullify(Pawtab(ityp)%tcorespl)
  nullify(Pawtab(ityp)%tphi)
  nullify(Pawtab(ityp)%tphitphj)
  nullify(Pawtab(ityp)%tvalespl)
  nullify(Pawtab(ityp)%Vee)
  nullify(Pawtab(ityp)%Vex)
  nullify(Pawtab(ityp)%zioneff)

  ! === Set all has_* flags to zero ===
  Pawtab(ityp)%has_kij  =0
  Pawtab(ityp)%has_nabla=0
  Pawtab(ityp)%usetcore =0
  Pawtab(ityp)%usetvale =0
 end do

end subroutine nullify_pawtab
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/init_paw_an
!! NAME
!!  init_paw_an
!!
!! FUNCTION
!!  Initialize a paw_an data type.
!!
!! SIDE EFFECTS
!!  Paw_an(:)<type(paw_an_type)>=PAW arrays given on ANgular mesh or ANgular moments.
!!                               Initialized in output
!!
!! PARENTS
!!      bethe_salpeter,paw_qpscgw,respfn,scfcv,scfcv3,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE


subroutine init_paw_an(natom,ntypat,nkxc1,nspden,cplex,pawxcdev,typat,Pawang,Pawtab,Paw_an,&
&                      has_vhartree,has_vxc,has_vxcval,has_kxc) ! Optional

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,nkxc1,ntypat,cplex,nspden,pawxcdev
 integer,optional,intent(in) :: has_vhartree,has_vxc,has_vxcval,has_kxc
!arrays
 integer,intent(in) :: typat(natom)
 type(Pawang_type),intent(in) :: Pawang
 type(Pawtab_type),intent(in) :: Pawtab(ntypat)
 type(Paw_an_type),intent(inout) :: Paw_an(:)

!Local variables-------------------------------
 integer :: iat,itypat,lm_size,v_size

! *************************************************************************

 DBG_ENTER("COLL")

 !@Paw_an_type

 !allocate(Paw_an(natom))
 !call nullify_paw_an(Paw_an)

 do iat=1,natom
  itypat=typat(iat)
  lm_size                =Pawtab(itypat)%lcut_size**2
  Paw_an(iat)%angl_size  =Pawang%angl_size
  Paw_an(iat)%cplex      =cplex
  Paw_an(iat)%lm_size    =lm_size
  Paw_an(iat)%mesh_size  =Pawtab(itypat)%mesh_size
  Paw_an(iat)%nkxc1      =nkxc1
  Paw_an(iat)%nspden     =nspden

  ! === Non-zero LM-moments of "one-center" densities/potentials ===
  ! * Filled in pawdenpot.
  allocate(Paw_an(iat)%lmselect(lm_size))

  v_size=Paw_an(iat)%lm_size ; if (pawxcdev==0) v_size=Paw_an(iat)%angl_size

 ! === XC potential inside the sphere ===
 ! * LM-moments of potential if pawxcdev/=0
 ! * (theta,phi) values of potential if pawxcdev=0
  Paw_an(iat)%has_vxc=0
  if (PRESENT(has_vxc)) then
   if (has_vxc>0) then
    Paw_an(iat)%has_vxc=1
    allocate(Paw_an(iat)%vxc1 (cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
    allocate(Paw_an(iat)%vxct1(cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
   end if
  end if

  ! ==========================
  ! === Optional arguments ===
  ! ==========================

  ! * XC potential inside PAW spheres generated by valence electrons.
  Paw_an(iat)%has_vxcval=0
  if (PRESENT(has_vxcval)) then
   if (has_vxcval>0) then
    Paw_an(iat)%has_vxcval=1
    allocate(Paw_an(iat)%vxc1_val (cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
    allocate(Paw_an(iat)%vxct1_val(cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
   end if
  end if

  ! * XC potential for local exact exchange inside the sphere.
  if (Pawtab(itypat)%useexexch>0) then
   allocate(Paw_an(iat)%vxc_ex(cplex*Pawtab(itypat)%mesh_size,v_size,nspden))
  end if

  ! * Hartree potential LM-moments inside the sphere.
  Paw_an(iat)%has_vhartree=0
  if (PRESENT(has_vhartree)) then
   if (has_vhartree>0) then
    Paw_an(iat)%has_vhartree=1
! FIXME what about vht1?
!MG This is the coding PRESENTLY used in pawdenpot but the commented code should be the correct one
!MT: don't agreee for nspden (there is no dependance of vH^(1) with nspden)
    allocate(Paw_an(iat)%vh1(cplex*Pawtab(itypat)%mesh_size,1,1))
    !$allocate(Paw_an(iat)%vh1 (cplex*Pawtab(itypat)%mesh_size,lm_size,nspden))
    !$allocate(Paw_an(iat)%vht1(cplex*Pawtab(itypat)%mesh_size,lm_size,nspden))
   end if
  end if

  ! xc kernels inside the sphere.
  Paw_an(iat)%has_kxc=0
  if (PRESENT(has_kxc)) then
   if (has_kxc>0) then
    Paw_an(iat)%has_kxc=1
    allocate(Paw_an(iat)%kxc1 (cplex*Pawtab(itypat)%mesh_size,v_size,nkxc1))
    allocate(Paw_an(iat)%kxct1(cplex*Pawtab(itypat)%mesh_size,v_size,nkxc1))
   end if
  end if

 end do !iat

 DBG_EXIT("COLL")

end subroutine init_paw_an
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/print_pawtab
!! NAME
!! print_pawtab
!!
!! FUNCTION
!!  Print out the content of a pawtab datastructure
!!
!! INPUTS
!!  Pawtab<pawtab_type> Only for PAW, TABulated data initialized at start
!!
!! OUTPUT
!!  Only writing
!!
!! PARENTS
!!      bethe_salpeter,screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine print_pawtab(Pawtab,header,unit,prtvol,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: unit,prtvol
 character(len=4),optional,intent(in) :: mode_paral
 character(len=*),optional,intent(in) :: header
!arrays
 type(Pawtab_type) :: Pawtab(:)

!Local variables-------------------------------
!scalars
 integer :: ityp,ntypat,my_unt,my_prtvol
 character(len=4) :: my_mode
 character(len=500) :: msg

! *************************************************************************

 my_unt   =std_out; if (PRESENT(unit      )) my_unt   =unit
 my_prtvol=0      ; if (PRESENT(prtvol    )) my_prtvol=prtvol
 my_mode  ='COLL' ; if (PRESENT(mode_paral)) my_mode  =mode_paral

 write(msg,'(6a)')&
&  ' ==================================== ',ch10,&
&  ' ==== Info on PAW TABulated data ==== ',ch10,&
&  ' ==================================== ',ch10
 if (PRESENT(header)) msg=' ==== '//TRIM(ADJUSTL(header))//' ==== '
 call wrtout(my_unt,msg,my_mode)

 ntypat=SIZE(Pawtab(:))

 do ityp=1,ntypat

  ! Print out integer values (dimensions)
  write(*,*)'                                 '
  write(*,*)'  ****************************** '
  write(*,*)'  **** Atom type ',ityp,' ****   '
  write(*,*)'  ****************************** '
  write(*,*)'  Number of (n,l) elements ....................... ',Pawtab(ityp)%basis_size
  write(*,*)'  Number of (l,m,n) elements ..................... ',Pawtab(ityp)%lmn_size
  write(*,*)'  Number of (i,j) elements (packed form) ......... ',Pawtab(ityp)%ij_size
  write(*,*)'  Max L+1 leading to non-zero Gaunt .............. ',Pawtab(ityp)%l_size
  write(*,*)'  Max L+1 leading to non-zero Gaunt (pawlcutd) ... ',Pawtab(ityp)%lcut_size
  write(*,*)'  lmn2_size ...................................... ',Pawtab(ityp)%lmn2_size
  write(*,*)'  lmnmix_sz ...................................... ',Pawtab(ityp)%lmnmix_sz
  write(*,*)'  Size of radial mesh ............................ ',Pawtab(ityp)%mesh_size
  write(*,*)'  No of Q-points for tcorespl and tvalespl ....... ',Pawtab(ityp)%mqgrid
  write(*,*)'  No of Q-points for the radial shape functions .. ',Pawtab(ityp)%mqgrid_shp
  write(*,*)'  Radial shape function type ..................... ',Pawtab(ityp)%shape_type
  write(*,*)'  shape_lambda ................................... ',Pawtab(ityp)%shape_lambda
  write(*,*)'  Use pseudized core density ..................... ',Pawtab(ityp)%usetcore
  write(*,*)'  Use pseudized valence density .................. ',Pawtab(ityp)%usetvale
  write(*,*)'  Option for the use of hat density in XC terms .. ',Pawtab(ityp)%usexcnhat
  write(*,*)'  Use LDA+U ...................................... ',Pawtab(ityp)%usepawu
  if (Pawtab(ityp)%usepawu/=0) then
   write(*,*)'  L on which U is applied ........................ ',Pawtab(ityp)%lpawu
  end if
  write(*,*)'  Use Local Exact exchange ....................... ',Pawtab(ityp)%useexexch
  if (Pawtab(ityp)%useexexch/=0) then
   write(*,*)'  L on which local exact-exchange is applied ..... ',Pawtab(ityp)%lexexch
  end if
  if (Pawtab(ityp)%usepawu/=0.or.Pawtab(ityp)%useexexch/=0) then
   write(*,*)'  Number of (i,j) elements for PAW+U or EXX ..... ',Pawtab(ityp)%ij_proj
   write(*,*)'  Number of projectors on which U or EXX acts .... ',Pawtab(ityp)%nproju
  end if

  ! "Has" flags
  write(*,*)'  Has kij   ...................................... ',Pawtab(ityp)%has_kij
  write(*,*)'  Has nabla ...................................... ',Pawtab(ityp)%has_nabla
  !
  ! Real scalars
  write(*,*)'  1/q d(tNcore(q))/dq for q=0 .....................',Pawtab(ityp)%dncdq0
  write(*,*)'  1/q d(tNvale(q))/dq for q=0 .....................',Pawtab(ityp)%dnvdq0
  write(*,*)'  XC energy for the core density ..................',Pawtab(ityp)%exccore
  write(*,*)'  Mixing of exact exchange (PBE0) .................',Pawtab(ityp)%exchmix
  write(*,*)'  Radius of the PAW sphere ........................',Pawtab(ityp)%rpaw
  write(*,*)'  Compensation charge radius (if >rshp, g(r)=0) ...',Pawtab(ityp)%rshp !(if r>rshp, g(r)=zero)
  if (Pawtab(ityp)%shape_type==2) then
   write(*,*)'  Sigma parameter in gaussian shape function ......',Pawtab(ityp)%shape_sigma !(shape_type=2)
  end if
  if (Pawtab(ityp)%usepawu/=0) then
   write(*,*)'  Value of the U parameter [eV] ...................',Pawtab(ityp)%upawu*Ha_eV
   write(*,*)'  Value of the J parameter [eV] ...................',Pawtab(ityp)%jpawu*Ha_eV
  end if

 end do ! ityp
 !
 ! The other (huge) arrays are not reported..

end subroutine print_pawtab
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/print_paw_ij
!! NAME
!! print_paw_ij
!!
!! FUNCTION
!!  Print out the content of a paw_ij datastructure
!!
!! INPUTS
!! [unit]=the unit number for output
!! [pawprtvol]=verbosity level
!! [mode_paral]=either "COLL" or "PERS"
!!
!! OUTPUT
!! (Only writing)
!!
!! NOTES
!!  The implementation of the routine is not yet completed.
!!  This implementation does not work if Paw_ij contains a 1st-order Dij (DFPT) at a non-zero q
!!
!! PARENTS
!!      sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine print_paw_ij(Paw_ij,unit,pawprtvol,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: pawprtvol
 integer,optional,intent(in) :: unit
 character(len=4),optional,intent(in) :: mode_paral
!arrays
 type(Paw_ij_type),intent(in) :: Paw_ij(:)

!Local variables-------------------------------
 character(len=7),parameter :: dspin(6)=(/"up     ","down   ","up-up  ","dwn-dwn","up-dwn ","dwn-up "/)
!scalars
 integer :: cplex,cplex_dij,iatom,idij,lmn2_size,lmn_size,natom,nspden,nsploop,nsppol,my_unt
 integer :: opt_sym,tmp_cplex_dij,my_prtvol
 character(len=4) :: my_mode
 character(len=500) :: msg
!arrays
 integer,allocatable :: idum(:)
 real(dp),pointer :: dij2p(:)

! *************************************************************************

 DBG_ENTER("COLL")

 my_unt   =std_out; if (PRESENT(unit      )) my_unt   =unit
 my_prtvol=0      ; if (PRESENT(pawprtvol )) my_prtvol=pawprtvol
 my_mode  ='COLL' ; if (PRESENT(mode_paral)) my_mode  =mode_paral

 ! TODO: to be consistent, my_unt and my_mode should be passed to print_ij
 ! moreover the pointers should be nullified when paw_ij is initialized

 natom  = SIZE(Paw_ij)
 nsppol = Paw_ij(1)%nsppol
 nspden = Paw_ij(1)%nspden
 nsploop= nsppol; if (Paw_ij(1)%ndij==4) nsploop=4

 do iatom=1,natom

  lmn_size  = Paw_ij(iatom)%lmn_size
  lmn2_size = Paw_ij(iatom)%lmn2_size
  cplex_dij = Paw_ij(iatom)%cplex_dij
  cplex     = Paw_ij(iatom)%cplex

  ! ====================================
  ! === Loop over density components ===
  ! ====================================
  do idij=1,nsploop

   ! * Print title.
   if (ABS(my_prtvol)>=1) then
    if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
     if (nspden==2.and.nsppol==1) then
      write(msg,'(2a,i3,2a)')ch10,&
&      ' >>>>>>>>>> Atom ',iatom,':',ch10,&
&      ' (antiferromagnetism case: only one spin component)'
     else
      write(msg,'(2a,i3,3a)') ch10,&
&      ' >>>>>>>>>> Atom ',iatom,' (component ',TRIM(dspin(idij+2*(nsploop/4))),'):'
     end if
     call wrtout(my_unt,msg,my_mode)
    end if
   end if

   !if (abs(my_prtvol)>=1) then
   ! if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
   !  write(msg, '(a)') '   ************ Dij atomic (Dij0) ***********'
   !  call wrtout(my_unt,msg,my_mode)
   !  call print_ij(Pawtab(itypat)%dij0,lmn2_size,1,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1)
   ! end if
   !end if

   if (abs(my_prtvol)>=1) then
    if (Paw_ij(iatom)%has_dijhartree==2) then
     if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
      write(msg, '(a)')'   ************** Dij Hartree ***************'
      call wrtout(my_unt,msg,my_mode)
      call print_ij(Paw_ij(iatom)%dijhartree,lmn2_size,cplex,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1)
     end if
    end if
   end if

   if (Paw_ij(iatom)%has_dijxc>0) then
    if ((abs(my_prtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
      write(msg,'(a)')'   ****************** Dij_xc + Dijhat_xc ****************'
      call wrtout(my_unt,msg,my_mode)
      if (idij<=nsppol.or.idij==2) then
       opt_sym=2; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:cplex_dij,idij)
      else
       opt_sym=1; tmp_cplex_dij=cplex_dij
       dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:1,idij)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   if (Paw_ij(iatom)%has_dijxc_val>0) then
    if ((abs(my_prtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
      write(msg,'(a)')'   ****************** Dij_xcval ****************'
      call wrtout(my_unt,msg,my_mode)
      if (idij<=nsppol.or.idij==2) then
       opt_sym=2; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijxc_val(1:cplex_dij*lmn2_size:cplex_dij,idij)
      else
       opt_sym=1; tmp_cplex_dij=cplex_dij
       dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:1,idij)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   if (Paw_ij(iatom)%has_dijhat>0) then
    if ((abs(my_prtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
      write(msg,'(a)')'   ************* Dij_hat (Veff_ij) **********'
      call wrtout(my_unt,msg,my_mode)
      !if ((idij<=nsppol.or.idij==2).and.cplex==1)then
      if ((idij<=nsppol.or.idij==2))then
       opt_sym=2; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijhat(1:cplex_dij*lmn2_size:cplex_dij,idij)
       !call print_ij(dijhat(1:lmn2_size),lmn2_size,1,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1)
      else
        opt_sym=1; tmp_cplex_dij=cplex_dij
        dij2p => Paw_ij(iatom)%dijxc(1:cplex_dij*lmn2_size:1,idij)
       !call print_ij(dijhat,lmn2_size,cplex_dij,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1,opt_sym=1)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   if (Paw_ij(iatom)%has_dijso>0) then
    if (abs(my_prtvol)>=1) then
     if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
      write(msg,'(a)')'   ************** Dij SpinOrbit ************'
      call wrtout(my_unt,msg,my_mode)
      dij2p =>  Paw_ij(iatom)%dijso(:,idij)
      call print_ij(dij2p,lmn2_size,cplex_dij,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1,opt_sym=3)
     end if
    end if
   end if

   if (Paw_ij(iatom)%has_dijU>0) then
    if ((abs(my_prtvol)>=1).and.(idij<=2.or.nspden==4)) then
     if (iatom==1.or.iatom==natom.or.my_prtvol<0) then
      write(msg,'(a)')'   ************* Dij_LDA+U (dijpawu) **********'
      call wrtout(my_unt,msg,my_mode)
      if (idij<=nsppol.or.idij==2) then
       opt_sym=2; tmp_cplex_dij=1
       dij2p => Paw_ij(iatom)%dijU(1:cplex_dij*lmn_size:cplex_dij,idij)
       !call print_ij(dijpawu(1:lmn2_size),lmn2_size,1,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1)
      else
       opt_sym=1; tmp_cplex_dij=cplex_dij
       dij2p => Paw_ij(iatom)%dijU(1:cplex_dij*lmn2_size:1,idij)
       !call print_ij(dijpawu,lmn2_size,cplex_dij,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1,opt_sym=1)
      end if
      call print_ij(dij2p,lmn2_size,tmp_cplex_dij,lmn_size,1,-1,idum,0,my_prtvol,idum,-1.d0,1,opt_sym=opt_sym)
     end if
    end if
   end if

   !TODO Dij_Exact_Exchange is not printed because there is no entry in the objects
   ! Add new entries in Paw_ij

  end do !idij
 end do !iat

 write(msg,'(a)')' '
 call wrtout(my_unt,msg,my_mode)

 DBG_ENTER("COLL")

end subroutine print_paw_ij
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/pawfgrtab_free
!! NAME
!! pawfgrtab_free
!!
!! FUNCTION
!!  Free all dynamic memory stored in a pawfgrtab datastructure
!!
!! SIDE EFFECTS
!!  Pawfgrt(natom) <type(pawfgrtab_type)>=atomic data given on fine rectangular grid
!!
!! PARENTS
!!      bethe_salpeter,calc_sigx_me,classify_bands,denfgr,respfn,scfcv
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_free(Pawfgrt)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Pawfgrtab_type),intent(inout) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom

! *************************************************************************

 DBG_ENTER("COLL")

 !@Pawfgrtab_type
 natom=SIZE(Pawfgrt)
 do iat=1,natom
  if (associated(Pawfgrt(iat)%ifftsph)) deallocate(Pawfgrt(iat)%ifftsph)
  if (associated(Pawfgrt(iat)%gylm   )) deallocate(Pawfgrt(iat)%gylm   )
  if (associated(Pawfgrt(iat)%gylmgr )) deallocate(Pawfgrt(iat)%gylmgr )
  if (associated(Pawfgrt(iat)%gylmgr2)) deallocate(Pawfgrt(iat)%gylmgr2)
  if (associated(Pawfgrt(iat)%nhatfr )) deallocate(Pawfgrt(iat)%nhatfr )
  if (associated(Pawfgrt(iat)%rfgd   )) deallocate(Pawfgrt(iat)%rfgd   )
  if (associated(Pawfgrt(iat)%expiqr))  deallocate(Pawfgrt(iat)%expiqr )
 end do

 DBG_EXIT("COLL")

end subroutine pawfgrtab_free
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/pawfgrtab_init
!! NAME
!! pawfgrtab_init
!!
!! FUNCTION
!!  Initialize a pawfgrtab datastructure
!!
!! OUTPUT
!!  Pawfgrt(natom) <type(pawfgrtab_type)>=atomic data given on fine rectangular grid
!!
!! PARENTS
!!      bethe_salpeter,calc_sigx_me,classify_bands,denfgr,respfn,scfcv
!!      screening,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_init(Pawfgrt,cplex,l_size_atm,nspden)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: cplex,nspden
!arrays
 integer,intent(in) :: l_size_atm(:)
 type(Pawfgrtab_type),intent(out) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom
 character(len=500) :: msg

! *************************************************************************

 natom=SIZE(Pawfgrt)
 if (natom/=SIZE(l_size_atm)) then
  msg='Sizes of assumed shape arrays do not match'
  MSG_BUG(msg)
 end if

 !@Pawfgrtab_type
 call pawfgrtab_nullify(Pawfgrt)

 do iat=1,natom
  Pawfgrt(iat)%cplex             = cplex
  Pawfgrt(iat)%nspden            = nspden
  Pawfgrt(iat)%l_size            = l_size_atm(iat)
  Pawfgrt(iat)%nfgd              = 0  ; allocate(Pawfgrt(iat)%ifftsph(0)    )
  Pawfgrt(iat)%gylm_allocated    = 0  ; allocate(Pawfgrt(iat)%gylm(0,0)     )
  Pawfgrt(iat)%gylmgr_allocated  = 0  ; allocate(Pawfgrt(iat)%gylmgr(0,0,0) )
  Pawfgrt(iat)%gylmgr2_allocated = 0  ; allocate(Pawfgrt(iat)%gylmgr2(0,0,0))
  Pawfgrt(iat)%nhatfr_allocated  = 0  ; allocate(Pawfgrt(iat)%nhatfr(0,0)   )
  Pawfgrt(iat)%rfgd_allocated    = 0  ; allocate(Pawfgrt(iat)%rfgd(0,0)     )
  Pawfgrt(iat)%expiqr_allocated  = 0  ; allocate(Pawfgrt(iat)%expiqr(0,0)   )
 end do

end subroutine pawfgrtab_init
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/pawfgrtab_print
!! NAME
!! pawfgrtab_print
!!
!! FUNCTION
!!  Reports basic info on the pawfgrtab datatype.
!!
!! INPUTS
!! Pawfgrt<pawfgrtab_type>=The datatype to be printed
!! [mode_paral]=either "COLL" or "PERS", "COLL" if None.
!! [unit]=Unit number for output, std_out if None.
!! [prtvol]=Verbosity level, lowest if None.
!!
!! OUTPUT
!! (only writing)
!!
!! PARENTS
!!      calc_sigx_me,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_print(Pawfgrt,unit,prtvol,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: prtvol,unit
 character(len=4),optional,intent(in) :: mode_paral
!arrays
 type(Pawfgrtab_type),intent(inout) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom,my_unt,my_prtvol
 character(len=4) :: my_mode
 character(len=500) :: msg

! *************************************************************************

 my_prtvol=0      ; if (PRESENT(prtvol    )) my_prtvol=prtvol
 my_unt   =std_out; if (PRESENT(unit      )) my_unt   =unit
 my_mode  ='COLL' ; if (PRESENT(mode_paral)) my_mode  =mode_paral

 natom=SIZE(Pawfgrt)

 write(msg,'(3a)')ch10,' === Content of the pawfgrtab datatype === ',ch10
 call wrtout(my_unt,msg,my_mode)
 do iat=1,natom
   write(msg,'(3(2a,i5))')ch10,&
&    ' > For atom number : ',iat,ch10,&
&    '    1+ Max l in Gaunt coefficients ',Pawfgrt(iat)%l_size,ch10,&
&    '    Number of fine FFT points in PAW sphere ',Pawfgrt(iat)%nfgd
   call wrtout(my_unt,msg,my_mode)

   if (my_prtvol>=3) then
     write(msg,'(a,6(a,i2,a))')ch10,&
&      '    rfgd_allocated    : ',Pawfgrt(iat)%rfgd_allocated,ch10,&
&      '    gylm_allocated    : ',Pawfgrt(iat)%gylm_allocated,ch10,&
&      '    gylmgr_allocated  : ',Pawfgrt(iat)%gylmgr_allocated,ch10,&
&      '    gylmgr2_allocated : ',Pawfgrt(iat)%gylmgr2_allocated,ch10,&
&      '    nhatgr_allocated  : ',Pawfgrt(iat)%nhatfr_allocated,ch10,&
&      '    expiqr_allocated  : ',Pawfgrt(iat)%expiqr_allocated,ch10
     call wrtout(my_unt,msg,my_mode)
   end if

!  These huge arrays are not printed out!
!  Pawfgrt(iat)%ifftsph
!  Pawfgrt(iat)%rfgd
!  Pawfgrt(iat)%gylm
!  Pawfgrt(iat)%gylmgr
!  Pawfgrt(iat)%gylmgr2
!  Pawfgrt(ia)%nhatfr
!  Pawfgrt(ia)%expiqr
 end do

end subroutine pawfgrtab_print
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/pawfgrtab_nullify
!! NAME
!! pawfgrtab_nullify
!!
!! FUNCTION
!!  Nullify the pointers in a pawfgrtab datastructure
!!
!! SIDE EFFECTS
!!  Pawfgrt(:) <type(pawfgrtab_type)>=atomic data given on fine rectangular grid
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      m_paw_toolbox
!!
!! CHILDREN
!!
!! SOURCE

subroutine pawfgrtab_nullify(Pawfgrt)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Pawfgrtab_type),intent(inout) :: Pawfgrt(:)

!Local variables-------------------------------
!scalars
 integer :: iat,natom

! *************************************************************************

 !@Pawfgrtab_type
 natom=SIZE(Pawfgrt)
 do iat=1,natom
  nullify(Pawfgrt(iat)%ifftsph)
  nullify(Pawfgrt(iat)%gylm   )
  nullify(Pawfgrt(iat)%gylmgr )
  nullify(Pawfgrt(iat)%gylmgr2)
  nullify(Pawfgrt(iat)%nhatfr )
  nullify(Pawfgrt(iat)%rfgd   )
  nullify(Pawfgrt(iat)%expiqr )
  Pawfgrt(iat)%nfgd              = 0
  Pawfgrt(iat)%gylm_allocated    = 0
  Pawfgrt(iat)%gylmgr_allocated  = 0
  Pawfgrt(iat)%gylmgr2_allocated = 0
  Pawfgrt(iat)%nhatfr_allocated  = 0
  Pawfgrt(iat)%rfgd_allocated    = 0
  Pawfgrt(iat)%expiqr_allocated  = 0
 end do

end subroutine pawfgrtab_nullify
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/reset_paw_an_flags
!! NAME
!! reset_paw_an_flags
!!
!! FUNCTION
!!  Reset the flags in a paw_an datastructure
!!
!! SIDE EFFECTS
!!  Paw_an_flags<type(Paw_an_flags_type)>=flags in a paw_an datastructure
!!                                        All "has" flags set to 0.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine reset_paw_an_flags(Paw_an_flags)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_flags_type),intent(inout) :: Paw_an_flags

! *************************************************************************

 ! @Paw_an_flags_type
 Paw_an_flags%has_kxc     =0
 Paw_an_flags%has_vhartree=0
 Paw_an_flags%has_vxc     =0
 Paw_an_flags%has_vxcval  =0

end subroutine reset_paw_an_flags
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/copy_paw_an_flags
!! NAME
!! copy_paw_an_flags
!!
!! FUNCTION
!!  Copy the flags in a paw_an datastructure
!!
!! INPUT
!!  an_flags_in<type(Paw_an_flags_type)>=(input) flags in a paw_an datastructure
!!  Copy a Paw_an_flags_type
!!
!! OUTPUT
!!  an_flags_out<type(Paw_an_flags_type)>=(output) flags in a paw_an datastructure
!!  Copy a Paw_an_flags_type
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine copy_paw_an_flags(an_flags_in, an_flags_out)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!arrays
 type(Paw_an_flags_type),intent(in)    :: an_flags_in
 type(Paw_an_flags_type),intent(inout) :: an_flags_out

! *************************************************************************

 ! @Paw_an_flags_type
 an_flags_out%has_kxc      = an_flags_in%has_kxc
 an_flags_out%has_vhartree = an_flags_in%has_vhartree
 an_flags_out%has_vxc      = an_flags_in%has_vxc
 an_flags_out%has_vxcval   = an_flags_in%has_vxcval

end subroutine copy_paw_an_flags
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/destroy_pawang
!! NAME
!! destroy_pawang
!!
!! FUNCTION
!!  Free all dynamic memory and reset all flags stored in a pawang datastructure
!!
!! SIDE EFFECTS
!!  Pawang <type(pawang_type)>=ANGular mesh discretization and related data
!!
!! PARENTS
!!      loper3,pawalloc
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_pawang(Pawang)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(Pawang_type),intent(inout) :: Pawang

!Local variables-------------------------------
 integer :: ierr

! *************************************************************************

 DBG_ENTER("COLL")

 !@Pawang_type
 if (associated(pawang%angwgth))   deallocate(pawang%angwgth)
 if (associated(pawang%anginit))   deallocate(pawang%anginit)
 if (associated(pawang%zarot))     deallocate(pawang%zarot)
 if (associated(pawang%gntselect)) deallocate(pawang%gntselect,STAT=ierr)
 if (associated(pawang%realgnt))   deallocate(pawang%realgnt,STAT=ierr)
 if (associated(pawang%ylmr))      deallocate(pawang%ylmr)
 if (associated(pawang%ylmrgr))    deallocate(pawang%ylmrgr)
 if (associated(pawang%ls_ylm))    deallocate(pawang%ls_ylm)

 pawang%angl_size =0
 pawang%ylm_size  =0
 pawang%gnt_option=0
 pawang%use_ls_ylm=0

 DBG_EXIT("COLL")

end subroutine destroy_pawang
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/init_pawang
!! NAME
!! init_pawang
!!
!! FUNCTION
!!  Initialize a pawang datastructure
!!
!! INPUTS
!!  gnt_option=flag activated if pawang%gntselect and pawang%realgnt have to be allocated
!!             also determine the size of these pointers
!!  lmax=maximum value of angular momentum l+1
!!  ngnt=number of non-zero Gaunt coefficients
!!  nphi,ntheta=dimensions of paw angular mesh
!!  nsym=number of symetries
!   pawxcdev=choice of XC development (0=no dev. (use of angular mesh) ; 1 or 2=dev. on moments)
!!  use_ls_ylm=flag activated if pawang%ls_ylm has to be allocated
!!  use_ylm=flag activated if pawang%ylmr and pawang%ylmrgr have to be allocated
!!  xclevel=XC functional level
!!
!! OUTPUT
!!  Pawang <type(pawang_type)>=ANGular mesh discretization and related data
!!
!! PARENTS
!!      loper3
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_pawang(Pawang,gnt_option,lmax,ngnt,nphi,nsym,ntheta,pawxcdev,use_ls_ylm,use_ylm,xclevel)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(Pawang_type),intent(inout) :: Pawang
 integer,intent(in) :: gnt_option,lmax,ngnt,nphi,nsym,ntheta,pawxcdev,use_ls_ylm,use_ylm,xclevel

!Local variables-------------------------------

! *************************************************************************

 !@Pawang_type

 Pawang%nphi=nphi
 Pawang%ntheta=ntheta
 Pawang%angl_size=0
 if (pawxcdev==0) Pawang%angl_size=ntheta*nphi
 if (Pawang%angl_size>0) then
   allocate(Pawang%anginit(3,Pawang%angl_size))
   allocate(Pawang%angwgth(Pawang%angl_size))
 else
   nullify(Pawang%anginit)
   nullify(Pawang%angwgth)
 end if

 Pawang%l_max=lmax
 Pawang%l_size_max=2*lmax-1
 if (use_ylm>0) then
   if (xclevel==2) Pawang%ylm_size=(Pawang%l_size_max+1)**2
   if (xclevel/=2) Pawang%ylm_size= Pawang%l_size_max   **2
   allocate(Pawang%ylmr(Pawang%ylm_size,Pawang%angl_size))
   if (xclevel==2) then
     allocate(Pawang%ylmrgr(3,Pawang%ylm_size,Pawang%angl_size))
   else
     nullify(Pawang%ylmrgr)
   end if
 else
   Pawang%ylm_size=0
   nullify(Pawang%ylmr)
   nullify(Pawang%ylmrgr)
 end if

 Pawang%ngnt=ngnt
 Pawang%gnt_option=gnt_option
 if (gnt_option>0) then
  if (gnt_option==2) then
    allocate(Pawang%gntselect((2*Pawang%l_size_max-1)**2,&
&     ((2*Pawang%l_max-1)**2)*((2*Pawang%l_max-1)**2+1)/2))
  else
    allocate(Pawang%gntselect((Pawang%l_size_max)**2,&
&     (Pawang%l_max**2)*(Pawang%l_max**2+1)/2))
  end if
  if (ngnt>0) then
    allocate(Pawang%realgnt(ngnt))
  else
    nullify(Pawang%realgnt)
  end if
 else
   nullify(Pawang%gntselect)
   nullify(Pawang%realgnt)
 end if

 Pawang%use_ls_ylm=use_ls_ylm
 if (use_ls_ylm>0) then
  allocate(pawang%ls_ylm(2,Pawang%l_max**2*(Pawang%l_max**2+1)/2,2))
 else
  nullify(pawang%ls_ylm)
 end if

 Pawang%nsym=nsym
 if (nsym>0) then
   allocate(Pawang%zarot(Pawang%l_size_max,Pawang%l_size_max,Pawang%l_max,nsym))
 else
   nullify(Pawang%zarot)
 end if

end subroutine init_pawang
!!***

!----------------------------------------------------------------------

!!****f* m_paw_toolbox/get_dimcprj
!! NAME
!! get_dimcprj
!!
!! FUNCTION
!!  Helper function returning the number of lmn components in the <p_{lmn}^i|\psi> for the i-th atom.
!!  Used to initialize the dimensioning array that is passed to the cprj_alloc routines when the
!!  Cprj_type structure is allocated and initialized.
!!
!! INPUT
!! Pawtab(ntypat)<pawtab_type>=PAW tabulated starting data.
!! Cryst<Crystal_structure>=Structure gathering info on the unit cell and the symmetries.
!! sort_mode(len=*)=String defining the sorting of the atoms in the Cprj arrays.
!!   Two modes are possible:
!!   -- "O[rdered]", if atoms are sorted by atom type.
!!   -- "R[andom]", if atoms are sorted randomly i.e. according the values of typat specified in the input file.
!!
!! OUTPUT
!!  dimcprj(natom)=Number of nlm elements in the <p_{lmn}^i|\psi> matrix elements for i=1,...,natom.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine get_dimcprj(Pawtab,Cryst,sort_mode,dimcprj)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 type(crystal_structure),intent(in) :: Cryst
 character(len=*),intent(in) :: sort_mode
!arrays
 integer,intent(out) :: dimcprj(Cryst%natom)
 type(Pawtab_type),intent(in) :: Pawtab(Cryst%ntypat)

!Local variables-------------------------------
 integer :: iatom,itypat
 character(len=500) :: msg

! *************************************************************************

 SELECT CASE (toupper(sort_mode(1:1)))

 CASE ("O") ! Ordered by atom-type

  iatom=0
  do itypat=1,Cryst%ntypat
   dimcprj(iatom+1:iatom+Cryst%nattyp(itypat))=Pawtab(itypat)%lmn_size
   iatom=iatom+Cryst%nattyp(itypat)
  end do

 CASE ("R") ! Randomly ordered (typat from input file)

  do iatom=1,Cryst%natom
   itypat=Cryst%typat(iatom)
   dimcprj(iatom)=Pawtab(itypat)%lmn_size
  end do

 CASE DEFAULT
  msg = " Wrong value for sort_mode: "//TRIM(sort_mode)
  MSG_ERROR(msg)
 END SELECT

end subroutine get_dimcprj
!!***

END MODULE m_paw_toolbox
!!***
