!{\src2tex{textfont=tt}}
!****m* ABINIT/m_wfs
!! NAME
!!  m_wfs
!!
!! FUNCTION
!!  This module contains the declaration of the wfs_descriptor object and its methods.
!!
!! COPYRIGHT
!! Copyright (C) 2008-2010 ABINIT group (MG, FB)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_wfs

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_xmpi
 use m_errors
 use m_iterators

 use m_fstrings,       only : toupper, starts_with
 use m_io_tools,       only : get_unit
 use m_numeric_tools,  only : imin_loc
 use m_blas,           only : xcopy, xdotc
 use m_fft_mesh,       only : print_ngfft, rotate_fft_mesh, ceigr
 use m_fftw3,          only : fftw3_fftpad_cplx
 use m_crystal,        only : crystal_structure
 use m_gsphere,        only : get_kg
 use m_bz_mesh,        only : bz_mesh_type, get_bz_item

 implicit none

 private 
!!***

 ! Flags giving the status of the local %ug, %ur %cprj buffers.
 integer,public,parameter :: WFD_NOWAVE   =0
 integer,public,parameter :: WFD_ALLOCATED=1 
 integer,public,parameter :: WFD_STORED   =2

 integer,public,parameter :: CPR_RANDOM   =1
 integer,public,parameter :: CPR_SORTED   =2

 ! Flags giving the status of the %ug buffers on the different nodes.
 integer,parameter :: NO_BKS  = 0
 integer,parameter :: HAS_BKS = 1

!----------------------------------------------------------------------

!!****t* m_wfs/kdata_t
!! NAME
!! kdata_t
!!
!! FUNCTION
!! Datatype storing k-dependent quantities and tables needed for performing the zero-padded FFT of wavefunctions.
!!
!! SOURCE

 type,public :: kdata_t

   integer :: istwfk
   ! Storage mode for this k point.

   integer :: npw
   ! Number of plane-waves for this k-point.
  
   integer :: dim_fnlylm
   ! Dimension of the fnlylm array.

   integer, pointer :: kg_k(:,:)
   ! kg_k(3,npw)
   ! G vector coordinates in reduced cordinates.

   integer,pointer :: igfft0(:)   SET2NULL
   ! igfft0(npw)
   ! Index of the G-sphere in the FFT box. 
                                                                            
   integer,pointer :: gbound(:,:)     SET2NULL
   ! gbound(2*mgfft+8,2))
   ! The boundary of the basis sphere of G vectors at a given k point.
   ! for use in improved zero padding of ffts in 3 dimensions.

   !$real(dp) :: kpoint(3)

   !real(dp),pointer :: ph1d(:,:)    SET2NULL
   ! ph1d(2,3*(2*mgfft+1)*natom) 
   ! 1-dim structure factor phase information.

   real(dp),pointer :: ph3d(:,:,:)
   ! ph3d(2,npw,natom)
   ! 3-dim structure factors, for each atom and each plane wave.

   real(dp),pointer :: phkxred(:,:)     SET2NULL
   ! phkxred(2,natom))
   ! e^{ik.Ra} for each atom. Packed according to the atom type (atindx).

   real(dp), pointer :: fnlylm(:,:,:,:)    SET2NULL
   ! fnlylm(npw,dim_fnlylm,lmnmax,ntypat)
   ! nonlocal form factors

   !$real(dp),pointer :: ylm(:,:)        SET2NULL
   ! ylm(npw,mpsang**2*useylm)
   ! Real spherical harmonics for each G

   !$integer :: ngfft(18)
   ! Information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft.

   !$integer :: mgfft  
   ! MAXVAL(ngfft(1:3)), used to dimension some arrays.

   !$integer :: nfftot 
   ! PRODUCT(ngfft(1:3)), ie the total number of FFT points. 

   !$integer :: nfft   
   ! The number of points treated by this node.

 end type kdata_t

 public :: kdata_init
 public :: kdata_nullify
 public :: kdata_free

 interface kdata_nullify
   module procedure nullify_kdata_0D
   module procedure nullify_kdata_1D
 end interface kdata_nullify

 interface kdata_free
   module procedure destroy_kdata_0D
   module procedure destroy_kdata_1D
 end interface kdata_free
!!***

!----------------------------------------------------------------------

!!****t* m_wfs/wave_t
!! NAME
!! wave_t
!!
!! FUNCTION
!!  Structure used to store a single wavefunction in reciprocal space and, optionally, its real space representation.
!!
!! SOURCE

 type,public :: wave_t

  !integer :: npw_k
  !integer :: nfft
  !integer :: nspinor
  !integer :: natom

  integer :: has_ug=WFD_NOWAVE
  ! Flag giving the status of ug.

  integer :: has_ur=WFD_NOWAVE
  ! Flag giving the status of ur.

  integer :: has_cprj=WFD_NOWAVE
  ! Flag giving the status of cprj.

  integer :: cprj_order=CPR_RANDOM
  ! Flag defining whether cprj are sorted by atom type or ordered according 
  ! to the typat variable used in the input file.

  complex(gwpc),pointer :: ug(:)  SET2NULL
  ! ug(npw_k*nspinor)
  ! The periodic part of the Bloch wavefunction in reciprocal space.

  complex(gwpc),pointer :: ur(:)  SET2NULL
  ! ur(nfft*nspinor)
  ! The periodic part of the Bloch wavefunction in real space.

  type(cprj_type),pointer :: Cprj(:,:)  SET2NULL
  ! Cprj(natom,nspinor)
  ! PAW projected wave function <Proj_i|Cnk> with all NL projectors.

 end type wave_t
!!***

 public :: wave_nullify
 public :: wave_init
 public :: wave_free

 interface wave_nullify
   module procedure nullify_wave_0D
   module procedure nullify_wave_3D
 end interface wave_nullify

 interface wave_init
   module procedure init_wave_0D
 end interface wave_init
                                     
 interface wave_free
   module procedure destroy_wave_0D
   module procedure destroy_wave_3D
 end interface wave_free

!!***

!----------------------------------------------------------------------

!!****t* m_wfs/wfs_descriptor
!! NAME
!! wfs_descriptor
!!
!! FUNCTION
!! Container gathering information on the set of wavefunctions treated by
!! this node as well as their distribution inside the MPI communicator.
!!
!! SOURCE

 type,public :: wfs_descriptor

  integer :: lmnmax
  integer :: mband              ! MAX(nband)
  integer :: mgfft              ! Maximum size of 1D FFTs
  !$integer :: mpsang
  integer :: natom
  integer :: nfft               ! Number of FFT points treated by this processor
  integer :: nfftot             ! Total number of points in the FFT grid
  integer :: nkibz              ! Number of irreducible k-points
  integer :: npwwfn             ! Number of G vectors for wavefunctions
  integer :: nspden             ! Number of independent spin-density components
  integer :: nspinor            ! Number of spinorial components
  integer :: nsppol             ! Number of independent spin polarizations
  integer :: ntypat
  integer :: paral_kgb          ! Option for kgb parallelism
  integer :: usepaw             ! 1 if PAW is used, 0 otherwise.

  integer :: comm               ! The MPI communicator.
  integer :: master             ! The rank of master node in comm.
  integer :: my_rank            ! The rank of my processor inside the MPI communicator comm.
  integer :: nproc              ! The number of processors in MPI comm.
  
  !$ integer :: cplex
  ! cplex= if 1 , wavefunctions are real, if 2 they are complex
  ! In systems with time-reversal and spatial inversion, wavefunctions are real.
  ! One might use this to reduce memory in wave_t.

  real(dp) :: ecut              ! Cutoff for plane wave basis set.

!arrays
  integer :: ngfft(18)
  ! Information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft

  integer :: nloalg(5)
  ! Governs the choice of the algorithm for nonlocal operator. See doc.

  integer,pointer :: igfft0(:)      SET2NULL
  ! igfft0(npwwfn) 
  ! Index of G vectors in the FFT array.
  ! TODO This is redundant and should be removed when k-centered G-sphere will be used.

  integer,pointer :: gvec(:,:)  SET2NULL        
  ! gvec(3,npwwfn)
  ! Reduced coordinates of the planewaves.
  ! TODO This is redundant and should be removed when k-centered G-sphere will be used.

  integer,pointer :: gbound(:,:) SET2NULL
  ! gbound(2*mgfft+8,2)
  ! The boundary of the basis sphere of G-vectors (at a given k point) 
  ! for use in improved zero padding of ffts in 3 dimensions. See sphereboundary.
  ! TODO This is redundant and should be removed when k-centered G-sphere will be used.

  integer :: mg0(3)   
  ! For each reduced direction gives the max G0 component to account for umklapp processes.
  ! TODO to be removed.

  integer,pointer :: irottb(:,:)  SET2NULL
  ! irottb(nfftot,nsym)
  ! Index of $R^{-1}(r-\tau)$ in the FFT box.

  integer,pointer :: istwfk(:)   SET2NULL
  ! istwfk(nkibz)
  ! Storage mode for this k-point. At present only istwfk==1 is supported.
  ! TODO This is redundant and should be removed when k-centered G-sphere will be used.

  integer,pointer :: nband(:,:)    SET2NULL
   ! nband(nkibz,nsppol)
   ! Number of bands at each k-point and spin.

  integer,pointer :: indlmn(:,:,:)  SET2NULL
  ! indlmn(6,lmnmax,ntypat)
  ! array giving l,m,n,lm,ln,spin for i=ln  (if useylm=0)
  !                                or i=lmn (if useylm=1)

  integer,pointer :: nlmn_atm(:)  SET2NULL
  ! nlmn_atm(natom)
  ! Number of (n,l,m) channels for each atom. Only for PAW

  integer,pointer :: nlmn_sort(:)  SET2NULL
  ! nlmn_sort(natom)
  ! Number of (n,l,m) channels for each atom (sorted by atom type). Only for PAW

  integer,pointer :: nlmn_type(:)  SET2NULL
  ! nlmn_type(ntypat)
  ! Number of (n,l,m) channels for each type of atom. Only for PAW.

  !$integer,pointer :: typat(:)  SET2NULL
  ! typat(natom) 
  ! The type of each atom.

! BEGIN dangerous variables introduced to simplify the transition to the k-centered basis set
! CAVEAT: __DO NOT__ USE THEM, Unless you really know what you are doing!

  integer,pointer  :: npwarr(:)   SET2NULL
  ! npwarr(nkibz)
  ! Number of plane waves for this k-point.
 
!END dangerous variables

  real(dp),pointer :: kibz(:,:)   SET2NULL
  ! kibz(3,nkibz)
  ! Reduced coordinates of the k-points in the IBZ.

  integer,pointer :: bks_tab(:,:,:,:)   SET2NULL
  ! bks_tab(mband,nkibz,nsppol,0:nproc-1)
  ! Global table used to keep trace of the distribution of the (b,k,s) on each node inside Wfd%comm.
  ! 1 if the node has this state. 0 otherwise.
  ! A node owns a wavefunction if the corresponding ug is allocated AND computed.
  ! If a node owns ur but not ug, or ug is just allocated then its entry in the table will be zero.

  real(dp),pointer :: ph1d(:,:)    SET2NULL
   ! ph1d(2,3*(2*mgfft+1)*natom) 
   ! 1-dim structure factor phase information.

  logical,pointer :: keep_ur(:,:,:) SET2NULL   
  ! keep(mband,nkibz,nsppol)
  ! Storage strategy: keep or not keep calculated u(r) in memory.

  !$type(mpicomm_t),pointer :: comm_spin(:)
  ! comm(0:nsppol)
  ! MPI communicators. 
  ! comm(1), comm(2) are the MPI communicators of the nodes treating the different spins.
  ! comm(0) is the MPI communicator enclosing all the nodes i.e. the union of the spin communicators. 

  type(kdata_t),pointer :: Kdata(:)  SET2NULL
   ! Kdata(nkibz)
   ! datatype storing k-dependent quantities.

  type(wave_t),pointer :: Wave(:,:,:)  SET2NULL
   ! Wave(mband,nkibz,nsppol)
   ! Array of structures storig the periodic part of the wavefunctions in reciprocal- and real-space.

  type(MPI_type) :: MPI_enreg
   ! The MPI_type structured datatype gather different information about the MPI parallelisation : 
   ! number of processors, the index of my processor, the different groups of processors, etc ...

 end type wfs_descriptor
!!***

 public :: wfd_init                ! Main creation method. 
 public :: wfd_destroy             ! Destructor.
 public :: wfd_reset_ur            ! Reinitialize memory storage of u(r).
 public :: wfd_get_ur              ! Get one wavefunction in real space from its (b,k,s) indeces.
 public :: wfd_get_cprj            ! Get one PAW projection <Proj_i|Cnk> with all NL projectors from its (b,k,s) indeces.
 public :: wfd_change_ngfft        ! Reinitialize internal FFT tables. 
 public :: wfd_nullify             ! Set all pointers to null()
 public :: wfd_print               ! Printout of basic info.
 public :: fft_onewfn              ! Helper function performing a single FFT from G to R.
 public :: wfd_mkall_ur            ! Calculate all ur owned by this node at once.
 public :: wfd_ug2cprj             ! Get PAW cprj from its (b,k,s) indeces.
 public :: wfd_ptr_ug              ! Return a pointer to ug from its (b,k,s) indeces. Use it carefully!
 public :: wfd_ihave_ug            ! .TRUE. if the node has this ug with the specified status.
 public :: wfd_ihave_ur            ! .TRUE. if the node has this ur with the specified status.
 public :: wfd_ihave_cprj          ! .TRUE. if the node has this cprj with the specified status.
 public :: wfd_mybands   
 public :: wfd_distribute_bands
 public :: wfd_iterator_bks
 public :: wfd_bks_distrb
 public :: wfd_update_bkstab
 public :: wfd_rotate
 public :: wfd_sanity_check
 public :: wfd_gamma2k
 public :: wfd_distribute_bbp
 public :: wfd_distribute_kb_kpbp
 public :: wfd_iam_master
 public :: wfd_test_ortho
 public :: wfd_barrier
 public :: wfd_sym_ur

!----------------------------------------------------------------------

CONTAINS  !==============================================================================
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/nullify_kdata_0D
!! NAME
!!  nullify_kdata_0D
!!
!! FUNCTION
!!  Set all pointers to null.
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine nullify_kdata_0D(Kdata)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(kdata_t),intent(inout) :: Kdata

!************************************************************************

 !@kdata_t

!integer pointers
 nullify(Kdata%kg_k   )
 nullify(Kdata%igfft0 )
 nullify(Kdata%gbound )

!real pointers
 nullify(Kdata%ph3d   )
 nullify(Kdata%phkxred)
 nullify(Kdata%fnlylm )

end subroutine nullify_kdata_0D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/nullify_kdata_1D
!! NAME
!!  nullify_kdata_1D
!!
!! FUNCTION
!!  Set all pointers to null.
!!

subroutine nullify_kdata_1D(Kdata)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(kdata_t),intent(inout) :: Kdata(:)

!Local variables ------------------------------
!scalars
 integer :: ik

!************************************************************************

 do ik=1,SIZE(Kdata)
   call nullify_kdata_0D(Kdata(ik))
 end do

end subroutine nullify_kdata_1D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/kdata_init
!! NAME
!!  kdata_init
!!
!! FUNCTION
!!  Main creation method for the kdata_t datatype.
!!
!! PARENTS
!!      bloch_interp,debug_tools,m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine kdata_init(Kdata,Cryst,Psps,kpoint,istwfk,ngfft,MPI_enreg,ecut,kg_k)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_ffts
 use interfaces_56_recipspace
 use interfaces_65_nonlocal
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istwfk
 real(dp),optional,intent(in) :: ecut
 type(crystal_structure),intent(in) :: Cryst
 type(pseudopotential_type),intent(in) :: Psps
 type(kdata_t),intent(inout) :: Kdata
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 integer,optional,target,intent(in) :: kg_k(:,:)
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: kpoint(3)

!Local variables ------------------------------
!scalars
 integer,parameter :: dum_unkg=0,dum_unylm=0,ider0=0,idir0=0
 integer :: mpw_,istat,npw_k,dimffnl,useylmgr,nkpg,iatom 
 integer :: mkmem_,nkpt_,optder,mgfft
 integer :: iatm,matblk
 real(dp) :: arg
 !character(len=500) :: msg
!arrays
 integer :: nband_(1),npwarr_(1) !dum_kg(3,0),
 integer,pointer :: my_kg_k(:,:)
 real(dp),allocatable :: ylm_k(:,:),ylmgr_k(:,:,:),kpg_k(:,:)
 !real(dp),pointer :: ph3d(:,:,:) !,fnlylm_k(:,:,:,:)
 real(dp),allocatable :: ph1d(:,:)
 logical,allocatable :: kg_mask(:)

!************************************************************************

 !@kdata_t
 Kdata%istwfk = istwfk     

 if (PRESENT(ecut)) then ! Calculate G-sphere from input ecut.
  call get_kg(kpoint,istwfk,ecut,Cryst%gmet,npw_k,my_kg_k) 
 else if (PRESENT(kg_k)) then ! Use input g-vectors.
   my_kg_k => kg_k
   npw_k = SIZE(kg_k,DIM=2) 
 else
   MSG_ERROR("Either ecut or kg_k must be present")
 end if

 Kdata%npw = npw_k
 allocate(Kdata%kg_k(3,npw_k)); Kdata%kg_k = my_kg_k

 if (PRESENT(ecut)) deallocate(my_kg_k) 
 my_kg_k => Kdata%kg_k

 mgfft = MAXVAL(ngfft(1:3))
 !
 ! Finds the boundary of the basis sphere of G vectors (for this k point) 
 ! for use in improved zero padding of ffts in 3 dimensions.
 allocate(Kdata%gbound(2*mgfft+8,2))
 call sphereboundary(Kdata%gbound,istwfk,my_kg_k,mgfft,npw_k)
 !
 ! Index of the G-sphere in the FFT box.
 allocate(Kdata%igfft0(npw_k))

 allocate(kg_mask(npw_k))
 call kgindex(Kdata%igfft0,my_kg_k,kg_mask,MPI_enreg,ngfft,npw_k)

 ABI_CHECK(ALL(kg_mask),"FFT para not yet implemented")
 deallocate(kg_mask)

 if (Psps%usepaw==1) then 
   !
   ! * Compute spherical harmonics for this k-point.
   useylmgr=0
   allocate(ylm_k(npw_k,Psps%mpsang**2),ylmgr_k(npw_k,3,Psps%mpsang**2*useylmgr)) 

   optder=0 ! only Ylm(K) are computed.
   mkmem_=1; mpw_=npw_k; nband_=0; nkpt_=1; npwarr_(1)=npw_k

   call initylmg(Cryst%gprimd,my_kg_k,kpoint,mkmem_,MPI_enreg,Psps%mpsang,mpw_,nband_,nkpt_,&
&    npwarr_,1,optder,Cryst%rprimd,dum_unkg,dum_unylm,ylm_k,ylmgr_k)
   !
   ! * Compute (k+G) vectors.
   nkpg=0
   allocate(kpg_k(npw_k,nkpg)); if (nkpg>0) call mkkpg(my_kg_k,kpg_k,kpoint,nkpg,npw_k)
   !
   ! * Compute nonlocal form factors fnl_ylm for all (k+G).
   dimffnl=1+3*ider0

   Kdata%dim_fnlylm = dimffnl
   allocate(Kdata%fnlylm(npw_k,dimffnl,Psps%lmnmax,Cryst%ntypat))

   call mkffnl(Psps%dimekb,dimffnl,Psps%ekb,Kdata%fnlylm,Psps%ffspl,&
&    Cryst%gmet,Cryst%gprimd,ider0,idir0,Psps%indlmn,my_kg_k,kpg_k,kpoint,Psps%lmnmax,&
&    Psps%lnmax,Psps%mpsang,Psps%mqgrid_ff,nkpg,npw_k,Cryst%ntypat,&
&    Psps%pspso,Psps%qgrid_ff,Cryst%rmet,Psps%usepaw,Psps%useylm,ylm_k,ylmgr_k)

   deallocate(kpg_k, STAT=istat)
   deallocate(ylm_k,ylmgr_k)

   allocate(Kdata%phkxred(2,Cryst%natom))
   do iatom=1,Cryst%natom
     iatm=Cryst%atindx(iatom)
     arg=two_pi*(DOT_PRODUCT(kpoint,Cryst%xred(:,iatom)))
     Kdata%phkxred(1,iatm)=DCOS(arg) 
     Kdata%phkxred(2,iatm)=DSIN(arg)
   end do
   !
   ! Calculate 1-dim structure factor phase information.
   mgfft = MAXVAL(ngfft(1:3))
   allocate(ph1d(2,3*(2*mgfft+1)*Cryst%natom))
   call getph(Cryst%atindx,Cryst%natom,ngfft(1),ngfft(2),ngfft(3),ph1d,Cryst%xred)

   matblk=Cryst%natom 
   allocate(Kdata%ph3d(2,npw_k,matblk))
   call ph1d3d(1,Cryst%natom,my_kg_k,matblk,Cryst%natom,npw_k,ngfft(1),ngfft(2),ngfft(3),Kdata%phkxred,ph1d,Kdata%ph3d)

   deallocate(ph1d)
 end if ! usepaw

end subroutine kdata_init
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/destroy_kdata_0D
!! NAME
!!  destroy_kdata_0D
!!
!! FUNCTION
!!  Deallocate memory
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine destroy_kdata_0D(Kdata)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(kdata_t),intent(inout) :: Kdata

!************************************************************************

 !@kdata_t
 if (associated(Kdata%kg_k  )) deallocate(Kdata%kg_k  )
 if (associated(Kdata%igfft0)) deallocate(Kdata%igfft0)
 if (associated(Kdata%gbound)) deallocate(Kdata%gbound)

 if (associated(Kdata%ph3d   )) deallocate(Kdata%ph3d   )
 if (associated(Kdata%phkxred)) deallocate(Kdata%phkxred)
 if (associated(Kdata%fnlylm )) deallocate(Kdata%fnlylm )

end subroutine destroy_kdata_0D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/destroy_kdata_1D
!! NAME
!!  destroy_kdata_1D
!!
!! FUNCTION
!!   Deallocate memory.
!!
!! PARENTS
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine destroy_kdata_1D(Kdata)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(kdata_t),intent(inout) :: Kdata(:)

!Local variables ------------------------------
!scalars
 integer :: ik

!************************************************************************

 do ik=1,SIZE(Kdata)
   call destroy_kdata_0d(Kdata(ik))
 end do

end subroutine destroy_kdata_1D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_init
!! NAME
!! wfd_init
!!
!! FUNCTION
!!  Initialize the object.
!!
!! INPUTS
!!  Cryst<crystal_structure>=Object defining the unit cell and its symmetries.
!!  Pawtab(ntypat*usepaw)<type(pawtab_type)>=PAW tabulated starting data.
!!  Psps<Pseudopotential_type>=datatype storing data on the pseudopotentials.
!!  ngfft(18)=All needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  nkibz=Number of irreducible k-points.
!!  npwwfn=Number of plane waves for u(G).
!!  nsppol=Number of independent spin polarizations.
!!  nspden=Number of density components.
!!  nspinor=Number of spinorial components.
!!  mband
!!  nband(nkibz,nsppol)
!!  keep_ur(mband,nkibz,nsppol)=Option for memory storage of u(r).
!!  paral_kgb=Option for band-FFT parallelism (not yet available)
!!  gvec(3,npwwfn)=G-vectors in reduced coordinates.
!!  istwfk(nkibz)=Storage mode.
!!  kibz(3,nkibz)=Reduced coordinates of the k-points.
!!  mg0(3)=Max reduced components for umklapps.
!!  nloalg(5)=Governs the choice of the algorithm for nonlocal operator. See doc.
!!  comm=MPI communicator.
!!
!! OUTPUT
!!  Initialize the object with basic dimensions, allocate also memory for u(g) and u(r) according to keep_ur
!!    %ug in reciprocal space are always allocated.
!!    %ur in real space only if keep_ur.
!!
!! PARENTS
!!      bethe_salpeter,screening,sigma
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_init(Wfd,Cryst,Pawtab,Psps,keep_ur,paral_kgb,npwwfn,mband,nband,nkibz,nsppol,bks_mask,&
&  nspden,nspinor,istwfk,kibz,ngfft,mg0,gvec,nloalg,comm)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_51_manage_mpi
 use interfaces_53_ffts
 use interfaces_56_recipspace
 use interfaces_65_nonlocal
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: paral_kgb,mband,comm
 integer,intent(in) :: nkibz,npwwfn,nsppol,nspden,nspinor
 type(crystal_structure),intent(in) :: Cryst
 type(pseudopotential_type),intent(in) :: Psps
 type(wfs_descriptor),intent(out) :: Wfd
!array
 integer,intent(in) :: ngfft(18),istwfk(nkibz),mg0(3),nband(nkibz,nsppol)
 integer,intent(in) :: gvec(3,npwwfn),nloalg(5)
 real(dp),intent(in) :: kibz(3,nkibz)
 logical,intent(in) :: bks_mask(mband,nkibz,nsppol)
 logical,intent(in) :: keep_ur(mband,nkibz,nsppol)
 type(Pawtab_type),intent(in) :: Pawtab(Cryst%ntypat*Psps%usepaw)

!Local variables ------------------------------
!scalars
 integer,parameter :: ider0=0,idir0=0,dum_unkg=0,dum_unylm=0,nfft0=0
 integer :: istat,ig,ik_ibz,spin,band
 integer :: ikg,exchn2n3d,istwf_k,mpw_,npw_k
 integer :: dimffnl,useylmgr,nkpg,iatom,itypat,iat
 integer :: mkmem_,nkpt_,optder
 integer :: iatm,matblk
 real(dp) :: ug_size,ur_size,cprj_size,gsq,g1,g2,g3,arg
 character(len=500) :: msg
 type(MPI_type) :: MPI_seq
!arrays
 integer :: dum_kg(3,0),nband_(1*nsppol),npwarr_(1)
 integer,pointer :: kg_k(:,:)
 real(dp) :: kpoint(3)
 real(dp),allocatable :: ylm_k(:,:),ylmgr_k(:,:,:),kpg_k(:,:)
 real(dp),pointer :: fnlylm_k(:,:,:,:),ph3d(:,:,:)
 real(dp),allocatable :: phkxred(:,:) 
 logical,allocatable :: kg_mask(:)
!************************************************************************

 DBG_ENTER("COLL")

 !@wfs_descriptor
 call wfd_nullify(Wfd)

 ! MPI info
 Wfd%comm = comm
 Wfd%my_rank = xcomm_rank(Wfd%comm)
 Wfd%nproc   = xcomm_size(Wfd%comm)
 Wfd%master = 0
 !
 ! Sequential MPI datatype to be passed to abinit routines.
 call initmpi_seq(Wfd%MPI_enreg)
 !
 ! === Basic dimensions ===
 Wfd%nkibz     = nkibz
 Wfd%nsppol    = nsppol
 Wfd%nspden    = nspden
 Wfd%nspinor   = nspinor
 Wfd%npwwfn    = npwwfn
 Wfd%paral_kgb = paral_kgb
 Wfd%nloalg    = nloalg

 Wfd%usepaw = Psps%usepaw
 Wfd%natom  = Cryst%natom
 Wfd%ntypat = Cryst%ntypat
 Wfd%lmnmax = Psps%lmnmax
                                                                  
 allocate(Wfd%indlmn(6,Wfd%lmnmax,Wfd%ntypat))
 Wfd%indlmn = Psps%indlmn

 if (Wfd%usepaw==1) then 
   allocate(Wfd%nlmn_atm(Cryst%natom),Wfd%nlmn_type(Cryst%ntypat))
   do iatom=1,Cryst%natom
     Wfd%nlmn_atm(iatom)=Pawtab(Cryst%typat(iatom))%lmn_size
   end do

   do itypat=1,Cryst%ntypat
     Wfd%nlmn_type(itypat)=Pawtab(itypat)%lmn_size
   end do

   allocate(Wfd%nlmn_sort(Cryst%natom)); iat=0 ! nlmn dims sorted by atom type.
   do itypat=1,Cryst%ntypat
     Wfd%nlmn_sort(iat+1:iat+Cryst%nattyp(itypat))=Pawtab(itypat)%lmn_size
     iat=iat+Cryst%nattyp(itypat)
   end do
 end if

 allocate(Wfd%keep_ur(mband,nkibz,nsppol)); Wfd%keep_ur=keep_ur   
 !
 ! Setup of the FFT mesh 
 Wfd%ngfft  = ngfft(:)
 Wfd%mgfft  = MAXVAL (Wfd%ngfft(1:3))
 Wfd%nfftot = PRODUCT(Wfd%ngfft(1:3)) 
 Wfd%nfft   = Wfd%nfftot ! TODO At present no FFT para
 !
 ! Calculate ecut from input gvec.
 Wfd%ecut=-one
 do ig=1,Wfd%npwwfn
   g1=REAL(gvec(1,ig))
   g2=REAL(gvec(2,ig))
   g3=REAL(gvec(3,ig))
   gsq=      Cryst%gmet(1,1)*g1**2+Cryst%gmet(2,2)*g2**2+Cryst%gmet(3,3)*g3**2+ &
&       two*(Cryst%gmet(1,2)*g1*g2+Cryst%gmet(1,3)*g1*g3+Cryst%gmet(2,3)*g2*g3)
   Wfd%ecut=MAX(Wfd%ecut,gsq)
 end do
 Wfd%ecut=two*Wfd%ecut*pi**2
 !
 ! * Index of the G-sphere in the FFT box.
 allocate(Wfd%igfft0(Wfd%npwwfn),kg_mask(Wfd%npwwfn))

 call kgindex(Wfd%igfft0,gvec,kg_mask,Wfd%MPI_enreg,ngfft,Wfd%npwwfn)

 ABI_CHECK(ALL(kg_mask),"FFT para not yet implemented")
 deallocate(kg_mask)
 !
 allocate(Wfd%kibz(3,Wfd%nkibz)); Wfd%kibz=kibz

 allocate(Wfd%istwfk(Wfd%nkibz)); Wfd%istwfk=istwfk
 ABI_CHECK(ALL(Wfd%istwfk==1),"istwfk/=1 not yet coded")

 ! * Get the number of planewaves npw_k
 allocate(Wfd%npwarr(Wfd%nkibz))

 exchn2n3d=0; ikg=0; mpw_=0 
 do ik_ibz=1,Wfd%nkibz
   istwf_k = Wfd%istwfk(ik_ibz)
   kpoint  = Wfd%kibz(:,ik_ibz)
   call kpgsph(Wfd%ecut,exchn2n3d,Cryst%gmet,ikg,ik_ibz,istwf_k,dum_kg,kpoint,0,Wfd%MPI_enreg,mpw_,npw_k)
   Wfd%npwarr(ik_ibz)= npw_k
 end do
 !
 ! TODO For the time being, continue to use Gamma-centered basis set.
 Wfd%npwarr = Wfd%npwwfn
 !Wfd%mpw= MAXVAL(Wfd%npwarr)

 allocate(Wfd%gvec(3,Wfd%npwwfn)); Wfd%gvec=gvec 
 !
 ! Finds the boundary of the basis sphere of G vectors (at a given k point) 
 ! for use in improved zero padding of ffts in 3 dimensions.
 allocate(Wfd%gbound(2*Wfd%mgfft+8,2))
 call sphereboundary(Wfd%gbound,Wfd%istwfk(1),Wfd%gvec,Wfd%mgfft,Wfd%npwwfn)

 ! FFT index of G-G0. Set to 0 if G-G0 falls outside the FFT box.
 Wfd%mg0 = mg0

 !allocate(Wfd%igmg0_fft(Wfd%npwwfn,2*mg0(1)+1,2*mg0(2)+1,2*mg0(3)+1) )
 !call cigfft(mg0,Wfd%npwwfn,Wfd%ngfft,Wfd%gvec,Wfd%igmg0_fft,ierr)

 allocate(Wfd%nband(nkibz,nsppol)); Wfd%nband=nband 

 Wfd%mband = mband 
 ABI_CHECK(MAXVAL(Wfd%nband)==mband,"wrong mband")

 ! === Allocate u(g) and, if required, also u(r) ===
 ug_size=nspinor*npwwfn*COUNT(bks_mask)
 write(msg,'(a,f12.1,a)')' Memory needed for storing ug= ',2*gwpc*ug_size*b2Mb,' [Mb]'
 call wrtout(std_out,msg,'PERS')

 if (Wfd%usepaw==1) then
   cprj_size=nspinor*SUM(Wfd%nlmn_atm)*COUNT(bks_mask)
   write(msg,'(a,f12.1,a)')' Memory needed for storing Cprj= ',dp*cprj_size*b2Mb,' [Mb]'
   call wrtout(std_out,msg,'PERS')
 end if

 ur_size=nspinor*Wfd%nfft*COUNT(Wfd%keep_ur)
 write(msg,'(a,f12.1,a)')' Memory needed for storing ur= ',2*gwpc*ur_size*b2Mb,' [Mb]'
 call wrtout(std_out,msg,'PERS')

 allocate(Wfd%Wave(Wfd%mband,Wfd%nkibz,Wfd%nsppol))
 call wave_nullify(Wfd%Wave)

 ! Allocate the wavefunctions in reciprocal space according to bks_mask.
 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz
     do band=1,Wfd%nband(ik_ibz,spin)
       if (bks_mask(band,ik_ibz,spin)) then
         !if (Wfd%keep_ur(band,ik_ibz,spin)) then 
         !  call init_wave_0D(Wfd%Wave(band,ik_ibz,spin),Wfd%usepaw,npwwfn,Wfd%nfft,nspinor,Wfd%natom,Wfd%nlmn_atm,CPR_RANDOM)
         !else
         call init_wave_0D(Wfd%Wave(band,ik_ibz,spin),Wfd%usepaw,npwwfn,nfft0,nspinor,Wfd%natom,Wfd%nlmn_atm,CPR_RANDOM)
         !end if
       end if
     end do
   end do
 end do

 ! Allocate the global table used to keep trace of the distribution, including a possible duplication.
 allocate(Wfd%bks_tab(Wfd%mband,nkibz,nsppol,0:Wfd%nproc-1)); Wfd%bks_tab=NO_BKS
 !
 ! ===================================================
 ! ==== Precalculate nonlocal form factors for PAW ====
 ! ===================================================
 !
 ! Calculate 1-dim structure factor phase information.
 allocate(Wfd%ph1d(2,3*(2*Wfd%mgfft+1)*Wfd%natom))
 call getph(Cryst%atindx,Wfd%natom,Wfd%ngfft(1),Wfd%ngfft(2),Wfd%ngfft(3),Wfd%ph1d,Cryst%xred)

 call initmpi_seq(MPI_seq)

 allocate(Wfd%Kdata(Wfd%nkibz))
 call kdata_nullify(Wfd%Kdata)    ! Nullify the pointers defined in the datatype.

 do ik_ibz=1,Wfd%nkibz

   kpoint  =  Wfd%kibz(:,ik_ibz)
   istwf_k =  Wfd%istwfk(ik_ibz)
   npw_k   =  Wfd%npwwfn           ! TODO these quantities should be k-dependent.

#if 0 
   ! TODO enable this call when k-centered G-spheres are used.
   !$if (wfd_ihave_ug(Wfd,0,ik_ibz,0)) then
   call kdata_init(Wfd%Kdata(ik_ibz),Cryst,Psps,kpoint,istwf_k,ngfft,Wfd%MPI_enreg,kg_k=Wfd%gvec)
   !endif
#else

   Wfd%Kdata(ik_ibz)%npw    = npw_k
   Wfd%Kdata(ik_ibz)%istwfk = istwf_k
   allocate(Wfd%Kdata(ik_ibz)%kg_k(3,npw_k))
   Wfd%Kdata(ik_ibz)%kg_k = Wfd%gvec
   kg_k   => Wfd%Kdata(ik_ibz)%kg_k

   if (Wfd%usepaw==1) then 
     !
     ! * Compute spherical harmonics for this k-point.
     useylmgr=0
     allocate(ylm_k(npw_k,Psps%mpsang**2),ylmgr_k(npw_k,3,Psps%mpsang**2*useylmgr)) 

     optder=0 ! only Ylm(K) are computed.
     mkmem_=1; mpw_=npw_k; nband_(:)=0; nkpt_=1; npwarr_(1)=npw_k

     call initylmg(Cryst%gprimd,kg_k,Wfd%kibz(:,ik_ibz),mkmem_,MPI_seq,Psps%mpsang,mpw_,nband_,nkpt_,&
&      npwarr_,Wfd%nsppol,optder,Cryst%rprimd,dum_unkg,dum_unylm,ylm_k,ylmgr_k)

     ! * Compute (k+G) vectors.
     nkpg=0
     allocate(kpg_k(npw_k,nkpg)); if (nkpg>0) call mkkpg(kg_k,kpg_k,kpoint,nkpg,npw_k)
     !
     ! * Compute nonlocal form factors fnl_ylm for all (k+G).
     dimffnl=1+3*ider0

     Wfd%Kdata(ik_ibz)%dim_fnlylm = dimffnl
     allocate(Wfd%Kdata(ik_ibz)%fnlylm(npw_k,dimffnl,Psps%lmnmax,Cryst%ntypat))
     fnlylm_k => Wfd%Kdata(ik_ibz)%fnlylm

     call mkffnl(Psps%dimekb,dimffnl,Psps%ekb,fnlylm_k,Psps%ffspl,&
&      Cryst%gmet,Cryst%gprimd,ider0,idir0,Psps%indlmn,kg_k,kpg_k,kpoint,Psps%lmnmax,&
&      Psps%lnmax,Psps%mpsang,Psps%mqgrid_ff,nkpg,npw_k,Cryst%ntypat,&
&      Psps%pspso,Psps%qgrid_ff,Cryst%rmet,Psps%usepaw,Psps%useylm,ylm_k,ylmgr_k)

     deallocate(kpg_k, STAT=istat)
     deallocate(ylm_k,ylmgr_k)

     allocate(phkxred(2,Cryst%natom))
     do iatom=1,Cryst%natom
       iatm=Cryst%atindx(iatom)
       arg=two_pi*(DOT_PRODUCT(kpoint,Cryst%xred(:,iatom)))
       phkxred(1,iatm)=DCOS(arg) 
       phkxred(2,iatm)=DSIN(arg)
     end do

     matblk=Wfd%natom 
     allocate(Wfd%Kdata(ik_ibz)%ph3d(2,npw_k,matblk))
     ph3d => Wfd%Kdata(ik_ibz)%ph3d
     call ph1d3d(1,Wfd%natom,kg_k,matblk,Wfd%natom,npw_k,Wfd%ngfft(1),Wfd%ngfft(2),Wfd%ngfft(3),phkxred,Wfd%ph1d,ph3d)

     deallocate(phkxred)

   end if ! usepaw
#endif
 end do

 DBG_EXIT("COLL")

end subroutine wfd_init
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_destroy
!! NAME
!!  wfd_destroy
!!
!! FUNCTION
!!  Free the memory allocated in the wfs_descriptor data type.
!!
!! PARENTS
!!      bethe_salpeter,screening,sigma
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_destroy(Wfd)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd
!************************************************************************

 DBG_ENTER("COLL")

 !@wfs_descriptor
 !
 ! integer pointers.
 if (associated(Wfd%igfft0   )) deallocate(Wfd%igfft0   )
 if (associated(Wfd%gvec     )) deallocate(Wfd%gvec     )
 if (associated(Wfd%gbound   )) deallocate(Wfd%gbound   )
 if (associated(Wfd%irottb   )) deallocate(Wfd%irottb   )
 if (associated(Wfd%istwfk   )) deallocate(Wfd%istwfk   )
 if (associated(Wfd%nband    )) deallocate(Wfd%nband    )
 if (associated(Wfd%indlmn   )) deallocate(Wfd%indlmn   )
 if (associated(Wfd%nlmn_atm )) deallocate(Wfd%nlmn_atm )
 if (associated(Wfd%nlmn_sort)) deallocate(Wfd%nlmn_sort)
 if (associated(Wfd%nlmn_type)) deallocate(Wfd%nlmn_type)
 if (associated(Wfd%npwarr   )) deallocate(Wfd%npwarr   )
 if (associated(Wfd%bks_tab  )) deallocate(Wfd%bks_tab  )
 !
 ! real pointers.
 if (associated(Wfd%kibz)) deallocate(Wfd%kibz)
 if (associated(Wfd%ph1d)) deallocate(Wfd%ph1d)
 !
 ! logical pointers.
 if (associated(Wfd%keep_ur)) deallocate(Wfd%keep_ur)
 !
 ! datatypes.
 if (associated(Wfd%Kdata)) then
   call kdata_free(Wfd%Kdata); deallocate(Wfd%Kdata)
 end if

 if (associated(Wfd%Wave)) then
   call wave_free(Wfd%Wave); deallocate(Wfd%Wave)
 end if

 call destroy_mpi_enreg(Wfd%MPI_enreg)

 DBG_EXIT("COLL")

end subroutine wfd_destroy
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_reset_ur
!! NAME
!!  wfd_reset_ur
!!
!! FUNCTION
!!  Reinitialize the storage mode of the ur treated by this node.
!!
!! PARENTS
!!      sigma
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_reset_ur(Wfd)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd

!************************************************************************

 where (Wfd%Wave(:,:,:)%has_ur == WFD_STORED) 
   Wfd%Wave(:,:,:)%has_ur = WFD_ALLOCATED
 end where

end subroutine wfd_reset_ur
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_get_ur
!! NAME
!!  wfd_get_ur
!!
!! FUNCTION
!!  Get a wave function in real space, either by doing a G-->R FFT 
!!  or by just retrieving the data already stored in Wfd.
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=the wavefunction descriptor.
!!  band=Band index.
!!  ik_ibz=Index of the k-point in the IBZ.
!!  spin=Spin index
!!
!! OUTPUT
!!  ur(Wfd%nfft*Wfd%nspinor)=The wavefunction in real space.
!!
!! PARENTS
!!      bloch_interp,calc_density,calc_exch,calc_sig_ppm_EET,calc_sigc_me
!!      calc_sigx_me,calc_vhxc_me,cchi0,cchi0q0,cexch_haydock,classify_bands
!!      cohsex_me,debug_tools,excden,exch,m_oscillators,m_wfs,trashme
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_get_ur(Wfd,band,ik_ibz,spin,ur)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays
 complex(gwpc),intent(out) :: ur(Wfd%nfft*Wfd%nspinor)

!Local variables ------------------------------
!scalars
 integer,parameter :: tim_fourdp=5,npw0=0
 integer :: npwwfn,nfft,nspinor,has_this_ur
 character(len=500) :: msg
!arrays 
 integer,pointer :: kg_k(:,:)
 complex(gwpc),pointer :: wave_ug(:)
!************************************************************************

 has_this_ur = Wfd%Wave(band,ik_ibz,spin)%has_ur

 SELECT CASE (has_this_ur) 

 CASE (WFD_NOWAVE, WFD_ALLOCATED) ! FFT is required.
   npwwfn = Wfd%npwwfn
   nfft   = Wfd%nfft
   nspinor= Wfd%nspinor

#ifdef DEBUG_MODE 
   ! TODO: here there is a problem somewhere in MSG_PERS_BUG
   if (.not.wfd_ihave_ug(Wfd,band,ik_ibz,spin,"Stored")) then
     write(msg,'(a,3(i0,1x),a)')" ug for (band, ik_ibz, spin): ",band,ik_ibz,spin," is not stored in memory!"
     MSG_PERS_ERROR(msg)
   end if
#endif

   wave_ug => Wfd%Wave(band,ik_ibz,spin)%ug 
   kg_k    => Wfd%Kdata(ik_ibz)%kg_k

   call fft_onewfn(Wfd%paral_kgb,Wfd%istwfk(ik_ibz),nspinor,npwwfn,nfft,wave_ug,ur,&
&    Wfd%igfft0,Wfd%ngfft,kg_k,Wfd%gbound,tim_fourdp,Wfd%MPI_enreg)

   if (Wfd%keep_ur(band,ik_ibz,spin)) then ! Store results
     if (has_this_ur==WFD_NOWAVE) then ! Allocate buffer for ur.
       call init_wave_0D(Wfd%Wave(band,ik_ibz,spin),Wfd%usepaw,npw0,Wfd%nfft,Wfd%nspinor,Wfd%natom,Wfd%nlmn_atm,CPR_RANDOM)
     end if
     call xcopy(Wfd%nfft*Wfd%nspinor,ur,1,Wfd%Wave(band,ik_ibz,spin)%ur,1)
     Wfd%Wave(band,ik_ibz,spin)%has_ur=WFD_STORED
   end if

 CASE (WFD_STORED) ! copy it back.
   call xcopy(Wfd%nfft*Wfd%nspinor,Wfd%Wave(band,ik_ibz,spin)%ur,1,ur,1)

 CASE DEFAULT
   write(msg,'(a,i0)')" Wrong has_ur: ",has_this_ur
   MSG_PERS_BUG(msg)
 END SELECT

end subroutine wfd_get_ur
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_nullify
!! NAME
!!  wfd_nullify
!!
!! FUNCTION
!!  Nullify the pointers of the data structure.
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_nullify(Wfd)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd
!************************************************************************

 !@wfs_descriptor

! integer
 nullify(Wfd%igfft0)
 nullify(Wfd%irottb)
 nullify(Wfd%istwfk)
 nullify(Wfd%nband )
 nullify(Wfd%indlmn)
 nullify(Wfd%nlmn_atm )
 nullify(Wfd%nlmn_sort)
 nullify(Wfd%nlmn_type)
 nullify(Wfd%npwarr)
 nullify(Wfd%gvec  )
 nullify(Wfd%gbound)

!integer arrays.
 nullify(Wfd%bks_tab)

!real arrays
 nullify(Wfd%kibz)
 nullify(Wfd%ph1d)

!logical arrays
 nullify(Wfd%keep_ur)

! pointers to datatypes.
 nullify(Wfd%Kdata)
 nullify(Wfd%Wave)

! datatypes 
 call nullify_mpi_enreg(Wfd%MPI_enreg)

end subroutine wfd_nullify
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_print
!! NAME
!! wfd_print
!!
!! FUNCTION
!!  Print the content of a wfs_descriptor datatype
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=The datatype.
!!  [header]=String to be printed as header for additional info.
!!  [unit]=Unit number for output
!!  [prtvol]=Verbosity level
!!  [mode_paral]=Either "COLL" or "PERS". Defaults to "COLL".
!!
!! OUTPUT
!!  Only printing 
!!
!! PARENTS
!!      bethe_salpeter,debug_tools,screening,sigma
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_print(Wfd,header,unit,prtvol,mode_paral)
    
 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,optional,intent(in) :: unit,prtvol
 character(len=4),optional,intent(in) :: mode_paral
 character(len=*),optional,intent(in) :: header
 type(wfs_descriptor),intent(in) :: Wfd

!Local variables-------------------------------
!scalars
 integer :: my_prtvol,my_unt
 character(len=4) :: my_mode
 character(len=500) :: msg      
! *************************************************************************

 my_unt   =std_out; if (PRESENT(unit      )) my_unt   =unit
 my_prtvol=0      ; if (PRESENT(prtvol    )) my_prtvol=prtvol
 my_mode  ='COLL' ; if (PRESENT(mode_paral)) my_mode  =mode_paral

 msg=' ==== Info on the Wfd% object ==== '
 if (PRESENT(header)) msg=' ==== '//TRIM(ADJUSTL(header))//' ==== '
 call wrtout(my_unt,msg,my_mode)

 write(msg,'(3(a,i5,a),a,i5,2a,f5.1)')&
&  '  Number of irreducible k-points ........ ',Wfd%nkibz,ch10,&
&  '  Number of spinorial components ........ ',Wfd%nspinor,ch10,&
&  '  Number of spin-density components ..... ',Wfd%nspden,ch10,&
&  '  Number of spin polarizations .......... ',Wfd%nsppol,ch10,&
&  '  Plane wave cutoff energy .............. ',Wfd%ecut 
 call wrtout(my_unt,msg,my_mode)

 write(msg,'(4(a,i0,a))')&
&  '  Number of reciprocal lattice vectors .. ',Wfd%npwwfn,ch10,&
&  '  Total number of FFT points ....... .... ',Wfd%nfftot,ch10,&
&  '  Number of FFT points treated by me .... ',Wfd%nfft,ch10,&
&  '  Parallelism over k-b-g (paral_kgb) .... ',Wfd%paral_kgb,ch10
 call wrtout(my_unt,msg,my_mode)

 call print_ngfft(Wfd%ngfft,'FFT mesh for wavefunctions',my_unt,my_mode,my_prtvol)

 !TODO
 ! Add addition info

end subroutine wfd_print
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/fft_onewfn
!! NAME
!! fft_onewfn
!!
!! FUNCTION
!! Low-level routine used to calculate ONE wavefunction in real space via FFT.
!!
!! INPUTS
!! nspinor=number of spinorial components
!! istwfk=Option describing the storage of the wavefunction. (at present must be 1)
!! igfft(npwwfn)=index of each plane wave in FFT grid
!! ngfft(18)=contain all needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!! nkibz=number of k points
!! npwwfn=number of plane waves
!! nsppol=number of independent spin polarizations 
!! tim_fourdp=4 if called from within screening ; =5 if called from within sigma
!! ug(nspinor*npwwfn)=wavefunctions in reciprocal space treated by this processor.
!! MPI_enreg= datatype containing information on parallelism to be passed to fourdp
!! gbound(2*mgfft+8,2)=Table for padded-FFT. See sphereboundary.
!! kg_k(3,npwwfn)=G-vectors in reduced coordinates
!!
!! OUTPUT
!!  ur(ngfft(1)*ngfft(2)*ngfft(3)*nspinor)=wavefunctions in real space.
!!
!! PARENTS
!!      bloch_interp,calc_sig_ppm_EET,m_wfs,trashme
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine fft_onewfn(paral_kgb,istwf_k,nspinor,npwwfn,nfftot,ug,ur,igfft,ngfft,kg_k,gbound,tim_fourdp,MPI_enreg)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_ffts
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: paral_kgb,npwwfn,nfftot,tim_fourdp,nspinor,istwf_k
 type(MPI_type),intent(inout) :: MPI_enreg
!arrays
 integer,intent(in) :: igfft(npwwfn) 
 integer,intent(in) :: ngfft(18),gbound(:,:) !2*mgfft+8,2)
 integer,intent(in) :: kg_k(3,npwwfn)
 complex(gwpc),intent(in) :: ug(npwwfn*nspinor)
 complex(gwpc),intent(out) :: ur(nfftot*nspinor)

!Local variables-------------------------------
!scalars
 integer :: nx,ny,nz,mgfft,ldx,ldy,ldz,fftalg,fftalga,fftalgc
 integer :: ispinor,ig,rspad,gspad 
 integer :: cplex=0,option=0,ix,iy,iz,ifft
 real(dp) :: weight=one
 character(len=500) :: msg
!arrays
 integer :: dum_kg_kout(0,0)
 real(dp) :: dum_denpot(0,0,0),dum_fofgout(0,0) 
 real(dp),allocatable :: fofgin(:,:),fofr(:,:,:,:)
 !complex(dpc),allocatable :: fr(:)
 complex(dpc),allocatable :: fg(:)

! *************************************************************************

 fftalg = ngfft(7); fftalga=fftalg/100; fftalgc=MOD(fftalg,10)
 nx=ngfft(1); ny=ngfft(2); nz=ngfft(3); mgfft=MAXVAL(ngfft(1:3))

 SELECT CASE (fftalga) 

 CASE (1) ! Goedecker Routines (only fftalgc==2 is coded)

   ldx=ngfft(4); ldy=ngfft(5); ldz=ngfft(6) ! Here augmentation is supported.

   allocate(fofgin(2,npwwfn),fofr(2,ldx,ldy,ldz))

   do ispinor=1,nspinor
     gspad=(ispinor-1)*npwwfn
     rspad=(ispinor-1)*nfftot

     do ig=1,npwwfn ! Have to convert from CPLX to REAL.
       fofgin(1,ig) = DBLE (ug(ig+gspad)) 
       fofgin(2,ig) = AIMAG(ug(ig+gspad))
     end do
 
     call sg_fftrisc(cplex,dum_denpot,fofgin,dum_fofgout,fofr,gbound,gbound,istwf_k,kg_k,dum_kg_kout,&
&      mgfft,ngfft,npwwfn,0,ldx,ldy,ldz,option,weight)

     !write(66,*)fofgin
     !call sg_fftrisc(cplex,dum_denpot,fofgin,fofgin,fofr,gbound,gbound,istwf_k,kg_k,kg_k,&
     !&mgfft,ngfft,npwwfn,npwwfn,ldx,ldy,ldz,3,weight)
     !write(67,*)fofgin

     do iz=1,nz     ! Fill the output array on the ngfft(1:3) mesh.
       do iy=1,ny
         do ix=1,nx
           ifft = ix + (iy-1)*nx + (iz-1)*nx*ny + rspad
           ur(ifft) = CMPLX(fofr(1,ix,iy,iz), fofr(2,ix,iy,iz), kind=gwpc)
         end do
       end do
     end do

     ! This call corresponds to fftalgc == 1
     !!!!!call sg_fftpad(ngfft(8),mgfft,ldx,ldy,ldz,nx,ny,nz,arr,ftarr,one,gbound)
     !MSG_ERROR("calling sg_fftpad! Results for R--G might be wrong")
     !call padded_fourwf_cplx(fg,ngfft,nx,ny,nz,ldx,ldy,ldz,mgfft,+1,gbound)
     !ur(rspad+1:rspad+nfftot)=fg
   end do ! ispinor

   deallocate(fofgin,fofr)
   RETURN

 CASE (3) ! FFTW3.

   allocate(fg(nfftot))

   do ispinor=1,nspinor
     gspad=(ispinor-1)*npwwfn
     rspad=(ispinor-1)*nfftot
     !
     fg=czero ! Fill the FFT array from the PW array
     do ig=1,npwwfn
       fg(igfft(ig))=ug(ig+gspad)
     end do
     !
     ! * FFT to give wfn in real space.
     ldx=nx; ldy=ny; ldz=nz ! TODO No augmentation at present 

     !call fftw3_c2c_ip(nx,ny,nz,ldx,ldy,ldz,1,+1,fg) ! Version without Padding
     call fftw3_fftpad_cplx(fg,nx,ny,nz,ldx,ldy,ldz,mgfft,+1,gbound)
     ur(rspad+1:rspad+nfftot)=fg
   end do ! ispinor
                                                                                
   deallocate(fg) 
   RETURN

 CASE DEFAULT
   write(msg,"(a,i0,a)")" fftalga= ",fftalga," not available for GW calculations."
   MSG_ERROR(msg)
 END SELECT

 RETURN

 ABI_UNUSED((/paral_kgb,tim_fourdp,MPI_enreg%nproc/)) ! FIXME

end subroutine fft_onewfn
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_mkall_ur
!! NAME
!! wfd_mkall_ur
!!
!! FUNCTION
!!  FFT transform from G to R the entire set of wavefunctions stored in the wfs_descriptor.
!!  Only those waves whose status is WFD_ALLOCATED are calculated unless force is used. 
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=Structure containing the wave functions for the GW.
!!
!! OUTPUT
!!  ncalc=Number of FFTs performed.
!!  [force]=If .TRUE. then force FFT even for waves whose status is WFD_STORED. 
!!
!! SIDE EFFECTS
!!  %ur
!!
!! PARENTS
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_mkall_ur(Wfd,ncalc,force)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(out) :: ncalc
 logical,optional,intent(in) :: force
 type(wfs_descriptor),intent(inout) :: Wfd

!Local variables ------------------------------
!scalars
 integer :: spin,ik_ibz,band
 logical :: do_fft
 integer,pointer :: kg_k(:,:)

!************************************************************************

! TODO FFTs should be done in bunches.
! 
 ncalc=0 !; if (.not.Wfd%keep_ur) RETURN

 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz    
     kg_k => Wfd%Kdata(ik_ibz)%kg_k

     do band=1,Wfd%nband(ik_ibz,spin)

       if (.not.Wfd%keep_ur(band,ik_ibz,spin)) CYCLE

       do_fft = wfd_ihave_ur(Wfd,band,ik_ibz,spin,"Allocated")
       if (PRESENT(force)) do_fft = (do_fft .or. wfd_ihave_ur(Wfd,band,ik_ibz,spin,"Stored"))

       if (do_fft) then
         call fft_onewfn(Wfd%paral_kgb,Wfd%istwfk(ik_ibz),Wfd%nspinor,Wfd%npwwfn,Wfd%nfft,&
&          Wfd%Wave(band,ik_ibz,spin)%ug,Wfd%Wave(band,ik_ibz,spin)%ur,Wfd%igfft0,Wfd%ngfft,&
&          kg_k,Wfd%gbound,0,Wfd%MPI_enreg)
         
         ncalc = ncalc + 1
         Wfd%Wave(band,ik_ibz,spin)%has_ur=WFD_STORED  ! Update the status
       end if

     end do
   end do
 end do

end subroutine wfd_mkall_ur
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_ug2cprj
!! NAME
!! wfd_ug2cprj
!!
!! FUNCTION
!!  Calculate the projected wave function <Proj_i|Cnk> with all NL projectors for a single
!!  k-point, band and spin.
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=Structure containing the wave functions for the GW.
!!  ik_ibz=Index of the required k-point
!!  spin=Required spin index.
!!  choice=chooses possible output:
!!    In addition to projected wave function:
!!    choice=1 => nothing else
!!          =2 => 1st gradients with respect to atomic position(s)
!!          =3 => 1st gradients with respect to strain(s)
!!          =23=> 1st gradients with respect to atm. pos. and strain(s)
!!          =4 => 2nd derivatives with respect to atomic pos.
!!          =24=> 1st and 2nd derivatives with respect to atomic pos.
!!          =5 => 1st gradients with respect to k wavevector
!!          =6 => 2nd derivatives with respect to strain and atm. pos.
!!  idir=direction of the derivative, i.e. dir. of - atom to be moved  in the case choice=2
!!                                                 - strain component  in the case choice=3
!!                                                 - k point direction in the case choice=5
!!       Compatible only with choice=2,3,5; if idir=0, all derivatives are computed
!!  natom
!!  Cryst
!!  [sorted]=Logical flags defining if the output Cprj has to be sorted by atom type or not.
!!    By default, Cprj matrix elements are unsorted.
!!
!! OUTPUT
!!  cwaveprj
!!
!! PARENTS
!!      classify_bands,m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_ug2cprj(Wfd,band,ik_ibz,spin,choice,idir,natom,Cryst,cwaveprj,sorted)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_abiutil
 use interfaces_65_nonlocal
!End of the abilint section

 implicit none

!Arguments -------------------------------
!scalars
 integer,intent(in) :: choice,idir,natom,band,ik_ibz,spin
 logical,optional,intent(in) :: sorted
 type(wfs_descriptor),intent(inout) :: Wfd
 type(crystal_structure),intent(in) :: Cryst
!arrays
 type(cprj_type),intent(inout) :: cwaveprj(natom,Wfd%nspinor)

!Local variables-------------------------------
!scalars
 integer :: cpopt,istwf_k,npw_k,nkpg,istat
 integer :: dimekb1,dimekb2,matblk,ia,iatm,dimffnl,itypat,iatom,isp
 real(dp) :: arg
 logical :: want_sorted
!arrays
 integer,pointer :: kg_k(:,:)
 integer,allocatable :: dimcprj_srt(:)
 real(dp) :: dummy_ekb(0,0)
 real(dp) :: kpoint(3)
 real(dp),allocatable :: kpg(:,:)
 real(dp),pointer :: phkxred(:,:) 
 real(dp),allocatable :: cwavef(:,:)
 !real(dp),allocatable :: ph1d(2,3*(2*mgfft+1)*natom)
 real(dp),pointer :: ph3d(:,:,:)    ! ph3d(2,npw_k,matblk)
 real(dp),pointer :: ffnl(:,:,:,:)  ! ffnl(npw_k,dimffnl,lmnmax,ntypat)
 type(cprj_type),allocatable :: Cprj_srt(:,:)

! *********************************************************************

 ! different form factors have to be calculated and stored in Wfd.
 ABI_CHECK(choice==1,"choice/=1 not coded")

 npw_k   =  Wfd%npwwfn            ! TODO these quantities should be k-dependent.
 istwf_k =  Wfd%istwfk(ik_ibz)

 kpoint  =  Wfd%kibz(:,ik_ibz)

 kg_k    => Wfd%Kdata(ik_ibz)%kg_k
 dimffnl =  Wfd%Kdata(ik_ibz)%dim_fnlylm 
 ph3d    => Wfd%Kdata(ik_ibz)%ph3d
 ffnl    => Wfd%Kdata(ik_ibz)%fnlylm
 !$phkxred => Wfd%Kdata(ik_ibz)%phkxred

! Compute (k+G) vectors
 nkpg=0
 !$if (choice==3.or.choice==2.or.choice==23) nkpg=3*Wfd%nloalg(5)
 !$if (choice==4.or.choice==24) nkpg=9*Wfd%nloalg(5)
 allocate(kpg(npw_k,nkpg)); if (nkpg>0) call mkkpg(kg_k,kpg,kpoint,nkpg,npw_k)

 ! Allocate and compute the arrays phkxred and ph3d
 allocate(phkxred(2,Cryst%natom))
 do ia=1,Cryst%natom
   iatm=min(Cryst%atindx(ia),Cryst%natom)
   arg=two_pi*(DOT_PRODUCT(kpoint,Cryst%xred(:,ia)))
   phkxred(1,iatm)=DCOS(arg) 
   phkxred(2,iatm)=DSIN(arg)
 end do

 matblk = Cryst%natom
 !
 ! Copy wavefunction in reciprocal space.
 allocate(cwavef(2,npw_k*Wfd%nspinor))
 cwavef(1,:) = DBLE (Wfd%Wave(band,ik_ibz,spin)%ug)
 cwavef(2,:) = AIMAG(Wfd%Wave(band,ik_ibz,spin)%ug)

 cpopt   = 0 ! Nothing is already calculated.
 dimekb1 = 0
 dimekb2 = 0

 want_sorted=.FALSE.; if (PRESENT(sorted)) want_sorted=sorted

 if (want_sorted) then ! Output cprj are sorted.

   call getcprj(choice,cpopt,cwavef,cwaveprj,dimekb1,dimekb2,dimffnl,dummy_ekb,ffnl,&
&    idir,Wfd%indlmn,istwf_k,kg_k,kpg,kpoint,Wfd%lmnmax,matblk,Wfd%mgfft,Wfd%MPI_enreg,&
&    Cryst%natom,Cryst%nattyp,Wfd%ngfft,nkpg,Wfd%nloalg,npw_k,Wfd%nspinor,Cryst%ntypat,&
&    phkxred,Wfd%ph1d,ph3d,Cryst%ucvol,Wfd%usepaw,1)

 else  ! Output cprj are unsorted.

   allocate(dimcprj_srt(Cryst%natom)); ia=0
   do itypat=1,Cryst%ntypat
     dimcprj_srt(ia+1:ia+Cryst%nattyp(itypat))=Wfd%nlmn_type(itypat)
     ia=ia+Cryst%nattyp(itypat)
   end do

   allocate(Cprj_srt(natom,Wfd%nspinor))
   call cprj_alloc(Cprj_srt,0,dimcprj_srt)
   deallocate(dimcprj_srt)
   !
   ! Calculate sorted cprj.
   call getcprj(choice,cpopt,cwavef,Cprj_srt,dimekb1,dimekb2,dimffnl,dummy_ekb,ffnl,&
&    idir,Wfd%indlmn,istwf_k,kg_k,kpg,kpoint,Wfd%lmnmax,matblk,Wfd%mgfft,Wfd%MPI_enreg,&
&    Cryst%natom,Cryst%nattyp,Wfd%ngfft,nkpg,Wfd%nloalg,npw_k,Wfd%nspinor,Cryst%ntypat,&
&    phkxred,Wfd%ph1d,ph3d,Cryst%ucvol,Wfd%usepaw,1)
   !
   ! Reorder cprj (sorted --> unsorted)
   do iatom=1,Cryst%natom
     iatm=Cryst%atindx(iatom)
     do isp=1,Wfd%nspinor
       cwaveprj(iatom,isp)%cp=Cprj_srt(iatm,isp)%cp
     end do
   end do

   call cprj_free(Cprj_srt); deallocate(Cprj_srt)
 end if

 deallocate(cwavef,phkxred)
 deallocate(kpg,STAT=istat)

end subroutine wfd_ug2cprj
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/nullify_wave_0D
!! NAME
!!  nullify_wave_0D
!!
!! FUNCTION
!!  Set pointers to null.
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine nullify_wave_0D(Wave)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(wave_t),intent(inout) :: Wave

!************************************************************************

 !@wave_t
 nullify(Wave%ug)
 nullify(Wave%ur)

! types
 nullify(Wave%Cprj)

end subroutine nullify_wave_0D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/nullify_wave_3D
!! NAME
!!  nullify_wave_3D
!!
!! FUNCTION
!!  Set all pointers to null.
!!
!! PARENTS
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine nullify_wave_3D(Wave)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wave_t),intent(inout) :: Wave(:,:,:)

!Local variables ------------------------------
!scalars
 integer :: i1,i2,i3

!************************************************************************

 do i3=1,SIZE(Wave,DIM=3)
   do i2=1,SIZE(Wave,DIM=2)
     do i1=1,SIZE(Wave,DIM=1)
       call nullify_wave_0D(Wave(i1,i2,i3))
     end do
   end do
 end do

end subroutine nullify_wave_3D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/init_wave_0D
!! NAME
!!  init_wave_0D
!!
!! FUNCTION
!!   Main creation method for the wave_t data type
!!
!! INPUTS
!!  usepaw=1 if PAW is used. 
!!  npw =Number of plane-waves for ug
!!  nfft=Number of FFT points for the real space wavefunction.
!!  nspinor=Number of spinor components.
!!  natom=Number of atoms in cprj matrix elements.
!!  nlmn_size(natom)=Number of (n,l,m) channel for each atom.  Ordering of atoms depends on cprj_order
!!  cprj_order=Flag defining the ordering of the atoms in the cprj matrix elements (CPR_RANDOM|CPR_SORTED).
!!    Use to know if we have to reorder the matrix elements when wfd_get_cprj is called.
!!
!! OUTPUT
!!  Wave<wave_t>=The structure completetly initialized.
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine init_wave_0D(Wave,usepaw,npw,nfft,nspinor,natom,nlmn_size,cprj_order)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_abiutil
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: npw,nfft,nspinor,usepaw,natom,cprj_order
 type(wave_t),intent(inout) :: Wave
!arrays
 integer,intent(in) :: nlmn_size(:)

!Local variables ------------------------------
!scalars
 integer,parameter :: ncpgr0=0  ! For the time being, no derivatives

!************************************************************************

 !@wave_t
 if (npw >0) then 
   allocate(Wave%ug(npw*nspinor)); Wave%has_ug=WFD_ALLOCATED; Wave%ug=czero
   if (usepaw==1) then
     allocate(Wave%Cprj(natom,nspinor))
     call cprj_alloc(Wave%Cprj,ncpgr0,nlmn_size)
     Wave%has_cprj=WFD_ALLOCATED
     Wave%cprj_order=cprj_order
   end if
 end if

 if (nfft>0) then 
   allocate(Wave%ur(nfft*nspinor)); Wave%ur=czero; Wave%has_ur=WFD_ALLOCATED
 end if

end subroutine init_wave_0D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/destroy_wave_0D
!! NAME
!!  destroy_wave_0D
!!
!! FUNCTION
!!  Main destruction method for the wave_t datatype.
!!
!! INPUTS
!!  [what]=String defining what has to be freed.
!!     "A" =Both ug and ur and Cprj. Default.
!!     "G" =Only ug.
!!     "R" =Only ur
!!     "C" =Only PAW Cprj.
!!
!! SIDE EFFECTS
!!  Memory in Wave is deallocated depending on what
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine destroy_wave_0D(Wave,what)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_abiutil
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 character(len=*),optional,intent(in) :: what
 type(wave_t),intent(inout) :: Wave

!Local variables ------------------------------
!scalars
 character(len=10) :: my_what

!************************************************************************

 !@wave_t
 my_what="ALL"; if (PRESENT(what)) my_what=toupper(what)

 if ( .not.starts_with(my_what,(/"A", "G", "R", "C"/) )) then
   MSG_ERROR("unknow what"//TRIM(what))
 end if

 if ( starts_with(my_what,(/"A", "G"/) )) then
   if (associated(Wave%ug)) deallocate(Wave%ug)
   Wave%has_ug=WFD_NOWAVE
 end if

 if ( starts_with(my_what,(/"A", "R"/) )) then
   if (associated(Wave%ur)) deallocate(Wave%ur)
   Wave%has_ur=WFD_NOWAVE
 end if

 if ( starts_with(my_what,(/"A", "C"/) )) then
   if (associated(Wave%Cprj)) then 
     call cprj_free(Wave%Cprj); deallocate(Wave%Cprj)
   end if
   Wave%has_cprj=WFD_NOWAVE
 end if

end subroutine destroy_wave_0D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/destroy_wave_3D
!! NAME
!!  destroy_wave_3D
!!
!! FUNCTION
!!   Destruction method used for a 3-D arrays of wave_t datatyps.
!!
!! INPUTS
!!  Wave(:,:,:)<wave_t>=The array of structures.
!!  [what]=String defining what has to be freed.
!!     "A"=Both ug and ur as PAW Cprj, if any. Default.
!!     "G"  =Only ug.
!!     "R"  =Only ur
!!     "C"  =Only PAW Cprj.
!!  [mask(:,:,:)]=Mask used to select the elements that have to be deallocated. All of them, if not specified.
!!
!! SIDE EFFECTS
!!  Memory in Wave is deallocated depending on what and mask.
!!
!! OUTPUT
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine destroy_wave_3D(Wave,what,mask)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 character(len=*),optional,intent(in) :: what
 type(wave_t),intent(inout) :: Wave(:,:,:)
!arrays
 logical,optional,intent(in) :: mask(:,:,:)

!Local variables ------------------------------
!scalars
 integer :: i1,i2,i3
 logical :: do_free 
 character(len=10) :: my_what

!************************************************************************

 my_what="ALL"; if (PRESENT(what)) my_what=toupper(what)

 do i3=1,SIZE(Wave,DIM=3)
   do i2=1,SIZE(Wave,DIM=2)
     do i1=1,SIZE(Wave,DIM=1)

       do_free=.TRUE.; if (PRESENT(mask)) do_free=mask(i1,i2,i3) 
       if (do_free) call destroy_wave_0D(Wave(i1,i2,i3),what=my_what)

     end do
   end do
 end do

end subroutine destroy_wave_3D
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_push_ug
!! NAME
!!  wfd_push_ug
!!
!! FUNCTION
!!  This routine changes the status of the object by saving the wavefunction in the correct
!!  slot inside Wfd%Wave. It also set the corresponding has_ug flag to WFD_STORED.
!!  If the status of the corresponding ur is (WFD_STORED|WFD_ALLOCATED), then an G->R FFT transform 
!!  is done (see also update_ur)
!!
!! INPUTS
!!   band=Band index.
!!   ik_ibz=k-point index
!!   spin=Spin index.
!!   Cryst<crystal_structure>=Object defining the unit cell and its symmetries.
!!   ug(Wfd%npwwfn*Wfd%nspinor)=The ug to be saved.
!!   [update_ur]=If .FALSE.: no G-->R transform is done even if ur is (WFD_STORED|WFD_ALLOCATED) so be careful.
!!               Defaults to .TRUE.
!!   [update_cprj]=If .FALSE.: <C|p_i> matrix elements are not recalculatedd even if cprj is (WFD_STORED|WFD_ALLOCATED) so be careful.
!!               Defaults to .TRUE.
!!
!! SIDE EFFECTS 
!!   Wfd<wfs_descriptor>=See above.
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_push_ug(Wfd,band,ik_ibz,spin,Cryst,ug,update_ur,update_cprj)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin,band
 logical,optional,intent(in) :: update_ur,update_cprj
 type(wfs_descriptor),intent(inout) :: Wfd
 type(crystal_structure),intent(in) :: Cryst
!arrays
 complex(gwpc),intent(in) :: ug(Wfd%npwwfn*Wfd%nspinor)

!Local variables ------------------------------
!scalars
 integer,parameter :: choice1=1,idir0=0,tim_fourdp=5
 logical :: do_update_ur,do_update_cprj,want_sorted
 character(len=500) :: msg
!arrays
 integer,pointer :: kg_k(:,:)

!************************************************************************

#ifdef DEBUG_MODE
 if (.not.wfd_ihave_ug(Wfd,band,ik_ibz,spin)) then
   write(msg,'(a,i0,a,3(i0,1x))')" Node ",Wfd%my_rank," doesn't have ug for (band, ik_ibz, spin): ",band,ik_ibz,spin
   MSG_PERS_ERROR(msg)
 end if
#endif

 !@wfs_descriptor
 Wfd%Wave(band,ik_ibz,spin)%ug = ug
 Wfd%Wave(band,ik_ibz,spin)%has_ug = WFD_STORED

 if (Wfd%usepaw==1.and.wfd_ihave_cprj(Wfd,band,ik_ibz,spin)) then ! Update the corresponding cprj if required.
   do_update_cprj=.TRUE.; if (PRESENT(update_cprj)) do_update_cprj=update_cprj
   if (do_update_cprj) then
     want_sorted = (Wfd%Wave(band,ik_ibz,spin)%cprj_order == CPR_SORTED)
     call wfd_ug2cprj(Wfd,band,ik_ibz,spin,choice1,idir0,Wfd%natom,Cryst,Wfd%Wave(band,ik_ibz,spin)%Cprj,sorted=want_sorted)
     Wfd%Wave(band,ik_ibz,spin)%has_cprj = WFD_STORED
   else 
     Wfd%Wave(band,ik_ibz,spin)%has_cprj = WFD_ALLOCATED
   end if
 end if

 if (wfd_ihave_ur(Wfd,band,ik_ibz,spin)) then ! Update the corresponding ur if required.
   do_update_ur=.TRUE.; if (PRESENT(update_ur)) do_update_ur=update_ur

   if (do_update_ur) then
     kg_k => Wfd%Kdata(ik_ibz)%kg_k 
     call fft_onewfn(Wfd%paral_kgb,Wfd%istwfk(ik_ibz),Wfd%nspinor,Wfd%npwwfn,Wfd%nfft,&
&      ug,Wfd%Wave(band,ik_ibz,spin)%ur,Wfd%igfft0,Wfd%ngfft,kg_k,Wfd%gbound,tim_fourdp,Wfd%MPI_enreg)
     Wfd%Wave(band,ik_ibz,spin)%has_ur = WFD_STORED
   else 
     Wfd%Wave(band,ik_ibz,spin)%has_ur = WFD_ALLOCATED
   end if

 end if

 RETURN
 ABI_UNUSED(msg)

end subroutine wfd_push_ug
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_rank_has_ug
!! NAME
!!  wfd_rank_has_ug
!!
!! FUNCTION
!!  This function is used to ask a particular processor whether it has a particular ug and with which status.
!!
!! INPUTS
!!   rank=The MPI rank of the processor.
!!   band=Band index.
!!   ik_ibz=k-point index
!!   spin=Spin index.
!!
!! NOTES
!!   A zero index can be used to inquire the status of a bunch of states.
!!   Thus (band,ik_ibz,spin) = (0,1,1) means: Do you have at least one band for the first k-point and the first spin. 
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_rank_has_ug(Wfd,rank,band,ik_ibz,spin) 

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin,rank
 logical :: wfd_rank_has_ug
 type(wfs_descriptor),intent(in) :: Wfd

!Local variables ------------------------------
!scalars
 integer :: nzeros,bks_flag
!arrays
 integer :: indeces(3) !,check(2)

!************************************************************************

 !check = (/WFD_ALLOCATED, WFD_STORED/)
 !if (PRESENT(how)) then
 !  if (starts_with(how,(/"A","a"/))) check = (/WFD_ALLOCATED, WFD_ALLOCATED/)
 !  if (starts_with(how,(/"S","s"/))) check = (/WFD_STORED, WFD_STORED/)
 !end if

 indeces = (/band,ik_ibz,spin/)
 bks_flag = HAS_BKS

 if ( ALL(indeces/=(/0,0,0/)) ) then
   wfd_rank_has_ug = (Wfd%bks_tab(band,ik_ibz,spin,rank) == bks_flag); RETURN
 else
   nzeros = COUNT(indeces==0)
   if (nzeros==3) MSG_ERROR("All indeces are zero!")
  
   if (band==0) then
     if (nzeros==1) wfd_rank_has_ug = ANY( Wfd%bks_tab(:,ik_ibz,spin,rank)==bks_flag)
     if (ik_ibz==0) wfd_rank_has_ug = ANY( Wfd%bks_tab(:,:,spin,rank)     ==bks_flag)
     if (spin  ==0) wfd_rank_has_ug = ANY( Wfd%bks_tab(:,ik_ibz,:,rank)   ==bks_flag)

   else if (ik_ibz==0) then
     if (nzeros==1) wfd_rank_has_ug = ANY( Wfd%bks_tab(band,:,spin,rank)==bks_flag)
     if (band  ==0) wfd_rank_has_ug = ANY( Wfd%bks_tab(:,:,spin,rank)   ==bks_flag)
     if (spin  ==0) wfd_rank_has_ug = ANY( Wfd%bks_tab(band,:,:,rank)   ==bks_flag)

   else
     if (nzeros==1) wfd_rank_has_ug = ANY( Wfd%bks_tab(band,ik_ibz,:,rank)==bks_flag)
     if (ik_ibz==0) wfd_rank_has_ug = ANY( Wfd%bks_tab(band,:,:,rank)     ==bks_flag)
     if (band  ==0) wfd_rank_has_ug = ANY( Wfd%bks_tab(:,ik_ibz,:,rank)   ==bks_flag)
   end if
 end if

end function wfd_rank_has_ug
!!***                                                                                                

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_ihave_ug
!! NAME
!!  wfd_ihave_ug
!!
!! FUNCTION
!!  This function is used to ask the processor whether it has a particular ug and with which status.
!!
!! INPUTS
!!   band=Band index.
!!   ik_ibz=k-point index
!!   spin=Spin index.
!!   [how]=string defining which status is checked.
!!     Possible mutually exclusive values: "Allocated", "Stored". 
!!     Only the first character is checked (no case-sensitive)
!!     By default the function returns .TRUE. if the wave is either WFD_ALLOCATED or WFD_STORED.
!!
!! NOTES
!!   A zero index can be used to inquire the status of a bunch of states.
!!   Thus (band,ik_ibz,spin) = (0,1,1) means: Do you have at least one band for the first k-point and the first spin. 
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_ihave_ug(Wfd,band,ik_ibz,spin,how) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 logical :: wfd_ihave_ug
 character(len=*),optional,intent(in) :: how
 type(wfs_descriptor),intent(in) :: Wfd

!************************************************************************

 wfd_ihave_ug = wfd_ihave(Wfd,"UG",band,ik_ibz,spin,how) 

end function wfd_ihave_ug
!!***                                                                                                

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_ihave_ur
!! NAME
!!  wfd_ihave_ur
!!
!! FUNCTION
!!  This function is used to ask the processor whether it has a particular ur and with which status.
!!
!! INPUTS
!!   band=Band index.
!!   ik_ibz=k-point index
!!   spin=Spin index.
!!   [how]=string defining which status is checked. By default the function returns
!!      .TRUE. if the wave is either WFD_ALLOCATED or WFD_STORED.
!!      Possible mutually exclusive values: "Allocated", "Stored". 
!!      Only the first character is checked (no case-sensitive)
!!
!! NOTES
!!   A zero index can be used to inquire the status of a bunch of states.
!!   Thus (band,ik_ibz,spin) = (0,1,1) means: Do you have at least one band for the first k-point and the first spin. 
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_ihave_ur(Wfd,band,ik_ibz,spin,how) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 logical :: wfd_ihave_ur
 character(len=*),optional,intent(in) :: how
 type(wfs_descriptor),intent(in) :: Wfd

!************************************************************************

 wfd_ihave_ur = wfd_ihave(Wfd,"UR",band,ik_ibz,spin,how) 

end function wfd_ihave_ur
!!***                                                                                                

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_ihave_cprj
!! NAME
!!  wfd_ihave_cprj
!!
!! FUNCTION
!!  This function is used to ask the processor whether it has a particular cprj and with which status.
!!
!! INPUTS
!!   band=Band index.
!!   ik_ibz=k-point index
!!   spin=Spin index.
!!   [how]=string defining which status is checked. By default the function returns
!!      .TRUE. if the wave is either WFD_ALLOCATED or WFD_STORED.
!!      Possible mutually exclusive values: "Allocated", "Stored". 
!!      Only the first character is checked (no case-sensitive)
!!
!! NOTES
!!   A zero index can be used to inquire the status of a bunch of states.
!!   Thus (band,ik_ibz,spin) = (0,1,1) means: Do you have at least one band for the first k-point and the first spin. 
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_ihave_cprj(Wfd,band,ik_ibz,spin,how) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 logical :: wfd_ihave_cprj
 character(len=*),optional,intent(in) :: how
 type(wfs_descriptor),intent(in) :: Wfd

!************************************************************************

 wfd_ihave_cprj = wfd_ihave(Wfd,"CPRJ",band,ik_ibz,spin,how) 

end function wfd_ihave_cprj
!!***                                                                                                

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_ihave
!! NAME
!!  wfd_ihave
!!
!! FUNCTION
!!  This function is used to ask the processor whether it has a particular (ug|ur|cprj) and with which status.
!!
!! INPUTS
!!   band=Band index.
!!   ik_ibz=k-point index
!!   spin=Spin index.
!!   what=String defining what has to be tested.
!!     ug 
!!     ur
!!     cprj
!!   [how]=string defining which status is checked.
!!     Possible mutually exclusive values: "Allocated", "Stored". 
!!     Only the first character is checked (no case-sensitive)
!!     By default the function returns .TRUE. if the wave is either WFD_ALLOCATED or WFD_STORED.
!!
!! NOTES
!!   A zero index can be used to inquire the status of a bunch of states.
!!   Thus (band,ik_ibz,spin) = (0,1,1) means: Do you have at least one band for the first k-point and the first spin. 
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_ihave(Wfd,what,band,ik_ibz,spin,how) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 logical :: wfd_ihave
 character(len=*),intent(in) :: what
 character(len=*),optional,intent(in) :: how
 type(wfs_descriptor),intent(in) :: Wfd

!Local variables ------------------------------
!scalars
 integer :: nzeros
 !character(len=500) :: msg
!arrays
 integer :: indeces(3),check(2)
 integer,pointer :: has_flags(:,:,:) 

!************************************************************************

 check = (/WFD_ALLOCATED, WFD_STORED/)
 if (PRESENT(how)) then
   if (starts_with(how,(/"A","a"/))) check = (/WFD_ALLOCATED, WFD_ALLOCATED/)
   if (starts_with(how,(/"S","s"/))) check = (/WFD_STORED, WFD_STORED/)
 end if

 indeces = (/band,ik_ibz,spin/)

 select case (toupper(what))
 case ("UG")
   has_flags => Wfd%Wave(:,:,:)%has_ug
 case ("UR")
   has_flags => Wfd%Wave(:,:,:)%has_ur
 case ("CPRJ")
   has_flags => Wfd%Wave(:,:,:)%has_cprj
 case default
   MSG_ERROR("Wrong what"//TRIM(what))
 end select

 if ( ALL(indeces/=(/0,0,0/)) ) then
   wfd_ihave = ( ANY(has_flags(band,ik_ibz,spin) == check )); RETURN
 else
   nzeros = COUNT(indeces==0)
   if (nzeros==3) MSG_ERROR("All indeces are zero!")
  
   if (band==0) then
     if (nzeros==1) wfd_ihave = ANY( has_flags(:,ik_ibz,spin)==check(1) .or.&
&                                    has_flags(:,ik_ibz,spin)==check(2) )

     if (ik_ibz==0) wfd_ihave = ANY( has_flags(:,:,spin)==check(1) .or.&
&                                    has_flags(:,:,spin)==check(2) )

     if (spin  ==0) wfd_ihave = ANY( has_flags(:,ik_ibz,:)==check(1) .or.&
&                                    has_flags(:,ik_ibz,:)==check(2) )

   else if (ik_ibz==0) then
     if (nzeros==1) wfd_ihave = ANY( has_flags(band,:,spin)==check(1) .or.&
&                                    has_flags(band,:,spin)==check(2) )

     if (band  ==0) wfd_ihave = ANY( has_flags(:,:,spin)==check(1) .or.&
&                                    has_flags(:,:,spin)==check(2) )

     if (spin  ==0) wfd_ihave = ANY( has_flags(band,:,:)==check(1) .or.&
&                                    has_flags(band,:,:)==check(2) )
   else
     if (nzeros==1) wfd_ihave = ANY( has_flags(band,ik_ibz,:)==check(1) .or.&
&                                    has_flags(band,ik_ibz,:)==check(2) ) 

     if (ik_ibz==0) wfd_ihave = ANY( has_flags(band,:,:)==check(1) .or.&
&                                    has_flags(band,:,:)==check(2) )

     if (band  ==0) wfd_ihave = ANY( has_flags(:,ik_ibz,:)==check(1) .or.&
&                                    has_flags(:,ik_ibz,:)==check(2) )
   end if
 end if

end function wfd_ihave
!!***                                                                                                

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_mybands
!! NAME
!!  wfd_mybands
!!
!! FUNCTION
!!  Return the list of band indeces of the ug owned by this node at given (k,s). 
!!
!! INPUTS
!!  ik_ibz=Index of the k-point in the IBZ 
!!  spin=spin index
!!  [how]=string defining which status is checked.
!!    Possible mutually exclusive values: "Allocated", "Stored". 
!!    Only the first character is checked (no case-sensitive)
!!    By default the list of bands whose status is either WFD_ALLOCATED or WFD_STORED is returned.
!!
!! OUTPUT
!!  how_manyb=The number of bands owned by this node
!!  my_band_list(Wfd%mband)=The first how_manyb values are the bands treated by this node.
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_mybands(Wfd,ik_ibz,spin,how_manyb,my_band_list,how) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin
 integer,intent(out) :: how_manyb
 character(len=*),optional,intent(in) :: how
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 integer,intent(out) :: my_band_list(Wfd%mband)

!Local variables ------------------------------
!scalars
 integer :: band
 logical :: do_have

!************************************************************************

 how_manyb=0; my_band_list=-1
 do band=1,Wfd%nband(ik_ibz,spin)
   if (PRESENT(how)) then   
     do_have = wfd_ihave_ug(Wfd,band,ik_ibz,spin,how=how) 
   else
     do_have = wfd_ihave_ug(Wfd,band,ik_ibz,spin) 
   end if
   if (do_have) then
     how_manyb = how_manyb +1
     my_band_list(how_manyb)=band
   end if
 end do

end subroutine wfd_mybands
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_bands_of_proc
!! NAME
!!  wfd_bands_of_proc
!!
!! FUNCTION
!!  Return the list of band index of the ug owned by a given processor at given (k,s). 
!!
!! INPUTS
!!  Wfd
!!  rank=The MPI rank of the processor.
!!  ik_ibz=Index of the k-point in the IBZ 
!!  spin=spin index
!!
!! OUTPUT
!!  how_manyb=The number of bands owned by this node
!!  rank_band_list(Wfd%mband)=The first how_manyb values are the bands treated by the node.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

subroutine wfd_bands_of_rank(Wfd,rank,ik_ibz,spin,how_manyb,rank_band_list) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin,rank
 integer,intent(out) :: how_manyb
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 integer,intent(out) :: rank_band_list(Wfd%mband)

!Local variables ------------------------------
!scalars
 integer :: band
 logical :: it_has

!************************************************************************

 how_manyb=0; rank_band_list=-1
 do band=1,Wfd%nband(ik_ibz,spin)
   it_has = wfd_rank_has_ug(Wfd,rank,band,ik_ibz,spin) 
   if (it_has) then
     how_manyb = how_manyb +1
     rank_band_list(how_manyb)=band
   end if
 end do

end subroutine wfd_bands_of_rank
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_get_ug
!! NAME
!!  wfd_get_ug
!!
!! FUNCTION
!!  Get a copy of a wave function in reciprocal space.
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=the data type
!!  band=the index of the band.
!!  ik_ibz=Index of the k-point in the IBZ 
!!  spin=spin index
!!
!! OUTPUT
!!  ug(Wfd%npwwfn*Wfd%nspinor)=The required wavefunction in reciprocal space.
!!
!! PARENTS
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_get_ug(Wfd,band,ik_ibz,spin,ug)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays
 complex(gwpc),intent(out) :: ug(Wfd%npwwfn*Wfd%nspinor)

!Local variables ------------------------------
!scalars
 character(len=500) :: msg
!************************************************************************

 if (wfd_ihave_ug(Wfd,band,ik_ibz,spin,"Stored")) then 
   !ug = Wfd%Wave(band,ik_ibz,spin)%ug
   call xcopy(Wfd%npwwfn*Wfd%nspinor,Wfd%Wave(band,ik_ibz,spin)%ug,1,ug,1)
 else
   write(msg,'(a,i0,a,3i0)')" Node ",Wfd%my_rank," doesn't have (band,ik_ibz,spin)=",band,ik_ibz,spin
   MSG_PERS_BUG(msg)
 end if

end subroutine wfd_get_ug
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_ptr_ug
!! NAME
!!  wfd_ptr_ug
!!
!! FUNCTION
!!  Returns a pointer to ug
!!  WARNING: Do not use the returned pointer to write on this location of memory.
!!   The status of the object should always be modified via the appropriate method.
!!   Use the pointer only if you want to avoid a copy and you are not going to change the ug!
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=the data type
!!  band=the index of the band.
!!  ik_ibz=Index of the k-point in the IBZ 
!!  spin=spin index
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_ptr_ug(Wfd,band,ik_ibz,spin)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 complex(gwpc),pointer :: wfd_ptr_ug(:)

!Local variables ------------------------------
!scalars
 character(len=500) :: msg   

!************************************************************************

#ifdef DEBUG_MODE
 if (.not.wfd_ihave_ug(Wfd,band,ik_ibz,spin)) then
   write(msg,'(a,i0,a,3(i0,1x))')" Node ",Wfd%my_rank," doesn't have ug for (band, ik_ibz, spin): ",band,ik_ibz,spin
   MSG_PERS_ERROR(msg)
 end if
#endif

 wfd_ptr_ug => Wfd%Wave(band,ik_ibz,spin)%ug

 RETURN
 ABI_UNUSED(msg)

end function wfd_ptr_ug
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_wave_free
!! NAME
!!  wfd_wave_free
!!
!! FUNCTION
!!  Free the set of waves specified by mask.
!!
!! INPUTS
!!  mask(Wfd%mband,Wfd%nkibz,Wfd%nsppol)=.TRUE. if the memory allocated for 
!!    this state has to be freed
!!  [what]=String specifying which array have to be deallocated. 
!!    Possible values (no case-sensitive).
!!      "All"= To free both ug and ur and PAW Cprj, if any. Default
!!      "G"  = Only ug
!!      "R"  = Only ur.
!!      "C"  = Only PAW Cprj.
!!
!! SIDE EFFECTS
!!  Wfd<wfs_descriptor>=See above.
!!
!! PARENTS
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_wave_free(Wfd,what,bks_mask)


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd
 character(len=*),optional,intent(in) :: what
!arrays
 logical,optional,intent(in) :: bks_mask(Wfd%mband,Wfd%nkibz,Wfd%nsppol)

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,spin,band !,ierr,istat
 logical :: do_free
 character(len=10) :: my_what
!************************************************************************

 my_what="ALL"; if (PRESENT(what)) my_what=toupper(what)

 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz
     do band=1,Wfd%mband

        do_free=.TRUE.; if (PRESENT(bks_mask)) do_free=bks_mask(band,ik_ibz,spin)
        if (do_free) then
          call destroy_wave_0D(Wfd%Wave(band,ik_ibz,spin),what=my_what)
          if ( starts_with(my_what,(/"A", "G"/) )) then ! Update the associated flags.
            Wfd%bks_tab(band,ik_ibz,spin,Wfd%my_rank) = NO_BKS
          end if
        end if

     end do
   end do
 end do

end subroutine wfd_wave_free
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_who_has_ug
!! NAME
!!  wfd_who_has_ug
!!
!! FUNCTION
!!  Return the number of processors having a particular (b,k,s) state as well as their MPI rank.
!!  Warning: Wfd%bks_tab is supposed to be up-to-date (see wfd_update_bkstab).
!!
!! INPUTS
!!  band=the index of the band.
!!  ik_ibz=Index of the k-point in the IBZ 
!!  spin=spin index
!!  [comm]=MPI communicator. Not used TODO.
!!
!! OUTPUT
!!  how_many=The number of nodes owing this ug state.
!!  proc_ranks(1:how_many)=Gives the MPI rank of the nodes owing the state.
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_who_has_ug(Wfd,band,ik_ibz,spin,how_many,proc_ranks,comm) 

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 integer,optional,intent(in) :: comm
 integer,intent(out) :: how_many
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 integer,intent(out) :: proc_ranks(Wfd%nproc)

!Local variables ------------------------------
!scalars
 integer :: irank,my_comm
!arrays

!************************************************************************

 my_comm=Wfd%comm
 if (PRESENT(comm)) then
   my_comm=comm
   MSG_ERROR("Not implemented error")
 end if

 how_many=0; proc_ranks=-1
 do irank=0,Wfd%nproc-1
   if (Wfd%bks_tab(band,ik_ibz,spin,irank)==HAS_BKS) then
     how_many = how_many +1
     proc_ranks(how_many)=irank
   end if
 end do

end subroutine wfd_who_has_ug
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_everybody_has_ug
!! NAME
!!  wfd_everybody_has_ug
!!
!! FUNCTION
!!  Return .TRUE. if all the nodes inside comm own the specified ug state.
!!
!! INPUTS
!!  band=the index of the band.
!!  ik_ibz=Index of the k-point in the IBZ 
!!  spin=spin index
!!  [comm]=MPI communicator. Not used TODO.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_everybody_has_ug(Wfd,band,ik_ibz,spin,comm) result(answer)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin 
 integer,optional,intent(in) :: comm
 logical :: answer
 type(wfs_descriptor),intent(in) :: Wfd
!arrays

!Local variables ------------------------------
!scalars
 integer :: my_comm,how_many,nzeros,ib
!arrays
 integer :: proc_ranks(Wfd%nproc)
 integer :: indeces(3) !,check(2)

!************************************************************************

 my_comm=Wfd%comm; if (PRESENT(comm)) my_comm=comm

 indeces = (/band,ik_ibz,spin/)

 if ( ALL(indeces/=(/0,0,0/)) ) then
   call wfd_who_has_ug(Wfd,band,ik_ibz,spin,how_many,proc_ranks,comm=my_comm) 
   answer = (how_many==Wfd%nproc); RETURN
 else
   nzeros = COUNT(indeces==0)
   if (nzeros==3) MSG_ERROR("All indeces are zero!")

   answer=.TRUE.
   MSG_WARNING("Some cases are not coded!")
   ! TODO 

   if (band==0) then

     if (nzeros==1)  then     ! All the bands for the given k-point and spin?
       ib=0
       do while(answer.and.ib<Wfd%nband(ik_ibz,spin))
         ib=ib+1
         call wfd_who_has_ug(Wfd,ib,ik_ibz,spin,how_many,proc_ranks,comm=my_comm) 
         answer = (how_many==Wfd%nproc)
       end do; RETURN

     else if (ik_ibz==0) then ! All the bands and all the k-points for the the given spin?

     else if (spin==0) then   ! All the bands and all the spins for the given k-point?

     end if

   else if (ik_ibz==0) then
     if (nzeros==1) then     ! All the k-points for the the given band and spin?

     else if (band==0) then  ! All the k-points and all the bands for the the given spin?

     else if (spin==0) then  ! All the k-points and all the spins for the the given band?

     end if

   else
     if (nzeros==1) then      ! All the spins for the the given band and k-point?

     else if (ik_ibz==0) then ! All the spins and all the k-points for the the given band?

     else if (band==0) then   ! All the spins and all the bands for the the given k-point?

     end if
   end if

   MSG_ERROR("Not implemented error")
 end if

end function wfd_everybody_has_ug
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_update_bkstab
!! NAME
!!  wfd_update_bkstab
!!
!! FUNCTION
!!  This routine should be called by all the nodes before any MPI operation involving the object.
!!  It updates the bks_tab storing information on the distribution of ug.
!!
!! SIDE EFFECTS
!!  Wfs%bks_tab
!!
!! PARENTS
!!      calc_density,gw_tools,m_io_kss,m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_update_bkstab(Wfd) 

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd

!Local variables ------------------------------
!scalars
 integer :: ierr 

!************************************************************************

 ! Fill my slice of the table.
 Wfd%bks_tab = NO_BKS
 where (Wfd%Wave(:,:,:)%has_ug == WFD_STORED)
   Wfd%bks_tab(:,:,:,Wfd%my_rank) = HAS_BKS
 end where

 call xsum_mpi(Wfd%bks_tab,Wfd%comm,ierr)

end subroutine wfd_update_bkstab
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_distribute_bands
!! NAME
!!  wfd_distribute_bands
!!
!! FUNCTION
!!  This routines distributes a set of band indeces taking into account the 
!!  distribution of the ug.
!!
!! INPUTS
!!  band=the index of the band.
!!  ik_ibz=Index of the k-point in the IBZ 
!!  spin=spin index
!!  [got(Wfd%nproc)]=The number of tasks already assigned to the nodes.
!!  [bmask(Wfd%mband)]=The routine will raise an error if one band index
!!    is not treated by any processor. bmask can be used to select the subset of 
!!    indeces that are expected to be available. 
!!
!! OUTPUT
!!   my_nband=The number of bands that will be treated by this node.
!!   my_band_list(1:my_nband)=The band indeces for this node
!!
!! PARENTS
!!      gw_tools,m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_distribute_bands(Wfd,ik_ibz,spin,my_nband,my_band_list,got,bmask)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin 
 integer,intent(out) :: my_nband
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 integer,intent(out) :: my_band_list(Wfd%mband)
 integer,optional,intent(inout) :: got(Wfd%nproc)
 logical,optional,intent(in) :: bmask(Wfd%mband)

!Local variables ------------------------------
!scalars
 integer :: band,how_many,idle
 character(len=500) :: msg
!arrays
 integer :: proc_ranks(Wfd%nproc),get_more(Wfd%nproc)
 logical :: rank_mask(Wfd%nproc)

!************************************************************************

 my_nband=0; my_band_list=0
 get_more=0; if (PRESENT(got)) get_more = got

 do band=1,Wfd%nband(ik_ibz,spin)
   if (PRESENT(bmask)) then
     if (.not.bmask(band)) CYCLE
   end if

   call wfd_who_has_ug(Wfd,band,ik_ibz,spin,how_many,proc_ranks) 

   if (how_many==1) then ! I am the only one owing this band. Add it to list.
     if (proc_ranks(1) == Wfd%my_rank) then 
       my_nband=my_nband + 1 
       my_band_list(my_nband) = band
     end if
   else if (how_many>0) then  ! This band is duplicated. Assign it trying to obtain a good load distribution.
     rank_mask=.FALSE.; rank_mask(proc_ranks(1:how_many)+1)=.TRUE.
     idle = imin_loc(get_more,mask=rank_mask)
     get_more(idle) = get_more(idle) + 1
     if (Wfd%my_rank==idle-1) then 
       my_nband=my_nband + 1 
       my_band_list(my_nband) = band
     end if
   else 
     write(msg,'(a,3(i0,1x))')" No processor has (band, ik_ibz, spin): ",band,ik_ibz,spin 
     MSG_PERS_ERROR(msg)
   end if
 end do

 if (PRESENT(got)) got = get_more

end subroutine wfd_distribute_bands
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_rotate
!! NAME
!! wfd_rotate
!!
!! FUNCTION
!!  This routine performs a linear trasformation of the wavefunctions stored in Wfd
!!  taking into account memory distribution. The transformation is done in reciprocal
!!  space therefore all the ug should be available. Wavefunctions in real space are then
!!  obtained via FFT. The implementation assumes that the matrix associated to the 
!!  linear transformation is sparse (No BLAS-3 calls here).
!!
!! INPUTS
!!  Cryst<crystal_structure>=Object defining the unit cell and its symmetries.
!!  m_lda_to_qp(mband,mband,nkibz,nsppol)= expansion of the QP amplitudes in terms of KS wavefunctions.
!!  [bmask(mband,nkibz,nsppol)]=The routine will raise an error if one band index
!!    is not treated by any processor. bmask can be used to select the subset of 
!!    indeces that are expected to be available. 
!!
!! SIDE EFFECTS 
!!   Wfd<wfs_descriptor>=See above.
!!
!! PARENTS
!!      screening,sigma
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE
!!

subroutine wfd_rotate(Wfd,Cryst,m_lda_to_qp,bmask)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd
 type(crystal_structure),intent(in) :: Cryst
!arrays
 complex(dpc),target,intent(in) :: m_lda_to_qp(Wfd%mband,Wfd%mband,Wfd%nkibz,Wfd%nsppol)
 logical,optional,intent(in) :: bmask(Wfd%mband,Wfd%nkibz,Wfd%nsppol)

!Local variables-------------------------------
!scalars
 integer :: band,ik_ibz,spin,ierr,icol,nnew,inew,my_nband,ib
 !character(len=500) :: msg
!arrays
 integer :: new_list(Wfd%mband),my_band_list(Wfd%mband)
 complex(dpc),pointer :: umat_sk(:,:)
 complex(gwpc) :: mcol(Wfd%mband)
 complex(gwpc),allocatable :: new_ug(:,:)

!************************************************************************

 DBG_ENTER("COLL")

 ! Update the distribution table, first.
 call wfd_update_bkstab(Wfd) 
 !
 ! === Calculate : $\Psi^{QP}_{r,b} = \sum_n \Psi^{KS}_{r,n} M_{n,b}$ ===
 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz

     umat_sk => m_lda_to_qp(:,:,ik_ibz,spin)

     ! Select only those states that are mixed by the (sparse) m_lda_to_qp.
     nnew=0; new_list=0
     !do icol=1,Wfd%mband
     do icol=1,Wfd%nband(ik_ibz,spin)
       mcol = m_lda_to_qp(:,icol,ik_ibz,spin) 
       mcol(icol) = mcol(icol) - cone
       if (ANY(ABS(mcol)>tol12)) then  ! Avoid a simple copy.
         nnew=nnew+1
         new_list(nnew)=icol
       end if
     end do
     if (nnew==0) CYCLE ! Nothing to do.
     !
     ! Retrieve the set of band indeces that have to be treated by 
     ! this node taking into accoung a possible duplication.
     !
     if (PRESENT(bmask)) then
       call wfd_distribute_bands(Wfd,ik_ibz,spin,my_nband,my_band_list,bmask=bmask(:,ik_ibz,spin)) 
     else 
       call wfd_distribute_bands(Wfd,ik_ibz,spin,my_nband,my_band_list) 
     end if
  
     !if (my_nband>0) then
     !  write(*,*)" At (ik_ibz,spin) ",ik_ibz,spin,&
     !  & ", rank ",Wfd%my_rank," will sum ",my_nband," bands, my_band_list: ",my_band_list(1:my_nband)
     !end if

     allocate(new_ug(Wfd%nspinor*Wfd%npwwfn,nnew)); new_ug=czero
     do inew=1,nnew
       icol = new_list(inew)
       do ib=1,my_nband
         band = my_band_list(ib)
         if (ABS(umat_sk(band,icol))>tol12) then
           new_ug(:,inew) = new_ug(:,inew) + umat_sk(band,icol)* Wfd%Wave(band,ik_ibz,spin)%ug
         end if
       end do
     end do

     call xsum_mpi(new_ug,Wfd%comm,ierr)
     ! =======================================
     ! === Update the input wave functions ===
     ! =======================================
     do inew=1,nnew
       band = new_list(inew)
       if (wfd_ihave_ug(Wfd,band,ik_ibz,spin)) call wfd_push_ug(Wfd,band,ik_ibz,spin,Cryst,new_ug(:,inew))
     end do

     deallocate(new_ug)

   end do !ik_ibz
 end do !spin

 ! TODO This is needed only if FFTs are not done in wfd_push_ug. Do not know which one is faster.
 !call wfd_reset_ur(Wfd)

 call xbarrier_mpi(Wfd%comm)

 DBG_EXIT("COLL")

end subroutine wfd_rotate
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_iterator_bks
!! NAME
!!  wfd_iterator_bks
!!
!! FUNCTION
!!  This routines returns an iterator used to loop over bands, k-points and spin indeces 
!!  taking into account the distribution of the ug.
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=
!!  bks_mask(Wfd%mband.Wfd%nkibz,Wfd%nsppol)= mask used to select the (b,k,s) indeces.
!!
!! OUTPUT
!!  iter_bks<iter2_t>=Iterator over the bands treated by this node for each k-point and spin.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_iterator_bks(Wfd,bks_mask) result(iter_bks)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 logical,optional,intent(in) :: bks_mask(Wfd%mband,Wfd%nkibz,Wfd%nsppol)
 !type(coeffi1_type),intent(out) :: iter_bks(Wfd%nkibz,Wfd%nsppol)
 type(iter2_t) :: iter_bks

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,spin,my_nband 
 !character(len=500) :: msg
!arrays
 integer :: my_band_list(Wfd%mband)

!************************************************************************

 call iter_alloc(iter_bks,(/Wfd%nkibz,Wfd%nsppol/))

 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz

     if (PRESENT(bks_mask)) then
       call wfd_distribute_bands(Wfd,ik_ibz,spin,my_nband,my_band_list,bmask=bks_mask(:,ik_ibz,spin))
     else 
       call wfd_distribute_bands(Wfd,ik_ibz,spin,my_nband,my_band_list)
     end if

     !iter_bks(ik_ibz,spin)%size = my_nband
     !allocate(iter_bks(ik_ibz,spin)%value(my_nband))
     !if (my_nband>0) iter_bks(ik_ibz,spin)%value = my_band_list(1:my_nband)

     call iter_push(iter_bks,ik_ibz,spin,my_band_list(1:my_nband))
   end do
 end do

end function wfd_iterator_bks
!!***

!----------------------------------------------------------------------


!!****f* m_wfs/wfd_bks_distrb
!! NAME
!!  wfd_bks_distrb
!!
!! FUNCTION
!!  This routines build a local logical table indexed by bands, k-points and spin that defines
!!  the distribution of the load inside the loops according to the availability of the ug.
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=
!!  [bks_mask(Wfd%mband,Wfd%nkibz,Wfd%nsppol)]=Mask used to skip selecter (b,k,s) entries.
!!  [got(Wfd%nproc)]=The number of tasks already assigned to the nodes.
!!
!! OUTPUT
!!  bks_distrbk(Wfd%mband,Wfd%nkibz,Wfd%nsppol)=Global table with the rank of the node treating (b,k,s)
!!
!! PARENTS
!!      calc_density
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_bks_distrb(Wfd,bks_distrb,got,bks_mask)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 integer,intent(out) :: bks_distrb(Wfd%mband,Wfd%nkibz,Wfd%nsppol)
 integer,optional,intent(inout) :: got(Wfd%nproc)
 logical,optional,intent(in) :: bks_mask(Wfd%mband,Wfd%nkibz,Wfd%nsppol)

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,spin,band,how_many,idle
 character(len=500) :: msg
!arrays
 integer :: get_more(Wfd%nproc),proc_ranks(Wfd%nproc)
 logical :: rank_mask(Wfd%nproc)

!************************************************************************

 get_more=0; if (PRESENT(got)) get_more=got

 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz
     do band=1,Wfd%nband(ik_ibz,spin)

       if (PRESENT(bks_mask)) then
         if (.not.bks_mask(band,ik_ibz,spin)) CYCLE
       end if

       call wfd_who_has_ug(Wfd,band,ik_ibz,spin,how_many,proc_ranks) 

       if (how_many==1) then ! I am the only one owing this band. Add it to list.
         bks_distrb(band,ik_ibz,spin) = proc_ranks(1)

       else if (how_many>1) then ! This band is duplicated. Assign it trying to obtain a good load distribution.
         rank_mask=.FALSE.; rank_mask(proc_ranks(1:how_many)+1)=.TRUE.
         idle = imin_loc(get_more,mask=rank_mask)
         get_more(idle) = get_more(idle) + 1
         bks_distrb(band,ik_ibz,spin) = proc_ranks(idle)

       else 
         call wfd_dump_errinfo(Wfd)
         write(msg,'(a,3(i0,1x))')" Nobody has (band, ik_ibz, spin): ",band,ik_ibz,spin
         MSG_ERROR(msg)
       end if

     end do
   end do
 end do

 if (PRESENT(got)) got=get_more

end subroutine wfd_bks_distrb
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_sanity_check
!! NAME
!!  wfd_sanity_check
!!
!! FUNCTION
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_sanity_check(Wfd) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,spin,band,mpi_ierr,ierr
 integer :: how_manyb,unt_dbg,irank
 character(len=500) :: msg
!arrays
 integer :: my_band_list(Wfd%mband)
 integer,pointer :: my_bkstab(:,:,:)

!************************************************************************
 
 call wfd_update_bkstab(Wfd)

 my_bkstab => Wfd%bks_tab(:,:,:,Wfd%my_rank)
 ierr=0

 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz
      do band=1,Wfd%nband(ik_ibz,spin)
        if (my_bkstab(band,ik_ibz,spin) == HAS_BKS .and. .not. wfd_ihave_ug(Wfd,band,ik_ibz,spin,"Stored") ) then
          write(msg,'(a,3(i0,1x))')" Found inconsistency in bks_tab for (band, ik_ibz, spin): ",band,ik_ibz,spin
          call wrtout(std_out,msg,"PERS")
          ierr=ierr+1
        end if
     end do
   end do
 end do

 call xsum_mpi(ierr,Wfd%comm,mpi_ierr)

 if (ierr/=0) then
   unt_dbg = get_unit()
   open(unit=unt_dbg,file="WFD_DEBUG")
   do irank=0,Wfd%nproc-1
            
     if (irank==Wfd%my_rank) then
       write(unt_dbg,*)" (k,b,s) stated owned by rank: ",Wfd%my_rank

       do spin=1,Wfd%nsppol
         do ik_ibz=1,Wfd%nkibz
            write(unt_dbg,*)" (spin,ik_ibz) ",spin,ik_ibz
            call wfd_mybands(Wfd,ik_ibz,spin,how_manyb,my_band_list,"Stored") 
            write(unt_dbg,*) (my_band_list(band),band=1,how_manyb)
          end do
       end do

     end if
   end do
   close(unt_dbg)
   call xbarrier_mpi(Wfd%comm)
   call leave_new("COLL")  ! Stop the code.
 end if

end subroutine wfd_sanity_check
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_dump_errinfo
!! NAME
!!  wfd_dump_errinfo
!!
!! FUNCTION
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=
!!
!! OUTPUT
!!
!! PARENTS
!!      m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_dump_errinfo(Wfd,onfile) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_27_toolbox_oop
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 logical,optional,intent(in) :: onfile
 type(wfs_descriptor),intent(in) :: Wfd
!arrays

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,spin,band
 integer :: how_manyb,unt_dbg
 character(len=10) :: strank
 !character(len=500) :: msg
 character(len=fnlen) :: fname_dbg
!arrays
 integer :: my_band_list(Wfd%mband)

!************************************************************************

 unt_dbg=std_out

 if (PRESENT(onfile)) then
   if (onfile) then
     call int2char(Wfd%my_rank,strank)
     fname_dbg = "WFD_DEBUG_RANK"//TRIM(strank); unt_dbg = get_unit()
     open(unit=unt_dbg,file=fname_dbg)
   end if
 end if

 write(unt_dbg,*)" (k,b,s) stated owned by rank: ",Wfd%my_rank
 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz
      write(unt_dbg,*)" ug stored at (spin, ik_ibz) ",spin,ik_ibz
      call wfd_mybands(Wfd,ik_ibz,spin,how_manyb,my_band_list,"Stored") 
      write(unt_dbg,*) (my_band_list(band),band=1,how_manyb)
    end do
 end do

end subroutine wfd_dump_errinfo
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_gamma2k
!! NAME
!! wfd_gamma2k
!!
!! FUNCTION
!!  Converts a block of wavefunctions from the gamma-centered basis set to the k-centered
!!  basis set used in abinit. It works for a single block of states at fixed spin and k-point. 
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=Structure containing the wave functions for the GW.
!!    %ecut=cutoff energy for GW wavefunctions.
!!    %kibz(3,nkibz)=K-points in reduced coordinates
!!    %istwfk(nkibz)=Storage mode for each k-point
!!    %npwwfn=Number of G-vectors in the gamma-centered basis set.
!!    %nspinor=Number of spinorial components.
!!    %gvec(3,npwwfn)=G-vectors in the gamma-centered basis set.
!!  ik_ibz=Index of the required k-point
!!  spin=Required spin index.
!!  ikg=Shift to be used in cg and. Mainly used to fill selected slices of the arrays.
!!  gmet(3,3)=Metric in reciprocal space.
!!
!! OUTPUT
!!  nmiss=Number of missing G-vectors, i.e. G-vectors in the k-centered basis set not contained 
!!        in the gamma-centered basis set used to describe the GW wavefunctions.
!!  See also SIDE EFFECTS
!!
!! SIDE EFFECTS
!!  cg(:,:)= cg(icg:) contains the states with Fourier components defined in the k-centered basis set. 
!!  kg_k(:,:)= contains the G-vectors centered on the k-point.
!!
!! PARENTS
!!      bloch_interp,gw2wfk
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_gamma2k(Wfd,ik_ibz,spin,ikg,kg_k,cg,gmet,nmiss)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_51_manage_mpi
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin,ikg
 integer,intent(out) :: nmiss
 type(wfs_descriptor),intent(in) :: Wfd
!arrays
 integer,pointer :: kg_k(:,:)
 real(dp),intent(in) :: gmet(3,3)
 real(dp),intent(inout)  :: cg(:,:) 

!Local variables-------------------------------
!scalars
 integer :: band,ig,igp,gw_spad,cg_spad,ispinor,icg,igw,istwf_k,cg_bpad,npw_k,mcg
 real(dp) :: ecut
 logical :: found
 type(MPI_type)  :: MPI_enreg_seq
!arrays
 integer :: gcur(3)
 integer,allocatable :: k2g(:)
 integer,pointer :: gvec(:,:)
 real(dp) :: kpt(3)

! *************************************************************************

 !* Fake MPI_type for the sequential part.
 call initmpi_seq(MPI_enreg_seq) 

 MSG_ERROR("Code has to be checked!")
 ABI_CHECK(ikg==0,"ikg/=0 not coded")

 ecut    = Wfd%ecut
 kpt     = Wfd%kibz(:,ik_ibz)
 istwf_k = Wfd%istwfk(ik_ibz)

 gvec    => Wfd%Kdata(ik_ibz)%kg_k

 ! * Calculate set of G"s for this k-point.
 call get_kg(kpt,istwf_k,ecut,gmet,npw_k,kg_k) 

 !ABI_CHECK(SIZE(kg,DIM=2)>=npw_k,"kg too small")

 ! * Make table for the correspondence btw the k-centered 
 !   and the Gamma-centered basis set.
 allocate(k2g(npw_k))
 !
 ! Loop over g-vectors, for this k point.
 ! Search selected vector in the gamma centered G-sphere.
 nmiss=0
 do ig=1,npw_k
   gcur(:)=kg_k(:,ig)
   igp=0 ; found=.FALSE.
   do while ((.not.found) .and. igp<Wfd%npwwfn)
     igp=igp+1; found=ALL(gcur(:)==gvec(:,igp))
   end do
   if (found) then ! Store it if found:
     k2g(ig) = igp
   else
     k2g(ig)=Wfd%npwwfn+1
     nmiss=nmiss+1 
   end if
 end do

 !$mcg = npw_k*Wfd%nspinor*(Wfd%my_maxb-Wfd%my_minb+1)
 ABI_CHECK(SIZE(cg,DIM=2)>=(mcg+ikg),"cg too small")

 do band=1,Wfd%nband(ik_ibz,spin) 
   !$cg_bpad=npw_k*Wfd%nspinor*(band-Wfd%my_minb)
   do ispinor=1,Wfd%nspinor
     cg_spad=(ispinor-1)*npw_k
     gw_spad=(ispinor-1)*Wfd%npwwfn
     do ig=1,npw_k
       icg = ig+cg_spad+cg_bpad+ikg
       igw = k2g(ig)+gw_spad
       if (k2g(ig)<Wfd%npwwfn+1) then
         cg(1,icg) = REAL (Wfd%Wave(band,ik_ibz,spin)%ug(igw))
         cg(2,icg) = AIMAG(Wfd%Wave(band,ik_ibz,spin)%ug(igw))
       else  ! not in the gamma-centered basis set, zeroing the Fourier component.
         cg(:,icg) = zero
       end if
     end do
   end do
 end do

 deallocate(k2g)

end subroutine wfd_gamma2k
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_distribute_bbp
!! NAME
!!  wfd_distribute_bbp
!!
!! FUNCTION
!!  This routines distributes as set of (b,b') indeces taking into account the MPI distribution of the ug.
!!  It is used to calculate matrix elements of the form <b,k,s|O|b',k,s>
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=
!!  ik_ibz=The index of the k-point in the IBZ.
!!  spin=Spin index.
!!  allup=String used to select or not the upper triangle. Possible values:
!!    "All"  =Entire (b,b') matrix will be distributed.
!!    "Upper"=Only the upper triangle is distributed.
!!  [got(%nproc)]=The number of tasks already assigned to the nodes. Used to optimize the work load.
!!  [bbp_mask(%mband,%mband)]= mask used to select a subset of (b,b') indeces.
!!
!! OUTPUT
!!  my_nbbp=The number of (b,b') indeces treated by this node.
!!  bbp_distrb(%mband%mband)=The rank of the node that will treat (b,b').
!!
!! PARENTS
!!      calc_vhxc_me,cchi0q0
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_distribute_bbp(Wfd,ik_ibz,spin,allup,my_nbbp,bbp_distrb,got,bbp_mask) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,spin
 integer,intent(out) :: my_nbbp
 type(wfs_descriptor),intent(in) :: Wfd
 character(len=*),intent(in) :: allup
!arrays
 integer,intent(out) :: bbp_distrb(Wfd%mband,Wfd%mband)
 integer,optional,intent(inout) :: got(Wfd%nproc)
 logical,optional,intent(in) :: bbp_mask(Wfd%mband,Wfd%mband)

!Local variables ------------------------------
!arrays
 integer :: loc_got(Wfd%nproc)

!************************************************************************

 ! Just a wrapper around wfd_distribute_kb_kpbp.
 loc_got=0; if (PRESENT(got)) loc_got = got

 if (PRESENT(bbp_mask)) then
   call wfd_distribute_kb_kpbp(Wfd,ik_ibz,ik_ibz,spin,allup,my_nbbp,bbp_distrb,loc_got,bbp_mask) 
 else
   call wfd_distribute_kb_kpbp(Wfd,ik_ibz,ik_ibz,spin,allup,my_nbbp,bbp_distrb,loc_got) 
 end if

end subroutine wfd_distribute_bbp
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_distribute_kb_kpbp
!! NAME
!!  wfd_distribute_kb_kpbp
!!
!! FUNCTION
!!  This routines distributes as set of (b,b') indeces taking into account the MPI distribution of the ug.
!!  It is used to calculate matrix elements of the form <b,k,s|O|b',k',s>
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=
!!  ik_ibz =The index of the k-point k  in the IBZ.
!!  ikp_ibz=The index of the k-point k' in the IBZ.
!!  spin=Spin index.
!!  allup=String used to select the upper triangle of the (b,b') matrix. Possible values:
!!    "All"  =Entire (b,b') matrix will be distributed.
!!    "Upper"=Only the upper triangle is distributed.
!!  [got(%nproc)]=The number of tasks already assigned to the nodes. Used to optimize the distribution of the tasks.
!!  [bbp_mask(%mband,%mband)]= mask used to select a subset of (b,b') indeces.
!!
!! OUTPUT
!!  my_nbbp=The number of (b,b') indeces treated by this node.
!!  bbp_distrb(%mband%mband)=The rank of the node that will treat (b,b').
!!
!! PARENTS
!!      cchi0,m_wfs
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_distribute_kb_kpbp(Wfd,ik_ibz,ikp_ibz,spin,allup,my_nbbp,bbp_distrb,got,bbp_mask) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ik_ibz,ikp_ibz,spin
 integer,intent(out) :: my_nbbp
 type(wfs_descriptor),intent(in) :: Wfd
 character(len=*),intent(in) :: allup
!arrays
 integer,intent(out) :: bbp_distrb(Wfd%mband,Wfd%mband)
 integer,optional,intent(inout) :: got(Wfd%nproc)
 logical,optional,intent(in) :: bbp_mask(Wfd%mband,Wfd%mband)

!Local variables ------------------------------
!scalars
 integer :: my_nband,ib1,ib2,pcb2,pcb1,howmany_b,howmany_bp
 integer :: rank,ncpus,idle,b1_stop,istat
 character(len=500) :: msg
!arrays
 integer :: rank_bandlist_k(Wfd%mband),rank_bandlist_kp(Wfd%mband) 
 integer :: get_more(Wfd%nproc),my_band_list_k(Wfd%mband)
 integer(i1b),allocatable :: whocan(:,:,:)
 logical :: rank_mask(Wfd%nproc),b_mask(Wfd%mband)

!************************************************************************

 allocate(whocan(Wfd%mband,Wfd%mband,Wfd%nproc), STAT=istat)
 ABI_CHECK(istat==0,"out of memory in whocan")
 whocan=0 !  Will be set to 1 if this node can calculate (b,b').

 do rank=0,Wfd%nproc-1
   call wfd_bands_of_rank(Wfd,rank,ik_ibz ,spin,howmany_b, rank_bandlist_k ) 
   call wfd_bands_of_rank(Wfd,rank,ikp_ibz,spin,howmany_bp,rank_bandlist_kp) 

   if (howmany_b>0.and.howmany_bp>0) then ! This proc has bands both at (k,s) and at (k',s), add it to the list.
     do pcb2=1,howmany_bp
       ib2 = rank_bandlist_kp(pcb2)
       do pcb1=1,howmany_b
         ib1 = rank_bandlist_k(pcb1)
         whocan(ib1,ib2,rank+1) = 1
       end do
     end do
   end if
 end do

 get_more=0; if (PRESENT(got)) get_more=got
 b1_stop=Wfd%nband(ik_ibz,spin)

 bbp_distrb = xmpi_undefined_rank

 do ib2=1,Wfd%nband(ikp_ibz,spin)
   b_mask = .TRUE.; if (PRESENT(bbp_mask)) b_mask = bbp_mask(:,ib2)
   if (ANY(b_mask)) then
     my_nband=0; my_band_list_k=0
     if (starts_with(allup,(/"U","u"/))) b1_stop = MIN(ib2,Wfd%nband(ik_ibz,spin)) ! Only the upper triangle of the (b1,b2) matrix.

     do ib1=1,b1_stop
       if (b_mask(ib1)) then
         ncpus = COUNT(whocan(ib1,ib2,:)==1)

         if (ncpus==1) then ! Only one node have the data required. This section should not interfere with the one below.
           bbp_distrb(ib1,ib2)=imin_loc(ABS(whocan(ib1,ib2,:)-1)) - 1

         else if (ncpus>1) then ! More nodes can calculate this element. Assign it, trying to obtain a good load distribution.
           rank_mask = (whocan(ib1,ib2,:)==1)
           idle = imin_loc(get_more,mask=rank_mask)
           get_more(idle) = get_more(idle) + 1
           bbp_distrb(ib1,ib2)=idle-1

         else 
           call wfd_dump_errinfo(Wfd)
           write(msg,'(a,5(i0,1x))')" Nobody has (band1, ik_ibz) (band2, ikp_ibz) spin: ",ib1,ik_ibz,ib2,ikp_ibz,spin
           MSG_ERROR(msg)
         end if

       end if
     end do ! ib1
   end if
 end do ! ib2

 deallocate(whocan)

 my_nbbp = COUNT(bbp_distrb==Wfd%my_rank)
 if (PRESENT(got)) got=get_more

end subroutine wfd_distribute_kb_kpbp
!!***

!----------------------------------------------------------------------


!!****f* m_wfs/wfd_get_cprj
!! NAME
!!  wfd_get_cprj
!!
!! FUNCTION
!!  Return a copy of Cprj either by calculating it on-the-fly or by just retrieving the data already stored in the data type.
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=the wavefunction descriptor.
!!  band=Band index.
!!  ik_ibz=Index of the k-point in the IBZ.
!!  spin=Spin index
!!  sorted=.TRUE. if the output cprj matrix elements have to be sorted by atom type.
!!
!! OUTPUT
!!  Cprj_out(Wfd%natom,Wfd%nspinor) <type(cprj_type)>=Unsorted matrix elements.
!!
!! PARENTS
!!      calc_density,calc_exch,calc_sigc_me,calc_sigx_me,calc_vhxc_me,cchi0
!!      cchi0q0,cexch_haydock,cohsex_me,debug_tools,exch,m_wfs,sigma,spectra
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_get_cprj(Wfd,band,ik_ibz,spin,Cryst,Cprj_out,sorted)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_abiutil
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_ibz,spin
 logical,intent(in) :: sorted
 type(wfs_descriptor),intent(inout) :: Wfd
 type(crystal_structure),intent(in) :: Cryst
!arrays
 type(cprj_type),intent(out) :: Cprj_out(Wfd%natom,Wfd%nspinor)

!Local variables ------------------------------
!scalars
 integer,parameter :: choice1=1,idir0=0
 integer :: want_order,iatom,sidx
 character(len=500) :: msg
!arrays 

!************************************************************************

 want_order=CPR_RANDOM; if (sorted) want_order=CPR_SORTED

 SELECT CASE (Wfd%Wave(band,ik_ibz,spin)%has_cprj) 

 CASE (WFD_NOWAVE, WFD_ALLOCATED) 

#ifdef DEBUG_MODE 
   ! TODO: here there is a problem somewhere in MSG_PERS_BUG
   if (.not.wfd_ihave_ug(Wfd,band,ik_ibz,spin,"Stored")) then
     write(msg,'(a,3(i0,1x),a)')" ug for (band, ik_ibz, spin): ",band,ik_ibz,spin," is not stored in memory!"
     MSG_PERS_ERROR(msg)
   end if
#endif

   call wfd_ug2cprj(Wfd,band,ik_ibz,spin,choice1,idir0,Wfd%natom,Cryst,Cprj_out,sorted=sorted)

   if (Wfd%Wave(band,ik_ibz,spin)%has_cprj==WFD_ALLOCATED .and. want_order == Wfd%Wave(band,ik_ibz,spin)%cprj_order) then
     call cprj_copy(Cprj_out,Wfd%Wave(band,ik_ibz,spin)%Cprj) 
     Wfd%Wave(band,ik_ibz,spin)%has_cprj=WFD_STORED
   else 
     !TODO Have to reorder.
     MSG_ERROR("Not implemented error")
   end if

 CASE (WFD_STORED) ! copy it back.
   
   if (want_order == Wfd%Wave(band,ik_ibz,spin)%cprj_order) then
     call cprj_copy(Wfd%Wave(band,ik_ibz,spin)%Cprj,Cprj_out) 

   else 
     select case (want_order)

     case (CPR_SORTED)
       do iatom=1,Cryst%natom
         sidx = Cryst%atindx(iatom) ! random --> sorted table.
         call cprj_copy(Wfd%Wave(band,ik_ibz,spin)%Cprj(iatom:iatom,:),Cprj_out(sidx:sidx,:))
       end do

     case (CPR_RANDOM)  
       do sidx=1,Cryst%natom
         iatom = Cryst%atindx1(sidx) ! sorted --> random table.
         call cprj_copy(Wfd%Wave(band,ik_ibz,spin)%Cprj(sidx:sidx,:),Cprj_out(iatom:iatom,:))
       end do

     case default
       write(msg,'(a,i0)')" Wrong value for want_order ",want_order
       MSG_PERS_ERROR(msg)
     end select

   end if

 CASE DEFAULT
   write(msg,'(a,i0)')" Wrong has_cprj: ",Wfd%Wave(band,ik_ibz,spin)%has_cprj
   MSG_PERS_BUG(msg)
 END SELECT

end subroutine wfd_get_cprj
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_change_ngfft
!! NAME
!!  wfd_change_ngfft
!!
!! FUNCTION
!!   Reallocate and reinitialize internal tables for performing FFTs of wavefunctions.
!!
!! INPUTS
!!  Cryst<crystal_structure>=Info on unit cell.
!!  Psps<pseudopotential_type>=Pseudopotential info.
!!  new_ngfft(18)=FFT descriptor for the new FFT mesh.
!!
!!  SIDE EFFECTS
!!  Wfd<wfs_descriptor>=Wavefunction descriptor with new internal tables for FFT defined by new_ngfft.
!!
!! PARENTS
!!      bloch_interp,calc_density,calc_exch,calc_sigc_me,calc_sigx_me
!!      calc_vhxc_me,cchi0,cchi0q0,cexch_haydock,cohsex_me,exch,sigma
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_change_ngfft(Wfd,Cryst,Psps,new_ngfft) 

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_53_ffts
 use interfaces_56_recipspace
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: new_ngfft(18)
 type(crystal_structure),intent(in) :: Cryst
 type(pseudopotential_type),intent(in) :: Psps
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays

!Local variables ------------------------------
!scalars
 integer,parameter :: npw0=0
 integer :: npw_k,ik_ibz,istwf_k !,band,spin,
 !character(len=500) :: msg
!arrays
 integer,allocatable :: kg_k(:,:)
 logical,allocatable :: kg_mask(:)

!************************************************************************

 !@wfs_descriptor
 if ( ALL(Wfd%ngfft(1:3) == new_ngfft(1:3)) ) RETURN ! Nothing to do.

 !MSG_COMMENT("Changing FFT mesh")
 !
 ! Change FFT dimensions.
 Wfd%ngfft  = new_ngfft
 Wfd%mgfft  = MAXVAL(new_ngfft(1:3))
 Wfd%nfftot = PRODUCT(new_ngfft(1:3))
 Wfd%nfft   = Wfd%nfftot ! No FFT parallelism.
 !
 ! Recalculate FFT tables. 
 ! TODO: this will be done in kdata_init when k-centered G-spheres will be used.

 if (associated(Wfd%ph1d)) deallocate(Wfd%ph1d)
 allocate(Wfd%ph1d(2,3*(2*Wfd%mgfft+1)*Cryst%natom))
 call getph(Cryst%atindx,Cryst%natom,Wfd%ngfft(1),Wfd%ngfft(2),Wfd%ngfft(3),Wfd%ph1d,Cryst%xred)

 if (associated(Wfd%gbound)) deallocate(Wfd%gbound)
 allocate(Wfd%gbound(2*Wfd%mgfft+8,2))
 call sphereboundary(Wfd%gbound,Wfd%istwfk(1),Wfd%gvec,Wfd%mgfft,Wfd%npwwfn)

 if (associated(Wfd%igfft0)) deallocate(Wfd%igfft0)
 allocate(Wfd%igfft0(Wfd%npwwfn),kg_mask(Wfd%npwwfn))
                                                                      
 call kgindex(Wfd%igfft0,Wfd%gvec,kg_mask,Wfd%MPI_enreg,Wfd%ngfft,Wfd%npwwfn)
                                                                      
 ABI_CHECK(ALL(kg_mask),"FFT para not yet implemented")
 deallocate(kg_mask)
 !END TODO
 !
 ! Reallocate ur buffers with correct dimensions.
 call destroy_wave_3D(Wfd%Wave,"R")

 !do spin=1,Wfd%nsppol
 !  do ik_ibz=1,Wfd%nkibz
 !    do band=1,Wfd%nband(ik_ibz,spin)
 !      keep = keep_ur(band,ik_ibz,spin)
 !      if (wfd_ihave_ug(Wfd,band,ik_ibz,spin,"Stored") .and. keep) then
 !        call init_wave_0D(Wfd%Wave(band,ik_ibz,spin),Wfd%usepaw,npw0,Wfd%nfft,Wfd%nspinor,Wfd%natom,Wfd%nlmn_atm,CPR_RANDOM)
 !        Wfd%keep_ur(band,ik_ibz,spin) = .TRUE.
 !      else
 !        Wfd%keep_ur(band,ik_ibz,spin) = .FALSE.
 !      end if
 !    end do
 !  end do
 !end do
 !
 ! Reinit Kdata_t
 do ik_ibz=1,Wfd%nkibz
   if (wfd_ihave_ug(Wfd,0,ik_ibz,0)) then
     istwf_k = Wfd%istwfk(ik_ibz)
     npw_k   = Wfd%Kdata(ik_ibz)%npw
     allocate(kg_k(3,npw_k)); kg_k = Wfd%Kdata(ik_ibz)%kg_k
     call kdata_free(Wfd%Kdata(ik_ibz))
     call kdata_init(Wfd%Kdata(ik_ibz),Cryst,Psps,Wfd%kibz(:,ik_ibz),istwf_k,new_ngfft,Wfd%MPI_enreg,kg_k=kg_k)
     deallocate(kg_k)
   end if
 end do

end subroutine wfd_change_ngfft
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_iam_master
!! NAME
!!  wfd_iam_master
!!
!! FUNCTION
!!  Returns true if this rank is the master node. spin index can be specified. 
!!
!! INPUTS
!!  Wfd<wfs_descriptor>
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_iam_master(Wfd,spin) result(ans)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: spin
 type(wfs_descriptor),intent(in) :: Wfd
 logical :: ans

!Local variables ------------------------------
!scalars
! character(len=500) :: msg
!arrays

!************************************************************************

 if (.not.PRESENT(spin)) then
   ans = (Wfd%my_rank == Wfd%master)
 else 
   !FIXME
   MSG_WARNING(" spin optional argument not coded, have to introduce MPI communicators")
 end if

end function wfd_iam_master
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_test_ortho
!! NAME
!! wfd_test_ortho
!!
!! FUNCTION
!!  Test the orthonormalization of the wavefunctions stored in Wfd.
!!
!! INPUTS
!!   Wfd<wfs_descriptor>=wavefunction descriptor.
!!  Cryst<crystal_structure>=Object defining the unit cell and its symmetries.
!!  Pawtab(ntypat*usepaw)<type(pawtab_type)>=PAW tabulated starting data.
!!
!! OUTPUT
!!
!! PARENTS
!!      m_io_kss
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_test_ortho(Wfd,Cryst,Pawtab,unit,mode_paral)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_53_abiutil
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in),optional :: unit
 character(len=4),optional,intent(in) :: mode_paral
 type(crystal_structure),intent(in) :: Cryst
 type(wfs_descriptor),intent(inout) :: Wfd
!array
 type(Pawtab_type),intent(in) :: Pawtab(Cryst%ntypat*Wfd%usepaw)

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,spin,band,band1,band2,ib,ib1,ib2,ierr,how_manyb,my_unt
 real(dp) :: glob_cinf,my_cinf,glob_csup,my_csup,glob_einf,my_einf,glob_esup,my_esup
 complex(dpc) :: cdum
 logical :: bands_are_spread
 character(len=4) :: my_mode
 character(len=500) :: msg
!arrays
 integer :: my_bandlist(Wfd%mband)
 real(dp) :: pawovlp(2)
 complex(gwpc),pointer :: ug1(:),ug2(:)
 character(len=6) :: tag_spin(2)
 type(Cprj_type),allocatable :: Cp1(:,:),Cp2(:,:)

!************************************************************************

 tag_spin(:)=(/'      ','      '/); if (Wfd%nsppol==2) tag_spin(:)=(/' UP   ',' DOWN '/) 

 my_unt   =std_out; if (PRESENT(unit      )) my_unt   =unit
 my_mode  ='COLL' ; if (PRESENT(mode_paral)) my_mode  =mode_paral
 !
 ! Update the kbs table storing the distribution of the ug.
 !call wfd_update_bkstab(Wfd) 

 if (Wfd%usepaw==1) then
   allocate(Cp1(Wfd%natom,Wfd%nspinor)); call cprj_alloc(Cp1,0,Wfd%nlmn_atm)
   allocate(Cp2(Wfd%natom,Wfd%nspinor)); call cprj_alloc(Cp2,0,Wfd%nlmn_atm)
 end if

 bands_are_spread = .FALSE.

 do spin=1,Wfd%nsppol
   my_einf=greatest_real; my_esup=zero
   my_cinf=greatest_real; my_csup=zero
   !
   do ik_ibz=1,Wfd%nkibz
     !
     ! Select my band indeces.
     call wfd_mybands(Wfd,ik_ibz,spin,how_manyb,my_bandlist,"Stored") 
     if (how_manyb/=Wfd%nband(ik_ibz,spin)) bands_are_spread = .TRUE.
     !
     ! 1) Normalization.
     do ib=1,how_manyb
       band = my_bandlist(ib)
       ug1 => Wfd%Wave(band,ik_ibz,spin)%ug
       cdum = xdotc(Wfd%npwwfn*Wfd%nspinor,ug1,1,ug1,1)
       if (Wfd%usepaw==1) then 
         call wfd_get_cprj(Wfd,band,ik_ibz,spin,Cryst,Cp1,sorted=.FALSE.)
         pawovlp = paw_overlap(Cp1,Cp1,Cryst%typat,Pawtab)
         cdum = cdum + CMPLX(pawovlp(1),pawovlp(2))
       end if
       if (REAL(cdum)<my_einf) my_einf=REAL(cdum)
       if (REAL(cdum)>my_esup) my_esup=REAL(cdum)
     end do

     call xmin_mpi(my_einf,glob_einf,Wfd%comm,ierr) ! TODO should use the communicator for this spin
     call xmax_mpi(my_esup,glob_esup,Wfd%comm,ierr)
     !
     ! 2) Orthogonality of wavefunctions.
     do ib1=1,how_manyb 
       band1 = my_bandlist(ib1)
       ug1 => Wfd%Wave(band1,ik_ibz,spin)%ug
       if (Wfd%usepaw==1) call wfd_get_cprj(Wfd,band1,ik_ibz,spin,Cryst,Cp1,sorted=.FALSE.)

       do ib2=ib1+1,how_manyb
         band2 = my_bandlist(ib2)
         ug2 => Wfd%Wave(band2,ik_ibz,spin)%ug
         if (Wfd%usepaw==1) call wfd_get_cprj(Wfd,band2,ik_ibz,spin,Cryst,Cp2,sorted=.FALSE.)
         cdum = xdotc(Wfd%npwwfn*Wfd%nspinor,ug1,1,ug2,1)
         if (Wfd%usepaw==1) then 
           pawovlp = paw_overlap(Cp1,Cp2,Cryst%typat,Pawtab) 
           cdum = cdum + CMPLX(pawovlp(1),pawovlp(2))
         end if
         if (ABS(cdum)<my_cinf) my_cinf=ABS(cdum)
         if (ABS(cdum)>my_csup) my_csup=ABS(cdum)

       end do !ib2
     end do !ib

     call xmin_mpi(my_cinf,glob_cinf,Wfd%comm,ierr) ! TODO should use the communicator for this spin
     call xmax_mpi(my_csup,glob_csup,Wfd%comm,ierr)
   end do ! ik_ibz
   !  
   ! === Output results for this spin ===
   write(msg,'(2a)')ch10,' test on the normalization of the wavefunctions'
   if (Wfd%nsppol==2) write(msg,'(3a)')ch10,' test on the normalization of the wavefunctions with spin ',tag_spin(spin)
   call wrtout(my_unt,msg,mode_paral) 
   write(msg,'(a,f9.6,a,a,f9.6)')&
&    ' min sum_G |a(n,k,G)| = ',glob_einf,ch10,&
&    ' max sum_G |a(n,k,G)| = ',glob_esup
   call wrtout(my_unt,msg,mode_paral) 

   write(msg,'(a)')' test on the orthogonalization of the wavefunctions'
   if (Wfd%nsppol==2) write(msg,'(2a)')' test on the orthogonalization of the wavefunctions with spin ',tag_spin(spin)
   call wrtout(my_unt,msg,mode_paral) 
   write(msg,'(a,f9.6,a,a,f9.6,a)')&
&    ' min sum_G a(n,k,G)* a(n",k,G) = ',glob_cinf,ch10,&
&    ' max sum_G a(n,k,G)* a(n",k,G) = ',glob_csup,ch10
   call wrtout(my_unt,msg,mode_paral) 

 end do ! spin

 if (bands_are_spread) then
   write(msg,'(6a)')&
&    ' rdkss : COMMENT -',ch10,&
&    '  Note that the test on the orthogonalization is not complete ',ch10,&
&    '  since bands are spread among different processors',ch10
   call wrtout(my_unt,msg,mode_paral) 
 end if

 if (Wfd%usepaw==1) then
   call cprj_free(Cp1); deallocate(Cp1)
   call cprj_free(Cp2); deallocate(Cp2)
 end if

end subroutine wfd_test_ortho
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_barrier
!! NAME
!!  wfd_barrier
!!
!! FUNCTION
!!
!! INPUTS
!!  Wfd<wfs_descriptor>
!!
!! PARENTS
!!      bloch_interp,calc_sigc_me,cchi0,cchi0q0
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_barrier(Wfd,spin)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,optional,intent(in) :: spin
 type(wfs_descriptor),intent(in) :: Wfd

!Local variables ------------------------------

!************************************************************************

 if (.not.PRESENT(spin)) then
   call xbarrier_mpi(Wfd%comm)
 else 
   !FIXME
   MSG_WARNING(" spin optional argument not coded, have to introduce MPI communicators")
 end if

end subroutine wfd_barrier
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_sym_ur
!! NAME
!!  wfd_sym_ur
!!
!! FUNCTION
!!  Symmetrize a wave function in real space
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=the wavefunction descriptor.
!!  Cryst<crystal_structure>=Structure describing the crystal structure and its symmetries.
!!  Kmesh<bz_mesh_type>=Structure describing the BZ sampling
!!  band=Band index.
!!  ik_bz=Index of the k-point in the BZ.
!!  spin=Spin index
!!
!! OUTPUT
!!  ur_kbz(Wfd%nfft*Wfd%nspinor)=The symmetrized wavefunction in real space.
!!
!! PARENTS
!!      bloch_interp,debug_tools
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_sym_ur(Wfd,Cryst,Kmesh,band,ik_bz,spin,ur_kbz)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: band,ik_bz,spin
 type(crystal_structure),intent(in) :: Cryst
 type(bz_mesh_type),intent(in) :: Kmesh
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays
 complex(gwpc),intent(out) :: ur_kbz(Wfd%nfft*Wfd%nspinor)

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,isym_k,itim_k,nr !,has_this_ur
 integer :: ispinor,spad,ir,ir2
 complex(dpc) :: ph_mkt,u2b,u2a
 logical :: isirred,iscompatibleFFT
 character(len=500) :: msg
!arrays 
 integer,save :: ngfft_save(18)=-1
 integer :: umklp(3)
 integer,pointer :: tabr_k(:)
 real(dp) :: kbz(3)
 real(dp),pointer :: spinrot_k(:)
 complex(dpc) :: spinrot_mat(2,2) 
 complex(dpc) :: emig0r(Wfd%nfft)
 complex(gwpc) :: ur_kibz(Wfd%nfft*Wfd%nspinor)

!************************************************************************

 ! u(r,b,kbz)=e^{-2i\pi kibz.(R^{-1}t} u (R{^-1}(r-t),b,kibz) 
 !           =e^{+2i\pi kibz.(R^{-1}t} u*({R^-1}(r-t),b,kibz) for time-reversal
 !
 ! Calculate the FFT index of $ R^{-1} (r-\tau) $ used to symmetrize u_Rk if not yet done or if the FFT mesh has changed.
 if (.not.associated(Wfd%irottb) .or. ANY(Wfd%ngfft /= ngfft_save) ) then
   ngfft_save = Wfd%ngfft 
   if (associated(Wfd%irottb)) deallocate(Wfd%irottb)
   allocate(Wfd%irottb(Wfd%nfftot,Cryst%nsym))
   call rotate_FFT_mesh(Cryst%nsym,Cryst%symrel,Cryst%tnons,Wfd%ngfft,Wfd%irottb,iscompatibleFFT)
   if (.not.iscompatibleFFT) then
     msg = "FFT mesh is not compatible with symmetries. Wavefunction symmetrization might be affected by large errors!"
     MSG_WARNING(msg)
   end if
 end if
 !
 ! * Get ik_ibz, non-symmorphic phase, ph_mkt, and symmetries from ik_bz.
 call get_BZ_item(Kmesh,ik_bz,kbz,ik_ibz,isym_k,itim_k,ph_mkt,umklp,isirred)
 
 if (isirred) then ! Avoid symmetrization if this point is irreducible.
   call wfd_get_ur(Wfd,band,ik_ibz,spin,ur_kbz); RETURN
 end if
 !
 ! Reconstruct ur in BZ from the corresponding wavefunction in IBZ.
 call wfd_get_ur(Wfd,band,ik_ibz,spin,ur_kibz)

 !call paw_symcprj(ik_bz,nspinor,1,Cryst,Kmesh,Psps,Pawtab,Pawang,Cprj_bz) 

 if (ANY(umklp/=0)) emig0r = ceigr(-umklp,Wfd%nfft,Wfd%ngfft) 

 tabr_k  => Wfd%irottb(:,isym_k)  ! Table for rotated FFT points

 if (Wfd%nspinor==1) then
   ur_kbz = ur_kibz(tabr_k)*ph_mkt
   if (itim_k==2) ur_kbz = CONJG(ur_kbz)
   if (ANY(umklp /=0)) ur_kbz = ur_kbz*emig0r
 else 

   MSG_ERROR("Implementation has to be tested")

   nr = Wfd%nfft
   spinrot_k => Cryst%spinrot(:,isym_k)
   !
   ! ==== Apply Time-reversal if required ====
   ! \psi_{-k}^1 =  (\psi_k^2)^*
   ! \psi_{-k}^2 = -(\psi_k^1)^*
   if (itim_k==1) then 
     ur_kbz = ur_kibz 
   else if (itim_k==2) then
     ur_kbz(1:nr)     = CONJG(ur_kibz(nr+1:2*nr))
     ur_kbz(nr+1:2*nr)=-CONJG(ur_kibz(1:nr))
   else 
     MSG_ERROR('Wrong i2 in spinor')
   end if
   !
   ! Rotate wavefunctions in real space.
   do ispinor=1,Wfd%nspinor
     spad=(ispinor-1)*nr
     do ir=1,nr
       ir2=tabr_k(ir)
       ur_kbz(ir+spad) = ur_kbz(ir2+spad)*ph_mkt
     end do 
   end do
   !
   ! Rotation in spinor space.
   spinrot_mat(1,1)= spinrot_k(1) + j_dpc*spinrot_k(4)
   spinrot_mat(1,2)= spinrot_k(3) + j_dpc*spinrot_k(2)
   spinrot_mat(2,1)=-spinrot_k(3) + j_dpc*spinrot_k(2)
   spinrot_mat(2,2)= spinrot_k(1) - j_dpc*spinrot_k(4)
                                                                               
   do ir=1,nr
     u2a=ur_kbz(ir) 
     u2b=ur_kbz(ir+nr) 
     ur_kbz(ir)   =spinrot_mat(1,1)*u2a+spinrot_mat(1,2)*u2b
     ur_kbz(ir+nr)=spinrot_mat(2,1)*u2a+spinrot_mat(2,2)*u2b
   end do

   if (ANY(umklp /=0)) then 
     ur_kbz(1:Wfd%nfft) = ur_kbz(1:Wfd%nfft)  *emig0r
     ur_kbz(Wfd%nfft+1:) = ur_kbz(Wfd%nfft+1:)*emig0r
   end if
 end if

end subroutine wfd_sym_ur
!!***

#if 0

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_realloc_ur
!! NAME
!!  wfd_realloc_ur
!!
!! FUNCTION
!!  Reallocate the ur buffer while changing the keep_ur table.
!!
!! PARENTS
!!
!! CHILDREN
!!      iter_alloc,iter_push,wfd_bands_of_rank,wfd_dump_errinfo
!!
!! SOURCE

subroutine wfd_realloc_ur(Wfd,new_keep_ur)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays 
 logical,intent(in) :: new_keep_ur(Wfd%mband,Wfd%nkibz,Wfd%nsppol)

!Local variables ------------------------------

!************************************************************************

 MSG_ERROR("Not implemented error")

 !if (.not.Wfd%keep_ur) RETURN
 !where (Wfd%Wave(:,:,:)%has_ur == WFD_STORED) 
 !  Wfd%Wave(:,:,:)%has_ur = WFD_ALLOCATED
 !end where

end subroutine wfd_realloc_ur
!!***

!----------------------------------------------------------------------

!!****f* m_wfs/wfd_iterator_bbks
!! NAME
!!  wfd_iterator_bbks
!!
!! FUNCTION
!!  This routines returns an iterator used to loop over the composite index (b,b',k,s)
!!  taking into account the distribution of the ug.
!!
!! INPUTS
!!  Wfd<wfs_descriptor>=
!!  [got(Wfd%nproc)]=The number of tasks already assigned to the nodes.
!!  [bbks_mask(Wfd%mband,Wfd%mband,Wfd%nkibz,Wfd%nsppol)]= mask used to select the (b,b',k,s) indeces.
!!
!! OUTPUT
!!  Iter_bbks<iter3_t>=Iterator over the composite index (b,b',k,s) treated by this node.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

function wfd_iterator_bbks(Wfd,allup,got,bbks_mask) result(Iter_bbks)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(wfs_descriptor),intent(in) :: Wfd
 character(len=*),intent(in) :: allup
!arrays
 integer,optional,intent(inout) :: got(Wfd%nproc)
 logical,optional,intent(in) :: bbks_mask(Wfd%mband,Wfd%mband,Wfd%nkibz,Wfd%nsppol)
 type(iter3_t) :: Iter_bbks

!Local variables ------------------------------
!scalars
 integer :: ik_ibz,spin,my_nband,ib1,ib2,pcb,how_manyb
 integer :: rank,ncpus,idle,rich_rank
 character(len=500) :: msg
!arrays
 integer :: rank_bandlist(Wfd%mband),get_more(Wfd%nproc)
 integer :: my_band_list(Wfd%mband)
 integer :: whocan(Wfd%mband,Wfd%mband,Wfd%nkibz,Wfd%nsppol,Wfd%nproc)
 logical :: rank_mask(Wfd%nproc)

!************************************************************************

 call iter_alloc(Iter_bbks,(/Wfd%mband,Wfd%nkibz,Wfd%nsppol/))

 whocan=0 ! 1 means that this node can calculate (b,b',k,s)

 do rank=0,Wfd%nproc-1

   do spin=1,Wfd%nsppol
     do ik_ibz=1,Wfd%nkibz

       call wfd_bands_of_rank(Wfd,rank,ik_ibz,spin,how_manyb,rank_bandlist,how="Stored") 

       if (how_manyb>0) then 

         do pcb=1,how_manyb
           ib2 = rank_bandlist(pcb)
           whocan(rank_bandlist,ib2,ik_ibz,spin,rank+1) = 1
         end do

         !if (starts_with(allup,(/"U","u"/))) then ! only upper triangle of b1,b2 matrix.
         !  do ib2=1,Wfd%nband(ik_ibz,spin)
         !    do ib1=1,ib2-1
         !      whocan(ib1,ib2,ik_ibz,spin) = 0
         !    end do
         !  end do
         !end if

         !if (PRESENT(bbks_mask)) then
         !  where (.not.bbks_mask(:,:,ik_ibz,spin))
         !    whocan(:,:,ik_ibz,spin) = 0
         !  end where
         !end if
       end if
       !if (PRESENT(bbks_mask)) then
       !call wfd_distribute_bands(Wfd,ik_ibz,spin,my_nband,rank_bandlist,bmask=bbks_mask(:,ik_ibz,spin))
       !else 
       !call wfd_distribute_bands(Wfd,ik_ibz,spin,my_nband,rank_bandlist)
       !end if
       !call iter_push(Iter_bbks,ik_ibz,spin,rank_bandlist(1:my_nband))
     end do
   end do

 end do !rank

 get_more=0; if (PRESENT(got)) get_more=got

 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz

     do ib2=1,Wfd%nband(ik_ibz,spin)

       my_nband=0; my_band_list=0
       do ib1=1,ib2-1
         ncpus = COUNT(whocan(ib1,ib2,ik_ibz,spin,:)/=0)

         if (ncpus==1) then  
           ! Only one node have the data required. This section should not interfere with the one below.
           rich_rank = imin_loc(ABS(whocan(ib1,ib2,ik_ibz,spin,:)-1)) - 1
           if (Wfd%my_rank==rich_rank) then
             my_nband=my_nband+1
             my_band_list(my_nband)=ib1
           end if

         else if (ncpus>1) then 
           ! More nodes might calculate this element. Assign it trying to obtain a good load distribution.
           rank_mask = (whocan(ib1,ib2,ik_ibz,spin,:)==1)
           idle = imin_loc(get_more,mask=rank_mask)
           get_more(idle) = get_more(idle) + 1
           if (Wfd%my_rank==idle-1) then 
             my_nband=my_nband + 1 
             my_band_list(my_nband) = band
           end if

         else 
           call wfd_dump_errinfo(Wfd)
           write(msg,'(a,4(i0,1x))')" Nobody has (band1, band2, ik_ibz, spin): ",ib1,ib2,ik_ibz,spin
           MSG_ERROR(msg)
         end if
       end do ! ib1

       call iter_push(Iter_bbks,ib2,ik_ibz,spin,my_bandlist(1:my_nband))
     end do ! ib2

   end do ! ik_ibz
 end do ! spin

 if (PRESENT(got)) got=get_more

end function wfd_iterator_bbks
!!***
#endif

!----------------------------------------------------------------------

END MODULE m_wfs
!!***
