!{\src2tex{textfont=tt}}
!!****f* ABINIT/mlwfovlp
!! NAME
!! mlwfovlp
!!
!! FUNCTION
!! Routine which computes overlap M_{mn}(k,b) and projection A_{mn}(k)
!! for Wannier code (www.wannier.org f90 version).
!! Various file are written (wannier90.*) which can be used to run a
!! separate wannier calculation with the wannier90 code.
!!
!! COPYRIGHT
!! Copyright (C) 2005-2010 ABINIT group (BAmadon,FJollet,TRangel)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  atindx1(natom)=index table for atoms, inverse of atindx (see scfcv.f)
!!  cg(2,mpw*nspinor*mband*mkmem*nsppol)=planewave coefficients of wavefunctions.
!!  cprj(natom,nspinor*mband*mkmem*nsppol)= <p_lmn|Cnk> coefficients for each WF |Cnk>
!!                                          and each |p_lmn> non-local projector
!!  dtset <type(dataset_type)>=all input variables for this dataset
!!  dtfil <type(datafiles_type)>=variables related to files
!!  ecut=cut-off energy for plane wave basis sphere (Ha)
!!  eigen(mband*nkpt*nsppol)=array for holding eigenvalues (hartree)
!!  gprimd(3,3)=dimensional reciprocal space primitive translations
!!  hdr <type(hdr_type)>=the header of wf, den and pot files
!!  kg(3,mpw*mkmem)=reduced planewave coordinates.
!!  mband=maximum number of bands
!!  mgfft=maximum size of 1D FFTs
!!  mgfftc=maximum size of 1D FFTs (coarse grid)
!!  mkmem =number of k points which can fit in memory; set to 0 if use disk
!!  mpi_enreg=informations about MPI parallelization
!!  mpw=maximum dimensioned size of npw.
!  natom=number of atoms in cell.
!!  nattyp(ntypat)= # atoms of each type.
!!  nfft=(effective) number of FFT grid points (for this processor) (see NOTES at beginning of scfcv)
!!  ngfft(18)=contain all needed information about 3D FFT (see NOTES at beginning of scfcv)
!!  nkpt=number of k points.
!!  npwarr(nkpt)=number of planewaves in basis at this k point
!!  nspinor=number of spinorial components of the wavefunctions
!!  nsppol=1 for unpolarized, 2 for spin-polarized
!!  ntypat=number of types of atoms in unit cell.
!!  prtvol=control print volume and debugging output
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!  rprimd(3,3)=dimensional primitive translations for real space (bohr)
!!  ucvol=unit cell volume (bohr**3)
!!  xred(3,natom)=reduced dimensionless atomic coordinates
!!
!! OUTPUT
!!  (only writing, printing)
!!
!! SIDE EFFECTS
!!  (only writing, printing)
!!
!! NOTES
!!
!! PARENTS
!!      outscfcv
!!
!! CHILDREN
!!      smatrix_pawinit,mlwfovlp_setup,mlwfovlp_pw
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

 subroutine mlwfovlp(atindx1,cg,cprj,dtset,dtfil,eigen,gprimd,hdr,kg,&
& mband,mgfftc,mkmem,mpi_enreg,mpw,natom,&
& nattyp,nfft,ngfft,nkpt,npwarr,nspinor,nsppol,ntypat,&
& pawang,pawrad,pawtab,prtvol,psps,rprimd,ucvol,xred)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use defs_wannier90
 use m_xmpi
 use m_wannier2abinit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_51_manage_mpi
 use interfaces_53_ffts
 use interfaces_66_paw
 use interfaces_67_common, except_this_one => mlwfovlp
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: mband,mgfftc,mkmem,mpw,natom,nfft,nkpt
 integer,intent(in) :: nsppol,ntypat,prtvol
 integer,intent(in) :: nspinor
 real(dp),intent(in) :: ucvol
 type(MPI_type),intent(inout) :: mpi_enreg
 type(dataset_type),intent(in) :: dtset
 type(datafiles_type),intent(in) :: dtfil
 type(hdr_type),intent(in) :: hdr
 type(pawang_type),intent(in) :: pawang
 type(pseudopotential_type),intent(in) :: psps
!arrays
 integer,intent(in) :: atindx1(natom)
 integer :: kg(3,mpw*mkmem),nattyp(ntypat),ngfft(18),npwarr(nkpt)
 real(dp),intent(in) :: cg(2,mpw*nspinor*mband*mkmem*nsppol)
 real(dp),intent(in) :: eigen(mband*nkpt*nsppol),gprimd(3,3),rprimd(3,3)
 real(dp),intent(in) :: xred(3,natom)
 type(cprj_type) :: cprj(natom,nspinor*mband*mkmem*nsppol)
 type(pawrad_type),intent(in) :: pawrad(psps%ntypat*psps%usepaw)
 type(pawtab_type),intent(in) :: pawtab(psps%ntypat*psps%usepaw)

!Local variables-------------------------------
!scalars
 integer :: band_index,cplex,i,iatom,iband,iband1,iband2,icgtemp,ierr
 integer :: ig,ii,ikg
 integer :: ikpt,ikpt1,ikpt2,ilmn,intot,isppol,itypat
 integer :: iun(nsppol),iun_plot,iwan,jband,jband1,jband2,jj,jj1,jj2,jj3
 integer :: lmn_size,lproj,lwanniersetup,mbandw,mwan,mgfft,n1
 integer :: n1tmp,n2,n2tmp,n3,n3tmp,n4,n5,n6,nband_inc,nband_k
 integer :: nntot,npw_k,num_bands,num_nnmax,nwan,spacing
 integer :: tim_fourwf
 integer :: master,nprocs,spaceComm,spin,rank
!integer :: j,k,l
 real(dp) :: corrvdw
 real(dp) :: weight,w90version
 logical :: gamma_only,leig,lmmn,lwannierrun,spinors,have_disentangled,have_wan2abi
 character(len=20) :: wfnname
 character(len=500) :: message
 character(len=fnlen) :: seed_name
 character(len=fnlen) :: fname,filew90_win,filew90_wout,filew90_amn(nsppol),filew90_ramn
 character(len=fnlen) :: filew90_mmn(nsppol),filew90_eig(nsppol)
!arrays
 integer :: g1temp(3),ngkpt(3)
 integer,allocatable :: g1(:,:,:),gbound(:,:),icg(:,:)
 integer,allocatable:: iwav(:,:,:),kg_k(:,:),ndimwin(:),ovikp(:,:)
 integer,allocatable :: proj_l(:),proj_m(:),proj_radial(:)
 real(dp) :: real_lattice(3,3)
 real(dp) :: recip_lattice(3,3),spreadw(3)
 real(dp),allocatable :: cm1(:,:,:,:,:,:),cm2_paw(:,:,:),csix(:,:),cwavef(:,:)
 real(dp),allocatable :: denpot(:,:,:)
 real(dp),allocatable :: eigenvalues_w(:,:,:),fofgout(:,:),fofr(:,:,:,:)
 real(dp),allocatable :: proj_site(:,:),proj_x(:,:),proj_z(:,:),proj_zona(:)
 real(dp),allocatable :: wann_centres(:,:),wann_spreads(:),xcart(:,:)
 complex(dpc),allocatable :: A_paw(:,:,:,:)
 complex(dpc),allocatable :: M_matrix(:,:,:,:,:),U_matrix(:,:,:)
 complex(dpc),allocatable :: U_matrix_opt(:,:,:)
 complex(dpc),pointer :: A_matrix(:,:,:,:)
 logical,allocatable :: band_in(:),lwindow(:,:)
 character(len=3),allocatable :: atom_symbols(:)
 type(WannierData) :: WanData
 logical,allocatable::just_augmentation(:)

!************************************************************************

!TODO
!MAGNETISM

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!1) Initialize variables and allocations
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!Some initialization and checks
!
 lwanniersetup=1 ! 1 is mandatory ( 0 is for debug)
!to use lwanniersetup=0, one would need
!to define which bands to exclude.
 lwannierrun=.true.   ! .false. and .true. are possible
 lmmn=.true.          ! .false. and .true. are possible
 leig=.true.          ! .false. and .true. are possible
!
 gamma_only=.false. !not yet implemented
 spinors=.false. !not yet implemented
 spin=3 !spin polarization, just used in case of nsppol==2
!1=> spin up  !not yet implemented
!2=> spin down !not yet implemented
!3=> both
!
!mpi initialization
!
 call xcomm_init  (MPI_enreg,spaceComm)
 call xmaster_init(MPI_enreg,master   )
 call xme_init    (MPI_enreg,rank     )
 call xproc_init(MPI_enreg,nprocs)
!write(*,'("master ",i3," rank ",i3," nprocs ",i3)') master,rank,nprocs
!
!Generate seed names for wannier90 files, and file names
!
 call mlwfovlp_seedname(dtfil%fnameabo_w90,filew90_win,filew90_wout,filew90_amn,&
& filew90_ramn,filew90_mmn,filew90_eig,nsppol,seed_name)
!
!Check the validity of input variables
!
 write(message, '(a,a,a,a)' ) ch10,&
& '   mlwfovlp:  you should give k-point in the full brillouin zone ',ch10,&
& '   with explicit k-points (or kptopt=3) and istwfk 1'
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')
!

 if(nspinor==2) then
   write(6,*) "WARNING: nspinor==2 is not supported in mlwfovlp"
   stop
 end if
 if(mkmem==0) then
   write(6,*) "WARNING: mkmem==0 is not supported in mlwfovlp"
   stop
 end if
 if(nsppol==2) then
!  if(spin==1) then
!  write(message, '(3a)' ) ch10,&
!  &   '   mlwfovlp:  Calculating matrices for spin-up polarization  ',ch10
!  elseif(spin==2) then
!  write(message, '(3a)' ) ch10,&
!  &   '   mlwfovlp:  Calculating matrices for spin-down polarization  ',ch10
!  end if
!  call wrtout(ab_out,message,'COLL')
!  call wrtout(std_out,  message,'COLL')
   if(spin .ne. 3) then
     write(message, '(3a)' ) ch10,&
&     '   mlwfovlp:  spin variable should be equal to 3  ',ch10
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
     stop
   end if
 end if
!
!get lattice parameters in wannier90 format
!
 do i=1, 3
   real_lattice(:,i)=Bohr_Ang*rprimd(i,:)
   recip_lattice(:,i)=two_pi*gprimd(i,:)/Bohr_Ang
 end do
!
 if(psps%npsp/=psps%ntypat) then
   write(6,*) "prb npsp"
   stop
 end if
!
!In MPI cases:
!Wannier90 is not parallelized so we need to set lwannierrun to .false.
!
 if( nprocs>1) then
   if (lwannierrun) then
     write(message,'(9a)') ch10,&
&     '   WARNING: mlwfovlp: calling to wannier90',ch10,&
&     '   cannot be done in parallel',ch10, &
&     '   We will skip this call.',ch10, &
&     '   Afterwards you can run wannier90 in standalone mode.',ch10
     call wrtout(std_out,message,'COLL')
!    
     lwannierrun=.false.
!    
   end if
 end if
!
!Allocations.
!
 num_nnmax=12 !limit fixed for compact structure in wannier_setup.
 allocate(g1(3,nkpt,num_nnmax),ovikp(nkpt,num_nnmax))
 allocate(atom_symbols(natom))
 allocate(xcart(3,natom))
 allocate(band_in(mband))
 allocate(proj_site(3,mband),proj_l(mband),proj_m(mband),proj_radial(mband))
 allocate(proj_x(3,mband),proj_z(3,mband),proj_zona(mband))
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!2) Call to  Wannier setup
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
 nullify(A_matrix)
 call mlwfovlp_setup(atom_symbols,band_in,dtset,gamma_only,&
& g1,lwanniersetup,mband,mbandw,natom,nband_inc,nkpt,&
& nntot,num_bands,num_nnmax,nwan,ovikp,&
& proj_l,proj_m,proj_radial,proj_site,proj_x,proj_z,proj_zona,&
& real_lattice,recip_lattice,rprimd,seed_name,spinors,xcart,xred)
!
 write(message, '(6a)' ) ch10,&
& '   mlwfovlp :  mlwfovlp_setup done -',ch10,&
& '   see ',trim(filew90_wout),' for details.'
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')
!
!some allocations after wannier90 setup
!
 allocate(eigenvalues_w(num_bands,nkpt,nsppol))
 allocate(M_matrix(num_bands,num_bands,nntot,nkpt,nsppol))
 allocate(A_matrix(num_bands,nwan,nkpt,nsppol))
 allocate(iwav(mband,nkpt,nsppol)) !indices for cg coefficients

 if(dtset%vdw_xc==10)then 
   allocate(csix(nwan,nwan))
 end if
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!3) Write Eigenvalues (file seed_name.eig)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
 if(leig) then
!  
!  Assign file unit numbers
   if(rank==master) then
     iun(1)=444
     if(nsppol==2) iun(2)=455
     do isppol=1,nsppol
       open(unit=iun(isppol),file=filew90_eig(isppol),form='formatted',status='unknown')
     end do
   end if !rank==master
!  Loop to write eigenvalues
   band_index=zero
   do isppol=1,nsppol
     do ikpt=1,nkpt
       nband_k=dtset%nband(ikpt+(isppol-1)*nkpt)
       if( isppol .ne. spin .and. spin .ne. 3) then
         band_index=band_index+nband_k
         cycle
       end if
       jband=0
       do iband=1,mbandw
         if(band_in(iband)) then
           jband=jband+1
!          Writing data 
           if(rank==master) write(iun(isppol), '(2i6,4x,f10.5)' ) jband,ikpt,Ha_eV*eigen(iband+band_index)
!          Finish writing, now save eigenvalues
           eigenvalues_w(jband,ikpt,isppol)=Ha_eV*eigen(iband+band_index)
         end if
       end do !iband
       band_index=band_index+nband_k
     end do !ikpt
   end do  !nsppol
   if(rank==master) then
     do isppol=1,nsppol
       close(iun(isppol))
     end do
     write(message, '(a,a)' ) ch10,&
&     '   mlwfovlp :  eigenvalues written'
     call wrtout(std_out,  message,'COLL')
   end if !master
 end if !leig
!else if( leig . and. lwannierun) then
!read .eig file

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!4) Calculate overlaps (file seed_name.mmn)
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!First calculate indices and shift
!
!write(6,*) "Computes shift for cg"
 write(message, '(a,a)' ) ch10,&
& '   mlwfovlp : compute shifts for g-points '
 call wrtout(std_out,  message,'COLL')
!----------------------------------------------------------------------
!Compute shifts for g points (icg,iwav)
!(here mbandw is not used, because shifts are internal
!variables of abinit)
!----------------------------------------------------------------------
!write(6,*) mpw*nspinor*mband*mkmem*nsppol
 allocate(icg(nsppol,nkpt))
 icg=0
 icgtemp=0
 iwav(:,:,:)=0
 do isppol=1,nsppol
   do ikpt=1,nkpt
!    
!    MPI:cycle over k-points not treated by this node
!    

     if (mpi_enreg%paral_compil_kpt==1 ) then
       if (nprocs>1 ) then !sometimes we can have just one processor
         if ( ABS(MPI_enreg%proc_distrb(ikpt,1,isppol)-rank)  /=0) CYCLE
       end if 
     end if
!    
!    write(*,*)'rank',rank,'ikpt',ikpt,'isppol',isppol
     nband_k=dtset%nband(ikpt+(isppol-1)*nkpt)
!    write(6,*) ikpt+(isppol-1)*nkpt,nkpt
     npw_k=npwarr(ikpt)
     do iband=1,nband_k
       if(iband.gt.mband) then
         write(6,*)"error mband",iband,mband,nband_k
         stop
       end if
       iwav(iband,ikpt,isppol)= &
&       (iband-1)*npw_k*nspinor+icgtemp 
!      write(*,*) "iwav", 'me',MPI_enreg%me,'isppol,ikpt,iband,iwav',isppol,ikpt,iband,iwav(iband,ikpt,isppol)
     end do ! iband
     icgtemp=icgtemp+ npw_k*nspinor*nband_k
!    icg(isppol,ikpt)=icgtemp
!    write(6,*) "icg", isppol,ikpt,icg(isppol,ikpt)
   end do  ! ikpt
 end do   ! isppol
!write(6,*) "shift for cg computed"
 deallocate(icg)
!
!Shifts computed.
!
 if( lmmn) then
!  
!  In case of parallelization write out cg for all k-points
!  
   if (nprocs > 1) then
!    
     if(prtvol>0) then
       write(message, '(3a)' ) ch10,&
&       '   mlwfovlp :  Creating temporary files with cg and cprj (PAW)',ch10
       call wrtout(ab_out,message,'COLL')
       call wrtout(std_out,  message,'COLL')
     end if
!    
     do isppol=1,nsppol
       do ikpt=1,nkpt
!        
!        MPI:cycle over k-points not treated by this node
!        
         if (mpi_enreg%paral_compil_kpt==1 ) then
           if (nprocs>1 ) then !sometimes we can have just one processor
             if ( ABS(MPI_enreg%proc_distrb(ikpt,1,isppol)-rank)  /=0) CYCLE
           end if
         end if
!        write(*,*)'writing kpt ',ikpt,'isppol',isppol,' by node ', rank
         write(wfnname,'(a,I5.5,".",I1)') trim(dtfil%fnametmp_cg),ikpt,isppol
         iun_plot=1000+ikpt+ikpt*(isppol-1)

         open (unit=iun_plot, file=wfnname,form='unformatted')
         npw_k=npwarr(ikpt)
         do iband=1,mbandw
           do ig=1,npw_k*nspinor
             write(iun_plot) (cg(i,ig+iwav(iband,ikpt,isppol)),i=1,2)
           end do
         end do
         close(iun_plot)
       end do !ikpt
     end do !isppol
!    
!    In the PAW case we also need to write out cprj into files
!    
     if(psps%usepaw==1) then
!      
!      big loop on atoms, kpts, bands and lmn
!      
       ikpt2=0
       do isppol=1,nsppol
         do ikpt=1,nkpt
!          
!          MPI:cycle over k-points not treated by this node
!          
           if (mpi_enreg%paral_compil_kpt==1 ) then
             if (nprocs>1 ) then !sometimes we can have just one processor
               if ( ABS(MPI_enreg%proc_distrb(ikpt,1,isppol)-MPI_enreg%me)  /=0) CYCLE
             end if
           end if
           ikpt2=ikpt2+1 !sums just on the k-points treated by this node
!          
           write(wfnname,'(a,I5.5,".",I1)') trim(dtfil%fnametmp_cprj),ikpt,isppol
           iun_plot=1000+ikpt
           open (unit=iun_plot, file=wfnname,form='unformatted')
!          
           do iband=1,mband*nspinor
             ig=iband+(ikpt2-1)*mband*nspinor +(isppol-1)*nkpt*mband*nspinor !index for cprj(:,ig)
!            
             do iatom=1,natom
               itypat=dtset%typat(iatom)
               lmn_size=pawtab(itypat)%lmn_size
!              
               do ilmn=1,lmn_size
                 write(iun_plot) (( cprj(iatom,ig)%cp(i,ilmn)),i=1,2)
               end do !ilmn
             end do !iatom
           end do !iband

           close(iun_plot)
         end do !ikpt
       end do !isppol
     end if !usepaw==1

!    
!    
   end if !MPI nprocs>1
!  
!  End of MPI preliminarities
!  Calculate PW contribution of overlaps
!  
   allocate(cm1(2,mbandw,mbandw,nntot,nkpt,nsppol))
   call mlwfovlp_pw(cg,cm1,g1,iwav,kg,mband,mband,&
&   mkmem,mpi_enreg,mpw,nfft,ngfft,nkpt,nntot,&
&   npwarr,nspinor,nsppol,ovikp,dtfil%fnametmp_cg)
   write(message, '(a,a)' ) ch10,&
&   '   mlwfovlp : PW part of overlap computed   '
   call wrtout(std_out,  message,'COLL')
!  
!  compute PAW Contribution and add it to PW contribution
!  
   if(psps%usepaw==1) then
     write(message, '(a,a)' ) ch10,&
&     '** smatrix_pawinit : PAW part of overlap  '
     call wrtout(std_out,  message,'COLL')
     allocate(cm2_paw(2,mbandw,mbandw))
     do isppol=1,nsppol
       do ikpt1=1,nkpt
!        
!        MPI:cycle over k-points not treated by this node
!        
         if (mpi_enreg%paral_compil_kpt==1 ) then
           if (nprocs>1 ) then !sometimes we can have just one processor
             if ( ABS(MPI_enreg%proc_distrb(ikpt1,1,isppol)-rank)  /=0) CYCLE
           end if
         end if
         write(message, '(a,i6,a,2i6)' ) &
&         '   processor',rank,' computes PAW part for kpt and spin',ikpt1,isppol
         call wrtout(std_out,  message,'COLL')

         do intot=1,nntot
           ikpt2= ovikp(ikpt1,intot)
           g1temp(:)=g1(:,ikpt1,intot)
           call smatrix_pawinit(atindx1,cm2_paw,cprj,ikpt1,ikpt2,isppol,&
&           g1temp,gprimd,dtset%kpt,mband,mbandw,mkmem,mpi_enreg,&
&           natom,dtset%nband,nkpt,nspinor,nsppol,dtset%ntypat,pawang,pawrad,pawtab,rprimd,&
&           dtfil%fnametmp_cprj,dtset%typat,xred)
           cm1(:,:,:,intot,ikpt1,isppol)=cm1(:,:,:,intot,ikpt1,isppol)+four_pi*cm2_paw(:,:,:)
!          cm1(:,:,:,intot,ikpt1,isppol)=four_pi*cm2_paw(:,:,:)
         end do ! intot
       end do ! ikpt1
     end do ! isppol
     deallocate(cm2_paw)
     write(message, '(a,a)' ) ch10,&
&     '   mlwfovlp : PAW part of overlap computed '
     call wrtout(std_out,  message,'COLL')
   end if ! usepaw
!  
   call leave_test(mpi_enreg)
   call xsum_mpi(cm1,spaceComm,ierr)
!  
!  write overlap for separate calculation of wannier functions
!  
   if(rank==master) then
     do isppol=1,nsppol !we write separate output files for each isppol
       iun(isppol)=220+isppol
       open(unit=iun(isppol),file=filew90_mmn(isppol),form='formatted',status='unknown')
       write(iun(isppol),*) "nnkp version 90"
       write(iun(isppol),*) num_bands,nkpt,nntot
     end do
   end if ! rank==master

   do isppol=1,nsppol 
     do ikpt1=1,nkpt
       do intot=1,nntot
         if( rank==master) write(iun(isppol),'(2i6,3x,3x,3i5)') ikpt1,ovikp(ikpt1,intot),(g1(jj,ikpt1,intot),jj=1,3)
         jband2=0
         do iband2=1,mbandw ! the first index is faster
           if(band_in(iband2)) then
             jband2=jband2+1
             jband1=0
             do iband1=1,mbandw
               if(band_in(iband1)) then
                 jband1=jband1+1
                 if(rank==master) write(iun(isppol),*) &
&                 cm1(1,iband1,iband2,intot,ikpt1,isppol),cm1(2,iband1,iband2,intot,ikpt1,isppol)
                 M_matrix(jband1,jband2,intot,ikpt1,isppol)=&
&                 cmplx(cm1(1,iband1,iband2,intot,ikpt1,isppol),cm1(2,iband1,iband2,intot,ikpt1,isppol))
!                write(2211,*) ikpt1,intot,iband1,iband2
!                write(2211,*) cm1(1,iband1,iband2,intot,ikpt1,isppol),cm1(2,iband1,iband2,intot,ikpt1,isppol)
               end if ! band_in(iband1)
             end do ! iband1
           end if ! band_in(iband2)
         end do ! iband2
       end do !intot
     end do !ikpt
     if( rank==master ) then
       close(iun(isppol))
       write(message, '(3a)' )  '   ',trim(filew90_mmn(isppol)),' written'
       call wrtout(std_out,  message,'COLL')
     end if !rank==master
   end do !isppol
!  
   deallocate(cm1)
!  
!  Write down part of the matrix to the output file
!  This is for the automatic tests
!  
   if(rank==master) then
     write(message, '(4a)' ) ch10,&
&     '   Writing top of the overlap matrix: M_mn(ikb,ik)',ch10,&
&     '   m=n=1:3, ikb=1, ik=1'
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
!    
!    just write down the first 3 elements
!    
     do isppol=1,nsppol
       write(message, '( " " )')
       if (nsppol>1 ) then
         if (isppol==1) write(message,'(2a)')trim(message),'   spin up:'
         if (isppol==2) write(message,'(2a)')trim(message),'   spin down:'
       end if
       do ii=1,3
         if(ii>num_bands) cycle
         write(message,'(3a)') trim(message),ch10,'    ( '
         do jj=1,3
           if(jj>num_bands)cycle
           write(message, '(a,2f11.6,a)') trim(message),&
&           M_matrix(ii,jj,1,1,1),' , '
         end do
         write(message,'(2a)') trim(message),'    ) '
       end do
       call wrtout(ab_out,message,'COLL')
       call wrtout(std_out,  message,'COLL')
     end do
!    
!    Now write down bottom of the matrix
!    
     write(message, '(4a)' ) ch10,&
&     '   Writing bottom of the overlap matrix: M_mn(ikb,ik)',ch10,&
&     '   m=n=num_bands-2:num_bands, ikb=nntot, ik=nkpt'
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
!    
     do isppol=1,nsppol
       write(message, '( " " )')
       if (nsppol>1 ) then
         if (isppol==1) write(message,'(2a)')trim(message),'   spin up:'
         if (isppol==2) write(message,'(2a)')trim(message),'   spin down:'
       end if
       do ii=num_bands-2,num_bands
         if(ii<1) cycle
         write(message,'(3a)') trim(message),ch10,'    ( '
         do jj=num_bands-2,num_bands
           if(jj<1)cycle
           write(message, '(a,2f11.6,a)') trim(message),&
&           M_matrix(ii,jj,nntot,nkpt,1),' , '
         end do !jj
         write(message,'(2a)') trim(message),'    ) '
       end do !ii
       call wrtout(ab_out,message,'COLL')
       call wrtout(std_out,  message,'COLL')
     end do !isppol
   end if !rank==master
!  
!  erase temporary files created for parallel runs
!  
   if (nprocs > 1) then
!    
     if(prtvol>0) then
       write(message, '(3a)' ) ch10,&
&       '   mlwfovlp :  Removing temporary files with cg and cprj (PAW)',ch10
       call wrtout(ab_out,message,'COLL')
       call wrtout(std_out,  message,'COLL')
     end if
!    
!    Just master  node will remove the files
!    
     if(rank==master) then
       do isppol=1,nsppol
         do ikpt=1,nkpt
           write(wfnname,'(a,I5.5,".",I1)') trim(dtfil%fnametmp_cg),ikpt,isppol
           call unlink(wfnname)
           if(psps%usepaw==1) then
             write(wfnname,'(a,I5.5,".",I1)') trim(dtfil%fnametmp_cprj),ikpt,isppol
             call unlink(wfnname)
           end if
         end do !ikpt
       end do !isppol
     end if
   end if !MPI nprocs>1
!  
 end if !lmmn
!if ( lmmn== .false. .and. lwannierun ) the
!read .mmn file
!
!Deallocate arrays  no longer used
!
 deallocate(ovikp,g1)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!5) Calculate initial projections
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

 if(dtset%w90iniprj/=0 )  then


!  
!  Set value for lproj (type of projections to be computed)
!  In PAW, options 5 and 6 are not in use.
!  5 means that there will be a contribution from inside the spheres and another from the PW part
!  6 means that we take into account just the inside-spheres contribution
!  2 means that PW part will be calculated
   
!  
   lproj=dtset%w90iniprj
   if(dtset%w90iniprj == 5 ) lproj=2 ! Necessary to calculate PW contribution
!  
   if( psps%usepaw==1 .and. (dtset%w90iniprj==2 .or. dtset%w90iniprj>4)) then
     allocate(just_augmentation(nwan))
     just_augmentation(:)=.false.
     if (dtset%w90iniprj==6) just_augmentation(:)=.true.
     if (dtset%w90iniprj==5) then
       do iwan=1,nwan
!        
!        Trick to skip the planewave contribution for some Wannier functions
!        (Not in production).
!        
         if(proj_radial(iwan) > 4) then
           just_augmentation(iwan)=.true.
           proj_radial(iwan)=proj_radial(iwan)-3
           write(message, '(2a,i4)' ) &
&           '   ','Skiping planewave contribution for Wannier function:',iwan
           call wrtout(std_out,  message,'COLL')
         end if !proj_radial>4
       end do !iwan
     end if !w90iniprj == 5
   end if !paw
!  
!  Call mlwfovlp_proj (plane waves part of projections)
!  
   if (dtset%w90iniprj/=6) then ! option 6 not yet in use
     call mlwfovlp_proj(A_matrix,band_in,cg,cprj,dtset,gprimd,just_augmentation,kg,&
&     lproj,mband,mkmem,mpi_enreg,mpw,natom,&
&     nattyp,nkpt,npwarr,&
&     nspinor,nsppol,ntypat,num_bands,nwan,pawtab,proj_l,proj_m,&
&     proj_radial,proj_site,proj_x,proj_z,proj_zona,psps,seed_name,ucvol)
     write(message, '(a,a,a,a)' ) ch10,&
&     '   mlwfovlp:  mlwfovlp_proj done -',ch10,&
&     '   Projectors computed.'
     call wrtout(std_out,  message,'COLL')
   end if !w90proj/=6
!  
!  Calculate inside-sphere part of projections (PAW)
!  
   if (psps%usepaw ==1 .and. ( dtset%w90iniprj>4)) then
     allocate(A_paw(num_bands,nwan,nkpt,nsppol))
     call mlwfovlp_projpaw(A_paw,band_in,cprj,just_augmentation,mband,mkmem,&
&     natom,dtset%nband,nkpt,&
&     num_bands,nspinor,nsppol,dtset%ntypat,nwan,pawrad,pawtab,&
&     proj_l,proj_m,proj_radial,proj_site,proj_x,proj_z,proj_zona,psps,&
&     rprimd,dtset%typat,xred)
!    
     write(message, '(a,a,a,a)' ) ch10,&
&     '   mlwfovlp:  mlwfovlp_proj_paw done -',ch10,&
&     '   Inside-spheres part of projectors computed.'
     call wrtout(std_out,  message,'COLL')
!    
!    Add in-sphere contribution to A_matrix
!    
!    
!    w90iniprj==5. Plane waves + augmentation contributions
!    
     if(dtset%w90iniprj==5) A_matrix(:,:,:,:)=A_matrix(:,:,:,:)+A_paw(:,:,:,:)
!    
!    w90iniprj==6. Just augmentation contribution
!    
     if(dtset%w90iniprj==6) A_matrix(:,:,:,:)=A_paw(:,:,:,:)
!    
!    deallocations
!    
     deallocate(A_paw,just_augmentation)
   end if !usepaw==1
!  
   call leave_test(mpi_enreg)
   call xsum_mpi(A_matrix,spaceComm,ierr)


!  
!  write      projections  to a file
!  
   if(rank==master) then
     if(dtset%w90iniprj==1) then
       iun(:)=219
       open(unit=iun(1),file=trim(filew90_ramn),form='formatted',status='unknown')
       write(iun(1),*) 'Projections from Abinit : mband,nkpt,nwan. indices: iband1,iwan,ikpt'
       write(iun(1),*) num_bands,nkpt,nwan
     else
       do isppol=1,nsppol 
         iun(isppol)=220+isppol
         open(unit=iun(isppol),file=trim(filew90_amn(isppol)),form='formatted',status='unknown')
         write(iun(isppol),*) 'Projections from Abinit : mband,nkpt,nwan. indices: iband1,iwan,ikpt'
         write(iun(isppol),*) num_bands,nkpt,nwan
       end do
     end if
!    
     do isppol=1,nsppol
       if(dtset%w90iniprj==1) then !when random projections
         if (isppol==2) cycle       !there will be just one output file
       end if
       do ikpt=1,nkpt
         do iwan=1,nwan
           jband=0
           do iband=1,mband
             if(band_in(iband)) then
               jband=jband+1
               write(iun(isppol),'(3i6,13x,3x,2f18.14)')jband,iwan,ikpt,A_matrix(jband,iwan,ikpt,isppol)
             end if !band_in
           end do !iband
         end do !iwan
       end do !ikpt
     end do !isppol
!    
     if(dtset%w90iniprj==1) then
       close(iun(1))
       write(message, '(3a)' ) &
&       '   ',trim(filew90_ramn),' written'
       call wrtout(std_out,  message,'COLL')
     else
       do isppol=1,nsppol
         close(iun(isppol))
         write(message, '(3a)' ) &
&         '   ',trim(filew90_amn(isppol)),' written'
         call wrtout(std_out,  message,'COLL')
       end do
     end if
   end if !rank==master

!  
!  
!  Write down part of the matrix to the output file
!  This is for the automatic tests
!  
   if(rank==master) then
     write(message, '(4a)' ) ch10,&
&     '   Writing top of the initial projections matrix: A_mn(ik)',ch10,&
&     '   m=1:3, n=1:3, ik=1'
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
!    
!    just write down the first 3 elements
!    
     do isppol=1,nsppol
       write(message, '( " " )')
       if (nsppol>1 ) then
         if (isppol==1) write(message,'(2a)')trim(message),'   spin up:'
         if (isppol==2) write(message,'(2a)')trim(message),'   spin down:'
       end if
       do ii=1,3
         if(ii>num_bands) cycle
         write(message,'(3a)') trim(message),ch10,'    ( '
         do jj=1,3
           if(jj>nwan)cycle
           write(message, '(a,2f11.6,a)') trim(message),&
&           A_matrix(ii,jj,1,1),' , '
         end do
         write(message,'(2a)') trim(message),'    ) '
       end do
       call wrtout(ab_out,message,'COLL')
       call wrtout(std_out,  message,'COLL')
     end do
!    
!    Now write down bottom of the matrix
!    
     write(message, '(4a)' ) ch10,&
&     '   Writing bottom of the initial projections matrix: A_mn(ik)',ch10,&
&     '   m=num_bands-2:num_bands, n=nwan-2:nwan, ik=nkpt'
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
!    
     do isppol=1,nsppol
       write(message, '( " " )')
       if (nsppol>1 ) then
         if (isppol==1) write(message,'(2a)')trim(message),'   spin up:'
         if (isppol==2) write(message,'(2a)')trim(message),'   spin down:'
       end if
       do ii=num_bands-2,num_bands
         if(ii<1) cycle
         write(message,'(3a)') trim(message),ch10,'    ( '
         do jj=nwan-2,nwan
           if(jj<1)cycle
           write(message, '(a,2f11.6,a)') trim(message),&
&           A_matrix(ii,jj,nkpt,1),' , '
         end do
         write(message,'(2a)') trim(message),'    ) '
       end do
       call wrtout(ab_out,message,'COLL')
       call wrtout(std_out,  message,'COLL')
     end do !isppol
   end if !rank==master
 end if !dtset%w90iniprj/=0
!
!Deallocations
!
 deallocate(proj_site,proj_l,proj_m,proj_radial)
 deallocate(proj_x,proj_z,proj_zona)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!6) write files for wannier function plot
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
 if( dtset%w90prtunk>0) then
   if(psps%usepaw==1) then
     write(message, '( a,a,a,a,a,a,a,a,a)')ch10,&
&     "   WARNING: The UNK matrices will not contain the correct wavefunctions ",ch10,&
&     "   since we are just writing the plane wave contribution.",ch10,&
&     "   The contribution from inside the spheres is missing. ",ch10,&
&     "   However, these files can be used for plotting purposes",ch10
     call wrtout(std_out,  message,'COLL')
   end if
!  
   spacing = dtset%w90prtunk
   write(message, '( 8a,i3,2a)')ch10,&
&   "   UNK files will be written.",ch10,&
&   "   According to the chosen value of w90prtunk",ch10,&
&   "   the wavefunctions are to be written ",ch10, &
&   "   at every ", spacing," records.",ch10
   call wrtout(std_out,  message,'COLL')
!  
   allocate(kg_k(3,mpw))
   n1=ngfft(1)
   n2=ngfft(2)
   n3=ngfft(3)
   n4=ngfft(4)
   n5=ngfft(5)
   n6=ngfft(6)
   cplex=1
   mgfft=mgfftc ! error
   do isppol=1,nsppol
     ikg=0
     do ikpt=1,nkpt
!      
!      MPI:cycle over k-points not treated by this node
!      
       if (mpi_enreg%paral_compil_kpt==1 ) then
         if (nprocs>1 ) then !sometimes we can have just one processor
           if ( ABS(MPI_enreg%proc_distrb(ikpt,1,isppol)-rank)  /=0) CYCLE
         end if
       end if
!      
       npw_k=npwarr(ikpt)
       kg_k(:,1:npw_k)=kg(:,1+ikg:npw_k+ikg)
       allocate(denpot(cplex*n4,n5,n6),cwavef(2,npw_k),&
&       fofr(2,n4,n5,n6),gbound(2*mgfft+8,2))
       allocate(fofgout(2,npw_k))
       iun_plot=1000+ikpt+ikpt*(isppol-1)
       write(wfnname,'("UNK",I5.5,".",I1)') ikpt, isppol
!      open (unit=iun_plot, file=wfnname,form='formatted')
       open(unit=iun_plot, file=wfnname,form='unformatted')
!      optimizing grid for UNK files
       n1tmp = n1/spacing
       n2tmp = n2/spacing
       n3tmp = n3/spacing
       if( mod(n1,spacing) /= 0) then
         n1tmp = n1tmp + 1
       end if
       if( mod(n2,spacing) /= 0) then
         n2tmp = n2tmp + 1
       end if
       if( mod(n3,spacing) /= 0) then
         n3tmp = n3tmp + 1
       end if
!      write(iun_plot,*) n1tmp,n2tmp,n3tmp,ikpt,nband_inc
       write(iun_plot) n1tmp,n2tmp,n3tmp,ikpt,nband_inc
!      gbound=zero
       call sphereboundary(gbound,dtset%istwfk(ikpt),kg_k,mgfft,npw_k)
       write(6,*) "  writes UNK file for ikpt, spin=",ikpt,isppol
       denpot(:,:,:)=zero
       weight = one
       do iband=1,mbandw
         if(band_in(iband)) then
           do ig=1,npw_k*nspinor
             cwavef(1,ig)=cg(1,ig+iwav(iband,ikpt,isppol))
             cwavef(2,ig)=cg(2,ig+iwav(iband,ikpt,isppol))
           end do
           tim_fourwf=0
           call fourwf(cplex,denpot,cwavef,fofgout,fofr,&
&           gbound,gbound,dtset%istwfk(ikpt),kg_k,kg_k,mgfft,&
&           mpi_enreg,1,ngfft,npw_k,npw_k,n4,n5,n6,0,dtset%paral_kgb,&
&           tim_fourwf,weight,weight)
!          do jj3=1,n3,spacing
!          do jj2=1,n2,spacing
!          do jj1=1,n1,spacing
!          write(iun_plot,*) fofr(1,jj1,jj2,jj3),&
!          & fofr(2,jj1,jj2,jj3)
!          end do !jj1
!          end do !jj2
!          end do !jj3
!          unformatted (must be one record)
           write(iun_plot) (((fofr(1,jj1,jj2,jj3),fofr(2,jj1,jj2,jj3),&
&           jj1=1,n1,spacing),jj2=1,n2,spacing),jj3=1,n3,spacing)
         end if !iband
       end do ! iband
       deallocate(cwavef,fofr,gbound,denpot,fofgout)
       ikg=ikg+npw_k
       close(iun_plot)
     end do  ! ikpt
   end do  ! nsppol
   deallocate(kg_k)
!  
   write(message, '(4a)' )ch10, &
&   '   ','UNK files written',ch10
   call wrtout(std_out,  message,'COLL')
 end if !dtset%w90prtunk
!
 deallocate(iwav)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!7) Call to  Wannier90
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
 if(lwannierrun) then
   write(message, '(8a)' ) ch10,&
&   '** mlwfovlp :   call wannier90 library subroutine wannier_run ',ch10,&
&   '   Calculation is running         ',ch10,&
&   '   see ',trim(filew90_wout),' for details.'
   call wrtout(std_out,  message,'COLL')
   if(lwanniersetup.ne.1) stop
   allocate(U_matrix(nwan,nwan,nkpt))
   allocate(U_matrix_opt(num_bands,nwan,nkpt))
   allocate(lwindow(num_bands,nkpt))
   allocate(wann_centres(3,nwan))
   allocate(wann_spreads(nwan))
!  write(6,*) seed_name
!  write(6,*) ngkpt
   ngkpt(1)=dtset%kptrlatt(1,1)
   ngkpt(2)=dtset%kptrlatt(2,2) ! ajouter test de verif que kptrlatt est bien diagonal
   ngkpt(3)=dtset%kptrlatt(3,3)
!  write(6,*) nkpt
!  write(6,*) rprimd*Bohr_Ang
!  write(6,*) two_pi*gprimd/Bohr_Ang
!  write(6,*) mband
!  write(6,*) "nwan",nwan
!  write(6,*) nntot
!  write(6,*) natom
!  write(6,*) atom_symbols
!  write(6,*) xcart
!  write(6,*) num_bands,num_bands,nntot,nkpt
!  write(6,*) wann_spreads
!  wann_spreads=2
!  do i=1, nkpt
!  do j=1, nntot
!  write(6,*) i,j
!  do k=1, num_bands
!  do l=1, num_bands
!  write(6,*) "m",M_matrix(l,k,j,i,1)
!  enddo
!  enddo
!  enddo
!  enddo



#if defined HAVE_WANNIER90
   call wannier_run(trim(seed_name),ngkpt,nkpt,&            !input
&  real_lattice,recip_lattice,dtset%kpt,num_bands,& !input
&  nwan,nntot,natom,atom_symbols,&                  !input
&  xcart*Bohr_Ang,gamma_only,M_matrix(:,:,:,:,1),A_matrix(:,:,:,1),eigenvalues_w(:,:,1),& !input
&  U_matrix,U_matrix_opt,lwindow_loc=lwindow,wann_centres_loc=wann_centres,&     !output
&  wann_spreads_loc=wann_spreads,spread_loc=spreadw)                            !output

!  CALL SILVESTRELLI'S APPROACH TO EVALUATE vdW INTERACTION ENERGY USING MLWF!!
!  ----------------------------------------------------------------------------------------------
   if (dtset%vdw_xc==10) then ! vdw_xc==10 starts the vdW interaction using MLWFs  
     call evdw_wannier(csix,corrvdw,nwan,dtset%vdw_nwan,dtset%vdw_supercell,rprimd,&
&     wann_centres,wann_spreads)
     deallocate(csix)
   end if                       
   
!  ----------------------------------------------------------------------------------------------

   have_wan2abi=.FALSE.
   if (have_wan2abi) then
!    === Create object to store and handle Wannier result. Then dump on file.
!    * We know if disentanglement has been done by looking at the output values of lwindow
!    Not elegant but it is the only way to avoid the parsing of the wannier input.
!    anyway this approach it is not bulletproof, it might fail if disentanglement and
!    all bands are inside the window!
     mwan=nwan ; w90version=1.1 ; have_disentangled=.not.ALL(lwindow)
     if (have_disentangled) then
       allocate(ndimwin(nkpt))
!      call read_chkunit(seed_name,nkpt,ndimwin,ierr)
!      if (ierr==-1) stop 'internal error, likely all bands are inside the window'
     end if
     if (.not.allocated(ndimwin)) allocate(ndimwin(0))

     call WrapInitWanData(WanData,mband,nkpt,nsppol,nntot,mwan,nwan,have_disentangled,eigen,&
&     band_in,lwindow,wann_centres,wann_spreads,spreadw,U_matrix,U_matrix_opt,Hdr,w90version,'Wannier90 Results')

     call DumpWanData(WanData,dtfil%fnameabo_wan,Dtset%accesswff)
     call DestroyWanData(WanData)
     deallocate(ndimwin)
   end if


#endif

   deallocate(M_matrix,A_matrix,eigenvalues_w,wann_centres,wann_spreads)
   deallocate(U_matrix,U_matrix_opt)
   deallocate(lwindow)
   write(message, '(7a)' ) ch10,&
&   '   mlwfovlp :  mlwfovlp_run completed -',ch10,&
&   '   see ',trim(filew90_wout),' for details.',ch10
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')

 end if !lwannierrun
!
!deallocation
!
 deallocate(band_in)
 deallocate(atom_symbols)
 deallocate(xcart)

!DEBUG
!write(6,*)' mlwfovlp : exit'
!stop
!ENDDEBUG
 contains
!!***
!!****f* mlwfovlp/read_chkunit
!! NAME
!! read_chkunit
!!
!! FUNCTION
!! Function which reads the .chk file produced by Wannier90
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE
 subroutine read_chkunit(seed_name,nkpt,ndimwin,ierr)

 use defs_basis
 use m_io_tools, only : get_unit

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkpt
 character(len=*),intent(in) :: seed_name
 integer,intent(out) :: ierr
!arrays
 integer,intent(out) :: ndimwin(nkpt)

!Local variables-------------------------------
!scalars
 integer :: chk_unit,ios,ikpt
 logical :: have_disentangled

!************************************************************************

   chk_unit=get_unit()
   fname=TRIM(seed_name)//'.chk'
   open(unit=chk_unit,file=fname,form='unformatted',status='old',iostat=ios)

   ierr=0
   read(chk_unit) ! header                                   ! Date and time
   read(chk_unit) ! ((real_lattice(i,j),i=1,3),j=1,3)        ! Real lattice
   read(chk_unit) ! ((recip_lattice(i,j),i=1,3),j=1,3)       ! Reciprocal lattice
   read(chk_unit) ! num_kpts
   read(chk_unit) ! ((kpt_latt(i,nkp),i=1,3),nkp=1,num_kpts) ! K-points
   read(chk_unit) ! nntot                  ! Number of nearest k-point neighbours
   read(chk_unit) ! num_wann               ! Number of wannier functions
   read(chk_unit) ! chkpt1                 ! Position of checkpoint
   read(chk_unit) have_disentangled        ! Whether a disentanglement has been performed
   if (have_disentangled) then
!    read(chk_unit) ! omega_invariant     ! Omega invariant
!    read(chk_unit) ((lwindow(i,nkp),i=1,num_bands),nkp=1,num_kpts)
     read(chk_unit) (ndimwin(ikpt),ikpt=1,nkpt)
!    read(chk_unit) (((u_matrix_opt(i,j,nkp),i=1,num_bands),j=1,num_wann),nkp=1,num_kpts)
   else
!    this is not expected. we should have disentanglement. Report the error.
     ierr=-1
   end if
!  read(chk_unit)  (((u_matrix(i,j,k),i=1,num_wann),j=1,num_wann),k=1,num_kpts)               ! U_matrix
!  read(chk_unit)  ((((m_matrix(i,j,k,l,1),i=1,num_wann),j=1,num_wann),k=1,nntot),l=1,num_kpts) ! M_matrix
!  read(chk_unit)  ((wannier_centres(i,j),i=1,3),j=1,num_wann)
   close(chk_unit)

 end subroutine read_chkunit


end subroutine mlwfovlp
!!***
