!{\src2tex{textfont=tt}}
!!****f* ABINIT/wfd_mkrho
!! NAME
!! wfd_mkrho
!!
!! FUNCTION
!! Calculate the charge density on the fine FFT grid in real space.
!!
!! COPYRIGHT
!! Copyright (C) 1999-2010 ABINIT group (GMR, VO, LR, RWG, MG, RShaltaf)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  ngfftf(18)=array containing all the information for the "fine" FFT.
!!  Cryst<Crystal_structure> Info on the crystalline structure
!!     %nsym=number of symmetry operations.
!!     %ucvol=unit cell volume.
!!  Psps<type(pseudopotential_type)>=variables related to pseudopotentials
!!  nfftf=Total number of points on the fine FFT grid (for this processor)
!!  Kmesh<bz_mesh_type>= Info on the k-sampling:
!!     %nibz=number of irreducible k-points.
!!     %nbz=number of k-points in the full Brillouin zone.
!!     %wt(nibz)=irreducible k-points weights.
!!     %timrev=2 if time-reversal symmetry can be used, 1 otherwise.
!!  Wfd<wfs_descriptor)=datatype gathering info on the wavefunctions.
!!    %npwwfn=Number of plane waves used to describe the wave functions.
!!    %nspinor=number of spinorial components.
!!    %nsppol=1 for unpolarized, 2 for spin-polarized calculations.
!!    %nspden=number of spin-density components.
!!
!! OUTPUT
!!  rhor(nfftf,nspden)=The density in the real space on the fine FFT grid.
!!   If nsppol==2, total charge in first half, spin-up component in second half.
!!
!! NOTES
!! In the case of PAW calculations:
!!    All computations are done on the fine FFT grid.
!!    All variables (nfftf,ngfftf,mgfftf) refer to this fine FFT grid.
!!    All arrays (densities/potentials...) are computed on this fine FFT grid.
!!    Developers have to be careful when introducing others arrays:
!!      they have to be stored on the fine FFT grid.
!! In the case of norm-conserving calculations:
!!    The mesh is the usual augmented FFT grid to treat correctly the convolution.
!!
!! PARENTS
!!      bethe_salpeter,screening,sigma
!!
!! CHILDREN
!!      cprj_alloc,cprj_free,pawaccrhoij,print_ij,rhoij_mpi_sum,wfd_bks_distrb
!!      wfd_get_cprj,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine wfd_mkrho(Wfd,Cryst,Psps,Kmesh,Bstr,ngfftf,nfftf,rhor)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_xmpi
 use m_errors
 use m_iterators

 use m_io_tools,  only : get_unit
 use m_crystal,   only : crystal_structure
 use m_bz_mesh,   only : bz_mesh_type
 use m_wfs,       only : wfs_descriptor, wfd_get_ur, wfd_update_bkstab, wfd_iterator_bks, wfd_change_ngfft

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
 use interfaces_56_recipspace
 use interfaces_67_common
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftf
 type(Bandstructure_type),intent(in) :: Bstr
 type(BZ_mesh_type),intent(in) :: Kmesh
 type(Crystal_structure),intent(in) :: Cryst
 type(Pseudopotential_type),intent(in) :: Psps
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays
 integer,intent(in) :: ngfftf(18)
 real(dp),intent(out) :: rhor(nfftf,Wfd%nspden)

!Local variables ------------------------------
!scalars
 integer,parameter :: tim_fourdp=0
 integer :: cplex,ib,ib_iter,ierr,ik,ir,is,n1,n2,n3,nfftotf,unt
 character(len=100) :: frmt
 character(len=500) :: msg
 character(len=fnlen) :: filnam
!arrays
 integer,allocatable :: irrzon(:,:,:)
 real(dp),allocatable :: phnons(:,:,:),rhog(:,:),rhor_down(:),rhor_mx(:),rhor_my(:)
 real(dp),pointer :: occfact(:,:,:)
 complex(dpc),allocatable :: wfr_x(:),wfr_y(:)
 complex(gwpc),allocatable,target :: wfr(:)
 complex(gwpc),pointer :: cwavef1(:),cwavef2(:)
 type(iter2_t) :: Iter_bks

!*************************************************************************

 DBG_ENTER("COLL")

 ! === Consistency check ===
 if (Wfd%nspden==4.and.Wfd%nspinor==1) then
   MSG_ERROR('nspden==4 and nspinor==1 not implemented')
 end if
                                                                                     
 ABI_CHECK(Wfd%nsppol==Bstr%nsppol,"mismatch in nspppol")

 occfact => Bstr%occ(1:Bstr%mband,1:Bstr%nkpt,1:Bstr%nsppol)

 if ( ANY(ngfftf(1:3) /= Wfd%ngfft(1:3)) ) call wfd_change_ngfft(Wfd,Cryst,Psps,ngfftf) 
 !
 ! === Calculate IBZ contribution to the charge density ===
 allocate(wfr(nfftf*Wfd%nspinor))

 if (Wfd%nspinor==2) then
   allocate(wfr_x(nfftf),wfr_y(nfftf))
   if (Wfd%nspden==4) then
     allocate(rhor_down(nfftf),rhor_mx(nfftf),rhor_my(nfftf))
     rhor_down=zero
     rhor_mx  =zero
     rhor_my  =zero
   else !TODO
     MSG_ERROR('nspden/=4 and nspinor=1 not implemeted')
   end if
 end if

 ! Update the (b,k,s) distribution table.
 call wfd_update_bkstab(Wfd) 

 ! ==== Calculate the unsymmetrized density ====
 rhor=zero
 Iter_bks = wfd_iterator_bks(Wfd,bks_mask=ABS(Bstr%occ)>=tol8)

 do is=1,Wfd%nsppol
   do ik=1,Wfd%nkibz
     do ib_iter=1,iter_len(Iter_bks,ik,is)
       ib = yield(Iter_bks,ib_iter,ik,is)

       !write(*,*)"wfd_mkrho: ",is,ik,ib
       call wfd_get_ur(Wfd,ib,ik,is,wfr)

       cwavef1 => wfr(1:nfftf)
       do ir=1,nfftf
         rhor(ir,is)=rhor(ir,is)+occfact(ib,ik,is)*CONJG(cwavef1(ir))*cwavef1(ir)*Kmesh%wt(ik)/Cryst%ucvol
       end do

       if (Wfd%nspinor==2) then
         cwavef2 => wfr(1+nfftf:2*nfftf)
         wfr_x(:)=cwavef1(:)+cwavef2(:)       ! $(\Psi^{1}+\Psi^{2})$
         wfr_y(:)=cwavef1(:)-j_dpc*cwavef2(:) ! $(\Psi^{1}-i\Psi^{2})$
         do ir=1,nfftf
           rhor_down(ir)=rhor_down(ir)+occfact(ib,ik,is)*CONJG(cwavef2(ir))*cwavef2(ir)*Kmesh%wt(ik)/Cryst%ucvol
           rhor_mx  (ir)=rhor_mx  (ir)+occfact(ib,ik,is)*CONJG(wfr_x  (ir))*wfr_x  (ir)*Kmesh%wt(ik)/Cryst%ucvol
           rhor_my  (ir)=rhor_my  (ir)+occfact(ib,ik,is)*CONJG(wfr_y  (ir))*wfr_y  (ir)*Kmesh%wt(ik)/Cryst%ucvol
         end do
       end if

     end do
   end do
 end do

 call iter_destroy(Iter_bks)
 call xsum_mpi(rhor,Wfd%comm,ierr)
 !
 ! === Symmetrization in G-space implementing also the AFM case ===
 n1=ngfftf(1)
 n2=ngfftf(2)
 n3=ngfftf(3)
 nfftotf=n1*n2*n3

 allocate(irrzon(nfftotf**(1-1/Cryst%nsym),2,(Wfd%nspden/Wfd%nsppol)-3*(Wfd%nspden/4)))
 allocate(phnons(2,nfftotf,(Wfd%nspden/Wfd%nsppol)-3*(Wfd%nspden/4)))

 if (Cryst%nsym/=1) then
   call irrzg(irrzon,Wfd%nspden,Wfd%nsppol,Cryst%nsym,n1,n2,n3,phnons,Cryst%symafm,Cryst%symrel,Cryst%tnons)
 end if

 cplex=1
 allocate(rhog(2,cplex*nfftf)) !this might be output

 call symrhg(cplex,Cryst%gprimd,irrzon,Wfd%MPI_enreg,nfftf,nfftotf,ngfftf,Wfd%nspden,Wfd%nsppol,&
&  Cryst%nsym,Wfd%paral_kgb,phnons,rhog,rhor,Cryst%rprimd,Cryst%symafm,Cryst%symrel)

 deallocate(rhog,phnons,irrzon)

 write(msg,'(a,f9.4)')' planewave contribution to nelect: ',SUM(rhor(:,1))*Cryst%ucvol/nfftf
 call wrtout(std_out,msg,'COLL')

 if (Wfd%nspden==4) then
   write(msg,'(a,3f9.4)')&
&     ' mx, my, mz: ',SUM(rhor(:,2))*Cryst%ucvol/nfftf,SUM(rhor(:,3))*Cryst%ucvol/nfftf,SUM(rhor(:,4))*Cryst%ucvol/nfftf
   call wrtout(std_out,msg,'COLL')
 end if

 deallocate(wfr)

 if (Wfd%nspinor==2) then
   deallocate(wfr_x,wfr_y)
   if (Wfd%nspden==4) deallocate(rhor_down,rhor_mx,rhor_my)
 end if

 if (.FALSE..and.Wfd%my_rank==Wfd%master) then
   filnam='__rhor__.dat'
   call isfile(filnam,'new')
   unt=get_unit(); open(unit=unt,file=filnam)
   write(frmt,*)'(2x,',Wfd%nspden,'(1x,f8.3))'
   do ir=1,nfftf
     write(unt,frmt)(rhor(ir,:))
   end do
   close(unt)
 end if

 DBG_EXIT("COLL")

end subroutine wfd_mkrho
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/test_charge
!! NAME
!! test_charge
!!
!! FUNCTION
!!  Reports info on the electronic charge as well as Drude plasma frequency.
!!  Mainly used in the GW part.
!!
!! INPUTS
!!  nelectron_exp=Expected total number of electrons (used to normalize the charge)
!!
!! OUTPUT
!!
!! PARENTS
!!      bethe_salpeter,mrgscr,screening,sigma
!!
!! CHILDREN
!!      cprj_alloc,cprj_free,pawaccrhoij,print_ij,rhoij_mpi_sum,wfd_bks_distrb
!!      wfd_get_cprj,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine test_charge(nfftf,nelectron_exp,nspden,rhor,ucvol,&
& usepaw,usexcnhat,usefinegrid,compch_sph,compch_fft,omegaplasma)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftf,nspden,usefinegrid,usepaw,usexcnhat
 real(dp),intent(in) :: compch_fft,compch_sph,ucvol,nelectron_exp
 real(dp),intent(out) :: omegaplasma
!arrays
 real(dp),intent(inout) :: rhor(nfftf,nspden)

!Local variables ------------------------------
!scalars
 real(dp) :: nelectron_tot,nelectron_fft 
 real(dp) :: nelectron_pw,nelectron_sph,rhoav,rs,nratio
 character(len=500) :: msg

!*************************************************************************

! ABI_UNUSED(usexcnhat)
if (usexcnhat==0)then
end if

 ! === For PAW output of compensation charges ===
 if (usepaw==1) then
!if (usepaw==1.and.usexcnhat>0) then ! TODO I still dont understand this if!
   write(msg,'(4a)')ch10,' PAW TEST:',ch10,' ==== Compensation charge inside spheres ============'
   if (compch_sph<greatest_real.and.compch_fft<greatest_real) &
&    write(msg,'(3a)')TRIM(msg),ch10,' The following values must be close...'
   if (compch_sph<greatest_real) &
&    write(msg,'(3a,f22.15)')TRIM(msg),ch10,' Compensation charge over spherical meshes = ',compch_sph
   if (compch_fft<greatest_real) then
     if (usefinegrid==1) then
       write(msg,'(3a,f22.15)')TRIM(msg),ch10,' Compensation charge over fine fft grid    = ',compch_fft
     else
       write(msg,'(3a,f22.15)')TRIM(msg),ch10,' Compensation charge over fft grid         = ',compch_fft
     end if
   end if
   call wrtout(ab_out,msg,'COLL')
   call wrtout(std_out,msg,'COLL')
   write(msg,'(a)')ch10
   call wrtout(ab_out,msg,'COLL')
   call wrtout(std_out,msg,'COLL')
 end if !PAW

 nelectron_pw =SUM(rhor(:,1))*ucvol/nfftf 
 nelectron_tot=nelectron_pw
 nratio       =nelectron_exp/nelectron_tot

 if (usepaw==1) then
   nelectron_sph=nelectron_pw+compch_sph
   nelectron_fft=nelectron_pw+compch_fft
   nelectron_tot=nelectron_sph
   nratio=(nelectron_exp-nelectron_sph)/nelectron_pw
 end if

 rhoav=nelectron_tot/ucvol ; rs=(three/(four_pi*rhoav))**third
 if (usepaw==0) then
  write(msg,'(2(a,f9.4))')&
&   ' Number of electrons calculated from density = ',nelectron_tot,'; Expected = ',nelectron_exp
 else
   write(msg,'(2(a,f9.4),a)')&
&   ' Total number of electrons per unit cell = ',nelectron_sph,' (Spherical mesh), ',nelectron_fft,' (FFT mesh)'
 end if
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')

!$write(msg,'(a,f9.4)')' Renormalizing smooth charge density using nratio = ',nratio
!$ rhor(:,:)=nratio*rhor(:,:)

 write(msg,'(a,f9.6)')' average of density, n = ',rhoav
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')
 write(msg,'(a,f9.4)')' r_s = ',rs
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')
 omegaplasma=SQRT(four_pi*rhoav)
 write(msg,'(a,f9.4,2a)')' omega_plasma = ',omegaplasma*Ha_eV,' [eV]',ch10
 call wrtout(std_out,msg,'COLL') ; call wrtout(ab_out,msg,'COLL')

end subroutine test_charge
!!***

!----------------------------------------------------------------------


!!****f* ABINIT/wfd_pawrhoij
!!
!! NAME
!! wfd_pawrhoij
!!
!! FUNCTION
!! Calculate the PAW quantities rhoij (augmentation occupancies)
!! Remember:for each atom, rho_ij=Sum_{n,k} {occ(n,k)*<Cnk|p_i><p_j|Cnk>}
!!
!! COPYRIGHT
!! Copyright (C) 1998-2010 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!!  atindx1(natom)=index table for atoms, inverse of atindx
!!  cprj(natom,nspinor*mband*mkmem*nsppol)= wave functions projected with non-local projectors:
!!                                   cprj_nk(i)=<p_i|Cnk> where p_i is a non-local projector.
!!  istwfk(nkpt)=parameter that describes the storage of wfs
!!  kptopt=option for the generation of k points
!!  mband=maximum number of bands
!!  natom=number of atoms in cell
!!  nkpt=number of k points
!!  nspinor=number of spinorial components of the wavefunctions
!!  nsppol=1 for unpolarized, 2 for spin-polarized
!!  occ(mband*nkpt*nsppol)=occupation number for each band for each k
!!  pawprtvol=control print volume and debugging output for PAW
!!
!! SIDE EFFECTS
!!  pawrhoij(natom) <type(pawrhoij_type)>= paw rhoij occupancies and related data
!!  On input: arrays dimensions
!!  On output:
!!    pawrhoij(:)%rhoij_(lmn2_size,nspden)=
!!          Sum_{n,k} {occ(n,k)*conjugate[cprj_nk(ii)].cprj_nk(jj)} (non symetrized)
!!
!! PARENTS
!!      paw_qpscgw
!!
!! CHILDREN
!!      cprj_alloc,cprj_free,pawaccrhoij,print_ij,rhoij_mpi_sum,wfd_bks_distrb
!!      wfd_get_cprj,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine wfd_pawrhoij(Wfd,Cryst,Bst,kptopt,pawrhoij,pawprtvol)

 use defs_basis
 use defs_datatypes
 use m_errors

 use m_crystal, only : crystal_structure
 use m_wfs,     only : wfs_descriptor, wfd_get_cprj, wfd_bks_distrb

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
 use interfaces_53_abiutil
 use interfaces_66_paw
!End of the abilint section

 implicit none

!Arguments ---------------------------------------------
!scalars
 integer,intent(in) :: kptopt,pawprtvol
 type(crystal_structure),intent(in) :: Cryst
 type(wfs_descriptor),intent(inout) :: Wfd
 type(bandstructure_type),intent(in) :: Bst
!arrays
 type(pawrhoij_type),intent(inout) :: pawrhoij(Wfd%natom)

!Local variables ---------------------------------------
!scalars
 integer :: cplex,iatom,band,ik_ibz
 integer :: spin,natinc,nband_k,nsp2,option,rhoij_cplex,lmn2_size,nspden
 logical :: usetimerev
 real(dp) :: occup,wtk_k
 character(len=500) :: msg
!arrays
 integer,allocatable :: idum(:)
 !real(dp) :: tsec(2)
 character(len=8),parameter :: dspin(6)=(/"up      ","down    ","dens (n)","magn (x)","magn (y)","magn (z)"/)
 type(cprj_type),allocatable :: cwaveprj(:,:)
 integer :: bks_distrb(Wfd%mband,Wfd%nkibz,Wfd%nsppol)
 integer :: got(Wfd%nproc)
 logical :: bks_mask(Wfd%mband,Wfd%nkibz,Wfd%nsppol)

!************************************************************************

 DBG_ENTER("COLL")

 !Initialize temporary file (if used)
 !iorder_cprj=0
 !call cprj_diskinit_r(Cryst%atindx1,natom,iorder_cprj,mkmem,natom,0,dimcprj,Wfd%nspinor,unpaw)

 ! Allocate temporary cwaveprj storage (sorted by atom type)
 allocate(cwaveprj(Wfd%natom,Wfd%nspinor))
 call cprj_alloc(cwaveprj,0,Wfd%nlmn_sort)

 ! Initialize output quantities if not already done.
 do iatom=1,Wfd%natom
   if (pawrhoij(iatom)%use_rhoij_==0) then
     rhoij_cplex     = pawrhoij(iatom)%cplex
     lmn2_size = pawrhoij(iatom)%lmn2_size
     nspden    = pawrhoij(iatom)%nspden
     allocate(pawrhoij(iatom)%rhoij_(rhoij_cplex*lmn2_size,nspden))
     pawrhoij(iatom)%use_rhoij_=1
   end if
   pawrhoij(iatom)%rhoij_=zero
 end do

 option=1; usetimerev=(kptopt>0.and.kptopt<3)

 ! Distribute (b,k,s).
 where (ABS(Bst%occ)>tol8)
   bks_mask=.TRUE.
 else where
   bks_mask=.FALSE.
 end where
 got=zero

 call wfd_bks_distrb(Wfd,bks_distrb,got,bks_mask)

 do spin=1,Wfd%nsppol
   do ik_ibz=1,Wfd%nkibz

     nband_k=Wfd%nband(ik_ibz,spin)
     wtk_k=Bst%wtk(ik_ibz)

     cplex=2; if (Wfd%istwfk(ik_ibz)>1) cplex=1

     do band=1,nband_k

       if (bks_distrb(band,ik_ibz,spin) == Wfd%my_rank) then
         !locc_test = (abs(Bst%occ(band,ik_ibz,spin))>tol8)
         occup = Bst%occ(band,ik_ibz,spin)

!        Extract cprj for current band
!        Must read cprj when mkmem=0 (even if unused) to have right pointer inside _PAW file
!         if (locc_test.or.mkmem==0) then
!           call cprj_get(Cryst%atindx1,cwaveprj,cprj,natom,ib_this_proc,ibg,ik_ibz,iorder_cprj,spin,&
!&           mband_cprj,mkmem,Wfd%mpi_enreg,natom,1,mband_cprj,Wfd%nspinor,Wfd%nsppol,unpaw)
!         end if

          ! cwaveprj are sorted by atom type.
          call wfd_get_cprj(Wfd,band,ik_ibz,spin,Cryst,cwaveprj,sorted=.TRUE.)

!         Accumulate contribution from (occupied) current band
          !if (locc_test) then
           call pawaccrhoij(Cryst%atindx1,cplex,cwaveprj,cwaveprj ,0,spin,Wfd%natom,&
&            Wfd%nspinor,occup,option,pawrhoij,usetimerev,wtk_k)
          !end if
       end if
     end do !band

   end do !ik_ibz
 end do !spin
 !
 ! Free temporary cwaveprj storage.
 call cprj_free(cwaveprj); deallocate(cwaveprj)
 !
 !==========================================
 ! MPI: need to exchange arrays between procs
 ! TODO it should be tested.
 call rhoij_mpi_sum(pawrhoij,Wfd%comm)
 !
 ! Print info.
 if (abs(pawprtvol)>=1) then
   natinc=1; if(Wfd%natom>1.and.pawprtvol>=0) natinc=Wfd%natom-1
   do iatom=1,Cryst%natom,natinc
     nsp2=pawrhoij(iatom)%nsppol;if (pawrhoij(iatom)%nspden==4) nsp2=4
     write(msg, '(4a,i3,a)') ch10," PAW TEST:",ch10,&
&     ' ====== Values of RHOIJ in wfd_pawrhoij (iatom=',iatom,') ======'
     if (pawrhoij(iatom)%nspden==2.and.pawrhoij(iatom)%nsppol==1) write(msg,'(3a)') trim(msg),ch10,&
&     '      (antiferromagnetism case: only one spin component)'
     call wrtout(std_out,msg,'COLL')
     do spin=1,nsp2
       if (pawrhoij(iatom)%nspden/=1) then
         write(msg, '(3a)') '   Component ',trim(dspin(spin+2*(pawrhoij(iatom)%nspden/4))),':'
         call wrtout(std_out,msg,'COLL')
       end if
       call print_ij(pawrhoij(iatom)%rhoij_(:,spin),pawrhoij(iatom)%lmn2_size,&
&       pawrhoij(iatom)%cplex,pawrhoij(iatom)%lmn_size,1,-1,idum,0,pawprtvol,idum,-1.d0,1)
     end do
   end do
 end if

 DBG_EXIT("COLL")

end subroutine wfd_pawrhoij
!!***
