!{\src2tex{textfont=tt}}
!!****f* ABINIT/m_ppmodel
!! NAME
!! m_ppmodel
!!
!! FUNCTION
!!  Module containing the definition of the PPmodel_type used to deal with 
!!  the plasmonpole technique. Methods to operate on the object are also provided.
!!
!! COPYRIGHT
!!  Copyright (C) 2008-2010 ABINIT group (MG, GMR, VO, LR, RWG, RS)
!!  This file is distributed under the terms of the
!!  GNU General Public License, see ~abinit/COPYING
!!  or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

MODULE m_ppmodel

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use defs_gw
 use m_errors
 use m_jolly_pointers

 use m_numeric_tools,  only : is_zero
 use m_gwdefs,         only : GW_Q0_DEFAULT
 use m_bz_mesh,        only : bz_mesh_type
 use m_gsphere,        only : gvectors_type
 use m_coulombian,     only : coulombian_type, cmod_qpg
 use m_fft_mesh,       only : cggfft

 implicit none

 private

 public :: ppmodel_symmetrize     ! Symmetrize PPm parameters in the BZ.
 public :: nullify_ppmodel        ! Nullify all pointers
 public :: init_ppmodel           ! Initialize dimensions and pointers
 public :: destroy_ppmodel        ! Destruction method.
 public :: setup_ppmodel          ! Main Driver 
 public :: getem1_from_PPm        ! Reconstruct e^{-1}(w) from PPm.
 public :: get_ppm_eigenvalues       

 integer,public,parameter :: PPM_NO_PLASMONPOLE  = 0
 integer,public,parameter :: PPM_GODBY_NEEDS     = 1
 integer,public,parameter :: PPM_HYBERTSEN_LOUIE = 2
 integer,public,parameter :: PPM_LINDEN_HORSH    = 3
 integer,public,parameter :: PPM_ENGEL_FARID     = 4

CONTAINS  !==============================================================================
!!***

!!****f* m_ppmodel/ppmodel_symmetrize
!! NAME
!!  PPmodel_symmetrize
!!
!! FUNCTION
!!  Symmetrize the plasmonpole matrix elements in the full BZ zone.
!!
!! INPUTS
!!  PPm<PPmodel_type>=data type containing information on the plasmonpole technique 
!!  Gsph<Gvectors_type>=data related to the G-sphere
!!    %grottb
!!    %phmSGt 
!!  Qmesh<BZ_mesh_type>=Info on the q-mesh
!!    %nbz=number if q-points in the BZ
!!    %tab(nbz)=index of the symmeric q-point in the IBZ, for each point in the BZ
!!    %tabo(nbz)=the operation that rotates q_ibz onto \pm q_bz (depending on tabi) 
!!    %tabi(nbz)=-1 if time-reversal has to be considered, 1 otherwise
!!  iq_bz=Index of the q-point in the BZ where PPmodel parameters have to be symmetrized
!!
!! OUTPUT
!!  botsq 
!!  otq
!!  eig (only if PPm%ppmodel==3)
!!
!! NOTES
!!  In the present implementation we are not considering a possible umklapp vector G0.
!!  In this case,indeed, the equation is different since we have to consider G-G0. 
!!  There is however a check in sigma
!! 
!!  * Remember the symmetry properties of \tilde\espilon^{-1}
!!    If q_bz=Sq_ibz+G0:
!! 
!!    $\epsilon^{-1}_{SG1-G0,SG2-G0}(q_bz) = e^{+iS(G2-G1).\tau}\epsilon^{-1}_{G1,G2)}(q)
!!
!!    If time-reversal symmetry can be used then :
!!    $\epsilon^{-1}_{G1,G2}(-q_bz) = e^{+i(G1-G2).\tau}\epsilon^{-1}_{-S^{-1}(G1+Go),-S^{-1}(G2+G0)}^*(q)
!! 
!! * Notice that eig is used only if PPm%model==3
!!
!! TODO
!!  Symmetrization can be skipped if iq_bz correspond to a point in the IBZ
!!
!! PARENTS
!!      calc_sigc_me
!!
!! CHILDREN
!!
!! SOURCE

subroutine ppmodel_symmetrize(PPm,Gsph,Qmesh,iq_bz,botsq,otq,eig) 

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iq_bz 
 type(PPmodel_type),intent(in) :: PPm
 type(Gvectors_type),intent(in) :: Gsph
 type(BZ_mesh_type),intent(in) :: Qmesh
 type(Jpt_gwpc_2D),intent(inout) :: botsq,otq,eig
!Local variables-------------------------------
!scalars
 integer :: ii,jj
 integer :: iq_ibz,iiq,isymq,iq_curr
 character(len=500) :: msg
!arrays
 integer,pointer :: grottb(:)
 complex(gwpc),pointer :: phsgt(:)
! *********************************************************************

 !@PPmodel_type

 ! Here there is a problem with the small q, still cannot use BZ methods
 iq_ibz=Qmesh%tab(iq_bz) ; isymq=Qmesh%tabo(iq_bz) ; iiq=(3-Qmesh%tabi(iq_bz))/2

 iq_curr=iq_ibz ; if (PPm%mqmem==0) iq_curr=1 

 grottb => Gsph%rottb (1:PPm%npwc,iiq,isymq)
 phsgt  => Gsph%phmSGt(1:PPm%npwc,isymq) 

 ! === Symmetrize PPM parameters, equations depend on model ===
 SELECT CASE (PPm%model)

 CASE (PPM_GODBY_NEEDS, PPM_HYBERTSEN_LOUIE)
   ! * Plasmon pole frequencies otq are invariant under symmetry 
   do jj=1,PPm%npwc
     do ii=1,PPm%npwc
      botsq%datum(grottb(ii),grottb(jj))=PPm%bigomegatwsq%datum(ii,jj,iq_curr)*phsgt(ii)*CONJG(phsgt(jj))
      otq%datum  (grottb(ii),grottb(jj))=PPm%omegatw%datum(ii,jj,iq_curr)  
     end do
   end do

 CASE (PPM_LINDEN_HORSH)
   ! * For notations see pag 22 of Quasiparticle Calculations in solid (Aulbur et al)
   !  If q_bz=Sq_ibz+G0 then:
   !
   ! $\omega^2_{ii}(q_bz) = \omega^2_{ii}(q)$        (otq array)
   ! $\alpha_{ii}(q_bz)   = \alpha_{ii}(q)$          (botq array
   ! $\Phi_{SG-G0}(q_bz)  = \Phi_{G}(q) e^{-iSG.t}$  (eigenvectors of e^{-1}, eig array) 
   !   
   do ii=1,PPm%npwc ! DM bands index
     otq%datum  (ii,1)=PPm%omegatw%datum     (ii,1,iq_curr)
     botsq%datum(ii,1)=PPm%bigomegatwsq%datum(ii,1,iq_curr)
     do jj=1,PPm%npwc
       eig%datum(grottb(jj),ii)=PPm%eigpot(jj,ii,iq_curr)*phsgt(jj)
     end do
   end do
   if (iiq==2) eig%datum(:,:)=CONJG(eig%datum(:,:)) ! Time-reversal

 CASE (PPM_ENGEL_FARID)
   ! * For notations see pag 23 of Quasiparticle Calculations in solid (Aulbur et al)
   ! If q_bz=Sq_ibz+G0 then:
   !   
   ! $\omega^2_{ii}(q_bz) = \omega^2_{ii}(q)$        (otq array)
   ! $y_{SG-G0}(q_bz)     = y_{G}(q) e^{-iSG.t}$     (y=Lx) 
   !   
   do ii=1,PPm%npwc ! DM bands index
     otq%datum(ii,1)=PPm%omegatw%datum(ii,1,iq_curr)
     do jj=1,PPm%npwc
      botsq%datum(grottb(jj),ii)=PPm%bigomegatwsq%datum(jj,ii,iq_curr)*phsgt(jj)
     end do
   end do

 CASE DEFAULT 
   write(msg,'(a,i6)')' Wrong value for PPm%model = ',PPm%model
   MSG_BUG(msg)

 END SELECT
 !
 ! * Take into account time-reversal symmetry.
 if (iiq==2) botsq%datum(:,:)=CONJG(botsq%datum(:,:))

end subroutine ppmodel_symmetrize
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/nullify_ppmodel
!! NAME
!!  nullify_ppmodel
!!
!! FUNCTION
!!  Nullify dynamic entities in a PPmodel_type object.
!! 
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      m_ppmodel
!!
!! CHILDREN
!!
!! SOURCE

subroutine nullify_ppmodel(PPm)

 use defs_basis

 implicit none

!Arguments ------------------------------------
 type(PPmodel_type),intent(inout) :: PPm
!Local variables-------------------------------
 integer :: istat
! *********************************************************************

 !@PPmodel_type
 call destroy_jpt(PPm%bigomegatwsq,istat)
 call destroy_jpt(PPm%omegatw     ,istat)
 nullify(PPm%eigpot)

end subroutine nullify_ppmodel
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/destroy_ppmodel
!! NAME
!!  destroy_ppmodel
!!
!! FUNCTION
!!  Deallocate all associated pointers defined in a variable of type PPmodel_type.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine destroy_ppmodel(PPm)

 use defs_basis

 implicit none

!Arguments ------------------------------------
 type(PPmodel_type),intent(inout) :: PPm
!Local variables-------------------------------
 integer :: istat
! *********************************************************************

 !@PPmodel_type
 call destroy_jpt(PPm%bigomegatwsq,istat)
 call destroy_jpt(PPm%omegatw     ,istat)
 if (associated(PPm%eigpot)) deallocate(PPm%eigpot)

end subroutine destroy_ppmodel
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/init_ppmodel
!! NAME
!!  init_ppmodel
!!
!! FUNCTION
!!  Initialize dimensions and other useful variables related to the PPmodel
!!
!! INPUTS
!!
!! OUTPUT
!!
!! PARENTS
!!      mrgscr,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine init_ppmodel(PPm,Er,ppmodel,drude_plsmf,gwcalctyp10)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 integer,intent(in) :: ppmodel,gwcalctyp10
 real(dp),intent(in) :: drude_plsmf
 type(PPmodel_type),intent(out) :: PPm
 type(Epsilonm1_results),intent(in) :: Er

!Local variables-------------------------------
 integer :: dim_q,istat,dt_shape(2,3)
 logical :: ltest
 character(len=500) :: msg      
! *********************************************************************

 DBG_ENTER("COLL")

 !@PPmodel_type
 call nullify_ppmodel(PPm)

 PPm%nqibz = Er%nqibz
 PPm%mqmem = Er%mqmem
 ltest = (PPm%mqmem==0.or.PPm%mqmem==PPm%nqibz)
 ABI_CHECK(ltest,'Wrong value for mqmem')
 PPm%npwc  = Er%npwe
 PPm%model = ppmodel

 PPm%drude_plsmf = drude_plsmf

 SELECT CASE (PPm%model)

 CASE (PPM_NO_PLASMONPOLE) 
   MSG_WARNING("Called with ppmodel==0")
   PPm%dm2_botsq = 0
   PPm%dm2_otq   = 0
   PPm%dm_eig    = 0 
   RETURN

 CASE (PPM_GODBY_NEEDS, PPM_HYBERTSEN_LOUIE)
   PPm%dm2_botsq = PPm%npwc
   PPm%dm2_otq   = PPm%npwc
   PPm%dm_eig    = 1 ! Should be set to 0, but g95 doesnt like zero-sized arrays

 CASE (PPM_LINDEN_HORSH)
   PPm%dm2_botsq = 1  
   PPm%dm2_otq   = 1
   PPm%dm_eig    = PPm%npwc

 CASE (PPM_ENGEL_FARID)
   PPm%dm2_botsq = PPm%npwc
   PPm%dm2_otq   = 1
   PPm%dm_eig    = 1 ! Should be set to 0, but g95 doesnt like zero-sized arrays

 CASE DEFAULT
   write(msg,'(a,i6)')' Wrong PPm%model = ',PPm%model
   MSG_BUG(msg)

 END SELECT
 !
 ! === Do we store full the q-mesh or out-of-memory solution? ===
 dim_q=PPm%nqibz ; if (PPm%mqmem==0) dim_q=1

 allocate(PPm%eigpot       (PPm%dm_eig,PPm%dm_eig,   dim_q), STAT=istat)
 !
 ! === If developer, save memory by using the same piece of memory for epsm1 and 
 ! the PPm parameters
 ! This is a VERY bad coding habit and, therefore, should be used only by developers.
 ltest = .not.(PPm%save_memory_devel .and. (PPm%model/=1 .or. .not.(gwcalctyp10==0.or.gwcalctyp10==8)) )
 ABI_CHECK(ltest,'save_memory_devel should be set to FALSE (hardcoded)')
 if (PPm%save_memory_devel) then
   PPm%bigomegatwsq%datum => Er%epsm1(:,:,1,:) 
   PPm%omegatw%datum      => Er%epsm1(:,:,2,:)
 else
   dt_shape(1,:) = 1 ; dt_shape(2,1) = PPm%npwc   ; dt_shape(2,2) = PPm%dm2_botsq ; dt_shape(2,3) = dim_q
   call allocate_jpt(PPm%bigomegatwsq,dt_shape,istat)
   dt_shape(1,:) = 1 ; dt_shape(2,1) = PPm%npwc   ; dt_shape(2,2) = PPm%dm2_otq   ; dt_shape(2,3) = dim_q
   call allocate_jpt(PPm%omegatw,dt_shape,istat)
 end if

 DBG_EXIT("COLL")
 
end subroutine init_ppmodel
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/setup_ppmodel
!! NAME
!! setup_ppmodel
!!
!! FUNCTION
!!  Initialize some values of several arrays of the Er% datastructure 
!!  that are used in case of plasmonpole calculations
!!  Just a wrapper around different plasmonpole routines.
!!
!! INPUTS
!!  paral_kgb=variable related to band parallelism
!!  Qmesh<bz_mesh_type>=the q-mesh used for the inverse dielectric matrix
!!    %nibz=number of irreducible q-points
!!    %ibz(3,%nibz)=the irred q-point
!!  Er<epsilonm1_results>=the inverse dielectric matrix 
!!    %nomega=number of frequencies in $\epsilon^{-1}$
!!    %epsm1=the inverse dielctric matrix 
!!    %omega=frequencies in epsm1
!!    %npwe=number of G vectors for the correlation part
!!  MPI_enreg<MPI_type>=informations about MPI parallelization
!!  ngfftf(18)=contain all needed information about the 3D fine FFT mesh, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  gmet(3,3)=reciprocal space metric ($\textrm{bohr}^{-2}$).
!!  gprimd(3,3)=dimensional primitive translations for reciprocal space ($\textrm{bohr}^{-1}$)
!!  nfftf=the number of points in the FFT mesh (for this processor)
!!  rhor_tot(nfftf)=the total charge in real space
!!  PPm<PPmodel_type>: 
!!    %ppmodel=the type of  plasmonpole model 
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  PPm<PPmodel_type>: 
!!  == if ppmodel 1 or 2 ==
!!   %omegatw and %bigomegatwsq=PPmodel parameters 
!!  == if ppmodel 3 ==
!!   %omegatw, %bigomegatwsq and %eigpot=PPmodel parameters
!!  == if ppmodel 4 ==
!!   %omegatw and %bigomegatwsq=PPmodel parameters 
!!
!! NOTES
!! TODO: rhor_tot should be replaced by rhog_tot
!! FFT parallelism won"t work 
!! Solve Issue with MPI_enreg
!!
!! PARENTS
!!      calc_sigc_me,mrgscr,sigma
!!
!! CHILDREN
!!
!! SOURCE

subroutine setup_ppmodel(PPm,paral_kgb,Qmesh,Er,nfftf,gvec,ngfftf,gmet,gprimd,rhor_tot,&
& iqiA) !Optional

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftf,paral_kgb
 integer,intent(in),optional :: iqiA
 type(BZ_mesh_type),intent(in) :: Qmesh
 type(Epsilonm1_results),intent(inout) :: Er
 type(PPmodel_type),intent(inout) :: PPm
!arrays
 integer,intent(in) :: gvec(3,Er%npwe),ngfftf(18)
 real(dp),intent(in) :: gmet(3,3),gprimd(3,3)
 real(dp),intent(inout) :: rhor_tot(nfftf)

!Local variables-------------------------------
!scalars
 integer :: nqiA
 logical :: ltest,single_q
 character(len=500) :: msg

! *************************************************************************

 DBG_ENTER("COLL")

 !@PPmodel_type
 !
 ! === if iqiA is present, then consider only one qpoint to save memory ===
 ! * This means the object has been already initialized
 nqiA=Qmesh%nibz; single_q=.FALSE.
 if (PRESENT(iqiA)) then 
   nqiA=1 ; single_q=.TRUE.
   ltest=PRESENT(iqiA)
   ABI_CHECK(ltest,'For single q-point mode, also iqiA must be present')
 end if
 !
 ! Allocate plasmonpole parameters 
 ! TODO ppmodel==1 by default, should be set to 0 if AC and CD
 SELECT CASE (PPm%model)

 CASE (PPM_NO_PLASMONPOLE)
   write(msg,'(a)')' Skipping Plasmompole model calculation' 
   call wrtout(std_out,msg,'COLL') 
   RETURN

 CASE (PPM_GODBY_NEEDS) 
   ! * Note: q-dependency enters only through epsilon^-1

   ! save_memory_devel is to be used only for developers (aggressive memory savings)!
   if (PPm%save_memory_devel) then
     call cppm1par(Er%npwe,nqiA,Er%nomega,Er%epsm1,Er%omega,PPm%drude_plsmf)
   else
     call cppm1par(Er%npwe,nqiA,Er%nomega,Er%epsm1,Er%omega,PPm%drude_plsmf,PPm%omegatw%datum,PPm%bigomegatwsq%datum)
   end if

 CASE (PPM_HYBERTSEN_LOUIE)
   if (.not.single_q) then
     call cppm2par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,PPm%omegatw%datum,&
&      ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh,gmet)
   else
     call cppm2par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,PPm%omegatw%datum,&
&      ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh,gmet,iqiA)
   end if

 CASE (PPM_LINDEN_HORSH)
   ! TODO Check better double precision, this routine is in a messy state
   if (.not.single_q) then
     call cppm3par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,&
&      PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,PPm%eigpot,Qmesh)
   else
     call cppm3par(paral_kgb,Er%npwe,nqiA,Er%nomega,Er%epsm1,PPm%bigomegatwsq%datum,&
&      PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,PPm%eigpot,Qmesh,iqiA)
   end if

 CASE (PPM_ENGEL_FARID)
   ! TODO Check better double precision, this routine is in a messy state
   if (.not.single_q) then
     call cppm4par(paral_kgb,Er%npwe,nqiA,Er%epsm1,Er%nomega,PPm%bigomegatwsq%datum,&
&     PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh)
   else
     call cppm4par(paral_kgb,Er%npwe,nqiA,Er%epsm1,Er%nomega,PPm%bigomegatwsq%datum,&
&     PPm%omegatw%datum,ngfftf,gvec,gprimd,rhor_tot,nfftf,Qmesh,iqiA)
   end if

 CASE DEFAULT
   write(msg,'(a,i6)')' Wrong PPm%model=',PPm%model
   MSG_BUG(msg)

 END SELECT

 DBG_EXIT("COLL")

end subroutine setup_ppmodel
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/getem1_from_ppm
!! NAME
!!  getem1_from_ppm
!!
!! FUNCTION
!!  Calculate the symmetrized inverse dielectric matrix from the parameters of the plasmon-pole model.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      m_ppmodel,mrgscr
!!
!! CHILDREN
!!
!! SOURCE

subroutine getem1_from_ppm(PPm,iqibz,zcut,nomega,omega,Vcp,em1q)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iqibz,nomega
 type(PPmodel_type),intent(in) :: PPm
 type(Coulombian_type),intent(in) :: Vcp
 real(dp),intent(in) :: zcut
!arrays
 complex(dpc),intent(in) :: omega(nomega)
 complex(dpc),intent(out) :: em1q(PPm%npwc,PPm%npwc,nomega)

!Local variables-------------------------------
!scalars
 integer :: ig1,ig2,io,idm
 real(dp) :: den 
 complex(dpc) :: qpg1,qpg2,ug1,ug2
 complex(dpc) :: delta,em1ggp,otw,zzpq,yg1,yg2,bot1,bot2,chig1g2
 character(len=500) :: msg
!arrays

! *************************************************************************

 ABI_CHECK(PPm%mqmem/=0,'mqmem==0 not implemented')

 !TODO zcut should be an entry in PPm
 delta=CMPLX(zero,zcut)

 select case (PPm%model)

 case (PPM_GODBY_NEEDS, PPM_HYBERTSEN_LOUIE)
 do io=1,nomega
 
  do ig2=1,PPm%npwc
   do ig1=1,PPm%npwc
    !den = omega(io)**2-REAL(PPm%omegatw(ig1,ig2,iqibz)**2)
    !if (den**2<zcut**2) den = omega(io)**2-REAL( (PPm%omegatw(ig1,ig2,iqibz)-delta)**2 )
    den = omega(io)**2-REAL( (PPm%omegatw%datum(ig1,ig2,iqibz)-delta)**2 )
    em1ggp = PPm%bigomegatwsq%datum(ig1,ig2,iqibz)/den
    if (ig1==ig2) em1ggp=em1ggp+one
    em1q(ig1,ig2,io)=em1ggp
    !em1q(ig1,ig2,io)=em1ggp*Vcp%vc_sqrt(ig1,iqibz)*Vcp%vc_sqrt(ig2,iqibz)
   end do
  end do

 end do !io

 case (PPM_LINDEN_HORSH)

  !TODO Check coefficients 
  do io=1,nomega

   do ig2=1,PPm%npwc
    do ig1=1,PPm%npwc

     em1ggp=czero
     do idm=1,PPm%npwc
      !den=omega(io)**2-(omegatw(ig1,ig2,iqibz)-delta)**2
      !em1w(io)=em1w(io)+eigvec(ig1,idm,iqibz)*conjg(eigvec(ig2,idm,iqibz))*bigomegatwsq(ig1,ig2,iqibz)/den
      ug1 =PPm%eigpot(ig1,idm,iqibz)
      ug2 =PPm%eigpot(ig2,idm,iqibz)
      otw =PPm%bigomegatwsq%datum(idm,1,iqibz)*PPm%omegatw%datum(idm,1,iqibz)
      zzpq=PPm%bigomegatwsq%datum(idm,1,iqibz)
      den=half*REAL(zzpq*otw*( one/(omega(io)-otw+delta) - one/(omega(io)+otw-delta) ))
      em1ggp=em1ggp+ug1*CONJG(ug2)*den
      !eigenvalues(idm,io)=one + half*REAL(zzpq*otw*( one/(omega(io)-otw+delta) - one/(omega(io)+otw-delta) ))
     end do

     if (ig2==ig1) em1ggp=em1ggp+one
     em1q(ig1,ig2,io)=em1ggp

    end do !ig1
   end do !ig2

  end do !iomega 

 case (PPM_ENGEL_FARID)
 ! Make e^-1

 do io=1,nomega

  do ig2=1,PPm%npwc
   qpg2=one/Vcp%vc_sqrt(ig2,iqibz)
   do ig1=1,PPm%npwc
    qpg1=one/Vcp%vc_sqrt(ig1,iqibz)

    chig1g2=czero 
    do idm=1,PPm%npwc
     otw =PPm%omegatw%datum(idm,1,iqibz)
     bot1=PPm%bigomegatwsq%datum(ig1,idm,iqibz)
     bot2=PPm%bigomegatwsq%datum(ig2,idm,iqibz)
     yg1=SQRT(otw/four_pi)*qpg1*bot1
     yg2=SQRT(otw/four_pi)*qpg2*bot2
     chig1g2=chig1g2 + yg1*CONJG(yg2)/(omega(io)**2-(otw-delta)**2)
    end do

    em1ggp=four_pi*chig1g2/(qpg1*qpg2)
    if (ig1==ig2) em1ggp=em1ggp+one
    em1q(ig1,ig2,io)=em1ggp !*Vcp%vc_sqrt(ig1,iqibz)*Vcp%vc_sqrt(ig2,iqibz)

   end do !ig1
  end do !ig2

 end do !iomega

 case default
  write(msg,'(a,i6)')" Wrong PPm%model = ",PPm%model
  MSG_BUG(msg)
 end select

end subroutine getem1_from_ppm
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/get_ppm_eigenvalues
!! NAME
!!  get_ppm_eigenvalues
!!
!! FUNCTION
!!  Constructs the inverse dielectri matrix starting from the plasmon-pole
!!  parameters and calculates the frequency-dependent eigenvalues for each 
!!  of the nomega frequencies specifies in the array omega.
!!
!! INPUTS
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      mrgscr
!!
!! CHILDREN
!!
!! SOURCE

subroutine get_ppm_eigenvalues(PPm,iqibz,zcut,nomega,omega,Vcp,eigenvalues)

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_linalg
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: iqibz,nomega
 type(PPmodel_type),intent(in) :: PPm
 type(Coulombian_type),intent(in) :: Vcp
 real(dp),intent(in) :: zcut
!arrays
 complex(dpc),intent(in) :: omega(nomega)
 complex(dpc),intent(out) :: eigenvalues(PPm%npwc,nomega)

!Local variables-------------------------------
!scalars
 integer :: info,lwork,istat,negw,ig1,ig2,idx,sdim,iomega
 character(len=500) :: msg
!arrays
 real(dp),allocatable :: ww(:),rwork(:)
 complex(dpc),allocatable :: work(:),Adpp(:),eigvec(:,:),wwc(:),vs(:,:),Afull(:,:)
 complex(dpc),allocatable :: em1q(:,:,:)
 logical,allocatable :: bwork(:)
 logical :: sortcplx !BUG in abilint

! *************************************************************************

 ABI_CHECK(PPm%mqmem/=0,'mqmem==0 not implemented')

 allocate(em1q(PPm%npwc,PPm%npwc,nomega))

 call getem1_from_PPm(PPm,iqibz,zcut,nomega,omega,Vcp,em1q)
 !print*,em1q

 do iomega=1,nomega

  print*,'iomega',iomega

  if (ABS(REAL(omega(iomega)))>0.00001) then
   !if (.TRUE.) then
   ! === Eigenvalues for a generic complex matrix ===
                                                                                                         
   lwork=4*2*PPm%npwc
   allocate(wwc(PPm%npwc),work(lwork),rwork(PPm%npwc),bwork(PPm%npwc))
   allocate(vs(PPm%npwc,PPm%npwc),STAT=istat)
   allocate(Afull(PPm%npwc,PPm%npwc),STAT=istat)
 print*,'done'
                                                                                                         
   Afull=em1q(:,:,iomega)
                                                                                                         
   !for the moment no sort, maybe here I should sort using the real part?
   call ZGEES('V','N',sortcplx,PPm%npwc,Afull,PPm%npwc,sdim,wwc,vs,PPm%npwc,work,lwork,rwork,bwork,info)
   if (info/=0) then 
    write(msg,'(2a,i10)')' get_ppm_eigenvalues : Error in ZGEES, diagonalizing complex matrix, info = ',info
    call wrtout(std_out,msg,'COLL') 
   end if
                                                                                                         
 print*,'done'
   eigenvalues(:,iomega)=wwc(:)
                                                                                                         
   deallocate(wwc,work,rwork,bwork)
   deallocate(vs)
   deallocate(Afull)

  else 
   ! === Hermitian Case ===
   lwork=2*PPm%npwc-1
   allocate(ww(PPm%npwc),work(lwork),rwork(3*PPm%npwc-2))
   allocate(eigvec(PPm%npwc,PPm%npwc))
   allocate(Adpp(PPm%npwc*(PPm%npwc+1)/2),STAT=istat)
   if (istat/=0) STOP ' get_ppm_eigenvalues : out of memory in Adpp'

   print*,'in hermitian'
 
   idx=0
   do ig2=1,PPm%npwc
    do ig1=1,ig2
     idx=idx+1
     Adpp(idx)=em1q(ig1,ig2,iomega)
    end do
   end do

   ! For the moment we require also the eigenvectors.
   call ZHPEV('V','U',PPm%npwc,Adpp,ww,eigvec,PPm%npwc,work,rwork,info)

   if (info/=0) then 
    write(msg,'(2a,i10)')' get_ppm_eigenvalues : Error diagonalizing matrix, info = ',info
    call wrtout(std_out,msg,'COLL') 
   end if
   negw = (COUNT((REAL(ww)<tol6)))
   if (negw/=0) then 
    write(msg,'(3a,i5,a,i3,a,f8.4)')&
&    ' get_ppm_eigenvalues : WARNING - ',ch10,&
&    ' Found negative eigenvalues. No. ',negw,' at iqibz= ',iqibz,' minval= ',MINVAL(REAL(ww))
   end if

   eigenvalues(:,iomega)=ww(:)

   deallocate(ww,work,rwork)
   deallocate(eigvec)
   deallocate(Adpp)
  end if

 end do !iomega

 deallocate(em1q)

end subroutine get_ppm_eigenvalues
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/cppm1par
!! NAME
!! cppm1par
!!
!! FUNCTION
!! Calculate the plasmon-pole parameters big-omega-twiddle-squared and omega-twiddle from
!! epsilon-twiddle^-1 calculated for nomega (usually 2) frequencies omega=0 and omega=iE0.
!!
!! INPUTS
!!  epsm1(npwc,npwc,nomega,nqibz)=dielectric matrix at nomega frequencies, and nqibz wavevectors
!!  npwc=number of plane waves
!!  nomega=number of frequencies (usually 2)
!!  nqibz=number of irreducible q-points
!!  omega(nomega)=frequencies
!!  omegaplasma=input variable or Drude plasma frequency
!!
!! OUTPUT
!!  bigomegatwsq(npwc,npwc,nqibz)=parameter of the plasmon-pole model (see gwa.pdf file)
!!  omegatw(npwc,npwc,nqibz)=parameter of the plasmon-pole model (see gwa.pdf file)
!!
!! PARENTS
!!      m_ppmodel
!!
!! CHILDREN
!!
!! SOURCE

subroutine cppm1par(npwc,nqibz,nomega,epsm1,omega,omegaplasma,&
& omegatw,bigomegatwsq) ! optional

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nomega,npwc,nqibz
 real(dp),intent(in) :: omegaplasma
!arrays
 complex(gwpc),target,intent(inout) :: epsm1(npwc,npwc,nomega,nqibz)
 complex(dpc),intent(in) :: omega(nomega)
 complex(gwpc),optional,target,intent(inout) :: bigomegatwsq(npwc,npwc,nqibz)
 complex(gwpc),optional,target,intent(inout) :: omegatw(npwc,npwc,nqibz)

!Local variables-------------------------------
!scalars
 integer :: ig,igp,io,io0,ioe0,iq
 real(dp) :: e0,minomega
 character(len=500) :: msg
 complex(gwpc) :: AA,omegatwsq,diff,ratio
 complex(gwpc) :: epsm1_io0,epsm1_ioe0
!arrays
 complex(gwpc),pointer :: bigomegatwsq_local(:,:,:),omegatw_local(:,:,:) 

! *************************************************************************

 DBG_ENTER("COLL")

 ! 
 ! === When bigomegatwsq and omegatw are not given, store the PPm parameters in place in epsm1
 ! This is dirty according to Matteo and therefore, should be activated only for developers.
 if (present(bigomegatwsq).and.present(omegatw)) then
   bigomegatwsq_local => bigomegatwsq
   omegatw_local      => omegatw
 else
   bigomegatwsq_local => epsm1(:,:,1,:)
   omegatw_local      => epsm1(:,:,2,:)
 end if
 !
 ! === Find omega=0 and omega=imag (closest to omegaplasma) where to fit ppm parameters ===
 minomega=1.0d-3 ; io0=0
 do io=1,nomega
   if (ABS(omega(io))<minomega) then
     io0=io; minomega=ABS(omega(io))
   end if
 end do
 ABI_CHECK(io0/=0,"omega=0 not found")

 minomega=1.0d-3 ; e0=200.0 ; ioe0=0
 do io=1,nomega
   if (REAL(omega(io))<minomega.and.AIMAG(omega(io))>minomega) then
     if (ABS(AIMAG(omega(io))-omegaplasma)<ABS(e0-omegaplasma)) then
       ioe0=io; e0=AIMAG(omega(io))
     end if
   end if
 end do

 write(msg,'(a,f9.4,a)')' Using omega_plasma = ',e0*Ha_eV,' [eV] '
 call wrtout(std_out,msg,'COLL')
 ABI_CHECK(ioe0/=0,"Imaginary omega not found")

 do iq=1,nqibz
   ! === Calculate plasmon-pole A parameter A=epsilon^-1(0)-delta ===
   do ig=1,npwc  
     do igp=1,npwc  

       epsm1_io0  = epsm1(ig,igp,io0,iq)
       epsm1_ioe0 = epsm1(ig,igp,ioe0,iq)

       AA=epsm1_io0
       if (ig==igp) AA=AA-one

       ! === Calculate plasmon-pole omega-twiddle-square parameter ===
       ! XG201009 Strangely, the next formula does not work with gcc43-debug 
       ! omegatwsq=(AA/(epsm1_io0-epsm1_ioe0)-one)*e0**2
       ! This seems to be due to precision issue at the level of division by a complex whose norm squared
       ! is below the smallest representable number.    
       ! After many trials, I have decided to shift the difference by a small number ... well, not so small ... 
       ! for numerical issues
       diff=epsm1_io0-epsm1_ioe0
       diff=diff+cmplx(tol10,tol10)
       ratio=AA/diff
       omegatwsq=(ratio-cone)*e0**2
       !
       ! If omega-twiddle-squared is negative,set omega-twiddle-squared to 1.0 (a reasonable way of treating
       ! such terms, in which epsilon**-1 was originally increasing along this part of the imaginary axis)
       ! (note: originally these terms were ignored in Sigma; this was changed on 6 March 1990.)

       if (REAL(omegatwsq)<=zero) omegatwsq=one
       !
       ! === Get omega-twiddle ===
       ! * Neglect the imag part (if one) in omega-twiddle-squared
       omegatw_local(ig,igp,iq)=SQRT(REAL(omegatwsq))
       !
       ! === Get big-omega-twiddle-squared=-omega-twiddle-squared AA ===
       bigomegatwsq_local(ig,igp,iq)=-AA*omegatw(ig,igp,iq)**2

!DEBUG
!     if(ig==1 .and. igp==1 .and. iq==1)then
!       write(std_out,*)' m_ppmodel cppm1par : '
!       write(std_out,*)' diff, AA, ratio, omegatwsq, e0 ',diff, AA, ratio, omegatwsq, e0
!     endif
!ENDDEBUG

     end do !igp
   end do !ig
 end do !iq

!DEBUG
! do ig=1,min(npwc,20)
!  do igp=1,min(npwc,20)
!   write(6,'(a,2i4,4f16.6)')' ig,igp,omegatw_local(ig,igp,iq),bigomegatwsq_local(ig,igp,iq)=',&
!&              ig,igp,omegatw_local(ig,igp,1),bigomegatwsq_local(ig,igp,1)
!  enddo
! enddo
!ENDDEBUG

 write(msg,'(2a,f15.12,2a,3i5,a)')ch10,&
&  ' cppm1par : omega twiddle minval [eV]  = ',MINVAL(ABS(omegatw))*Ha_eV,ch10,&
&  '            omega twiddle min location = ',MINLOC(ABS(omegatw)),ch10
 call wrtout(std_out,msg,'COLL')

 DBG_EXIT("COLL")

end subroutine cppm1par
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/cppm2par
!! NAME
!! cppm2par
!!
!! FUNCTION
!!  Calculate plasmon-pole parameters using Hybertsen and Louie model (PRB 34, 5390 (1986))
!!
!! INPUTS
!!  iqiA(optional)= the index in the IBZ of the q-point where the ppmodel parameters have to be evaluated
!!  nqiA=number of irreducible points asked usually nqiA=niqbz. 
!!   It should be set to 1 if a single q-point is required (optional argument iqiA is needed)
!!  epsm1(npwc,npwc,nomega,nqiA)=symmetrized inverse dielectric matrix at nomega frequencies, and nq wavevectors
!!  gmet(3,3)=metric in reciprocal space
!!  ngfftf(18)=contain all needed information about the 3D fine FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  npwc=number of plane waves in epsm1
!!  nomega=number of frequencies (usually 2 but this plasmon-pole requires only the static symmetrized dielectric matrices
!!  Qmesh<BZ_mesh_type>=datatype gathering information on the q point sampling. see defs_datatypes.F90
!!    %nqibz=number of q points
!!    %ibz(3,nqibz)=irred q-points
!!  omega(nomega)=frequencies
!!  rhor(nfftf)=charge density on the real space FFT grid
!!  nfftf= total number of points in the fine FFT mesh  (for this processor)
!!
!! OUTPUT
!!  bigomegatwsq(npwc,npwc,nqiA)= squared bare plasma frequencies
!!   \Omega^2_{G1 G2}(q) = 4\pi \frac {(q+G1).(q+G2)}/{|q+G1|^2} n(G1-G2)
!!
!!  omegatw(npwc,npwc,nqiA)= plasmon frequencies \tilde\omega_{G1 G2}(q) where:
!!  \tilde\omega^2_{G1 G2}(q) = 
!!    \frac {\Omega^2_{G1 G2}(q)} {\delta_{G1 G2}-\tilde\epsilon^{-1}_{G1 G2} (q, \omega=0)}
!!
!! PARENTS
!!      m_ppmodel
!!
!! CHILDREN
!!
!! SOURCE

subroutine cppm2par(paral_kgb,npwc,nqiA,nomega,epsm1,bigomegatwsq,omegatw,ngfftf,gvec,gprimd,rhor,nfftf,Qmesh,gmet,&
& iqiA) ! Optional

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_53_ffts
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nomega,npwc,nqiA,nfftf,paral_kgb
 integer,optional,intent(in) :: iqiA
 type(BZ_mesh_type),intent(in) :: Qmesh
!arrays
 integer,intent(in) :: gvec(3,npwc)
 integer,intent(in) :: ngfftf(18)
 real(dp),intent(in) :: gmet(3,3),gprimd(3,3)
 real(dp),intent(inout) :: rhor(nfftf)
 complex(gwpc),intent(in) :: epsm1(npwc,npwc,nomega,nqiA)
 complex(gwpc),intent(inout) :: bigomegatwsq(npwc,npwc,nqiA)
 complex(gwpc),intent(inout) :: omegatw(npwc,npwc,nqiA)

!Local variables-------------------------------
!scalars
 integer :: ig,igp,iq,istat,nimwp
 integer :: ngfft1,ngfft2,ngfft3
 real(dp) :: lambda,phi
 logical,parameter :: use_symmetrized=.TRUE.,check_imppf=.FALSE.
 logical :: ltest
 character(len=500) :: msg
 type(MPI_type) :: mpi_enreg
!arrays
 integer,allocatable :: igfft(:,:)
 real(dp),pointer :: qibz(:,:)
 real(dp),allocatable :: qratio(:,:,:),qplusg(:),rhog_dp(:,:)
 complex(gwpc),allocatable :: omegatwsq(:,:) 
 complex(gwpc),allocatable :: rhog(:),rhogg(:,:),temp(:,:)  !MG these should be double precision TODO
!*************************************************************************

 qibz => Qmesh%ibz(1:3,1:Qmesh%nibz)

 if (PRESENT(iqiA)) then 
   ABI_CHECK(nqiA==1,'nqiA must be 1')
   ltest=(iqiA>0.and.iqiA<=Qmesh%nibz)
   ABI_CHECK(ltest,'iqiA out of range')
   qibz => Qmesh%ibz(1:3,iqiA:iqiA)
 end if
 !
 ! === Calculate qratio(npwec,npvec,nqiA) = (q+G).(q+Gp)/|q+G|^2 ===
 allocate(qratio(npwc,npwc,nqiA),stat=istat) 
 if (istat/=0) then 
   MSG_ERROR("out-of-memory in qratio")
 end if

 call cqratio(npwc,gvec,nqiA,qibz,gmet,gprimd,qratio)
 !
 ! === Compute the density in G space rhor(R)--> rhog(G) ===
 allocate(rhog_dp(2,nfftf),rhog(nfftf))
 ngfft1=ngfftf(1) 
 ngfft2=ngfftf(2) 
 ngfft3=ngfftf(3)

 call fourdp(1,rhog_dp,rhor,-1,mpi_enreg,nfftf,ngfftf,paral_kgb,0)
 rhog(1:nfftf)=CMPLX(rhog_dp(1,1:nfftf),rhog_dp(2,1:nfftf))
 !
 ! Calculate the FFT index of each (G-Gp) vector and assign 
 ! the value of the correspondent density simultaneously
 allocate(igfft(npwc,npwc),rhogg(npwc,npwc))
 call cggfft(npwc,ngfft1,ngfft2,ngfft3,gvec,igfft)
 do ig=1,npwc
   do igp=1,npwc
     rhogg(ig,igp)=rhog(igfft(ig,igp))
   end do
 end do
 rhogg(:,:)=four_pi*rhogg(:,:)
 deallocate(igfft,rhog_dp,rhog)
 !
 ! Calculate GPP parameters
 ! unsymmetrized epsm1 -> epsm1=|q+Gp|/|q+G|*epsm1
 !
 allocate(qplusg(npwc),temp(npwc,npwc),omegatwsq(npwc,npwc),stat=istat)
 if (istat/=0) then 
   ABI_DIE('out of memory in qplusg')
 end if

 do iq=1,nqiA
   temp(:,:)=-epsm1(:,:,1,iq)
   ! 
   ! RS still not obvious for me whether one shall use the symmetrized inverse DM or the unsymmetrized one
   ! the default here is to use the symmetrized one, I must discuss this with XG
   ! 
   ! MG it turns out that using the symmetrized inverse DM in the plasmon-pole
   ! equations give the same results for the squared plasmon frequencies omegatwsq while the 
   ! squared bare plasma frequencies bigomegatwsq related to the symmetrized dielectric matrix 
   ! are obtained multipling by |q+G1|/|q+G2|   
   ! 
   if (.not.use_symmetrized) then
     call cmod_qpg(nqiA,iq,qibz,npwc,gvec,gprimd,qplusg) !MG TODO here take care of small q
     do ig=1,npwc
       do igp=1,npwc
         temp(ig,igp)=qplusg(igp)/qplusg(ig)*temp(ig,igp)
       end do
     end do
   end if

   nimwp=0
   do ig=1,npwc
     temp(ig,ig)=temp(ig,ig)+one
     do igp=1,npwc
       bigomegatwsq(ig,igp,iq)=rhogg(ig,igp)*qratio(ig,igp,iq)
       omegatwsq(ig,igp)=bigomegatwsq(ig,igp,iq)/temp(ig,igp)
       !   
       ! Set omegatw to any arbitrary number to avoid dealing with undefined numbers like (INF)
       ! simply ignore all cases of omegatw with imaginary values
       ! in principle these correspond to cases where the imaginary part of epsm1 does not have
       ! a well defined peak. The imaginary part of epsm1 in these cases oscillates  with a small amplitude
       ! since the amplitude A_GGpr=-pi/2*bigomegatwsq/omegatw, 
       ! it follows that bigomegatwsq shall be set to zero for these cases
       !   
       if ( REAL(omegatwsq(ig,igp))<= tol12 .or. AIMAG(omegatwsq(ig,igp))**2*tol12> REAL(omegatwsq(ig,igp))**2) then
         bigomegatwsq(ig,igp,iq)=(0.,0.)
         omegatw(ig,igp,iq)=(ten,0.)
         nimwp=nimwp+1
         if (check_imppf) then 
           write(msg,'(a,i3,2i8)')' imaginary plasmon frequency at : ',iq,ig,igp
           call wrtout(std_out,msg,'COLL')
         end if 
       else 
         ! this part has been added to deal with systems without inversion symmetry
         ! this new implementation gives the same results as the previous one if 
         ! omegatwsq is a pure real number and has the advantage of being an improved
         ! approach for systems without an inversion center.
         lambda=ABS(omegatwsq(ig,igp))
         phi=ATAN(AIMAG(omegatwsq(ig,igp))/REAL(omegatwsq(ig,igp)))
         omegatw(ig,igp,iq)=SQRT(lambda/COS(phi))
         bigomegatwsq(ig,igp,iq)=bigomegatwsq(ig,igp,iq)*(1.-(0.,1.)*TAN(phi))
         ! Uncomment the following line and comment the previous to restore the old version.
         !omegatw(ig,igp,iq)=sqrt(real(omegatwsq(ig,igp)))
       end if
     end do
   end do

   write(msg,'(a,3f12.6,a,i8,a,i8)')&
&    ' at q-point : ',qibz(:,iq),&
&    ' number of imaginary plasmonpole frequencies = ',nimwp,' / ',npwc**2
   call wrtout(std_out,msg,'COLL')

 end do !iq

 write(msg,'(2a,f12.8,2a,3i5)')ch10,&
&  ' cppm2par : omega twiddle minval [eV]  = ',MINVAL(ABS(omegatw))*Ha_eV,ch10,&
&  '            omega twiddle min location = ',MINLOC(ABS(omegatw))
 call wrtout(std_out,msg,'COLL')

 deallocate(omegatwsq,rhogg,temp,qplusg,qratio)

end subroutine cppm2par
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/cppm3par
!! NAME
!! cppm3par
!!
!! FUNCTION
!! Calculate the plasmon-pole parameters using the von Linden-Horsh model (PRB 37, 8351, 1988)
!! (see also Pag 22 of Quasiparticle Calculations in Solids. Aulbur et. al)  
!!
!! INPUTS
!!  iqiA(optional)= the index in the IBZ of the q-point where the ppmodel parameters have to be evaluated
!!  nqiA=number of irreducible points asked usually nqiA=niqbz, 
!!   It should be set to 1 if a single q-point is required (optional argument iqiA is needed)
!! epsm1(npwc,npwc,nomega,nqiA)= symmetrized inverse dielectric 
!!  matrix at nomega frequencies, and nqiA wavevectors
!! MPI_enreg=information about MPI parallelization
!! ngfftf(18)=contain all needed information about 3D fine FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!! npwc=number of plane waves in epsm1
!! qratio=(q+G1).(q+G2)/(|q+G1|.|q+G2|)
!! Qmesh<BZ_mesh_type>=datatype gathering information on the q point sampling. see defs_datatypes.F90
!!   %nqibz=number of irreducible q-points
!!   %qibz(3,nqibz)=irreducible q-points.
!! rho(nfftf)=charge density on the real space FFT grid
!! nfftf=number of points in the FFT grid (for this processor)
!! gvec(3,npwc)= G vectors in reduced coordinates
!!
!! OUTPUT
!!  omegatw(npwc,npwc,nqiA)= plasmon pole positions
!!  bigomegatwsq(npwc,npwc,nqiA)=(E_{q,ii}^{-1}-1)*omegatw
!!   where E^{-1} is the eigenvalue of the inverse dielectric matrix
!!  eigtot(npwc,npwc,nqiA)=the eigvectors of the symmetrized inverse dielectric matrix 
!!   (first index for G, second index for bands)
!!
!! PARENTS
!!      m_ppmodel
!!
!! CHILDREN
!!
!! SOURCE

subroutine cppm3par(paral_kgb,npwc,nqiA,nomega,epsm1,bigomegatwsq,omegatw,ngfftf,gvec,gprimd,rho,nfftf,eigtot,Qmesh,&
& iqiA) ! Optional


!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_53_ffts
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftf,nomega,npwc,nqiA,paral_kgb
 integer,intent(in),optional :: iqiA
 type(BZ_mesh_type),intent(in) :: Qmesh
!arrays
 integer,intent(in) :: gvec(3,npwc),ngfftf(18)
 real(dp),intent(in) :: gprimd(3,3)
 real(dp),intent(inout) :: rho(nfftf)
 complex(gwpc),intent(in) :: epsm1(npwc,npwc,nomega,nqiA)
 complex(gwpc),intent(inout) :: bigomegatwsq(npwc,1,nqiA),eigtot(npwc,npwc,nqiA)
 complex(gwpc),intent(inout) :: omegatw(npwc,1,nqiA)

!Local variables-------------------------------
!TODO these should be dp
!scalars
 integer :: idx,ierr,ig,igp,ii,iq,istat,jj,ngfft1,ngfft2,ngfft3
 real(dp) :: num,qpg_dot_qpgp
 complex(dpc) :: conjg_eig
 logical :: ltest
 character(len=500) :: msg
 type(MPI_type) :: MPI_enreg
!arrays
 integer,allocatable :: igfft(:,:)
 real(dp) :: b1(3),b2(3),b3(3),gppq(3),gpq(3)
 real(dp),allocatable :: eigval(:),qplusg(:),rhog_dp(:,:),zhpev2(:)
 real(dp),pointer :: qibz(:,:)
 complex(dpc),allocatable :: eigvec(:,:),matr(:),mm(:,:,:),rhog(:),rhogg(:,:)
 complex(dpc),allocatable :: zhpev1(:),zz(:,:)

!*************************************************************************

 if (PRESENT(iqiA)) then 
   ABI_CHECK(nqiA==1,'nqiA must be 1')
   ltest=(iqiA>0.and.iqiA<=Qmesh%nibz)
   ABI_CHECK(ltest,'iqiA out of range')
   !qibz => Qmesh%ibz(1:3,iqia:iqia)
   allocate(qibz(3,1))
   qibz = Qmesh%ibz(1:3,iqia:iqia)
   if (iqia==1) qibz(:,1)=GW_Q0_DEFAULT
 else 
   allocate(qibz(3,Qmesh%nibz))
   qibz(:,:) = Qmesh%ibz
   qibz(:,1) = GW_Q0_DEFAULT
   !qibz => Qmesh%ibz(1:3,1:Qmesh%nibz)
 end if

 b1=two_pi*gprimd(:,1)
 b2=two_pi*gprimd(:,2)
 b3=two_pi*gprimd(:,3)

 ngfft1=ngfftf(1) ; ngfft2=ngfftf(2) ; ngfft3=ngfftf(3)

 allocate(rhog_dp(2,nfftf),rhog(nfftf),stat=istat)      
 allocate(igfft(npwc,npwc),rhogg(npwc,npwc),stat=istat) 
 if (istat/=0) then 
   msg = "out of memory in igfft and rhogg"
   ABI_DIE(msg)
 end if
 !
 ! === Compute the density in G space rhog(r)--> rho(G) ===
 ! FIXME this has to be fixed, rho(G) should be passed instead of doing FFT for each q
 ! Moreover MPI_enreg is local ????? 
 call fourdp(1,rhog_dp,rho,-1,MPI_enreg,nfftf,ngfftf,paral_kgb,0)

 rhog(1:nfftf)=CMPLX(rhog_dp(1,1:nfftf),rhog_dp(2,1:nfftf))
 !
 ! Calculate the FFT index of each (G-Gp) vector and assign the value
 ! of the correspondent density simultaneously
 call cggfft(npwc,ngfft1,ngfft2,ngfft3,gvec,igfft)

 do ig=1,npwc
   do igp=1,npwc
     if (igfft(ig,igp)>nfftf) then
       MSG_BUG("Cannot find rho(G-Gpr)")
     end if
     rhogg(ig,igp)=rhog(igfft(ig,igp))
   end do
 end do
 !
 ! mm(G,Gp) = (q+G) \cdot (q+Gp) n(G-Gp)
 allocate(mm(npwc,npwc,nqiA),stat=istat) 
 if (istat/=0) then
   ABI_DIE('out of memory')
 end if

 do iq=1,nqiA
   do ig=1,npwc
     if (ALL(ABS(qibz(:,iq))<1.0e-3)) then
       ! To be discussed with Riad, here we should use the small q 
       ! to be consistent and consider the limit q-->0
       gpq(:)=gvec(:,ig)
     else
       gpq(:)=gvec(:,ig)+qibz(:,iq)
     end if
     do igp=1,npwc
       if (ALL(ABS(qibz(:,iq))<1.0e-3)) then
         gppq(:)=gvec(:,igp)
       else
         gppq(:)=gvec(:,igp)+qibz(:,iq)
       end if
       qpg_dot_qpgp=zero
       do ii=1,3
         qpg_dot_qpgp=qpg_dot_qpgp+&
&          ( gpq(1)*b1(ii) +gpq(2)*b2(ii) +gpq(3)*b3(ii))*&
&          (gppq(1)*b1(ii)+gppq(2)*b2(ii)+gppq(3)*b3(ii))
       end do
       mm(ig,igp,iq)=rhogg(ig,igp)*qpg_dot_qpgp
     end do !igp
   end do !ig
 end do !iq

 deallocate(rhog_dp,rhog,igfft)
 ! === Now we have rhogg,rho0 ===
 !
 ! Calculate the dielectric matrix eigenvalues and vectors
 ! Use only the static epsm1 i.e., only the w=0 part (eps(:,:,1,:))
 allocate(eigval(npwc),eigvec(npwc,npwc),stat=istat) ! eigenvalues and vectors of DM
 if (istat/=0) stop 'eigvec out of memory'
 allocate(zz(npwc,nqiA),stat=istat) ; if (istat/=0) stop 'zz of memory'
 zz(:,:)=czero
 allocate(qplusg(npwc))

 do iq=1,nqiA
   !
   ! Store the susceptibility matrix in upper mode before calling zhpev for each iq value
   allocate(matr(npwc*(npwc+1)/2),stat=istat) ; if(istat/=0) stop 'matr of memory'

   idx=1
   do ii=1,npwc
     do jj=1,ii
       matr(idx)=epsm1(jj,ii,1,iq); idx=idx+1
     end do
   end do

   allocate(zhpev2(3*npwc-2),zhpev1(2*npwc-1),stat=istat)
   if (istat/=0) stop 'zhpev1 of memory' ! working arrays for lapack
   call ZHPEV('V','U',npwc,matr,eigval,eigvec,npwc,zhpev1,zhpev2,ierr)
   deallocate(matr,zhpev2,zhpev1)

   if (ierr<0) then
     write (msg,'(2a,i4,a)')&
&      ' Failed to calculate the eigenvalues and eigenvectors of the dielectric matrix ',ch10,&
&      ierr*(-1),'-th argument in the matrix has an illegal value. '
     MSG_ERROR(msg)
   end if

   if (ierr>0) then
     write(msg,'(3a,i4,2a)')&
&      ' Failed to calculate the eigenvalues and eigenvectors of the dielectric matrix ',ch10,&
&      ' the algorithm failed to converge; ierr = ', ierr,ch10,&
&      ' off-diagonal elements of an intermediate tridiagonal form did not converge to zero. '
     MSG_ERROR(msg)
   end if
   !
   ! Calculate the PPM parameters and the eigenpotentials needed for 
   ! the calculation of the generalized overlap matrix
   ! Note: the eigenpotentials has to be calculated on the FFT (G-Gp) index
   !
   ! Save eigenvectors of \tilde\epsilon^{-1}
   ! MG well it is better to save \Theta otherwise 
   ! we have to calculare \Theta for each band, spin, k-point but oh well
   eigtot(:,:,iq)=eigvec(:,:)

   call cmod_qpg(nqiA,iq,qibz,npwc,gvec,gprimd,qplusg) !MG TODO here take care of small q
   !
   ! Basic Equation:
   ! 
   ! \Theta_{q,ii}(G)=\Psi_{q,ii}(G)/|q+G|
   ! where \Psi_{q,ii}(G) is the eigenvector of \tilde\epsilon^{-1} 

   ! \tilde\omega_{ii,q}^2= 4\pi (1-eigenval(ii,q))) 
   ! \sum_{G,Gp} \Theta^*_{q,ii}(G) (q+G)\cdot(q+Gp) n(G-Gp) \Theta_{q,ii}(Gp) 

   do ii=1,npwc !DM band
     ! Calculate \Theta_{q,ii}(G)
     ! why the first element is not modified? if the problem is the small value of qplusg(1)
     ! we could multiply by sqrt(mod((q+G)(q+G'))) and then add the sing at the end 
     if(iq==1)then
       eigvec(2:,ii)=eigvec(2:,ii)/qplusg(2:)
     else
       eigvec(:,ii)=eigvec(:,ii)/qplusg(:)
     end if
     do ig=1,npwc
       conjg_eig=CONJG(eigvec(ig,ii))
       do igp=1,npwc
         if(iq==1 .and. ig==1 .and. igp==1)then
           zz(ii,iq)=zz(ii,iq)+conjg_eig*rhogg(ig,igp)*eigvec(igp,ii)
         else
           zz(ii,iq)=zz(ii,iq)+conjg_eig*mm(ig,igp,iq)*eigvec(igp,ii)
         end if
       end do
     end do

     num=one-eigval(ii)
     if (num<=zero) then
!      here I think we should set bigomegatwsq=0 and omegatw to an arbitrary value
!      maybe we can output a warning TO BE discussed with Riad 
       if (ABS(num)<1.0d-4) then
         num=1.0d-5
       else
         msg = "Found one or more imaginary plasmon pole energies."
         MSG_ERROR(msg)
       end if
     end if

     omegatw(ii,1,iq)=SQRT(4*pi*REAL(zz(ii,iq))/num)
     ! this should be \alpha = 2\pi omegatw * (1-eigenval) 
     ! MG check this, in the review I found a factor 2\pi, maybe it is reintroduced later
     bigomegatwsq(ii,1,iq)=num*omegatw(ii,1,iq)
   end do
 end do !iq

 deallocate(rhogg,mm,eigval,zz,eigvec,qplusg)

 write(msg,'(2a,f12.8,2a,3i5)')ch10,&
&  ' cppm3par : omega twiddle minval [eV]  = ',MINVAL(ABS(omegatw))*Ha_eV,ch10,&
&  '            omega twiddle min location = ',MINLOC(ABS(omegatw))
 call wrtout(std_out,msg,'COLL')

 deallocate(qibz)

end subroutine cppm3par
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/cppm4par
!! NAME
!! cppm4par
!!
!! FUNCTION
!! Calculate the plasmon-pole parameters using Engel and Farid model (PRB47,15931,1993)
!! See also Quasiparticle Calculations in Solids, Aulbur et al. (pag. 23)
!!
!! INPUTS
!!  iqiA(optional)= the index in the IBZ of the q-point where the ppmodel parameters have to be evaluated
!!  nqiA=number of irreducible points asked usually nqiA=niqbz. 
!!   It should be set to 1 if a single q-point is required (optional argument iqiA is needed)
!!  epsm1(npwc,npwc,nomega,nqiA)=symmetrized inverse dielectric matrix at 
!!   nomega frequencies, and nqiA wavevectors
!!  gprimd(3,3)=dimensional primitive translations for reciprocal space ($\textrm{bohr}^{-1}$)
!!  ngfftf(18)=contain all needed information about 3D fine FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  npwc=number of plane waves in epsm1
!!  nomega=number of frequencies (usually 2 but this model requires only \omega=0)
!!  Qmesh<BZ_mesh_type>=datatype gathering information on the q point sampling. see defs_datatypes.F90
!!    %nqibz=number of qibz vectors in the IBZ
!!    %qibz(3,nqibz)=irreducible q-points.
!!  rho(nfftf)=charge density on the real space FFT grid
!!  gvec(3,npwc)=G vectors in reduced coordinated 
!!
!! OUTPUT
!!  bigomegatwsq(npwc,npwc,nqiA)=plasmon-pole strength
!!  omegatw(npwc,npwc,nqiA)=plasmon-pole frequencies
!!
!! PARENTS
!!      m_ppmodel
!!
!! CHILDREN
!!
!! SOURCE

subroutine cppm4par(paral_kgb,npwc,nqiA,epsm1,nomega,bigomegatwsq,omegatw,ngfftf,gvec,gprimd,rho,nfftf,Qmesh,&
& iqia) ! Optional

 use defs_basis

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_53_ffts
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfftf,nomega,npwc,nqiA,paral_kgb
 integer,intent(in),optional :: iqiA
 type(BZ_mesh_type),intent(in) :: Qmesh
!arrays
 integer,intent(in) :: gvec(3,npwc),ngfftf(18)
 real(dp),intent(in) :: gprimd(3,3)
 real(dp),intent(inout) :: rho(nfftf)
 complex(gwpc),intent(in) :: epsm1(npwc,npwc,nomega,nqiA)
 complex(gwpc),intent(inout) :: bigomegatwsq(npwc,npwc,nqiA),omegatw(npwc,1,nqiA)

!Local variables-------------------------------
!scalars
 !integer :: idx,jj 
 integer :: ierr,ig,igp,ii,iq,istat,lowork,ngfft1,ngfft2,ngfft3
 real(dp) :: qpg_dot_qpgp
 logical :: ltest
 character(len=500) :: msg
 character(len=80) :: bar
 type(MPI_type) :: MPI_enreg
!arrays
 integer,allocatable :: igfft(:,:)
 real(dp) :: b1(3),b2(3),b3(3),gppq(3),gpq(3)
 real(dp),allocatable :: eigval(:),qplusg(:),rhog_dp(:,:),rwork(:)
 !real(dp),allocatable :: eigval1(:),zhpev2(:)
 real(dp),pointer :: qibz(:,:)
 complex(dpc),allocatable :: chi(:,:,:),chitmp(:,:),chitmps(:,:),eigvec(:,:)
 !complex(dpc),allocatable :: matr(:),eigvec1(:,:),zhpev1(:)
 complex(dpc),allocatable :: mm(:,:,:),mtemp(:,:),rhog(:)
 complex(dpc),allocatable :: rhogg(:,:),tmp1(:),work(:),zz2(:,:)

!*************************************************************************

 DBG_ENTER("COLL")

 !FIXME It seems that q has to be non zero to avoid problems with LAPACK 
 ! Remove GW_Q0_DEFAULT should be passed in input, 
 ! Rewrite Everything so that the rouine works for a single q

 if (PRESENT(iqiA)) then 
   ABI_CHECK(nqiA==1,'nqiA must be 1')
   ltest=(iqiA>0.and.iqiA<=Qmesh%nibz)
   ABI_CHECK(ltest,'iqiA out of range')
   !qibz => Qmesh%ibz(1:3,iqia:iqia)
   allocate(qibz(3,1))
   qibz = Qmesh%ibz(1:3,iqia:iqia)
   if (iqiA==1) qibz(:,1)=GW_Q0_DEFAULT
 else 
   allocate(qibz(3,Qmesh%nibz))
   qibz(:,:) = Qmesh%ibz
   qibz(:,1) = GW_Q0_DEFAULT
   !qibz => Qmesh%ibz(1:3,1:Qmesh%nibz)
 end if

 b1=two_pi*gprimd(:,1)
 b2=two_pi*gprimd(:,2)
 b3=two_pi*gprimd(:,3)
 !
 ! === Calculate density in G space rhog(G) ===
 allocate(rhog_dp(2,nfftf),rhog(nfftf),stat=istat)      ; if (istat/=0) stop 'rhog_dp/rhog out of memory'
 allocate(igfft(npwc,npwc),rhogg(npwc,npwc),stat=istat) ; if (istat/=0) stop 'igfft/rhogg out of memory'
 !
 ! Conduct FFT tho(r)-->rhog(G)
 ! FIXME this has to be fixed, rho(G) should be passed instead of doing FFT for each q
 ! Moreover MPI_enreg is local ????? 
 call fourdp(1,rhog_dp,rho,-1,MPI_enreg,nfftf,ngfftf,paral_kgb,0)

 rhog(1:nfftf)=CMPLX(rhog_dp(1,1:nfftf),rhog_dp(2,1:nfftf))
 deallocate(rhog_dp)
 !
 ! Calculate the FFT index of each (G-Gp) vector and assign the value
 ! of the correspondent density simultaneously
 ngfft1=ngfftf(1) ; ngfft2=ngfftf(2) ; ngfft3=ngfftf(3)
 call cggfft(npwc,ngfft1,ngfft2,ngfft3,gvec,igfft)

 do ig=1,npwc
   do igp=1,npwc
     if (igfft(ig,igp)>nfftf) then
       ! well by definition igfft <= nfftf
       MSG_BUG("Cannot find rho(G-Gpr)")
     end if
     rhogg(ig,igp)=rhog(igfft(ig,igp))
   end do
 end do
 deallocate(igfft,rhog)
 !
 ! Now we have rhogg, calculate the M matrix (q+G1).(q+G2) n(G1-G2)
 allocate(mm(npwc,npwc,nqiA),stat=istat) ; if (istat/=0) stop 'mm out of memory'

 do iq=1,nqiA
   do ig=1,npwc
     gpq(:)=gvec(:,ig)+qibz(:,iq)
     do igp=1,npwc
       gppq(:)=gvec(:,igp)+qibz(:,iq)
       qpg_dot_qpgp=zero
       do ii=1,3
         qpg_dot_qpgp=qpg_dot_qpgp+&
&          ( gpq(1)*b1(ii) +gpq(2)*b2(ii) +gpq(3)*b3(ii))*&
&          (gppq(1)*b1(ii)+gppq(2)*b2(ii)+gppq(3)*b3(ii))
       end do
       mm(ig,igp,iq)=rhogg(ig,igp)*qpg_dot_qpgp
     end do !igp
   end do !ig
 end do !iq

 !MG TODO too much memory in chi, we can do all this stuff inside a loop
 allocate(chitmp(npwc,npwc),chi(npwc,npwc,nqiA),stat=istat) ; if (istat/=0) stop 'cppm4par out of memory'
 allocate(qplusg(npwc))
 !
 ! Extract the full polarizability from \tilde \epsilon^{-1}
 ! \tilde\epsilon^{-1}_{G1 G2} = \delta_{G1 G2} + 4\pi \frac{\chi_{G1 G2}}{|q+G1| |q+G2|}
 do iq=1,nqiA
   chitmp(:,:)=epsm1(:,:,1,iq)
   call cmod_qpg(nqiA,iq,qibz,npwc,gvec,gprimd,qplusg) !MG TODO here take care of small q
   do ig=1,npwc
     chitmp(ig,ig)=chitmp(ig,ig)-one
   end do
   do ig=1,npwc
     do igp=1,npwc
      chi(ig,igp,iq)=chitmp(ig,igp)*qplusg(ig)*qplusg(igp)/four_pi
     end do
   end do
 end do
 deallocate(chitmp)

!DEBUG
!do iq=1,nqiA
!allocate(eigval1(npwc),stat=istat)
!if(istat/=0) stop 'eigval1 out of memory'
!allocate(eigvec1(npwc,npwc),stat=istat)
!if(istat/=0) stop 'eigvec1 out of memory'
!allocate(matr(npwc*(npwc+1)/2))
!if(istat/=0) stop 'matr out of memory'
!allocate(zhpev2(3*npwc-2),stat=istat)
!if(istat/=0) stop 'zhpev2 of memory'
!allocate(zhpev1(2*npwc-1),stat=istat)
!if(istat/=0) stop 'zhpev1 of memory' ! woking arrays for lapack
!
!idx=1
!do ii=1,npwc
!do jj=1,ii
!matr(idx)=chi(jj,ii,iq)
!idx=idx+1
!end do
!end do
!
!call zhpev('v','u',npwc,matr,eigval1,eigvec1,npwc,&
!&   zhpev1,zhpev2,ierr)
!
!deallocate(zhpev1,zhpev2,matr,eigval1,eigvec1)
!end do
!ENDDEBUG
 !
 ! === Solve chi*X = Lambda M*X where Lambda=-1/em(q)**2 ===
 do iq=1,nqiA

   allocate(eigval(npwc),eigvec(npwc,npwc),stat=istat)
   allocate(mtemp(npwc,npwc),chitmps(npwc,npwc),stat=istat) ! temp working arrays
   allocate(work(2*npwc-1),rwork(3*npwc-2),stat=istat) 
   if(istat/=0) stop 'rwork out of memory'
   !
   ! Copy chi and mm into working arrays
   chitmps(:,:)=chi(:,:,iq)
   mtemp(:,:)=mm(:,:,iq)
   lowork=2*npwc-1

   call ZHEGV(1,'V','U',npwc,chitmps,npwc,mtemp,npwc,eigval,work,lowork,rwork,ierr)
   
   if (ierr/=0) then 
     write(msg,'(a,i3)')' ZHEGV reported info = ',ierr
     MSG_ERROR(msg)
   end if 
   !
   ! Eigenvectors are normalized as : X_i^* M X_j = \delta_{ij}    
   eigvec(:,:)=chitmps(:,:)
   deallocate(mtemp,chitmps,work,rwork) 
   !
   ! === Calculate the plasmon pole parameters ===
   allocate(tmp1(npwc),stat=istat) !eigenvectors
   if (istat/=0) stop 'tmp1 out of memory'
   allocate(zz2(npwc,npwc),stat=istat) ! checking
   if (istat/=0) stop 'zz out of memory'
   !
   ! good check:
   ! the lowest plasmon energy on gamma should be
   ! close to experimental plasma energy within an error of 10 %
   ! this error can be reduced further if one includes the non local
   ! commutators in the calculation of the polarizability at q==0
   zz2(:,:)=(0.0,0.0)
   call cmod_qpg(nqiA,iq,qibz,npwc,gvec,gprimd,qplusg) !MG TODO here take care of small q

   do ii=1,npwc
     !
     ! keeping in mind that the above matrix is negative definite
     ! we might have a small problem with the eigval that correspond to large G vectors
     ! i.e. DM band index, where the eigevalues become very small with
     ! possibility of being small positive numbers (due to numerical problems)
     ! thus as a caution one can use the following condition
     ! this will not affect the result since such a huge plasmon energy give almost zero
     ! contribution to the self correlation energy

     if (eigval(ii)>=zero) then
       eigval(ii) = -1.0d-4 
       if (eigval(ii)>1.0d-3) then
         eigval(ii) = -1.0d-22
         write(msg,'(a,i6,a,es16.6)')&
&          ' Imaginary plasmon pole eigenenergy, eigenvector number ',ii,' with eigval',eigval(ii),ch10
         MSG_ERROR(msg)
       end if
     end if
     !
     ! === Save plasmon energies ===
     omegatw(ii,1,iq)=SQRT(-1/eigval(ii))
     !
     ! Calculate and save scaled plasmon-pole eigenvectors 
     ! defined as \sqrt{4\pi} \frac{Mx}{\sqrt{\tilde\omega} |q+G|}
     tmp1(:)=eigvec(:,ii)

     do ig=1,npwc
       do igp=1,npwc
         zz2(ig,ii)=zz2(ig,ii)+mm(ig,igp,iq)*tmp1(igp) ! z--->y
       end do
       bigomegatwsq(ig,ii,iq)=SQRT(four_pi)*zz2(ig,ii)/SQRT(omegatw(ii,1,iq))
       bigomegatwsq(ig,ii,iq)=bigomegatwsq(ig,ii,iq)/qplusg(ig)
     end do

   end do
   deallocate(tmp1,eigvec,eigval,zz2)
 end do !iq

 deallocate(qplusg,chi,rhogg,mm)

 bar=REPEAT('-',80)
 write(msg,'(3a)')bar,ch10,' plasmon energies vs q vector shown for lowest 10 bands                 '
 call wrtout(ab_out,msg,'COLL')
 do iq=1,nqiA
   write(msg,'(2x,i3,5x,10f7.3)')iq,(REAL(omegatw(ig,1,iq))*Ha_eV,ig=1,10)
   call wrtout(ab_out,msg,'COLL')
 end do
 write(msg,'(a)')bar
 call wrtout(ab_out,msg,'COLL')

 write(msg,'(2a,f12.8,2a,3i5)')ch10,&
&  ' cppm4par : omega twiddle minval [eV]  = ',MINVAL(ABS(omegatw))*Ha_eV,ch10,&
&  '            omega twiddle min location = ',MINLOC(ABS(omegatw))
 call wrtout(std_out,msg,'COLL')

 deallocate(qibz)

 DBG_EXIT("COLL")

end subroutine cppm4par
!!***

!----------------------------------------------------------------------

!!****f* m_ppmodel/cqratio
!! NAME
!! cqratio
!!
!! FUNCTION
!!  Calculate qratio(G,Gp,q)= (q+G)\cdot(q+Gp) / |q+G|^2 needed for Hybertsen-Louie and Plasmonpole model
!!
!! INPUTS
!!  gmet(3,3)=metric in reciprocal space
!!  gprimd(3,3)=reciprocal lattice vectors
!!  gvec(3,npwc)=reduced coordinates of the plane waves 
!!  npwc=number of planewaves considered (used for the correlation part)
!!  nq=number of q points
!!  q(3,nq)=coordinates of q points
!!
!! OUTPUT
!!  qratio(npwc,npwc,nq)=(q+G).(q+Gp) needed for Hybertsen-Louie and 
!!  von der Linden-Horsh plasmonpole models
!!
!! PARENTS
!!      m_ppmodel
!!
!! CHILDREN
!!
!! SOURCE

subroutine cqratio(npwc,gvec,nq,q,gmet,gprimd,qratio)

 use defs_basis

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: npwc,nq
!arrays
 integer,intent(in) :: gvec(3,npwc)
 real(dp),intent(in) :: gmet(3,3),gprimd(3,3),q(3,nq)
 real(dp),intent(out) :: qratio(npwc,npwc,nq)

!Local variables ------------------------------
!scalars
 integer :: ig,igp,ii,iq
 real(dp) :: qpg_dot_qpgp
!arrays
 real(dp) :: b1(3),b2(3),b3(3),gppq(3),gpq(3),norm(npwc)

!************************************************************************

 b1=two_pi*gprimd(:,1)
 b2=two_pi*gprimd(:,2)
 b3=two_pi*gprimd(:,3)

 norm(:)=zero ; qratio(:,:,:)=zero

 !this loops have to be rewritten!!!!
 do iq=1,nq
   do ig=1,npwc
     gpq(:)=gvec(:,ig)+q(:,iq)
     norm(ig)=two_pi*SQRT(DOT_PRODUCT(gpq,MATMUL(gmet,gpq)))
!    norm(ig)=normv(gpq,gmet,'g')
   end do
   do ig=1,npwc
     gpq(:)=gvec(:,ig)+q(:,iq)
     do igp=1,npwc
       gppq(:)=gvec(:,igp)+q(:,iq)
       qpg_dot_qpgp=zero
!      qpg_dot_qpgp=vdotw(gpq,gppq,gmet,'g')
       do ii=1,3
         qpg_dot_qpgp=qpg_dot_qpgp+&
&          ( gpq(1)*b1(ii) +  gpq(2)*b2(ii) + gpq(3)*b3(ii))*&
&          (gppq(1)*b1(ii) + gppq(2)*b2(ii) +gppq(3)*b3(ii))
       end do
!      
!      Now calculate qratio = (q+G).(q+Gp)/|q+G|^2 
!      when |q+G|^2 and (q+G).(q+Gp) are both zero
!      set (q+G).(q+Gp)/|q+G|^2 = 1
!      when |q+G|^2 is zero and |q+Gp| is not zero
!      set (q+G).(q+Gp)/|q+G|^2 = 0
!      
       if (norm(ig)<0.001) then
         if (norm(igp)<0.001) then
!          Case q=0 and G=Gp=0
           qratio(ig,igp,iq)=one
         else
!          Case q=0 and G=0 and Gp !=0
           qratio(ig,igp,iq)=zero
         end if
       else if (norm(igp)<0.001) then
!        Case q=0 and G= !0 and Gp=0
         qratio(ig,igp,iq)=zero
       else
         qratio(ig,igp,iq)=qpg_dot_qpgp/norm(ig)**2
       end if

     end do
   end do
 end do 

end subroutine cqratio
!!***

!----------------------------------------------------------------------

END MODULE m_ppmodel
!!***
