!{\src2tex{textfont=tt}}
!!****f* ABINIT/setup_bse
!! NAME
!!  setup_bse
!!
!! FUNCTION
!!  This routine performs the initialization of basic objects and quantities used for Bethe-Salpeter calculations.
!!  In particular the excparam data type that defines the parameters of the calculation is completely
!!  initialized starting from the content of Dtset and the parameters read from the external KSS and SCR (SUSC) file.
!!
!! COPYRIGHT
!! Copyright (C) 1992-2009 EXC group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida)
!! Copyright (C) 2009-2010 ABINIT group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida, M.Giantomassi)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! codvsn=Code version
!! ngfft_gw(18)=Information about 3D FFT for density and potentials, see ~abinit/doc/input_variables/vargs.htm#ngfft
!! acell(3)=Length scales of primitive translations (bohr)
!! rprim(3,3)=Dimensionless real space primitive translations.
!! Dtset<dataset_type>=All input variables for this dataset.
!!  Some of them might be redefined here TODO
!! Dtfil=filenames and unit numbers used in abinit.
!! Psps <type(pseudopotential_type)>=variables related to pseudopotentials
!! Pawtab(Psps%ntypat*Dtset%usepaw)<type(pawtab_type)>=PAW tabulated starting data
!!
!! OUTPUT
!! Er<type(Epsilonm1_results)=Datatype gathering info, dimensions and data for the symmetrized inverse dielectric matrix. 
!! Cryst<type(crystal_structure)=Info on the crystalline Structure. 
!! Kmesh<type(BZ_mesh_type)=Structure defining the k-sampling for the wavefunctions.
!! Qmesh<type(BZ_mesh_type)=Structure defining the q-sampling for the symmetrized inverse dielectric matrix.
!! Gsph_Max<type(Gvectors_type)=Data type gathering info on the G-sphere for wave functions and e^{-1}, 
!! KS_BSt<type(Bandstructure_type)=The KS band structure (energies, occupancies, k-weights...)
!! Vcp<type(Coulombian_type)>=Structure gathering information on the Coulomb interaction in reciprocal space,
!!   including a possible cutoff in real space.
!! transtab(:,:,:)<POINTER>=TAble mapping (k,v,c) onto the sequential index of the transition. Allocated inside the routine.
!! ngfft_osc(18)=Contain all needed information about the 3D FFT for the oscillator matrix elements.
!!   See ~abinit/doc/input_variables/vargs.htm#ngfft
!! gwenergy(:,:,:)<POINTER>=The (complex) GW energies either read from file or obtained via the scissors operator.
!!   Allocated inside the routine.
!! Trans(:)<type(transition)><POINTER>=For each transition, the index of the k-point, band indeces and the transition energy.
!!   Allocated inside the routine.
!! gw_gap=The value of the GW gap.
!! Bsp<type(excparam)>=Basic parameters defining the Bethe-Salpeter run. Completely initialed in output.
!! Hdr_kss<type(Hdr_type)>=The header of the KSS file.
!! Hdr_kss<type(Hdr_type)>=Local header initialized from the parameters used for the Bethe-Salpeter calculation.
!! BS_files<excfiles>=Files used in the calculation.
!!
!! PARENTS
!!      bethe_salpeter
!!
!! CHILDREN
!!      bstruct_clean,bstruct_init,copy_bandstructure,get_bz_item,get_ng0sh
!!      hdr_check,hdr_init,hdr_update,hdr_vs_dtset,init_crystal_from_hdr
!!      init_er_from_file,init_gvectors_type,init_kmesh,init_transitions
!!      initmpi_seq,matrginv,metric,mkrdim,nullify_bs_parameters
!!      nullify_bz_mesh,print_bandstructure,print_bs_files,print_bs_parameters
!!      print_bz_mesh,print_crystal,print_gvectors,print_ngfft,print_scrhdr
!!      rdgw,rhoij_alloc,rhoij_copy,rhoij_free,setmesh,setup_coulombian,testkss
!!      wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine setup_bse(codvsn,acell,rprim,ngfftf,ngfft_osc,Dtset,Dtfil,BS_files,Psps,Pawtab,BSp,&
& Cryst,Kmesh,Qmesh,KS_BSt,gwenergy,Hdr_kss,Gsph_Max,Vcp,Hdr_bse,Er,Trans,transtab,gw_gap,spaceComm)

 use defs_basis
 use m_bs_defs
 use defs_datatypes
 use defs_abitypes
 use defs_gw
 use m_errors

 use m_gwdefs,        only : GW_Q0_DEFAULT
 use m_fstrings,      only : toupper
 use m_io_tools,      only : existent_file
 use m_geometry,      only : normv
 use m_header,        only : hdr_init, hdr_get_nelect_byocc
 use m_crystal,       only : print_crystal, idx_spatial_inversion, crystal_structure !, isymmorphic, 
 use m_crystal_io,    only : init_crystal_from_hdr
 use m_bz_mesh,       only : bz_mesh_type, init_kmesh, nullify_BZ_mesh, get_ng0sh, print_BZ_mesh, get_BZ_item
 use m_ebands,        only : bstruct_init, print_bandstructure, copy_bandstructure, bstruct_clean
 use m_coulombian,    only : coulombian_type, setup_coulombian
 use m_fft_mesh,      only : setmesh, print_ngfft
 use m_gsphere,       only : gvectors_type, init_gvectors_type, print_gvectors
 use m_screening,     only : init_er_from_file
 use m_io_screening,  only : print_scrhdr
 use m_io_kss,        only : testkss
 use m_qparticles,    only : rdgw

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
 use interfaces_42_geometry
 use interfaces_51_manage_mpi
 use interfaces_53_abiutil
 use interfaces_62_iowfdenpot
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: spaceComm
 complex(dpc),intent(out) :: gw_gap
 character(len=6),intent(in) :: codvsn
 type(dataset_type),intent(inout) :: Dtset 
 type(datafiles_type),intent(in) :: Dtfil
 type(pseudopotential_type),intent(in) :: Psps
 type(excparam),intent(inout) :: Bsp
 type(hdr_type),intent(out) :: Hdr_kss,Hdr_bse
 type(Epsilonm1_results),intent(out) :: Er
 type(crystal_structure),intent(out) :: Cryst
 type(BZ_mesh_type),intent(out) :: Kmesh,Qmesh
 type(Gvectors_type),intent(out) :: Gsph_Max
 type(Bandstructure_type),intent(out) :: KS_BSt
 type(Pawtab_type),intent(in) :: Pawtab(Psps%ntypat*Dtset%usepaw)
 type(Coulombian_type),intent(out) :: Vcp
 type(excfiles),intent(out) :: BS_files
!arrays
 integer,pointer :: transtab(:,:,:)
 integer,intent(in) :: ngfftf(18)
 integer,intent(out) :: ngfft_osc(18)
 real(dp),intent(in) :: acell(3),rprim(3,3)
 complex(dpc),pointer :: gwenergy(:,:,:)
 type(transition),pointer :: Trans(:)

!Local variables ------------------------------
!scalars
 integer :: bantot,enforce_sym,ib,ibtot,ik_ibz,isppol,itypat,jj,method
 integer :: mpsang_kss,nbnds_kss,ng_kss,nsym_kss,io,istat
 integer :: nfftot_osc
 integer :: nqlwl,iq
 integer :: stripewidth
 integer :: ish,jsh, osl
 integer :: pertcase_,restart,restartpaw,timrev
 integer :: iqbz,isym,iq_ibz,itim,ig
 integer :: mg0sh,mqmem
 real(dp) :: ucvol,nelr,qnorm
 real(dp) :: tpula,mvec1,mvec2
 logical :: ltest,only_one_kpt,remove_inv
 logical :: linv,linvk,shiftk
 logical :: en_order
 character(len=500) :: msg
 character(len=fnlen) :: em1file
 character(len=fnlen) :: gwfile
 type(Bandstructure_type) :: QP_BSt
 type(MPI_type) :: MPI_enreg_seq
!arrays
 integer :: ng0sh_opt(3)
 integer,allocatable :: npwarr(:),nlmn(:)
 integer,allocatable :: shlimtmp(:) 
 integer,pointer :: gvec_p(:,:)
 real(dp) :: gg(3),qpt_bz(3)
 real(dp) :: gmet(3,3),gprimd(3,3),rmet(3,3),rprimd(3,3),sq(3)
 real(dp) :: qred2cart(3,3),qcart2red(3,3)
 real(dp),allocatable :: doccde(:),eigen(:),occfact(:),qlwl(:,:)
 real(dp),allocatable :: igwene(:,:,:)
 real(dp),pointer :: energies_p(:,:,:)
 type(Pawrhoij_type),allocatable :: Pawrhoij(:)

!************************************************************************

 DBG_ENTER("COLL")

 call nullify_bs_parameters(BSp)

 ! === Check for calculations that are not implemented ===
 ltest=ALL(Dtset%nband(1:Dtset%nkpt*Dtset%nsppol)==Dtset%nband(1))
 ABI_CHECK(ltest,'Dtset%nband must be constant')
 ABI_CHECK(Dtset%nsppol==1,"nsppol==2 not implemented")
 ABI_CHECK(Dtset%nspinor==1,"nspinor==2 not coded")

 !* Fake MPI_type for the sequential part.
 call initmpi_seq(MPI_enreg_seq)

 ! === Dimensional primitive translations rprimd (from input), gprimd, metrics and unit cell volume ===
 call mkrdim(acell,rprim,rprimd)  
 call metric(gmet,gprimd,-1,rmet,rprimd,ucvol) 

 ! === Define consistently npw, nsh, and ecut for wavefunctions and W ===
 !call setshells(Dtset%ecutwfn,Dtset%npwwfn,Dtset%nshwfn,Dtset%nsym,gmet,gprimd,Dtset%symrel,'wfn',ucvol)
 !call setshells(Dtset%ecuteps,Dtset%npweps,Dtset%nsheps,Dtset%nsym,gmet,gprimd,Dtset%symrel,'eps',ucvol)

 call testkss(Dtfil%fnameabi_kss,Dtset%accesswff,Dtset%localrdwf,nsym_kss,nbnds_kss,&
&  ng_kss,mpsang_kss,gvec_p,energies_p,Hdr_kss,spaceComm) 

 nelr = hdr_get_nelect_byocc(Hdr_kss)

 ! === Create crystal_structure data type ===
 remove_inv=(nsym_kss/=Hdr_kss%nsym) 
 timrev=  2 ! This information is not reported in the header
            ! 1 => do not use time-reversal symmetry 
            ! 2 => take advantage of time-reversal symmetry
                                                            
 call init_crystal_from_hdr(Cryst,Hdr_kss,timrev,remove_inv)
 call print_crystal(Cryst)

! here we reorder the shell, because most likely are badly written by abinit
 call wrtout(std_out,'reordering shells',"COLL")

 allocate(shlimtmp(ng_kss))
 do ig=1,ng_kss
   shlimtmp(ig)=ig
 end do

 jsh   = 1
 mvec1 = zero
 do ish=2,ng_kss
   osl = shlimtmp(ish-1) + 1
   gg(:)  = gvec_p(:, osl)
   mvec2 = normv(gvec_p(:,osl),Cryst%gmet,"G")
!  print *, ish, mvec2, gg

   if (abs(mvec2 - mvec1) > tol5) then
     jsh=jsh + 1
     mvec1 = mvec2
   end if
    
   shlimtmp(jsh) = shlimtmp(ish)
 end do
 BSp%nsh = jsh

 allocate(BSp%shlim(BSp%nsh))
 BSp%shlim(:)=shlimtmp(1:BSp%nsh)
 deallocate(shlimtmp)
 
 ltest=(Psps%mpsang==mpsang_kss) 
 ABI_CHECK(ltest,'Psps%mpsang/=mpsang_kss')

 call hdr_vs_dtset(Hdr_kss,Dtset) 

 ! ==== Set up of the k-points and tables in the whole BZ ===
 call nullify_BZ_mesh(Kmesh)
 call init_kmesh(Kmesh,Cryst,Hdr_kss%nkpt,Hdr_kss%kptns,Dtset%kptopt)

 call print_BZ_mesh(Kmesh,"K-mesh for the wavefunctions",std_out,Dtset%prtvol,"COLL")
 call print_BZ_mesh(Kmesh,"K-mesh for the wavefunctions",ab_out, 0,           "COLL")

 if (Dtset%getscr/=0.or.Dtset%irdscr/=0) then
   em1file=Dtfil%fnameabi_scr
 else if (Dtset%getsuscep/=0.or.Dtset%irdsuscep/=0) then
   em1file=Dtfil%fnameabi_sus
   MSG_ERROR("(get|ird)suscep not implemented")
 else 
   em1file=Dtfil%fnameabi_scr
   msg="you should not be here, inconsistency in getscr, irdscr, irdsuscep , getsuscep"
   MSG_ERROR(msg)
   !FIXME this has to be cleaned, in tgw2_3 Dtset%get* and Dtset%ird* are  not defined
 end if

 ! Initialize Er
 mqmem=1  !TODO out-of-core solution not implemented yet.
 call init_Er_from_file(Er,em1file,mqmem,Dtset%npweps,Dtset%accesswff,Dtset%localrdwf,spaceComm)

 call print_ScrHdr(Er%Hscr,prtvol=Dtset%prtvol)

 Dtset%npweps=Er%npwe ! Have to change %npweps if it was larger than dim on disk.

 ! === Setup of q-mesh in the whole BZ ===
 ! * Stop if a nonzero umklapp is needed to reconstruct the BZ. In this case, indeed, 
 !   epsilon^-1(Sq) should be symmetrized in csigme using a different expression (G-G_o is needed)

 call nullify_BZ_mesh(Qmesh)
 call init_kmesh(Qmesh,Cryst,Er%nqibz,Er%qibz,Dtset%kptopt)

 call print_BZ_mesh(Qmesh,"Q-mesh for the screening function",std_out,Dtset%prtvol,"COLL")
 call print_BZ_mesh(Qmesh,"Q-mesh for the screening function",ab_out ,0           ,"COLL")

 do iqbz=1,Qmesh%nbz
   call get_BZ_item(Qmesh,iqbz,qpt_bz,iq_ibz,isym,itim)
   sq = (3-2*itim)*MATMUL(Cryst%symrec(:,:,isym),Qmesh%ibz(:,iq_ibz))
   if (ANY(ABS(Qmesh%bz(:,iqbz)-sq )>1.0d-4)) then
     write(std_out,*) sq,Qmesh%bz(:,iqbz) 
     write(msg,'(a,3f6.3,a,3f6.3,2a,9i3,a,i2,2a)')&
&      ' qpoint ',Qmesh%bz(:,iqbz),' is the symmetric of ',Qmesh%ibz(:,iq_ibz),ch10,&
&      ' through operation ',Cryst%symrec(:,:,isym),' and itim ',itim,ch10,&
&      ' however a non zero umklapp G_o vector is required and this is not yet allowed'
     MSG_ERROR(msg)
   end if
 end do 

 ! Define the algorithm for solving the BSE.
 BSp%use_haydock = (Dtset%bs_algorithm == 2)
 if (BSp%use_haydock) then 
   BSp%niter       = Dtset%bs_haydock_niter
   BSp%haydock_tol = Dtset%bs_haydock_tol
 else
   BSp%niter       = 0
   BSp%haydock_tol = HUGE(one)
 end if

 ! Setup of the frequency mesh for the absorption spectrum.
 BSp%omegai = Dtset%bs_freq_mesh(1)
 BSp%omegae = Dtset%bs_freq_mesh(2)
 BSp%domega = Dtset%bs_freq_mesh(3)
 BSp%nomega = (BSp%omegae - BSp%omegai)/BSp%domega + 1
 !?BSp%broad  = Dtset%bs_broad
 BSp%broad  = Dtset%zcut

 ! The frequency mesh.
 allocate(BSp%omega(BSp%nomega))
 do io=1,BSp%nomega
   BSp%omega(io) = (BSp%omegai + (io-1)*BSp%domega)
 end do

 ! Possible cutoff on the transitions.
 BSp%ircut     = Dtset%bs_eh_cutoff(1)
 BSp%uvcut     = Dtset%bs_eh_cutoff(2)
 BSp%stripecut = Dtset%bs_eh_cutoff(3) 

 ! Shall we include non-local field effects?
 SELECT CASE (Dtset%bs_exchange_term)
 CASE (0)
   BSp%lftype = "NLF"
 CASE (1)
   BSp%lftype = "LF"
 CASE DEFAULT 
   write(msg,'(a,i0)')" Wrong bs_exchange_term: ",Dtset%bs_exchange_term
   MSG_ERROR(msg)
 END SELECT

 SELECT CASE (Dtset%bs_calctype)
 CASE (1)
   BSp%type = "RPA"
 CASE (2)
   BSp%type = "GW"
 CASE (3)
   BSp%type = "EXC"
 CASE DEFAULT
   write(msg,'(a,i3)')" Wrong value for bs_calctype: ",Dtset%bs_calctype
   MSG_ERROR(msg)
 END SELECT

 ! Treatment of the off-diagonal coupling block.
 SELECT CASE (Dtset%bs_coupling)
 CASE (0)
   BSp%htype = "RESONANT"
 CASE (1)
   BSp%htype = "COUPLING"
 CASE DEFAULT
   write(msg,'(a,i0)')" Wrong bs_coupling: ",Dtset%bs_coupling
   MSG_ERROR(msg)
 END SELECT 

 ! Treatment of the Coulomb contribution.
 SELECT CASE (Dtset%bs_coulomb_term)
 CASE (0)
   BSp%wtype = "WDIAG"
 CASE (1)
   BSp%wtype = 'WFULL'
 CASE (2)
   msg= " Model dielectric function not yet coded"
   MSG_ERROR(msg)
   !$BSp%wtype = 'wmodel' 
 CASE DEFAULT
   write(msg,'(a,i0)')" Wrong bs_coulomb_term: ",Dtset%bs_coulomb_term
   MSG_ERROR(msg)
 END SELECT

 ! Dimensions and parameters of the calculation.
 ! TODO one should add npwx as well
 BSp%npweps=Dtset%npweps
 BSp%npwwfn=Dtset%npwwfn

 BSp%nel   = Dtset%nelect  
 BSp%lomo  = Dtset%bs_eh_basis_set(1)
 
 if (Bsp%lomo > Bsp%nel/2) then ! correct only for unpolarized semiconductors 
   write(msg,'(a,i0,a,f8.3)') " Bsp%lomo = ",Bsp%lomo," cannot be greater than nelec/2 = ",Bsp%nel/2
   MSG_ERROR(msg)
 end if

 BSp%nbnds = -1
 BSp%nkibz = -1 
 BSp%nop   = Dtset%nsym

 shiftk = .FALSE.
 if (shiftk) then
   BSp%ninvk = 1
   BSp%nopk = 1
 else
   BSp%inv = .TRUE.
   BSp%ninv = 2
   if(.not.BSp%inv) BSp%ninv = 1
   BSp%nopk = -1
   BSp%ninvk = 2
 end if
 !
 ! ==============================================
 ! ==== Setup of the q for the optical limit ====
 ! ==============================================
 qred2cart = two_pi*Cryst%gprimd 
 qcart2red = qred2cart
 call matrginv(qcart2red,3,3)

 tpula = sqrt(2.0) * sqrt(Cryst%rprimd(1,1)**2+Cryst%rprimd(2,1)**2+Cryst%rprimd(3,1)**2)
 tpula = two_pi / tpula

 if (Dtset%gw_nqlwl==0) then
   BSp%nq = 6
   BSp%q(:,1) = (/one,zero,zero/)  ! (100)
   BSp%q(:,2) = (/zero,one,zero/)  ! (010)
   BSp%q(:,3) = (/zero,zero,one/)  ! (001)
   BSp%qcc(:,1) = MATMUL(qred2cart,BSp%q(:,1))
   BSp%qcc(:,2) = MATMUL(qred2cart,BSp%q(:,2))
   BSp%qcc(:,3) = MATMUL(qred2cart,BSp%q(:,3))

   BSp%qcc(1,4) = 1.0 ; BSp%qcc(2,4) = 0.0 ; BSp%qcc(3,4) = 0.0 ! (x)
   BSp%qcc(1,5) = 0.0 ; BSp%qcc(2,5) = 1.0 ; BSp%qcc(3,5) = 0.0 ! (y)
   BSp%qcc(1,6) = 0.0 ; BSp%qcc(2,6) = 0.0 ; BSp%qcc(3,6) = 1.0 ! (z)
   BSp%q(:,4) = MATMUL(qcart2red,BSp%qcc(:,4))
   BSp%q(:,5) = MATMUL(qcart2red,BSp%qcc(:,5))
   BSp%q(:,6) = MATMUL(qcart2red,BSp%qcc(:,6))
   
   do iq=1,BSp%nq ! normalization on cartesian coordinates polarization vectors
    qnorm = SQRT(DOT_PRODUCT(BSp%qcc(:,iq),BSp%qcc(:,iq)))
    BSp%qcc(:,iq) = BSp%qcc(:,iq) / qnorm
    BSp%q(:,iq) = MATMUL(qcart2red,BSp%qcc(:,iq))
   end do

   BSp%qcc2pioa(:,:) = BSp%qcc(:,:) / tpula

 else
   MSG_ERROR("not coded") !TODO
   BSp%nq = Dtset%gw_nqlwl
   BSp%qcc(:,1) = MATMUL(qred2cart,BSp%q(:,1))
!  normalization on cartesian coordinates polarization vectors
   qnorm = SQRT(DOT_PRODUCT(BSp%qcc(:,1),BSp%qcc(:,1)))
   BSp%qcc(:,1) = BSp%qcc(:,1) / qnorm
   BSp%q(:,1) = MATMUL(qcart2red,BSp%qcc(:,1))
   BSp%qcc2pioa(:,1) = BSp%qcc(:,1) / tpula
 end if

 ! Shall we use the scissors operator to open the gap?
 BSp%soshift=.FALSE.; BSp%soenergy = zero
 if (ABS(Dtset%soenergy)>tol6) then
   BSp%soshift = .TRUE.
   BSp%soenergy = Dtset%soenergy
 end if

 ! Define some logical flags according to the type of calculation.
 BSp%RPA = .FALSE.
 BSp%GW  = .FALSE.
 BSp%EXC = .FALSE.

 SELECT CASE (toupper(BSp%type))
 CASE ("RPA")
   BSp%RPA         = .TRUE.
   BSp%COULOMBTERM = .FALSE.
   BSp%GWTERM      = .FALSE.
   msg = 'RPA calculation'
 CASE ("GW")
   BSp%GW          = .TRUE.
   BSp%COULOMBTERM = .FALSE.
   BSp%GWTERM      = .TRUE.
   msg = 'GW calculation'
 CASE ("EXC")
   BSp%EXC         = .TRUE.
   BSp%COULOMBTERM = .TRUE.
   BSp%GWTERM      = .TRUE.
   msg = 'EXCITONIC calculation '
 CASE DEFAULT
   write(msg,'(2a)')"Wrong p%type: ",TRIM(BSp%type)
   MSG_ERROR(msg)
 END SELECT

 call wrtout(std_out,msg,"COLL")
 
 BSp%LF  = .FALSE.
 BSp%NLF = .FALSE.

 SELECT CASE (toupper(BSp%lftype))
 CASE ("NLF")
   BSp%EXCHANGETERM = .FALSE.
   BSp%NLF          = .TRUE.
   msg = 'WITHOUT LOCAL FIELDS'
 CASE ("LF")
   BSp%EXCHANGETERM = .TRUE.
   BSp%LF           = .TRUE.
   msg = 'LOCAL FIELDS'
 CASE DEFAULT
   write(msg,'(2a)')"Wrong BSp%lftype: ",TRIM(BSp%lftype)
   MSG_ERROR(msg)
 END SELECT 

 call wrtout(std_out,msg,"COLL")
 
 BSp%COUPLING  = .FALSE.
 BSp%RESONANT  = .FALSE.

 SELECT CASE (toupper(BSp%htype))
 CASE ("RESONANT")
   BSp%COUPLING = .FALSE.
   BSp%RESONANT = .TRUE.
   msg = 'RESONANT ONLY'
 CASE ("COUPLING")
   BSp%COUPLING = .TRUE.
   msg = 'COUPLING'
 CASE DEFAULT
   write(msg,'(2a)')"Wrong BSp%htype: ",TRIM(BSp%htype)
   MSG_ERROR(msg)
 END SELECT

 call wrtout(std_out,msg,"COLL")
 
 if (.not.BSp%COULOMBTERM .and. .not.BSp%COUPLING) then
   MSG_WARNING(' ANTIRESONANT EXCLUDED')
 end if
 
 SELECT CASE (toupper(BSp%wtype))
 CASE ("WFULL")
   BSp%WDIAG = .FALSE.
   BSp%WFULL = .TRUE.
   msg = ' W FULL'
 CASE ("WDIAG")
   BSp%WDIAG = .TRUE.
   BSp%WFULL = .FALSE.
   msg = ' W DIAGONAL ONLY'
 CASE DEFAULT
   write(msg,'(2a)')"Wrong BSp%wtype: ",TRIM(BSp%wtype)
   MSG_ERROR(msg)
 END SELECT

 call wrtout(std_out,msg,"COLL")

!now test input parameters from input and kss file and assume some defaults

!MG NOW Have to copy some values to BSp%
 BSp%nkibz  = Kmesh%nibz  !We might allow for a smaller number of points....
 BSp%npwwfn = Dtset%npwwfn
 BSp%npweps = Dtset%npweps 
 BSp%npwvec = MAX(BSp%npwwfn,BSp%npweps)  !RECHECK THIS

 !TODO add new dim for exchange part and consider the possibility of 
 !having npwsigx > npwwfn (see setup_sigma).

 if (BSp%npwwfn>ng_kss) then
   BSp%npwwfn=ng_kss 
   write(msg,'(2a,(a,i8,a))')&
&    ' Number of G-vectors in KSS file found less than required',ch10,&
&    '  calculation will proceed with npwwfn  = ',BSp%npwwfn,ch10
   MSG_WARNING(msg)
 end if

!test if shells are closed
 do ish=1,BSp%nsh
   if (BSp%npweps >  BSp%shlim(ish)) cycle
   if (BSp%npweps == BSp%shlim(ish)) then
     EXIT
   else  ! that means npweps is in between two closed shells
     BSp%npweps=BSp%shlim(ish-1)
     write(msg,'(3a,i0)')&
&      ' npweps is not a closed shell',ch10,&
&      ' calculation continues with npweps =', BSp%shlim(ish-1)
     MSG_COMMENT(msg)
     EXIT
   end if
 end do

 do ish=1,BSp%nsh
   if (BSp%npwwfn  > BSp%shlim(ish)) cycle
   if (BSp%npwwfn == BSp%shlim(ish)) then
     EXIT
   else  ! that means npwwfn is in between two closed shells
     BSp%npwwfn=BSp%shlim(ish-1)
     write(msg,'(3a,i0)')&
&      ' npwwfn is not a closed shell',ch10,&
&      ' calculation continues with npweps =', BSp%shlim(ish-1)
     MSG_COMMENT(msg)
     EXIT 
   end if
 end do
     
!calculate NPWVEC as the biggest between npweps and npwwfn.
!MG RECHECK this part.
 BSp%npwvec=MAX(BSp%npwwfn,BSp%npweps)

 if (ng_kss < BSp%npwvec) then
   write(msg,'(a,i0,a)')" KSS file contains only ", ng_kss, " planewaves"
   MSG_WARNING(msg)
   BSp%npwvec = ng_kss
   if (BSp%npwwfn > ng_kss) then
     BSp%npwwfn = ng_kss
     write(msg,'(a,i0)') '       the calculation will be done with NPWWFN = ',BSp%npwwfn
     call wrtout(std_out,msg,"COLL")
   end if
   if (BSp%npweps > ng_kss) then
     BSp%npweps = ng_kss
     write(msg,'(a,i0)') '       the calculation will be done with npweps = ',BSp%npweps
     call wrtout(std_out,msg,"COLL")
   end if
 end if

 BSp%nop = Cryst%nsym

 if (BSp%nopk == -1) BSp%nopk = Cryst%nsym
 if (Cryst%nsym < BSp%nopk) then
   write(*,*) 'ERROR: qplda file contains only ', Cryst%nsym, ' symmetries'
   write(*,*) '       instead of ', BSp%nopk, ' required;'
   BSp%nopk = Cryst%nsym
   write(*,*) '       the calculation will be done with NOP = ',BSp%nopk
 else if (Cryst%nsym > BSp%nopk) then
   write(*,*) 'WARNING: qplda file contains ', Cryst%nsym, ' symmetries;'
   write(*,*) '         assuming expert user;'
   write(*,*) '         the calculation will be done with the first',BSp%nopk, ' symmetries of the file.'
 end if

 BSp%nel = Dtset%nelect  
 if (ABS(BSp%nel-nelr)>tol6) then
   write(*,*) 'WARNING: kss file contains ', nelr, ' electrons;'
   write(*,*) '         assuming expert user;'
   write(*,*) '         the calculation will be done with ', BSp%nel
   MSG_ERROR("not tested")
 end if 

 if (nbnds_kss < Dtset%nband(1)) then
   write(msg,'(2(a,i0),3a,i0)')&
&    ' KSS file contains only ', nbnds_kss, ' levels instead of ', Dtset%nband(1), ' required;',ch10,&
&    ' The calculation will be done with nbands = ',nbnds_kss
   MSG_WARNING(msg)
   Dtset%nband(:) = nbnds_kss
 end if

 BSp%nbnds = Dtset%nband(1) ! TODO Note the change in the meaning of input variables
 !BSp%nbnds = Dtset%bs_eh_basis_set(2)

 if (BSp%nbnds<=Dtset%nelect/2) then
   write(msg,'(2a,2(a,i0))')&
&    ' BSp%nbnds cannot be smaller than homo ',ch10,&
&    ' while BSp%nbnds = ',BSp%nbnds,' and Dtset%nelect= ',Dtset%nelect
   MSG_ERROR(msg)
 end if

 if (BSp%ninv == 1) then
   linv = .false.
 else
   BSp%ninv = 2
   linv = .true.
 end if
      
 if (BSp%ninvk == 1) then
   linvk = .false.
 else
   BSp%ninvk = 2
   linvk = .true.
 end if

 ! === Make biggest G-sphere of BSP%npwvec vectors ===
 only_one_kpt=.FALSE. !(Kmesh%nbz==1)
 call init_Gvectors_type(only_one_kpt,Gsph_Max,Cryst,BSp%npwvec,gvec_p,Cryst%gmet,gprimd)
 deallocate(gvec_p)

 call print_Gvectors(Gsph_Max,unit=std_out,prtvol=Dtset%prtvol)

 nqlwl=Er%Hscr%nqlwl

 if (nqlwl==0) then 
   nqlwl=1
   allocate(qlwl(3,nqlwl)) 
   qlwl(:,nqlwl)= GW_Q0_DEFAULT
   write(msg,'(3a,i2,a,3f9.6)')&
&    " The Header of the screening file does not contain the list of q-point for the optical limit ",ch10,&
&    " Using nqlwl= ",nqlwl," and qlwl = ",qlwl(:,1)  
   MSG_COMMENT(msg)
 else 
   allocate(qlwl(3,nqlwl)) 
   qlwl = Er%Hscr%qlwl
 end if
 !write(*,*)"nqlwl and qlwl for Coulomb singularity and e^-1",nqlwl,qlwl

 call setup_coulombian(Dtset,Gsph_Max,Qmesh,Kmesh,BSp%npwvec,nqlwl,qlwl,Cryst%rprimd,ngfftf,Vcp,spaceComm)
 
 deallocate(qlwl)

 bantot=SUM(Dtset%nband(1:Dtset%nkpt*Dtset%nsppol))                                                       
 allocate(doccde(bantot),eigen(bantot),occfact(bantot))
 doccde=zero; eigen=zero; occfact=zero 

 jj=0; ibtot=0
 do isppol=1,Dtset%nsppol
   do ik_ibz=1,Dtset%nkpt
     do ib=1,Hdr_kss%nband(ik_ibz+(isppol-1)*Dtset%nkpt)
       ibtot=ibtot+1
       if (ib<=BSP%nbnds) then  
         jj=jj+1
         occfact(jj)=Hdr_kss%occ(ibtot)
         eigen  (jj)=energies_p(ib,ik_ibz,isppol)
       end if
     end do
   end do
 end do

 deallocate(energies_p)

 ! * Make sure that Dtset%wtk==Kmesh%wt due to the dirty treatment of 
 !   symmetry operations in the old GW code (symmorphy and inversion) 
 ltest=(ALL(ABS(Dtset%wtk(1:Kmesh%nibz)-Kmesh%wt(1:Kmesh%nibz))<tol6))
 ABI_CHECK(ltest,'Mismatch between Dtset%wtk and Kmesh%wt')

 allocate(npwarr(Dtset%nkpt)); npwarr(:)=BSP%npwwfn

 call bstruct_init(bantot,KS_BSt,Dtset%nelect,doccde,eigen,Dtset%istwfk,Kmesh%ibz,Dtset%nband,&
&  Kmesh%nibz,npwarr,Dtset%nsppol,Dtset%nspinor,Dtset%tphysel,Dtset%tsmear,Dtset%occopt,occfact,Kmesh%wt) 

 !TODO call update_occ here
 ! Occupancies might be zero if NSCF 
 !$call update_occ(KS_BSt,fixmom,stmbias,Dtset%prtvol)

 call print_bandstructure(KS_BSt,"Band structure read from the KSS file",unit=std_out,prtvol=Dtset%prtvol)

 deallocate(doccde,eigen,npwarr)

 ! === Create Sigma header === 
 pertcase_=0
 call hdr_init(KS_BSt,codvsn,Dtset,Hdr_bse,Pawtab,pertcase_,Psps)

 ! === Get Pawrhoij from the header of the KSS file ===
 allocate(Pawrhoij(Cryst%natom*Dtset%usepaw))
 if (Dtset%usepaw==1) then
   allocate(nlmn(Cryst%ntypat))
   do itypat=1,Cryst%ntypat
     nlmn(itypat)=Pawtab(itypat)%lmn_size
   end do
   call rhoij_alloc(1,nlmn,Dtset%nspden,Dtset%nspinor,Dtset%nsppol,Pawrhoij,Cryst%typat)
   deallocate(nlmn)
   call rhoij_copy(Hdr_kss%Pawrhoij,Pawrhoij)
 end if

 call hdr_update(bantot,1.0d20,1.0d20,Hdr_bse,Cryst%natom,1.0d20,&
&  Cryst%rprimd,occfact,MPI_enreg_seq,Pawrhoij,Dtset%usepaw,Cryst%xred)

! call hdr_update(bantot,1.0d20,1.0d20,Hdr_bse,Cryst%natom,1.0d20,&
!& Cryst%rprimd,KS_BSt%occ,Pawrhoij,Dtset%usepaw,Cryst%xred)

 deallocate(occfact)

 ! This is just to do a check, the file format is wrong!
 call hdr_check(1002,1002,Hdr_bse,Hdr_kss,'COLL',restart,restartpaw)

 if (Dtset%usepaw==1) call rhoij_free(Pawrhoij) 
 deallocate(Pawrhoij)

 ! === Find optimal value for G-sphere enlargment due to oscillator matrix elements ===
 ! * Here I have to be sure that Qmesh%bz is always inside the BZ, not always true size bz is buggy
 ! * -one is used because we loop over all the possibile differences, unlike screening
 mg0sh=5
 call get_ng0sh(Kmesh%nbz,Kmesh%bz,Kmesh%nbz,Kmesh%bz,Qmesh%nbz,Qmesh%bz,Cryst%gmet,-one,mg0sh,ng0sh_opt)
 BSp%mg0(:)=ng0sh_opt(:)

 ! === Setup of the FFT mesh for the oscilator strengths === 
 ! * ngfft_osc(7:18)==Dtset%ngfft(7:18) which is initialized before entering screening.
 ! * Here we redefine ngfft_osc(1:6) according to the following options :
 !
 ! method==0 --> FFT grid read from fft.in (debugging purpose)
 ! method==1 --> Normal FFT mesh
 ! method==2 --> Slightly augmented FFT grid to calculate exactly rho_tw_g (see setmesh.F90)
 ! method==3 --> Doubled FFT grid, same as the the FFT for the density,
 !
 ! enforce_sym==1 ==> Enforce a FFT mesh compatible with all the symmetry operation and FFT library
 ! enforce_sym==0 ==> Find the smallest FFT grid compatbile with the library, do not care about symmetries
 !
 ngfft_osc(1:18)=Dtset%ngfft(1:18); method=2
 if (Dtset%fftgw==00 .or. Dtset%fftgw==01) method=0
 if (Dtset%fftgw==10 .or. Dtset%fftgw==11) method=1
 if (Dtset%fftgw==20 .or. Dtset%fftgw==21) method=2
 if (Dtset%fftgw==30 .or. Dtset%fftgw==31) method=3
 enforce_sym=MOD(Dtset%fftgw,10) 
 
 call setmesh(gmet,Gsph_Max%gvec,ngfft_osc,BSp%npwvec,BSp%npweps,BSp%npwwfn,nfftot_osc,method,BSp%mg0,Cryst,enforce_sym)
 nfftot_osc=PRODUCT(ngfft_osc(1:3))

 call print_ngfft(ngfft_osc,"FFT mesh for oscillator matrix elements",std_out,"COLL",prtvol=Dtset%prtvol)

!find BSp%homo, the highest occupied band
! TODO spin
 do ib=0,BSp%nbnds-1
   if (KS_BSt%occ(ib+1,1,1) < 0.001) then
    BSp%homo=ib; EXIT
   end if
 end do

 BSp%lumo  = BSp%homo + 1
 BSp%humo  = BSp%nbnds

 BSp%nbndv = BSp%homo  - BSp%lomo + 1
 BSp%nbndc = BSp%nbnds - BSp%homo

 BSp%nkbz = Kmesh%nbz

 allocate(gwenergy(BSp%nbnds,BSp%nkibz,Dtset%nsppol), STAT=istat)
 gwenergy(:,:,:) = KS_BSt%eig(:,:,:)
      
 if (BSp%GWTERM) then ! Read GW energies.
   !
   if (BSp%soshift) then
     write(msg,'(a,f8.2,a)')' Applying a scissors operator ',BSp%soenergy*Ha_eV," [eV]."
     call wrtout(std_out,msg,"COLL")
     gwenergy(:BSp%homo,:,:) = KS_BSt%eig(:BSp%homo,:,:)
     gwenergy(BSp%lumo:,:,:) = KS_BSt%eig(BSp%lumo:,:,:) + BSp%soenergy
   else
     gwfile=TRIM(Dtfil%filnam_ds(4))//'_GW'   ! TODO here I should introduce variable getgw
     gwfile="__in.gw__"

     if (.not.existent_file(gwfile)) then
       msg = " File "//TRIM(gwfile)//" not found. Aborting now"
       MSG_ERROR(msg)
     end if

     call copy_bandstructure(KS_BSt,QP_bst)
     allocate(igwene(QP_bst%mband,QP_bst%nkpt,QP_bst%nsppol))
     call rdgw(QP_Bst,gwfile,igwene,extrapolate=.FALSE.) ! here gwenergy is real
     !write(*,*)"igwene",igwene
     !gwenergy = QP_bst%eig + igwene
     gwenergy = QP_bst%eig  ! Do not include qp linewidth!
     deallocate(igwene)
     call bstruct_clean(QP_bst) ! TODO this should be output but I have to add a new array to store im(e)

     do isppol=1,Dtset%nsppol
       write(std_out,*) ' k       GW energies [eV]'
       do ik_ibz=1,BSp%nkibz
         write(std_out,'(i3,7x,10f7.2/50(10x,10f7.2/))')ik_ibz,(DBLE(gwenergy(ib,ik_ibz,isppol))*Ha_eV,ib=1,BSp%nbnds)
       end do
       write(std_out,*) ' k       Im GW energies [eV]'
       do ik_ibz=1,BSp%nkibz
         write(std_out,'(i3,7x,10f7.2/50(10x,10f7.2/))')ik_ibz,(AIMAG(gwenergy(ib,ik_ibz,isppol))*Ha_eV,ib=1,BSp%nbnds)
       end do
     end do

   end if ! soshift
   !
 end if ! BSp%GWTERM

 !write(*,*)"nkbz, nbndv, nbndc ",BSp%nkbz,BSp%nbndv,BSp%nbndc
 allocate(Trans   (BSp%nkbz*BSp%nbndv*BSp%nbndc), STAT=istat)
 if (istat/=0) stop 'out of memory, transtab'
 allocate(transtab(BSp%nkbz,BSp%nbndv,BSp%nbndc), STAT=istat)
 if (istat/=0) stop 'out of memory, transtab'

 !en_order=.TRUE.; if (BSp%use_haydock) en_order=.FALSE.
 en_order=.FALSE. ! Transitions are ALWAYS ordered in c-v-k mode with k the slowest index.
                                                                                            
 call init_transitions(BSp,Dtset%nsppol,gwenergy,Kmesh%tab,stripewidth,Trans,transtab,gw_gap,en_order)
 !call wtrans(BSp,nsppol,gwenergy,Kmesh%bz,Kmesh%tab,Trans)

 msg=' Fundamental parameters for the solution of the Bethe-Salpeter equation:'
 call print_bs_parameters(BSp,unit=std_out,header=msg,mode_paral="COLL",prtvol=Dtset%prtvol) 
 call print_bs_parameters(BSp,unit=ab_out, header=msg,mode_paral="COLL") 

 ! TODO direct diago with non-symmorphic ops seems OK.
 ! Haydock should be checked in more detail

 ! non-symmorphic phases in Wfr are missing
 !if (.not.isymmorphic(Cryst).and. Kmesh%nibz/=Kmesh%nbz) then 
 !  msg=" Code cannot exploit non-symmorphic operations, check k-tables."
 !  MSG_WARNING(msg)
 !end if

 ! non-symmorphic phases in W are missing 
 !if (.not.isymmorphic(Cryst).and. Qmesh%nibz/=Qmesh%nbz) then 
 !  msg=" Code cannot exploit non-symmorphic operations, check q-tables." 
 !  MSG_WARNING(msg)
 !end if

 if ( ANY (Cryst%symrec(:,:,1) /= RESHAPE ( (/1,0,0,0,1,0,0,0,1/),(/3,3/) )) .or. &
&     ANY( ABS(Cryst%tnons(:,1)) > tol6) ) then
   write(msg,'(3a,9i2,2a,3f6.3,2a)')&
&    " The first symmetry operation should be the Identity with zero tnons while ",ch10,&
&    " symrec(:,:,1) = ",Cryst%symrec(:,:,1),ch10,&
&    " tnons(:,1)    = ",Cryst%tnons(:,1),ch10,&
&    " This is not allowed, sym_rhotwgq0 should be changed."
   MSG_ERROR(msg)
 end if

 ! Search for previous output files. TODO Here I have to introduce 
 ! get_bsham, ird_bsham version, getbseig and irdbseig

 BS_files%exh = 'out.exh'
 inquire(file='in.exh',exist=BS_files%hx)
 if (BS_files%hx) BS_files%exh = 'in.exh'

 BS_files%exc = 'out.exc'
 inquire(file='in.exc',exist=BS_files%cx)
 if (BS_files%cx) BS_files%exc = 'in.exc'

 ! in_eig is the name of the input file with eigenvalues and eigenvectors 
 ! constructed from getbseig or irdbseig. out_eig is the name of the output file
 ! produced by this dataset. in_eig_exists checks for the presence of the input file.

 BS_files%in_eig  = Dtfil%fnameabi_bseig
 BS_files%out_eig = Dtfil%fnameabo_bseig

 BS_files%in_eig_exists = existent_file(BS_files%in_eig)

 if (BS_files%in_eig_exists) then ! Perform some consistency checks.
   MSG_ERROR("Reading of BS_EIG file not yet coded")
 end if

 BS_files%exovl = 'out.exovl'
 inquire(file='in.exovl',exist=BS_files%ox)
 if (BS_files%ox)  BS_files%exovl = 'in.exovl'

 call print_bs_files(BS_files,unit=std_out)

! ==========================================================
! ==== Final check on the parameters of the calculation ====
! ==========================================================
 if (Bsp%use_haydock .and. Bsp%htype /= "RESONANT") then
   msg = " Haydock method is only compatible with RESONANT calculations"
   MSG_ERROR(msg)
 end if

 DBG_EXIT("COLL")

end subroutine setup_bse
!!***
