!{\src2tex{textfont=tt}}
!!****f* ABINIT/build_spectra
!! NAME
!!  build_spectra
!!
!! FUNCTION
!!  Driver routine for the computation of optical spectra.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida, M.Giantomassi)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  ntrans=Total number of transitions.
!!  nsppol=Number of independent spin polarizations.
!!  nspinor=Number of spinorial components.
!!  usepaw=1 for PAW calculations, 0 otherwise.
!!  inclvkb=If different from 0, [Vnl,r] is included in the calculation of the matrix elements of the velocity operator.
!!  omegaplasma=Drude plasma frequency.
!!  Dtfil<datafiles_type)>=variables related to file names and unit numbers.
!!  Bsp<type(excparam)>=Data type gathering the paramenters used for the Bethe-Salpeter calculation.
!!  BS_files<type(excfiles)>=filenames used in the Bethe-Salpeter part.
!!  Kmesh<type(BZ_mesh_type)>=the k-point sampling for the wave functions.
!!  Cryst<type(crystal_structure)>=Structure defining the crystalline structure.
!!  energy(BSp%nbnds,BSp%nkibz,nsppol)=The KS energies.
!!  gwenergy(BSp%nbnds,BSp%nkibz,nsppol)=The complex QP energies.
!!  trans(ntrans)<type(transition)>=Useful indeces definig each transition i.e., (k,v,c).
!!  Psps <type(pseudopotential_type)>=variables related to pseudopotentials.
!!  Pawtab(Cryst%ntypat*usepaw)<type(pawtab_type)>=PAW tabulated starting data
!!  Hur(Cryst%natom*usepaw)<type(HUr_commutator)>=Only for PAW and LDA+U, quantities used to evaluate the commutator [H_u,r].
!!  Wfd<wfs_descriptor>=Handler for the wavefunctions.
!!  spaceComm=MPI communicator.
!!
!! OUTPUT
!!  No output. The routine calls specialized routines where the computation and the output of the spectra is done.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine build_spectra(BSp,nspinor,nsppol,usepaw,Dtfil,BS_files,Cryst,Kmesh,energy,gwenergy,&
& Psps,Pawtab,ntrans,trans,inclvkb,Wfd,Hur,omegaplasma,spaceComm)

 use defs_basis 
 use m_bs_defs
 use defs_datatypes   
 use defs_abitypes   
 use m_xmpi
 use m_errors

 use m_crystal,         only : crystal_structure 
 use m_bz_mesh,         only : bz_mesh_type
 use m_commutator_vkbr, only : kb_potential
 use m_paw_commutator,  only : HUr_commutator
 use m_wfs,             only : wfs_descriptor

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_69_bse, except_this_one => build_spectra
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: ntrans,nsppol,nspinor,usepaw,inclvkb,spaceComm
 real(dp),intent(in) :: omegaplasma
 type(Datafiles_type),intent(in) :: Dtfil
 type(excparam),intent(in) :: BSp
 type(excfiles),intent(in) :: BS_files
 type(pseudopotential_type),intent(in) :: Psps
 type(BZ_mesh_type),intent(in) :: Kmesh
 type(crystal_structure),intent(in) :: Cryst
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays
 real(dp),intent(in) :: energy(BSp%nbnds,BSp%nkibz,nsppol)
 complex(dpc),intent(in) :: gwenergy(BSp%nbnds,BSp%nkibz,nsppol)
 type(transition),intent(in) :: trans(ntrans)
 type(pawtab_type),intent(in) :: Pawtab(Cryst%ntypat*usepaw)
 type(HUr_commutator),intent(in) :: Hur(Cryst%natom*usepaw)
 
!Local variables ------------------------------
!scalars
 integer,parameter :: NPOL=6
 integer :: nprocs,my_rank,master,iq,io,nh2
 character(len=500) :: msg
!arrays
 complex(dpc),allocatable :: epsrpanlf(:,:),epsgwnlf(:,:)
 complex(dpc),allocatable :: epsexc(:,:),fcvk(:,:,:,:,:)
      
!************************************************************************

 nprocs  = xcomm_size(spaceComm)
 my_rank = xcomm_rank(spaceComm)
 master=0

! =================================================
! === Calculate fcv(k)=<c k|e^{-iqr}|v k> in BZ ===
! =================================================
 allocate(fcvk(BSp%nbnds,BSp%nbnds,BSp%nkbz,nsppol,NPOL))

 call cfcvk(BSp%nbnds,BSp%nkibz,BSp%nkbz,Kmesh,nspinor,nsppol,energy,BSp,&
&  inclvkb,usepaw,Cryst,Wfd,Psps,Pawtab,Hur,fcvk,spaceComm)

! ========================
! === Make EPS EXCITON ===
! ========================
 allocate(epsexc(BSp%nomega,NPOL+2))
 epsexc(:,:)=czero

 if (BSp%COUPLING) then
   call wrtout(std_out," Calculating absorption strength with full coupling","COLL")
   nh2 = 2*ntrans
   do iq=1,BSp%nq ! TODO add possibility of reading in_eig
     call cepsexccoupl(BS_files%out_eig,BS_files,nsppol,BSp%nkbz,BSp%nbnds,ntrans,nh2,trans,fcvk(:,:,:,:,iq),Cryst%ucvol,&
&      BSp%nomega,BSp%omegai,BSp%domega,BSp%broad,epsexc(:,iq))
   end do

 else
   call wrtout(std_out," Calculating excitonic epsilon with antiresonant","COLL")
   do iq=1,BSp%nq ! TODO add possibility of reading in_eig
     call cepsexc(BS_files%out_eig,nsppol,BSp%nkbz,BSp%nbnds,ntrans,trans,fcvk(:,:,:,:,iq),Cryst%ucvol,&
&      BSp%nomega,BSp%omegai,BSp%domega,BSp%broad,epsexc(:,iq))
   end do
 end if

! =======================================================
! === Make EPS RPA and GW without local-field effects ===
! =======================================================
 call wrtout(std_out," Calculating RPA NLF and QP NLF epsilon","COLL")

 allocate(epsrpanlf(BSp%nomega,NPOL+2))
 allocate(epsgwnlf(BSp%nomega,NPOL+2))

 do iq=1,BSp%nq
   call cepsrpagw(BSp%nbnds,BSp%lomo,BSp%homo,Kmesh,nsppol,energy,gwenergy,fcvk(:,:,:,:,iq),Cryst%ucvol,&
&    BSp%nomega,BSp%omegai,BSp%domega,BSp%broad,epsrpanlf(:,iq),epsgwnlf(:,iq))
 end do

 deallocate(fcvk)

! ==========================
! === Write out Epsilon2 ===
! ==========================
 !dsffact = zero
 !dsffact = vlngth_dp(BSp%qpg,REAL(ls%b1,dp),REAL(ls%b2,dp),REAL(ls%b3,dp))**2 / (4 * pi**2 * BSp%nel/Cryst%ucvol)

 !this is just for the automatic tests, It will be removed when fldiff 
 !will be able to compare two optical spectral
 do iq=1,BSp%nq
   write(msg,'(2a,i3,a)')ch10," Polarization no. ",iq," omega, epsrpanlf, epsgwnlfm epsexc "
   call wrtout(ab_out,msg,"COLL")
   do io=1,BSp%nomega
     !write(msg,'(7es12.4)')REAL(BSp%omega(io))*Ha_eV,epsrpanlf(io,iq),epsgwnlf(io,iq),epsexc(io,iq)
     write(msg,'(7f9.4)')REAL(BSp%omega(io))*Ha_eV,epsrpanlf(io,iq),epsgwnlf(io,iq),epsexc(io,iq)
     call wrtout(ab_out,msg,"COLL")
   end do
 end do

 if (my_rank==master) then ! Master node writes final results on file.
   call wreps(BSp,Dtfil,epsexc,epsrpanlf,epsgwnlf)
 end if
 
 call wrtout(std_out," Checking Kramers Kronig on Excitonic Macroscopic Epsilon","COLL")
 call check_kramerskronig(BSp%nomega,REAL(BSp%omega(:)),epsexc(:,1))

 call wrtout(std_out," Checking Kramers Kronig on RPA NLF Macroscopic Epsilon","COLL")
 call check_kramerskronig(BSp%nomega,REAL(BSp%omega(:)),epsrpanlf(:,1))

 call wrtout(std_out," Checking Kramers Kronig on GW NLF Macroscopic Epsilon","COLL")
 call check_kramerskronig(BSp%nomega,REAL(BSp%omega(:)),epsgwnlf(:,1))

 call wrtout(std_out," Checking f-sum rule on Excitonic Macroscopic Epsilon","COLL")

 if(BSp%LF) then 
   MSG_WARNING(' f-sum rule should be checked without LF')
 end if
 call check_fsumrule(BSp%nomega,REAL(BSp%omega(:)),AIMAG(epsexc(:,1)),omegaplasma)

 call wrtout(std_out," Checking f-sum rule on RPA NLF Macroscopic Epsilon","COLL")
 call check_fsumrule(BSp%nomega,REAL(BSp%omega(:)),AIMAG(epsrpanlf(:,1)),omegaplasma)

 call wrtout(std_out," Checking f-sum rule on GW NLF Macroscopic Epsilon","COLL")
 call check_fsumrule(BSp%nomega,REAL(BSp%omega(:)),AIMAG(epsgwnlf(:,1)),omegaplasma)

 deallocate(epsrpanlf,epsgwnlf,epsexc)

 call xbarrier_mpi(spaceComm)

end subroutine build_spectra
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/cfcvk
!! NAME
!!  cfcvk
!!
!! FUNCTION
!!  Calculate all optical matrix elements in the IBZ.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida, M.Giantomassi)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! npwvec=Maximum between npweps and npwwfn
!! npwwfn=Number of plane wave for the wave functions.
!! nsppol=Number of independent spin polarizations.
!! usepaw=1 for PAW, 0 otherwise.
!! nspinor=Number of spinorial components.
!! nbnds=Total number of bands.
!! nkibz=Number of irreducible k-points.
!! nkbz=Number of points in the full Brillouin zone.
!! inclvkb=if different from 0, [Vnl,r] is included in the calculation of the matrix element of the velocity operator
!!   No meaning for PAW.
!! Bsp<type(excparam)>=Parameters for the Bethe-Salpeter run.
!! Kmesh<type(bz_mesh_type)>=Info on the k-point sampling for wave functions. 
!! Cryst<type(crystal_structure)>=Structure defining the crystalline structure.
!! energy(nbnds,nkibz,nsppol)=KS energies.
!! Pawtab(Cryst%ntypat*usepaw)<type(pawtab_type)>=PAW tabulated starting data
!! Psps <type(pseudopotential_type)>=variables related to pseudopotentials.
!! Hur(Cryst%natom*usepaw)<type(HUr_commutator)>=Only for PAW and LDA+U, quantities used to evaluate the commutator [H_u,r].
!! Wfd<wfs_descriptor>=Handler for the wavefunctions.
!!   %gvec(3,npwvec)=Reduced coordinates of the plane waves.
!! spaceComm=MPI Communicator.
!!
!! OUTPUT
!! fcvk(nbnds,nbnds,nkbz,nsppol,BSp%nq)=Matrix elements <c k|e^{+iqr}|v k> for the different qs.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine cfcvk(nbnds,nkibz,nkbz,Kmesh,nspinor,nsppol,energy,BSp,&
&  inclvkb,usepaw,Cryst,Wfd,Psps,Pawtab,Hur,fcvk,spaceComm)

 use defs_basis
 use defs_datatypes
 use m_bs_defs
 use m_xmpi
 use m_errors

 use m_bz_mesh,           only : bz_mesh_type,get_BZ_item
 use m_crystal,           only : crystal_structure
 use m_paw_commutator,    only : HUr_commutator, paw_ihr_comm
 use m_commutator_vkbr,   only : kb_potential, nullify_kb_potential, destroy_kb_potential, init_kb_potential, &
&                                add_vnlr_commutator, nc_ihr_comm
 use m_wfs,               only : wfs_descriptor, wfd_get_cprj

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
 use interfaces_51_manage_mpi
 use interfaces_53_abiutil
 use interfaces_68_gw
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nsppol,usepaw,nspinor,nbnds
 integer,intent(in) :: nkibz,nkbz,inclvkb,spaceComm
 type(excparam),intent(in) :: BSp
 type(bz_mesh_type),intent(in) :: Kmesh
 type(crystal_structure),intent(in) :: Cryst
 type(pseudopotential_type),intent(in) :: Psps
 type(wfs_descriptor),intent(inout) :: Wfd
!arrays
 real(dp),intent(in) ::  energy(nbnds,nkibz,nsppol)
 complex(dpc),intent(out) :: fcvk(nbnds,nbnds,nkbz,nsppol,BSp%nq)
 type(pawtab_type),intent(in) :: Pawtab(Cryst%ntypat*usepaw)
 type(HUr_commutator),intent(in) :: Hur(Cryst%natom*usepaw)

!Local variables ------------------------------
!scalars
 integer :: ik_bz,ik_ibz,itim_k,isym_k,iq,ib_c,ib_v,ierr,nprocs,my_rank,master
 integer :: spad,i1,i2,isppol,iband
 real(dp) :: ediff
 complex(dpc) :: emcvk
 type(kb_potential) :: KBgrad_k
!arrays
 integer,allocatable :: task_distrib(:,:,:,:)
 real(dp) :: mat_dp(3,3),qrot(3),b1(3),b2(3),b3(3),kbz(3)
 complex(dpc),allocatable :: ir_kibz(:,:,:,:,:)
 complex(gwpc),pointer :: ug(:,:)
 complex(gwpc) :: ihrc(3,nspinor**2)
 type(Cprj_type),allocatable :: Cp_v(:,:),Cp_c(:,:)

!************************************************************************

 call wrtout(std_out," Calculating optical matrix elements in the IBZ","COLL")
 ABI_CHECK(nspinor==1,"nspinor==2 not coded")

 nprocs  = xcomm_size(spaceComm)
 my_rank = xcomm_rank(spaceComm)
 master=0

 !rcart2red = TRANSPOSE(Cryst%gprimd)

 allocate(task_distrib(nbnds,nbnds,nkibz,nsppol))
 call fill_task_distrib_4D(nprocs,task_distrib)

 if (usepaw==1) then
   allocate(Cp_v(Wfd%natom,Wfd%nspinor)); call cprj_alloc(Cp_v,0,Wfd%nlmn_atm)
   allocate(Cp_c(Wfd%natom,Wfd%nspinor)); call cprj_alloc(Cp_c,0,Wfd%nlmn_atm)
 end if

 if (inclvkb==1) then
   MSG_WARNING("inclvkb==1 not coded,using inclvkb==2")
 end if
 !
 ! Calculate the matrix elements of ir in the IBZ.
 call nullify_kb_potential(KBgrad_k)

 allocate(ir_kibz(3,nbnds,nbnds,nkibz,nsppol))
 ir_kibz=czero

 do ik_ibz=1,nkibz
   do isppol=1,nsppol

    if ( ALL(task_distrib(:,:,ik_ibz,isppol)/=my_rank) ) CYCLE

    if (inclvkb/=0.and.usepaw==0) then ! Prepare term i <n,k|[Vnl,r]|n"k>
      call init_kb_potential(KBgrad_k,Cryst,Psps,inclvkb,Wfd%npwwfn,Kmesh%ibz(:,ik_ibz),Wfd%gvec)
    end if

    ! Select u_k(G).
    allocate(ug(Wfd%npwwfn*Wfd%nspinor,nbnds))
    do iband=1,nbnds
      ug(:,iband) = Wfd%Wave(iband,ik_ibz,isppol)%ug(:)
    end do

    ! Note: spinorial case is not coded therefore we work with ihrc(:,1).
    ! TODO: The lower triangle can be Reconstructed by symmetry.
    do ib_v=1,nbnds ! Loop over bands
      if (usepaw==1) call wfd_get_cprj(Wfd,ib_v,ik_ibz,isppol,Cryst,Cp_v,sorted=.FALSE.)

      do ib_c=1,nbnds

       if (task_distrib(ib_v,ib_c,ik_ibz,isppol)/=my_rank) CYCLE

       if (usepaw==0) then  ! Calculate matrix elements of i[H,r] for NC pseudopotentials.        
         ihrc = nc_ihr_comm(nspinor,Wfd%npwwfn,inclvkb,KBgrad_k,ug(:,ib_c),ug(:,ib_v),Wfd%gvec) 

       else ! Matrix elements of i[H,r] for PAW.
         spad=(nspinor-1)
         i1=ib_c; if (nspinor==2) i1=(2*ib_c-1)
         i2=ib_v; if (nspinor==2) i2=(2*ib_v-1)

         if (usepaw==1) call wfd_get_cprj(Wfd,ib_c,ik_ibz,isppol,Cryst,Cp_c,sorted=.FALSE.)

         ihrc = paw_ihr_comm(isppol,nspinor,Wfd%npwwfn,Kmesh%ibz(:,ik_ibz),Cryst,Pawtab,&
&          ug(:,ib_c),ug(:,ib_v),Wfd%gvec,Cp_c,Cp_v,HUr)

         !igradred_paw(1,:)=MATMUL(rcart2red,igradpaw_cart(1,:))
         !igradred_paw(2,:)=MATMUL(rcart2red,igradpaw_cart(2,:))
         !ihrc(:,1)=ihrc(:,1)-CMPLX(igradred_paw(1,:),igradred_paw(2,:),dpc)
       end if

       ! Save matrix elements of i*r in the IBZ
       ediff = energy(ib_c,ik_ibz,isppol) - energy(ib_v,ik_ibz,isppol)
       if (ABS(ediff)<tol16) ediff=tol6  ! Treat a possible degeneracy between v and c.
       ir_kibz(:,ib_c,ib_v,ik_ibz,isppol) = ihrc(:,1)/ediff

      end do !ib_c
    end do !ib_v

    call destroy_kb_potential(KBgrad_k)

    deallocate(ug)

   end do !isppol
 end do !ik_ibz

 ! Collect results on each node.
 call xsum_mpi(ir_kibz,spaceComm,ierr)

 deallocate(task_distrib)
 if (usepaw==1) then
   call cprj_free(Cp_v); deallocate(Cp_v)
   call cprj_free(Cp_c); deallocate(Cp_c)
 end if

 ! Calculate Fcv(kBZ) in the full Brilouin zone.

 ! Symmetrization of the matrix elements of the position operator.
 ! <Sk b|r|Sk b'> = R <k b|r|k b'> + \tau \delta_{bb'}
 !   where S is one of the symrec operations in reciprocal space, R is the 
 !   corresponding operation in real space, \tau being the associated fractional translations.
 !
 ! q.Mcv( Sk) =  S^{-1}q. Mcv(k) 
 ! q.Mcv(-Sk) = -S^{-1}q. CONJG(Mcv(k)) if time-reversal is used.

 b1=Cryst%gprimd(:,1)*two_pi
 b2=Cryst%gprimd(:,2)*two_pi
 b3=Cryst%gprimd(:,3)*two_pi

 fcvk = czero
 do iq=1,BSp%nq

   do isppol=1,nsppol
     do ik_bz=1,nkbz
      !
      ! * Get ik_ibz, and symmetries index from ik_bz.
      call get_BZ_item(Kmesh,ik_bz,kbz,ik_ibz,isym_k,itim_k)
      
      mat_dp = DBLE(Cryst%symrec(:,:,isym_k))
      call matrginv(mat_dp,3,3) ! Invert

      qrot = (3-2*itim_k) * MATMUL(mat_dp,BSp%q(:,iq))

      do ib_v=1,nbnds !  Loops over the bands C and V start
        do ib_c=1,nbnds
          !if (ib_c==ib_v) CYCLE 
          emcvk = pdtqrc(qrot,ir_kibz(1,ib_c,ib_v,ik_ibz,isppol),b1,b2,b3)
          if (itim_k==2) emcvk = CONJG(emcvk)
          fcvk(ib_c,ib_v,ik_bz,isppol,iq) = emcvk
        end do !ib_c
      end do !ib_v

     end do !ik_bz
   end do !isppol

 end do ! iq

 deallocate(ir_kibz)

 call xbarrier_mpi(spaceComm)

end subroutine cfcvk
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/cepsrpagw
!! NAME
!!  cepsrpagw
!!
!! FUNCTION
!!  Make epsilon within RPA and GW. 
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida, M.Giantomassi)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! nkbz=Number of points in the BZ 
!! nbnds=Number of bands
!! lomo=Lowest occupied state
!! nbndo=Number of occupied states.
!! nsppol=Number of independent spin polarizations.
!! nomega=Number of frequencies
!! ucvol=Unit cell volume.
!! dE=Frequency step.
!! eta=Amplitude of the complex imaginary shift.
!! emin=minimum frequency for the spectra.
!! energy(nbnds,nkibz,nsppol)=The KS energies.
!! gwenergy(nbnds,nkibz,nsppol)=The complex QP energies.
!! fcvk(nbnds,nbnds,nkbz)=Matrix elements <b k|e^{-iqr}|b" k> for a given q in the full BZ.
!!
!! OUTPUT
!!  epsrpanlf(nomega)=RPA spectrum without local-field effects.
!!  epsgwnlf(nomega)=GW spectrum without local-field effects (KS energies replaced by QP energies).
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine cepsrpagw(nbnds,lomo,nbndo,Kmesh,nsppol,energy,gwenergy,fcvk,ucvol,nomega,emin,dE,eta,epsrpanlf,epsgwnlf)

 use defs_basis
 use m_errors

 use m_bz_mesh,  only : bz_mesh_type

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nbnds,lomo,nbndo,nsppol,nomega
 real(dp),intent(in) :: ucvol
 real(dp),intent(in) :: dE,eta,emin
 type(BZ_mesh_type),intent(in) :: Kmesh
!arrays
 real(dp),intent(in) :: energy(nbnds,Kmesh%nibz,nsppol)
 complex(dpc),intent(in) :: gwenergy(nbnds,Kmesh%nibz,nsppol)
 complex(dpc),intent(in) :: fcvk(nbnds,nbnds,Kmesh%nbz,nsppol)
 complex(dpc),intent(out) :: epsrpanlf(nomega),epsgwnlf(nomega)

!Local variables ------------------------------
!scalars
 integer :: ii,ib_v,ib_c,ik_bz,ik_ibz,isppol
 real(dp) :: FAQ,ediff
 complex(dpc) :: omega,ctemp,egwdiff

!************************************************************************

 epsrpanlf(:) = czero
 epsgwnlf (:) = czero
 !
 ! Sum over all QP transitions.
 do isppol=1,nsppol
   do ik_bz=1,Kmesh%nbz
     ik_ibz = Kmesh%tab(ik_bz)
     do ib_v=lomo,nbndo
       do ib_c=nbndo+1,nbnds

        egwdiff = gwenergy(ib_c,ik_ibz,isppol) - gwenergy(ib_v,ik_ibz,isppol)
        ediff   =   energy(ib_c,ik_ibz,isppol) -   energy(ib_v,ik_ibz,isppol)
        ctemp = fcvk(ib_c,ib_v,ik_bz,isppol)

        do ii=1,nomega
          omega = emin + (ii-1)*dE + (0,1)*eta
          epsrpanlf(ii) = epsrpanlf(ii)  + ctemp * CONJG(ctemp) * (one/(ediff-omega) + one/(ediff+omega))
          epsgwnlf (ii) = epsgwnlf (ii)  + ctemp * CONJG(ctemp) * (one/(egwdiff-omega) + one/(egwdiff+omega))
        end do

       end do !ib_c
     end do !ib_v
   end do !ik_bz
 end do !isppol

 FAQ=four_pi/(ucvol*Kmesh%nbz); if (nsppol==1) FAQ=two*FAQ

 epsrpanlf(:) = cone + FAQ*epsrpanlf(:)
 epsgwnlf (:) = cone + FAQ*epsgwnlf(:)

end subroutine cepsrpagw
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/cepsexc
!! NAME
!!  cepsexc
!!
!! FUNCTION
!!  This routine calculates the macroscopic dielectric function with excitonic effects.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida, M.Giantomassi)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! filbseig=Name of the file containing excitoning eigenstates and eigenvalues.
!! nkbz=Number of points in the BZ 
!! nsppol=Number of independent polarizations.
!! nbnds=Number of bands
!! ntrans=Total number of transitions.
!! nomega=Number of frequencies
!! ucvol=Volume of the unit cell.
!! dE=Frequency step.
!! eta=Amplitude of the complex imaginary shift.
!! emin=minimum frequency for the spectra.
!! fcvk(nbnds,nbnds,nkbz,nsppol)=Matrix elements <b k|e^{-iqr}|b" k> for a given q in the full BZ.
!! trans(ntrans)<type(transition)>It gives the (k,v,c) index and the energy for each transition. 
!!
!! OUTPUT
!!  epsexc(nomega)=Macroscopic dielectric function with excitonic effects.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine cepsexc(filbseig,nsppol,nkbz,nbnds,ntrans,trans,fcvk,ucvol,nomega,emin,dE,eta,epsexc)

 use defs_basis
 use m_bs_defs
 use m_errors

 use m_io_tools, only : get_unit

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbz,nbnds,ntrans,nomega,nsppol
 real(dp),intent(in) :: ucvol
 real(dp),intent(in) :: dE,eta,emin
 character(len=*),intent(in) :: filbseig
!arrays
 complex(dpc),intent(in) :: fcvk(nbnds,nbnds,nkbz,nsppol)
 complex(dpc),intent(out) :: epsexc(nomega)
 type(transition),intent(in) :: trans(ntrans)

!Local variables ------------------------------
!scalars
 integer :: ll,it,ii,ib_v,ib_c,ik_bz,itemp,ios,eig_unt
 real(dp) :: gwgap,excgap,FAQ
 complex(dpc) :: eps,omega,ctemp
 character(len=500) :: msg
!arrays
 real(dp),allocatable :: exc_ene(:)
 real(dp),allocatable :: ostrength(:)
 complex(dpc),allocatable :: cexevl(:)
 complex(dpc),allocatable :: exc_state(:)

!************************************************************************

 ABI_CHECK(nsppol==1,"nsppol==2 not coded")

 FAQ=four_pi/(ucvol*nkbz); if (nsppol==1) FAQ=two*FAQ

 call wrtout(std_out," Reading excitonic states from file "//TRIM(filbseig),"COLL")

 eig_unt = get_unit() 
 open(unit=eig_unt,file=filbseig,form='unformatted',status='old',iostat=ios)

 msg="Opening file "//TRIM(filbseig)//" as old unformatted"
 ABI_CHECK(ios==0,msg) 

 read(eig_unt)itemp
 ABI_CHECK(itemp==ntrans,'Wrong exeig file')

 allocate(cexevl(ntrans))
 read(eig_unt) cexevl(:) ! Read eigenvalues, ignore possible (small imaginary part)

 allocate(exc_ene(ntrans))
 exc_ene(:) = cexevl(:)
 deallocate(cexevl)

 ! Calculate oscillator strength.
 allocate(exc_state(ntrans),ostrength(ntrans))

 do ll=1,ntrans ! loop on excitonic eigenstates
   read(eig_unt) exc_state(:)

   ctemp = czero 
   do it=1,ntrans ! loop on transition t = (k,v,c)
     ik_bz = Trans(it)%k
     ib_v  = Trans(it)%v
     ib_c  = Trans(it)%c
     ctemp = ctemp + CONJG(fcvk(ib_c,ib_v,ik_bz,1)) * exc_state(it)
   end do ! it
   ostrength(ll) = ctemp*CONJG(ctemp)
 end do ! ll

 deallocate(exc_state)

 close(eig_unt)

 do ii=1,nomega
   omega = emin + (ii-1)*dE + (0,1)*eta
   eps = cone

   do ll=1,ntrans ! Sum over all exciton eigenstates
     eps = eps + FAQ * ostrength(ll) * (one/(exc_ene(ll) - omega) - one/(-exc_ene(ll) - omega))
   end do ! ll
   epsexc(ii) = eps
 end do ! ii

 gwgap = trans(1)%en
 if (exc_ene(1) < exc_ene(ntrans)) then
   excgap = exc_ene(1)
 else
   excgap = exc_ene(ntrans)
 end if

 deallocate(ostrength,exc_ene)

end subroutine cepsexc
!!***

!----------------------------------------------------------------------

!!****f* ABINIT/cepsexccoupl
!! NAME
!!  cepsexccoupl
!!
!! FUNCTION
!!  Make epsilon EXCITONIC with full COUPLING.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (L.Reining, V.Olevano, F.Sottile, S.Albrecht, G.Onida, M.Giantomassi)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! filbseig=Name of the file containing excitoning eigenstates and eigenvalues.
!! nkbz=Number of points in the BZ 
!! nbnds=Number of bands
!! ntrans=Total number of transitions.
!! nh2=Rank of the entire excitonic Hamiltonian including the coupling block.
!! nomega=Number of frequencies
!! nsppol=Number of independent spin polarizations.
!! ucvol=Unit cell volume.
!! dE=Frequency step.
!! eta=Amplitude of the complex imaginary shift.
!! emin=minimum frequency for the spectra.
!! BS_files<type(excfiles)>File names used in the Bethe-Salpeter code.
!! trans(ntrans)<type(transition)>It gives the (k,v,c) index and the energy for each transition. 
!! fcvk(nbnds,nbnds,nkbz,nsppol)=Matrix elements <b k|e^{-iqr}|b" k> for a given q in the full BZ.
!!
!! OUTPUT
!!  epsexc(nomega)=Macroscopic dielectric function with excitonic effects calculated including the COUPLING.
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

subroutine cepsexccoupl(filbseig,BS_files,nsppol,nkbz,nbnds,ntrans,nh2,trans,fcvk,ucvol,nomega,emin,dE,eta,epsexc)

 use defs_basis
 use m_bs_defs
 use m_errors

 use m_io_tools,  only : get_unit
 use m_blas,      only : xdotu

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nkbz,nbnds,ntrans,nh2,nomega,nsppol
 real(dp),intent(in) :: ucvol,dE,eta,emin
 type(excfiles),intent(in) :: BS_files
 character(len=*),intent(in) :: filbseig
!arrays
 type(transition),intent(in) :: trans(ntrans)
 complex(dpc),intent(in) :: fcvk(nbnds,nbnds,nkbz,nsppol)
 complex(dpc),intent(out) :: epsexc(nomega)

!Local variables ------------------------------
!scalars
 integer :: mi,it,itp,ii,ib_v,ib_c,ik_bz,itemp,eig_unt,ios,ovlp_unt
 real(dp) :: gwgap,excgap,FAQ
 complex(dpc) :: eps,omega,fam,famp
 character(len=500) :: msg
!arrays
 complex(dpc),allocatable :: Ami(:),E(:),Sm1mi(:),msfap(:),fa(:),fap(:)

!************************************************************************

 ABI_CHECK(nsppol==1,"nsppol==2 not coded")
 ABI_CHECK(nh2==ntrans*2,"Wrong nh2 or ntrans")

 FAQ=four_pi/(ucvol*nkbz); if (nsppol==1) FAQ=two*FAQ

 call wrtout(std_out," Reading excitonic states from file "//TRIM(filbseig),"COLL")

 eig_unt = get_unit()
 open(unit=eig_unt,file=filbseig,form='unformatted',status='old',iostat=ios)

 msg=" Opening file "//TRIM(filbseig)//" as old-unformatted."
 ABI_CHECK(ios==0,msg)

 read(eig_unt) itemp
 ABI_CHECK(itemp==nh2,"wrong file")

 allocate(E(nh2))
 read(eig_unt) E(:)

 allocate(Ami(nh2))
 allocate(fa(nh2),fap(nh2))

 do mi=1,nh2 ! Loop on excitonic eigenvalues mi
   read(eig_unt) Ami(:)
   fam  = czero
   famp = czero

   do it=1,ntrans ! Loop on transition t = (k,v,c)
     itp   = ntrans + it
     ik_bz = trans(it)%k
     ib_v  = trans(it)%v
     ib_c  = trans(it)%c

     fam = fam + CONJG(fcvk(ib_c,ib_v,ik_bz,1)) * Ami(it) &
&              + CONJG(fcvk(ib_v,ib_c,ik_bz,1)) * Ami(itp)

     famp = famp - fcvk(ib_c,ib_v,ik_bz,1) * CONJG(Ami(it)) &
&                + fcvk(ib_v,ib_c,ik_bz,1) * CONJG(Ami(itp))
   end do ! it
   fa (mi) = fam
   fap(mi) = famp
 end do ! mi

 deallocate(Ami)
 close(eig_unt)

 call wrtout(std_out," Reading overlap matrix from file "//TRIM(BS_files%exovl),"COLL")

 ovlp_unt = get_unit()
 open(unit=ovlp_unt,file=BS_files%exovl,form='unformatted',status='old',iostat=ios)

 msg=" Opening file "//TRIM(BS_files%exovl)//" as old-unformatted."
 ABI_CHECK(ios==0,msg)

 read(ovlp_unt) itemp
 ABI_CHECK(itemp==nh2,"wrong file")

!Sum over mi prime

 allocate(Sm1mi(nh2),msfap(nh2))
 do mi=1,nh2
   read(ovlp_unt) Sm1mi
   msfap(mi) = xdotu(nh2,Sm1mi,1,fap,1)
 end do

 deallocate(Sm1mi)
 deallocate(fap)

 close(ovlp_unt)

! === Calculate excitonic epsilon with coupling ===
 do ii=1,nomega
   omega = emin + (ii-1)*dE + (0,1)*eta
   eps = czero

   do mi=1,nh2    ! sum over all exciton eigenstates
     eps = eps - fa(mi) * msfap(mi) / (E(mi) - omega)
   end do
   epsexc(ii) = one + FAQ * eps
 end do

 gwgap = trans(1)%en
 excgap = 10000000.0
 do mi=1,nh2
   if (ABS(REAL(E(mi))) < excgap) excgap = ABS(REAL(E(mi)))
 end do

 deallocate(E,msfap)
 deallocate(fa)

end subroutine cepsexccoupl
!!***
