!{\src2tex{textfont=tt}}
!!****f* ABINIT/elphon
!!
!! NAME
!! elphon
!!
!! FUNCTION
!! This routine extracts the electron phonon coupling matrix
!! elements and calculates related properties - Tc, phonon linewidths...
!!
!! COPYRIGHT
!! Copyright (C) 2004-2010 ABINIT group (MVer, MG)
!! This file is distributed under the terms of the
!! GNU General Public Licence, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!   anaddb_dtset=dataset with input variables
!!     anaddb_dtset%a2fsmear = smearing for alpha2F function
!!     anaddb_dtset%brav = type of Bravais lattice
!!     anaddb_dtset%dipdip  =dipole dipole interaction flag
!!     anaddb_dtset%elphsmear = smearing width for gaussian integration
!!           or buffer in energy for calculations with tetrahedra (telphint=0)
!!     anaddb_dtset%elph_fermie = input value of Fermi energy
!!           0 means use value from wfk file
!!     anaddb_dtset%enunit = governs the units to be used for the output of
!!           the phonon frequencies and e-ph quantities
!!     anaddb_dtset%gkk2write= flag to write out gkk2 matrix elements to disk
!!     anaddb_dtset%gkk_rptwrite= flag to write out real space gkk_rpt matrix elements to disk
!!     anaddb_dtset%gkqwrite= flag to write out gkq matrix elements to disk
!!     anaddb_dtset%ep_b_min= first band taken into account in FS integration (if telphint==2)
!!     anaddb_dtset%ep_b_max= last band taken into account in FS integration (if telphint==2) 
!!     anaddb_dtset%prtfsurf = integer flag for the output of the Fermi surface (XCrysden file format)
!!     anaddb_dtset%prtnest = integer flag for the calculation of the nesting function
!!     anaddb_dtset%ifcflag = flag for IFC matrices in anaddb calling routine
!!           the IFCs are presumed to be known!
!!     anaddb_dtset%ifltransport= flag for transport properties (no=0: yes=1 )
!!     anaddb_dtset%kptrlatt=kpoint grid generating vectors, as in abinit
!!     anaddb_dtset%kptrlatt_fine=kpoint grid generating vectors, for fine grid used in FS integration
!!     anaddb_dtset%mustar = parameter for Coulombic pseudo-potential in McMillan T_c calculation
!!     anaddb_dtset%ngqpt(3)=integers defining the number of points in the qpt sampling
!!     anaddb_dtset%nqpath=number of vertices in the path in reciprocal space, for band structure
!!           and phonon linewidth output
!!     anaddb_dtset%nqshft= number of shift vectors for defining the sampling of q points
!!     anaddb_dtset%ntemper = number of temperature points to calculate, from tempermin to 
!!           tempermin+ntemper*temperinc
!!     anaddb_dtset%qpath=vertices in the path in reciprocal space, for band structure
!!           and phonon linewidth output
!!     anaddb_dtset%q1shft(3,4) =qpoint shifts considered
!!     anaddb_dtset%telphint = flag for integration over the FS with 0=tetrahedra 1=gaussians
!!     anaddb_dtset%tempermin = minimum temperature at which resistivity etc are calculated (in K)
!!     anaddb_dtset%temperinc = interval temperature grid on which resistivity etc are calculated (in K)
!!     anaddb_dtset%ep_keepbands = flag to keep gamma matrix dependence on electronic bands
!!
!!   acell_in(3)= input length scales of cell (bohr)
!!   amu(ntypat)=mass of the atoms (atomic mass unit)
!!   atmfrc  = inter-atomic force constants from anaddb
!!   dielt(3,3) = dielectric tensor
!!   dyewq0(3,3,natom)=atomic self-interaction correction to the
!!        dynamical matrix (only when anaddb_dtset%dipdip=1)
!!     or buffer in energy for calculations with tetrahedra (telphint=0)
!!     0 means use value from wfk file
!!   elph_base_name=base name for output files
!!   gmet(3,3) =metric in reciprocal space (telphint=1)
!!   gprim(3,3) =dimensionless basis vectors of reciprocal space
!!   indsym = mapping of atoms btw themselves under symmetry
!!   mpert =maximum number of ipert
!!   natom=number of atoms in cell
!!   nrpt =number of real space points used to integrate IFC (for
!!        interpolation of dynamical matrices)
!!   nsym=number of space group symmetries
!!   ntypat = number of types of atoms
!!   rcan(3,natom) =canonical positions of atoms
!!   rmet(3,3)=metric tensor in real space (bohr^2)
!!   rprim_in(3,3)= input primitive translation vectors
!!   rpt(3,nprt) =canonical positions of R points in the unit cell
!!   symrec(3,3,nsym)=3x3 matrices of the group symmetries (reciprocal space)
!!   symrel(3,3,nsym)=3x3 matrices of the group symmetries (real space)
!!   tnons(3,nsym)=fractional nonsymmorphic translations
!!   trans(3,natom) = Atomic translations : xred = rcan + trans
!!   typat(natom)=type integer for each atom in cell
!!   ucvol=unit cell volume in bohr**3
!!   unitgkk = fortran unit for file containing matrix elements, from mrggkk
!!   wghatm(natom,natom,nrpt) =Weight for the pair of atoms and the R vector
!!   xred(3,natom)=fractional dimensionless atomic coordinates
!!   zeff(3,3,natom) =effective charge on each atom, versus electric
!!        field and atomic displacement
!!
!! OUTPUT
!!
!! NOTES
!!  inspired to a large extent by epcouple.f from the DecAFT package by J. Kay Dewhurst
!!  most inputs taken from mkifc.f
!!  in anaddb anaddb_dtset%ifcflag must be 1 such that the IFC are calculated in atmfrc prior to calling elphon
!!
!!  brav not taken into account propely in all of the code. (MG?)
!!
!!  could choose to make a full 3 dimensional kpt array (:,:,:). Easier for many operations
!!
!! PARENTS
!!      anaddb
!!
!! CHILDREN
!!
!! NOTES
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine elphon(anaddb_dtset,acell_in,amu,atmfrc,&
&  ddkfilename,dielt,dyewq0,elph_base_name,gmet,&
&  gprim,indsym,mpert,mpi_enreg,natom,nrpt,nsym,ntypat,&
&  rcan,rmet,rprim_in,rpt,symrec,symrel,tnons,trans,typat,ucvol,&
&  unitgkk,wghatm,xred,zeff)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use defs_elphon
 use m_header,          only : hdr_clean
 use m_io_tools
 use m_kptrank

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
 use interfaces_18_timing
 use interfaces_32_util
 use interfaces_42_geometry
 use interfaces_56_recipspace
 use interfaces_59_io_mpi
 use interfaces_62_occeig
 use interfaces_77_ddb, except_this_one => elphon
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: mpert,natom
 integer,intent(in) :: nrpt,nsym,ntypat
 integer,intent(in) :: unitgkk
 real(dp),intent(in) :: ucvol
 character(len=fnlen) :: elph_base_name
 character(len=fnlen),intent(in) :: ddkfilename
 type(MPI_type),intent(inout) :: mpi_enreg
 type(anaddb_dataset_type) :: anaddb_dtset
!arrays
 integer,intent(in) :: indsym(4,nsym,natom)
 integer,intent(in) :: symrec(3,3,nsym),symrel(3,3,nsym),typat(natom)
 real(dp),intent(in) :: acell_in(3),amu(ntypat),atmfrc(2,3,natom,3,natom,nrpt)
 real(dp),intent(in) :: dielt(3,3)
 real(dp),intent(in) :: dyewq0(3,3,natom),gmet(3,3),gprim(3,3)
 real(dp),intent(in) :: rcan(3,natom),rmet(3,3),rprim_in(3,3)
 real(dp),intent(in) :: rpt(3,nrpt),tnons(3,nsym),trans(3,natom)
 real(dp),intent(in) :: wghatm(natom,natom,nrpt),xred(3,natom),zeff(3,3,natom)

!Local variables-------------------------------
!scalars
 integer :: ikpt_phon,iatom, ikpt_fine,ierr
 integer :: iband,ibandp, ieliash,ii,ikpt
 integer :: iqpt,irpt,isppol,istat
 integer :: n1wf,nband
 integer :: neliash
 integer :: onegkksize
 integer :: timrev, unitfskgrid
 integer :: qtor
 integer :: idir
 integer :: iFSkpq, symrankkpt
 integer :: ep_prt_wtk ! eventually to be made into an input variable
 integer :: rdwr, fform

 real(dp) :: max_occ,realdp_ex
 real(dp) :: res,ss
 real(dp) :: tcpu, twall, tcpui, twalli
 logical :: make_gkk2,use_afm,use_tr
 character(len=500) :: message
 character(len=fnlen) :: fname
 type(elph_tr_type) :: elph_tr_ds
 type(elph_type) :: elph_ds
 type(hdr_type) :: hdr
 type(hdr_type) :: hdr1
 type(phon_type) :: phon_ds
!arrays
 integer :: qptrlatt(3,3)
 integer,allocatable :: FSfullpqtofull(:,:)
 integer,allocatable :: irredtoGS_phon(:)
 integer,allocatable :: irredtoGS_fine(:)
 integer,allocatable :: gkk_flag(:,:,:,:,:)
 integer,allocatable :: qpttoqpt(:,:,:)
 real(dp) :: acell(3)
 real(dp) :: ftwghtgkk(natom,nrpt),gprimd(3,3)
 real(dp) :: kpt(3),rprim(3,3)
 real(dp) :: rprimd(3,3),shiftk(3)
 real(dp),allocatable :: a2f_1d(:),delta(:,:),dos_phon(:)
 real(dp),allocatable :: dummy_eig(:,:,:)
 real(dp),allocatable :: eigenGS(:,:,:)
 real(dp),allocatable :: eigenGS_fine(:,:,:)
 real(dp),allocatable :: qptirred(:,:)
 real(dp),allocatable :: zz(:,:)

! *************************************************************************

 call timein(tcpui,twalli)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-begin elphon at tcpu',tcpui,'  and twall',twalli,' sec'
 call wrtout(std_out,message,'COLL')

!==================================
!Initialization of some variables
!==================================

 call elph_ds_nullify(elph_ds)

 call elph_tr_ds_nullify(elph_tr_ds)

 neliash = 10                     !maximum number of iterations to converge Tc
 elph_ds%mustar=anaddb_dtset%mustar            !input mustar
 elph_ds%nbranch = 3*natom        !number of phonon modes = 3 * natom
 elph_ds%ep_keepbands = anaddb_dtset%ep_keepbands  !flag to sum over bands
 elph_ds%a2fsmear = anaddb_dtset%a2fsmear   !smearing for Eliashber functions

 elph_ds%tuniformgrid = 1

 elph_ds%na2f = 400               !maximum number of Matsubara frequencies.
!The precise number used depends on the value of Tc:
!they span $w_n = (2n+1) \pi T_c$  where $abs(w_n) < w_{cutoff}$
!ie $|n| < n_{cutoff} = ( \frac{w_{cutoff}}{\pi T_c} ) / 2$
 write (*,*) ' elphon : na2f = ', elph_ds%na2f
 
!use time reversal symmetry always when possible for kpoint reduction,
!and suppose it has been used in WF generation
!not used for the moment: values are always taken from input files.
 timrev = 1

!save gkk data for full kpoints to file on disk

 elph_ds%gkqwrite=anaddb_dtset%gkqwrite
 elph_ds%gkk_rptwrite=anaddb_dtset%gkk_rptwrite
 elph_ds%gkk2write=anaddb_dtset%gkk2write

!This should never be turned off: symmetrization of elphon matrix elements
!in complete_gkk. See get_all_gkq
 elph_ds%symgkq=anaddb_dtset%symgkq

 elph_ds%elph_base_name = trim(elph_base_name)

!normalize input rprim and acell.
 do ii=1,3
   ss = sqrt(rprim_in(1,ii)**2+rprim_in(2,ii)**2+rprim_in(3,ii)**2)
   rprim(:,ii) = rprim_in(:,ii)/ss
   acell(ii) = acell_in(ii) * ss
 end do

!make dimension-ful rprimd and gprimd for transformation of derivatives to cartesian coordinates.
 call mkrdim(acell,rprim,rprimd)
 call matr3inv(rprimd,gprimd)

!===================
!Check some inputs
!===================

 if (nsym==1) then
   write (message,'(7a)')ch10,&
&   ' elphon: COMMENT- ',ch10,&
&   ' Symmetries are not used! ',ch10,&
&   ' Full matrix elements must be supplied for all perturbations and qpoints!',ch10
   call wrtout(std_out,message,'COLL')
   call wrtout(ab_out,message,'COLL')
   
   if (abs(tnons(1,1))+abs(tnons(2,1))+abs(tnons(3,1)) > tol10) then
     write (message,'(4a)')ch10,&
&     ' elphon : ERROR-',ch10,&
&     ' tnons should be (0,0,0) for unique symmetry Id'
     call wrtout(std_out,message,'COLL')
     call leave_new('COLL')
   end if
 end if

 if (anaddb_dtset%ifcflag/=1) then
   write(message,'(6a,i4)')ch10,&
&   ' elphon : ERROR-',ch10,&
&   ' ifcflag should be set to one,',ch10,&
&   ' the IFC matrices are supposed to exist.',anaddb_dtset%ifcflag
   call wrtout(std_out,message,'COLL')
   call wrtout(ab_out,message,'COLL')
 end if

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon begin setup after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

!=================================
!Set up the full grid of qpoints
!=================================
 call ep_setupqpt (anaddb_dtset,elph_ds,gmet,nsym,qptrlatt,rprimd,symrec,symrel,timrev)

!================================================================
!Summed weights for gkk FT. Is wghatm symmetric on the 2 atoms?
!present version does not use ftwghtgkk
!================================================================
 do iatom=1,natom
   do irpt=1,nrpt
     ftwghtgkk(iatom,irpt) = wghatm(iatom,iatom,irpt)
   end do
 end do

!====================================
!Read the GS header of the GKK file
!this will give the phon grid of k
!and the Fermi surface integration weights
!====================================
 
 write (message,'(2a)')ch10,' elphon : reading and checking the GS header of the GKK file'
 call wrtout (std_out,message,'COLL')

 call rchkGSheader(hdr,natom,nband,unitgkk)

 elph_ds%nsppol=hdr%nsppol
 elph_ds%nspinor=hdr%nspinor

!in spinor or spin polarized case, orbitals have occupation <= 1 instead of 2
 max_occ = one
 if (hdr%nspinor == 2) max_occ = half ! this accounts for the doubling of the num of bands, even though spin channels are not well defined
 if (elph_ds%nsppol > 1) max_occ = one
 write (*,*) ' max_occ factor  ', max_occ

 elph_ds%occ_factor = one
 if (hdr%nspinor == 1 .and. hdr%nsppol == 1) then
   elph_ds%occ_factor = one
 else if (hdr%nspinor == 2) then
   elph_ds%occ_factor = two
 else if (hdr%nsppol == 2) then
   elph_ds%occ_factor = one
 end if

!==================================================
!Set elph_ds%fermie: either comes from anaddb input file or from wfk file
!==================================================
 elph_ds%fermie = hdr%fermie
 if (abs(anaddb_dtset%elph_fermie) > tol10) then
   elph_ds%fermie = anaddb_dtset%elph_fermie
   write(message,'(a,f12.6)')' Fermi level set by the user at :',elph_ds%fermie
   call wrtout(std_out,message,'COLL')
 end if

!==================================================
!Read GS eigenvalues for each irreducible kpt and
!number of 1WF files contributing to the GKK file
!==================================================

 allocate(eigenGS(nband,hdr%nkpt,elph_ds%nsppol),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating eigenGS'
 do isppol=1,elph_ds%nsppol
   do ikpt=1,hdr%nkpt
     read(unitgkk) eigenGS(:,ikpt,isppol)
   end do
 end do

!read number of 1WF files contributing to the GKK file
 read(unitgkk) n1wf
 write(message,'(a,i4)')' elphon : number of perturbations in the gkk file = ',n1wf
 call wrtout(std_out,message,'COLL')


!====================================================================
!Setup of the phon k-grid :
!1) get bands near Ef
!====================================================================

 call get_fs_bands(eigenGS,elph_ds,hdr)

 if (elph_ds%ep_keepbands == 0) then !we are summing over bands
   elph_ds%ngkkband = 1
 else if (elph_ds%ep_keepbands == 1) then
!  keep the band dependency btw elph_ds%minFSband and elph_ds%maxFSband
   elph_ds%ngkkband = elph_ds%nFSband
 else
   write(message,'(4a,i4)')ch10,&
&   ' elphon : BUG- ',ch10,     &
&   ' ep_keepbands must be 0 or 1 while it is : ',elph_ds%ep_keepbands
   call wrtout(std_out,message,'COLL')
   call leave_new('COLL')
 end if
 write(message,'(a,i4,2x,i4)')' elphon : minFSband, maxFSband = ',elph_ds%minFSband,elph_ds%maxFSband
 call wrtout(std_out,message,'COLL')

 allocate(elph_ds%k_phon%kptirr(3,elph_ds%k_phon%nkptirr),stat=istat)
 if (istat /= 0) then
   write (message, '(a)') 'elphon: error in allocating elph_ds%k_phon%kptirr'
   call wrtout(std_out,message,'COLL')
   call leave_new('COLL')
 end if
 allocate(irredtoGS_phon(elph_ds%k_phon%nkptirr),stat=istat)
 if (istat /= 0) then
   write (message, '(a)') 'elphon: error in allocating irredtoGS_phon'
   call wrtout(std_out,message,'COLL')
   call leave_new('COLL')
 end if

!====================================================================
!2) order irred k-points 
!====================================================================
 call order_fs_kpts(elph_ds%k_phon%kptirr,irredtoGS_phon,&
& hdr,elph_ds%k_phon%nkptirr)

!==========================================
!3) reconstruct full kgrid from irred kpoints,
!==========================================
 call mkFSkgrid (elph_ds%k_phon, nsym, symrec, timrev) 

!====================================================================
!4) setup weights for integration (gaussian or tetrahedron method) 
!====================================================================
 allocate(elph_ds%k_phon%wtk(elph_ds%nFSband,elph_ds%k_phon%nkpt,elph_ds%nsppol),stat=istat)
 call ep_fs_weights(anaddb_dtset%ep_b_min, anaddb_dtset%ep_b_max, eigenGS, anaddb_dtset%elphsmear, &
& elph_ds%fermie, gprimd, irredtoGS_phon, anaddb_dtset%kptrlatt, max_occ, elph_ds%minFSband, nband, elph_ds%nFSband, &
& elph_ds%nsppol, anaddb_dtset%telphint, elph_ds%k_phon)

!=====================================================
!get kpt info from the fine grid part 
!=====================================================
 if (anaddb_dtset%ep_alter_int_gam == 0) then
!  simply copy over _phon variables
   elph_ds%k_fine%nkpt = elph_ds%k_phon%nkpt
   elph_ds%k_fine%nkptirr = elph_ds%k_phon%nkptirr

   allocate(elph_ds%k_fine%kptirr(3,elph_ds%k_fine%nkptirr),stat=istat)
   elph_ds%k_fine%kptirr = elph_ds%k_phon%kptirr
   allocate(elph_ds%k_fine%wtkirr(elph_ds%k_fine%nkptirr))
   elph_ds%k_fine%wtkirr = elph_ds%k_phon%wtkirr
   allocate(irredtoGS_fine(elph_ds%k_fine%nkptirr),stat=istat)
   irredtoGS_fine = irredtoGS_phon

   allocate(elph_ds%k_fine%wtk(elph_ds%nFSband,elph_ds%k_fine%nkpt,elph_ds%nsppol),stat=istat)
   elph_ds%k_fine%wtk = elph_ds%k_phon%wtk
   allocate(elph_ds%k_fine%kpt(3,elph_ds%k_fine%nkpt),stat=istat)
   elph_ds%k_fine%kpt = elph_ds%k_phon%kpt 

   call copy_kptrank(elph_ds%k_phon%kptrank_t, elph_ds%k_fine%kptrank_t)

   allocate (elph_ds%k_fine%irr2full(elph_ds%k_fine%nkptirr), stat=istat)
   elph_ds%k_fine%irr2full = elph_ds%k_phon%irr2full
   allocate(elph_ds%k_fine%full2irr(3,elph_ds%k_fine%nkpt),stat=istat)
   elph_ds%k_fine%full2irr = elph_ds%k_phon%full2irr
   allocate(elph_ds%k_fine%full2full(2,nsym,elph_ds%k_fine%nkpt),stat=istat)
   elph_ds%k_fine%full2full = elph_ds%k_phon%full2full

 else
!  read in the first header for the gkk part
   unitfskgrid = get_unit()
   open (unit=unitfskgrid,file='finegrid_GKK',form='unformatted',status='old')
   rewind(unitfskgrid)
   rdwr = 5 !read in header of file without rewinding it
   call hdr_io(fform,hdr1,rdwr,unitfskgrid)
   if (fform == 0) then
     write (message,'(4a)')ch10,' elphon : ERROR- :',ch10,&
&     ' fine grid GKK header was mis-read. fform == 0'
     call wrtout(std_out,message,'COLL')
     call leave_new('COLL')
   end if

   allocate(eigenGS_fine(nband,hdr1%nkpt,elph_ds%nsppol),stat=istat)
   if (istat /= 0) stop 'elphon: error in allocating eigenGS_fine'
   do isppol=1,elph_ds%nsppol
     do ikpt=1,hdr1%nkpt
       read(unitfskgrid) eigenGS_fine(:,ikpt,isppol)
     end do
   end do

   elph_ds%k_fine%nkptirr = hdr1%nkpt
   allocate(elph_ds%k_fine%kptirr(3,elph_ds%k_fine%nkptirr),stat=istat)
   allocate(irredtoGS_fine(elph_ds%k_fine%nkptirr),stat=istat)

   call order_fs_kpts(elph_ds%k_fine%kptirr,irredtoGS_fine,&
&   hdr1,elph_ds%k_fine%nkptirr)

   call mkFSkgrid (elph_ds%k_fine, nsym, symrec, timrev)

   allocate(elph_ds%k_fine%wtk(elph_ds%nFSband,elph_ds%k_fine%nkpt,elph_ds%nsppol),stat=istat)
   if (istat /= 0) stop 'elphon: error in allocating elph_ds%k_fine%wtk'
   call ep_fs_weights(anaddb_dtset%ep_b_min, anaddb_dtset%ep_b_max, eigenGS_fine, anaddb_dtset%elphsmear, &
&   elph_ds%fermie, gprimd, irredtoGS_fine, anaddb_dtset%kptrlatt, max_occ, elph_ds%minFSband, nband, elph_ds%nFSband, &
&   elph_ds%nsppol, anaddb_dtset%telphint, elph_ds%k_fine)

   deallocate (eigenGS_fine)

 end if ! alter_int_gam

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon k and q grids have been setup after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

!====================================================================
!5) calculate DOS at Ef
!====================================================================
 allocate(elph_ds%n0(elph_ds%nsppol),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating elph_ds%n0'

!SPPOL sum over spin channels to get total DOS
!channels decoupled => use separate values for DOS_up(Ef) resp down
 do isppol=1,elph_ds%nsppol
   elph_ds%n0(isppol) = sum(elph_ds%k_fine%wtk(:,:,isppol))/elph_ds%k_fine%nkpt
 end do

 if (elph_ds%nsppol == 1) then
   write (*,*) ' elphon : the estimated DOS(E_Fermi) = ', elph_ds%n0(1), ' states/Ha/spin '
   write (*,*) ' elphon : the total FS weight and # of kpoints = ',sum(elph_ds%k_fine%wtk),elph_ds%k_fine%nkpt
 else if (elph_ds%nsppol == 2) then
   write (*,*) ' elphon : the spin up   DOS(E_Fermi) = ', elph_ds%n0(1), ' states/Ha/spin '
   write (*,*) ' elphon : the spin down DOS(E_Fermi) = ', elph_ds%n0(2), ' states/Ha/spin '
   write (*,*) ' elphon : total DOS(E_Fermi) = ', elph_ds%n0(1)+elph_ds%n0(2), ' states/Ha '
   write (*,*) ' elphon : the spin up   FS weight and # of kpoints = ',&
&   sum(elph_ds%k_fine%wtk(:,:,1)),elph_ds%k_fine%nkpt
   write (*,*) ' elphon : the spin down FS weight and # of kpoints = ',&
&   sum(elph_ds%k_fine%wtk(:,:,2)),elph_ds%k_fine%nkpt
 else
   write (*,*) 'bad value for nsppol ', elph_ds%nsppol
   stop
 end if

 allocate(elph_ds%gkk_intweight(elph_ds%ngkkband,elph_ds%k_fine%nkpt,elph_ds%nsppol),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating elph_ds%gkk_intweight'

 if (elph_ds%ep_keepbands == 0) then
!  use trivial integration weights  for single band,
!  since average over bands is done in normsq_gkk
   elph_ds%gkk_intweight(1,:,:) = one

 else if (elph_ds%ep_keepbands == 1) then
!  use elph_ds%k_fine%wtk since average over bands is not done in normsq_gkk
   elph_ds%gkk_intweight(:,:,:) = elph_ds%k_fine%wtk(:,:,:)
 else
   write(message,'(4a,i4)')ch10,' elphon : BUG- ',ch10,            &
&   ' ep_keepbands must be 0 or 1 while it is : ',elph_ds%ep_keepbands
   call wrtout(std_out,message,'COLL')
   call leave_new('COLL')
 end if

 ep_prt_wtk = 0
 if (ep_prt_wtk == 1) then
   do iband=1, elph_ds%ngkkband
     do ikpt_fine=1, elph_ds%k_fine%nkpt
       write (300,*) ikpt_fine, elph_ds%gkk_intweight(iband,ikpt_fine,1)
     end do
   end do
 end if


 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon weights and DOS setup after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

!Output of the Fermi Surface
 if (anaddb_dtset%prtfsurf == 1) then
   fname=trim(elph_ds%elph_base_name) // '_BXSF'

!  FIXME
!  shiftk is defined neither in the anaddb nor in the hdr data type
!  an incorrect FS will be produced in case of a shifted k-grid used during the GS calculation
!  check if we are using a unshifthed kgrid, obviously doesnt work in case
!  of multiple shifts containg a zero translation but in this case prtbxsf should work
   shiftk=one
   do ii=1,hdr%nkpt
     if (all(hdr%kptns(:,ii) == zero)) shiftk=zero
   end do

!  the input argument in printbxsf has shape (nband,nkpt,nspin), to be on the safe side we use:
   allocate (dummy_eig(nband,hdr%nkpt,elph_ds%nsppol),stat=istat)
   if (istat /= 0) stop 'elphon: error in allocating dummy_eig'
   dummy_eig(:,:,:) = eigenGS(:,:,:)

   use_afm=(hdr%nsppol==1.and.hdr%nspden==2)
!  MG FIXME warning time reversal is always assumed to be present. 
!  the header should report this information.

   use_tr=(timrev==1)

   call printbxsf(dummy_eig,zero,elph_ds%fermie,gprimd,&
&   anaddb_dtset%kptrlatt,nband,hdr%nkpt,hdr%kptns,&
&   nsym,use_afm,symrec,hdr%symafm,use_tr,elph_ds%nsppol,shiftk,1,fname,ierr)

   deallocate (dummy_eig)
 end if !anaddb_dtset%prtfsurf

 deallocate (eigenGS)

!=========================================================    
!Get equivalence between a kpt_phon pair and a qpt in qpt_full
!only works if the qpt grid is complete (identical to
!the kpt one, with a basic shift of (0,0,0)
!=========================================================    

 allocate(FSfullpqtofull(elph_ds%k_phon%nkpt,elph_ds%nqpt_full),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating FSfullpqtofull'
 allocate(qpttoqpt(2,nsym,elph_ds%nqpt_full),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating qpttoqpt'

 write (message,'(2a)')ch10,' elphon : calling mkqptequiv to set up the FS qpoint set'
 call wrtout(std_out,message,'COLL')

 call mkqptequiv (FSfullpqtofull,elph_ds%k_phon%kpt,elph_ds%k_phon%nkpt,&
& elph_ds%nqpt_full,nsym,qpttoqpt,elph_ds%qpt_full,symrec)


!==========================================
!Set up dataset for phonon interpolations
!==========================================

 call setup_phon_ds(phon_ds,anaddb_dtset%dipdip,mpert,nsym,natom,ntypat,nrpt,&
& ucvol,indsym,symrel,typat,acell,amu,atmfrc,dielt,dyewq0,gprim,gmet,&
& xred,zeff,rcan,rmet,rprim,rprimd,rpt,trans,wghatm)

!transfer ifltransport flag to structure
 elph_tr_ds%ifltransport=anaddb_dtset%ifltransport
!transfer name of files file for ddk
 elph_tr_ds%ddkfilename=ddkfilename


!reduce qpt_full to correct zone
 do iqpt=1,elph_ds%nqpt_full
   call wrap2_pmhalf(elph_ds%qpt_full(1,iqpt),kpt(1),res)
   call wrap2_pmhalf(elph_ds%qpt_full(2,iqpt),kpt(2),res)
   call wrap2_pmhalf(elph_ds%qpt_full(3,iqpt),kpt(3),res)
   elph_ds%qpt_full(:,iqpt)=kpt
 end do

!test density of k+q grid: the following should be close to n0 squared
!FIXME: generalize for sppol
 res = zero
 do ikpt_fine = 1, elph_ds%k_fine%nkpt
   do iqpt = 1, elph_ds%nqpt_full
     kpt = elph_ds%k_fine%kpt(:,ikpt_fine) + elph_ds%qpt_full(:,iqpt)
     call get_rank_1kpt (kpt,symrankkpt,elph_ds%k_fine%kptrank_t)
     iFSkpq = elph_ds%k_fine%kptrank_t%invrank(symrankkpt)
     do iband = 1, elph_ds%ngkkband
       do ibandp = 1, elph_ds%ngkkband
         res = res + elph_ds%gkk_intweight(iband,ikpt_fine,1)*elph_ds%gkk_intweight(ibandp,iFSkpq,1)
       end do
     end do 
   end do
 end do
 res = res / elph_ds%k_fine%nkpt/elph_ds%k_fine%nkpt
 write (*,*) 'elphon: integrated value of intweight for given k and q grid : ', res, res / elph_ds%n0(1)**2

 res = zero
 do ikpt_fine = 1, elph_ds%k_fine%nkpt
   do iqpt = 1, elph_ds%k_fine%nkpt
     kpt = elph_ds%k_fine%kpt(:,ikpt_fine) + elph_ds%k_fine%kpt(:,iqpt)
     call get_rank_1kpt (kpt,symrankkpt,elph_ds%k_fine%kptrank_t)
     iFSkpq = elph_ds%k_fine%kptrank_t%invrank(symrankkpt)
     do iband = 1, elph_ds%ngkkband
       do ibandp = 1, elph_ds%ngkkband
         res = res + elph_ds%gkk_intweight(iband,ikpt_fine,1)*elph_ds%gkk_intweight(ibandp,iFSkpq,1)
       end do
     end do
   end do
 end do
 res = res / elph_ds%k_fine%nkpt/elph_ds%k_fine%nkpt
 write (*,*) 'elphon: integrated value of intweight for double k grid : ', res, res / elph_ds%n0(1)**2


!===================================================
!Allocate all important arrays for FS integrations
!===================================================

 allocate(gkk_flag(elph_ds%nbranch,elph_ds%nbranch,elph_ds%k_phon%nkpt,&
& elph_ds%nsppol,elph_ds%nqpt_full),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating gkk_flag'

!Record sizes for matrices on disk: complex and real versions (for real and recip space resp!)
 onegkksize = 2*elph_ds%nbranch*elph_ds%nbranch*&
& elph_ds%ngkkband*elph_ds%ngkkband*&
& elph_ds%nsppol*kind(realdp_ex)
 elph_tr_ds%onegkksize=onegkksize

 allocate (qptirred(3,n1wf),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating qptirred'

 write (message,'(5a)')                         &
& ' elphon : preliminary setup completed ',ch10,&
& '          calling get_all_gkq to read in',&
& ' all the e-ph matrix elements',ch10
 call wrtout(std_out,message,'COLL')

!flag to do scalar product in gkq before interpolation:
!should also used in interpolate_gkk and mkph_linwid
 elph_ds%ep_scalprod=anaddb_dtset%ep_scalprod
 if (elph_ds%ep_scalprod==0) then
   write (*,*) ' elphon: will NOT perform scalar product with phonon'
   write (*,*) '  displacement vectors in read_gkk. ep_scalprod==0'
 else if (elph_ds%ep_scalprod==1) then
   write (*,*) ' elphon: will perform scalar product with phonon'
   write (*,*) '  displacement vectors in read_gkk. ep_scalprod==1'
 else
   write (*,*) ' elphon: illegal value for ep_scalprod'
   stop
 end if

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon begin gkq construction after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

 call get_all_gkq (elph_ds,FSfullpqtofull,&
& gkk_flag,gprimd,indsym,natom,&
& nband,nsym,n1wf,onegkksize,phon_ds,&
& qpttoqpt,rprimd,symrec,symrel,tnons,&
& anaddb_dtset%ep_prt_yambo,&
& unitgkk)


 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon end gkq construction after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

 if (elph_tr_ds%ifltransport==1 )then
   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon begin gkq_tr construction after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall

   call get_veloc_tr(elph_ds,mpi_enreg,nband,elph_tr_ds)

   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon end gkq_tr construction after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall
 end if

!============================================================================
!Evaluate lambda and omega_log using the weighted sum over the irred q-points
!found in the GKK file. All the data we need are stored in elph_ds%qgrid_data
!============================================================================
 write(message,'(a)')' elphon: calling outelph '
 call wrtout(std_out,message,'COLL')
 fname=trim(elph_ds%elph_base_name) // '_QPTS'
 call outelph(elph_ds,anaddb_dtset%enunit,fname)

!========================================================
!Get FS averaged gamma matrices and Fourier transform to real space 
!========================================================
 
 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon begin integration of gkq after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

 if (anaddb_dtset%ep_alter_int_gam == 1) then
   call integrate_gamma_alt(elph_ds,elph_tr_ds,gprim,gprimd,indsym,anaddb_dtset%kptrlatt,&
&   natom,nrpt,nsym,qpttoqpt,rprimd,rpt,symrec,symrel,wghatm)

 else
!  NOTE: gprim (not gprimd) is used for all FT interpolations,
!  to be consistent with the dimensions of the rpt, which come from anaddb.
   call integrate_gamma(elph_ds,FSfullpqtofull,nrpt)

   if (elph_ds%symgkq ==1) then
!    complete the gamma_qpt here instead of the gkk previously
     call complete_gamma(elph_ds,elph_ds%gamma_qpt, &
&     gprimd,indsym,natom,nsym,qpttoqpt,rprimd, &
&     symrec,symrel)
   end if

!  Now FT to real space too
   qtor = 1 ! q --> r
   do isppol=1,elph_ds%nsppol
     call ftgam(wghatm,elph_ds%gamma_qpt(:,:,isppol,:),elph_ds%gamma_rpt(:,:,isppol,:),gprim,natom,&
&     elph_ds%nqpt_full,nrpt,qtor,rpt,elph_ds%qpt_full)
   end do
 end if

!DEBUG
 if (anaddb_dtset%ep_alter_int_gam == 1) then
   do ii=1, elph_ds%nbranch*elph_ds%nbranch
     write (100,*) ' ibr gamma ', ii, elph_ds%gamma_qpt(1,ii,1,:)
     write (100,*)
   end do
   do ikpt_phon=1, elph_ds%k_phon%nkpt
     write (200,'(I6,3e16.6,I12)') ikpt_phon, elph_ds%k_phon%kpt(:,ikpt_phon),&
&     elph_ds%k_phon%kptrank_t%rank(ikpt_phon)
   end do 
 end if
!ENDDEBUG

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon end integration and completion of gkq after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall


!==========================================================
!calculate transport matrix elements, integrated over FS
!in ep_alter_int_gam == 1 case the gamma_qpt_trout are
!already calculated in integrate_gamma_alt
!==========================================================
 if (elph_tr_ds%ifltransport==1 .and. anaddb_dtset%ep_alter_int_gam == 0 )then
   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon begin integrate gkq_tr after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall

   call integrate_gamma_tr(elph_ds,FSfullpqtofull,nrpt,elph_tr_ds)

   do idir=1,9
     call complete_gamma(elph_ds,elph_tr_ds%gamma_qpt_trout(:,idir,:,:,:), &
&     gprimd,indsym,natom,nsym,qpttoqpt,rprimd, &
&     symrec,symrel)

     call complete_gamma(elph_ds,elph_tr_ds%gamma_qpt_trin(:,idir,:,:,:), &
&     gprimd,indsym,natom,nsym,qpttoqpt,rprimd, &
&     symrec,symrel)
   end do

!  Now FT to real space too
   qtor = 1 ! q --> r
   do isppol=1,elph_ds%nsppol
     do idir=1,9
       call ftgam(wghatm,elph_tr_ds%gamma_qpt_trout(:,idir,:,isppol,:),&
&       elph_tr_ds%gamma_rpt_trout(:,idir,:,isppol,:),gprim,natom,&
&       elph_ds%nqpt_full,nrpt,qtor,rpt,elph_ds%qpt_full)

       call ftgam(wghatm,elph_tr_ds%gamma_qpt_trin(:,idir,:,isppol,:),&
&       elph_tr_ds%gamma_rpt_trin(:,idir,:,isppol,:),gprim,natom,&
&       elph_ds%nqpt_full,nrpt,qtor,rpt,elph_ds%qpt_full)
     end do
   end do

   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon end integrate gkq_tr after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall
 end if

!==============================================================
!Calculate phonon linewidths, interpolating on chosen qpoints
!==============================================================

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon begin linewidths after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

 call mkph_linwid(anaddb_dtset%ep_alter_int_gam, elph_ds,&
& gprim,gprimd,gmet,anaddb_dtset%kptrlatt_fine,natom,nrpt,anaddb_dtset%nqpath,&
& phon_ds,anaddb_dtset%qpath,rpt,wghatm)

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon end linewidths after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

!==============================================================
!the nesting factor calculation
!FIXME: this could go higher up, before the call to get_all_gkq
!you only need the kpt and weight info
!==============================================================
 if (anaddb_dtset%prtnest==1 .or. anaddb_dtset%prtnest==2) then
   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon begin mknesting after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall

   write(message,'(a)')&
&   ' elphon : calling mknesting to interpolate the nesting factor'
   call wrtout(std_out,message,'COLL')

   call mknesting(elph_ds%k_phon%nkpt,elph_ds%k_phon%kpt,anaddb_dtset%kptrlatt,elph_ds%nFSband,&
&   elph_ds%k_phon%wtk,anaddb_dtset%nqpath,anaddb_dtset%qpath,elph_ds%nqpt_full, &
&   elph_ds%qpt_full,elph_ds%elph_base_name,&
&   gprimd,gmet,anaddb_dtset%prtnest,qptrlatt)

   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon end mknesting after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall
 end if
 
!======================================================
!Calculate alpha^2 F integrating over fine kpt_phon grid
!======================================================

 allocate(a2f_1d(elph_ds%na2f),dos_phon(elph_ds%na2f),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating a2f_1d,dos_phon'
 
 write(message,'(a)')' elphon : calling mka2f '
 call wrtout(std_out,message,'COLL')

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon begin mka2f after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

 call mka2f(anaddb_dtset%ep_alter_int_gam, a2f_1d,dos_phon,elph_ds,&
& gprim,gprimd,anaddb_dtset%kptrlatt_fine,anaddb_dtset%mustar,&
& natom,nrpt,phon_ds,rpt,wghatm)

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon end mka2f after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

!calculate transport spectral function and coefficients
 if (elph_tr_ds%ifltransport==1 )then
   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon begin mka2f_tr after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall

   call mka2f_tr(anaddb_dtset%ep_alter_int_gam, elph_ds,gprim,gprimd,ucvol,natom,nrpt,&
&   anaddb_dtset%ntemper,anaddb_dtset%tempermin,anaddb_dtset%temperinc,&
&   phon_ds,rpt,wghatm,elph_tr_ds)

   call timein(tcpu,twall)
   write(message, '(a,f11.3,a,f11.3,a)' )&
&   '-elphon end mka2f_tr after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
   call wrtout(std_out,message,'COLL')
   tcpui = tcpu
   twalli = twall

 end if

!evaluate a2F only using the input Q-grid (without using interpolated matrices)
!SCOPE: test the validity of the Fourier interpolation
 write(message,'(a)')' elphon : calling mka2fQgrid'
 call wrtout(std_out,message,'COLL')
 fname=trim(elph_ds%elph_base_name) // '_A2F_QGRID'
 call mka2fQgrid(elph_ds,fname)

!=============================================
!Eliashberg equation in 1-D (isotropic case)
!=============================================

 write (message,'(2a)')ch10,&
& ' Solving the 1-D Eliashberg equation (isotropic case)'
 call wrtout(std_out,message,'COLL')

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon begin eliashberg_1d after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

 call eliashberg_1d(a2f_1d,elph_ds,anaddb_dtset%mustar)

 call timein(tcpu,twall)
 write(message, '(a,f11.3,a,f11.3,a)' )&
& '-elphon end eliashberg_1d after tcpu',tcpu-tcpui,'  and twall',twall-twalli,' sec'
 call wrtout(std_out,message,'COLL')
 tcpui = tcpu
 twalli = twall

 deallocate (a2f_1d,dos_phon)

!MJV: 20070805 should exit here. None of the rest is tested or used yet to my knowledge

!========================================================================
!Now gkk contains the matrix elements of dH(1)/dxi i=1,2,3
!for kpoints on the FS but qpoints only in the given grid {Q}.
!
!1.) Need to complete the gkk elements for q and k\prime=k+q not 
!in the set of {k+Q} by Fourier interpolation on the Q.
!
!2.) Need to complete the dynamical matrices and phonon freqs for
!all q between points on the FS.
!
!3.) With the eigenvectors e_ph of the dyn mats, do the scalar product
!e_ph . gkk, which implies the gkk are turned to the eigenbasis of
!the phonons. Before the (non eigen-) modes are ordered
!atom1 xred1 atom1 xred2 atom1 xred3
!atom2 xred1 atom2 xred2 atom2 xred3 ...
!=======================================================================

 make_gkk2=.false.
 
 if (.not. make_gkk2) then
   write(message,'(2a)')ch10,&
&   ' elphon : skipping full g(k,k") interpolation '
   call wrtout(std_out,message,'COLL')
 else

!  ==========================================================
!  FT of recip space gkk matrices to real space (gkk_rpt)
!  NOTE: could be made into FFT, couldnt it? If shifts are
!  used with a homogeneous grid
!  ==========================================================
   write (message,'(2a,i5)')ch10,&
&   ' elphon : Fourier transform (q --> r) of the gkk matrices using nrpt = ',nrpt
   call wrtout(std_out,message,'COLL')

   call get_all_gkr(elph_ds,gprim,natom,nrpt,onegkksize,rpt,elph_ds%qpt_full,wghatm)

!  =========================================================
!  complete gkk2 for all qpts between points
!  on full kpt grid (interpolation from real space values)
!  =========================================================

   write(message,'(2a)')ch10,&
&   ' elphon : Calling get_all_gkk2 to calculate gkk2 for q points over the full k grid'
   call wrtout(std_out,message,'COLL')
   
   call get_all_gkk2(elph_ds,elph_ds%k_phon%kptirr,elph_ds%k_phon%kpt,   &
&   natom,nrpt, &
&   phon_ds,rcan,&
&   wghatm)
 end if


 allocate(zz(elph_ds%na2f,elph_ds%k_phon%nkpt),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating zz'
 allocate(delta(elph_ds%na2f,elph_ds%k_phon%nkpt),stat=istat)
 if (istat /= 0) stop 'elphon: error in allocating delta'
 
!=====================================================
!Here should be the anisotropic Eliashberg equations.
!=====================================================

!initialize delta function

!initialize T_c

!initialize delta

!iterate for calculation of T_c
 do ieliash=1,neliash

!  ===========================
!  calculate lambda function
!  ===========================

!  ========================================
!  integrate lambda over FS -> Z function
!  ========================================
   
!  ========================================
!  integrate delta*Z over FS -> new delta
!  ========================================
   
!  update T_c

 end do

!end iterate ieliash

!output T_c and related quantities


!clean and deallocate junk

 call elph_ds_clean(elph_ds)
 call elph_tr_ds_clean(elph_tr_ds)

 call clean_phon_ds(phon_ds)

 call hdr_clean(hdr)

 end subroutine elphon
!!***

!!****f* ABINIT/elph_ds_clean
!!
!! NAME
!!   elph_ds_clean
!!
!! FUNCTION
!!   deallocate remaining arrays in the elph_ds datastructure
!!
!! COPYRIGHT
!! Copyright (C) 2010 ABINIT group (MVer)
!! This file is distributed under the terms of the
!! GNU General Public Licence, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  elph_ds = elphon datastructure
!!
!! OUTPUT
!!
!! NOTES
!!
!! PARENTS
!!      elphon
!!
!! CHILDREN
!!
!! NOTES
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine elph_ds_clean(elph_ds)

 use defs_elphon
 use m_kptrank

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(elph_type), intent(inout) :: elph_ds

! *************************************************************************

 if (associated(elph_ds%qirredtofull)) deallocate(elph_ds%qirredtofull)
 if (associated(elph_ds%wtq)) deallocate(elph_ds%wtq)
 if (associated(elph_ds%n0)) deallocate(elph_ds%n0)
 if (associated(elph_ds%qpt_full)) deallocate(elph_ds%qpt_full)
 if (associated(elph_ds%gkk_intweight)) deallocate(elph_ds%gkk_intweight)
 if (associated(elph_ds%gkk_qpt)) deallocate(elph_ds%gkk_qpt)
 if (associated(elph_ds%gkk_rpt)) deallocate(elph_ds%gkk_rpt)
 if (associated(elph_ds%gkk2)) deallocate(elph_ds%gkk2)
 if (associated(elph_ds%gamma_qpt)) deallocate(elph_ds%gamma_qpt)
 if (associated(elph_ds%gamma_rpt)) deallocate(elph_ds%gamma_rpt)
 if (associated(elph_ds%phfrq)) deallocate(elph_ds%phfrq)
 if (associated(elph_ds%a2f)) deallocate(elph_ds%a2f)
 if (associated(elph_ds%qgrid_data)) deallocate(elph_ds%qgrid_data)

 if (associated(elph_ds%k_phon%wtk)) deallocate(elph_ds%k_phon%wtk)
 if (associated(elph_ds%k_phon%wtkirr)) deallocate(elph_ds%k_phon%wtkirr)
 if (associated(elph_ds%k_phon%kpt)) deallocate(elph_ds%k_phon%kpt)
 if (associated(elph_ds%k_phon%kptirr)) deallocate(elph_ds%k_phon%kptirr)
 if (associated(elph_ds%k_phon%irr2full)) deallocate(elph_ds%k_phon%irr2full)
 if (associated(elph_ds%k_phon%full2irr)) deallocate(elph_ds%k_phon%full2irr)
 if (associated(elph_ds%k_phon%full2full)) deallocate(elph_ds%k_phon%full2full)
 call destroy_kptrank (elph_ds%k_phon%kptrank_t)

 if (associated(elph_ds%k_fine%wtk)) deallocate(elph_ds%k_fine%wtk)
 if (associated(elph_ds%k_fine%wtkirr)) deallocate(elph_ds%k_fine%wtkirr)
 if (associated(elph_ds%k_fine%kpt)) deallocate(elph_ds%k_fine%kpt)
 if (associated(elph_ds%k_fine%kptirr)) deallocate(elph_ds%k_fine%kptirr)
 if (associated(elph_ds%k_fine%irr2full)) deallocate(elph_ds%k_fine%irr2full)
 if (associated(elph_ds%k_fine%full2irr)) deallocate(elph_ds%k_fine%full2irr)
 if (associated(elph_ds%k_fine%full2full)) deallocate(elph_ds%k_fine%full2full)
 call destroy_kptrank (elph_ds%k_fine%kptrank_t)

end subroutine elph_ds_clean
!!***

!!****f* ABINIT/elph_tr_ds_clean
!!
!! NAME
!!   elph_tr_ds_clean
!!
!! FUNCTION
!!   deallocate remaining arrays in the elph_tr_ds datastructure
!!
!! COPYRIGHT
!! Copyright (C) 2010 ABINIT group (MVer)
!! This file is distributed under the terms of the
!! GNU General Public Licence, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  elph_tr_ds = elphon transport datastructure
!!
!! OUTPUT
!!
!! NOTES
!!
!! PARENTS
!!      elphon
!!
!! CHILDREN
!!
!! NOTES
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine elph_tr_ds_clean(elph_tr_ds)

 use defs_elphon

 implicit none 

!Arguments ------------------------------------
!scalars
 type(elph_tr_type), intent(inout) :: elph_tr_ds
 
! *************************************************************************

 if (associated(elph_tr_ds%el_veloc)) deallocate(elph_tr_ds%el_veloc)
 if (associated(elph_tr_ds%eta_trin)) deallocate(elph_tr_ds%eta_trin)
 if (associated(elph_tr_ds%eta_trout)) deallocate(elph_tr_ds%eta_trout)
 if (associated(elph_tr_ds%gamma_qpt_tr)) deallocate(elph_tr_ds%gamma_qpt_tr)
 if (associated(elph_tr_ds%gamma_qpt_trin)) deallocate(elph_tr_ds%gamma_qpt_trin)
 if (associated(elph_tr_ds%gamma_qpt_trout)) deallocate(elph_tr_ds%gamma_qpt_trout)
 if (associated(elph_tr_ds%gamma_rpt_tr)) deallocate(elph_tr_ds%gamma_rpt_tr)
 if (associated(elph_tr_ds%gamma_rpt_trin)) deallocate(elph_tr_ds%gamma_rpt_trin)
 if (associated(elph_tr_ds%gamma_rpt_trout)) deallocate(elph_tr_ds%gamma_rpt_trout)
 if (associated(elph_tr_ds%a2f_1d_tr)) deallocate(elph_tr_ds%a2f_1d_tr)
 if (associated(elph_tr_ds%a2f_1d_trin)) deallocate(elph_tr_ds%a2f_1d_trin)
 if (associated(elph_tr_ds%a2f_1d_trout)) deallocate(elph_tr_ds%a2f_1d_trout)


end subroutine elph_tr_ds_clean
!!***

!!****f* ABINIT/elph_ds_nullify
!!
!! NAME
!!   elph_ds_nullify
!!
!! FUNCTION
!!   nullify all arrays in the elph_ds datastructure
!!
!! COPYRIGHT
!! Copyright (C) 2010 ABINIT group (MVer)
!! This file is distributed under the terms of the
!! GNU General Public Licence, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  elph_ds = elphon datastructure
!!
!! OUTPUT
!!
!! NOTES
!!
!! PARENTS
!!      elphon
!!
!! CHILDREN
!!
!! NOTES
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine elph_ds_nullify(elph_ds)

 use defs_elphon

 implicit none

!Arguments ------------------------------------
!scalars
 type(elph_type), intent(inout) :: elph_ds

! *************************************************************************

 nullify(elph_ds%qirredtofull)
 nullify(elph_ds%wtq)
 nullify(elph_ds%n0)
 nullify(elph_ds%qpt_full)
 nullify(elph_ds%gkk_intweight)
 nullify(elph_ds%gkk_qpt)
 nullify(elph_ds%gkk_rpt)
 nullify(elph_ds%gkk2)
 nullify(elph_ds%gamma_qpt)
 nullify(elph_ds%gamma_rpt)
 nullify(elph_ds%phfrq)
 nullify(elph_ds%a2f)
 nullify(elph_ds%qgrid_data)

 nullify(elph_ds%k_phon%wtk)
 nullify(elph_ds%k_phon%wtkirr)
 nullify(elph_ds%k_phon%kpt)
 nullify(elph_ds%k_phon%kptirr)
 nullify(elph_ds%k_phon%irr2full)
 nullify(elph_ds%k_phon%full2irr)
 nullify(elph_ds%k_phon%full2full)

 nullify(elph_ds%k_fine%wtk)
 nullify(elph_ds%k_fine%wtkirr)
 nullify(elph_ds%k_fine%kpt)
 nullify(elph_ds%k_fine%kptirr)
 nullify(elph_ds%k_fine%irr2full)
 nullify(elph_ds%k_fine%full2irr)
 nullify(elph_ds%k_fine%full2full)

end subroutine elph_ds_nullify
!!***

!!****f* ABINIT/elph_tr_ds_nullify
!!
!! NAME
!!   elph_tr_ds_nullify
!!
!! FUNCTION
!!   deallocate remaining arrays in the elph_tr_ds datastructure
!!
!! COPYRIGHT
!! Copyright (C) 2010 ABINIT group (MVer)
!! This file is distributed under the terms of the
!! GNU General Public Licence, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  elph_tr_ds = elphon transport datastructure
!!
!! OUTPUT
!!
!! NOTES
!!
!! PARENTS
!!      elphon
!!
!! CHILDREN
!!
!! NOTES
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine elph_tr_ds_nullify(elph_tr_ds)

 use defs_elphon

 implicit none


!Arguments ------------------------------------
!scalars
 type(elph_tr_type), intent(inout) :: elph_tr_ds
 
! *************************************************************************

 nullify(elph_tr_ds%el_veloc)
 nullify(elph_tr_ds%eta_trin)
 nullify(elph_tr_ds%eta_trout)
 nullify(elph_tr_ds%gamma_qpt_tr)
 nullify(elph_tr_ds%gamma_qpt_trin)
 nullify(elph_tr_ds%gamma_qpt_trout)
 nullify(elph_tr_ds%gamma_rpt_tr)
 nullify(elph_tr_ds%gamma_rpt_trin)
 nullify(elph_tr_ds%gamma_rpt_trout)
 nullify(elph_tr_ds%a2f_1d_tr)
 nullify(elph_tr_ds%a2f_1d_trin)
 nullify(elph_tr_ds%a2f_1d_trout)


end subroutine elph_tr_ds_nullify
!!***

