!{\src2tex{textfont=tt}}
!!****f* ABINIT/mover
!! NAME
!! mover
!!
!! FUNCTION
!! Move ion or change acell acording to forces and stresses
!!
!! COPYRIGHT
!! Copyright (C) 1998-2010 ABINIT group (DCA, XG, GMR, SE)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! INPUTS
!!  amass(natom)=mass of each atom, in unit of electronic mass (=amu*1822...)
!!  atindx(natom)=index table for atoms (see scfcv.f)
!!  atindx1(natom)=index table for atoms, inverse of atindx (see scfcv.f)
!!  cpus= cpu time limit in seconds)
!!  dtfil <type(datafiles_type)>=variables related to files
!!  dtset <type(dataset_type)>=all input variables for this dataset
!!   | mband=maximum number of bands
!!   | mgfft=maximum size of 1D FFTs
!!   | mkmem =number of k points which can fit in memory; set to 0 if use disk
!!   |  angular momentum for nonlocal pseudopotential
!!   | mpw=maximum dimensioned size of npw.
!!   | natom=number of atoms in unit cell
!!   |  except on first call (hartree/bohr); updated on output
!!   | nfft=(effective) number of FFT grid points (for this processor)
!!   |      for the "coarse" grid (see NOTES below)
!!   | nkpt=number of k points.
!!   | nspden=number of spin-density components
!!   | nsppol=1 for unpolarized, 2 for spin-polarized
!!   | nsym=number of symmetry elements in space group
!!  ecore=core psp energy (part of total energy) (hartree)
!!  hh=time step for molecular dynamics in atomic time units
!!   (1 atomic time unit=2.418884e-17 seconds)
!!  kg(3,mpw*mkmem)=reduced planewave coordinates.
!!  mpi_enreg=informations about MPI parallelization
!!  nfftf=(effective) number of FFT grid points (for this processor)
!!       for the "fine" grid (see NOTES below)
!!  npwarr(nkpt)=number of planewaves in basis and boundary at this k point.
!!  nspinor=number of spinorial components of the wavefunctions
!!  nattyp(ntypat)= # atoms of each type.
!!  paw_dmft  <type(paw_dmft_type)>= paw+dmft related data
!!  pawang <type(pawang_type)>=paw angular mesh and related data
!!  pawfgr <type(pawfgr_type)>=fine grid parameters and related data
!!  pawrad(ntypat*usepaw) <type(pawrad_type)>=paw radial mesh and related data
!!  pawtab(ntypat*usepaw) <type(pawtab_type)>=paw tabulated starting data
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!   | mpsang= 1+maximum angular momentum for nonlocal pseudopotentials
!!  pwind(pwind_alloc,2,3) = array used to compute
!!           the overlap matrix smat between k-points (see initberry.f)
!!  pwind_alloc = first dimension of pwind
!!  pwnsfac(2,pwind_alloc) = phase factors for non-symmorphic translations
!!                           (see initberry.f)
!!  rprimd(3,3)=dimensional primitive translations (bohr)
!!  ylm(mpw*mkmem,mpsang*mpsang*useylm)= real spherical harmonics for each G and k point
!!  ylmgr(mpw*mkmem,3,mpsang*mpsang*useylm)= gradients of real spherical harmonics
!!
!! OUTPUT
!!  results_gs <type(results_gs_type)>=results (energy and its components,
!!   forces and its components, the stress tensor) of a ground-state computation
!!  eigen(mband*nkpt*nsppol)=array for holding eigenvalues (hartree)
!!  resid(mband*nkpt*nsppol)=residuals for each band over all k points.
!!
!! SIDE EFFECTS
!! Rest of i/o is related to lda
!!  acell(3)=length scales of primitive translations (bohr)
!!  cg(2,mpw*nspinor*mband*mkmem*nsppol)=array for planewave coefficients of wavefunctions.
!!  dtefield <type(efield_type)> = variables related to Berry phase
!!      calculations (see initberry.f)
!!  ecore=core psp energy (part of total energy) (hartree)
!!  electronpositron <type(electronpositron_type)>=quantities for the electron-positron annihilation
!!  hdr <type(hdr_type)>=the header of wf, den and pot files
!!  indsym(4,nsym,natom)=indirect indexing array for atom labels
!!  initialized= if 0 the initialisation of the gstate run is not yet finished
!!  irrzon(nfft**(1-1/nsym),2,(nspden/nsppol)-3*(nspden/4))=irreducible zone data
!!  pawrhoij(natom*usepaw) <type(pawrhoij_type)>= -PAW only- atomic occupancies
!!  occ(mband*nkpt*nsppol=occupation number for each band (usually 2) at each k point.
!!  phnons(2,nfft**(1-1/nsym),nspden/nsppol)=nonsymmorphic translation phases
!!  rhog(2,nfftf)=array for Fourier transform of electron density
!!  rhor(nfftf,nspden)=array for electron density in electrons/bohr**3.
!!  scf_history <type(scf_history_type)>=arrays obtained from previous SCF cycles
!!  symrec(3,3,nsym)=symmetry operations in reciprocal space
!!  taug(2,nfftf*dtset%usekden)=array for Fourier transform of kinetic energy density
!!  taur(nfftf,nspden*dtset%usekden)=array for kinetic energy density
!!  wffnew,wffnow= struct info for wf disk files.
!!  vel(3,natom)=old value of velocity; updated on output
!!  xred(3,natom)=reduced dimensionless atomic coordinates; updated on output
!!  xred_old(3,natom)=work space for old xred
!!
!! NOTES
!! This subroutine uses the arguments natom, xred, vel, fcart, amass,
!! vis, and dtion (the last two contained in dtset) to make
!! molecular dynamics updates.  The rest of the lengthy
!! argument list supports the underlying lda computation
!! of forces, returned from subroutine scfcv
!!
!! USE OF FFT GRIDS:
!! =================
!! In case of PAW:
!! ---------------
!!    Two FFT grids are used:
!!    - A "coarse" FFT grid (defined by ecut)
!!      for the application of the Hamiltonian on the plane waves basis.
!!      It is defined by nfft, ngfft, mgfft, ...
!!      Hamiltonian, wave-functions, density related to WFs (rhor here), ...
!!      are expressed on this grid.
!!    - A "fine" FFT grid (defined) by ecutdg)
!!      for the computation of the density inside PAW spheres.
!!      It is defined by nfftf, ngfftf, mgfftf, ...
!!      Total density, potentials, ...
!!      are expressed on this grid.
!! In case of norm-conserving:
!! ---------------------------
!!    - Only the usual FFT grid (defined by ecut) is used.
!!      It is defined by nfft, ngfft, mgfft, ...
!!      For compatibility reasons, (nfftf,ngfftf,mgfftf)
!!      are set equal to (nfft,ngfft,mgfft) in that case.
!!
!! PARENTS
!!      gstate
!!
!! CHILDREN
!!      chkexi,leave_new,prtxvf,scfcv,status,wrtout,xredxcart
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine mover(ab_scfcv_in,ab_xfh,acell,amass,cg,dtefield,&
  & dtfil,dtset,eigen,electronpositron,hdr,&
  & initialized,irrzon,mpi_enreg,nfftf,&
  & occ,paw_dmft,pawfgr,pawrhoij,&
  & rec_set,resid,results_gs,&
  & rhog,rhor,rprimd,scf_history,symrec,taug,taur,wffnew,&
  & wffnow,vel,wvl,xred,xred_old)

use defs_basis
use defs_datatypes
use defs_abitypes
use defs_scftypes
use defs_rectypes
use defs_wvltypes
use m_paw_dmft, only: paw_dmft_type
use m_electronpositron, only : electronpositron_type
use m_wffile

use defs_mover

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_42_geometry
 use interfaces_42_geomoptim
 use interfaces_56_recipspace
 use interfaces_67_common
 use interfaces_95_drive, except_this_one => mover
!End of the abilint section

implicit none

!Arguments ------------------------------------
!scalars
integer,intent(inout) :: initialized,nfftf
type(MPI_type),intent(inout) :: mpi_enreg
type(datafiles_type),intent(inout),target :: dtfil
type(dataset_type),intent(inout),target :: dtset
type(efield_type),intent(inout) :: dtefield
type(electronpositron_type),pointer :: electronpositron
type(hdr_type),intent(inout) :: hdr
type(paw_dmft_type) :: paw_dmft
type(pawfgr_type) :: pawfgr
type(results_gs_type),intent(out) :: results_gs
type(recursion_type),intent(inout) ::rec_set
type(scf_history_type),intent(inout) :: scf_history
type(ab_scfcv_args_in),intent(inout) :: ab_scfcv_in
type(wffile_type),intent(inout) :: wffnew,wffnow
type(wvl_data),intent(inout) :: wvl
type(ab_xfh_type),intent(inout) :: ab_xfh
!arrays
!no_abirules
!nfft**(1-1/nsym) is 1 if nsym==1, and nfft otherwise
integer, intent(inout) :: irrzon(dtset%nfft**(1-1/dtset%nsym),2,(dtset%nspden/dtset%nsppol)-3*(dtset%nspden/4))
integer, intent(inout) :: symrec(3,3,dtset%nsym)
real(dp),intent(inout) :: acell(3)
real(dp), intent(in),target :: amass(dtset%natom)
real(dp), intent(inout) :: cg(2,dtset%mpw*ab_scfcv_in%nspinor*dtset%mband*dtset%mkmem*dtset%nsppol)
real(dp), intent(out) :: eigen(dtset%mband*dtset%nkpt*dtset%nsppol)
!nfft**(1-1/nsym) is 1 if nsym==1, and nfft otherwise
real(dp), intent(inout) :: occ(dtset%mband*dtset%nkpt*dtset%nsppol)
real(dp), intent(out) :: resid(dtset%mband*dtset%nkpt*dtset%nsppol)
real(dp), pointer :: rhog(:,:),rhor(:,:)
real(dp), pointer :: taug(:,:),taur(:,:)
real(dp), intent(inout) :: xred(3,dtset%natom),xred_old(3,dtset%natom)
real(dp), intent(inout) :: vel(3,dtset%natom),rprimd(3,3)
type(pawrhoij_type), intent(inout) :: pawrhoij(mpi_enreg%natom*ab_scfcv_in%psps%usepaw)

!Local variables-------------------------------
!scalars
type(ab_movehistory) :: hist
type(ab_movetype) :: ab_mover
integer,parameter :: level=102
integer itime,icycle,iexit,ncycle,kk,option
character(len=500) :: message
character(len=60) :: method
character(len=10) :: type4xml
character(len=8) :: crit4xml
character(len=8) :: stat4xml
real(dp) :: ucvol,favg
logical :: isFconv ! If the convergence is needed
logical :: DEBUG=.FALSE.

!arrays
real(dp) :: xcart(3,dtset%natom)
real(dp) :: fred_corrected(3,dtset%natom)
real(dp) :: rprim(3,3)
real(dp) :: gprimd(3,3)
real(dp) :: gmet(3,3)
real(dp) :: rmet(3,3)
type(macro_uj_type) :: dtpawuj(0) !ndtpawuj=0

! ***************************************************************

!Table of contents
!
!01. Initialization of indexes and allocations of arrays
!02. Particularities of each predictor
!03. General initializations for all predictors
!04. First output before any itime or icycle
!05. Compute xred and fill the history of the first SCFCV
!06. Loop for itime (From 1 to ntime)
!07. Loop for icycle (From 1 to ncycles)
!08. Output for each icycle (and itime)
!09. => Call to SCFCV routine and fill history with forces
!10. Output after SCFCV
!11. => Call to each predictor
!12. Use the history  to extract the new values
!13. End loop icycle
!14. End loop itime
!15. Set the final values of xcart and xred
!
!Conditions:
!
!Routines (XXX_init and XXX_post) should not produce output, only
!mathematical expresions are expected on those routines
!
!This routine (mover.F90) should produce output, no mathematical
!expresions should appear.
!
!History:
!
!History of several variables are under the type "ab_movehistory"
!It contains:
!* mxhist                  : Maximun size of history
!* ihist                   : index of history
!* histA(3,mxhist)         : Acell
!* histE(mxhist)           : Energy
!* histR(3,3,mxhist)       : Rprimd
!* histS(6,mxhist)         : Strten
!* histV(3,natom,mxhist)   : Velocity
!* histXF(3,natom,4,mxhist): Xcart,Xred,Fcart,Fred
!
!IONMOV values:
!
!1.  Molecular dynamics without viscosity (vis=0)
!1.  Molecular dynamics with viscosity (vis/=0)
!2.  Broyden-Fletcher-Goldfard-Shanno method (forces)
!3.  Broyden-Fletcher-Goldfard-Shanno method (forces,Tot energy)
!4.  Conjugate gradient of potential and ionic degrees of freedom
!5.  Simple relaxation of ionic positions
!6.  Verlet algorithm for molecular dynamics
!7.  Verlet algorithm blocking every atom where dot(vel,force)<0
!8.  Verlet algorithm with a nose-hoover thermostat
!9.  Langevin molecular dynamics
!10. BFGS with delocalized internal coordinates
!11. Conjugate gradient algorithm
!12. Isokinetic ensemble molecular dynamics
!13. Isothermal/isenthalpic ensemble molecular dynamics
!14. Symplectic algorithm Runge-Kutta-Nyström SRKNa14
!20. Ionic positions relaxation using DIIS
!30. Self consistent phonon structure using a supercell

!write(*,*) 'mover 01'
!###########################################################
!### 01. Initialization of indexes and allocations of arrays

 if (dtset%ionmov>50) dtset%ionmov=100-dtset%ionmov

!Copy the information from the Dataset (dtset)
!to the ab_mover structure
!DIIS memory (Use by pred_diisrelax only)
 ab_mover%diismemory=>dtset%diismemory
!Delta Time for IONs
 ab_mover%dtion=>dtset%dtion
!include a JELLium SLAB in the cell
 ab_mover%jellslab=>dtset%jellslab
!Number of ATOMs
 ab_mover%natom=>dtset%natom
!Number of CONstraint EQuations
 ab_mover%nconeq=>dtset%nconeq
!Number of SYMmetry operations
 ab_mover%nsym=>dtset%nsym
!Number of Type of atoms
 ab_mover%ntypat=>dtset%ntypat
!OPTimize the CELL shape and dimensions
 ab_mover%optcell=>dtset%optcell
!RESTART Xcart and Fred
 ab_mover%restartxf=>dtset%restartxf
!Molecular Dynamics Final Temperature
 ab_mover%mdftemp=>dtset%mdftemp
!Molecular Dynamics Initial Temperature
 ab_mover%mditemp=>dtset%mditemp
!NOT DOCUMENTED
 ab_mover%noseinert=>dtset%noseinert
!Phonon Temperature (Use by pred_scphon only)
 ab_mover%scphon_temp=>dtset%scphon_temp
!STRess PRECONditioner
 ab_mover%strprecon=>dtset%strprecon
!VIScosity
 ab_mover%vis=>dtset%vis
!Indices of AToms that are FIXed
 ab_mover%iatfix=>dtset%iatfix
!SYMmetry in REaL space
 ab_mover%symrel=>dtset%symrel
!Supercell_multiplicity (Use by pred_scphon only)
 ab_mover%scphon_supercell=>dtset%scphon_supercell
!Mass of each atom (NOT IN DTSET)
 ab_mover%amass=>amass
!STRess TARGET
 ab_mover%strtarget=>dtset%strtarget
!TYPe of ATom
 ab_mover%typat=>dtset%typat
!Filename for Hessian matrix (NOT IN DTSET)
 ab_mover%fnameabi_hes=>dtfil%fnameabi_hes
!Filename for _PCINFO file
 ab_mover%filnam_ds3=>dtfil%filnam_ds(3)
!Filename for _PHFRQ file
 ab_mover%fnameabi_phfrq=>dtfil%fnameabi_phfrq
!Filename for _PHVEC file
 ab_mover%fnameabi_phvec=>dtfil%fnameabi_phvec

!Z number of each NUCLeus
 ab_mover%znucl=>dtset%znucl

!!DEBUG
!call ab_movetype_print(ab_mover,ab_out)
!!DEBUG

!acell and rprimd are never change except if optcell/=0
 if (ab_mover%optcell/=0)then
   hist%isARused=.TRUE.
 else
   hist%isARused=.FALSE.
 end if

!Velocities are never change except for ionmov=1,6,7,8
 hist%isVused=.FALSE.

!In general convergence is needed
 isFconv=.TRUE.

!write(*,*) 'mover 02'
!###########################################################
!### 02. Particularities of each predictor

!This is the initialization for ionmov==1
!-----------------------------------------
 if(dtset%ionmov==1) then
   hist%mxhist=dtset%ntime+4
   isFconv=.FALSE.     ! Convergence is not used
   hist%isVused=.TRUE. ! Velocities are used
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=4 ! Number of internal cycles for first itime
!  Values use in XML Output
   type4xml='moldyn'
   crit4xml='none'
!  Name of method
   if (abs(ab_mover%vis)<=1.d-8) then
     write(method,'(a)')&
     'Molecular dynamics without viscosity (vis=0)'
   else
     write(method,'(a,1p,e12.5,a)')&
     'Molecular dynamics with viscosity (vis=',&
&     ab_mover%vis,')'
   end if
!  This is the initialization for ionmov==2,3
!  -------------------------------------------
 else if (dtset%ionmov==2.or.dtset%ionmov==3)then
   if(dtset%ionmov==2) hist%mxhist=dtset%ntime+1
   if(dtset%ionmov==3) hist%mxhist=dtset%ntime+1
   ncycle=1
!  Values use in XML Output
   type4xml='bfgs'
   crit4xml='tolmxf'
!  Name of method
   if (dtset%ionmov==2) then
     write(method,'(a)')&
     'Broyden-Fletcher-Goldfard-Shanno method (forces)'
   else
     write(method,'(a)')&
     'Broyden-Fletcher-Goldfard-Shanno method (forces,Tot energy)'
   end if
!  This is the initialization for ionmov==4,5
!  -------------------------------------------
 else if (dtset%ionmov==4.or.dtset%ionmov==5)then
   hist%mxhist=dtset%ntime+1
   ncycle=1
!  Values use in XML Output
   type4xml='simple'
   crit4xml='tolmxf'
!  Name of method
   if (dtset%ionmov==4) then
     write(method,'(a)')&
     'Conjugate gradient of potential and ionic degrees of freedom'
   else
     write(method,'(a)')&
     'Simple relaxation of ionic positions'
   end if
!  This is the initialization for ionmov==6
!  ------------------------------------------
 else if (dtset%ionmov==6)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   hist%isVused=.TRUE. ! Velocities are used
   ncycle=1
!  Values use in XML Output
   type4xml='verlet'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Verlet algorithm for molecular dynamics'
!  This is the initialization for ionmov==7
!  ------------------------------------------
 else if (dtset%ionmov==7)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   hist%isVused=.TRUE. ! Velocities are used
   ncycle=1
!  Values use in XML Output
   type4xml='verlet'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Verlet algorithm blocking every atom where dot(vel,force)<0'
!  This is the initialization for ionmov==8
!  ------------------------------------------
 else if (dtset%ionmov==8)then
   hist%mxhist=dtset%ntime+1
   hist%isVused=.TRUE.
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='nose'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Verlet algorithm with a nose-hoover thermostat'
!  This is the initialization for ionmov==9
!  ------------------------------------------
 else if (dtset%ionmov==9)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='langevin'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Langevin molecular dynamics'
!  This is the initialization for ionmov==10
!  -------------------------------------------
 else if (dtset%ionmov==10)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='delocint'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'BFGS with delocalized internal coordinates'
!  This is the initialization for ionmov==11
!  -------------------------------------------
 else if (dtset%ionmov==11)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='cg'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Conjugate gradient algorithm'
!  This is the initialization for ionmov==12
!  -------------------------------------------
 else if (dtset%ionmov==12)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='isokin'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Isokinetic ensemble molecular dynamics'
!  This is the initialization for ionmov==13
!  -------------------------------------------
 else if (dtset%ionmov==13)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='isother'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Isothermal/isenthalpic ensemble molecular dynamics'
!  This is the initialization for ionmov==14
!  -------------------------------------------
 else if (dtset%ionmov==14)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='srkna14'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Symplectic algorithm Runge-Kutta-Nyström SRKNa14'
!  This is the initialization for ionmov==20
!  -------------------------------------------
 else if (dtset%ionmov==20)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='diisrelax'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Ionic positions relaxation using DIIS'
!  This is the initialization for ionmov==30
!  -------------------------------------------
 else if (dtset%ionmov==30)then
   hist%mxhist=dtset%ntime+1
!  TEMPORARLY optcell is not allow
   hist%isARused=.FALSE.
   ncycle=1
!  Values use in XML Output
   type4xml='scphon'
   crit4xml='tolmxf'
!  Name of method
   write(method,'(a)')&
   'Self consistent phonon structure using a supercell'
 end if

!write(*,*) 'mover 03'
!###########################################################
!### 03. General initializations for all predictors

!!     if(DEBUG)then
!write (ab_out,*) '---BEFORE ANYTHING---'

!write (ab_out,*) 'XRED'
!do kk=1,ab_mover%natom
!write (ab_out,*) xred(:,kk)
!end do
!write(ab_out,*) 'RPRIMD'
!do kk=1,3
!write(ab_out,*) rprimd(:,kk)
!end do
!!     end if

!Initialize indexes
 hist%ihist=1
 iexit=0

!Allocate all the histories
 allocate(hist%histA(3,hist%mxhist))
 allocate(hist%histE(hist%mxhist))
 allocate(hist%histR(3,3,hist%mxhist))
 allocate(hist%histS(6,hist%mxhist))
 allocate(hist%histV(3,dtset%natom,hist%mxhist))
 allocate(hist%histXF(3,dtset%natom,4,hist%mxhist))

!call status(0,dtfil%filstat,iexit,level,'Entering move')

!write(*,*) 'mover 04'
!###########################################################
!### 04. First output before any itime or icycle

 write(message,'(a,a,i2,a,a,a,80a,a)')&
& ch10,'=== [ionmov=',dtset%ionmov,'] ',method,&
& ch10,('=',kk=1,80),ch10
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,message,'COLL')

!write(*,*) 'mover 05'
!###########################################################
!### 05. Compute xcart and fill the history of the first SCFCV

!Compute xcart from xred, and rprimd
 call xredxcart(dtset%natom,1,rprimd,xcart,xred)

!Fill history with the values of xred,xcart,acell and rprimd
 call var2hist(ab_mover,hist,acell,rprimd,xcart,xred)

 hist%histV(:,:,hist%ihist)=vel(:,:)

!write(*,*) 'mover 06'
!###########################################################
!### 06. Loop for itime (From 1 to ntime)
 do itime=1,dtset%ntime
!  call status(1000*itime,dtfil%filstat,iexit,level,'Loop itime')

!  write(*,*) 'mover 07'
!  ###########################################################
!  ### 07. Loop for icycle (From 1 to ncycles)
   do icycle=1,ncycle
!    call status(1000*itime+icycle,dtfil%filstat,iexit,level,'Loop icycle')

!    write(*,*) 'mover 08'
!    ###########################################################
!    ### 08. Output for each icycle (and itime)

     write(message,'(a,i2,a,i2,a,i2,a,i2,a,a,80a)')&
&     '--- Iteration: (',itime,'/',dtset%ntime,&
&     ') Internal Cycle: (',icycle,'/',ncycle,')',ch10,&
&     ('-',kk=1,80)
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,message,'COLL')

     call prtxfase(ab_mover,hist,std_out,icycle,itime,mover_BEFORE)

!    DEBUG (XRA BEFORE SCFCV)
     if(DEBUG)then
       write (ab_out,*) '---XRA BEFORE SCFCV---'
       write (ab_out,*) 'XCART'
       do kk=1,dtset%natom
         write (ab_out,*) xcart(:,kk)
       end do
       write (ab_out,*) 'XRED'
       do kk=1,dtset%natom
         write (ab_out,*) xred(:,kk)
       end do
       if (dtset%ionmov==1)then
         write (ab_out,*) 'VEL'
         do kk=1,dtset%natom
           write (ab_out,*) hist%histV(:,kk,hist%ihist)
         end do
       end if
       write(ab_out,*) 'RPRIMD'
       do kk=1,3
         write(ab_out,*) rprimd(:,kk)
       end do
       write(ab_out,*) 'ACELL'
       write(ab_out,*) acell(:)
     end if

!    write(*,*) 'mover 09'
!    ###########################################################
!    ### 09. => Call to SCFCV routine and fill history with forces
     write(message,'(a,3a,a,72a)')&
&     ch10,('-',kk=1,3),&
&     'SELF-CONSISTENT-FIELD CONVERGENCE',('-',kk=1,44)
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,message,'COLL')
     ab_scfcv_in%ndtpawuj=0
     ab_scfcv_in%iapp=itime
     if(itime==1 .and. icycle/=ncycle )ab_scfcv_in%iapp=-icycle-1
     if (dtset%ionmov==4.or.dtset%ionmov==5) ab_scfcv_in%iapp=itime
     call scfcv_new(ab_scfcv_in,cg,dtefield,dtfil,&
&     dtpawuj,dtset,eigen,electronpositron,hdr,&
&     initialized,irrzon,mpi_enreg,&
&     nfftf,occ,&
&     paw_dmft,pawfgr,pawrhoij,&
&     rec_set,&
&     resid,results_gs,rhog,rhor,rprimd,&
&     scf_history,symrec,taug,taur,wffnew,wffnow,&
&     wvl,xred,xred_old)

!    ANOMALOUS SITUATION
!    This is the only case where rprimd could change inside scfcv
!    It generates an weird condition, we start with a certain
!    value for rprimd before scfcv and after we finish with
!    a different value.
!    Notice that normally scfcv should not change rprimd
!    And even worse if optcell==0
!    The solution here is to recompute acell and rprim
!    and store those values in the present record
!    even if initially those values were not exactly
!    the values entering in scfcv
!    
!    One test case with these condition is bigdft/t10
     if (dtset%usewvl == 1) then
       call mkradim(acell,rprim,rprimd)
       hist%histA(:,hist%ihist)=acell(:)
       hist%histR(:,:,hist%ihist)=rprimd(:,:)
     end if

!    ANOMALOUS SITUATIONS
!    * In ionmov 4 & 5 xred could change inside SCFCV
!    So we need to take the values from the output
!    
!    * Inside scfcv.F90 there is a call to symzat.F90
!    for the first SCF cycle symzat could change xred
!    so we need always take xred convert to xcart and
!    store in the history

     if (dtset%ionmov<10)then
       call xredxcart(dtset%natom,1,rprimd,xcart,xred)
       hist%histXF(:,:,1,hist%ihist)=xcart(:,:)
       hist%histXF(:,:,2,hist%ihist)=xred(:,:)
     end if

     hist%histV(:,:,hist%ihist)=vel(:,:)
     hist%histE(hist%ihist)=results_gs%etotal
     hist%histXF(:,:,3,hist%ihist)=results_gs%fcart(:,:)
     hist%histXF(:,:,4,hist%ihist)=results_gs%fred(:,:)
     hist%histS(:,hist%ihist)=results_gs%strten(:)

     if(DEBUG)then
       write (ab_out,*) '---BEFORE XFH_UPDATE---'

       write (ab_out,*) 'XRED'
       do kk=1,ab_mover%natom
         write (ab_out,*) xred(:,kk)
       end do
       write (ab_out,*) 'FRED'
       do kk=1,ab_mover%natom
         write (ab_out,*) results_gs%fred(:,kk)
       end do
       write(ab_out,*) 'RPRID'
       do kk=1,3
         write(ab_out,*) rprimd(:,kk)
       end do
       write(ab_out,*) 'RPRIM'
       do kk=1,3
         write(ab_out,*) rprim(:,kk)
       end do
       write(ab_out,*) 'ACELL'
       write(ab_out,*) acell(:)
     end if

     if(ab_xfh%nxfh==0.or.itime/=1) then

       call mkradim(acell,rprim,rprimd)
!      Get rid of mean force on whole unit cell, but only if no
!      generalized constraints are in effect
       if(ab_mover%nconeq==0)then
         do kk=1,3
           favg=sum(results_gs%fred(kk,:))/dble(ab_mover%natom)
           fred_corrected(kk,:)=results_gs%fred(kk,:)-favg
           if(ab_mover%jellslab/=0.and.kk==3)&
&           fred_corrected(kk,:)=results_gs%fred(kk,:)
         end do
       else
         fred_corrected(:,:)=results_gs%fred(:,:)
       end if

       call xfh_update(ab_xfh,acell,fred_corrected,ab_mover%natom,rprim,&
&       results_gs%strten,xred)
     end if

!    call status(1000*itime+icycle,dtfil%filstat,iexit,level,'Calling Post')

!    write(*,*) 'mover 10'
!    ###########################################################
!    ### 10. Output after SCFCV
     write(message,'(3a,a,72a,a)')&
&     ('-',kk=1,3),&
&     'OUTPUT',('-',kk=1,71),ch10
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,message,'COLL')

     call prtxfase(ab_mover,hist,ab_out,icycle,itime,mover_AFTER)
     call prtxfase(ab_mover,hist,std_out,icycle,itime,mover_AFTER)

!    DEBUG (XRA AFTER SCFCV)
     if(DEBUG)then
       write (ab_out,*) '---XRA AFTER SCFCV---'
       write (ab_out,*) 'XCART'
       do kk=1,dtset%natom
         write (ab_out,*) xcart(:,kk)
       end do
       write (ab_out,*) 'XRED'
       do kk=1,dtset%natom
         write (ab_out,*) xred(:,kk)
       end do
       if (dtset%ionmov==1)then
         write (ab_out,*) 'VEL'
         do kk=1,dtset%natom
           write (ab_out,*) hist%histV(:,kk,hist%ihist)
         end do
       end if
       write(ab_out,*) 'RPRIMD'
       do kk=1,3
         write(ab_out,*) rprimd(:,kk)
       end do
       write(ab_out,*) 'ACELL'
       write(ab_out,*) acell(:)
     end if

!    write(*,*) 'mover 11'
!    ###########################################################
!    ### 11. => Test Convergence of forces and stresses

     if (itime==dtset%ntime.and.icycle==ncycle)then
       iexit=1
       stat4xml="Failed"
     else
       stat4xml="Succeded"
     end if

!    Only if convergence is needed
     if(isFconv)then
       if ((dtset%ionmov/=4.and.dtset%ionmov/=5).or.mod(itime,2)==1)then
         call fconv(results_gs%fcart, dtset%iatfix, &
&         iexit, itime, dtset%natom, dtset%ntime,&
&         ab_mover%optcell,&
&         dtset%strfact, dtset%strtarget, results_gs%strten,&
&         dtset%tolmxf)
       end if
     end if

     if (itime==dtset%ntime.and.icycle==ncycle) iexit=1

     if(iexit/=0) exit

!    write(*,*) 'mover 12'
!    ###########################################################
!    ### 12. => Call to each predictor

     if(dtset%ionmov==1) then
       call pred_moldyn(ab_mover,hist,icycle,itime,ncycle,dtset%ntime)
     else if (dtset%ionmov==2.or.dtset%ionmov==3)then
       call pred_bfgs(ab_mover,ab_xfh,hist,dtset%ionmov,itime)
     else if (dtset%ionmov==4.or.dtset%ionmov==5)then
       call pred_simple(ab_mover,hist)
     else if (dtset%ionmov==6.or.dtset%ionmov==7)then
       call pred_verlet(ab_mover,hist,dtset%ionmov,itime,dtset%ntime)
     else if (dtset%ionmov==8)then
       call pred_nose(ab_mover,hist,itime,dtset%ntime)
     else if (dtset%ionmov==9)then
       call pred_langevin(ab_mover,hist,dtset%ionmov,itime,dtset%ntime)
     else if (dtset%ionmov==10.or.dtset%ionmov==11)then
       call pred_delocint(ab_mover,ab_xfh,hist,dtset%ionmov,itime)
     else if (dtset%ionmov==20)then
       call pred_diisrelax(ab_mover,hist,itime,dtset%ntime)
     else if (dtset%ionmov==30)then
       call pred_scphon(ab_mover,hist,itime,dtset%ntime,iexit,dtset%tolmxf)
     end if


!    write(*,*) 'mover 13'
!    ###########################################################
!    ### 13. Use the history  to extract the new values
!    ###     acell, rprimd, xcart and xred

     call hist2var(ab_mover,hist,acell,rprimd,xcart,xred)

     if(dtset%optcell/=0)then

       call metric(gmet,gprimd,-1,rmet,rprimd,ucvol)

!      If metric has changed since the initialization,
!      update the Ylm's
       if (ab_scfcv_in%psps%useylm==1)then
         option=0;
         if (dtset%iscf>0) option=1
         call initylmg(gprimd,ab_scfcv_in%kg,dtset%kptns,dtset%mkmem,mpi_enreg,&
&         ab_scfcv_in%psps%mpsang,dtset%mpw,dtset%nband,dtset%nkpt,&
&         ab_scfcv_in%npwarr,dtset%nsppol,option,rprimd,dtfil%unkg,&
&         dtfil%unylm,ab_scfcv_in%ylm,ab_scfcv_in%ylmgr)
       end if

     end if

     vel(:,:)=hist%histV(:,:,hist%ihist)

!    write(*,*) 'mover 14'
!    ###########################################################
!    ### 14. End loop icycle
   end do ! do icycle=1,ncycle

   if(iexit/=0)exit
!  write(*,*) 'mover 15'
!  ###########################################################
!  ### 15. End loop itime
 end do ! do itime=1,dtset%ntime

!write(*,*) 'mover 16'
!###########################################################
!### 16. Set the final values of xcart and xred with the last
!###     computed values (not the last predicted)

 acell(:)=hist%histA(:,hist%ihist)
 rprimd(:,:)=hist%histR(:,:,hist%ihist)
 xcart(:,:)=hist%histXF(:,:,1,hist%ihist)
 xred(:,:)=hist%histXF(:,:,2,hist%ihist)
 vel(:,:)=hist%histV(:,:,hist%ihist)

 if(DEBUG)then
   write (ab_out,*) '---LAST VALUES BEFORE EXIT---'
   write (ab_out,*) 'ihist=',hist%ihist
   write (ab_out,*) 'itime=',itime
   write (ab_out,*) 'XCART'
   do kk=1,dtset%natom
     write (ab_out,*) xcart(:,kk)
   end do
   write (ab_out,*) 'XRED'
   do kk=1,dtset%natom
     write (ab_out,*) xred(:,kk)
   end do
   if (dtset%ionmov==1)then
     write (ab_out,*) 'VEL'
     do kk=1,dtset%natom
       write (ab_out,*) hist%histV(:,kk,hist%ihist)
     end do
   end if
   write(ab_out,*) 'RPRIMD'
   do kk=1,3
     write(ab_out,*) rprimd(:,kk)
   end do
   write(ab_out,*) 'ACELL'
   write(ab_out,*) acell(:)
 end if

!write(*,*) 'mover 17'
!###########################################################
!### 17. XML Output at the end

!XML output of the status
 if (mpi_enreg%me == 0 .and. dtset%prtxml == 1) then
   write(ab_xml_out, "(3a)") '    <geometryMinimisation type="',&
&   trim(type4xml),'">'
   write(ab_xml_out, "(5a)") '      <status cvState="',&
&   trim(stat4xml) ,&
&   '" stop-criterion="',trim(crit4xml),'" />'
   write(ab_xml_out, "(3a)") '    </geometryMinimisation>'
 end if

!write(*,*) 'mover 18'
!###########################################################
!### 18. Deallocate hist and ab_mover datatypes

 call ab_movetype_nullify(ab_mover)
 call ab_movehistory_nullify(hist)

end subroutine mover
!!***
