!{\src2tex{textfont=tt}}
!!****f* ABINIT/pred_scphon
!! NAME
!! pred_scphon
!!
!! FUNCTION
!! Ionmov predictor (30) Self consistent phonon structure
!!
!! IONMOV 30:
!!  Using a supercell, calculate a self consistent phonon structure
!!  as in PRL 100 095901 (2008). The initial phonon eigenvectors and
!!  eigenvalues are read in, and then atoms are displaced according
!!  to the normal modes populated at a given temperature until
!!  convergence of the vibrational free energy (or so I hope)
!!
!! Other references:
!!   Computational Materials Science 44 (2009) 888-894
!!   Phys Rev B 78, 184304 (2008)
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors,
!! see ~abinit/doc/developers/contributors.txt .
!!
!! INPUT (in)
!! ab_mover <type(ab_movetype)> : Datatype with all the information
!!                                needed by the preditor
!! itime  : Index of the present iteration
!! ntime  : Maximal number of iterations
!! ionmov : (2 or 3) Specific kind of BFGS
!!
!! OUTPUT (out)
!!
!! SIDE EFFECTS (inout)
!! hist <type(ab_movehistory)> : History of positions,forces
!!                               acell, rprimd, stresses
!!
!! NOTES
!!
!! PARENTS
!!      mover
!!
!! CHILDREN
!!      chkrprimd,destroy_primcell_ddb_info,hist2var,init_phondos,metric,mkrdim
!!      print_phonfreq,read_primcell_ddb_info,scphon_build_qsym_map
!!      scphon_check_fcart,scphon_free_energy,scphon_freq_to_normmode
!!      scphon_ft_fcart,scphon_interpolate_phonon_and_dos
!!      scphon_make_phonon_dos,scphon_new_frequencies,scphon_phonon_init
!!      scphon_qpoint_init,scphon_supercell_vectors_init,scphon_update_xcart
!!      var2hist,wrtout,xredxcart
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine pred_scphon(ab_mover,hist,itime,ntime,&
& iexit,tolmxf)

! define dp,sixth,third,etc...
use defs_basis
! type(ab_movetype), type(ab_movehistory), type(ab_xfh_type)
use defs_mover
! type(primcell_ddb_info)
use m_primcell_ddb_info
! type(phonon_dos_type)
use m_phdos

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_42_geometry
 use interfaces_42_geomoptim
 use interfaces_95_drive, except_this_one => pred_scphon
!End of the abilint section

implicit none

!Arguments ------------------------------------
!scalars
type(ab_movetype),intent(in) :: ab_mover
type(ab_movehistory),intent(inout) :: hist
integer,  intent(in)  :: itime,ntime
integer,  intent(out) :: iexit
real(dp), intent(in)  :: tolmxf

!Local variables-------------------------------
!scalars
integer  :: ii,jj,kk
real(dp),save :: ucvol0
real(dp) :: ucvol
real(dp) :: etotal
real(dp) :: favg
logical  :: DEBUG=.TRUE.
integer,save  :: natom_primitive_cell,nphononq,nsym_primitive_cell
real(dp) :: free_energy,old_free_energy
character(len=32) :: statusOut
character(len=500) :: message
character(len=fnlen) :: ddb_info_filename
type(phonon_dos_type) :: t_phonon_dos
type(primcell_ddb_info) :: pcell

!arrays
integer,allocatable,save :: pcell_atom_in_supercell(:)
integer,allocatable,save :: qsym_map(:,:,:)
integer,allocatable,save :: symrec_primitive_cell(:,:,:)
real(dp),save :: acell0(3) ! Initial acell
real(dp),save :: rprimd0(3,3) ! Initial rprimd
real(dp) :: acell(3)
real(dp) :: rprimd(3,3),rprim(3,3)
real(dp) :: gprimd(3,3)
real(dp) :: gmet(3,3)
real(dp) :: rmet(3,3)
real(dp) :: fred(3,ab_mover%natom),fred_corrected(3,ab_mover%natom)
real(dp) :: xred(3,ab_mover%natom),xcart(3,ab_mover%natom)
real(dp) :: strten(6)
real(dp) :: fcart(3,ab_mover%natom)
real(dp),allocatable,save :: amass_pcell(:)
real(dp),allocatable,save :: forces_on_atoms_ft(:,:,:)
real(dp),allocatable,save :: normal_mode_displacements(:,:)
real(dp),allocatable,save :: normal_mode_displacements_old(:,:)
real(dp),allocatable,save :: phonon_eigval(:,:)
real(dp),allocatable,save :: phonon_eigval2_averaged(:,:)
real(dp),allocatable,save :: phonon_eigval_ref(:,:)
real(dp),allocatable,save :: phonon_eigvec_ref(:,:,:,:),phononq(:,:)
real(dp),allocatable,save :: sqrt_amass_pcell(:)
real(dp),allocatable,save :: supercell_vectors(:,:)
real(dp),allocatable,save :: xcart0(:,:)
! Displacements of atoms from equilibrium positions
! written U_R in the PRL
!real(dp) :: atom_displacements(3,ab_mover%natom)
real(dp),allocatable,save :: cartesian_displacements(:,:)

!write(*,*) 'scphon 01'
! ##########################################################
! ### 01. Initial settings, only for first iteration

if(itime==1)then
  write (message,'(a)') ' scphon: enter '
  call wrtout(ab_out,message,'COLL')
  call wrtout(std_out,  message,'COLL')

! --> ab_mover%scphon_supercell
!
! The number of times the primitive unit cell has been replicated
!  in each dimension
! FIXME: this could be inferred from the supercell size, avoiding
!  possible mismatches of scphon_supercell with the real one...

  write (message,'(a,3I6)')&
    & ' SC phonons: Found supercell multiplicity of',&
    & ab_mover%scphon_supercell
  call wrtout(ab_out,message,'COLL')
  call wrtout(std_out,  message,'COLL')

! This is the number of atoms for the primitive unit cell, in
!  which the phonons were calculated
  natom_primitive_cell=nint(dble(ab_mover%natom)&
    & /dble(ab_mover%scphon_supercell(1))&
    & /dble(ab_mover%scphon_supercell(2))&
    & /dble(ab_mover%scphon_supercell(3)))
  write (message,'(a,I6,a)') 'Deduced that there are ',&
    & natom_primitive_cell,&
    & ' atoms in the primitive unit cell'
  call wrtout(ab_out,message,'COLL')
  call wrtout(std_out,  message,'COLL')
  allocate(amass_pcell(natom_primitive_cell))
  allocate(sqrt_amass_pcell(natom_primitive_cell))
  amass_pcell(:) = ab_mover%amass(1:natom_primitive_cell)
  sqrt_amass_pcell(:) = sqrt(amass_pcell(:))
  do ii=1,natom_primitive_cell
    write (message,'(a,I6,a,E20.10)') ' mass of atom ', ii,&
      & ' = ', amass_pcell(ii)
    call wrtout(ab_out,message,'COLL')
    call wrtout(std_out,  message,'COLL')
  end do

! Set energies for DOS output and integration
!  t_phonon_dos%dossmear  =0.00001   ! about 2. cm-1
!  t_phonon_dos%omega_step=0.0000047 ! about 1. cm-1
!  t_phonon_dos%nomega=1000

! In the old version appears psps%ntypat instead of
! ab_mover%ntypat := dtset%ntypat

  call init_phondos(t_phonon_dos,ab_mover%ntypat,ab_mover%natom,1,&
    & 1000,1,1,smallest_real,greatest_real,0.0000047_dp,0.00001_dp)

end if

!write(*,*) 'scphon 02'
! ##########################################################
! ### 02. Several allocations also for first time
! ###     These allocations are dependent of "nphononq" or
! ###     "natom_primitive_cell"

if(itime==1)then

! Number of qpoints for phonon integration should be = natom
! Could be reduced by symmetries for reciprocal space,
!  but then you need to reconstruct the real space atomic
!  displacements below
! This is not implemented yet.
  nphononq=ab_mover%natom/natom_primitive_cell

! Phonon q vectors we will consider. Should correspond to
!  the superlattice of atoms we are using, eg 4x4x4 supercell
  allocate(phononq(3,nphononq))

! eigvec of reference phonon system
  allocate(phonon_eigvec_ref(2,3*natom_primitive_cell,&
    & 3*natom_primitive_cell,nphononq))

! eigval of reference phonon system
  allocate(phonon_eigval_ref(3*natom_primitive_cell,nphononq))

! eigval of present phonon system, extracted from instantaneous
!  forces
  allocate(phonon_eigval(3*natom_primitive_cell,nphononq))

! eigval of present phonon system: averaged value of phonon_eigval
  allocate(phonon_eigval2_averaged(3*natom_primitive_cell,nphononq))

! Classical displacements along each of the 3*natom normal modes
!  written A_ks in the reference PRL, but without the 1/sqrt(mass)
!  factor
  allocate(normal_mode_displacements(3*natom_primitive_cell,&
    & nphononq))
  allocate(normal_mode_displacements_old(3*natom_primitive_cell,&
    & nphononq))

! Forces on each atom fcart are fourier transformed into the
! following array (in cartesian coordinates)
  allocate(forces_on_atoms_ft(2,3*natom_primitive_cell,nphononq))

! For each atom, the vector to the unit cell it is in
  allocate(pcell_atom_in_supercell(ab_mover%natom))

! The absolute displacement of the atoms, in cartesian
! coordinates, since the beginning of time
  allocate(cartesian_displacements(3,ab_mover%natom))

  allocate(supercell_vectors(3,ab_mover%natom))
  allocate(xcart0(3,ab_mover%natom))


end if

!write(*,*) 'scphon 04'
! ##########################################################
! ### 04. Obtain the present values from the history

call hist2var(ab_mover,hist,acell,rprimd,xcart,xred)

fred(:,:)=hist%histXF(:,:,4,hist%ihist)
fcart(:,:)=hist%histXF(:,:,3,hist%ihist)
etotal=hist%histE(hist%ihist)
strten(:)=hist%histS(:,hist%ihist)

! Compute rprim from rprimd and acell
do kk=1,3
  do jj=1,3
    rprim(jj,kk)=rprimd(jj,kk)/acell(kk)
  end do
end do

call metric(gmet,gprimd,-1,rmet,rprimd,ucvol)

! Save initial values
if (itime==1)then
  acell0(:)=acell(:)
  rprimd0(:,:)=rprimd(:,:)
  ucvol0=ucvol
end if

! Get rid of mean force on whole unit cell, but only if no
! generalized constraints are in effect
if(ab_mover%nconeq==0)then
  do ii=1,3
    favg=sum(fred(ii,:))/dble(ab_mover%natom)
    fred_corrected(ii,:)=fred(ii,:)-favg
    if(ab_mover%jellslab/=0.and.ii==3)&
      & fred_corrected(ii,:)=fred(ii,:)
  end do
else
  fred_corrected(:,:)=fred(:,:)
end if

!write(*,*) 'scphon 05'
! ##########################################################
! ### 05. Initializations dependent of xcart, xred and
! ###     rprimd

if(itime==1)then

! Get generic phonon info from anaddb run, for later interpolation
  write (*,*) ' entering read_primcell'
  ! Should send dtfil%filnam_ds(3)
  ddb_info_filename=trim(ab_mover%filnam_ds3)//'_PCINFO'
  ddb_info_filename=trim(ddb_info_filename)
  call read_primcell_ddb_info(ddb_info_filename,pcell)

! Initialize the supercell vector grid (pointing to each unit
! cell in the supercell)
  call scphon_supercell_vectors_init(ab_mover%natom,&
    & natom_primitive_cell,pcell,pcell_atom_in_supercell,&
    & ab_mover%scphon_supercell,supercell_vectors,xred)

   write (message,'(a)') '  xred = '
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
   do ii=1,ab_mover%natom
     write (message,'(3E20.10)')  xred(:,ii)
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
   end do

! Initialize the phonon q grid based on the supercell size
  call scphon_qpoint_init (nphononq,phononq,ab_mover%scphon_supercell)

! Initialize the reference phonon eigenvectors and eigenvalues
! from file
  call scphon_phonon_init (ab_mover%fnameabi_phfrq,&
    & ab_mover%fnameabi_phvec,natom_primitive_cell,&
    & nphononq,phonon_eigvec_ref,phonon_eigval_ref)

! Set the initial phonon frequency average
  phonon_eigval2_averaged(:,:)=sign(one,phonon_eigval_ref(:,:))&
    & *phonon_eigval_ref(:,:)**2

! Get the symmetry operations for the primitive unit cells
  nsym_primitive_cell=pcell%nsym
  allocate (symrec_primitive_cell(3,3,nsym_primitive_cell))
  symrec_primitive_cell(:,:,1:nsym_primitive_cell) = pcell%symrec

! Mapping from 1 qpoint to another under symmetries
  allocate(qsym_map(nphononq,nsym_primitive_cell,2))
! Initialize the mapping from one qpoint to another under the
!  symops
  call scphon_build_qsym_map(nphononq,nsym_primitive_cell,phononq,&
    & qsym_map,symrec_primitive_cell)

! The variable ab_mover%scphon_temp is the temperature
! we want the phonons to be at.

! Initialize the first normal mode displacements
  call scphon_freq_to_normmode (qsym_map(:,1,2),&
    & natom_primitive_cell,normal_mode_displacements,&
    & nphononq,phonon_eigval_ref,ab_mover%scphon_temp)
  write (message,'(a)')&
    & ' Have initialized normal_mode_displacements'
  call wrtout(ab_out,message,'COLL')
  call wrtout(std_out,  message,'COLL')

! Transform xred to cartesian coordinates.
  call xredxcart(ab_mover%natom, 1, rprimd, xcart, xred)

! Initial positions
  xcart0=xcart

! Update xcart with difference between old and new normal mode
!  displacements also initializes cartesian_displacements
  cartesian_displacements=zero
  call scphon_update_xcart (sqrt_amass_pcell,&
    & cartesian_displacements,ab_mover%natom,natom_primitive_cell,&
    & normal_mode_displacements,nphononq,pcell_atom_in_supercell,&
    & phonon_eigvec_ref,phononq,supercell_vectors,xcart,xcart0)

! In first iteration, apply full normal mode displacement
  normal_mode_displacements_old=normal_mode_displacements

  call print_phonfreq(-1,natom_primitive_cell,nphononq,&
    & phonon_eigval_ref)

! Set old_free_energy to make sure at least 2 iterations are
!  performed.
  old_free_energy=greatest_real

! Transform to xcart ->  xred 
   call xredxcart(ab_mover%natom, -1, rprimd, xcart, xred)

   write (message,'(a)') '  input xred = '
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
   do ii=1,ab_mover%natom
     write (message,'(3E20.10)')  xred(:,ii)
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
   end do
   write (message,'(a)') ch10
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')

end if

!write(*,*) 'scphon 05'
! ##########################################################
! ### 05. Compute phonons

if(itime>1)then

! Check if fcart is in the opposite direction to
!  cartesian_displacements for each atom
call scphon_check_fcart(cartesian_displacements,&
  & fcart,ab_mover%natom)

! Fourier transform forces to recirocal space (inside BZ)
!  FIXME: could be optimized perhaps, make it an FFT, but the
!  grid will never be very big.
call scphon_ft_fcart(sqrt_amass_pcell,fcart,ab_mover%natom,&
  & natom_primitive_cell,nphononq,phononq,forces_on_atoms_ft,&
  & pcell_atom_in_supercell,supercell_vectors)

! Determine new frequencies according to forces
call scphon_new_frequencies(forces_on_atoms_ft,itime,&
  & natom_primitive_cell,normal_mode_displacements,nphononq,&
  & nsym_primitive_cell,pcell,phonon_eigvec_ref,&
  & phonon_eigval2_averaged,phonon_eigval,phononq,qsym_map)

! Calculate the DOS
! 1. The following code is for the bare input phonon frequencies
!    (no interpolation)
t_phonon_dos%nomega=1000
if (associated(t_phonon_dos%phdos)) deallocate(t_phonon_dos%phdos)
allocate (t_phonon_dos%phdos(t_phonon_dos%nomega))
call scphon_make_phonon_dos (t_phonon_dos%dossmear,&
  & natom_primitive_cell,t_phonon_dos%nomega,nphononq,&
  & t_phonon_dos%omega_max,t_phonon_dos%omega_min,&
  & t_phonon_dos%phdos,phonon_eigval)

! 2. Interpolate phonon freq first, then calculate DOS
!    this will give nomega a new value!
call scphon_interpolate_phonon_and_dos (natom_primitive_cell,&
  & nphononq,pcell,t_phonon_dos,phonon_eigval,phonon_eigvec_ref,&
  & phononq,ab_mover%scphon_supercell)

! Calculate vibrational free energy
call scphon_free_energy(free_energy,itime,t_phonon_dos,&
  & ab_mover%scphon_temp)

! Check whether Free Energy diff is below tolerance; if so, exit
!  from the itime loop
! NOTE: tolmxf is usually in Ha/bohr, so we should probably have
!  a conversion factor like for the stress. F usually down to
!  1e-5 and energy down to 1e-10 or so...
if (abs(free_energy-old_free_energy) < tolmxf) then
  write (message,'(a)')&
    & ' convergence in vibrational free energy has been reached: '
  call wrtout(ab_out,message,'COLL')
  call wrtout(std_out,  message,'COLL')
  write (message,'(E15.5,a,E15.5)') free_energy-old_free_energy,&
    & ' < ', tolmxf
  call wrtout(ab_out,message,'COLL')
  call wrtout(std_out,  message,'COLL')
  statusOut = "OK"
  iexit = 1
end if

iexit = 0
if (itime == ntime) then
  iexit = 1
  statusOut = "Failed"
end if

! Determine new classical normal mode displacements
call scphon_freq_to_normmode (qsym_map(:,1,2),&
  & natom_primitive_cell,normal_mode_displacements,nphononq,&
  & phonon_eigval,ab_mover%scphon_temp)

! Update xcart with difference between old and new normal mode
!  displacements
call scphon_update_xcart (sqrt_amass_pcell,&
  & cartesian_displacements,ab_mover%natom,natom_primitive_cell,&
  & normal_mode_displacements-normal_mode_displacements_old,&
  & nphononq,pcell_atom_in_supercell,phonon_eigvec_ref,phononq,&
  & supercell_vectors,xcart,xcart0)

! Save present displacements for next step
normal_mode_displacements_old=normal_mode_displacements

old_free_energy=free_energy

end if

!write(*,*) 'scphon 06'
! ##########################################################
! ### 06. Update the history with the prediction

! Increase indexes
hist%ihist=hist%ihist+1

if(DEBUG)then
  write (ab_out,*) '---XRA before xfpack option 2---'
  call chkrprimd(acell,rprim,rprimd,ab_out)
  write(ab_out,*) 'RPRIM'
  do kk=1,3
    write(ab_out,*) rprim(:,kk)
  end do
  write(ab_out,*) 'RPRIMD'
  do kk=1,3
    write(ab_out,*) rprimd(:,kk)
  end do
  write(ab_out,*) 'ACELL'
  write(ab_out,*) acell(:)
end if

if(ab_mover%optcell/=0)then
  call mkrdim(acell,rprim,rprimd)
  call metric(gmet,gprimd,-1,rmet,rprimd,ucvol)
end if

! DEBUG (XRA AFTER PREDICTION)
if(DEBUG)then
  write (ab_out,*) '---XRA AFTER PREDICTION ---'
  call chkrprimd(acell,rprim,rprimd,ab_out)
  write (ab_out,*) 'XCART'
  do kk=1,ab_mover%natom
    write (ab_out,*) xcart(:,kk)
  end do
  write (ab_out,*) 'XRED'
  do kk=1,ab_mover%natom
    write (ab_out,*) xred(:,kk)
  end do
  write (ab_out,*) 'RPRIM'
  do kk=1,3
    write (ab_out,*) rprim(:,kk)
  end do
  write(ab_out,*) 'RPRIMD'
  do kk=1,3
    write(ab_out,*) rprimd(:,kk)
  end do
  write(ab_out,*) 'ACELL'
  write(ab_out,*) acell(:)
end if

! Compute xcart from xred, and rprimd
call xredxcart(ab_mover%natom,1,rprimd,xcart,xred)

! Fill the history with the variables
! xcart, xred, acell, rprimd
call var2hist(ab_mover,hist,acell,rprimd,xcart,xred)

hist%histV(:,:,hist%ihist)=hist%histV(:,:,hist%ihist-1)

!write(*,*) 'scphon 07'
! ##########################################################
! ### 07. Deallocate arrays

if (itime==ntime-1)then

! Free working arrays
  deallocate(phononq)
  deallocate(phonon_eigvec_ref)
  deallocate(phonon_eigval_ref)
  deallocate(phonon_eigval)
  deallocate(phonon_eigval2_averaged)
  deallocate(normal_mode_displacements)
  deallocate(normal_mode_displacements_old)
  deallocate(forces_on_atoms_ft)
  deallocate(qsym_map)

  call destroy_primcell_ddb_info(pcell)

end if

end subroutine pred_scphon
!!***

!*********************************************************
!*********************************************************
! OTHER ROUTINES
!*********************************************************
!*********************************************************

