!{\src2tex{textfont=tt}}
!!****f* ABINIT/pred_verlet
!! NAME
!! pred_verlet
!!
!! FUNCTION
!! Ionmov predictors (6 & 7) Verlet algorithm
!!
!! IONMOV 6:
!! Given a starting point xred that is a vector of length 3*natom
!! (reduced nuclei coordinates), a velocity vector (in cartesian
!! coordinates), and unit cell parameters (acell and rprim -
!! without velocities in the present implementation), the Verlet
!! dynamics is performed, using the gradient of the energy
!! (atomic forces and stress : fred or fcart and stress) as
!! calculated by the routine scfcv.
!!
!! Some atoms can be kept fixed, while the propagation of unit cell
!! parameters is only performed if optcell/=0.
!! No more than ab_mover%ntime steps are performed.
!! The time step is governed by dtion, contained in ab_mover
!! (coming from dtset).
!! Returned quantities are xred, and eventually acell and rprim
!! (new ones!).
!!
!! IONMOV 7:
!! Block every atom for which the scalar product of velocity and
!! forces is negative, in order to reach the minimum.
!! The convergence requirement on the atomic forces, ab_mover%tolmxf,
!! allows an early exit.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2010 ABINIT group (DCA, XG, GMR, JCC, SE)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors,
!! see ~abinit/doc/developers/contributors.txt .
!!
!! INPUT (in)
!! ab_mover <type(ab_movetype)> : Datatype with all the information
!!                                needed by the preditor
!! itime  : Index of the present iteration
!! ntime  : Maximal number of iterations
!! ionmov : (2 or 3) Specific kind of BFGS
!!
!! OUTPUT (out)
!!
!! SIDE EFFECTS (inout)
!! hist <type(ab_movehistory)> : History of positions,forces
!!                               acell, rprimd, stresses
!!
!! NOTES
!!
!! PARENTS
!!      mover
!!
!! CHILDREN
!!      hessupdt,metric,mkrdim,wrtout,xfpack,xredxcart
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine pred_verlet(ab_mover,hist,ionmov,itime,ntime)

! define dp,sixth,third,etc...
 use defs_basis
! type(ab_movetype), type(ab_movehistory)
 use defs_mover

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_42_geometry
 use interfaces_42_geomoptim
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(ab_movetype),intent(in) :: ab_mover
 type(ab_movehistory),intent(inout) :: hist
 integer, intent(in) :: itime
 integer, intent(in) :: ntime
 integer, intent(in) :: ionmov

!Local variables-------------------------------
!scalars
 integer  :: ndim,option
 integer  :: ii,jj,kk
 integer  :: istopped,nstopped
 real(dp) :: taylor
 real(dp) :: ucvol,ucvol0,ucvol_next
 real(dp) :: etotal
 real(dp) :: diag,favg
 real(dp) :: ekin_corr,scprod
!arrays
 real(dp),allocatable,save :: hessin(:,:)
 real(dp),allocatable,save :: vin(:),vin_prev(:),vin_next(:)
 real(dp),allocatable,save :: vout(:),vout_prev(:)
 real(dp) :: acell(3),acell0(3)
 real(dp) :: acell_next(3)
 real(dp) :: rprimd(3,3),rprimd0(3,3),rprim(3,3)
 real(dp) :: rprimd_next(3,3),rprim_next(3,3)
 real(dp) :: gprimd(3,3)
 real(dp) :: gmet(3,3)
 real(dp) :: rmet(3,3)
 real(dp) :: fred(3,ab_mover%natom)
 real(dp) :: fcart(3,ab_mover%natom)
 real(dp) :: fred_corrected(3,ab_mover%natom)
 real(dp) :: xred(3,ab_mover%natom),xcart(3,ab_mover%natom)
 real(dp) :: xcart_next(3,ab_mover%natom)
 real(dp) :: xred_next(3,ab_mover%natom)
 real(dp) :: vel(3,ab_mover%natom)
 real(dp) :: vel_nexthalf(3,ab_mover%natom)
 real(dp) :: vel_prevhalf(3,ab_mover%natom)
 real(dp) :: strten(6)
 real(dp) :: stopped(ab_mover%natom)
 character(len=500) :: message

write(*,*) 'bfgs 01'
! ##########################################################
! ### 01. Compute the dimension of vectors (ndim)

 ndim=3*ab_mover%natom
 if(ab_mover%optcell==1 .or.&
      & ab_mover%optcell==4 .or.&
      & ab_mover%optcell==5 .or.&
      & ab_mover%optcell==6) ndim=ndim+1
 if(ab_mover%optcell==2 .or.&
      & ab_mover%optcell==3) ndim=ndim+6
 if(ab_mover%optcell==7 .or.&
      & ab_mover%optcell==8 .or.&
      & ab_mover%optcell==9) ndim=ndim+3

write(*,*) 'bfgs 02'
! ##########################################################
! ### 02. Allocate the vectors vin, vout and hessian matrix

 if (.not.allocated(vin)) allocate(vin(ndim))
 if (.not.allocated(vin_next)) allocate(vin_next(ndim))
 if (.not.allocated(vout)) allocate(vout(ndim))
 if (.not.allocated(vin_prev)) allocate(vin_prev(ndim))
 if (.not.allocated(vout_prev)) allocate(vout_prev(ndim))
 if (.not.allocated(hessin)) allocate(hessin(ndim,ndim))

write(*,*) 'bfgs 03'
! ##########################################################
! ### 03. Obtain the present values from the history

 xred(:,:)=hist%histXF(:,:,2,hist%ihist)
 fred(:,:)=hist%histXF(:,:,4,hist%ihist)
 etotal=hist%histE(hist%ihist)
 vel=hist%histV(:,:,1)

 acell(:)=hist%histA(:,hist%ihist)
 rprimd(:,:)=hist%histR(:,:,hist%ihist)
 strten(:)=hist%histS(:,hist%ihist)

 !Compute rprim from rprimd and acell
 do kk=1,3
    do jj=1,3
       rprim(jj,kk)=rprimd(jj,kk)/acell(jj)
    end do
 end do

 write(*,*) 'RPRIMD'
 do ii=1,3
    write(*,*) rprimd(:,ii)
 end do
 call metric(gmet,gprimd,-1,rmet,rprimd,ucvol)
 write(*,*) 'RMET'
 do ii=1,3
    write(*,*) rmet(:,ii)
 end do

!Save initial values
 acell0(:)=acell(:)
 rprimd0(:,:)=rprimd(:,:)
 ucvol0=ucvol

!  Get rid of mean force on whole unit cell, but only if no
!  generalized constraints are in effect
   if(ab_mover%nconeq==0)then
     do ii=1,3
       favg=sum(fred(ii,:))/dble(ab_mover%natom)
       fred_corrected(ii,:)=fred(ii,:)-favg
       if(ab_mover%jellslab/=0.and.ii==3)&
            & fred_corrected(ii,:)=fred(ii,:)
     end do
   else
     fred_corrected(:,:)=fred(:,:)
   end if

write(*,*) 'bfgs 04'
! ##########################################################
! ### 04. Fill the vectors vin and vout

!Initialize input vectors : first vin, then vout
 option=1
 call xfpack(acell, acell0, fred, ab_mover%natom, ndim,&
& ab_mover%nsym, ab_mover%optcell, option, rprim, rprimd0,&
& ab_mover%strtarget, strten, ab_mover%symrel, ucvol, ucvol0,&
& vin, vout, xred)
 option=3
 call xfpack(acell, acell0, fred_corrected, ab_mover%natom, ndim,&
& ab_mover%nsym, ab_mover%optcell, option, rprim, rprimd0,&
& ab_mover%strtarget, strten, ab_mover%symrel, ucvol, ucvol0,&
& vin, vout, xred)

 write(*,*) 'Entering to BFGS'
 write(*,*) 'VECTOR INPUT (vin)'
 do ii=1,ndim,3
    write(*,*) ii,vin(ii:ii+2)
 end do

 write(*,*) 'VECTOR OUTPUT (vout)'
 do ii=1,ndim,3
    write(*,*) ii,vout(ii:ii+2)
 end do

write(*,*) 'bfgs 05'
! ##########################################################
! ### 05. Initialize or update the hessian matrix

!Initialise the Hessian matrix using gmet
 if (itime==1)then
!Initialize inverse hessian with identity matrix
!in cartesian coordinates, which makes use of metric tensor gmet
!in reduced coordinates.
    hessin(:,:)=zero
    do ii=1,ab_mover%natom
       do kk=1,3
          do jj=1,3
!            Warning : implemented in reduced coordinates
             if (ab_mover%iatfix(kk,ii)==0 .and.&
                  & ab_mover%iatfix(jj,ii)==0 )then
                hessin(kk+3*(ii-1),jj+3*(ii-1))=gmet(kk,jj)
             end if
          end do
       end do
    end do
    if(ab_mover%optcell/=0)then
!      These values might lead to too large changes in some cases ...
       diag=ab_mover%strprecon*30.0_dp/ucvol
       if(ab_mover%optcell==1) diag=diag/three
       do ii=3*ab_mover%natom+1,ndim
          hessin(ii,ii)=diag
       end do
    end if
 end if

 if(itime>1)then
!    Update the hessian matrix, by taking into account the
!    current pair (x,f) and the previous one.
    call hessupdt(hessin,ab_mover%iatfix,ab_mover%natom,ndim,vin,&
&     vin_prev,vout,vout_prev)
 end if

 write(*,*) 'Hessian matrix'
 do kk=1,ndim
    do jj=1,ndim,3
       write(*,*) hessin(jj:jj+2,kk)
    end do
 end do

!DEBUG
 write(*,*) 'Exiting from BFGS BEFORE'
 write(*,*) 'VECTOR INPUT (vin)'
 do ii=1,ndim,3
    write(*,*) ii,vin(ii:ii+2)
 end do
 write(*,*) 'VECTOR OUTPUT (vout)'
 do ii=1,ndim,3
    write(*,*) ii,vout(ii:ii+2)
 end do
!DEBUG

write(*,*) 'bfgs 06'
! ##########################################################
! ### 06. Compute the next values

! %%% VERLET ALGORITHM %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

! Compute next atomic coordinates and cell parameters, using
! Verlet algorithm
! First propagate the position, without acceleration
if(itime/=0)then
  vin_next(:)=2*vin(:)-vin_prev(:)
  taylor=one
else
! Initialisation : no vin_prev is available, but the ionic velocity
! is available, in cartesian coordinates
! Convert input xred (reduced coordinates) to xcart (cartesian)
  call xredxcart(ab_mover%natom,1,rprimd,xcart,xred)
! Uses the velocity
  xcart_next(:,:)=xcart(:,:)+ab_mover%dtion*vel(:,:)
! Convert back to xred_next (reduced coordinates)
  call xredxcart(ab_mover%natom,-1,rprimd,xcart_next,xred_next)
! Impose no change of acell, ucvol, rprim, and rprimd
  acell_next(:)=acell(:)
  ucvol_next=ucvol
  rprim_next(:,:)=rprim(:,:)
  rprimd_next(:,:)=rprimd(:,:)
! Store all these next values in vin_next
  option=1
  call xfpack(acell_next,acell0,fred_corrected,ab_mover%natom,&
    & ndim,ab_mover%nsym,ab_mover%optcell,option,rprim_next,&
    & rprimd0,ab_mover%strtarget,strten,ab_mover%symrel,&
    & ucvol_next,ucvol0,&
    & vin_next,vout,xred_next)
  taylor=half
end if
! Now, take into account the acceleration
do ii=1,ndim
! Note the minus sign: the forces are minus the gradients,
! contained in vout.
  vin_next(:)=vin_next(:)-ab_mover%dtion**2*hessin(:,ii)*&
    & vout(ii)*taylor
end do
! Implement fixing of atoms : put back old values for fixed
! components
do kk=1,ab_mover%natom
  do jj=1,3
! Warning : implemented in reduced coordinates
    if (ab_mover%iatfix(jj,kk) == 1) then
      vin_next(jj+(kk-1)*3)=vin(jj+(kk-1)*3)
    end if
  end do
end do

! Now, compute the velocity at the next half-step
! Get xred_next, and eventually acell_next, ucvol_next, rprim_next and
! rprimd_next, from vin_next
option=2
call xfpack(acell_next,acell0,fred,ab_mover%natom,ndim,&
  & ab_mover%nsym,ab_mover%optcell,option,rprim_next,rprimd0,&
  & ab_mover%strtarget,strten,ab_mover%symrel,ucvol_next,ucvol0,vin_next,&
  & vout,xred_next)
if(ab_mover%optcell/=0)then
  call mkrdim(acell_next,rprim_next,rprimd_next)
  call metric(gmet,gprimd,-1,rmet,rprimd_next,ucvol_next)
else
  rprimd_next(:,:)=rprimd(:,:)
end if
! Convert input xred_next (reduced coordinates) to
! xcart_next (cartesian)
call xredxcart(ab_mover%natom,1,rprimd_next,xcart_next,xred_next)
! Compute the velocity at half of the new step
vel_nexthalf(:,:)=(xcart_next(:,:)-xcart(:,:))/ab_mover%dtion

! If needed, compute the velocity at present position
if(itime/=0)then
  vel(:,:)=(vel_nexthalf(:,:)+vel_prevhalf(:,:))*0.5_dp
end if

! %%% VERLET ALGORITHM BLOCKING ATOMS %%%%%%%%%%%%%%%%%%%%%%

!  Here, stop the atoms for which the scalar product of velocity
!  and force is negative, and recompute the kinetic energy.
   if(ionmov==7)then
     stopped(:)=0
     do ii=1,ab_mover%natom
       scprod=fcart(1,ii)*vel(1,ii)+&
&       fcart(2,ii)*vel(2,ii)+&
&       fcart(3,ii)*vel(3,ii)
       if(scprod<0.0_dp .and. itime/=0)then
         stopped(ii)=1
!        Shift the velocities of the previous half-step and current half-step,
!        so that the acceleration is correct but the present velocity vanishes.
         vel_prevhalf(:,ii)=vel_prevhalf(:,ii)-vel(:,ii)
         vel_nexthalf(:,ii)=vel_nexthalf(:,ii)-vel(:,ii)
         vel(:,ii)=0.0_dp
         xcart_next(:,ii)=xcart(:,ii)+ab_mover%dtion*vel_nexthalf(:,ii)
       end if
     end do

!    Establish a list of stopped atoms
     nstopped=sum(stopped(:))

     if(nstopped/=0)then
       write(message,'(a)') ' List of stopped atoms (ionmov=7) :'
       call wrtout(ab_out,message,'COLL')
       istopped=1
       do ii=1,ab_mover%natom
         if(stopped(ii)==1)then
           stopped(istopped)=ii
           istopped=istopped+1
         end if
       end do
       do ii=1,nstopped,16
         write(message, '(16i4)' )stopped(ii:min(ii+15,nstopped))
         call wrtout(ab_out,message,'COLL')
       end do
!      Now, compute the corrected kinetic energy
!      Generate xred_next from xcart_next
       call xredxcart(ab_mover%natom,-1,rprimd_next,xcart_next,xred_next)
!      Store xred_next, and eventual acell_next and rprim_next in vin
       option=1
       call xfpack(acell_next,acell0,fred_corrected,&
         & ab_mover%natom,ndim,ab_mover%nsym,ab_mover%optcell,&
         & option,rprim_next,rprimd0,ab_mover%strtarget,strten,&
         & ab_mover%symrel,ucvol_next,ucvol0,vin_next,vout,&
         & xred_next)
       ekin_corr=0.0_dp
       do ii=1,ab_mover%natom
         do jj=1,3
!          Warning : the fixing of atomis is implemented in reduced
!          coordinates, so that this expression is wrong
           if (ab_mover%iatfix(jj,ii) == 0) then
             ekin_corr=ekin_corr+0.5_dp*ab_mover%amass(ii)*vel(jj,ii)**2
           end if
         end do
       end do
!      End of test nstopped/=0
     end if

!    End of test ionmov==7
   end if


write(*,*) 'bfgs 07'
! ##########################################################
! ### 07. Update the history with the prediction

! Increase indexes
hist%ihist=hist%ihist+1

!Transfer vin  to xred, acell and rprim
 option=2
 call xfpack(acell, acell0, fred, ab_mover%natom, ndim,&
& ab_mover%nsym, ab_mover%optcell, option, rprim, rprimd0,&
& ab_mover%strtarget, strten, ab_mover%symrel, ucvol, ucvol0,&
& vin, vout, xred)

      write(*,*) 'RPRIM original'
      do ii=1,3
         write(*,*) rprim(:,ii)
      end do


 !Compute rprimd from rprim and acell
 do kk=1,3
    do jj=1,3
       rprimd(jj,kk)=rprim(jj,kk)*acell(jj)
    end do
 end do

      write(*,*) 'RPRIMD computed'
      do ii=1,3
         write(*,*) rprimd(:,ii)
      end do

 !Compute xcart from xred, and rprimd
 call xredxcart(ab_mover%natom,1,rprimd,xcart,xred)

 hist%histXF(:,:,1,hist%ihist)=xcart(:,:)
 hist%histXF(:,:,2,hist%ihist)=xred(:,:)
 hist%histV(:,:,hist%ihist)=hist%histV(:,:,hist%ihist-1)

 hist%histA(:,hist%ihist)=acell(:)
 hist%histR(:,:,hist%ihist)=rprimd(:,:)

!DEBUG
 write(*,*) 'Exiting from BFGS'
 write(*,*) 'VECTOR INPUT (vin)'
 do ii=1,ndim,3
    write(*,*) ii,vin(ii:ii+2)
 end do
!DEBUG

write(*,*) 'bfgs 08'
! ##########################################################
! ### 08. Deallocate in the last iteration

 if(itime==ntime)then
    if (allocated(vin)) deallocate(vin)
    if (allocated(vout)) deallocate(vout)
    if (allocated(vin_prev)) deallocate(vin_prev)
    if (allocated(vout_prev)) deallocate(vout_prev)
    if (allocated(hessin)) deallocate(hessin)
 end if

end subroutine pred_verlet
!!***
