!{\src2tex{textfont=tt}}
!!****f* ABINIT/scphon
!! NAME
!! scphon
!!
!! FUNCTION
!!  Using a supercell, calculate a self consistent phonon structure
!!  as in PRL 100 095901 (2008). The initial phonon eigenvectors and
!!  eigenvalues are read in, and then atoms are displaced according
!!  to the normal modes populated at a given temperature until
!!  convergence of the vibrational free energy (or so I hope)
!!
!! Other references:
!!   Computational Materials Science 44 (2009) 888-894
!!   Phys Rev B 78, 184304 (2008)
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  amass(natom)=mass of each atom, in unit of electronic mass (=amu*1822...)
!!  atindx(natom)=index table for atoms (see scfcv.f)
!!  atindx1(natom)=index table for atoms, inverse of atindx (see scfcv.f)
!!  cpus= cpu time limit in seconds
!!  dtfil <type(datafiles_type)>=variables related to files
!!  dtset <type(dataset_type)>=all input variables for this dataset
!!   | mband=maximum number of bands
!!   | mgfft=maximum size of 1D FFTs for the "coarse" grid (see NOTES below)
!!   | mkmem =number of k points which can fit in memory; set to 0 if use disk
!!   | mpw=maximum dimensioned size of npw.
!!   | natom=number of atoms in cell.
!!   | nfft=(effective) number of FFT grid points (for this processor)
!!   |      for the "coarse" grid (see NOTES below)
!!   | nkpt=number of k points
!!   | nspden=number of spin-density components
!!   | nsppol=1 for unpolarized, 2 for spin-polarized
!!   | nsym=number of symmetry elements in space group
!!  ecore=core psp energy (part of total energy) (hartree)
!!  kg(3,mpw*mkmem)=reduced planewave coordinates.
!!  mpi_enreg=informations about MPI parallelization
!!  nattyp(ntypat)= # atoms of each type.
!!  nfftf=(effective) number of FFT grid points (for this processor)
!!       for the "fine" grid (see NOTES below)
!!  npwarr(nkpt)=number of planewaves in basis at this k point
!!  nspinor=number of spinorial components of the wavefunctions
!!  paw_dmft  <type(paw_dmft_type)>= paw+dmft related data
!!  pawang <type(pawang_type)>=paw angular mesh and related data
!!  pawfgr <type(pawfgr_type)>=fine grid parameters and related data
!!  pawrad(ntypat*usepaw) <type(pawrad_type)>=paw radial mesh and related data
!!  pawtab(ntypat*usepaw) <type(pawtab_type)>=paw tabulated starting data
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!   | mpsang= 1+maximum angular momentum for nonlocal pseudopotentials
!!  pwind(pwind_alloc,2,3) = array used to compute
!!           the overlap matrix smat between k-points (see initberry.f)
!!  pwind_alloc = first dimension of pwind
!!  pwnsfac(2,pwind_alloc) = phase factors for non-symmorphic translations
!!                           (see initberry.f)
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!  ylm(mpw*mkmem,mpsang*mpsang*useylm)= real spherical harmonics for each G and k point
!!  ylmgr(mpw*mkmem,3,mpsang*mpsang*useylm)= gradients of real spherical harmonics
!!
!! OUTPUT
!!  resid(mband*nkpt*nsppol)=residuals for each band over all k points and spins
!!
!! SIDE EFFECTS
!!  cg(2,mpw*nspinor*mband*mkmem*nsppol)=updated wavefunctions;  if mkmem>=nkpt,
!!         these are kept in a disk file.
!!  dtefield <type(efield_type)> = variables related to Berry phase
!!      calculations (see initberry.f)
!!  eigen(mband*nkpt*nsppol)=array for holding eigenvalues (hartree)
!!  electronpositron <type(electronpositron_type)>=quantities for the electron-positron annihilation
!!  hdr <type(hdr_type)>=the header of wf, den and pot files
!!  indsym(4,nsym,natom)=indirect indexing array for atom labels
!!  initialized= if 0 the initialization of the gstate run is not yet finished
!!  irrzon(nfft**(1-1/nsym),2,(nspden/nsppol)-3*(nspden/4))=irreducible zone data
!!  occ(mband*nkpt*nsppol)=occupation number for each band (often 2) at each k point
!!  pawrhoij(natom*usepaw) <type(pawrhoij_type)>= -PAW only- atomic occupancies
!!  phnons(2,nfft**(1-1/nsym),(nspden/nsppol)-3*(nspden/4))=nonsymmorphic translation phases
!!  results_gs <type(results_gs_type)>=results (energy and its components,
!!   forces and its components, the stress tensor) of a ground-state computation
!!   (should be made a pure output quantity)
!!  rhog(2,nfftf)=array for Fourier transform of electron density
!!  rhor(nfftf,nspden)=array for electron density in el./bohr**3
!!  scf_history <type(scf_history_type)>=arrays obtained from previous SCF cycles
!!  symrec(3,3,nsym)=symmetry operations in reciprocal space
!!  taug(2,nfftf*dtset%usekden)=array for Fourier transform of kinetic energy density
!!  taur(nfftf,nspden*dtset%usekden)=array for kinetic energy density
!!  wffnew,wffnow=struct info for wf disk files.
!!  wvl <type(wvl_data)>=all wavelets data.
!!  xred(3,natom)=reduced dimensionless atomic coordinates
!!  xred_old(3,natom)= at input, previous reduced dimensionless atomic coordinates
!!                     at output, current xred is transferred to xred_old
!!
!! PARENTS
!!      gstate
!!
!! CHILDREN
!!      fappnd,prtxvf
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon(amass, atindx, atindx1, cg, cpus, dtefield,&
& dtfil, dtset, ecore, eigen, electronpositron, hdr, indsym, initialized,&
& irrzon, kg, mpi_enreg, nattyp, nfftf, npwarr, nspinor, occ,&
& pawang, paw_dmft, pawfgr, pawrad, pawrhoij, pawtab, phnons, psps,&
& pwind, pwind_alloc, pwnsfac, rec_set, resid, results_gs, rhog, rhor,&
& rprimd, scf_history, fatvshift, symrec, taug, taur, wffnew, wffnow,&
& wvl, xred, xred_old, ylm, ylmgr)

 use defs_basis
 use defs_datatypes
 use m_wffile
 use defs_abitypes
 use defs_scftypes
 use defs_rectypes
 use defs_wvltypes
 use m_electronpositron, only : electronpositron_type
 use m_primcell_ddb_info
 use m_phdos
 use m_paw_dmft, only: paw_dmft_type

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_42_geometry
 use interfaces_67_common
 use interfaces_95_drive, except_this_one => scphon
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: pwind_alloc
 integer,intent(inout) :: initialized,nfftf,nspinor
 real(dp),intent(in) :: cpus,ecore
 type(MPI_type),intent(inout) :: mpi_enreg
 type(datafiles_type),intent(inout) :: dtfil
 type(dataset_type),intent(inout) :: dtset
 type(efield_type),intent(inout) :: dtefield
 type(electronpositron_type),pointer :: electronpositron
 type(hdr_type),intent(inout) :: hdr
 type(paw_dmft_type) :: paw_dmft
 type(pawang_type),intent(in) :: pawang
 type(pawfgr_type) :: pawfgr
 type(pseudopotential_type),intent(in) :: psps
 type(results_gs_type),intent(inout) :: results_gs
  type(recursion_type),intent(inout) ::rec_set
 type(scf_history_type),intent(inout) :: scf_history
 real(dp),intent(in) :: fatvshift
 type(wffile_type),intent(inout) :: wffnew,wffnow
 type(wvl_data),intent(inout) :: wvl
!arrays
 integer,intent(in) :: atindx(dtset%natom),atindx1(dtset%natom)
 integer,intent(inout) :: indsym(4,dtset%nsym,dtset%natom)
  !no_abirules
  integer, intent(inout) :: irrzon(dtset%nfft**(1-1/dtset%nsym),2,(dtset%nspden/dtset%nsppol)-3*(dtset%nspden/4))
   !(nfft**(1-1/nsym) is 1 if nsym==1, and nfft otherwise)
  integer, intent(in) :: kg(3,dtset%mpw*dtset%mkmem)
  integer, intent(in) :: nattyp(psps%ntypat),npwarr(dtset%nkpt),pwind(pwind_alloc,2,3)
  integer, intent(inout) :: symrec(3,3,dtset%nsym)
  real(dp), intent(in) :: amass(dtset%natom)
  real(dp), intent(inout) :: rprimd(3,3)
  real(dp), intent(inout) :: cg(2,dtset%mpw*nspinor*dtset%mband*dtset%mkmem*dtset%nsppol)
  real(dp), intent(inout) :: eigen(dtset%mband*dtset%nkpt*dtset%nsppol)
  real(dp), intent(inout) :: occ(dtset%mband*dtset%nkpt*dtset%nsppol)
  real(dp), intent(inout) :: phnons(2,dtset%nfft**(1-1/dtset%nsym),(dtset%nspden/dtset%nsppol)-3*(dtset%nspden/4))
   !(nfft**(1-1/nsym) is 1 if nsym==1, and nfft otherwise)
  real(dp), intent(in) :: pwnsfac(2,pwind_alloc)
  real(dp), pointer :: rhog(:,:),rhor(:,:)
  real(dp), pointer :: taug(:,:),taur(:,:)
  real(dp), intent(out) :: resid(dtset%mband*dtset%nkpt*dtset%nsppol)
  real(dp), intent(inout) :: xred(3,dtset%natom),xred_old(3,dtset%natom)
  real(dp), intent(in) :: ylm(dtset%mpw*dtset%mkmem,psps%mpsang*psps%mpsang*psps%useylm)
  real(dp), intent(in) :: ylmgr(dtset%mpw*dtset%mkmem,3,psps%mpsang*psps%mpsang*psps%useylm)
  type(pawrad_type), intent(in) :: pawrad(psps%ntypat*psps%usepaw)
  type(pawrhoij_type), intent(inout) :: pawrhoij(mpi_enreg%natom*psps%usepaw)
  type(pawtab_type), intent(in) :: pawtab(psps%ntypat*psps%usepaw)

!Local variables-------------------------------
!scalars
 integer,parameter :: ndtpawuj=0
 integer :: iatom,iStep,iexit
 integer :: natom_primitive_cell,nphononq,nsym_primitive_cell
 real(dp) :: free_energy,old_free_energy,scphon_temp
 character(len=32) :: statusOut
 character(len=500) :: message
 character(len=fnlen) :: ddb_info_filename
 type(phonon_dos_type) :: t_phonon_dos
 type(primcell_ddb_info) :: pcell
!arrays
 integer :: supercell_multiplicity(3)
 integer,allocatable :: pcell_atom_in_supercell(:),qsym_map(:,:,:)
 integer,allocatable :: symrec_primitive_cell(:,:,:)
 real(dp),allocatable :: amass_pcell(:)
 real(dp),allocatable :: cartesian_displacements(:,:),forces_on_atoms_ft(:,:,:)
 real(dp),allocatable :: normal_mode_displacements(:,:)
 real(dp),allocatable :: normal_mode_displacements_old(:,:),phonon_eigval(:,:)
 real(dp),allocatable :: phonon_eigval2_averaged(:,:),phonon_eigval_ref(:,:)
 real(dp),allocatable :: phonon_eigvec_ref(:,:,:,:),phononq(:,:)
 real(dp),allocatable :: sqrt_amass_pcell(:),supercell_vectors(:,:),vel(:,:)
 real(dp),allocatable :: xcart(:,:),xcart0(:,:)
 type(macro_uj_type)  :: dtpawuj(ndtpawuj)

! ************************************************************************

!write(*,*) 'scphon 01'
!##########################################################
!### 01. Initial settings, only for first iteration

 write (message,'(a)') ' scphon: enter '
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')

!the number of times the primitive unit cell has been replicated in each
!dimension
!FIXME: this could be inferred from the supercell size, avoiding possible mismatches
!of scphon_supercell with the real one...
 supercell_multiplicity = dtset%scphon_supercell
 write (message,'(a,3I6)') ' SC phonons: Found supercell multiplicity of', &
& supercell_multiplicity
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')

!this is the number of atoms for the primitive unit cell, in which the phonons
!were calculated
 natom_primitive_cell=nint(dble(dtset%natom)&
& /dble(supercell_multiplicity(1))&
& /dble(supercell_multiplicity(2))&
& /dble(supercell_multiplicity(3)))
 write (message,'(a,I6,a)') 'Deduced that there are ', natom_primitive_cell, &
& ' atoms in the primitive unit cell'
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')
 allocate(amass_pcell(natom_primitive_cell))
 allocate(sqrt_amass_pcell(natom_primitive_cell))
 amass_pcell(:) = amass(1:natom_primitive_cell)
 sqrt_amass_pcell(:) = sqrt(amass_pcell(:))
 do iatom=1,natom_primitive_cell
   write (message,'(a,I6,a,E20.10)') ' mass of atom ', iatom, ' = ', amass_pcell(iatom)
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
 end do

!the temperature we want the phonons to be at
 scphon_temp = dtset%scphon_temp

!set energies for DOS output and integration
!t_phonon_dos%dossmear  =0.00001   ! about 2. cm-1
!t_phonon_dos%omega_step=0.0000047 ! about 1. cm-1
!t_phonon_dos%nomega=1000
 call init_phondos(t_phonon_dos,psps%ntypat,dtset%natom,1,1000,1,1,&
& smallest_real,greatest_real,0.0000047_dp,0.00001_dp)


!END input variables

!write(*,*) 'scphon 02'
!##########################################################
!### 02. Several allocations also for first time

!
!Allocate working arrays
!
 allocate(xcart(3, dtset%natom))
 allocate(xcart0(3, dtset%natom))
 allocate(vel(3, dtset%natom))
!Transform xred to cartesian coordinates.
 call xredxcart(dtset%natom, 1, rprimd, xcart, xred)
!initial positions
 xcart0=xcart

!number of qpoints for phonon integration should be = natom
!could be reduced by symmetries for reciprocal space,
!but then you need to reconstruct the real space atomic displacements below
!This is not implemented yet.
 nphononq=dtset%natom/natom_primitive_cell

!for each atom, the vector to the unit cell it is in
 allocate(supercell_vectors(3,dtset%natom))
 allocate(pcell_atom_in_supercell(dtset%natom))

!phonon q vectors we will consider. Should correspond to the superlattice
!of atoms we are using, eg 4x4x4 supercell
 allocate(phononq(3,nphononq))

!displacements of atoms from equilibrium positions
!written U_R in the PRL
!allocate(atom_displacements(3,dtset%natom))


!the following depend on natom_primitive_cell

!eigvec of reference phonon system
 allocate(phonon_eigvec_ref(2,3*natom_primitive_cell,3*natom_primitive_cell,nphononq))

!eigval of reference phonon system
 allocate(phonon_eigval_ref(3*natom_primitive_cell,nphononq))

!eigval of present phonon system, extracted from instantaneous forces
 allocate(phonon_eigval(3*natom_primitive_cell,nphononq))

!eigval of present phonon system: averaged value of phonon_eigval
 allocate(phonon_eigval2_averaged(3*natom_primitive_cell,nphononq))

!classical displacements along each of the 3*natom normal modes
!written A_ks in the reference PRL, but without the 1/sqrt(mass) factor
 allocate(normal_mode_displacements(3*natom_primitive_cell,nphononq))
 allocate(normal_mode_displacements_old(3*natom_primitive_cell,nphononq))

!the absolute displacement of the atoms, in cartesian coordinates, since
!the beginning of time
 allocate(cartesian_displacements(3,dtset%natom))

!forces on each atom fcart are fourier transformed into the following array
!(in cartesian coordinates)
 allocate(forces_on_atoms_ft(2,3*natom_primitive_cell,nphononq))


!
!Initializations
!

!get generic phonon info from anaddb run, for later interpolation
 write (*,*) ' entering read_primcell'
 ddb_info_filename=trim(dtfil%filnam_ds(3))//'_PCINFO'
 ddb_info_filename=trim(ddb_info_filename)
 call read_primcell_ddb_info(ddb_info_filename,pcell)

!initialize the supercell vector grid (pointing to each unit cell in the supercell)
 call scphon_supercell_vectors_init(dtset%natom,natom_primitive_cell,&
& pcell,pcell_atom_in_supercell,supercell_multiplicity,&
& supercell_vectors,xred)

 write (message,'(a)') '  xred = '
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')
 do iatom=1,dtset%natom
   write (message,'(3E20.10)')  xred(:,iatom)
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
 end do

!initialize the phonon q grid based on the supercell size
 call scphon_qpoint_init (nphononq,phononq,supercell_multiplicity)

!initialize the reference phonon eigenvectors and eigenvalues from file
 call scphon_phonon_init (dtfil%fnameabi_phfrq,&
& dtfil%fnameabi_phvec,natom_primitive_cell,&
& nphononq,phonon_eigvec_ref,phonon_eigval_ref)

!set the initial phonon frequency average
 phonon_eigval2_averaged(:,:)=sign(one,phonon_eigval_ref(:,:))*phonon_eigval_ref(:,:)**2

!get the symmetry operations for the primitive unit cells
 nsym_primitive_cell=pcell%nsym
 allocate (symrec_primitive_cell(3,3,nsym_primitive_cell))
 symrec_primitive_cell(:,:,1:nsym_primitive_cell) = pcell%symrec

!mapping from 1 qpoint to another under symmetries
 allocate(qsym_map(nphononq,nsym_primitive_cell,2))
!initialize the mapping from one qpoint to another under the symops
 call scphon_build_qsym_map(nphononq,nsym_primitive_cell,phononq,&
& qsym_map,symrec_primitive_cell)

!initialize the first normal mode displacements
 call scphon_freq_to_normmode (qsym_map(:,1,2),natom_primitive_cell,normal_mode_displacements,&
& nphononq,phonon_eigval_ref,scphon_temp)
 write (message,'(a)') ' Have initialized normal_mode_displacements'
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')

!update xcart with difference between old and new normal mode displacements
!also initializes cartesian_displacements
 cartesian_displacements=zero
 call scphon_update_xcart (sqrt_amass_pcell,cartesian_displacements,dtset%natom,natom_primitive_cell,&
& normal_mode_displacements,&
& nphononq,pcell_atom_in_supercell,phonon_eigvec_ref,phononq,supercell_vectors,xcart,xcart0)

!in first iteration, apply full normal mode displacement
 normal_mode_displacements_old=normal_mode_displacements

 call print_phonfreq(-1,natom_primitive_cell,nphononq,phonon_eigval_ref)

!set old_free_energy to make sure at least 2 iterations are performed.
 old_free_energy=greatest_real

!
!Begin scphon relaxation.
!
 do iStep = 1, dtset%ntime, 1
   write(message, '(a,a,i3,a)' ) ch10, ' SCPHON STEP NUMBER ', iStep,&
&   '  ---------------------------------------------------------'
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')


!  transform to reduced coordinates for call to scfcv
   call xredxcart(dtset%natom, -1, rprimd, xcart, xred)
   write (message,'(a)') '  input xred = '
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
   do iatom=1,dtset%natom
     write (message,'(3E20.10)')  xred(:,iatom)
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
   end do
   write (message,'(a)') ch10
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')

!  
!  Main call: scfcv calculates electronic structure and forces, and so on.
!  
   call scfcv_tmp(atindx, atindx1, cg, cpus, dtefield, dtfil,dtpawuj, &
&   dtset, ecore, eigen, electronpositron, hdr, istep, indsym, initialized,&
&   irrzon, kg, mpi_enreg, nattyp, ndtpawuj, nfftf, npwarr, nspinor, occ,&
&   paw_dmft, pawang, pawfgr, pawrad, pawrhoij, pawtab,&
&   phnons, psps, pwind, pwind_alloc, pwnsfac, rec_set, resid, results_gs, rhog, &
&   rhor, rprimd, scf_history, fatvshift,symrec, taug,taur,wffnew, wffnow, wvl, &
&   xred, xred_old, ylm, ylmgr)

!  Output coordinates and forces (not velocities, prtvel = 0) and total energy
   call prtxvf(results_gs%fcart,results_gs%fred,dtset%iatfix, ab_out, dtset%natom, &
&   0, vel, xcart,xred)
   call prtxvf(results_gs%fcart,results_gs%fred,dtset%iatfix, 06 , dtset%natom, &
&   0, vel, xcart,xred)


!  Check if fcart is in the opposite direction to cartesian_displacements for
!  each atom
   call scphon_check_fcart(cartesian_displacements,results_gs%fcart,dtset%natom)

!  Fourier transform forces to recirocal space (inside BZ)
!  FIXME: could be optimized perhaps, make it an FFT, but the grid will never be
!  very big.
   call scphon_ft_fcart(sqrt_amass_pcell,results_gs%fcart,dtset%natom,natom_primitive_cell,nphononq,phononq,&
&   forces_on_atoms_ft,pcell_atom_in_supercell,supercell_vectors)

!  Determine new frequencies according to forces
   call scphon_new_frequencies(forces_on_atoms_ft,istep,natom_primitive_cell,&
&   normal_mode_displacements,nphononq,nsym_primitive_cell,pcell,phonon_eigvec_ref,&
&   phonon_eigval2_averaged,phonon_eigval,phononq,qsym_map)

!  calculate the DOS
!  1. the following code is for the bare input phonon frequencies (no interpolation)
   t_phonon_dos%nomega=1000
   if (associated(t_phonon_dos%phdos)) deallocate(t_phonon_dos%phdos)
   allocate (t_phonon_dos%phdos(t_phonon_dos%nomega))
   call scphon_make_phonon_dos (t_phonon_dos%dossmear,natom_primitive_cell,&
&   t_phonon_dos%nomega,nphononq, t_phonon_dos%omega_max,t_phonon_dos%omega_min,&
&   t_phonon_dos%phdos,phonon_eigval)

!  2. interpolate phonon freq first, then calculate DOS
!  this will give nomega a new value!
   call scphon_interpolate_phonon_and_dos (natom_primitive_cell,&
&   nphononq,pcell,t_phonon_dos,phonon_eigval,phonon_eigvec_ref,&
&   phononq,supercell_multiplicity)


!  calculate vibrational free energy
   call scphon_free_energy(free_energy,istep,t_phonon_dos,scphon_temp)

!  Check whether Free Energy diff is below tolerance; if so, exit
!  from the istep loop
!  NOTE: tolmxf is usually in Ha/bohr, so we should probably have a conversion
!  factor like for the stress. F usually down to 1e-5 and energy down to 1e-10
!  or so...
   if (abs(free_energy-old_free_energy) < dtset%tolmxf) then
     write (message,'(a)') ' convergence in vibrational free energy has been reached: '
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
     write (message,'(E15.5,a,E15.5)') free_energy-old_free_energy, ' < ', dtset%tolmxf
     call wrtout(ab_out,message,'COLL')
     call wrtout(std_out,  message,'COLL')
     statusOut = "OK"
     iexit = 1
     exit
   end if

   iexit = 0
   if (iStep == dtset%ntime) then
     iexit = 1
     statusOut = "Failed"
   end if

!  determine new classical normal mode displacements
   call scphon_freq_to_normmode (qsym_map(:,1,2),natom_primitive_cell,normal_mode_displacements,&
&   nphononq,phonon_eigval,scphon_temp)

!  update xcart with difference between old and new normal mode displacements
   call scphon_update_xcart (sqrt_amass_pcell,cartesian_displacements,&
&   dtset%natom,natom_primitive_cell,&
&   normal_mode_displacements-normal_mode_displacements_old,&
&   nphononq,pcell_atom_in_supercell,phonon_eigvec_ref,phononq,supercell_vectors,xcart,xcart0)

!  save present displacements for next step
   normal_mode_displacements_old=normal_mode_displacements

   old_free_energy=free_energy

 end do !                       end istep do

 call xredxcart(dtset%natom, -1, rprimd, xcart, xred)

!Free working arrays
 deallocate(xcart)
 deallocate(xcart0)
 deallocate(vel)
 deallocate(supercell_vectors)
 deallocate(phononq)
 deallocate(phonon_eigvec_ref)
 deallocate(phonon_eigval_ref)
 deallocate(phonon_eigval)
 deallocate(phonon_eigval2_averaged)
 deallocate(normal_mode_displacements)
 deallocate(normal_mode_displacements_old)
 deallocate(forces_on_atoms_ft)
 deallocate(qsym_map)

 call destroy_primcell_ddb_info(pcell)


!XML output of the status
 if (mpi_enreg%me == 0 .and. dtset%prtxml == 1) then
   write(ab_xml_out, "(A)") '    <geometryMinimisation type="scphon">'
   write(ab_xml_out, "(A,A,A)") '      <status cvState="', trim(statusOut) , &
&   '" stop-criterion="tolmxf" />'
   write(ab_xml_out, "(A)") '    </geometryMinimisation>'
 end if


end subroutine scphon
!!***

!!****f* ABINIT/scphon_phonon_init
!! NAME
!! scphon_phonon_init
!!
!! FUNCTION
!! Return phonon frequencies and eigenvectors initialized in
!! phonon_eigvec_ref and phonon_eigval_ref, read in from files
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! dtfil= datatype for file names and units
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!!
!! OUTPUT
!! phonon_eigval_ref=phonon eigenfrequencies, from the anaddb equil run
!! phonon_eigvec_ref=reference phonon eigenvectors, from the anaddb equil run
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_phonon_init (fnameabi_phfrq,fnameabi_phvec,&
&  natom_primitive_cell,nphononq,phonon_eigvec_ref,&
&  phonon_eigval_ref)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom_primitive_cell,nphononq
 character(len=fnlen),intent(in) :: fnameabi_phfrq
 character(len=fnlen),intent(in) :: fnameabi_phvec

!arrays
 real(dp),intent(out) :: phonon_eigval_ref(3*natom_primitive_cell,nphononq)
 real(dp),intent(out) :: phonon_eigvec_ref(2,3*natom_primitive_cell,3*natom_primitive_cell,nphononq)

!Local variables-------------------------------
!scalars
 integer :: imode_primitive_cell,indx,iq,jmode_primitive_cell
 integer :: phfrq_unit,phvec_unit
!character(len=500) :: message

! ************************************************************************

!read in _PHFRQ and _PHVEC files from anaddb output
!

!first eigenvalues
 phfrq_unit=300
 open (unit=phfrq_unit,file=fnameabi_phfrq)

!read in header of 3 lines
 read(phfrq_unit,*)
 read(phfrq_unit,*)
 read(phfrq_unit,*)

!for each qpoint, read in index frq_1 frq_2 ... frq_3natom
 do iq=1,nphononq
   read (phfrq_unit,'(i6)',ADVANCE='NO') indx
   do imode_primitive_cell=1,3*natom_primitive_cell
     read (phfrq_unit,'(e16.8,2x)',ADVANCE='NO') phonon_eigval_ref(imode_primitive_cell,iq)
   end do
   read (phfrq_unit,*)
 end do

 close (phfrq_unit)

!now eigenvectors
 phvec_unit=300
 open (unit=phvec_unit,file=fnameabi_phvec)

!read in header of 3 lines
 read(phvec_unit,*)
 read(phvec_unit,*)
 read(phvec_unit,*)

!for each qpoint, read in index frq_1 frq_2 ... frq_3natom
 do iq=1,nphononq
   read (phvec_unit,'(i6)',ADVANCE='NO') indx
   do imode_primitive_cell=1,3*natom_primitive_cell
     do jmode_primitive_cell=1,3*natom_primitive_cell
       read (phvec_unit,'(2e25.15,2x)',ADVANCE='NO') phonon_eigvec_ref(:,jmode_primitive_cell,imode_primitive_cell,iq)
     end do
   end do
   read (phvec_unit,*)
 end do

 close (phvec_unit)

end subroutine scphon_phonon_init
!!***

!!****f* ABINIT/scphon_qpoint_init
!! NAME
!! scphon_qpoint_init
!!
!! FUNCTION
!! Initialize the qpoint grid which should be read in for the frequencies
!! and eigenvectors of the equilibrium primitive unit cell, and used for FT
!! of supercell quantities.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! supercell_multiplicity=number of times the primitive unit cell is repeated
!!
!! OUTPUT
!! phononq= phonon q vectors, should be the same as those used in anaddb run
!!   (reduced coordinates)
!! FIXME: add a check to make sure this is the case.
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_qpoint_init (nphononq,phononq,supercell_multiplicity)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_32_util
 use interfaces_56_recipspace
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nphononq
!arrays
 integer,intent(in) :: supercell_multiplicity(3)
 real(dp),intent(out) :: phononq(3,nphononq)

!Local variables-------------------------------
!scalars
 integer :: brav,iout,iqpt,mqpt,nqpt,nqshft,option
 real(dp) :: res
!character(len=500) :: message
!arrays
 integer :: qptrlatt(3,3)
 real(dp) :: kpt(3),qshift(3)

! *************************************************************************

!always impose no shift in q
 nqshft=1
 qshift=(/zero,zero,zero/)

 qptrlatt = 0
 qptrlatt(1,1) = supercell_multiplicity(1)
 qptrlatt(2,2) = supercell_multiplicity(2)
 qptrlatt(3,3) = supercell_multiplicity(3)

 mqpt= qptrlatt(1,1)*qptrlatt(2,2)*qptrlatt(3,3) &
& +qptrlatt(1,2)*qptrlatt(2,3)*qptrlatt(3,1) &
& +qptrlatt(1,3)*qptrlatt(2,1)*qptrlatt(3,2) &
& -qptrlatt(1,2)*qptrlatt(2,1)*qptrlatt(3,3) &
& -qptrlatt(1,3)*qptrlatt(2,2)*qptrlatt(3,1) &
& -qptrlatt(1,1)*qptrlatt(2,3)*qptrlatt(3,2)

 iout = 6
 option=1
 brav=1
 call smpbz(brav,iout,qptrlatt,mqpt,nqpt,&
& nqshft,option,qshift,phononq)

 if (mqpt /= nqpt .or. nqpt /= nphononq) then
   write (*,*) 'scphon_qpoint_init: Error in num of qpt: mqpt,nqpt,nphononq = ',&
&   mqpt,nqpt,nphononq
   stop
 end if

!FIXME: should check these qpt are in the same order as the frequencies and
!eigenvectors!!!!
!
!reduce spqpt to correct zone
 do iqpt=1,nphononq
   call wrap2_pmhalf(phononq(1,iqpt),kpt(1),res)
   call wrap2_pmhalf(phononq(2,iqpt),kpt(2),res)
   call wrap2_pmhalf(phononq(3,iqpt),kpt(3),res)
   phononq(:,iqpt) = kpt
 end do

end subroutine scphon_qpoint_init
!!***

!!****f* ABINIT/scphon_ft_fcart
!! NAME
!! scphon_ft_fcart
!!
!! FUNCTION
!! Fourier Transform cartesian forces on all supercell atoms, with respect to
!! the supercell lattice vectors (ie multiples of the primitive unit cell which
!! are contained in the supercell). This returns a force on each atom in the
!! primitive unit cell, for each q-vector in the dual grid of the supercell.
!!
!! The force is divided by the square root of the mass of the appropriate atom, in prevision of the
!! calculation of new frequencies.
!!
!! The dual grid should be the same as the input q-point grid on which the
!! equilibrium phonons were calculated.
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! sqrt_amass_pcell= masses of the atoms in the primitive unit cell, in atomic units
!! fcart= forces on all supercell atoms, in cartesian coordinates
!! natom= number of atoms in the full supercell
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! phononq= phonon q vectors used in anaddb run (reduced coordinates)
!! pcell_atom_in_supercell= mapping of atoms to an atom index in the primitive
!!   unit cell
!! supercell_vectors= vector for each atom in the supercell, which points
!!   to the unit cell it is contained in, in integer units of the primitive cell
!!
!! OUTPUT
!! forces_on_atoms_ft= FT of cartesian forces on atoms, wrt the superlattice
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_ft_fcart(sqrt_amass_pcell,fcart,natom,natom_primitive_cell,nphononq,phononq,&
&   forces_on_atoms_ft,pcell_atom_in_supercell,supercell_vectors)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_16_hideleave
!End of the abilint section

 implicit none

!Arguments ------------------------------------
 !scalar
!scalars
 integer,intent(in) :: natom,natom_primitive_cell,nphononq
!arrays
 integer,intent(in) :: pcell_atom_in_supercell(natom)
 real(dp),intent(in) :: fcart(3,natom),phononq(3,nphononq)
 real(dp),intent(in) :: sqrt_amass_pcell(natom_primitive_cell)
 real(dp),intent(in) :: supercell_vectors(3,natom)
 real(dp),intent(out) :: forces_on_atoms_ft(2,3*natom_primitive_cell,nphononq)

!Local variables-------------------------------
!scalars
 integer :: iatom,iatom_in_pcell,idir,indx_pcell,iq
 real(dp) :: argument
 character(len=500) :: message

! *************************************************************************

  write(ab_out,*) natom_primitive_cell,nphononq,natom

 if (natom_primitive_cell*nphononq /= natom) then
   write (message,'(a,a)') 'Error: number of phonon q times number of atoms ',&
&   ' in unit cell should be equal to number of atoms in supercell'
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
   call leave_new('COLL')
 end if

 forces_on_atoms_ft = zero
 do iq=1,nphononq
   do iatom=1,natom
     iatom_in_pcell=pcell_atom_in_supercell(iatom)

     argument=sum(supercell_vectors(1:3,iatom)*phononq(1:3,iq))
!    abinit does not add 2 pi factor into k vectors
     argument=argument*two_pi

     do idir=1,3
       indx_pcell=idir+(iatom_in_pcell-1)*3
!      Presumes fcart is real (which is certain), so its FT has some inversion
!      symmetry
       forces_on_atoms_ft(1,indx_pcell,iq)=forces_on_atoms_ft(1,indx_pcell,iq)+&
&       fcart(idir,iatom)*cos(argument)/sqrt_amass_pcell(iatom_in_pcell)
       forces_on_atoms_ft(2,indx_pcell,iq)=forces_on_atoms_ft(2,indx_pcell,iq)+&
&       fcart(idir,iatom)*sin(argument)/sqrt_amass_pcell(iatom_in_pcell)
     end do
   end do
 end do

!looks like this is the choice of normalization in the PRL
 forces_on_atoms_ft = forces_on_atoms_ft/sqrt(dble(nphononq))

!write (*,*) ' done with FT '
!
!! there was a factor missing of 1/natom
!test_inverse_ft = zero
!do iatom=1,natom
!do iq=1,nphononq
!argument=sum(phononq(:,iq)*supercell_vectors(:,iatom))
!argument=-two_pi*argument
!test_inverse_ft(1,:,iatom) = test_inverse_ft(1,:,iatom) &
!&     + forces_on_atoms_ft(1,:,iq)*cos(argument) &
!&     - forces_on_atoms_ft(2,:,iq)*sin(argument)
!test_inverse_ft(2,:,iatom) = test_inverse_ft(2,:,iatom) &
!&     + forces_on_atoms_ft(2,:,iq)*cos(argument) &
!&     + forces_on_atoms_ft(1,:,iq)*sin(argument)
!end do
!write (*,*) 'ratio of fcart and 2xFTed quantity (for real part), and Im part: '
!idir=1
!write (*,*) fcart(idir,iatom)/(tol12+test_inverse_ft(1,idir,iatom)), test_inverse_ft(2,idir,iatom)
!idir=2
!write (*,*) fcart(idir,iatom)/(tol12+test_inverse_ft(1,idir,iatom)), test_inverse_ft(2,idir,iatom)
!idir=3
!write (*,*) fcart(idir,iatom)/(tol12+test_inverse_ft(1,idir,iatom)), test_inverse_ft(2,idir,iatom)
!end do

 write (*,*) ' Re(FT of fcart) / sqrt(M) : '
 write (*,'(3(E20.10,2x))') forces_on_atoms_ft(1,:,:)
 write (*,*)

end subroutine scphon_ft_fcart
!!***

!!****f* ABINIT/scphon_new_frequencies
!! NAME
!! scphon_new_frequencies
!!
!! FUNCTION
!! Calculate new frequencies from forces on supercell atoms, then symmetrize
!! them and add them to the averaged frequencies
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! istep= number of the present iteration of the SC phonon calculations, for
!!   printing purposes
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! normal_mode_displacements= calculated displacements of canonical coordinates
!!   (normal modes) of phonons at desired temperature.
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! nsym_primitive_cell= number of symmetries in the primitive unit cell
!! pcell=container type with ancillary variables and dimensions from anaddb run
!! phonon_eigvec_ref=reference phonon eigenvectors, from the anaddb equil run
!! phononq= phonon q vectors used in anaddb run (reduced coordinates)
!! qsym_map= map of qpoints onto one another, by sym ops:
!!   q_{qmap(iq,isym)} = S_{isym} q_{iq}
!!
!! OUTPUT
!! phonon_eigval=phonon eigenfrequencies, updated inside SC phonon run
!!
!! SIDE EFFECTS
!! phonon_eigval2_averaged= phonon frequencies squared, averaged over all
!! iterations to date
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_new_frequencies(forces_on_atoms_ft,istep,natom_primitive_cell,&
&    normal_mode_displacements,nphononq,nsym_primitive_cell,pcell,phonon_eigvec_ref,&
&    phonon_eigval2_averaged,phonon_eigval,phononq,qsym_map)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_primcell_ddb_info

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_42_geometry
 use interfaces_72_response
 use interfaces_95_drive, except_this_one => scphon_new_frequencies
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 type(primcell_ddb_info),intent(inout) :: pcell
 integer,intent(in) :: istep,natom_primitive_cell,nphononq,nsym_primitive_cell
!arrays
 integer,intent(in) :: qsym_map(nphononq,nsym_primitive_cell,2)
 real(dp),intent(in) :: forces_on_atoms_ft(2,3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: normal_mode_displacements(3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: phonon_eigvec_ref(2,3*natom_primitive_cell,3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: phononq(3,nphononq)
 real(dp),intent(inout) :: phonon_eigval2_averaged(3*natom_primitive_cell,nphononq)
 real(dp),intent(out) :: phonon_eigval(3*natom_primitive_cell,nphononq)

!Local variables-------------------------------
!scalars
 integer :: iatom,idir,imode_primitive_cell,iq,iq_image,isym
 integer :: itimrev,msize,multiplicity_q,  iatom2
 real(dp) :: timrev_sign
!arrays
 integer :: symqpoint_flag(nphononq)

 real(dp) :: scalprod_eigvec_force(2)
 real(dp) :: symfreq(3*natom_primitive_cell)
 real(dp) :: tmp_phonon_eigval(3*natom_primitive_cell,nphononq)

 real(dp) :: rprimd(3,3)

 real(dp), allocatable :: dynmat(:,:,:,:,:,:)
 character (len=500) :: message

!************************************************************************

 call mkrdim(pcell%acell,pcell%rprim,rprimd)

!calculate SQUARE of new frequencies from forces
!NOTE: only real part of scalar product is kept
 phonon_eigval = zero
 do iq=1,nphononq
   do iatom=1,natom_primitive_cell
     do idir=1,3
       imode_primitive_cell=idir+3*(iatom-1)
       if (abs(normal_mode_displacements(imode_primitive_cell,iq)) < tol12) cycle

!      here sum is over all atoms and directions
       scalprod_eigvec_force(1)=sum(phonon_eigvec_ref(1,:,imode_primitive_cell,iq)*forces_on_atoms_ft(1,:,iq)) &
&       -sum(phonon_eigvec_ref(2,:,imode_primitive_cell,iq)*forces_on_atoms_ft(2,:,iq))
       scalprod_eigvec_force(2)=sum(phonon_eigvec_ref(1,:,imode_primitive_cell,iq)*forces_on_atoms_ft(2,:,iq)) &
&       +sum(phonon_eigvec_ref(2,:,imode_primitive_cell,iq)*forces_on_atoms_ft(1,:,iq))
       write (*,*) 'scphon_new_frequencies: scal prd = ', scalprod_eigvec_force(1), scalprod_eigvec_force(2)
       phonon_eigval(imode_primitive_cell,iq)= -scalprod_eigvec_force(1) &
&       / normal_mode_displacements(imode_primitive_cell,iq)
     end do
   end do
 end do

 write (*,*) '  new frequencies^2 '
 write (*,'(6E20.10)')  phonon_eigval
 write (*,*)

 tmp_phonon_eigval = sign(one,phonon_eigval)*sqrt(abs(phonon_eigval))
 write (*,*) '  new frequencies '
 write (*,'(6E20.10)')  tmp_phonon_eigval
 write (*,*)

 call print_phonfreq(istep,natom_primitive_cell,nphononq,tmp_phonon_eigval)


!Symmetrize new frequencies (still in recip space)
 allocate (dynmat(2,3,natom_primitive_cell,3,natom_primitive_cell,1))
 symqpoint_flag = 0
 do iq=1,nphononq
!  if qpoint already symmetrized skip it
   if (symqpoint_flag(iq) == 1) cycle

   symfreq(:)=zero
   multiplicity_q=0
!  sum over images of qpoint (I think mode number is conserved btw equivalent
!  qpoints - the only alternative is for degenerate modes, so we do not care)
   do itimrev=1,2
     do isym=1,nsym_primitive_cell

       iq_image = qsym_map(iq,isym,itimrev)
       if (iq_image==0) cycle

!      add this contribution
       symfreq(:) = symfreq(:) + phonon_eigval(:,iq_image)
       multiplicity_q=multiplicity_q+1
     end do
     timrev_sign=-one
   end do

!  average and take square root
   symfreq(:) = symfreq/dble(multiplicity_q)
   symfreq(:) = sign(one,symfreq)*sqrt(abs(symfreq))

!  DEBUG
   write (400,*) 'before symdyma'
   write (400,'(3E20.10)') symfreq
!  ENDDEBUG

   msize=3*pcell%mpert*3*pcell%mpert
   dynmat=zero
!  cast the freq into dynamical matrix
!  only called for 1 qpoint, iq
   call scphon_freq_to_dynmat(dynmat,natom_primitive_cell,&
&   1,symfreq,phonon_eigvec_ref(:,:,:,iq))


!  DEBUG
   write (message,'(a)') 'dynamical matrix before symdyma'
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
   do iatom2=1,natom_primitive_cell
     do idir=1,3
       do iatom=1,natom_primitive_cell
         write (message,'(3(2E12.3,2x))') dynmat(:,:,iatom,idir,iatom2,1)
         call wrtout(ab_out,message,'COLL')
         call wrtout(std_out,  message,'COLL')
       end do
     end do
   end do
   write(*,*) 'phononq = ', phononq(:,iq)
!  ENDDEBUG

!  symmetrize the dynamical matrix
   call symdyma(dynmat,pcell%indsym,natom_primitive_cell,pcell%nsym,&
&   phononq(:,iq),rprimd,pcell%symrel)


!  DEBUG
   write (message,'(a)') 'dynamical matrix after symdyma'
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
   do iatom2=1,natom_primitive_cell
     do idir=1,3
       do iatom=1,natom_primitive_cell
         write (message,'(3(2E12.3,2x))') dynmat(:,:,iatom,idir,iatom2,1)
         call wrtout(ab_out,message,'COLL')
         call wrtout(std_out,  message,'COLL')
       end do
     end do
   end do
!  ENDDEBUG

!  re-extract the frequencies from the dynamical matrix, diagonalizing
   call scphon_dynmat_to_freq2(dynmat,natom_primitive_cell,&
&   1,symfreq,phonon_eigvec_ref(:,:,:,iq))

!  DEBUG
   write (400,*) 'after symdyma'
   write (400,'(3E20.10)') symfreq
!  ENDDEBUG

!  copy symmetrized value to images and flag them as done
   do itimrev=1,2
     do isym=1,nsym_primitive_cell
       iq_image=qsym_map(iq,isym,itimrev)
       if (iq_image==0) cycle

       phonon_eigval(:,iq_image) = symfreq(:)
       symqpoint_flag(iq_image) = 1
     end do
   end do

 end do ! iq

 deallocate (dynmat)

!call flush (400)

 write (*,*) '  symmetrized frequencies^2 '
 write (*,'(6E20.10)')  phonon_eigval
 write (*,*)

!Update average frequencies (averaged over all iterations up to now, counting
!the first iteration with the equilibrium phonon freq in step 0)

 phonon_eigval2_averaged=(dble(istep)*phonon_eigval2_averaged + phonon_eigval)/dble(istep+1)
 write (*,*) '  averaged frequencies^2 '
 write (*,'(6E20.10)')  phonon_eigval2_averaged
 write (*,*)

!return to ACTUAL frequencies
 phonon_eigval(:,:) = sign(one,phonon_eigval2_averaged(:,:))*sqrt( abs(phonon_eigval2_averaged(:,:)) )

 write (*,*) '  averaged frequencies ', istep
 write (*,'(6E20.10)')  phonon_eigval
 write (*,*)

end subroutine scphon_new_frequencies
!!***




!!****f* ABINIT/scphon_freq_to_normmode
!! NAME
!! scphon_freq_to_normmode
!!
!! FUNCTION
!! From updated phonon frequencies, and temperature, calculate displacement
!! of normal modes of phonons, adding an arbitrary sign (+- displacement)
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! phonon_eigval=phonon eigenfrequencies, updated inside SC phonon run
!! scphon_temp= phononic temperature in Ha
!!
!! OUTPUT
!! normal_mode_displacements= calculated displacements of canonical coordinates
!!   (normal modes) of phonons at desired temperature.
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE
! initialize the first normal mode displacements

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_freq_to_normmode (minusq_map,natom_primitive_cell,normal_mode_displacements,&
&   nphononq,phonon_eigval,scphon_temp)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_28_numeric_noabirule
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom_primitive_cell,nphononq
 real(dp),intent(in) :: scphon_temp
!arrays
 integer,intent(in) :: minusq_map(nphononq)
 real(dp),intent(in) :: phonon_eigval(3*natom_primitive_cell,nphononq)
 real(dp),intent(out) :: normal_mode_displacements(3*natom_primitive_cell,nphononq)

!Local variables-------------------------------
!scalars
 integer :: iminusq,imode_primitive_cell,iq,iseed=-10
 real(dp) :: bose_factor,random_sign

! *************************************************************************

!NOTE: what do we do about negative modes?

!NOTE: the only subtle part is choosing the sign of the displacement randomly
!Also presumes the mode is symmetric, which is only true for purely harmonic
!modes. As we have real displacements here, it may be an issue.
!One could also use a continuous variable instead of just +-1 for the
!amplitude of the normal coordinates...

 normal_mode_displacements=zero
 do iq=1, nphononq
!  
!  skip -q if it has already been filled
!  
   if ( abs(normal_mode_displacements(1,iq)) > tol10) cycle

   do imode_primitive_cell=1,3*natom_primitive_cell
!    skip gamma point acoustic modes
     if (abs(phonon_eigval(imode_primitive_cell,iq)) < tol10) cycle

!    for negative modes, use absolute value, until they eventually stabilize
!    also used (but not stated) in PRL
     bose_factor=one/(exp(abs(phonon_eigval(imode_primitive_cell,iq))/scphon_temp)-one)

!    always reseed random function
     random_sign=-one
     if (uniformrandom(iseed) > half) random_sign = one
!    write (*,*) ' iq im rand sign ',iq,imode_primitive_cell,random_sign
     normal_mode_displacements(imode_primitive_cell,iq) = random_sign *&
&     sqrt( (half+bose_factor) / abs(phonon_eigval(imode_primitive_cell,iq)) )
!    
!    NOTE: this is probably wrong in the PRL eq 6, as an atom mass is associated with a true
!    phonon mode, which moves all atoms a priori. The mass should be inserted
!    elsewhere, when the eigenvector comes in, which does have components by idir,
!    iatom
!    the sqrt(M) should be associated to the components of the eigenvectors, when
!    the cartesian displacements are calculated.
!    
   end do
!  
!  add normal_mode_displacement for -q
!  
   iminusq = minusq_map(iq)
   if (iminusq /= 0) then
     normal_mode_displacements(:,iminusq) = normal_mode_displacements(:,iq)
   end if

 end do

 write (*,*) '  new normal_mode_displacements '
 write (*,'(6E20.10)')  normal_mode_displacements
 write (*,*)

end subroutine scphon_freq_to_normmode
!!***



!!****f* ABINIT/scphon_update_xcart
!! NAME
!! scphon_update_xcart
!!
!! FUNCTION
!! From normal mode displacements, calculate the cartesian displacements
!! for all atoms in the supercell, and update xcart
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! sqrt_amass_pcell= masses of the atoms in the primitive unit cell, in atomic units
!! natom= number of atoms in the full supercell
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! normal_mode_displacements= calculated displacements of canonical coordinates
!!   (normal modes) of phonons at desired temperature.
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! pcell_atom_in_supercell= mapping of atoms to an atom index in the primitive
!!   unit cell
!! phonon_eigvec_ref=reference phonon eigenvectors, from the anaddb equil run
!! phononq= phonon q vectors used in anaddb run (reduced coordinates)
!! supercell_vectors= vector for each atom in the supercell, which points
!!   to the unit cell it is contained in, in integer units of the primitive cell
!!   lattice vectors
!! xcart0= initial positions of atoms, before perturbations
!!
!! OUTPUT
!! cartesian_displacements= displacements of all atoms in the supercell, in
!!   cartesian coordinates
!! xcart= old, then new positions of all atoms, in cartesian coordinates
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE
! update xcart with difference between old and new normal mode displacements

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_update_xcart (sqrt_amass_pcell,cartesian_displacements,natom,&
&   natom_primitive_cell,normal_mode_displacements,&
&   nphononq,pcell_atom_in_supercell,phonon_eigvec_ref,phononq,supercell_vectors,xcart,xcart0)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,natom_primitive_cell,nphononq
!arrays
 integer,intent(in) :: pcell_atom_in_supercell(natom)
 real(dp),intent(in) :: normal_mode_displacements(3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: phonon_eigvec_ref(2,3*natom_primitive_cell,3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: phononq(3,nphononq)
 real(dp),intent(in) :: sqrt_amass_pcell(natom_primitive_cell)
 real(dp),intent(in) :: supercell_vectors(3,natom),xcart0(3,natom)
 real(dp),intent(inout) :: cartesian_displacements(3,natom),xcart(3,natom)

!Local variables-------------------------------
! the instantaneous change in the cartesian displacements
!scalars
 integer :: iatom,iatom_in_pcell,imode_primitive_cell,iq
 real(dp) :: argument,cosarg,sinarg
!arrays
 real(dp),allocatable :: delta_cartesian_displacements(:,:)

! *************************************************************************

 allocate(delta_cartesian_displacements(3,natom))

!
!NOTE: only real part of cartesian displacement is taken here.
!what happens if there is an imaginary part?
!
 delta_cartesian_displacements=zero
 do iatom=1,natom
   iatom_in_pcell=pcell_atom_in_supercell(iatom)
   do iq=1,nphononq
     argument=-two_pi * sum(phononq(:,iq)*supercell_vectors(:,iatom))
     cosarg=cos(argument)
     sinarg=sin(argument)

     do imode_primitive_cell=1,3*natom_primitive_cell
       delta_cartesian_displacements(:,iatom)=delta_cartesian_displacements(:,iatom) + &
&       normal_mode_displacements(imode_primitive_cell,iq)*&
&       (phonon_eigvec_ref(1,(iatom_in_pcell-1)*3+1:iatom_in_pcell*3,imode_primitive_cell,iq)*cosarg  &
&       -phonon_eigvec_ref(2,(iatom_in_pcell-1)*3+1:iatom_in_pcell*3,imode_primitive_cell,iq)*sinarg) &
&       / sqrt_amass_pcell(iatom_in_pcell)
     end do
   end do
 end do
!Normalization chosen in PRL and confirmed by Souvatzis
 delta_cartesian_displacements = delta_cartesian_displacements &
& /sqrt(dble(nphononq))
!delta_cartesian_displacements = delta_cartesian_displacements &
!& /dble(nphononq)

!!this is needed for normalization of cartesian coordinates
!! NO - the phonon displacement vectors are already normalized correctly
!delta_cartesian_displacements = delta_cartesian_displacements &
!& /sqrt(dble(natom_primitive_cell))

 xcart = xcart0 + delta_cartesian_displacements
 cartesian_displacements=delta_cartesian_displacements

 write (*,*) '  new cartesian displacements = '
 write (*,'(3E20.10)')  delta_cartesian_displacements
 write (*,*)
 write (*,*) '  new xcart = '
 write (*,'(3E20.10)')  xcart
 write (*,*)

end subroutine scphon_update_xcart
!!***




!!****f* ABINIT/scphon_build_qsym_map
!! NAME
!! scphon_build_qsym_map
!!
!! FUNCTION
!! Build up map of transformation of qpoints into one another under symmetry
!! operations, and eventually time reversal.
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! nsym_primitive_cell= number of symmetries in the primitive unit cell
!! phononq= phonon q vectors used in anaddb run (reduced coordinates)
!! symrec_primitive_cell= reciprocal space symmetry operations
!!
!! OUTPUT
!! qsym_map= map of qpoints onto one another, by sym ops:
!!   q_{qmap(iq,isym)} = S_{isym} q_{iq}
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_build_qsym_map(nphononq,nsym_primitive_cell,phononq,&
&    qsym_map,symrec_primitive_cell)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nphononq,nsym_primitive_cell
!arrays
 integer,intent(in) :: symrec_primitive_cell(3,3,nsym_primitive_cell)
 integer,intent(out) :: qsym_map(nphononq,nsym_primitive_cell,2)
 real(dp),intent(in) :: phononq(3,nphononq)

!Local variables-------------------------------
!scalars
 integer :: dummy_prtvol,found,iq,iq_image,isym,itimrev,maxtimrev
 real(dp) :: shift,timrev_sign
 character(len=500) :: message
!arrays
 real(dp) :: image_qpoint(3),tmpvec(3)

! *************************************************************************

 dummy_prtvol=0
 qsym_map=0

!write (*,'(9I6)') symrec_primitive_cell

 do iq=1,nphononq

!  find small group of qpoint: returns
!  symq(4,2,nsym)= (integer) three first numbers define the G vector ;
!  fourth number is zero if the q-vector is not preserved, is 1 otherwise
!  second index is one without time-reversal symmetry, two with
!  call symq3(nsym_primitive_cell,phononq(:,iq),symq,symrec_primitive_cell,qtimrev,dummy_prtvol)

!  write (*,*) 'making qmap: ', iq, sum(symq(4,1,:)), qtimrev
!  if (qtimrev==0) maxtimrev=1
!  if (qtimrev==1) maxtimrev=2

   maxtimrev=2
   timrev_sign=one

!  find image of present qpoint through symmetry isym
   do itimrev=1,maxtimrev
     do isym=1,nsym_primitive_cell
!      if (symq(4,itimrev,isym)==0) cycle
       image_qpoint(:) = timrev_sign*&
&       (dble(symrec_primitive_cell(:,1,isym))*phononq(1,iq)&
&       +dble(symrec_primitive_cell(:,2,isym))*phononq(2,iq)&
&       +dble(symrec_primitive_cell(:,3,isym))*phononq(3,iq))
       call wrap2_pmhalf (image_qpoint(1),tmpvec(1),shift)
       call wrap2_pmhalf (image_qpoint(2),tmpvec(2),shift)
       call wrap2_pmhalf (image_qpoint(3),tmpvec(3),shift)
       found=0
       do iq_image=1,nphononq
!        write (*,'(3E20.8)') image_qpoint, phononq(:,iq_image)
!        write (*,'(E20.8)') sum((phononq(:,iq_image)-image_qpoint(:))**2)
         if (sum((phononq(:,iq_image)-tmpvec(:))**2) < tol10) then
           found=iq_image
           exit
         end if
       end do
       if (found==0) then
         write(message,'(a,I6,3E20.10)') 'Warning: sym qpoint not found ', isym, image_qpoint
         call wrtout(ab_out,message,'COLL')
         call wrtout(std_out,    message,'COLL')
         write(message,'(a,a)') ' is your qgrid compatible with all symmetries',&
&         ' of the primitive unit cell?'
         call wrtout(ab_out,message,'COLL')
         call wrtout(std_out,    message,'COLL')
!        call leave_new('COLL')
       end if
       qsym_map(iq,isym,itimrev) = found
     end do
     timrev_sign=-one
   end do
 end do

!write (*,*) '  qsym_map  with no timrev = '
!write (*,*) '  q1 q2 q3 ... on each line'
!do isym=1,nsym_primitive_cell
!write (*,'(8I6)')  qsym_map(:,isym,1)
!end do
!write (*,*)
!write (*,*) '  qsym_map  with timrev = '
!do isym=1,nsym_primitive_cell
!write (*,'(8I6)')  qsym_map(:,isym,2)
!end do

end subroutine scphon_build_qsym_map
!!***

!!****f* ABINIT/scphon_supercell_vectors_init
!! NAME
!! scphon_supercell_vectors_init
!!
!! FUNCTION
!! Calculate the integer vectors, for each atom in the supercell, which point to
!! the primitive unit cell the atom is contained in. Also output an array which
!! gives the equivalent atom in the primitive cell basis, if there are several
!! atoms in the primitive cell.
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! natom= number of atoms in the full supercell
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! pcell=container type with ancillary variables and dimensions from anaddb run
!! supercell_multiplicity=number of times the primitive unit cell is repeated
!!   along each axis, in the supercell
!! xred= reduced coordinates of all atoms in the supercell
!!
!! OUTPUT
!! pcell_atom_in_supercell= mapping of atoms to an atom index in the primitive
!!   unit cell
!! supercell_vectors= vector for each atom in the supercell, which points
!!   to the unit cell it is contained in, in integer units of the primitive cell
!!   lattice vectors
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_supercell_vectors_init(natom,natom_primitive_cell,&
&   pcell,pcell_atom_in_supercell,&
&   supercell_multiplicity,supercell_vectors,xred)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_primcell_ddb_info

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom,natom_primitive_cell
 type(primcell_ddb_info),intent(inout) :: pcell
!arrays
 integer,intent(in) :: supercell_multiplicity(3)
 integer,intent(out) :: pcell_atom_in_supercell(natom)
 real(dp),intent(in) :: xred(3,natom)
 real(dp),intent(out) :: supercell_vectors(3,natom)

!Local variables-------------------------------
!scalars
 integer :: iatom,iatom_primcell,idir,ii
 character(len=500) :: message
!arrays
 real(dp) :: relative_position(3)

! *************************************************************************

 write (message,'(a)') ' supercell vectors :  '
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')

 write (message,*) ' supercell multiplicity:',supercell_multiplicity(:)
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')

 write(ab_out,*) 'xred:'
 do ii=1,natom
   write(ab_out,*) xred(:,ii)
 end do

 do iatom=1,natom
!  for each atom find unit cell which it should belong to.
   do idir=1,3
     supercell_vectors(idir,iatom)=dble(floor(&
&     xred(idir,iatom)*supercell_multiplicity(idir) + 0.01))
     if (supercell_vectors(idir,iatom) >= supercell_multiplicity(idir) .or. &
&     supercell_vectors(idir,iatom) < 0) then
       write (message,'(a,2I6,2x,I6)') 'error in atomic disposition ',&
&       iatom,idir,supercell_vectors(idir,iatom)
       call wrtout(ab_out,message,'COLL')
       call wrtout(std_out,  message,'COLL')
       stop
     end if
   end do
!  find which atom in the primitive cell corresponds to the present one in the
!  supercell
   pcell_atom_in_supercell(iatom)=-1
!  1. if the ordering is preserved, just modulo natom_primitive_cell
   pcell_atom_in_supercell(iatom) = mod(iatom-1,natom_primitive_cell) + 1

!  2. otherwise really need to seek primcell atom
   relative_position=xred(:,iatom)-dble(supercell_vectors(:,iatom))/dble(supercell_multiplicity(:))
   do iatom_primcell=1,natom_primitive_cell
     if (sum((relative_position(:)-pcell%xred(:,iatom_primcell))**2) < tol8) then
       pcell_atom_in_supercell(iatom) = iatom_primcell
       exit
     end if
   end do
   if (pcell_atom_in_supercell(iatom) < 1  .or. &
&   pcell_atom_in_supercell(iatom) > natom_primitive_cell) then
     write (*,*) 'Error: pcell atom index is out of bounds '
     stop
   end if

   write (message,'(3E20.8)') supercell_vectors (:,iatom)
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
   write (message,'(a,I6)') ' equiv primcell atom ', pcell_atom_in_supercell (iatom)
   call wrtout(ab_out,message,'COLL')
   call wrtout(std_out,  message,'COLL')
 end do

 write(ab_out,*) 'supercell_vectors:'
 do ii=1,natom
   write(ab_out,*) supercell_vectors(:,ii)
 end do

end subroutine scphon_supercell_vectors_init
!!***



!!****f* ABINIT/scphon_check_fcart
!! NAME
!! scphon_check_fcart
!!
!! FUNCTION
!! Check that the cartesian displacements presently being imposed on the atoms
!! in the supercell are consistent with the calculated forces and the hypothesis
!! that the phonons are harmonic, or at least that the equilibrium structure is
!! a minimum of energy: takes the scalar product of the displacement by the
!! force, for each atom, and this should be negative
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! cartesian_displacements= displacement of all atoms from their initial
!!   equilibrium positions
!! fcart= forces on all atoms, in cartesian coordinates
!! natom= number of atoms in the full supercell
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_check_fcart(cartesian_displacements,fcart,natom)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom
!arrays
 real(dp),intent(in) :: cartesian_displacements(3,natom),fcart(3,natom)

!Local variables-------------------------------
!scalars
 integer :: iatom
 real(dp) :: scprod

! *************************************************************************

 write(*,*) 'natom=',natom

 write (*,'(a)') 'cartesian_displacements(3,natom) = '
 do iatom=1,natom
   write (*,'(3E20.10)')  cartesian_displacements(3,iatom)
 end do

 write (*,'(a)') 'fcart(3,natom) = '
 do iatom=1,natom
   write (*,'(3E20.10)')  fcart(3,iatom)
 end do

 write(*,*) ' Calculate scalar product of force times displacement.'
 write(*,*) '  Should be negative'
 do iatom=1,natom
   scprod=sum(fcart(:,iatom)*cartesian_displacements(:,iatom))
   write(*,*) 'atom ', iatom, ' F.dR = ', scprod
 end do


end subroutine scphon_check_fcart
!!***


!!****f* ABINIT/scphon_free_energy
!! NAME
!! scphon_free_energy
!!
!! FUNCTION
!! Calculate the phonon Free energy, from the input Density of States.
!!
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! istep= number of the present iteration of the SC phonon calculations, for
!!   printing purposes
!! phonon_dos=array containing the phonon DOS
!! scphon_temp= phononic temperature in Ha
!!
!! OUTPUT
!! free_energy= value of the Free energy
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_free_energy(free_energy,istep,t_phonon_dos,scphon_temp)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_phdos

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istep
 real(dp),intent(in) :: scphon_temp
 real(dp),intent(out) :: free_energy
 type(phonon_dos_type),intent(inout) :: t_phonon_dos

!Local variables-------------------------------
!scalars
 integer :: ifreq
 real(dp) :: freq,inv_temp
 character(len=500) :: message
!arrays
 real(dp),allocatable :: free_energy_int(:),free_energy_integrand(:)

! *************************************************************************

 inv_temp = one / scphon_temp

!integrate free energy from DOS
 allocate (free_energy_integrand(t_phonon_dos%nomega), free_energy_int(t_phonon_dos%nomega))
 free_energy_integrand=zero
 do ifreq=1,t_phonon_dos%nomega
   freq=t_phonon_dos%omega_min+dble(ifreq-1)*(t_phonon_dos%omega_max-t_phonon_dos%omega_min)/dble(t_phonon_dos%nomega-1)
!  0 point energy
   free_energy_integrand(ifreq)=t_phonon_dos%phdos(ifreq)*freq*half
!  rest of the free energy
   if ((freq*inv_temp) < 30._dp .and. (freq*inv_temp) > tol12) then
     free_energy_integrand(ifreq)=free_energy_integrand(ifreq) &
&     + t_phonon_dos%phdos(ifreq)*scphon_temp*log(one-exp(-freq*inv_temp))
!    else if((freq*inv_temp) < tol12) then
!    free_energy_integrand(ifreq)=free_energy_integrand(ifreq) &
!    &     + t_phonon_dos%phdos(ifreq)*scphon_temp*
   end if
!  write (*,*) ' freen integrand ', freq, free_energy_integrand(ifreq)
 end do
 free_energy_int=zero
 call simpson_int(t_phonon_dos%nomega,t_phonon_dos%omega_step,free_energy_integrand,free_energy_int)
 free_energy = free_energy_int(t_phonon_dos%nomega)


 write (message,'(a,I6,a,E10.3,a)') ' Free energy at iteration ', istep, &
& ' and temperature T= ',scphon_temp/kb_HaK, ' K is:'
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')
 write (message,'(a,i6,E20.8)') 'FREEEN ',  istep, free_energy
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')

 deallocate (free_energy_int,free_energy_integrand)

end subroutine scphon_free_energy
!!***


!!****f* ABINIT/scphon_make_phonon_dos
!! NAME
!! scphon_make_phonon_dos
!!
!! FUNCTION
!! Simple Gaussian summation of the density of states for the phonon frequencies
!! given in input. This is robust, but superceded by
!! scphon_interpolate_phonon_and_dos which does much better by interpolating the
!! phonon frequencies explicitly before summing the DOS.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! dos_smearing= smearing width used to smooth the phonon DOS
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nfreq_int= number of frequencies in the grid on which the phonon DOS is
!!   calculated
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! pcell=container type with ancillary variables and dimensions from anaddb run
!! phonon_eigval=phonon eigenfrequencies, updated inside SC phonon run
!!
!! OUTPUT
!! maxfreq=maximum frequency for which the DOS is calculated
!! minfreq=minimum frequency for which the DOS is calculated
!! phonon_dos=array containing the phonon DOS
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_make_phonon_dos (dos_smearing,natom_primitive_cell,&
&    nfreq_int,nphononq,maxfreq,minfreq,phonon_dos,phonon_eigval)

 use defs_basis
 use defs_datatypes
 use defs_abitypes

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_32_util
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom_primitive_cell,nfreq_int,nphononq
 real(dp),intent(in) :: dos_smearing
 real(dp),intent(out) :: maxfreq,minfreq
!arrays
 real(dp),intent(in) :: phonon_eigval(3*natom_primitive_cell,nphononq)
 real(dp),intent(out) :: phonon_dos(nfreq_int)

!Local variables-------------------------------
!scalars
 integer :: ifreq,imode,iq,unit_phondos
 real(dp) :: domega,freq,gauss_prefactor,inv_dos_smearing
!arrays
 real(dp) :: phonon_dos_int(nfreq_int)

! *************************************************************************

 inv_dos_smearing = one / dos_smearing
 gauss_prefactor = one/dos_smearing/sqrt(pi)

 maxfreq=1.1_dp*maxval(phonon_eigval)
 minfreq=minval(phonon_eigval)

!add a few negative frequencies to account for smearing of acoustic modes
 minfreq=minfreq-dos_smearing*five

 domega=(maxfreq-minfreq)/dble(nfreq_int-1)

!calculate phonon DOS from (small) number of frequencies we have
 phonon_dos= zero
 do iq=1,nphononq
   do imode=1,3*natom_primitive_cell
     do ifreq=1,nfreq_int
       freq=(minfreq+dble(ifreq-1)*(maxfreq-minfreq)/dble(nfreq_int-1) &
&       - phonon_eigval(imode,iq))*inv_dos_smearing
       if (abs(freq) > seven) cycle
       phonon_dos(ifreq) = phonon_dos(ifreq) + gauss_prefactor*exp(-freq*freq)
     end do
   end do
 end do

!normalize for number of qpoints
 phonon_dos=phonon_dos/dble(nphononq)

 unit_phondos=401
 open (unit=unit_phondos,file='phonondos.dat')
 write (unit_phondos,*) '#'
 write (unit_phondos,*) '#  phonon dos calculated from self-consistent phonon spectrum'
 write (unit_phondos,*) '#'
 do ifreq=1,nfreq_int
   freq=minfreq+dble(ifreq-1)*(maxfreq-minfreq)/dble(nfreq_int-1)
   write (unit_phondos,*) freq, phonon_dos(ifreq)
 end do

 phonon_dos_int=zero
 call simpson_int(nfreq_int,domega,phonon_dos,phonon_dos_int)

 write (unit_phondos,'(a,F10.3)') '# integral = ', phonon_dos_int(nfreq_int)
 close(unit_phondos)

end subroutine scphon_make_phonon_dos
!!***



!!****f* ABINIT/scphon_interpolate_phonon_and_dos
!! NAME
!! scphon_interpolate_phonon_and_dos
!!
!! FUNCTION
!! Interpolate the phonon Density of States, from frequencies updated inside SC
!! phonon run
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! pcell=container type with ancillary variables and dimensions from anaddb run
!! phonon_eigval=phonon eigenfrequencies, updated inside SC phonon run
!! phonon_eigvec_ref=reference phonon eigenvectors, from the anaddb equil run
!! phononq= phonon q vectors used in anaddb run (reduced coordinates)
!! supercell_multiplicity=number of times the primitive unit cell is repeated
!!   along each axis, in the supercell
!!
!! OUTPUT
!! t_phonon_dos=type containing the DOS, partial DOS, and dimensions
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_interpolate_phonon_and_dos (natom_primitive_cell,&
&    nphononq,pcell,t_phonon_dos,phonon_eigval,phonon_eigvec_ref,&
&    phononq,supercell_multiplicity)

 use defs_basis
 use defs_datatypes
 use defs_abitypes
 use m_phdos
 use m_primcell_ddb_info

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_77_ddb
 use interfaces_95_drive, except_this_one => scphon_interpolate_phonon_and_dos
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom_primitive_cell,nphononq
 type(phonon_dos_type),intent(inout) :: t_phonon_dos
 type(primcell_ddb_info),intent(inout) :: pcell
!arrays
 integer,intent(in) :: supercell_multiplicity(3)
 real(dp),intent(in) :: phonon_eigval(3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: phonon_eigvec_ref(2,3*natom_primitive_cell,3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: phononq(3,nphononq)

!Local variables-------------------------------
!scalars
 integer :: msize
 real(dp) :: tcpui,twalli
 type(anaddb_dataset_type) :: dummy_anaddb_dtset
!arrays
 integer :: ngqpt(3)
 real(dp),allocatable :: atmfrc(:,:,:,:,:,:),dynmat(:,:,:,:,:,:)

! *************************************************************************

!Source code

 write (*,*) ' in scphon_interpolate_phonon_and_dos'

!test if brav was used in initial anaddb run
 if(pcell%brav /= 1) stop ' scphon_interpolate_phonon_and_dos : Error - brav/=1 not coded '

!
!The following initializations will be moved to a central routine later
!

!this is the fine grid on which the phonons are interpolated
 ngqpt(:) = 20

!fill defaults for anaddb_dtset and other inputs to mkphdos
 dummy_anaddb_dtset%prtdos = 1 ! use gaussian integ by default
 dummy_anaddb_dtset%dosdeltae = t_phonon_dos%omega_step
 dummy_anaddb_dtset%dossmear = t_phonon_dos%dossmear
 dummy_anaddb_dtset%brav = 1   ! do not use brav symmetrization for the moment
 dummy_anaddb_dtset%ngqpt = 0  ! for brav==1 the other 6 values should not be used
 dummy_anaddb_dtset%ngqpt(1:3) = supercell_multiplicity
 dummy_anaddb_dtset%nqshft = 1 ! force 1 shift to 000
 dummy_anaddb_dtset%q1shft = zero
 dummy_anaddb_dtset%ng2qpt = ngqpt
 dummy_anaddb_dtset%q2shft = zero
 dummy_anaddb_dtset%dipdip = 1    ! include dipole dipole interaction term?
 dummy_anaddb_dtset%symdynmat = 1 ! force symmetrization of dynamical matrix

!
!END initializations indep of presently calculated frequencies
!

!reconstitute dynamical matrices on qpoints we know
 msize=3*pcell%mpert*3*pcell%mpert
 allocate (dynmat(2,3,natom_primitive_cell,3,natom_primitive_cell,nphononq))
 call scphon_freq_to_dynmat(dynmat,natom_primitive_cell,&
& nphononq,phonon_eigval,phonon_eigvec_ref)

!calculate atomic force constants
 allocate(atmfrc(2,3,natom_primitive_cell,3,natom_primitive_cell,pcell%nrpt))
 call ftiaf9(atmfrc,dynmat,pcell%gprim,pcell%natom,nphononq,&
& pcell%nrpt,1,pcell%rpt,phononq,pcell%wghatm)

!interpolate the DOS (should also work for tetrahedron method)
 tcpui=zero
 twalli=zero
 call mkphdos(pcell%acell,pcell%amu,dummy_anaddb_dtset,atmfrc,pcell%dielt,pcell%dyewq0,&
& pcell%gmet,pcell%gprim,pcell%indsym,&
& pcell%mpert,pcell%msym,pcell%natom,pcell%nrpt,pcell%nsym,&
& t_phonon_dos%ntypat,t_phonon_dos,pcell%rmet,pcell%rprim,&
& pcell%rpt,pcell%symrec,pcell%symrel,tcpui,&
& pcell%trans,twalli,pcell%typat,pcell%ucvol,pcell%wghatm,pcell%xred,pcell%zeff)

 call print_phondos(t_phonon_dos)

end subroutine scphon_interpolate_phonon_and_dos
!!***

!{\src2tex{textfont=tt}}
!!****f* ABINIT/scphon_freq_to_dynmat
!! NAME
!! scphon_freq_to_dynmat
!!
!! FUNCTION
!!  From the updated frequencies and constant, reference, phonon eigenvectors
!!  this routine recalculates the dynamical matrices, which will be used
!!  for phonon interpolation.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! ------ Removed in beautification MS ------
!! msize=maximum size of dynamical matrix, not used at present
!! ------------------------------------------
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! ------ Removed in beautification MS ------
!! pcell=container type with ancillary variables and dimensions from anaddb run
!! ------------------------------------------
!! phonon_eigval=phonon eigenfrequencies, updated inside SC phonon run
!! phonon_eigvec_ref=reference phonon eigenvectors, from the anaddb equil run
!!
!! OUTPUT
!! dynmat=dynamical matrix recalculated from phonon_eigval and phonon_eigvec_ref
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine scphon_freq_to_dynmat(dynmat,natom_primitive_cell,&
&     nphononq,phonon_eigval,phonon_eigvec_ref)

 use defs_basis
 use m_primcell_ddb_info

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: natom_primitive_cell,nphononq !,msize
!type(primcell_ddb_info),intent(inout) :: pcell
!arrays
 real(dp),intent(in) :: phonon_eigval(3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: phonon_eigvec_ref(2,3*natom_primitive_cell,3*natom_primitive_cell,nphononq)
 real(dp),intent(out) :: dynmat(2,3,natom_primitive_cell,3,natom_primitive_cell,nphononq)

!Local variables-------------------------------
!scalars
 integer :: imode,iq,nmode
!character(len=500) :: message
!arrays
 real(dp) :: tmpeigvec(2,3*natom_primitive_cell,3*natom_primitive_cell)
 real(dp) :: tmpmat(2,3*natom_primitive_cell,3*natom_primitive_cell)
 real(dp) :: tmpmat2(2,3*natom_primitive_cell,3*natom_primitive_cell),z_one(2)
 real(dp) :: z_zero(2)

! *************************************************************************

 nmode=3*natom_primitive_cell
 z_one=(/one,zero/)
 z_zero=zero

!loop over qpoints
 do iq=1,nphononq

   tmpeigvec=reshape(phonon_eigvec_ref(:,:,:,iq),&
&   (/2,3*natom_primitive_cell,3*natom_primitive_cell/))

!  square eigenvalues
   tmpmat=zero
   do imode=1,3*natom_primitive_cell
     tmpmat(1,imode,imode)=sign(one,phonon_eigval(imode,iq))*phonon_eigval(imode,iq)**2
   end do

!  ZGEMM (TRANSA,TRANSB,M,N,K,ALPHA,A,LDA,B,LDB,BETA,C,LDC)
!  multiply by eigvec left (no complex conjugation)
   tmpmat2=zero
   call zgemm ('N','N',nmode,nmode,nmode,z_one,&
&   tmpeigvec,nmode,&
&   tmpmat,nmode,&
&   z_zero,tmpmat2,nmode)

!  multiply by eigvec right, with trans conjugate
   tmpmat=zero
   call zgemm ('N','C',nmode,nmode,nmode,z_one,&
&   tmpmat2,nmode,&
&   tmpeigvec,nmode,&
&   z_zero,tmpmat,nmode)

   dynmat(:,:,:,:,:,iq)=reshape(tmpmat,(/2,3,natom_primitive_cell,3,natom_primitive_cell/))

!  include sqrt of masses?
!  dynmat(:,:,iatom2,:,iatom,iq) = dynmat(:,:,iatom2,:,iatom,iq)  &
!  &         / sqrt(pcell%amu(pcell%typat(iatom))*pcell%amu(pcell%typat(iatom2)) &
!  &         / amu_emass
 end do ! iq


end subroutine scphon_freq_to_dynmat
!!***


!{\src2tex{textfont=tt}}
!!****f* ABINIT/scphon_dynmat_to_freq2
!! NAME
!! scphon_dynmat_to_freq2
!!
!! FUNCTION
!!  From the dynamical matrices, calculate the corresponding frequencies (squared)
!!  and see if reference phonon eigenvectors are still eigenvectors.
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! dynmat=dynamical matrix recalculated from phonon_eigval and phonon_eigvec_ref
!! ------ Removed in beautification MS ------
!! msize=maximum size of dynamical matrix, not used at present
!! ------------------------------------------
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! ------- Removed in beautification because unused MS ------
!! pcell=container type with ancillary variables and dimensions from anaddb run
!! ----------------------------------------------------------
!! phonon_eigvec_ref=reference phonon eigenvectors, from the anaddb equil run
!!
!! OUTPUT
!! phonon_eigval=phonon eigenfrequencies, updated inside SC phonon run
!!
!! SIDE EFFECTS
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE
subroutine scphon_dynmat_to_freq2(dynmat,natom_primitive_cell,&
&     nphononq,phonon_eigval,phonon_eigvec_ref)

 use defs_basis
 use m_primcell_ddb_info

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_14_hidewrite
!End of the abilint section

 implicit none

!Arguments -------------------------------

 !scalars
 integer, intent(in) :: natom_primitive_cell,nphononq ! ,msize
 !type(primcell_ddb_info),intent(inout) :: pcell
 !arrays
 real(dp),intent(in) :: phonon_eigvec_ref(2,3*natom_primitive_cell,&
&                                           3*natom_primitive_cell,nphononq)
 real(dp),intent(in) :: dynmat(2,3,natom_primitive_cell,3,natom_primitive_cell,nphononq)
 real(dp),intent(out) :: phonon_eigval(3*natom_primitive_cell,nphononq)

!Local variables -------------------------

 integer :: iq,imode,nmode,imode2!,iatom
 real(dp) :: z_one(2), z_zero(2)
 real(dp) :: tmpmat(2,3*natom_primitive_cell,3*natom_primitive_cell)
 real(dp) :: tmpmat2(2,3*natom_primitive_cell,3*natom_primitive_cell)
 real(dp) :: tmpeigvec(2,3*natom_primitive_cell,3*natom_primitive_cell)
 character(len=500) :: message

! *********************************************************************

 nmode=3*natom_primitive_cell
 z_one=(/one,zero/)
 z_zero=zero

 write (message,'(a)') 'off diagonal elements of evec* dynmat evec'
 call wrtout(ab_out,message,'COLL')
 call wrtout(std_out,  message,'COLL')

!loop over qpoints
 do iq=1,nphononq

   tmpeigvec=reshape(phonon_eigvec_ref(:,:,:,iq),&
&   (/2,3*natom_primitive_cell,3*natom_primitive_cell/))

   tmpmat = reshape(dynmat(:,:,:,:,:,iq),(/2,3*natom_primitive_cell,3*natom_primitive_cell/))

!  ZGEMM (TRANSA,TRANSB,M,N,K,ALPHA,A,LDA,B,LDB,BETA,C,LDC)
!  multiply by eigvec left with trans conjugate
   tmpmat2=zero
   call zgemm ('C','N',nmode,nmode,nmode,z_one,&
&   tmpeigvec,nmode,&
&   tmpmat,nmode,&
&   z_zero,tmpmat2,nmode)

!  multiply by eigvec right, (no complex conjugation)
   tmpmat=zero
   call zgemm ('N','N',nmode,nmode,nmode,z_one,&
&   tmpmat2,nmode,&
&   tmpeigvec,nmode,&
&   z_zero,tmpmat,nmode)

!  should we check imaginary part too?
   phonon_eigval(imode,iq) = tmpmat(1,imode,imode)

   do imode=1,nmode
     do imode2=1,nmode
       if (imode2==imode) cycle
       if (abs(tmpmat(1,imode,imode2)) > tol10) then
         write (message,'(2I6,2x,2E12.3)') imode,imode2,tmpmat(:,imode,imode2)
         call wrtout(ab_out,message,'COLL')
         call wrtout(std_out,  message,'COLL')
       end if
     end do
   end do
 end do ! iq

end subroutine scphon_dynmat_to_freq2
!!***

!!****f* ABINIT/print_phonfreq
!! NAME
!! print_phonfreq
!!
!! FUNCTION
!!   print phonon frequencies to standardized file SCphon_TIMx
!!
!! COPYRIGHT
!! Copyright (C) 2009-2010 ABINIT group (MJV)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!! istep=iteration number in SC phonon run -1 gives TIM0, as for Broyden algorithm
!! natom_primitive_cell=number of atoms in primitive cell (not supercell used
!!   for SC phonon calculation)
!! nphononq=number of phonon q-vectors input from anaddb run at equilibrium
!!   geometry
!! phonon_eigval=phonon eigenfrequencies, updated inside SC phonon run
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  writes to file
!!
!! PARENTS
!!      scphon
!!
!! CHILDREN
!!      fappnd
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine print_phonfreq(istep,natom_primitive_cell,nphononq,phonon_eigval)

 use defs_basis
 use m_io_tools

!This section has been created automatically by the script Abilint (TD).
!Do not modify the following lines by hand.
 use interfaces_62_iowfdenpot
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istep,natom_primitive_cell,nphononq
!arrays
 real(dp),intent(in) :: phonon_eigval(3*natom_primitive_cell,nphononq)

!Local variables-------------------------------
! scfrq_unit= file unit for output of SC phonon frequencies as a function of iteration
! FIXME: replace by call to getunit
! should become input from files file out radix
!scalars
 integer :: imode_primitive_cell,iq,scfrq_unit
 character(len=fnlen) :: outfilename_radix
 character(len=fnlen) :: phonon_freq_filename

! *************************************************************************

 outfilename_radix="SCphon"
 call fappnd(phonon_freq_filename,outfilename_radix,istep)
 phonon_freq_filename = trim(phonon_freq_filename)//"_PHFRQ"
 scfrq_unit = get_unit()
 open (unit=scfrq_unit,file=phonon_freq_filename)
 write (scfrq_unit,*) '#'
 write (scfrq_unit,*) '# phonon frequencies (in Ha) on qph1l list of qpoints'
 write (scfrq_unit,*) '#'

 do iq=1,nphononq
   write (scfrq_unit,'(i6)',ADVANCE='NO') iq
   do imode_primitive_cell=1,3*natom_primitive_cell
     write (scfrq_unit,'(E16.8,2x)',ADVANCE='NO') phonon_eigval(imode_primitive_cell,iq)
   end do
   write (scfrq_unit,*)
 end do

 close (scfrq_unit)

end subroutine print_phonfreq
!!***
